/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.migrationhubconfig;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.migrationhubconfig.internal.MigrationHubConfigServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.migrationhubconfig.model.AccessDeniedException;
import software.amazon.awssdk.services.migrationhubconfig.model.CreateHomeRegionControlRequest;
import software.amazon.awssdk.services.migrationhubconfig.model.CreateHomeRegionControlResponse;
import software.amazon.awssdk.services.migrationhubconfig.model.DeleteHomeRegionControlRequest;
import software.amazon.awssdk.services.migrationhubconfig.model.DeleteHomeRegionControlResponse;
import software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsRequest;
import software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsResponse;
import software.amazon.awssdk.services.migrationhubconfig.model.DryRunOperationException;
import software.amazon.awssdk.services.migrationhubconfig.model.GetHomeRegionRequest;
import software.amazon.awssdk.services.migrationhubconfig.model.GetHomeRegionResponse;
import software.amazon.awssdk.services.migrationhubconfig.model.InternalServerErrorException;
import software.amazon.awssdk.services.migrationhubconfig.model.InvalidInputException;
import software.amazon.awssdk.services.migrationhubconfig.model.MigrationHubConfigException;
import software.amazon.awssdk.services.migrationhubconfig.model.ServiceUnavailableException;
import software.amazon.awssdk.services.migrationhubconfig.model.ThrottlingException;
import software.amazon.awssdk.services.migrationhubconfig.transform.CreateHomeRegionControlRequestMarshaller;
import software.amazon.awssdk.services.migrationhubconfig.transform.DeleteHomeRegionControlRequestMarshaller;
import software.amazon.awssdk.services.migrationhubconfig.transform.DescribeHomeRegionControlsRequestMarshaller;
import software.amazon.awssdk.services.migrationhubconfig.transform.GetHomeRegionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MigrationHubConfigAsyncClient}.
 *
 * @see MigrationHubConfigAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMigrationHubConfigAsyncClient implements MigrationHubConfigAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMigrationHubConfigAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMigrationHubConfigAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * This API sets up the home region for the calling account only.
     * </p>
     *
     * @param createHomeRegionControlRequest
     * @return A Java Future containing the result of the CreateHomeRegionControl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when a request fails due to temporary unavailability of
     *         the service.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>DryRunOperationException Exception raised to indicate that authorization of an action was successful,
     *         when the <code>DryRun</code> flag is set to true.</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubConfigException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubConfigAsyncClient.CreateHomeRegionControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhub-config-2019-06-30/CreateHomeRegionControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateHomeRegionControlResponse> createHomeRegionControl(
            CreateHomeRegionControlRequest createHomeRegionControlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createHomeRegionControlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHomeRegionControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHub Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHomeRegionControl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateHomeRegionControlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateHomeRegionControlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateHomeRegionControlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateHomeRegionControlRequest, CreateHomeRegionControlResponse>()
                            .withOperationName("CreateHomeRegionControl").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateHomeRegionControlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createHomeRegionControlRequest));
            CompletableFuture<CreateHomeRegionControlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation deletes the home region configuration for the calling account. The operation does not delete
     * discovery or migration tracking data in the home region.
     * </p>
     *
     * @param deleteHomeRegionControlRequest
     * @return A Java Future containing the result of the DeleteHomeRegionControl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when a request fails due to temporary unavailability of
     *         the service.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubConfigException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubConfigAsyncClient.DeleteHomeRegionControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhub-config-2019-06-30/DeleteHomeRegionControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteHomeRegionControlResponse> deleteHomeRegionControl(
            DeleteHomeRegionControlRequest deleteHomeRegionControlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteHomeRegionControlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteHomeRegionControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHub Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHomeRegionControl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteHomeRegionControlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteHomeRegionControlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteHomeRegionControlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteHomeRegionControlRequest, DeleteHomeRegionControlResponse>()
                            .withOperationName("DeleteHomeRegionControl").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteHomeRegionControlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteHomeRegionControlRequest));
            CompletableFuture<DeleteHomeRegionControlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This API permits filtering on the <code>ControlId</code> and <code>HomeRegion</code> fields.
     * </p>
     *
     * @param describeHomeRegionControlsRequest
     * @return A Java Future containing the result of the DescribeHomeRegionControls operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when a request fails due to temporary unavailability of
     *         the service.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubConfigException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubConfigAsyncClient.DescribeHomeRegionControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhub-config-2019-06-30/DescribeHomeRegionControls"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeHomeRegionControlsResponse> describeHomeRegionControls(
            DescribeHomeRegionControlsRequest describeHomeRegionControlsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeHomeRegionControlsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeHomeRegionControlsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHub Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeHomeRegionControls");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeHomeRegionControlsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeHomeRegionControlsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeHomeRegionControlsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeHomeRegionControlsRequest, DescribeHomeRegionControlsResponse>()
                            .withOperationName("DescribeHomeRegionControls").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeHomeRegionControlsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeHomeRegionControlsRequest));
            CompletableFuture<DescribeHomeRegionControlsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the calling account’s home region, if configured. This API is used by other AWS services to determine the
     * regional endpoint for calling AWS Application Discovery Service and Migration Hub. You must call
     * <code>GetHomeRegion</code> at least once before you call any other AWS Application Discovery Service and AWS
     * Migration Hub APIs, to obtain the account's Migration Hub home region.
     * </p>
     *
     * @param getHomeRegionRequest
     * @return A Java Future containing the result of the GetHomeRegion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException Exception raised when an internal, configuration, or dependency error is
     *         encountered.</li>
     *         <li>ServiceUnavailableException Exception raised when a request fails due to temporary unavailability of
     *         the service.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InvalidInputException Exception raised when the provided input violates a policy constraint or is
     *         entered in the wrong format or data type.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubConfigException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubConfigAsyncClient.GetHomeRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhub-config-2019-06-30/GetHomeRegion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetHomeRegionResponse> getHomeRegion(GetHomeRegionRequest getHomeRegionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getHomeRegionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getHomeRegionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHub Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetHomeRegion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetHomeRegionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetHomeRegionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetHomeRegionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetHomeRegionRequest, GetHomeRegionResponse>()
                            .withOperationName("GetHomeRegion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetHomeRegionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getHomeRegionRequest));
            CompletableFuture<GetHomeRegionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final MigrationHubConfigServiceClientConfiguration serviceClientConfiguration() {
        return new MigrationHubConfigServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MigrationHubConfigException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DryRunOperation")
                                .exceptionBuilderSupplier(DryRunOperationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        MigrationHubConfigServiceClientConfigurationBuilder serviceConfigBuilder = new MigrationHubConfigServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
