/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.migrationhubstrategy;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.migrationhubstrategy.model.AccessDeniedException;
import software.amazon.awssdk.services.migrationhubstrategy.model.ConflictException;
import software.amazon.awssdk.services.migrationhubstrategy.model.DependencyException;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetApplicationComponentDetailsRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetApplicationComponentDetailsResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetApplicationComponentStrategiesRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetApplicationComponentStrategiesResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetAssessmentRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetAssessmentResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetImportFileTaskRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetImportFileTaskResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetLatestAssessmentIdRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetLatestAssessmentIdResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetPortfolioPreferencesRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetPortfolioPreferencesResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetPortfolioSummaryRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetPortfolioSummaryResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetRecommendationReportDetailsRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetRecommendationReportDetailsResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetServerDetailsRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetServerDetailsResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetServerStrategiesRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.GetServerStrategiesResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.InternalServerException;
import software.amazon.awssdk.services.migrationhubstrategy.model.ListApplicationComponentsRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.ListApplicationComponentsResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.ListCollectorsRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.ListCollectorsResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.ListImportFileTaskRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.ListImportFileTaskResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.ListServersRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.ListServersResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.MigrationHubStrategyException;
import software.amazon.awssdk.services.migrationhubstrategy.model.MigrationHubStrategyRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.PutPortfolioPreferencesRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.PutPortfolioPreferencesResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.ResourceNotFoundException;
import software.amazon.awssdk.services.migrationhubstrategy.model.ServiceLinkedRoleLockClientException;
import software.amazon.awssdk.services.migrationhubstrategy.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.migrationhubstrategy.model.StartAssessmentRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.StartAssessmentResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.StartImportFileTaskRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.StartImportFileTaskResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.StartRecommendationReportGenerationRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.StartRecommendationReportGenerationResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.StopAssessmentRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.StopAssessmentResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.ThrottlingException;
import software.amazon.awssdk.services.migrationhubstrategy.model.UpdateApplicationComponentConfigRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.UpdateApplicationComponentConfigResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.UpdateServerConfigRequest;
import software.amazon.awssdk.services.migrationhubstrategy.model.UpdateServerConfigResponse;
import software.amazon.awssdk.services.migrationhubstrategy.model.ValidationException;
import software.amazon.awssdk.services.migrationhubstrategy.paginators.GetServerDetailsPublisher;
import software.amazon.awssdk.services.migrationhubstrategy.paginators.ListApplicationComponentsPublisher;
import software.amazon.awssdk.services.migrationhubstrategy.paginators.ListCollectorsPublisher;
import software.amazon.awssdk.services.migrationhubstrategy.paginators.ListImportFileTaskPublisher;
import software.amazon.awssdk.services.migrationhubstrategy.paginators.ListServersPublisher;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetApplicationComponentDetailsRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetApplicationComponentStrategiesRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetAssessmentRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetImportFileTaskRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetLatestAssessmentIdRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetPortfolioPreferencesRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetPortfolioSummaryRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetRecommendationReportDetailsRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetServerDetailsRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.GetServerStrategiesRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.ListApplicationComponentsRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.ListCollectorsRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.ListImportFileTaskRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.ListServersRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.PutPortfolioPreferencesRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.StartAssessmentRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.StartImportFileTaskRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.StartRecommendationReportGenerationRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.StopAssessmentRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.UpdateApplicationComponentConfigRequestMarshaller;
import software.amazon.awssdk.services.migrationhubstrategy.transform.UpdateServerConfigRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MigrationHubStrategyAsyncClient}.
 *
 * @see MigrationHubStrategyAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMigrationHubStrategyAsyncClient implements MigrationHubStrategyAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMigrationHubStrategyAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final MigrationHubStrategyServiceClientConfiguration serviceClientConfiguration;

    protected DefaultMigrationHubStrategyAsyncClient(MigrationHubStrategyServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Retrieves details about an application component.
     * </p>
     *
     * @param getApplicationComponentDetailsRequest
     * @return A Java Future containing the result of the GetApplicationComponentDetails operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified ID in the request is not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.GetApplicationComponentDetails
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetApplicationComponentDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetApplicationComponentDetailsResponse> getApplicationComponentDetails(
            GetApplicationComponentDetailsRequest getApplicationComponentDetailsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getApplicationComponentDetailsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApplicationComponentDetails");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetApplicationComponentDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetApplicationComponentDetailsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetApplicationComponentDetailsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetApplicationComponentDetailsRequest, GetApplicationComponentDetailsResponse>()
                            .withOperationName("GetApplicationComponentDetails")
                            .withMarshaller(new GetApplicationComponentDetailsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getApplicationComponentDetailsRequest));
            CompletableFuture<GetApplicationComponentDetailsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of all the recommended strategies and tools for an application component running on a server.
     * </p>
     *
     * @param getApplicationComponentStrategiesRequest
     * @return A Java Future containing the result of the GetApplicationComponentStrategies operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified ID in the request is not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.GetApplicationComponentStrategies
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetApplicationComponentStrategies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetApplicationComponentStrategiesResponse> getApplicationComponentStrategies(
            GetApplicationComponentStrategiesRequest getApplicationComponentStrategiesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getApplicationComponentStrategiesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApplicationComponentStrategies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetApplicationComponentStrategiesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetApplicationComponentStrategiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetApplicationComponentStrategiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetApplicationComponentStrategiesRequest, GetApplicationComponentStrategiesResponse>()
                            .withOperationName("GetApplicationComponentStrategies")
                            .withMarshaller(new GetApplicationComponentStrategiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getApplicationComponentStrategiesRequest));
            CompletableFuture<GetApplicationComponentStrategiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the status of an on-going assessment.
     * </p>
     *
     * @param getAssessmentRequest
     * @return A Java Future containing the result of the GetAssessment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified ID in the request is not found.</li>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.GetAssessment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetAssessment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAssessmentResponse> getAssessment(GetAssessmentRequest getAssessmentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAssessmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAssessment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAssessmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAssessmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAssessmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAssessmentRequest, GetAssessmentResponse>()
                            .withOperationName("GetAssessment")
                            .withMarshaller(new GetAssessmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAssessmentRequest));
            CompletableFuture<GetAssessmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the details about a specific import task.
     * </p>
     *
     * @param getImportFileTaskRequest
     * @return A Java Future containing the result of the GetImportFileTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified ID in the request is not found.</li>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.GetImportFileTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetImportFileTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetImportFileTaskResponse> getImportFileTask(GetImportFileTaskRequest getImportFileTaskRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getImportFileTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetImportFileTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetImportFileTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetImportFileTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetImportFileTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetImportFileTaskRequest, GetImportFileTaskResponse>()
                            .withOperationName("GetImportFileTask")
                            .withMarshaller(new GetImportFileTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getImportFileTaskRequest));
            CompletableFuture<GetImportFileTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve the latest ID of a specific assessment task.
     * </p>
     *
     * @param getLatestAssessmentIdRequest
     * @return A Java Future containing the result of the GetLatestAssessmentId operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.GetLatestAssessmentId
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetLatestAssessmentId"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLatestAssessmentIdResponse> getLatestAssessmentId(
            GetLatestAssessmentIdRequest getLatestAssessmentIdRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLatestAssessmentIdRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLatestAssessmentId");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLatestAssessmentIdResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLatestAssessmentIdResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLatestAssessmentIdResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLatestAssessmentIdRequest, GetLatestAssessmentIdResponse>()
                            .withOperationName("GetLatestAssessmentId")
                            .withMarshaller(new GetLatestAssessmentIdRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getLatestAssessmentIdRequest));
            CompletableFuture<GetLatestAssessmentIdResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves your migration and modernization preferences.
     * </p>
     *
     * @param getPortfolioPreferencesRequest
     * @return A Java Future containing the result of the GetPortfolioPreferences operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified ID in the request is not found.</li>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.GetPortfolioPreferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetPortfolioPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPortfolioPreferencesResponse> getPortfolioPreferences(
            GetPortfolioPreferencesRequest getPortfolioPreferencesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPortfolioPreferencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPortfolioPreferences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPortfolioPreferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPortfolioPreferencesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPortfolioPreferencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPortfolioPreferencesRequest, GetPortfolioPreferencesResponse>()
                            .withOperationName("GetPortfolioPreferences")
                            .withMarshaller(new GetPortfolioPreferencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPortfolioPreferencesRequest));
            CompletableFuture<GetPortfolioPreferencesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves overall summary including the number of servers to rehost and the overall number of anti-patterns.
     * </p>
     *
     * @param getPortfolioSummaryRequest
     * @return A Java Future containing the result of the GetPortfolioSummary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.GetPortfolioSummary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetPortfolioSummary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPortfolioSummaryResponse> getPortfolioSummary(
            GetPortfolioSummaryRequest getPortfolioSummaryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPortfolioSummaryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPortfolioSummary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPortfolioSummaryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPortfolioSummaryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPortfolioSummaryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPortfolioSummaryRequest, GetPortfolioSummaryResponse>()
                            .withOperationName("GetPortfolioSummary")
                            .withMarshaller(new GetPortfolioSummaryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPortfolioSummaryRequest));
            CompletableFuture<GetPortfolioSummaryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves detailed information about the specified recommendation report.
     * </p>
     *
     * @param getRecommendationReportDetailsRequest
     * @return A Java Future containing the result of the GetRecommendationReportDetails operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified ID in the request is not found.</li>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.GetRecommendationReportDetails
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetRecommendationReportDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRecommendationReportDetailsResponse> getRecommendationReportDetails(
            GetRecommendationReportDetailsRequest getRecommendationReportDetailsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getRecommendationReportDetailsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecommendationReportDetails");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRecommendationReportDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRecommendationReportDetailsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRecommendationReportDetailsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRecommendationReportDetailsRequest, GetRecommendationReportDetailsResponse>()
                            .withOperationName("GetRecommendationReportDetails")
                            .withMarshaller(new GetRecommendationReportDetailsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getRecommendationReportDetailsRequest));
            CompletableFuture<GetRecommendationReportDetailsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves detailed information about a specified server.
     * </p>
     *
     * @param getServerDetailsRequest
     * @return A Java Future containing the result of the GetServerDetails operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified ID in the request is not found.</li>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.GetServerDetails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetServerDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServerDetailsResponse> getServerDetails(GetServerDetailsRequest getServerDetailsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServerDetailsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServerDetails");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServerDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetServerDetailsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServerDetailsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServerDetailsRequest, GetServerDetailsResponse>()
                            .withOperationName("GetServerDetails")
                            .withMarshaller(new GetServerDetailsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getServerDetailsRequest));
            CompletableFuture<GetServerDetailsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves detailed information about a specified server.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getServerDetails(software.amazon.awssdk.services.migrationhubstrategy.model.GetServerDetailsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.GetServerDetailsPublisher publisher = client.getServerDetailsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.GetServerDetailsPublisher publisher = client.getServerDetailsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.migrationhubstrategy.model.GetServerDetailsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.migrationhubstrategy.model.GetServerDetailsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getServerDetails(software.amazon.awssdk.services.migrationhubstrategy.model.GetServerDetailsRequest)}
     * operation.</b>
     * </p>
     *
     * @param getServerDetailsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified ID in the request is not found.</li>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.GetServerDetails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetServerDetails"
     *      target="_top">AWS API Documentation</a>
     */
    public GetServerDetailsPublisher getServerDetailsPaginator(GetServerDetailsRequest getServerDetailsRequest) {
        return new GetServerDetailsPublisher(this, applyPaginatorUserAgent(getServerDetailsRequest));
    }

    /**
     * <p>
     * Retrieves recommended strategies and tools for the specified server.
     * </p>
     *
     * @param getServerStrategiesRequest
     * @return A Java Future containing the result of the GetServerStrategies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified ID in the request is not found.</li>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.GetServerStrategies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/GetServerStrategies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServerStrategiesResponse> getServerStrategies(
            GetServerStrategiesRequest getServerStrategiesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServerStrategiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServerStrategies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServerStrategiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetServerStrategiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServerStrategiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServerStrategiesRequest, GetServerStrategiesResponse>()
                            .withOperationName("GetServerStrategies")
                            .withMarshaller(new GetServerStrategiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getServerStrategiesRequest));
            CompletableFuture<GetServerStrategiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of all the application components (processes).
     * </p>
     *
     * @param listApplicationComponentsRequest
     * @return A Java Future containing the result of the ListApplicationComponents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ServiceLinkedRoleLockClientException Exception to indicate that the service-linked role (SLR) is
     *         locked.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.ListApplicationComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/ListApplicationComponents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListApplicationComponentsResponse> listApplicationComponents(
            ListApplicationComponentsRequest listApplicationComponentsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationComponentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplicationComponents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApplicationComponentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListApplicationComponentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListApplicationComponentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApplicationComponentsRequest, ListApplicationComponentsResponse>()
                            .withOperationName("ListApplicationComponents")
                            .withMarshaller(new ListApplicationComponentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listApplicationComponentsRequest));
            CompletableFuture<ListApplicationComponentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of all the application components (processes).
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listApplicationComponents(software.amazon.awssdk.services.migrationhubstrategy.model.ListApplicationComponentsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.ListApplicationComponentsPublisher publisher = client.listApplicationComponentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.ListApplicationComponentsPublisher publisher = client.listApplicationComponentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.migrationhubstrategy.model.ListApplicationComponentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.migrationhubstrategy.model.ListApplicationComponentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listApplicationComponents(software.amazon.awssdk.services.migrationhubstrategy.model.ListApplicationComponentsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listApplicationComponentsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ServiceLinkedRoleLockClientException Exception to indicate that the service-linked role (SLR) is
     *         locked.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.ListApplicationComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/ListApplicationComponents"
     *      target="_top">AWS API Documentation</a>
     */
    public ListApplicationComponentsPublisher listApplicationComponentsPaginator(
            ListApplicationComponentsRequest listApplicationComponentsRequest) {
        return new ListApplicationComponentsPublisher(this, applyPaginatorUserAgent(listApplicationComponentsRequest));
    }

    /**
     * <p>
     * Retrieves a list of all the installed collectors.
     * </p>
     *
     * @param listCollectorsRequest
     * @return A Java Future containing the result of the ListCollectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.ListCollectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/ListCollectors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCollectorsResponse> listCollectors(ListCollectorsRequest listCollectorsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCollectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCollectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCollectorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCollectorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCollectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCollectorsRequest, ListCollectorsResponse>()
                            .withOperationName("ListCollectors")
                            .withMarshaller(new ListCollectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listCollectorsRequest));
            CompletableFuture<ListCollectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of all the installed collectors.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listCollectors(software.amazon.awssdk.services.migrationhubstrategy.model.ListCollectorsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.ListCollectorsPublisher publisher = client.listCollectorsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.ListCollectorsPublisher publisher = client.listCollectorsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.migrationhubstrategy.model.ListCollectorsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.migrationhubstrategy.model.ListCollectorsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCollectors(software.amazon.awssdk.services.migrationhubstrategy.model.ListCollectorsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listCollectorsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.ListCollectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/ListCollectors"
     *      target="_top">AWS API Documentation</a>
     */
    public ListCollectorsPublisher listCollectorsPaginator(ListCollectorsRequest listCollectorsRequest) {
        return new ListCollectorsPublisher(this, applyPaginatorUserAgent(listCollectorsRequest));
    }

    /**
     * <p>
     * Retrieves a list of all the imports performed.
     * </p>
     *
     * @param listImportFileTaskRequest
     * @return A Java Future containing the result of the ListImportFileTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.ListImportFileTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/ListImportFileTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListImportFileTaskResponse> listImportFileTask(ListImportFileTaskRequest listImportFileTaskRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listImportFileTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListImportFileTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListImportFileTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListImportFileTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListImportFileTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListImportFileTaskRequest, ListImportFileTaskResponse>()
                            .withOperationName("ListImportFileTask")
                            .withMarshaller(new ListImportFileTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listImportFileTaskRequest));
            CompletableFuture<ListImportFileTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of all the imports performed.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listImportFileTask(software.amazon.awssdk.services.migrationhubstrategy.model.ListImportFileTaskRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.ListImportFileTaskPublisher publisher = client.listImportFileTaskPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.ListImportFileTaskPublisher publisher = client.listImportFileTaskPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.migrationhubstrategy.model.ListImportFileTaskResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.migrationhubstrategy.model.ListImportFileTaskResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listImportFileTask(software.amazon.awssdk.services.migrationhubstrategy.model.ListImportFileTaskRequest)}
     * operation.</b>
     * </p>
     *
     * @param listImportFileTaskRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.ListImportFileTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/ListImportFileTask"
     *      target="_top">AWS API Documentation</a>
     */
    public ListImportFileTaskPublisher listImportFileTaskPaginator(ListImportFileTaskRequest listImportFileTaskRequest) {
        return new ListImportFileTaskPublisher(this, applyPaginatorUserAgent(listImportFileTaskRequest));
    }

    /**
     * <p>
     * Returns a list of all the servers.
     * </p>
     *
     * @param listServersRequest
     * @return A Java Future containing the result of the ListServers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.ListServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/ListServers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServersResponse> listServers(ListServersRequest listServersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListServersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServersRequest, ListServersResponse>()
                            .withOperationName("ListServers").withMarshaller(new ListServersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listServersRequest));
            CompletableFuture<ListServersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all the servers.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listServers(software.amazon.awssdk.services.migrationhubstrategy.model.ListServersRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.ListServersPublisher publisher = client.listServersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubstrategy.paginators.ListServersPublisher publisher = client.listServersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.migrationhubstrategy.model.ListServersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.migrationhubstrategy.model.ListServersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listServers(software.amazon.awssdk.services.migrationhubstrategy.model.ListServersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listServersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.ListServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/ListServers"
     *      target="_top">AWS API Documentation</a>
     */
    public ListServersPublisher listServersPaginator(ListServersRequest listServersRequest) {
        return new ListServersPublisher(this, applyPaginatorUserAgent(listServersRequest));
    }

    /**
     * <p>
     * Saves the specified migration and modernization preferences.
     * </p>
     *
     * @param putPortfolioPreferencesRequest
     * @return A Java Future containing the result of the PutPortfolioPreferences operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>ConflictException Exception to indicate that there is an ongoing task when a new task is created.
     *         Return when once the existing tasks are complete.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.PutPortfolioPreferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/PutPortfolioPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutPortfolioPreferencesResponse> putPortfolioPreferences(
            PutPortfolioPreferencesRequest putPortfolioPreferencesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putPortfolioPreferencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutPortfolioPreferences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutPortfolioPreferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutPortfolioPreferencesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutPortfolioPreferencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutPortfolioPreferencesRequest, PutPortfolioPreferencesResponse>()
                            .withOperationName("PutPortfolioPreferences")
                            .withMarshaller(new PutPortfolioPreferencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putPortfolioPreferencesRequest));
            CompletableFuture<PutPortfolioPreferencesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts the assessment of an on-premises environment.
     * </p>
     *
     * @param startAssessmentRequest
     * @return A Java Future containing the result of the StartAssessment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ServiceQuotaExceededException The AWS account has reached its quota of imports. Contact AWS Support
     *         to increase the quota for this account.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.StartAssessment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/StartAssessment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartAssessmentResponse> startAssessment(StartAssessmentRequest startAssessmentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startAssessmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartAssessment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartAssessmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartAssessmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartAssessmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartAssessmentRequest, StartAssessmentResponse>()
                            .withOperationName("StartAssessment")
                            .withMarshaller(new StartAssessmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startAssessmentRequest));
            CompletableFuture<StartAssessmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a file import.
     * </p>
     *
     * @param startImportFileTaskRequest
     * @return A Java Future containing the result of the StartImportFileTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ServiceQuotaExceededException The AWS account has reached its quota of imports. Contact AWS Support
     *         to increase the quota for this account.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.StartImportFileTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/StartImportFileTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartImportFileTaskResponse> startImportFileTask(
            StartImportFileTaskRequest startImportFileTaskRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startImportFileTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartImportFileTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartImportFileTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartImportFileTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartImportFileTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartImportFileTaskRequest, StartImportFileTaskResponse>()
                            .withOperationName("StartImportFileTask")
                            .withMarshaller(new StartImportFileTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startImportFileTaskRequest));
            CompletableFuture<StartImportFileTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts generating a recommendation report.
     * </p>
     *
     * @param startRecommendationReportGenerationRequest
     * @return A Java Future containing the result of the StartRecommendationReportGeneration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>ConflictException Exception to indicate that there is an ongoing task when a new task is created.
     *         Return when once the existing tasks are complete.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.StartRecommendationReportGeneration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/StartRecommendationReportGeneration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartRecommendationReportGenerationResponse> startRecommendationReportGeneration(
            StartRecommendationReportGenerationRequest startRecommendationReportGenerationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startRecommendationReportGenerationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartRecommendationReportGeneration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartRecommendationReportGenerationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, StartRecommendationReportGenerationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartRecommendationReportGenerationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartRecommendationReportGenerationRequest, StartRecommendationReportGenerationResponse>()
                            .withOperationName("StartRecommendationReportGeneration")
                            .withMarshaller(new StartRecommendationReportGenerationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startRecommendationReportGenerationRequest));
            CompletableFuture<StartRecommendationReportGenerationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops the assessment of an on-premises environment.
     * </p>
     *
     * @param stopAssessmentRequest
     * @return A Java Future containing the result of the StopAssessment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The user does not have permission to perform the action. Check the AWS Identity
     *         and Access Management (IAM) policy associated with this user.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.StopAssessment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/StopAssessment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopAssessmentResponse> stopAssessment(StopAssessmentRequest stopAssessmentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopAssessmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopAssessment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopAssessmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopAssessmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopAssessmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopAssessmentRequest, StopAssessmentResponse>()
                            .withOperationName("StopAssessment")
                            .withMarshaller(new StopAssessmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopAssessmentRequest));
            CompletableFuture<StopAssessmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the configuration of an application component.
     * </p>
     *
     * @param updateApplicationComponentConfigRequest
     * @return A Java Future containing the result of the UpdateApplicationComponentConfig operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified ID in the request is not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.UpdateApplicationComponentConfig
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/UpdateApplicationComponentConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateApplicationComponentConfigResponse> updateApplicationComponentConfig(
            UpdateApplicationComponentConfigRequest updateApplicationComponentConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateApplicationComponentConfigRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApplicationComponentConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateApplicationComponentConfigResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateApplicationComponentConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateApplicationComponentConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateApplicationComponentConfigRequest, UpdateApplicationComponentConfigResponse>()
                            .withOperationName("UpdateApplicationComponentConfig")
                            .withMarshaller(new UpdateApplicationComponentConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateApplicationComponentConfigRequest));
            CompletableFuture<UpdateApplicationComponentConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the configuration of the specified server.
     * </p>
     *
     * @param updateServerConfigRequest
     * @return A Java Future containing the result of the UpdateServerConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified ID in the request is not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The request body isn't valid.</li>
     *         <li>InternalServerException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MigrationHubStrategyException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MigrationHubStrategyAsyncClient.UpdateServerConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhubstrategy-2020-02-19/UpdateServerConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateServerConfigResponse> updateServerConfig(UpdateServerConfigRequest updateServerConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServerConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHubStrategy");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServerConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateServerConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateServerConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateServerConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateServerConfigRequest, UpdateServerConfigResponse>()
                            .withOperationName("UpdateServerConfig")
                            .withMarshaller(new UpdateServerConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateServerConfigRequest));
            CompletableFuture<UpdateServerConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final MigrationHubStrategyServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MigrationHubStrategyException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependencyException")
                                .exceptionBuilderSupplier(DependencyException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceLinkedRoleLockClientException")
                                .exceptionBuilderSupplier(ServiceLinkedRoleLockClientException::builder).httpStatusCode(400)
                                .build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends MigrationHubStrategyRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
