/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mobile;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mobile.model.AccountActionRequiredException;
import software.amazon.awssdk.services.mobile.model.BadRequestException;
import software.amazon.awssdk.services.mobile.model.CreateProjectRequest;
import software.amazon.awssdk.services.mobile.model.CreateProjectResponse;
import software.amazon.awssdk.services.mobile.model.DeleteProjectRequest;
import software.amazon.awssdk.services.mobile.model.DeleteProjectResponse;
import software.amazon.awssdk.services.mobile.model.DescribeBundleRequest;
import software.amazon.awssdk.services.mobile.model.DescribeBundleResponse;
import software.amazon.awssdk.services.mobile.model.DescribeProjectRequest;
import software.amazon.awssdk.services.mobile.model.DescribeProjectResponse;
import software.amazon.awssdk.services.mobile.model.ExportBundleRequest;
import software.amazon.awssdk.services.mobile.model.ExportBundleResponse;
import software.amazon.awssdk.services.mobile.model.ExportProjectRequest;
import software.amazon.awssdk.services.mobile.model.ExportProjectResponse;
import software.amazon.awssdk.services.mobile.model.InternalFailureException;
import software.amazon.awssdk.services.mobile.model.LimitExceededException;
import software.amazon.awssdk.services.mobile.model.ListBundlesRequest;
import software.amazon.awssdk.services.mobile.model.ListBundlesResponse;
import software.amazon.awssdk.services.mobile.model.ListProjectsRequest;
import software.amazon.awssdk.services.mobile.model.ListProjectsResponse;
import software.amazon.awssdk.services.mobile.model.MobileException;
import software.amazon.awssdk.services.mobile.model.MobileRequest;
import software.amazon.awssdk.services.mobile.model.NotFoundException;
import software.amazon.awssdk.services.mobile.model.ServiceUnavailableException;
import software.amazon.awssdk.services.mobile.model.TooManyRequestsException;
import software.amazon.awssdk.services.mobile.model.UnauthorizedException;
import software.amazon.awssdk.services.mobile.model.UpdateProjectRequest;
import software.amazon.awssdk.services.mobile.model.UpdateProjectResponse;
import software.amazon.awssdk.services.mobile.paginators.ListBundlesPublisher;
import software.amazon.awssdk.services.mobile.paginators.ListProjectsPublisher;
import software.amazon.awssdk.services.mobile.transform.CreateProjectRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.DeleteProjectRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.DescribeBundleRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.DescribeProjectRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.ExportBundleRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.ExportProjectRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.ListBundlesRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.ListProjectsRequestMarshaller;
import software.amazon.awssdk.services.mobile.transform.UpdateProjectRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MobileAsyncClient}.
 *
 * @see MobileAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMobileAsyncClient implements MobileAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMobileAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMobileAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an AWS Mobile Hub project.
     * </p>
     *
     * @param createProjectRequest
     *        Request structure used to request a project be created.
     * @return A Java Future containing the result of the CreateProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>NotFoundException No entity can be found with the specified identifier.</li>
     *         <li>LimitExceededException There are too many AWS Mobile Hub projects in the account or the account has
     *         exceeded the maximum number of resources in some AWS service. You should create another sub-account using
     *         AWS Organizations or remove some resources and retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.CreateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/CreateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProjectResponse> createProject(CreateProjectRequest createProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProjectRequest, CreateProjectResponse>()
                            .withOperationName("CreateProject")
                            .withMarshaller(new CreateProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createProjectRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createProjectRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delets a project in AWS Mobile Hub.
     * </p>
     *
     * @param deleteProjectRequest
     *        Request structure used to request a project be deleted.
     * @return A Java Future containing the result of the DeleteProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>NotFoundException No entity can be found with the specified identifier.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.DeleteProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/DeleteProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProjectResponse> deleteProject(DeleteProjectRequest deleteProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProjectRequest, DeleteProjectResponse>()
                            .withOperationName("DeleteProject")
                            .withMarshaller(new DeleteProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteProjectRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteProjectRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the bundle details for the requested bundle id.
     * </p>
     *
     * @param describeBundleRequest
     *        Request structure to request the details of a specific bundle.
     * @return A Java Future containing the result of the DescribeBundle operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>NotFoundException No entity can be found with the specified identifier.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.DescribeBundle
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/DescribeBundle" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeBundleResponse> describeBundle(DescribeBundleRequest describeBundleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBundleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBundle");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBundleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeBundleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBundleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBundleRequest, DescribeBundleResponse>()
                            .withOperationName("DescribeBundle")
                            .withMarshaller(new DescribeBundleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeBundleRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeBundleRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeBundleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details about a project in AWS Mobile Hub.
     * </p>
     *
     * @param describeProjectRequest
     *        Request structure used to request details about a project.
     * @return A Java Future containing the result of the DescribeProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>NotFoundException No entity can be found with the specified identifier.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.DescribeProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/DescribeProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeProjectResponse> describeProject(DescribeProjectRequest describeProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeProjectResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeProjectRequest, DescribeProjectResponse>()
                            .withOperationName("DescribeProject")
                            .withMarshaller(new DescribeProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeProjectRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeProjectRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates customized software development kit (SDK) and or tool packages used to integrate mobile web or mobile
     * app clients with backend AWS resources.
     * </p>
     *
     * @param exportBundleRequest
     *        Request structure used to request generation of custom SDK and tool packages required to integrate mobile
     *        web or app clients with backed AWS resources.
     * @return A Java Future containing the result of the ExportBundle operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>NotFoundException No entity can be found with the specified identifier.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.ExportBundle
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/ExportBundle" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ExportBundleResponse> exportBundle(ExportBundleRequest exportBundleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportBundleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportBundle");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportBundleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ExportBundleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportBundleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportBundleRequest, ExportBundleResponse>()
                            .withOperationName("ExportBundle").withMarshaller(new ExportBundleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(exportBundleRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = exportBundleRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ExportBundleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports project configuration to a snapshot which can be downloaded and shared. Note that mobile app push
     * credentials are encrypted in exported projects, so they can only be shared successfully within the same AWS
     * account.
     * </p>
     *
     * @param exportProjectRequest
     *        Request structure used in requests to export project configuration details.
     * @return A Java Future containing the result of the ExportProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>NotFoundException No entity can be found with the specified identifier.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.ExportProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/ExportProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ExportProjectResponse> exportProject(ExportProjectRequest exportProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ExportProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportProjectRequest, ExportProjectResponse>()
                            .withOperationName("ExportProject")
                            .withMarshaller(new ExportProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(exportProjectRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = exportProjectRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ExportProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all available bundles.
     * </p>
     *
     * @param listBundlesRequest
     *        Request structure to request all available bundles.
     * @return A Java Future containing the result of the ListBundles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.ListBundles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/ListBundles" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListBundlesResponse> listBundles(ListBundlesRequest listBundlesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBundlesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBundles");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListBundlesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListBundlesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListBundlesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListBundlesRequest, ListBundlesResponse>()
                            .withOperationName("ListBundles").withMarshaller(new ListBundlesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listBundlesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listBundlesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListBundlesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all available bundles.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listBundles(software.amazon.awssdk.services.mobile.model.ListBundlesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mobile.paginators.ListBundlesPublisher publisher = client.listBundlesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mobile.paginators.ListBundlesPublisher publisher = client.listBundlesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mobile.model.ListBundlesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mobile.model.ListBundlesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listBundles(software.amazon.awssdk.services.mobile.model.ListBundlesRequest)} operation.</b>
     * </p>
     *
     * @param listBundlesRequest
     *        Request structure to request all available bundles.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.ListBundles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/ListBundles" target="_top">AWS API
     *      Documentation</a>
     */
    public ListBundlesPublisher listBundlesPaginator(ListBundlesRequest listBundlesRequest) {
        return new ListBundlesPublisher(this, applyPaginatorUserAgent(listBundlesRequest));
    }

    /**
     * <p>
     * Lists projects in AWS Mobile Hub.
     * </p>
     *
     * @param listProjectsRequest
     *        Request structure used to request projects list in AWS Mobile Hub.
     * @return A Java Future containing the result of the ListProjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.ListProjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/ListProjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListProjectsResponse> listProjects(ListProjectsRequest listProjectsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProjects");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListProjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProjectsRequest, ListProjectsResponse>()
                            .withOperationName("ListProjects").withMarshaller(new ListProjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listProjectsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listProjectsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListProjectsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists projects in AWS Mobile Hub.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listProjects(software.amazon.awssdk.services.mobile.model.ListProjectsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mobile.paginators.ListProjectsPublisher publisher = client.listProjectsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mobile.paginators.ListProjectsPublisher publisher = client.listProjectsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mobile.model.ListProjectsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mobile.model.ListProjectsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProjects(software.amazon.awssdk.services.mobile.model.ListProjectsRequest)} operation.</b>
     * </p>
     *
     * @param listProjectsRequest
     *        Request structure used to request projects list in AWS Mobile Hub.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.ListProjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/ListProjects" target="_top">AWS API
     *      Documentation</a>
     */
    public ListProjectsPublisher listProjectsPaginator(ListProjectsRequest listProjectsRequest) {
        return new ListProjectsPublisher(this, applyPaginatorUserAgent(listProjectsRequest));
    }

    /**
     * <p>
     * Update an existing project.
     * </p>
     *
     * @param updateProjectRequest
     *        Request structure used for requests to update project configuration.
     * @return A Java Future containing the result of the UpdateProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalFailureException The service has encountered an unexpected error condition which prevents it
     *         from servicing the request.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable. The request should be retried
     *         after some time delay.</li>
     *         <li>UnauthorizedException Credentials of the caller are insufficient to authorize the request.</li>
     *         <li>TooManyRequestsException Too many requests have been received for this AWS account in too short a
     *         time. The request should be retried after some time delay.</li>
     *         <li>BadRequestException The request cannot be processed because some parameter is not valid or the
     *         project state prevents the operation from being performed.</li>
     *         <li>NotFoundException No entity can be found with the specified identifier.</li>
     *         <li>AccountActionRequiredException Account Action is required in order to continue the request.</li>
     *         <li>LimitExceededException There are too many AWS Mobile Hub projects in the account or the account has
     *         exceeded the maximum number of resources in some AWS service. You should create another sub-account using
     *         AWS Organizations or remove some resources and retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MobileException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MobileAsyncClient.UpdateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mobile-2017-07-01/UpdateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProjectResponse> updateProject(UpdateProjectRequest updateProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Mobile");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProjectRequest, UpdateProjectResponse>()
                            .withOperationName("UpdateProject")
                            .withMarshaller(new UpdateProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateProjectRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateProjectRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MobileException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccountActionRequiredException")
                                .exceptionBuilderSupplier(AccountActionRequiredException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends MobileRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
