/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mturk.model;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.mturk.transform.AssignmentMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The Assignment data structure represents a single assignment of a HIT to a Worker. The assignment tracks the Worker's
 * efforts to complete the HIT, and contains the results for later retrieval.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Assignment implements StructuredPojo, ToCopyableBuilder<Assignment.Builder, Assignment> {
    private final String assignmentId;

    private final String workerId;

    private final String hitId;

    private final String assignmentStatus;

    private final Instant autoApprovalTime;

    private final Instant acceptTime;

    private final Instant submitTime;

    private final Instant approvalTime;

    private final Instant rejectionTime;

    private final Instant deadline;

    private final String answer;

    private final String requesterFeedback;

    private Assignment(BuilderImpl builder) {
        this.assignmentId = builder.assignmentId;
        this.workerId = builder.workerId;
        this.hitId = builder.hitId;
        this.assignmentStatus = builder.assignmentStatus;
        this.autoApprovalTime = builder.autoApprovalTime;
        this.acceptTime = builder.acceptTime;
        this.submitTime = builder.submitTime;
        this.approvalTime = builder.approvalTime;
        this.rejectionTime = builder.rejectionTime;
        this.deadline = builder.deadline;
        this.answer = builder.answer;
        this.requesterFeedback = builder.requesterFeedback;
    }

    /**
     * <p>
     * A unique identifier for the assignment.
     * </p>
     * 
     * @return A unique identifier for the assignment.
     */
    public String assignmentId() {
        return assignmentId;
    }

    /**
     * <p>
     * The ID of the Worker who accepted the HIT.
     * </p>
     * 
     * @return The ID of the Worker who accepted the HIT.
     */
    public String workerId() {
        return workerId;
    }

    /**
     * <p>
     * The ID of the HIT.
     * </p>
     * 
     * @return The ID of the HIT.
     */
    public String hitId() {
        return hitId;
    }

    /**
     * <p>
     * The status of the assignment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #assignmentStatus}
     * will return {@link AssignmentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #assignmentStatusAsString}.
     * </p>
     * 
     * @return The status of the assignment.
     * @see AssignmentStatus
     */
    public AssignmentStatus assignmentStatus() {
        return AssignmentStatus.fromValue(assignmentStatus);
    }

    /**
     * <p>
     * The status of the assignment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #assignmentStatus}
     * will return {@link AssignmentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #assignmentStatusAsString}.
     * </p>
     * 
     * @return The status of the assignment.
     * @see AssignmentStatus
     */
    public String assignmentStatusAsString() {
        return assignmentStatus;
    }

    /**
     * <p>
     * If results have been submitted, AutoApprovalTime is the date and time the results of the assignment results are
     * considered Approved automatically if they have not already been explicitly approved or rejected by the Requester.
     * This value is derived from the auto-approval delay specified by the Requester in the HIT. This value is omitted
     * from the assignment if the Worker has not yet submitted results.
     * </p>
     * 
     * @return If results have been submitted, AutoApprovalTime is the date and time the results of the assignment
     *         results are considered Approved automatically if they have not already been explicitly approved or
     *         rejected by the Requester. This value is derived from the auto-approval delay specified by the Requester
     *         in the HIT. This value is omitted from the assignment if the Worker has not yet submitted results.
     */
    public Instant autoApprovalTime() {
        return autoApprovalTime;
    }

    /**
     * <p>
     * The date and time the Worker accepted the assignment.
     * </p>
     * 
     * @return The date and time the Worker accepted the assignment.
     */
    public Instant acceptTime() {
        return acceptTime;
    }

    /**
     * <p>
     * If the Worker has submitted results, SubmitTime is the date and time the assignment was submitted. This value is
     * omitted from the assignment if the Worker has not yet submitted results.
     * </p>
     * 
     * @return If the Worker has submitted results, SubmitTime is the date and time the assignment was submitted. This
     *         value is omitted from the assignment if the Worker has not yet submitted results.
     */
    public Instant submitTime() {
        return submitTime;
    }

    /**
     * <p>
     * If the Worker has submitted results and the Requester has approved the results, ApprovalTime is the date and time
     * the Requester approved the results. This value is omitted from the assignment if the Requester has not yet
     * approved the results.
     * </p>
     * 
     * @return If the Worker has submitted results and the Requester has approved the results, ApprovalTime is the date
     *         and time the Requester approved the results. This value is omitted from the assignment if the Requester
     *         has not yet approved the results.
     */
    public Instant approvalTime() {
        return approvalTime;
    }

    /**
     * <p>
     * If the Worker has submitted results and the Requester has rejected the results, RejectionTime is the date and
     * time the Requester rejected the results.
     * </p>
     * 
     * @return If the Worker has submitted results and the Requester has rejected the results, RejectionTime is the date
     *         and time the Requester rejected the results.
     */
    public Instant rejectionTime() {
        return rejectionTime;
    }

    /**
     * <p>
     * The date and time of the deadline for the assignment. This value is derived from the deadline specification for
     * the HIT and the date and time the Worker accepted the HIT.
     * </p>
     * 
     * @return The date and time of the deadline for the assignment. This value is derived from the deadline
     *         specification for the HIT and the date and time the Worker accepted the HIT.
     */
    public Instant deadline() {
        return deadline;
    }

    /**
     * <p>
     * The Worker's answers submitted for the HIT contained in a QuestionFormAnswers document, if the Worker provides an
     * answer. If the Worker does not provide any answers, Answer may contain a QuestionFormAnswers document, or Answer
     * may be empty.
     * </p>
     * 
     * @return The Worker's answers submitted for the HIT contained in a QuestionFormAnswers document, if the Worker
     *         provides an answer. If the Worker does not provide any answers, Answer may contain a QuestionFormAnswers
     *         document, or Answer may be empty.
     */
    public String answer() {
        return answer;
    }

    /**
     * <p>
     * The feedback string included with the call to the ApproveAssignment operation or the RejectAssignment operation,
     * if the Requester approved or rejected the assignment and specified feedback.
     * </p>
     * 
     * @return The feedback string included with the call to the ApproveAssignment operation or the RejectAssignment
     *         operation, if the Requester approved or rejected the assignment and specified feedback.
     */
    public String requesterFeedback() {
        return requesterFeedback;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(assignmentId());
        hashCode = 31 * hashCode + Objects.hashCode(workerId());
        hashCode = 31 * hashCode + Objects.hashCode(hitId());
        hashCode = 31 * hashCode + Objects.hashCode(assignmentStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(autoApprovalTime());
        hashCode = 31 * hashCode + Objects.hashCode(acceptTime());
        hashCode = 31 * hashCode + Objects.hashCode(submitTime());
        hashCode = 31 * hashCode + Objects.hashCode(approvalTime());
        hashCode = 31 * hashCode + Objects.hashCode(rejectionTime());
        hashCode = 31 * hashCode + Objects.hashCode(deadline());
        hashCode = 31 * hashCode + Objects.hashCode(answer());
        hashCode = 31 * hashCode + Objects.hashCode(requesterFeedback());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Assignment)) {
            return false;
        }
        Assignment other = (Assignment) obj;
        return Objects.equals(assignmentId(), other.assignmentId()) && Objects.equals(workerId(), other.workerId())
                && Objects.equals(hitId(), other.hitId())
                && Objects.equals(assignmentStatusAsString(), other.assignmentStatusAsString())
                && Objects.equals(autoApprovalTime(), other.autoApprovalTime())
                && Objects.equals(acceptTime(), other.acceptTime()) && Objects.equals(submitTime(), other.submitTime())
                && Objects.equals(approvalTime(), other.approvalTime()) && Objects.equals(rejectionTime(), other.rejectionTime())
                && Objects.equals(deadline(), other.deadline()) && Objects.equals(answer(), other.answer())
                && Objects.equals(requesterFeedback(), other.requesterFeedback());
    }

    @Override
    public String toString() {
        return ToString.builder("Assignment").add("AssignmentId", assignmentId()).add("WorkerId", workerId())
                .add("HITId", hitId()).add("AssignmentStatus", assignmentStatusAsString())
                .add("AutoApprovalTime", autoApprovalTime()).add("AcceptTime", acceptTime()).add("SubmitTime", submitTime())
                .add("ApprovalTime", approvalTime()).add("RejectionTime", rejectionTime()).add("Deadline", deadline())
                .add("Answer", answer()).add("RequesterFeedback", requesterFeedback()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AssignmentId":
            return Optional.ofNullable(clazz.cast(assignmentId()));
        case "WorkerId":
            return Optional.ofNullable(clazz.cast(workerId()));
        case "HITId":
            return Optional.ofNullable(clazz.cast(hitId()));
        case "AssignmentStatus":
            return Optional.ofNullable(clazz.cast(assignmentStatusAsString()));
        case "AutoApprovalTime":
            return Optional.ofNullable(clazz.cast(autoApprovalTime()));
        case "AcceptTime":
            return Optional.ofNullable(clazz.cast(acceptTime()));
        case "SubmitTime":
            return Optional.ofNullable(clazz.cast(submitTime()));
        case "ApprovalTime":
            return Optional.ofNullable(clazz.cast(approvalTime()));
        case "RejectionTime":
            return Optional.ofNullable(clazz.cast(rejectionTime()));
        case "Deadline":
            return Optional.ofNullable(clazz.cast(deadline()));
        case "Answer":
            return Optional.ofNullable(clazz.cast(answer()));
        case "RequesterFeedback":
            return Optional.ofNullable(clazz.cast(requesterFeedback()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        AssignmentMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Assignment> {
        /**
         * <p>
         * A unique identifier for the assignment.
         * </p>
         * 
         * @param assignmentId
         *        A unique identifier for the assignment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assignmentId(String assignmentId);

        /**
         * <p>
         * The ID of the Worker who accepted the HIT.
         * </p>
         * 
         * @param workerId
         *        The ID of the Worker who accepted the HIT.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workerId(String workerId);

        /**
         * <p>
         * The ID of the HIT.
         * </p>
         * 
         * @param hitId
         *        The ID of the HIT.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hitId(String hitId);

        /**
         * <p>
         * The status of the assignment.
         * </p>
         * 
         * @param assignmentStatus
         *        The status of the assignment.
         * @see AssignmentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssignmentStatus
         */
        Builder assignmentStatus(String assignmentStatus);

        /**
         * <p>
         * The status of the assignment.
         * </p>
         * 
         * @param assignmentStatus
         *        The status of the assignment.
         * @see AssignmentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssignmentStatus
         */
        Builder assignmentStatus(AssignmentStatus assignmentStatus);

        /**
         * <p>
         * If results have been submitted, AutoApprovalTime is the date and time the results of the assignment results
         * are considered Approved automatically if they have not already been explicitly approved or rejected by the
         * Requester. This value is derived from the auto-approval delay specified by the Requester in the HIT. This
         * value is omitted from the assignment if the Worker has not yet submitted results.
         * </p>
         * 
         * @param autoApprovalTime
         *        If results have been submitted, AutoApprovalTime is the date and time the results of the assignment
         *        results are considered Approved automatically if they have not already been explicitly approved or
         *        rejected by the Requester. This value is derived from the auto-approval delay specified by the
         *        Requester in the HIT. This value is omitted from the assignment if the Worker has not yet submitted
         *        results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoApprovalTime(Instant autoApprovalTime);

        /**
         * <p>
         * The date and time the Worker accepted the assignment.
         * </p>
         * 
         * @param acceptTime
         *        The date and time the Worker accepted the assignment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceptTime(Instant acceptTime);

        /**
         * <p>
         * If the Worker has submitted results, SubmitTime is the date and time the assignment was submitted. This value
         * is omitted from the assignment if the Worker has not yet submitted results.
         * </p>
         * 
         * @param submitTime
         *        If the Worker has submitted results, SubmitTime is the date and time the assignment was submitted.
         *        This value is omitted from the assignment if the Worker has not yet submitted results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder submitTime(Instant submitTime);

        /**
         * <p>
         * If the Worker has submitted results and the Requester has approved the results, ApprovalTime is the date and
         * time the Requester approved the results. This value is omitted from the assignment if the Requester has not
         * yet approved the results.
         * </p>
         * 
         * @param approvalTime
         *        If the Worker has submitted results and the Requester has approved the results, ApprovalTime is the
         *        date and time the Requester approved the results. This value is omitted from the assignment if the
         *        Requester has not yet approved the results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder approvalTime(Instant approvalTime);

        /**
         * <p>
         * If the Worker has submitted results and the Requester has rejected the results, RejectionTime is the date and
         * time the Requester rejected the results.
         * </p>
         * 
         * @param rejectionTime
         *        If the Worker has submitted results and the Requester has rejected the results, RejectionTime is the
         *        date and time the Requester rejected the results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rejectionTime(Instant rejectionTime);

        /**
         * <p>
         * The date and time of the deadline for the assignment. This value is derived from the deadline specification
         * for the HIT and the date and time the Worker accepted the HIT.
         * </p>
         * 
         * @param deadline
         *        The date and time of the deadline for the assignment. This value is derived from the deadline
         *        specification for the HIT and the date and time the Worker accepted the HIT.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deadline(Instant deadline);

        /**
         * <p>
         * The Worker's answers submitted for the HIT contained in a QuestionFormAnswers document, if the Worker
         * provides an answer. If the Worker does not provide any answers, Answer may contain a QuestionFormAnswers
         * document, or Answer may be empty.
         * </p>
         * 
         * @param answer
         *        The Worker's answers submitted for the HIT contained in a QuestionFormAnswers document, if the Worker
         *        provides an answer. If the Worker does not provide any answers, Answer may contain a
         *        QuestionFormAnswers document, or Answer may be empty.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder answer(String answer);

        /**
         * <p>
         * The feedback string included with the call to the ApproveAssignment operation or the RejectAssignment
         * operation, if the Requester approved or rejected the assignment and specified feedback.
         * </p>
         * 
         * @param requesterFeedback
         *        The feedback string included with the call to the ApproveAssignment operation or the RejectAssignment
         *        operation, if the Requester approved or rejected the assignment and specified feedback.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requesterFeedback(String requesterFeedback);
    }

    static final class BuilderImpl implements Builder {
        private String assignmentId;

        private String workerId;

        private String hitId;

        private String assignmentStatus;

        private Instant autoApprovalTime;

        private Instant acceptTime;

        private Instant submitTime;

        private Instant approvalTime;

        private Instant rejectionTime;

        private Instant deadline;

        private String answer;

        private String requesterFeedback;

        private BuilderImpl() {
        }

        private BuilderImpl(Assignment model) {
            assignmentId(model.assignmentId);
            workerId(model.workerId);
            hitId(model.hitId);
            assignmentStatus(model.assignmentStatus);
            autoApprovalTime(model.autoApprovalTime);
            acceptTime(model.acceptTime);
            submitTime(model.submitTime);
            approvalTime(model.approvalTime);
            rejectionTime(model.rejectionTime);
            deadline(model.deadline);
            answer(model.answer);
            requesterFeedback(model.requesterFeedback);
        }

        public final String getAssignmentId() {
            return assignmentId;
        }

        @Override
        public final Builder assignmentId(String assignmentId) {
            this.assignmentId = assignmentId;
            return this;
        }

        public final void setAssignmentId(String assignmentId) {
            this.assignmentId = assignmentId;
        }

        public final String getWorkerId() {
            return workerId;
        }

        @Override
        public final Builder workerId(String workerId) {
            this.workerId = workerId;
            return this;
        }

        public final void setWorkerId(String workerId) {
            this.workerId = workerId;
        }

        public final String getHITId() {
            return hitId;
        }

        @Override
        public final Builder hitId(String hitId) {
            this.hitId = hitId;
            return this;
        }

        public final void setHITId(String hitId) {
            this.hitId = hitId;
        }

        public final String getAssignmentStatus() {
            return assignmentStatus;
        }

        @Override
        public final Builder assignmentStatus(String assignmentStatus) {
            this.assignmentStatus = assignmentStatus;
            return this;
        }

        @Override
        public final Builder assignmentStatus(AssignmentStatus assignmentStatus) {
            this.assignmentStatus(assignmentStatus.toString());
            return this;
        }

        public final void setAssignmentStatus(String assignmentStatus) {
            this.assignmentStatus = assignmentStatus;
        }

        public final Instant getAutoApprovalTime() {
            return autoApprovalTime;
        }

        @Override
        public final Builder autoApprovalTime(Instant autoApprovalTime) {
            this.autoApprovalTime = autoApprovalTime;
            return this;
        }

        public final void setAutoApprovalTime(Instant autoApprovalTime) {
            this.autoApprovalTime = autoApprovalTime;
        }

        public final Instant getAcceptTime() {
            return acceptTime;
        }

        @Override
        public final Builder acceptTime(Instant acceptTime) {
            this.acceptTime = acceptTime;
            return this;
        }

        public final void setAcceptTime(Instant acceptTime) {
            this.acceptTime = acceptTime;
        }

        public final Instant getSubmitTime() {
            return submitTime;
        }

        @Override
        public final Builder submitTime(Instant submitTime) {
            this.submitTime = submitTime;
            return this;
        }

        public final void setSubmitTime(Instant submitTime) {
            this.submitTime = submitTime;
        }

        public final Instant getApprovalTime() {
            return approvalTime;
        }

        @Override
        public final Builder approvalTime(Instant approvalTime) {
            this.approvalTime = approvalTime;
            return this;
        }

        public final void setApprovalTime(Instant approvalTime) {
            this.approvalTime = approvalTime;
        }

        public final Instant getRejectionTime() {
            return rejectionTime;
        }

        @Override
        public final Builder rejectionTime(Instant rejectionTime) {
            this.rejectionTime = rejectionTime;
            return this;
        }

        public final void setRejectionTime(Instant rejectionTime) {
            this.rejectionTime = rejectionTime;
        }

        public final Instant getDeadline() {
            return deadline;
        }

        @Override
        public final Builder deadline(Instant deadline) {
            this.deadline = deadline;
            return this;
        }

        public final void setDeadline(Instant deadline) {
            this.deadline = deadline;
        }

        public final String getAnswer() {
            return answer;
        }

        @Override
        public final Builder answer(String answer) {
            this.answer = answer;
            return this;
        }

        public final void setAnswer(String answer) {
            this.answer = answer;
        }

        public final String getRequesterFeedback() {
            return requesterFeedback;
        }

        @Override
        public final Builder requesterFeedback(String requesterFeedback) {
            this.requesterFeedback = requesterFeedback;
            return this;
        }

        public final void setRequesterFeedback(String requesterFeedback) {
            this.requesterFeedback = requesterFeedback;
        }

        @Override
        public Assignment build() {
            return new Assignment(this);
        }
    }
}
