/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mturk.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateHitWithHitTypeRequest extends MTurkRequest implements
        ToCopyableBuilder<CreateHitWithHitTypeRequest.Builder, CreateHitWithHitTypeRequest> {
    private final String hitTypeId;

    private final Integer maxAssignments;

    private final Long lifetimeInSeconds;

    private final String question;

    private final String requesterAnnotation;

    private final String uniqueRequestToken;

    private final ReviewPolicy assignmentReviewPolicy;

    private final ReviewPolicy hitReviewPolicy;

    private final String hitLayoutId;

    private final List<HITLayoutParameter> hitLayoutParameters;

    private CreateHitWithHitTypeRequest(BuilderImpl builder) {
        super(builder);
        this.hitTypeId = builder.hitTypeId;
        this.maxAssignments = builder.maxAssignments;
        this.lifetimeInSeconds = builder.lifetimeInSeconds;
        this.question = builder.question;
        this.requesterAnnotation = builder.requesterAnnotation;
        this.uniqueRequestToken = builder.uniqueRequestToken;
        this.assignmentReviewPolicy = builder.assignmentReviewPolicy;
        this.hitReviewPolicy = builder.hitReviewPolicy;
        this.hitLayoutId = builder.hitLayoutId;
        this.hitLayoutParameters = builder.hitLayoutParameters;
    }

    /**
     * <p>
     * The HIT type ID you want to create this HIT with.
     * </p>
     * 
     * @return The HIT type ID you want to create this HIT with.
     */
    public String hitTypeId() {
        return hitTypeId;
    }

    /**
     * <p>
     * The number of times the HIT can be accepted and completed before the HIT becomes unavailable.
     * </p>
     * 
     * @return The number of times the HIT can be accepted and completed before the HIT becomes unavailable.
     */
    public Integer maxAssignments() {
        return maxAssignments;
    }

    /**
     * <p>
     * An amount of time, in seconds, after which the HIT is no longer available for users to accept. After the lifetime
     * of the HIT elapses, the HIT no longer appears in HIT searches, even if not all of the assignments for the HIT
     * have been accepted.
     * </p>
     * 
     * @return An amount of time, in seconds, after which the HIT is no longer available for users to accept. After the
     *         lifetime of the HIT elapses, the HIT no longer appears in HIT searches, even if not all of the
     *         assignments for the HIT have been accepted.
     */
    public Long lifetimeInSeconds() {
        return lifetimeInSeconds;
    }

    /**
     * <p>
     * The data the person completing the HIT uses to produce the results.
     * </p>
     * <p>
     * Constraints: Must be a QuestionForm data structure, an ExternalQuestion data structure, or an HTMLQuestion data
     * structure. The XML question data must not be larger than 64 kilobytes (65,535 bytes) in size, including
     * whitespace.
     * </p>
     * <p>
     * Either a Question parameter or a HITLayoutId parameter must be provided.
     * </p>
     * 
     * @return The data the person completing the HIT uses to produce the results. </p>
     *         <p>
     *         Constraints: Must be a QuestionForm data structure, an ExternalQuestion data structure, or an
     *         HTMLQuestion data structure. The XML question data must not be larger than 64 kilobytes (65,535 bytes) in
     *         size, including whitespace.
     *         </p>
     *         <p>
     *         Either a Question parameter or a HITLayoutId parameter must be provided.
     */
    public String question() {
        return question;
    }

    /**
     * <p>
     * An arbitrary data field. The RequesterAnnotation parameter lets your application attach arbitrary data to the HIT
     * for tracking purposes. For example, this parameter could be an identifier internal to the Requester's application
     * that corresponds with the HIT.
     * </p>
     * <p>
     * The RequesterAnnotation parameter for a HIT is only visible to the Requester who created the HIT. It is not shown
     * to the Worker, or any other Requester.
     * </p>
     * <p>
     * The RequesterAnnotation parameter may be different for each HIT you submit. It does not affect how your HITs are
     * grouped.
     * </p>
     * 
     * @return An arbitrary data field. The RequesterAnnotation parameter lets your application attach arbitrary data to
     *         the HIT for tracking purposes. For example, this parameter could be an identifier internal to the
     *         Requester's application that corresponds with the HIT. </p>
     *         <p>
     *         The RequesterAnnotation parameter for a HIT is only visible to the Requester who created the HIT. It is
     *         not shown to the Worker, or any other Requester.
     *         </p>
     *         <p>
     *         The RequesterAnnotation parameter may be different for each HIT you submit. It does not affect how your
     *         HITs are grouped.
     */
    public String requesterAnnotation() {
        return requesterAnnotation;
    }

    /**
     * <p>
     * A unique identifier for this request which allows you to retry the call on error without creating duplicate HITs.
     * This is useful in cases such as network timeouts where it is unclear whether or not the call succeeded on the
     * server. If the HIT already exists in the system from a previous call using the same UniqueRequestToken,
     * subsequent calls will return a AWS.MechanicalTurk.HitAlreadyExists error with a message containing the HITId.
     * </p>
     * <note>
     * <p>
     * Note: It is your responsibility to ensure uniqueness of the token. The unique token expires after 24 hours.
     * Subsequent calls using the same UniqueRequestToken made after the 24 hour limit could create duplicate HITs.
     * </p>
     * </note>
     * 
     * @return A unique identifier for this request which allows you to retry the call on error without creating
     *         duplicate HITs. This is useful in cases such as network timeouts where it is unclear whether or not the
     *         call succeeded on the server. If the HIT already exists in the system from a previous call using the same
     *         UniqueRequestToken, subsequent calls will return a AWS.MechanicalTurk.HitAlreadyExists error with a
     *         message containing the HITId. </p> <note>
     *         <p>
     *         Note: It is your responsibility to ensure uniqueness of the token. The unique token expires after 24
     *         hours. Subsequent calls using the same UniqueRequestToken made after the 24 hour limit could create
     *         duplicate HITs.
     *         </p>
     */
    public String uniqueRequestToken() {
        return uniqueRequestToken;
    }

    /**
     * <p>
     * The Assignment-level Review Policy applies to the assignments under the HIT. You can specify for Mechanical Turk
     * to take various actions based on the policy.
     * </p>
     * 
     * @return The Assignment-level Review Policy applies to the assignments under the HIT. You can specify for
     *         Mechanical Turk to take various actions based on the policy.
     */
    public ReviewPolicy assignmentReviewPolicy() {
        return assignmentReviewPolicy;
    }

    /**
     * <p>
     * The HIT-level Review Policy applies to the HIT. You can specify for Mechanical Turk to take various actions based
     * on the policy.
     * </p>
     * 
     * @return The HIT-level Review Policy applies to the HIT. You can specify for Mechanical Turk to take various
     *         actions based on the policy.
     */
    public ReviewPolicy hitReviewPolicy() {
        return hitReviewPolicy;
    }

    /**
     * <p>
     * The HITLayoutId allows you to use a pre-existing HIT design with placeholder values and create an additional HIT
     * by providing those values as HITLayoutParameters.
     * </p>
     * <p>
     * Constraints: Either a Question parameter or a HITLayoutId parameter must be provided.
     * </p>
     * 
     * @return The HITLayoutId allows you to use a pre-existing HIT design with placeholder values and create an
     *         additional HIT by providing those values as HITLayoutParameters. </p>
     *         <p>
     *         Constraints: Either a Question parameter or a HITLayoutId parameter must be provided.
     */
    public String hitLayoutId() {
        return hitLayoutId;
    }

    /**
     * <p>
     * If the HITLayoutId is provided, any placeholder values must be filled in with values using the HITLayoutParameter
     * structure. For more information, see HITLayout.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return If the HITLayoutId is provided, any placeholder values must be filled in with values using the
     *         HITLayoutParameter structure. For more information, see HITLayout.
     */
    public List<HITLayoutParameter> hitLayoutParameters() {
        return hitLayoutParameters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hitTypeId());
        hashCode = 31 * hashCode + Objects.hashCode(maxAssignments());
        hashCode = 31 * hashCode + Objects.hashCode(lifetimeInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(question());
        hashCode = 31 * hashCode + Objects.hashCode(requesterAnnotation());
        hashCode = 31 * hashCode + Objects.hashCode(uniqueRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(assignmentReviewPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(hitReviewPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(hitLayoutId());
        hashCode = 31 * hashCode + Objects.hashCode(hitLayoutParameters());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateHitWithHitTypeRequest)) {
            return false;
        }
        CreateHitWithHitTypeRequest other = (CreateHitWithHitTypeRequest) obj;
        return Objects.equals(hitTypeId(), other.hitTypeId()) && Objects.equals(maxAssignments(), other.maxAssignments())
                && Objects.equals(lifetimeInSeconds(), other.lifetimeInSeconds()) && Objects.equals(question(), other.question())
                && Objects.equals(requesterAnnotation(), other.requesterAnnotation())
                && Objects.equals(uniqueRequestToken(), other.uniqueRequestToken())
                && Objects.equals(assignmentReviewPolicy(), other.assignmentReviewPolicy())
                && Objects.equals(hitReviewPolicy(), other.hitReviewPolicy())
                && Objects.equals(hitLayoutId(), other.hitLayoutId())
                && Objects.equals(hitLayoutParameters(), other.hitLayoutParameters());
    }

    @Override
    public String toString() {
        return ToString.builder("CreateHitWithHitTypeRequest").add("HITTypeId", hitTypeId())
                .add("MaxAssignments", maxAssignments()).add("LifetimeInSeconds", lifetimeInSeconds())
                .add("Question", question()).add("RequesterAnnotation", requesterAnnotation())
                .add("UniqueRequestToken", uniqueRequestToken()).add("AssignmentReviewPolicy", assignmentReviewPolicy())
                .add("HITReviewPolicy", hitReviewPolicy()).add("HITLayoutId", hitLayoutId())
                .add("HITLayoutParameters", hitLayoutParameters()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "HITTypeId":
            return Optional.ofNullable(clazz.cast(hitTypeId()));
        case "MaxAssignments":
            return Optional.ofNullable(clazz.cast(maxAssignments()));
        case "LifetimeInSeconds":
            return Optional.ofNullable(clazz.cast(lifetimeInSeconds()));
        case "Question":
            return Optional.ofNullable(clazz.cast(question()));
        case "RequesterAnnotation":
            return Optional.ofNullable(clazz.cast(requesterAnnotation()));
        case "UniqueRequestToken":
            return Optional.ofNullable(clazz.cast(uniqueRequestToken()));
        case "AssignmentReviewPolicy":
            return Optional.ofNullable(clazz.cast(assignmentReviewPolicy()));
        case "HITReviewPolicy":
            return Optional.ofNullable(clazz.cast(hitReviewPolicy()));
        case "HITLayoutId":
            return Optional.ofNullable(clazz.cast(hitLayoutId()));
        case "HITLayoutParameters":
            return Optional.ofNullable(clazz.cast(hitLayoutParameters()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends MTurkRequest.Builder, CopyableBuilder<Builder, CreateHitWithHitTypeRequest> {
        /**
         * <p>
         * The HIT type ID you want to create this HIT with.
         * </p>
         * 
         * @param hitTypeId
         *        The HIT type ID you want to create this HIT with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hitTypeId(String hitTypeId);

        /**
         * <p>
         * The number of times the HIT can be accepted and completed before the HIT becomes unavailable.
         * </p>
         * 
         * @param maxAssignments
         *        The number of times the HIT can be accepted and completed before the HIT becomes unavailable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxAssignments(Integer maxAssignments);

        /**
         * <p>
         * An amount of time, in seconds, after which the HIT is no longer available for users to accept. After the
         * lifetime of the HIT elapses, the HIT no longer appears in HIT searches, even if not all of the assignments
         * for the HIT have been accepted.
         * </p>
         * 
         * @param lifetimeInSeconds
         *        An amount of time, in seconds, after which the HIT is no longer available for users to accept. After
         *        the lifetime of the HIT elapses, the HIT no longer appears in HIT searches, even if not all of the
         *        assignments for the HIT have been accepted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifetimeInSeconds(Long lifetimeInSeconds);

        /**
         * <p>
         * The data the person completing the HIT uses to produce the results.
         * </p>
         * <p>
         * Constraints: Must be a QuestionForm data structure, an ExternalQuestion data structure, or an HTMLQuestion
         * data structure. The XML question data must not be larger than 64 kilobytes (65,535 bytes) in size, including
         * whitespace.
         * </p>
         * <p>
         * Either a Question parameter or a HITLayoutId parameter must be provided.
         * </p>
         * 
         * @param question
         *        The data the person completing the HIT uses to produce the results. </p>
         *        <p>
         *        Constraints: Must be a QuestionForm data structure, an ExternalQuestion data structure, or an
         *        HTMLQuestion data structure. The XML question data must not be larger than 64 kilobytes (65,535 bytes)
         *        in size, including whitespace.
         *        </p>
         *        <p>
         *        Either a Question parameter or a HITLayoutId parameter must be provided.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder question(String question);

        /**
         * <p>
         * An arbitrary data field. The RequesterAnnotation parameter lets your application attach arbitrary data to the
         * HIT for tracking purposes. For example, this parameter could be an identifier internal to the Requester's
         * application that corresponds with the HIT.
         * </p>
         * <p>
         * The RequesterAnnotation parameter for a HIT is only visible to the Requester who created the HIT. It is not
         * shown to the Worker, or any other Requester.
         * </p>
         * <p>
         * The RequesterAnnotation parameter may be different for each HIT you submit. It does not affect how your HITs
         * are grouped.
         * </p>
         * 
         * @param requesterAnnotation
         *        An arbitrary data field. The RequesterAnnotation parameter lets your application attach arbitrary data
         *        to the HIT for tracking purposes. For example, this parameter could be an identifier internal to the
         *        Requester's application that corresponds with the HIT. </p>
         *        <p>
         *        The RequesterAnnotation parameter for a HIT is only visible to the Requester who created the HIT. It
         *        is not shown to the Worker, or any other Requester.
         *        </p>
         *        <p>
         *        The RequesterAnnotation parameter may be different for each HIT you submit. It does not affect how
         *        your HITs are grouped.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requesterAnnotation(String requesterAnnotation);

        /**
         * <p>
         * A unique identifier for this request which allows you to retry the call on error without creating duplicate
         * HITs. This is useful in cases such as network timeouts where it is unclear whether or not the call succeeded
         * on the server. If the HIT already exists in the system from a previous call using the same
         * UniqueRequestToken, subsequent calls will return a AWS.MechanicalTurk.HitAlreadyExists error with a message
         * containing the HITId.
         * </p>
         * <note>
         * <p>
         * Note: It is your responsibility to ensure uniqueness of the token. The unique token expires after 24 hours.
         * Subsequent calls using the same UniqueRequestToken made after the 24 hour limit could create duplicate HITs.
         * </p>
         * </note>
         * 
         * @param uniqueRequestToken
         *        A unique identifier for this request which allows you to retry the call on error without creating
         *        duplicate HITs. This is useful in cases such as network timeouts where it is unclear whether or not
         *        the call succeeded on the server. If the HIT already exists in the system from a previous call using
         *        the same UniqueRequestToken, subsequent calls will return a AWS.MechanicalTurk.HitAlreadyExists error
         *        with a message containing the HITId. </p> <note>
         *        <p>
         *        Note: It is your responsibility to ensure uniqueness of the token. The unique token expires after 24
         *        hours. Subsequent calls using the same UniqueRequestToken made after the 24 hour limit could create
         *        duplicate HITs.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder uniqueRequestToken(String uniqueRequestToken);

        /**
         * <p>
         * The Assignment-level Review Policy applies to the assignments under the HIT. You can specify for Mechanical
         * Turk to take various actions based on the policy.
         * </p>
         * 
         * @param assignmentReviewPolicy
         *        The Assignment-level Review Policy applies to the assignments under the HIT. You can specify for
         *        Mechanical Turk to take various actions based on the policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assignmentReviewPolicy(ReviewPolicy assignmentReviewPolicy);

        /**
         * <p>
         * The Assignment-level Review Policy applies to the assignments under the HIT. You can specify for Mechanical
         * Turk to take various actions based on the policy.
         * </p>
         * This is a convenience that creates an instance of the {@link ReviewPolicy.Builder} avoiding the need to
         * create one manually via {@link ReviewPolicy#builder()}.
         *
         * When the {@link Consumer} completes, {@link ReviewPolicy.Builder#build()} is called immediately and its
         * result is passed to {@link #assignmentReviewPolicy(ReviewPolicy)}.
         * 
         * @param assignmentReviewPolicy
         *        a consumer that will call methods on {@link ReviewPolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #assignmentReviewPolicy(ReviewPolicy)
         */
        default Builder assignmentReviewPolicy(Consumer<ReviewPolicy.Builder> assignmentReviewPolicy) {
            return assignmentReviewPolicy(ReviewPolicy.builder().applyMutation(assignmentReviewPolicy).build());
        }

        /**
         * <p>
         * The HIT-level Review Policy applies to the HIT. You can specify for Mechanical Turk to take various actions
         * based on the policy.
         * </p>
         * 
         * @param hitReviewPolicy
         *        The HIT-level Review Policy applies to the HIT. You can specify for Mechanical Turk to take various
         *        actions based on the policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hitReviewPolicy(ReviewPolicy hitReviewPolicy);

        /**
         * <p>
         * The HIT-level Review Policy applies to the HIT. You can specify for Mechanical Turk to take various actions
         * based on the policy.
         * </p>
         * This is a convenience that creates an instance of the {@link ReviewPolicy.Builder} avoiding the need to
         * create one manually via {@link ReviewPolicy#builder()}.
         *
         * When the {@link Consumer} completes, {@link ReviewPolicy.Builder#build()} is called immediately and its
         * result is passed to {@link #hitReviewPolicy(ReviewPolicy)}.
         * 
         * @param hitReviewPolicy
         *        a consumer that will call methods on {@link ReviewPolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hitReviewPolicy(ReviewPolicy)
         */
        default Builder hitReviewPolicy(Consumer<ReviewPolicy.Builder> hitReviewPolicy) {
            return hitReviewPolicy(ReviewPolicy.builder().applyMutation(hitReviewPolicy).build());
        }

        /**
         * <p>
         * The HITLayoutId allows you to use a pre-existing HIT design with placeholder values and create an additional
         * HIT by providing those values as HITLayoutParameters.
         * </p>
         * <p>
         * Constraints: Either a Question parameter or a HITLayoutId parameter must be provided.
         * </p>
         * 
         * @param hitLayoutId
         *        The HITLayoutId allows you to use a pre-existing HIT design with placeholder values and create an
         *        additional HIT by providing those values as HITLayoutParameters. </p>
         *        <p>
         *        Constraints: Either a Question parameter or a HITLayoutId parameter must be provided.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hitLayoutId(String hitLayoutId);

        /**
         * <p>
         * If the HITLayoutId is provided, any placeholder values must be filled in with values using the
         * HITLayoutParameter structure. For more information, see HITLayout.
         * </p>
         * 
         * @param hitLayoutParameters
         *        If the HITLayoutId is provided, any placeholder values must be filled in with values using the
         *        HITLayoutParameter structure. For more information, see HITLayout.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hitLayoutParameters(Collection<HITLayoutParameter> hitLayoutParameters);

        /**
         * <p>
         * If the HITLayoutId is provided, any placeholder values must be filled in with values using the
         * HITLayoutParameter structure. For more information, see HITLayout.
         * </p>
         * 
         * @param hitLayoutParameters
         *        If the HITLayoutId is provided, any placeholder values must be filled in with values using the
         *        HITLayoutParameter structure. For more information, see HITLayout.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hitLayoutParameters(HITLayoutParameter... hitLayoutParameters);

        /**
         * <p>
         * If the HITLayoutId is provided, any placeholder values must be filled in with values using the
         * HITLayoutParameter structure. For more information, see HITLayout.
         * </p>
         * This is a convenience that creates an instance of the {@link List<HITLayoutParameter>.Builder} avoiding the
         * need to create one manually via {@link List<HITLayoutParameter>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<HITLayoutParameter>.Builder#build()} is called immediately
         * and its result is passed to {@link #hitLayoutParameters(List<HITLayoutParameter>)}.
         * 
         * @param hitLayoutParameters
         *        a consumer that will call methods on {@link List<HITLayoutParameter>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hitLayoutParameters(List<HITLayoutParameter>)
         */
        Builder hitLayoutParameters(Consumer<HITLayoutParameter.Builder>... hitLayoutParameters);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends MTurkRequest.BuilderImpl implements Builder {
        private String hitTypeId;

        private Integer maxAssignments;

        private Long lifetimeInSeconds;

        private String question;

        private String requesterAnnotation;

        private String uniqueRequestToken;

        private ReviewPolicy assignmentReviewPolicy;

        private ReviewPolicy hitReviewPolicy;

        private String hitLayoutId;

        private List<HITLayoutParameter> hitLayoutParameters = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateHitWithHitTypeRequest model) {
            super(model);
            hitTypeId(model.hitTypeId);
            maxAssignments(model.maxAssignments);
            lifetimeInSeconds(model.lifetimeInSeconds);
            question(model.question);
            requesterAnnotation(model.requesterAnnotation);
            uniqueRequestToken(model.uniqueRequestToken);
            assignmentReviewPolicy(model.assignmentReviewPolicy);
            hitReviewPolicy(model.hitReviewPolicy);
            hitLayoutId(model.hitLayoutId);
            hitLayoutParameters(model.hitLayoutParameters);
        }

        public final String getHITTypeId() {
            return hitTypeId;
        }

        @Override
        public final Builder hitTypeId(String hitTypeId) {
            this.hitTypeId = hitTypeId;
            return this;
        }

        public final void setHITTypeId(String hitTypeId) {
            this.hitTypeId = hitTypeId;
        }

        public final Integer getMaxAssignments() {
            return maxAssignments;
        }

        @Override
        public final Builder maxAssignments(Integer maxAssignments) {
            this.maxAssignments = maxAssignments;
            return this;
        }

        public final void setMaxAssignments(Integer maxAssignments) {
            this.maxAssignments = maxAssignments;
        }

        public final Long getLifetimeInSeconds() {
            return lifetimeInSeconds;
        }

        @Override
        public final Builder lifetimeInSeconds(Long lifetimeInSeconds) {
            this.lifetimeInSeconds = lifetimeInSeconds;
            return this;
        }

        public final void setLifetimeInSeconds(Long lifetimeInSeconds) {
            this.lifetimeInSeconds = lifetimeInSeconds;
        }

        public final String getQuestion() {
            return question;
        }

        @Override
        public final Builder question(String question) {
            this.question = question;
            return this;
        }

        public final void setQuestion(String question) {
            this.question = question;
        }

        public final String getRequesterAnnotation() {
            return requesterAnnotation;
        }

        @Override
        public final Builder requesterAnnotation(String requesterAnnotation) {
            this.requesterAnnotation = requesterAnnotation;
            return this;
        }

        public final void setRequesterAnnotation(String requesterAnnotation) {
            this.requesterAnnotation = requesterAnnotation;
        }

        public final String getUniqueRequestToken() {
            return uniqueRequestToken;
        }

        @Override
        public final Builder uniqueRequestToken(String uniqueRequestToken) {
            this.uniqueRequestToken = uniqueRequestToken;
            return this;
        }

        public final void setUniqueRequestToken(String uniqueRequestToken) {
            this.uniqueRequestToken = uniqueRequestToken;
        }

        public final ReviewPolicy.Builder getAssignmentReviewPolicy() {
            return assignmentReviewPolicy != null ? assignmentReviewPolicy.toBuilder() : null;
        }

        @Override
        public final Builder assignmentReviewPolicy(ReviewPolicy assignmentReviewPolicy) {
            this.assignmentReviewPolicy = assignmentReviewPolicy;
            return this;
        }

        public final void setAssignmentReviewPolicy(ReviewPolicy.BuilderImpl assignmentReviewPolicy) {
            this.assignmentReviewPolicy = assignmentReviewPolicy != null ? assignmentReviewPolicy.build() : null;
        }

        public final ReviewPolicy.Builder getHITReviewPolicy() {
            return hitReviewPolicy != null ? hitReviewPolicy.toBuilder() : null;
        }

        @Override
        public final Builder hitReviewPolicy(ReviewPolicy hitReviewPolicy) {
            this.hitReviewPolicy = hitReviewPolicy;
            return this;
        }

        public final void setHITReviewPolicy(ReviewPolicy.BuilderImpl hitReviewPolicy) {
            this.hitReviewPolicy = hitReviewPolicy != null ? hitReviewPolicy.build() : null;
        }

        public final String getHITLayoutId() {
            return hitLayoutId;
        }

        @Override
        public final Builder hitLayoutId(String hitLayoutId) {
            this.hitLayoutId = hitLayoutId;
            return this;
        }

        public final void setHITLayoutId(String hitLayoutId) {
            this.hitLayoutId = hitLayoutId;
        }

        public final Collection<HITLayoutParameter.Builder> getHITLayoutParameters() {
            return hitLayoutParameters != null ? hitLayoutParameters.stream().map(HITLayoutParameter::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder hitLayoutParameters(Collection<HITLayoutParameter> hitLayoutParameters) {
            this.hitLayoutParameters = HITLayoutParameterListCopier.copy(hitLayoutParameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder hitLayoutParameters(HITLayoutParameter... hitLayoutParameters) {
            hitLayoutParameters(Arrays.asList(hitLayoutParameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder hitLayoutParameters(Consumer<HITLayoutParameter.Builder>... hitLayoutParameters) {
            hitLayoutParameters(Stream.of(hitLayoutParameters).map(c -> HITLayoutParameter.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setHITLayoutParameters(Collection<HITLayoutParameter.BuilderImpl> hitLayoutParameters) {
            this.hitLayoutParameters = HITLayoutParameterListCopier.copyFromBuilder(hitLayoutParameters);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateHitWithHitTypeRequest build() {
            return new CreateHitWithHitTypeRequest(this);
        }
    }
}
