/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mturk.model;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.mturk.transform.QualificationRequestMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The QualificationRequest data structure represents a request a Worker has made for a Qualification.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class QualificationRequest implements StructuredPojo,
        ToCopyableBuilder<QualificationRequest.Builder, QualificationRequest> {
    private final String qualificationRequestId;

    private final String qualificationTypeId;

    private final String workerId;

    private final String test;

    private final String answer;

    private final Instant submitTime;

    private QualificationRequest(BuilderImpl builder) {
        this.qualificationRequestId = builder.qualificationRequestId;
        this.qualificationTypeId = builder.qualificationTypeId;
        this.workerId = builder.workerId;
        this.test = builder.test;
        this.answer = builder.answer;
        this.submitTime = builder.submitTime;
    }

    /**
     * <p>
     * The ID of the Qualification request, a unique identifier generated when the request was submitted.
     * </p>
     * 
     * @return The ID of the Qualification request, a unique identifier generated when the request was submitted.
     */
    public String qualificationRequestId() {
        return qualificationRequestId;
    }

    /**
     * <p>
     * The ID of the Qualification type the Worker is requesting, as returned by the CreateQualificationType operation.
     * </p>
     * 
     * @return The ID of the Qualification type the Worker is requesting, as returned by the CreateQualificationType
     *         operation.
     */
    public String qualificationTypeId() {
        return qualificationTypeId;
    }

    /**
     * <p>
     * The ID of the Worker requesting the Qualification.
     * </p>
     * 
     * @return The ID of the Worker requesting the Qualification.
     */
    public String workerId() {
        return workerId;
    }

    /**
     * <p>
     * The contents of the Qualification test that was presented to the Worker, if the type has a test and the Worker
     * has submitted answers. This value is identical to the QuestionForm associated with the Qualification type at the
     * time the Worker requests the Qualification.
     * </p>
     * 
     * @return The contents of the Qualification test that was presented to the Worker, if the type has a test and the
     *         Worker has submitted answers. This value is identical to the QuestionForm associated with the
     *         Qualification type at the time the Worker requests the Qualification.
     */
    public String test() {
        return test;
    }

    /**
     * <p>
     * The Worker's answers for the Qualification type's test contained in a QuestionFormAnswers document, if the type
     * has a test and the Worker has submitted answers. If the Worker does not provide any answers, Answer may be empty.
     * </p>
     * 
     * @return The Worker's answers for the Qualification type's test contained in a QuestionFormAnswers document, if
     *         the type has a test and the Worker has submitted answers. If the Worker does not provide any answers,
     *         Answer may be empty.
     */
    public String answer() {
        return answer;
    }

    /**
     * <p>
     * The date and time the Qualification request had a status of Submitted. This is either the time the Worker
     * submitted answers for a Qualification test, or the time the Worker requested the Qualification if the
     * Qualification type does not have a test.
     * </p>
     * 
     * @return The date and time the Qualification request had a status of Submitted. This is either the time the Worker
     *         submitted answers for a Qualification test, or the time the Worker requested the Qualification if the
     *         Qualification type does not have a test.
     */
    public Instant submitTime() {
        return submitTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(qualificationRequestId());
        hashCode = 31 * hashCode + Objects.hashCode(qualificationTypeId());
        hashCode = 31 * hashCode + Objects.hashCode(workerId());
        hashCode = 31 * hashCode + Objects.hashCode(test());
        hashCode = 31 * hashCode + Objects.hashCode(answer());
        hashCode = 31 * hashCode + Objects.hashCode(submitTime());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof QualificationRequest)) {
            return false;
        }
        QualificationRequest other = (QualificationRequest) obj;
        return Objects.equals(qualificationRequestId(), other.qualificationRequestId())
                && Objects.equals(qualificationTypeId(), other.qualificationTypeId())
                && Objects.equals(workerId(), other.workerId()) && Objects.equals(test(), other.test())
                && Objects.equals(answer(), other.answer()) && Objects.equals(submitTime(), other.submitTime());
    }

    @Override
    public String toString() {
        return ToString.builder("QualificationRequest").add("QualificationRequestId", qualificationRequestId())
                .add("QualificationTypeId", qualificationTypeId()).add("WorkerId", workerId()).add("Test", test())
                .add("Answer", answer()).add("SubmitTime", submitTime()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "QualificationRequestId":
            return Optional.ofNullable(clazz.cast(qualificationRequestId()));
        case "QualificationTypeId":
            return Optional.ofNullable(clazz.cast(qualificationTypeId()));
        case "WorkerId":
            return Optional.ofNullable(clazz.cast(workerId()));
        case "Test":
            return Optional.ofNullable(clazz.cast(test()));
        case "Answer":
            return Optional.ofNullable(clazz.cast(answer()));
        case "SubmitTime":
            return Optional.ofNullable(clazz.cast(submitTime()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        QualificationRequestMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, QualificationRequest> {
        /**
         * <p>
         * The ID of the Qualification request, a unique identifier generated when the request was submitted.
         * </p>
         * 
         * @param qualificationRequestId
         *        The ID of the Qualification request, a unique identifier generated when the request was submitted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder qualificationRequestId(String qualificationRequestId);

        /**
         * <p>
         * The ID of the Qualification type the Worker is requesting, as returned by the CreateQualificationType
         * operation.
         * </p>
         * 
         * @param qualificationTypeId
         *        The ID of the Qualification type the Worker is requesting, as returned by the CreateQualificationType
         *        operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder qualificationTypeId(String qualificationTypeId);

        /**
         * <p>
         * The ID of the Worker requesting the Qualification.
         * </p>
         * 
         * @param workerId
         *        The ID of the Worker requesting the Qualification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workerId(String workerId);

        /**
         * <p>
         * The contents of the Qualification test that was presented to the Worker, if the type has a test and the
         * Worker has submitted answers. This value is identical to the QuestionForm associated with the Qualification
         * type at the time the Worker requests the Qualification.
         * </p>
         * 
         * @param test
         *        The contents of the Qualification test that was presented to the Worker, if the type has a test and
         *        the Worker has submitted answers. This value is identical to the QuestionForm associated with the
         *        Qualification type at the time the Worker requests the Qualification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder test(String test);

        /**
         * <p>
         * The Worker's answers for the Qualification type's test contained in a QuestionFormAnswers document, if the
         * type has a test and the Worker has submitted answers. If the Worker does not provide any answers, Answer may
         * be empty.
         * </p>
         * 
         * @param answer
         *        The Worker's answers for the Qualification type's test contained in a QuestionFormAnswers document, if
         *        the type has a test and the Worker has submitted answers. If the Worker does not provide any answers,
         *        Answer may be empty.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder answer(String answer);

        /**
         * <p>
         * The date and time the Qualification request had a status of Submitted. This is either the time the Worker
         * submitted answers for a Qualification test, or the time the Worker requested the Qualification if the
         * Qualification type does not have a test.
         * </p>
         * 
         * @param submitTime
         *        The date and time the Qualification request had a status of Submitted. This is either the time the
         *        Worker submitted answers for a Qualification test, or the time the Worker requested the Qualification
         *        if the Qualification type does not have a test.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder submitTime(Instant submitTime);
    }

    static final class BuilderImpl implements Builder {
        private String qualificationRequestId;

        private String qualificationTypeId;

        private String workerId;

        private String test;

        private String answer;

        private Instant submitTime;

        private BuilderImpl() {
        }

        private BuilderImpl(QualificationRequest model) {
            qualificationRequestId(model.qualificationRequestId);
            qualificationTypeId(model.qualificationTypeId);
            workerId(model.workerId);
            test(model.test);
            answer(model.answer);
            submitTime(model.submitTime);
        }

        public final String getQualificationRequestId() {
            return qualificationRequestId;
        }

        @Override
        public final Builder qualificationRequestId(String qualificationRequestId) {
            this.qualificationRequestId = qualificationRequestId;
            return this;
        }

        public final void setQualificationRequestId(String qualificationRequestId) {
            this.qualificationRequestId = qualificationRequestId;
        }

        public final String getQualificationTypeId() {
            return qualificationTypeId;
        }

        @Override
        public final Builder qualificationTypeId(String qualificationTypeId) {
            this.qualificationTypeId = qualificationTypeId;
            return this;
        }

        public final void setQualificationTypeId(String qualificationTypeId) {
            this.qualificationTypeId = qualificationTypeId;
        }

        public final String getWorkerId() {
            return workerId;
        }

        @Override
        public final Builder workerId(String workerId) {
            this.workerId = workerId;
            return this;
        }

        public final void setWorkerId(String workerId) {
            this.workerId = workerId;
        }

        public final String getTest() {
            return test;
        }

        @Override
        public final Builder test(String test) {
            this.test = test;
            return this;
        }

        public final void setTest(String test) {
            this.test = test;
        }

        public final String getAnswer() {
            return answer;
        }

        @Override
        public final Builder answer(String answer) {
            this.answer = answer;
            return this;
        }

        public final void setAnswer(String answer) {
            this.answer = answer;
        }

        public final Instant getSubmitTime() {
            return submitTime;
        }

        @Override
        public final Builder submitTime(Instant submitTime) {
            this.submitTime = submitTime;
            return this;
        }

        public final void setSubmitTime(Instant submitTime) {
            this.submitTime = submitTime;
        }

        @Override
        public QualificationRequest build() {
            return new QualificationRequest(this);
        }
    }
}
