/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mturk.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The QualificationRequirement data structure describes a Qualification that a Worker must have before the Worker is
 * allowed to accept a HIT. A requirement may optionally state that a Worker must have the Qualification in order to
 * preview the HIT, or see the HIT in search results.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class QualificationRequirement implements SdkPojo, Serializable,
        ToCopyableBuilder<QualificationRequirement.Builder, QualificationRequirement> {
    private static final SdkField<String> QUALIFICATION_TYPE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(QualificationRequirement::qualificationTypeId)).setter(setter(Builder::qualificationTypeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QualificationTypeId").build())
            .build();

    private static final SdkField<String> COMPARATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(QualificationRequirement::comparatorAsString)).setter(setter(Builder::comparator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Comparator").build()).build();

    private static final SdkField<List<Integer>> INTEGER_VALUES_FIELD = SdkField
            .<List<Integer>> builder(MarshallingType.LIST)
            .getter(getter(QualificationRequirement::integerValues))
            .setter(setter(Builder::integerValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IntegerValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Locale>> LOCALE_VALUES_FIELD = SdkField
            .<List<Locale>> builder(MarshallingType.LIST)
            .getter(getter(QualificationRequirement::localeValues))
            .setter(setter(Builder::localeValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LocaleValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Locale> builder(MarshallingType.SDK_POJO)
                                            .constructor(Locale::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> REQUIRED_TO_PREVIEW_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(QualificationRequirement::requiredToPreview)).setter(setter(Builder::requiredToPreview))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequiredToPreview").build()).build();

    private static final SdkField<String> ACTIONS_GUARDED_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(QualificationRequirement::actionsGuardedAsString)).setter(setter(Builder::actionsGuarded))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActionsGuarded").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(QUALIFICATION_TYPE_ID_FIELD,
            COMPARATOR_FIELD, INTEGER_VALUES_FIELD, LOCALE_VALUES_FIELD, REQUIRED_TO_PREVIEW_FIELD, ACTIONS_GUARDED_FIELD));

    private static final long serialVersionUID = 1L;

    private final String qualificationTypeId;

    private final String comparator;

    private final List<Integer> integerValues;

    private final List<Locale> localeValues;

    private final Boolean requiredToPreview;

    private final String actionsGuarded;

    private QualificationRequirement(BuilderImpl builder) {
        this.qualificationTypeId = builder.qualificationTypeId;
        this.comparator = builder.comparator;
        this.integerValues = builder.integerValues;
        this.localeValues = builder.localeValues;
        this.requiredToPreview = builder.requiredToPreview;
        this.actionsGuarded = builder.actionsGuarded;
    }

    /**
     * <p>
     * The ID of the Qualification type for the requirement.
     * </p>
     * 
     * @return The ID of the Qualification type for the requirement.
     */
    public String qualificationTypeId() {
        return qualificationTypeId;
    }

    /**
     * <p>
     * The kind of comparison to make against a Qualification's value. You can compare a Qualification's value to an
     * IntegerValue to see if it is LessThan, LessThanOrEqualTo, GreaterThan, GreaterThanOrEqualTo, EqualTo, or
     * NotEqualTo the IntegerValue. You can compare it to a LocaleValue to see if it is EqualTo, or NotEqualTo the
     * LocaleValue. You can check to see if the value is In or NotIn a set of IntegerValue or LocaleValue values.
     * Lastly, a Qualification requirement can also test if a Qualification Exists or DoesNotExist in the user's
     * profile, regardless of its value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #comparator} will
     * return {@link Comparator#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #comparatorAsString}.
     * </p>
     * 
     * @return The kind of comparison to make against a Qualification's value. You can compare a Qualification's value
     *         to an IntegerValue to see if it is LessThan, LessThanOrEqualTo, GreaterThan, GreaterThanOrEqualTo,
     *         EqualTo, or NotEqualTo the IntegerValue. You can compare it to a LocaleValue to see if it is EqualTo, or
     *         NotEqualTo the LocaleValue. You can check to see if the value is In or NotIn a set of IntegerValue or
     *         LocaleValue values. Lastly, a Qualification requirement can also test if a Qualification Exists or
     *         DoesNotExist in the user's profile, regardless of its value.
     * @see Comparator
     */
    public Comparator comparator() {
        return Comparator.fromValue(comparator);
    }

    /**
     * <p>
     * The kind of comparison to make against a Qualification's value. You can compare a Qualification's value to an
     * IntegerValue to see if it is LessThan, LessThanOrEqualTo, GreaterThan, GreaterThanOrEqualTo, EqualTo, or
     * NotEqualTo the IntegerValue. You can compare it to a LocaleValue to see if it is EqualTo, or NotEqualTo the
     * LocaleValue. You can check to see if the value is In or NotIn a set of IntegerValue or LocaleValue values.
     * Lastly, a Qualification requirement can also test if a Qualification Exists or DoesNotExist in the user's
     * profile, regardless of its value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #comparator} will
     * return {@link Comparator#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #comparatorAsString}.
     * </p>
     * 
     * @return The kind of comparison to make against a Qualification's value. You can compare a Qualification's value
     *         to an IntegerValue to see if it is LessThan, LessThanOrEqualTo, GreaterThan, GreaterThanOrEqualTo,
     *         EqualTo, or NotEqualTo the IntegerValue. You can compare it to a LocaleValue to see if it is EqualTo, or
     *         NotEqualTo the LocaleValue. You can check to see if the value is In or NotIn a set of IntegerValue or
     *         LocaleValue values. Lastly, a Qualification requirement can also test if a Qualification Exists or
     *         DoesNotExist in the user's profile, regardless of its value.
     * @see Comparator
     */
    public String comparatorAsString() {
        return comparator;
    }

    /**
     * Returns true if the IntegerValues property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasIntegerValues() {
        return integerValues != null && !(integerValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The integer value to compare against the Qualification's value. IntegerValue must not be present if Comparator is
     * Exists or DoesNotExist. IntegerValue can only be used if the Qualification type has an integer value; it cannot
     * be used with the Worker_Locale QualificationType ID. When performing a set comparison by using the In or the
     * NotIn comparator, you can use up to 15 IntegerValue elements in a QualificationRequirement data structure.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasIntegerValues()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The integer value to compare against the Qualification's value. IntegerValue must not be present if
     *         Comparator is Exists or DoesNotExist. IntegerValue can only be used if the Qualification type has an
     *         integer value; it cannot be used with the Worker_Locale QualificationType ID. When performing a set
     *         comparison by using the In or the NotIn comparator, you can use up to 15 IntegerValue elements in a
     *         QualificationRequirement data structure.
     */
    public List<Integer> integerValues() {
        return integerValues;
    }

    /**
     * Returns true if the LocaleValues property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasLocaleValues() {
        return localeValues != null && !(localeValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The locale value to compare against the Qualification's value. The local value must be a valid ISO 3166 country
     * code or supports ISO 3166-2 subdivisions. LocaleValue can only be used with a Worker_Locale QualificationType ID.
     * LocaleValue can only be used with the EqualTo, NotEqualTo, In, and NotIn comparators. You must only use a single
     * LocaleValue element when using the EqualTo or NotEqualTo comparators. When performing a set comparison by using
     * the In or the NotIn comparator, you can use up to 30 LocaleValue elements in a QualificationRequirement data
     * structure.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLocaleValues()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The locale value to compare against the Qualification's value. The local value must be a valid ISO 3166
     *         country code or supports ISO 3166-2 subdivisions. LocaleValue can only be used with a Worker_Locale
     *         QualificationType ID. LocaleValue can only be used with the EqualTo, NotEqualTo, In, and NotIn
     *         comparators. You must only use a single LocaleValue element when using the EqualTo or NotEqualTo
     *         comparators. When performing a set comparison by using the In or the NotIn comparator, you can use up to
     *         30 LocaleValue elements in a QualificationRequirement data structure.
     */
    public List<Locale> localeValues() {
        return localeValues;
    }

    /**
     * <p>
     * DEPRECATED: Use the <code>ActionsGuarded</code> field instead. If RequiredToPreview is true, the question data
     * for the HIT will not be shown when a Worker whose Qualifications do not meet this requirement tries to preview
     * the HIT. That is, a Worker's Qualifications must meet all of the requirements for which RequiredToPreview is true
     * in order to preview the HIT. If a Worker meets all of the requirements where RequiredToPreview is true (or if
     * there are no such requirements), but does not meet all of the requirements for the HIT, the Worker will be
     * allowed to preview the HIT's question data, but will not be allowed to accept and complete the HIT. The default
     * is false. This should not be used in combination with the <code>ActionsGuarded</code> field.
     * </p>
     * 
     * @return DEPRECATED: Use the <code>ActionsGuarded</code> field instead. If RequiredToPreview is true, the question
     *         data for the HIT will not be shown when a Worker whose Qualifications do not meet this requirement tries
     *         to preview the HIT. That is, a Worker's Qualifications must meet all of the requirements for which
     *         RequiredToPreview is true in order to preview the HIT. If a Worker meets all of the requirements where
     *         RequiredToPreview is true (or if there are no such requirements), but does not meet all of the
     *         requirements for the HIT, the Worker will be allowed to preview the HIT's question data, but will not be
     *         allowed to accept and complete the HIT. The default is false. This should not be used in combination with
     *         the <code>ActionsGuarded</code> field.
     */
    public Boolean requiredToPreview() {
        return requiredToPreview;
    }

    /**
     * <p>
     * Setting this attribute prevents Workers whose Qualifications do not meet this QualificationRequirement from
     * taking the specified action. Valid arguments include "Accept" (Worker cannot accept the HIT, but can preview the
     * HIT and see it in their search results), "PreviewAndAccept" (Worker cannot accept or preview the HIT, but can see
     * the HIT in their search results), and "DiscoverPreviewAndAccept" (Worker cannot accept, preview, or see the HIT
     * in their search results). It's possible for you to create a HIT with multiple QualificationRequirements (which
     * can have different values for the ActionGuarded attribute). In this case, the Worker is only permitted to perform
     * an action when they have met all QualificationRequirements guarding the action. The actions in the order of least
     * restrictive to most restrictive are Discover, Preview and Accept. For example, if a Worker meets all
     * QualificationRequirements that are set to DiscoverPreviewAndAccept, but do not meet all requirements that are set
     * with PreviewAndAccept, then the Worker will be able to Discover, i.e. see the HIT in their search result, but
     * will not be able to Preview or Accept the HIT. ActionsGuarded should not be used in combination with the
     * <code>RequiredToPreview</code> field.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #actionsGuarded}
     * will return {@link HITAccessActions#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #actionsGuardedAsString}.
     * </p>
     * 
     * @return Setting this attribute prevents Workers whose Qualifications do not meet this QualificationRequirement
     *         from taking the specified action. Valid arguments include "Accept" (Worker cannot accept the HIT, but can
     *         preview the HIT and see it in their search results), "PreviewAndAccept" (Worker cannot accept or preview
     *         the HIT, but can see the HIT in their search results), and "DiscoverPreviewAndAccept" (Worker cannot
     *         accept, preview, or see the HIT in their search results). It's possible for you to create a HIT with
     *         multiple QualificationRequirements (which can have different values for the ActionGuarded attribute). In
     *         this case, the Worker is only permitted to perform an action when they have met all
     *         QualificationRequirements guarding the action. The actions in the order of least restrictive to most
     *         restrictive are Discover, Preview and Accept. For example, if a Worker meets all
     *         QualificationRequirements that are set to DiscoverPreviewAndAccept, but do not meet all requirements that
     *         are set with PreviewAndAccept, then the Worker will be able to Discover, i.e. see the HIT in their search
     *         result, but will not be able to Preview or Accept the HIT. ActionsGuarded should not be used in
     *         combination with the <code>RequiredToPreview</code> field.
     * @see HITAccessActions
     */
    public HITAccessActions actionsGuarded() {
        return HITAccessActions.fromValue(actionsGuarded);
    }

    /**
     * <p>
     * Setting this attribute prevents Workers whose Qualifications do not meet this QualificationRequirement from
     * taking the specified action. Valid arguments include "Accept" (Worker cannot accept the HIT, but can preview the
     * HIT and see it in their search results), "PreviewAndAccept" (Worker cannot accept or preview the HIT, but can see
     * the HIT in their search results), and "DiscoverPreviewAndAccept" (Worker cannot accept, preview, or see the HIT
     * in their search results). It's possible for you to create a HIT with multiple QualificationRequirements (which
     * can have different values for the ActionGuarded attribute). In this case, the Worker is only permitted to perform
     * an action when they have met all QualificationRequirements guarding the action. The actions in the order of least
     * restrictive to most restrictive are Discover, Preview and Accept. For example, if a Worker meets all
     * QualificationRequirements that are set to DiscoverPreviewAndAccept, but do not meet all requirements that are set
     * with PreviewAndAccept, then the Worker will be able to Discover, i.e. see the HIT in their search result, but
     * will not be able to Preview or Accept the HIT. ActionsGuarded should not be used in combination with the
     * <code>RequiredToPreview</code> field.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #actionsGuarded}
     * will return {@link HITAccessActions#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #actionsGuardedAsString}.
     * </p>
     * 
     * @return Setting this attribute prevents Workers whose Qualifications do not meet this QualificationRequirement
     *         from taking the specified action. Valid arguments include "Accept" (Worker cannot accept the HIT, but can
     *         preview the HIT and see it in their search results), "PreviewAndAccept" (Worker cannot accept or preview
     *         the HIT, but can see the HIT in their search results), and "DiscoverPreviewAndAccept" (Worker cannot
     *         accept, preview, or see the HIT in their search results). It's possible for you to create a HIT with
     *         multiple QualificationRequirements (which can have different values for the ActionGuarded attribute). In
     *         this case, the Worker is only permitted to perform an action when they have met all
     *         QualificationRequirements guarding the action. The actions in the order of least restrictive to most
     *         restrictive are Discover, Preview and Accept. For example, if a Worker meets all
     *         QualificationRequirements that are set to DiscoverPreviewAndAccept, but do not meet all requirements that
     *         are set with PreviewAndAccept, then the Worker will be able to Discover, i.e. see the HIT in their search
     *         result, but will not be able to Preview or Accept the HIT. ActionsGuarded should not be used in
     *         combination with the <code>RequiredToPreview</code> field.
     * @see HITAccessActions
     */
    public String actionsGuardedAsString() {
        return actionsGuarded;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(qualificationTypeId());
        hashCode = 31 * hashCode + Objects.hashCode(comparatorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(integerValues());
        hashCode = 31 * hashCode + Objects.hashCode(localeValues());
        hashCode = 31 * hashCode + Objects.hashCode(requiredToPreview());
        hashCode = 31 * hashCode + Objects.hashCode(actionsGuardedAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof QualificationRequirement)) {
            return false;
        }
        QualificationRequirement other = (QualificationRequirement) obj;
        return Objects.equals(qualificationTypeId(), other.qualificationTypeId())
                && Objects.equals(comparatorAsString(), other.comparatorAsString())
                && Objects.equals(integerValues(), other.integerValues()) && Objects.equals(localeValues(), other.localeValues())
                && Objects.equals(requiredToPreview(), other.requiredToPreview())
                && Objects.equals(actionsGuardedAsString(), other.actionsGuardedAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("QualificationRequirement").add("QualificationTypeId", qualificationTypeId())
                .add("Comparator", comparatorAsString()).add("IntegerValues", integerValues())
                .add("LocaleValues", localeValues()).add("RequiredToPreview", requiredToPreview())
                .add("ActionsGuarded", actionsGuardedAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "QualificationTypeId":
            return Optional.ofNullable(clazz.cast(qualificationTypeId()));
        case "Comparator":
            return Optional.ofNullable(clazz.cast(comparatorAsString()));
        case "IntegerValues":
            return Optional.ofNullable(clazz.cast(integerValues()));
        case "LocaleValues":
            return Optional.ofNullable(clazz.cast(localeValues()));
        case "RequiredToPreview":
            return Optional.ofNullable(clazz.cast(requiredToPreview()));
        case "ActionsGuarded":
            return Optional.ofNullable(clazz.cast(actionsGuardedAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<QualificationRequirement, T> g) {
        return obj -> g.apply((QualificationRequirement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, QualificationRequirement> {
        /**
         * <p>
         * The ID of the Qualification type for the requirement.
         * </p>
         * 
         * @param qualificationTypeId
         *        The ID of the Qualification type for the requirement.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder qualificationTypeId(String qualificationTypeId);

        /**
         * <p>
         * The kind of comparison to make against a Qualification's value. You can compare a Qualification's value to an
         * IntegerValue to see if it is LessThan, LessThanOrEqualTo, GreaterThan, GreaterThanOrEqualTo, EqualTo, or
         * NotEqualTo the IntegerValue. You can compare it to a LocaleValue to see if it is EqualTo, or NotEqualTo the
         * LocaleValue. You can check to see if the value is In or NotIn a set of IntegerValue or LocaleValue values.
         * Lastly, a Qualification requirement can also test if a Qualification Exists or DoesNotExist in the user's
         * profile, regardless of its value.
         * </p>
         * 
         * @param comparator
         *        The kind of comparison to make against a Qualification's value. You can compare a Qualification's
         *        value to an IntegerValue to see if it is LessThan, LessThanOrEqualTo, GreaterThan,
         *        GreaterThanOrEqualTo, EqualTo, or NotEqualTo the IntegerValue. You can compare it to a LocaleValue to
         *        see if it is EqualTo, or NotEqualTo the LocaleValue. You can check to see if the value is In or NotIn
         *        a set of IntegerValue or LocaleValue values. Lastly, a Qualification requirement can also test if a
         *        Qualification Exists or DoesNotExist in the user's profile, regardless of its value.
         * @see Comparator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Comparator
         */
        Builder comparator(String comparator);

        /**
         * <p>
         * The kind of comparison to make against a Qualification's value. You can compare a Qualification's value to an
         * IntegerValue to see if it is LessThan, LessThanOrEqualTo, GreaterThan, GreaterThanOrEqualTo, EqualTo, or
         * NotEqualTo the IntegerValue. You can compare it to a LocaleValue to see if it is EqualTo, or NotEqualTo the
         * LocaleValue. You can check to see if the value is In or NotIn a set of IntegerValue or LocaleValue values.
         * Lastly, a Qualification requirement can also test if a Qualification Exists or DoesNotExist in the user's
         * profile, regardless of its value.
         * </p>
         * 
         * @param comparator
         *        The kind of comparison to make against a Qualification's value. You can compare a Qualification's
         *        value to an IntegerValue to see if it is LessThan, LessThanOrEqualTo, GreaterThan,
         *        GreaterThanOrEqualTo, EqualTo, or NotEqualTo the IntegerValue. You can compare it to a LocaleValue to
         *        see if it is EqualTo, or NotEqualTo the LocaleValue. You can check to see if the value is In or NotIn
         *        a set of IntegerValue or LocaleValue values. Lastly, a Qualification requirement can also test if a
         *        Qualification Exists or DoesNotExist in the user's profile, regardless of its value.
         * @see Comparator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Comparator
         */
        Builder comparator(Comparator comparator);

        /**
         * <p>
         * The integer value to compare against the Qualification's value. IntegerValue must not be present if
         * Comparator is Exists or DoesNotExist. IntegerValue can only be used if the Qualification type has an integer
         * value; it cannot be used with the Worker_Locale QualificationType ID. When performing a set comparison by
         * using the In or the NotIn comparator, you can use up to 15 IntegerValue elements in a
         * QualificationRequirement data structure.
         * </p>
         * 
         * @param integerValues
         *        The integer value to compare against the Qualification's value. IntegerValue must not be present if
         *        Comparator is Exists or DoesNotExist. IntegerValue can only be used if the Qualification type has an
         *        integer value; it cannot be used with the Worker_Locale QualificationType ID. When performing a set
         *        comparison by using the In or the NotIn comparator, you can use up to 15 IntegerValue elements in a
         *        QualificationRequirement data structure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder integerValues(Collection<Integer> integerValues);

        /**
         * <p>
         * The integer value to compare against the Qualification's value. IntegerValue must not be present if
         * Comparator is Exists or DoesNotExist. IntegerValue can only be used if the Qualification type has an integer
         * value; it cannot be used with the Worker_Locale QualificationType ID. When performing a set comparison by
         * using the In or the NotIn comparator, you can use up to 15 IntegerValue elements in a
         * QualificationRequirement data structure.
         * </p>
         * 
         * @param integerValues
         *        The integer value to compare against the Qualification's value. IntegerValue must not be present if
         *        Comparator is Exists or DoesNotExist. IntegerValue can only be used if the Qualification type has an
         *        integer value; it cannot be used with the Worker_Locale QualificationType ID. When performing a set
         *        comparison by using the In or the NotIn comparator, you can use up to 15 IntegerValue elements in a
         *        QualificationRequirement data structure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder integerValues(Integer... integerValues);

        /**
         * <p>
         * The locale value to compare against the Qualification's value. The local value must be a valid ISO 3166
         * country code or supports ISO 3166-2 subdivisions. LocaleValue can only be used with a Worker_Locale
         * QualificationType ID. LocaleValue can only be used with the EqualTo, NotEqualTo, In, and NotIn comparators.
         * You must only use a single LocaleValue element when using the EqualTo or NotEqualTo comparators. When
         * performing a set comparison by using the In or the NotIn comparator, you can use up to 30 LocaleValue
         * elements in a QualificationRequirement data structure.
         * </p>
         * 
         * @param localeValues
         *        The locale value to compare against the Qualification's value. The local value must be a valid ISO
         *        3166 country code or supports ISO 3166-2 subdivisions. LocaleValue can only be used with a
         *        Worker_Locale QualificationType ID. LocaleValue can only be used with the EqualTo, NotEqualTo, In, and
         *        NotIn comparators. You must only use a single LocaleValue element when using the EqualTo or NotEqualTo
         *        comparators. When performing a set comparison by using the In or the NotIn comparator, you can use up
         *        to 30 LocaleValue elements in a QualificationRequirement data structure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localeValues(Collection<Locale> localeValues);

        /**
         * <p>
         * The locale value to compare against the Qualification's value. The local value must be a valid ISO 3166
         * country code or supports ISO 3166-2 subdivisions. LocaleValue can only be used with a Worker_Locale
         * QualificationType ID. LocaleValue can only be used with the EqualTo, NotEqualTo, In, and NotIn comparators.
         * You must only use a single LocaleValue element when using the EqualTo or NotEqualTo comparators. When
         * performing a set comparison by using the In or the NotIn comparator, you can use up to 30 LocaleValue
         * elements in a QualificationRequirement data structure.
         * </p>
         * 
         * @param localeValues
         *        The locale value to compare against the Qualification's value. The local value must be a valid ISO
         *        3166 country code or supports ISO 3166-2 subdivisions. LocaleValue can only be used with a
         *        Worker_Locale QualificationType ID. LocaleValue can only be used with the EqualTo, NotEqualTo, In, and
         *        NotIn comparators. You must only use a single LocaleValue element when using the EqualTo or NotEqualTo
         *        comparators. When performing a set comparison by using the In or the NotIn comparator, you can use up
         *        to 30 LocaleValue elements in a QualificationRequirement data structure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localeValues(Locale... localeValues);

        /**
         * <p>
         * The locale value to compare against the Qualification's value. The local value must be a valid ISO 3166
         * country code or supports ISO 3166-2 subdivisions. LocaleValue can only be used with a Worker_Locale
         * QualificationType ID. LocaleValue can only be used with the EqualTo, NotEqualTo, In, and NotIn comparators.
         * You must only use a single LocaleValue element when using the EqualTo or NotEqualTo comparators. When
         * performing a set comparison by using the In or the NotIn comparator, you can use up to 30 LocaleValue
         * elements in a QualificationRequirement data structure.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Locale>.Builder} avoiding the need to
         * create one manually via {@link List<Locale>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Locale>.Builder#build()} is called immediately and its
         * result is passed to {@link #localeValues(List<Locale>)}.
         * 
         * @param localeValues
         *        a consumer that will call methods on {@link List<Locale>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #localeValues(List<Locale>)
         */
        Builder localeValues(Consumer<Locale.Builder>... localeValues);

        /**
         * <p>
         * DEPRECATED: Use the <code>ActionsGuarded</code> field instead. If RequiredToPreview is true, the question
         * data for the HIT will not be shown when a Worker whose Qualifications do not meet this requirement tries to
         * preview the HIT. That is, a Worker's Qualifications must meet all of the requirements for which
         * RequiredToPreview is true in order to preview the HIT. If a Worker meets all of the requirements where
         * RequiredToPreview is true (or if there are no such requirements), but does not meet all of the requirements
         * for the HIT, the Worker will be allowed to preview the HIT's question data, but will not be allowed to accept
         * and complete the HIT. The default is false. This should not be used in combination with the
         * <code>ActionsGuarded</code> field.
         * </p>
         * 
         * @param requiredToPreview
         *        DEPRECATED: Use the <code>ActionsGuarded</code> field instead. If RequiredToPreview is true, the
         *        question data for the HIT will not be shown when a Worker whose Qualifications do not meet this
         *        requirement tries to preview the HIT. That is, a Worker's Qualifications must meet all of the
         *        requirements for which RequiredToPreview is true in order to preview the HIT. If a Worker meets all of
         *        the requirements where RequiredToPreview is true (or if there are no such requirements), but does not
         *        meet all of the requirements for the HIT, the Worker will be allowed to preview the HIT's question
         *        data, but will not be allowed to accept and complete the HIT. The default is false. This should not be
         *        used in combination with the <code>ActionsGuarded</code> field.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requiredToPreview(Boolean requiredToPreview);

        /**
         * <p>
         * Setting this attribute prevents Workers whose Qualifications do not meet this QualificationRequirement from
         * taking the specified action. Valid arguments include "Accept" (Worker cannot accept the HIT, but can preview
         * the HIT and see it in their search results), "PreviewAndAccept" (Worker cannot accept or preview the HIT, but
         * can see the HIT in their search results), and "DiscoverPreviewAndAccept" (Worker cannot accept, preview, or
         * see the HIT in their search results). It's possible for you to create a HIT with multiple
         * QualificationRequirements (which can have different values for the ActionGuarded attribute). In this case,
         * the Worker is only permitted to perform an action when they have met all QualificationRequirements guarding
         * the action. The actions in the order of least restrictive to most restrictive are Discover, Preview and
         * Accept. For example, if a Worker meets all QualificationRequirements that are set to
         * DiscoverPreviewAndAccept, but do not meet all requirements that are set with PreviewAndAccept, then the
         * Worker will be able to Discover, i.e. see the HIT in their search result, but will not be able to Preview or
         * Accept the HIT. ActionsGuarded should not be used in combination with the <code>RequiredToPreview</code>
         * field.
         * </p>
         * 
         * @param actionsGuarded
         *        Setting this attribute prevents Workers whose Qualifications do not meet this QualificationRequirement
         *        from taking the specified action. Valid arguments include "Accept" (Worker cannot accept the HIT, but
         *        can preview the HIT and see it in their search results), "PreviewAndAccept" (Worker cannot accept or
         *        preview the HIT, but can see the HIT in their search results), and "DiscoverPreviewAndAccept" (Worker
         *        cannot accept, preview, or see the HIT in their search results). It's possible for you to create a HIT
         *        with multiple QualificationRequirements (which can have different values for the ActionGuarded
         *        attribute). In this case, the Worker is only permitted to perform an action when they have met all
         *        QualificationRequirements guarding the action. The actions in the order of least restrictive to most
         *        restrictive are Discover, Preview and Accept. For example, if a Worker meets all
         *        QualificationRequirements that are set to DiscoverPreviewAndAccept, but do not meet all requirements
         *        that are set with PreviewAndAccept, then the Worker will be able to Discover, i.e. see the HIT in
         *        their search result, but will not be able to Preview or Accept the HIT. ActionsGuarded should not be
         *        used in combination with the <code>RequiredToPreview</code> field.
         * @see HITAccessActions
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HITAccessActions
         */
        Builder actionsGuarded(String actionsGuarded);

        /**
         * <p>
         * Setting this attribute prevents Workers whose Qualifications do not meet this QualificationRequirement from
         * taking the specified action. Valid arguments include "Accept" (Worker cannot accept the HIT, but can preview
         * the HIT and see it in their search results), "PreviewAndAccept" (Worker cannot accept or preview the HIT, but
         * can see the HIT in their search results), and "DiscoverPreviewAndAccept" (Worker cannot accept, preview, or
         * see the HIT in their search results). It's possible for you to create a HIT with multiple
         * QualificationRequirements (which can have different values for the ActionGuarded attribute). In this case,
         * the Worker is only permitted to perform an action when they have met all QualificationRequirements guarding
         * the action. The actions in the order of least restrictive to most restrictive are Discover, Preview and
         * Accept. For example, if a Worker meets all QualificationRequirements that are set to
         * DiscoverPreviewAndAccept, but do not meet all requirements that are set with PreviewAndAccept, then the
         * Worker will be able to Discover, i.e. see the HIT in their search result, but will not be able to Preview or
         * Accept the HIT. ActionsGuarded should not be used in combination with the <code>RequiredToPreview</code>
         * field.
         * </p>
         * 
         * @param actionsGuarded
         *        Setting this attribute prevents Workers whose Qualifications do not meet this QualificationRequirement
         *        from taking the specified action. Valid arguments include "Accept" (Worker cannot accept the HIT, but
         *        can preview the HIT and see it in their search results), "PreviewAndAccept" (Worker cannot accept or
         *        preview the HIT, but can see the HIT in their search results), and "DiscoverPreviewAndAccept" (Worker
         *        cannot accept, preview, or see the HIT in their search results). It's possible for you to create a HIT
         *        with multiple QualificationRequirements (which can have different values for the ActionGuarded
         *        attribute). In this case, the Worker is only permitted to perform an action when they have met all
         *        QualificationRequirements guarding the action. The actions in the order of least restrictive to most
         *        restrictive are Discover, Preview and Accept. For example, if a Worker meets all
         *        QualificationRequirements that are set to DiscoverPreviewAndAccept, but do not meet all requirements
         *        that are set with PreviewAndAccept, then the Worker will be able to Discover, i.e. see the HIT in
         *        their search result, but will not be able to Preview or Accept the HIT. ActionsGuarded should not be
         *        used in combination with the <code>RequiredToPreview</code> field.
         * @see HITAccessActions
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HITAccessActions
         */
        Builder actionsGuarded(HITAccessActions actionsGuarded);
    }

    static final class BuilderImpl implements Builder {
        private String qualificationTypeId;

        private String comparator;

        private List<Integer> integerValues = DefaultSdkAutoConstructList.getInstance();

        private List<Locale> localeValues = DefaultSdkAutoConstructList.getInstance();

        private Boolean requiredToPreview;

        private String actionsGuarded;

        private BuilderImpl() {
        }

        private BuilderImpl(QualificationRequirement model) {
            qualificationTypeId(model.qualificationTypeId);
            comparator(model.comparator);
            integerValues(model.integerValues);
            localeValues(model.localeValues);
            requiredToPreview(model.requiredToPreview);
            actionsGuarded(model.actionsGuarded);
        }

        public final String getQualificationTypeId() {
            return qualificationTypeId;
        }

        @Override
        public final Builder qualificationTypeId(String qualificationTypeId) {
            this.qualificationTypeId = qualificationTypeId;
            return this;
        }

        public final void setQualificationTypeId(String qualificationTypeId) {
            this.qualificationTypeId = qualificationTypeId;
        }

        public final String getComparatorAsString() {
            return comparator;
        }

        @Override
        public final Builder comparator(String comparator) {
            this.comparator = comparator;
            return this;
        }

        @Override
        public final Builder comparator(Comparator comparator) {
            this.comparator(comparator == null ? null : comparator.toString());
            return this;
        }

        public final void setComparator(String comparator) {
            this.comparator = comparator;
        }

        public final Collection<Integer> getIntegerValues() {
            return integerValues;
        }

        @Override
        public final Builder integerValues(Collection<Integer> integerValues) {
            this.integerValues = IntegerListCopier.copy(integerValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder integerValues(Integer... integerValues) {
            integerValues(Arrays.asList(integerValues));
            return this;
        }

        public final void setIntegerValues(Collection<Integer> integerValues) {
            this.integerValues = IntegerListCopier.copy(integerValues);
        }

        public final Collection<Locale.Builder> getLocaleValues() {
            return localeValues != null ? localeValues.stream().map(Locale::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder localeValues(Collection<Locale> localeValues) {
            this.localeValues = LocaleListCopier.copy(localeValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder localeValues(Locale... localeValues) {
            localeValues(Arrays.asList(localeValues));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder localeValues(Consumer<Locale.Builder>... localeValues) {
            localeValues(Stream.of(localeValues).map(c -> Locale.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setLocaleValues(Collection<Locale.BuilderImpl> localeValues) {
            this.localeValues = LocaleListCopier.copyFromBuilder(localeValues);
        }

        public final Boolean getRequiredToPreview() {
            return requiredToPreview;
        }

        @Override
        public final Builder requiredToPreview(Boolean requiredToPreview) {
            this.requiredToPreview = requiredToPreview;
            return this;
        }

        public final void setRequiredToPreview(Boolean requiredToPreview) {
            this.requiredToPreview = requiredToPreview;
        }

        public final String getActionsGuardedAsString() {
            return actionsGuarded;
        }

        @Override
        public final Builder actionsGuarded(String actionsGuarded) {
            this.actionsGuarded = actionsGuarded;
            return this;
        }

        @Override
        public final Builder actionsGuarded(HITAccessActions actionsGuarded) {
            this.actionsGuarded(actionsGuarded == null ? null : actionsGuarded.toString());
            return this;
        }

        public final void setActionsGuarded(String actionsGuarded) {
            this.actionsGuarded = actionsGuarded;
        }

        @Override
        public QualificationRequirement build() {
            return new QualificationRequirement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
