/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mturk;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mturk.model.AcceptQualificationRequestRequest;
import software.amazon.awssdk.services.mturk.model.AcceptQualificationRequestResponse;
import software.amazon.awssdk.services.mturk.model.ApproveAssignmentRequest;
import software.amazon.awssdk.services.mturk.model.ApproveAssignmentResponse;
import software.amazon.awssdk.services.mturk.model.AssociateQualificationWithWorkerRequest;
import software.amazon.awssdk.services.mturk.model.AssociateQualificationWithWorkerResponse;
import software.amazon.awssdk.services.mturk.model.CreateAdditionalAssignmentsForHitRequest;
import software.amazon.awssdk.services.mturk.model.CreateAdditionalAssignmentsForHitResponse;
import software.amazon.awssdk.services.mturk.model.CreateHitRequest;
import software.amazon.awssdk.services.mturk.model.CreateHitResponse;
import software.amazon.awssdk.services.mturk.model.CreateHitTypeRequest;
import software.amazon.awssdk.services.mturk.model.CreateHitTypeResponse;
import software.amazon.awssdk.services.mturk.model.CreateHitWithHitTypeRequest;
import software.amazon.awssdk.services.mturk.model.CreateHitWithHitTypeResponse;
import software.amazon.awssdk.services.mturk.model.CreateQualificationTypeRequest;
import software.amazon.awssdk.services.mturk.model.CreateQualificationTypeResponse;
import software.amazon.awssdk.services.mturk.model.CreateWorkerBlockRequest;
import software.amazon.awssdk.services.mturk.model.CreateWorkerBlockResponse;
import software.amazon.awssdk.services.mturk.model.DeleteHitRequest;
import software.amazon.awssdk.services.mturk.model.DeleteHitResponse;
import software.amazon.awssdk.services.mturk.model.DeleteQualificationTypeRequest;
import software.amazon.awssdk.services.mturk.model.DeleteQualificationTypeResponse;
import software.amazon.awssdk.services.mturk.model.DeleteWorkerBlockRequest;
import software.amazon.awssdk.services.mturk.model.DeleteWorkerBlockResponse;
import software.amazon.awssdk.services.mturk.model.DisassociateQualificationFromWorkerRequest;
import software.amazon.awssdk.services.mturk.model.DisassociateQualificationFromWorkerResponse;
import software.amazon.awssdk.services.mturk.model.GetAccountBalanceRequest;
import software.amazon.awssdk.services.mturk.model.GetAccountBalanceResponse;
import software.amazon.awssdk.services.mturk.model.GetAssignmentRequest;
import software.amazon.awssdk.services.mturk.model.GetAssignmentResponse;
import software.amazon.awssdk.services.mturk.model.GetFileUploadUrlRequest;
import software.amazon.awssdk.services.mturk.model.GetFileUploadUrlResponse;
import software.amazon.awssdk.services.mturk.model.GetHitRequest;
import software.amazon.awssdk.services.mturk.model.GetHitResponse;
import software.amazon.awssdk.services.mturk.model.GetQualificationScoreRequest;
import software.amazon.awssdk.services.mturk.model.GetQualificationScoreResponse;
import software.amazon.awssdk.services.mturk.model.GetQualificationTypeRequest;
import software.amazon.awssdk.services.mturk.model.GetQualificationTypeResponse;
import software.amazon.awssdk.services.mturk.model.ListAssignmentsForHitRequest;
import software.amazon.awssdk.services.mturk.model.ListAssignmentsForHitResponse;
import software.amazon.awssdk.services.mturk.model.ListBonusPaymentsRequest;
import software.amazon.awssdk.services.mturk.model.ListBonusPaymentsResponse;
import software.amazon.awssdk.services.mturk.model.ListHiTsForQualificationTypeRequest;
import software.amazon.awssdk.services.mturk.model.ListHiTsForQualificationTypeResponse;
import software.amazon.awssdk.services.mturk.model.ListHiTsRequest;
import software.amazon.awssdk.services.mturk.model.ListHiTsResponse;
import software.amazon.awssdk.services.mturk.model.ListQualificationRequestsRequest;
import software.amazon.awssdk.services.mturk.model.ListQualificationRequestsResponse;
import software.amazon.awssdk.services.mturk.model.ListQualificationTypesRequest;
import software.amazon.awssdk.services.mturk.model.ListQualificationTypesResponse;
import software.amazon.awssdk.services.mturk.model.ListReviewPolicyResultsForHitRequest;
import software.amazon.awssdk.services.mturk.model.ListReviewPolicyResultsForHitResponse;
import software.amazon.awssdk.services.mturk.model.ListReviewableHiTsRequest;
import software.amazon.awssdk.services.mturk.model.ListReviewableHiTsResponse;
import software.amazon.awssdk.services.mturk.model.ListWorkerBlocksRequest;
import software.amazon.awssdk.services.mturk.model.ListWorkerBlocksResponse;
import software.amazon.awssdk.services.mturk.model.ListWorkersWithQualificationTypeRequest;
import software.amazon.awssdk.services.mturk.model.ListWorkersWithQualificationTypeResponse;
import software.amazon.awssdk.services.mturk.model.MTurkException;
import software.amazon.awssdk.services.mturk.model.MTurkRequest;
import software.amazon.awssdk.services.mturk.model.NotifyWorkersRequest;
import software.amazon.awssdk.services.mturk.model.NotifyWorkersResponse;
import software.amazon.awssdk.services.mturk.model.RejectAssignmentRequest;
import software.amazon.awssdk.services.mturk.model.RejectAssignmentResponse;
import software.amazon.awssdk.services.mturk.model.RejectQualificationRequestRequest;
import software.amazon.awssdk.services.mturk.model.RejectQualificationRequestResponse;
import software.amazon.awssdk.services.mturk.model.RequestErrorException;
import software.amazon.awssdk.services.mturk.model.SendBonusRequest;
import software.amazon.awssdk.services.mturk.model.SendBonusResponse;
import software.amazon.awssdk.services.mturk.model.SendTestEventNotificationRequest;
import software.amazon.awssdk.services.mturk.model.SendTestEventNotificationResponse;
import software.amazon.awssdk.services.mturk.model.ServiceException;
import software.amazon.awssdk.services.mturk.model.UpdateExpirationForHitRequest;
import software.amazon.awssdk.services.mturk.model.UpdateExpirationForHitResponse;
import software.amazon.awssdk.services.mturk.model.UpdateHitReviewStatusRequest;
import software.amazon.awssdk.services.mturk.model.UpdateHitReviewStatusResponse;
import software.amazon.awssdk.services.mturk.model.UpdateHitTypeOfHitRequest;
import software.amazon.awssdk.services.mturk.model.UpdateHitTypeOfHitResponse;
import software.amazon.awssdk.services.mturk.model.UpdateNotificationSettingsRequest;
import software.amazon.awssdk.services.mturk.model.UpdateNotificationSettingsResponse;
import software.amazon.awssdk.services.mturk.model.UpdateQualificationTypeRequest;
import software.amazon.awssdk.services.mturk.model.UpdateQualificationTypeResponse;
import software.amazon.awssdk.services.mturk.paginators.ListAssignmentsForHITIterable;
import software.amazon.awssdk.services.mturk.paginators.ListBonusPaymentsIterable;
import software.amazon.awssdk.services.mturk.paginators.ListHITsForQualificationTypeIterable;
import software.amazon.awssdk.services.mturk.paginators.ListHITsIterable;
import software.amazon.awssdk.services.mturk.paginators.ListQualificationRequestsIterable;
import software.amazon.awssdk.services.mturk.paginators.ListQualificationTypesIterable;
import software.amazon.awssdk.services.mturk.paginators.ListReviewPolicyResultsForHITIterable;
import software.amazon.awssdk.services.mturk.paginators.ListReviewableHITsIterable;
import software.amazon.awssdk.services.mturk.paginators.ListWorkerBlocksIterable;
import software.amazon.awssdk.services.mturk.paginators.ListWorkersWithQualificationTypeIterable;
import software.amazon.awssdk.services.mturk.transform.AcceptQualificationRequestRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ApproveAssignmentRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.AssociateQualificationWithWorkerRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.CreateAdditionalAssignmentsForHitRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.CreateHitRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.CreateHitTypeRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.CreateHitWithHitTypeRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.CreateQualificationTypeRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.CreateWorkerBlockRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.DeleteHitRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.DeleteQualificationTypeRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.DeleteWorkerBlockRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.DisassociateQualificationFromWorkerRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.GetAccountBalanceRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.GetAssignmentRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.GetFileUploadUrlRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.GetHitRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.GetQualificationScoreRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.GetQualificationTypeRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListAssignmentsForHitRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListBonusPaymentsRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListHiTsForQualificationTypeRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListHiTsRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListQualificationRequestsRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListQualificationTypesRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListReviewPolicyResultsForHitRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListReviewableHiTsRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListWorkerBlocksRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListWorkersWithQualificationTypeRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.NotifyWorkersRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.RejectAssignmentRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.RejectQualificationRequestRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.SendBonusRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.SendTestEventNotificationRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.UpdateExpirationForHitRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.UpdateHitReviewStatusRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.UpdateHitTypeOfHitRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.UpdateNotificationSettingsRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.UpdateQualificationTypeRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MTurkClient}.
 *
 * @see MTurkClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMTurkClient implements MTurkClient {
    private static final Logger log = Logger.loggerFor(DefaultMTurkClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMTurkClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * The <code>AcceptQualificationRequest</code> operation approves a Worker's request for a Qualification.
     * </p>
     * <p>
     * Only the owner of the Qualification type can grant a Qualification request for that type.
     * </p>
     * <p>
     * A successful request for the <code>AcceptQualificationRequest</code> operation returns with no errors and an
     * empty body.
     * </p>
     *
     * @param acceptQualificationRequestRequest
     * @return Result of the AcceptQualificationRequest operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.AcceptQualificationRequest
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/AcceptQualificationRequest"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AcceptQualificationRequestResponse acceptQualificationRequest(
            AcceptQualificationRequestRequest acceptQualificationRequestRequest) throws ServiceException, RequestErrorException,
            AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AcceptQualificationRequestResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AcceptQualificationRequestResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptQualificationRequestRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptQualificationRequest");

            return clientHandler
                    .execute(new ClientExecutionParams<AcceptQualificationRequestRequest, AcceptQualificationRequestResponse>()
                            .withOperationName("AcceptQualificationRequest").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(acceptQualificationRequestRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AcceptQualificationRequestRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>ApproveAssignment</code> operation approves the results of a completed assignment.
     * </p>
     * <p>
     * Approving an assignment initiates two payments from the Requester's Amazon.com account
     * </p>
     * <ul>
     * <li>
     * <p>
     * The Worker who submitted the results is paid the reward specified in the HIT.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Mechanical Turk fees are debited.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the Requester's account does not have adequate funds for these payments, the call to ApproveAssignment returns
     * an exception, and the approval is not processed. You can include an optional feedback message with the approval,
     * which the Worker can see in the Status section of the web site.
     * </p>
     * <p>
     * You can also call this operation for assignments that were previous rejected and approve them by explicitly
     * overriding the previous rejection. This only works on rejected assignments that were submitted within the
     * previous 30 days and only if the assignment's related HIT has not been deleted.
     * </p>
     *
     * @param approveAssignmentRequest
     * @return Result of the ApproveAssignment operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ApproveAssignment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ApproveAssignment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ApproveAssignmentResponse approveAssignment(ApproveAssignmentRequest approveAssignmentRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ApproveAssignmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ApproveAssignmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, approveAssignmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ApproveAssignment");

            return clientHandler.execute(new ClientExecutionParams<ApproveAssignmentRequest, ApproveAssignmentResponse>()
                    .withOperationName("ApproveAssignment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(approveAssignmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ApproveAssignmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>AssociateQualificationWithWorker</code> operation gives a Worker a Qualification.
     * <code>AssociateQualificationWithWorker</code> does not require that the Worker submit a Qualification request. It
     * gives the Qualification directly to the Worker.
     * </p>
     * <p>
     * You can only assign a Qualification of a Qualification type that you created (using the
     * <code>CreateQualificationType</code> operation).
     * </p>
     * <note>
     * <p>
     * Note: <code>AssociateQualificationWithWorker</code> does not affect any pending Qualification requests for the
     * Qualification by the Worker. If you assign a Qualification to a Worker, then later grant a Qualification request
     * made by the Worker, the granting of the request may modify the Qualification score. To resolve a pending
     * Qualification request without affecting the Qualification the Worker already has, reject the request with the
     * <code>RejectQualificationRequest</code> operation.
     * </p>
     * </note>
     *
     * @param associateQualificationWithWorkerRequest
     * @return Result of the AssociateQualificationWithWorker operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.AssociateQualificationWithWorker
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/AssociateQualificationWithWorker"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateQualificationWithWorkerResponse associateQualificationWithWorker(
            AssociateQualificationWithWorkerRequest associateQualificationWithWorkerRequest) throws ServiceException,
            RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateQualificationWithWorkerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateQualificationWithWorkerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateQualificationWithWorkerRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateQualificationWithWorker");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateQualificationWithWorkerRequest, AssociateQualificationWithWorkerResponse>()
                            .withOperationName("AssociateQualificationWithWorker").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(associateQualificationWithWorkerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateQualificationWithWorkerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>CreateAdditionalAssignmentsForHIT</code> operation increases the maximum number of assignments of an
     * existing HIT.
     * </p>
     * <p>
     * To extend the maximum number of assignments, specify the number of additional assignments.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * HITs created with fewer than 10 assignments cannot be extended to have 10 or more assignments. Attempting to add
     * assignments in a way that brings the total number of assignments for a HIT from fewer than 10 assignments to 10
     * or more assignments will result in an <code>AWS.MechanicalTurk.InvalidMaximumAssignmentsIncrease</code>
     * exception.
     * </p>
     * </li>
     * <li>
     * <p>
     * HITs that were created before July 22, 2015 cannot be extended. Attempting to extend HITs that were created
     * before July 22, 2015 will result in an <code>AWS.MechanicalTurk.HITTooOldForExtension</code> exception.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param createAdditionalAssignmentsForHitRequest
     * @return Result of the CreateAdditionalAssignmentsForHIT operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.CreateAdditionalAssignmentsForHIT
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/CreateAdditionalAssignmentsForHIT"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateAdditionalAssignmentsForHitResponse createAdditionalAssignmentsForHIT(
            CreateAdditionalAssignmentsForHitRequest createAdditionalAssignmentsForHitRequest) throws ServiceException,
            RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAdditionalAssignmentsForHitResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateAdditionalAssignmentsForHitResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createAdditionalAssignmentsForHitRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAdditionalAssignmentsForHIT");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateAdditionalAssignmentsForHitRequest, CreateAdditionalAssignmentsForHitResponse>()
                            .withOperationName("CreateAdditionalAssignmentsForHIT").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createAdditionalAssignmentsForHitRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateAdditionalAssignmentsForHitRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>CreateHIT</code> operation creates a new Human Intelligence Task (HIT). The new HIT is made available
     * for Workers to find and accept on the Amazon Mechanical Turk website.
     * </p>
     * <p>
     * This operation allows you to specify a new HIT by passing in values for the properties of the HIT, such as its
     * title, reward amount and number of assignments. When you pass these values to <code>CreateHIT</code>, a new HIT
     * is created for you, with a new <code>HITTypeID</code>. The HITTypeID can be used to create additional HITs in the
     * future without needing to specify common parameters such as the title, description and reward amount each time.
     * </p>
     * <p>
     * An alternative way to create HITs is to first generate a HITTypeID using the <code>CreateHITType</code> operation
     * and then call the <code>CreateHITWithHITType</code> operation. This is the recommended best practice for
     * Requesters who are creating large numbers of HITs.
     * </p>
     * <p>
     * CreateHIT also supports several ways to provide question data: by providing a value for the <code>Question</code>
     * parameter that fully specifies the contents of the HIT, or by providing a <code>HitLayoutId</code> and associated
     * <code>HitLayoutParameters</code>.
     * </p>
     * <note>
     * <p>
     * If a HIT is created with 10 or more maximum assignments, there is an additional fee. For more information, see <a
     * href="https://requester.mturk.com/pricing">Amazon Mechanical Turk Pricing</a>.
     * </p>
     * </note>
     *
     * @param createHitRequest
     * @return Result of the CreateHIT operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.CreateHIT
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/CreateHIT" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateHitResponse createHIT(CreateHitRequest createHitRequest) throws ServiceException, RequestErrorException,
            AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateHitResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateHitResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHIT");

            return clientHandler.execute(new ClientExecutionParams<CreateHitRequest, CreateHitResponse>()
                    .withOperationName("CreateHIT").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createHitRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new CreateHitRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>CreateHITType</code> operation creates a new HIT type. This operation allows you to define a standard
     * set of HIT properties to use when creating HITs. If you register a HIT type with values that match an existing
     * HIT type, the HIT type ID of the existing type will be returned.
     * </p>
     *
     * @param createHitTypeRequest
     * @return Result of the CreateHITType operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.CreateHITType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/CreateHITType" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateHitTypeResponse createHITType(CreateHitTypeRequest createHitTypeRequest) throws ServiceException,
            RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateHitTypeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateHitTypeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHitTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHITType");

            return clientHandler.execute(new ClientExecutionParams<CreateHitTypeRequest, CreateHitTypeResponse>()
                    .withOperationName("CreateHITType").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createHitTypeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateHitTypeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>CreateHITWithHITType</code> operation creates a new Human Intelligence Task (HIT) using an existing
     * HITTypeID generated by the <code>CreateHITType</code> operation.
     * </p>
     * <p>
     * This is an alternative way to create HITs from the <code>CreateHIT</code> operation. This is the recommended best
     * practice for Requesters who are creating large numbers of HITs.
     * </p>
     * <p>
     * CreateHITWithHITType also supports several ways to provide question data: by providing a value for the
     * <code>Question</code> parameter that fully specifies the contents of the HIT, or by providing a
     * <code>HitLayoutId</code> and associated <code>HitLayoutParameters</code>.
     * </p>
     * <note>
     * <p>
     * If a HIT is created with 10 or more maximum assignments, there is an additional fee. For more information, see <a
     * href="https://requester.mturk.com/pricing">Amazon Mechanical Turk Pricing</a>.
     * </p>
     * </note>
     *
     * @param createHitWithHitTypeRequest
     * @return Result of the CreateHITWithHITType operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.CreateHITWithHITType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/CreateHITWithHITType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateHitWithHitTypeResponse createHITWithHITType(CreateHitWithHitTypeRequest createHitWithHitTypeRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateHitWithHitTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateHitWithHitTypeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHitWithHitTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHITWithHITType");

            return clientHandler.execute(new ClientExecutionParams<CreateHitWithHitTypeRequest, CreateHitWithHitTypeResponse>()
                    .withOperationName("CreateHITWithHITType").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createHitWithHitTypeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateHitWithHitTypeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>CreateQualificationType</code> operation creates a new Qualification type, which is represented by a
     * <code>QualificationType</code> data structure.
     * </p>
     *
     * @param createQualificationTypeRequest
     * @return Result of the CreateQualificationType operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.CreateQualificationType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/CreateQualificationType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateQualificationTypeResponse createQualificationType(CreateQualificationTypeRequest createQualificationTypeRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateQualificationTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateQualificationTypeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createQualificationTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateQualificationType");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateQualificationTypeRequest, CreateQualificationTypeResponse>()
                            .withOperationName("CreateQualificationType").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createQualificationTypeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateQualificationTypeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>CreateWorkerBlock</code> operation allows you to prevent a Worker from working on your HITs. For
     * example, you can block a Worker who is producing poor quality work. You can block up to 100,000 Workers.
     * </p>
     *
     * @param createWorkerBlockRequest
     * @return Result of the CreateWorkerBlock operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.CreateWorkerBlock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/CreateWorkerBlock"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateWorkerBlockResponse createWorkerBlock(CreateWorkerBlockRequest createWorkerBlockRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateWorkerBlockResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateWorkerBlockResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkerBlockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkerBlock");

            return clientHandler.execute(new ClientExecutionParams<CreateWorkerBlockRequest, CreateWorkerBlockResponse>()
                    .withOperationName("CreateWorkerBlock").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createWorkerBlockRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateWorkerBlockRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>DeleteHIT</code> operation is used to delete HIT that is no longer needed. Only the Requester who
     * created the HIT can delete it.
     * </p>
     * <p>
     * You can only dispose of HITs that are in the <code>Reviewable</code> state, with all of their submitted
     * assignments already either approved or rejected. If you call the DeleteHIT operation on a HIT that is not in the
     * <code>Reviewable</code> state (for example, that has not expired, or still has active assignments), or on a HIT
     * that is Reviewable but without all of its submitted assignments already approved or rejected, the service will
     * return an error.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * HITs are automatically disposed of after 120 days.
     * </p>
     * </li>
     * <li>
     * <p>
     * After you dispose of a HIT, you can no longer approve the HIT's rejected assignments.
     * </p>
     * </li>
     * <li>
     * <p>
     * Disposed HITs are not returned in results for the ListHITs operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * Disposing HITs can improve the performance of operations such as ListReviewableHITs and ListHITs.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param deleteHitRequest
     * @return Result of the DeleteHIT operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.DeleteHIT
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/DeleteHIT" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteHitResponse deleteHIT(DeleteHitRequest deleteHitRequest) throws ServiceException, RequestErrorException,
            AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteHitResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteHitResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteHitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHIT");

            return clientHandler.execute(new ClientExecutionParams<DeleteHitRequest, DeleteHitResponse>()
                    .withOperationName("DeleteHIT").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteHitRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new DeleteHitRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>DeleteQualificationType</code> deletes a Qualification type and deletes any HIT types that are
     * associated with the Qualification type.
     * </p>
     * <p>
     * This operation does not revoke Qualifications already assigned to Workers because the Qualifications might be
     * needed for active HITs. If there are any pending requests for the Qualification type, Amazon Mechanical Turk
     * rejects those requests. After you delete a Qualification type, you can no longer use it to create HITs or HIT
     * types.
     * </p>
     * <note>
     * <p>
     * DeleteQualificationType must wait for all the HITs that use the deleted Qualification type to be deleted before
     * completing. It may take up to 48 hours before DeleteQualificationType completes and the unique name of the
     * Qualification type is available for reuse with CreateQualificationType.
     * </p>
     * </note>
     *
     * @param deleteQualificationTypeRequest
     * @return Result of the DeleteQualificationType operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.DeleteQualificationType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/DeleteQualificationType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteQualificationTypeResponse deleteQualificationType(DeleteQualificationTypeRequest deleteQualificationTypeRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteQualificationTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteQualificationTypeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteQualificationTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteQualificationType");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteQualificationTypeRequest, DeleteQualificationTypeResponse>()
                            .withOperationName("DeleteQualificationType").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteQualificationTypeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteQualificationTypeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>DeleteWorkerBlock</code> operation allows you to reinstate a blocked Worker to work on your HITs. This
     * operation reverses the effects of the CreateWorkerBlock operation. You need the Worker ID to use this operation.
     * If the Worker ID is missing or invalid, this operation fails and returns the message “WorkerId is invalid.” If
     * the specified Worker is not blocked, this operation returns successfully.
     * </p>
     *
     * @param deleteWorkerBlockRequest
     * @return Result of the DeleteWorkerBlock operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.DeleteWorkerBlock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/DeleteWorkerBlock"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteWorkerBlockResponse deleteWorkerBlock(DeleteWorkerBlockRequest deleteWorkerBlockRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteWorkerBlockResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteWorkerBlockResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkerBlockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkerBlock");

            return clientHandler.execute(new ClientExecutionParams<DeleteWorkerBlockRequest, DeleteWorkerBlockResponse>()
                    .withOperationName("DeleteWorkerBlock").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteWorkerBlockRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteWorkerBlockRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>DisassociateQualificationFromWorker</code> revokes a previously granted Qualification from a user.
     * </p>
     * <p>
     * You can provide a text message explaining why the Qualification was revoked. The user who had the Qualification
     * can see this message.
     * </p>
     *
     * @param disassociateQualificationFromWorkerRequest
     * @return Result of the DisassociateQualificationFromWorker operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.DisassociateQualificationFromWorker
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/DisassociateQualificationFromWorker"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateQualificationFromWorkerResponse disassociateQualificationFromWorker(
            DisassociateQualificationFromWorkerRequest disassociateQualificationFromWorkerRequest) throws ServiceException,
            RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateQualificationFromWorkerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateQualificationFromWorkerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateQualificationFromWorkerRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateQualificationFromWorker");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateQualificationFromWorkerRequest, DisassociateQualificationFromWorkerResponse>()
                            .withOperationName("DisassociateQualificationFromWorker").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateQualificationFromWorkerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateQualificationFromWorkerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>GetAccountBalance</code> operation retrieves the amount of money in your Amazon Mechanical Turk
     * account.
     * </p>
     *
     * @param getAccountBalanceRequest
     * @return Result of the GetAccountBalance operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.GetAccountBalance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/GetAccountBalance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAccountBalanceResponse getAccountBalance(GetAccountBalanceRequest getAccountBalanceRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAccountBalanceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAccountBalanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccountBalanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccountBalance");

            return clientHandler.execute(new ClientExecutionParams<GetAccountBalanceRequest, GetAccountBalanceResponse>()
                    .withOperationName("GetAccountBalance").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getAccountBalanceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAccountBalanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>GetAssignment</code> operation retrieves the details of the specified Assignment.
     * </p>
     *
     * @param getAssignmentRequest
     * @return Result of the GetAssignment operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.GetAssignment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/GetAssignment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetAssignmentResponse getAssignment(GetAssignmentRequest getAssignmentRequest) throws ServiceException,
            RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAssignmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAssignmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAssignmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAssignment");

            return clientHandler.execute(new ClientExecutionParams<GetAssignmentRequest, GetAssignmentResponse>()
                    .withOperationName("GetAssignment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getAssignmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAssignmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>GetFileUploadURL</code> operation generates and returns a temporary URL. You use the temporary URL to
     * retrieve a file uploaded by a Worker as an answer to a FileUploadAnswer question for a HIT. The temporary URL is
     * generated the instant the GetFileUploadURL operation is called, and is valid for 60 seconds. You can get a
     * temporary file upload URL any time until the HIT is disposed. After the HIT is disposed, any uploaded files are
     * deleted, and cannot be retrieved. Pending Deprecation on December 12, 2017. The Answer Specification structure
     * will no longer support the <code>FileUploadAnswer</code> element to be used for the QuestionForm data structure.
     * Instead, we recommend that Requesters who want to create HITs asking Workers to upload files to use Amazon S3.
     * </p>
     *
     * @param getFileUploadUrlRequest
     * @return Result of the GetFileUploadURL operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.GetFileUploadURL
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/GetFileUploadURL"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetFileUploadUrlResponse getFileUploadURL(GetFileUploadUrlRequest getFileUploadUrlRequest) throws ServiceException,
            RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetFileUploadUrlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetFileUploadUrlResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFileUploadUrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFileUploadURL");

            return clientHandler.execute(new ClientExecutionParams<GetFileUploadUrlRequest, GetFileUploadUrlResponse>()
                    .withOperationName("GetFileUploadURL").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getFileUploadUrlRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetFileUploadUrlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>GetHIT</code> operation retrieves the details of the specified HIT.
     * </p>
     *
     * @param getHitRequest
     * @return Result of the GetHIT operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.GetHIT
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/GetHIT" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetHitResponse getHIT(GetHitRequest getHitRequest) throws ServiceException, RequestErrorException,
            AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetHitResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetHitResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getHitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetHIT");

            return clientHandler.execute(new ClientExecutionParams<GetHitRequest, GetHitResponse>().withOperationName("GetHIT")
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(getHitRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetHitRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>GetQualificationScore</code> operation returns the value of a Worker's Qualification for a given
     * Qualification type.
     * </p>
     * <p>
     * To get a Worker's Qualification, you must know the Worker's ID. The Worker's ID is included in the assignment
     * data returned by the <code>ListAssignmentsForHIT</code> operation.
     * </p>
     * <p>
     * Only the owner of a Qualification type can query the value of a Worker's Qualification of that type.
     * </p>
     *
     * @param getQualificationScoreRequest
     * @return Result of the GetQualificationScore operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.GetQualificationScore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/GetQualificationScore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetQualificationScoreResponse getQualificationScore(GetQualificationScoreRequest getQualificationScoreRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetQualificationScoreResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetQualificationScoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQualificationScoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQualificationScore");

            return clientHandler.execute(new ClientExecutionParams<GetQualificationScoreRequest, GetQualificationScoreResponse>()
                    .withOperationName("GetQualificationScore").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getQualificationScoreRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetQualificationScoreRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>GetQualificationType</code>operation retrieves information about a Qualification type using its ID.
     * </p>
     *
     * @param getQualificationTypeRequest
     * @return Result of the GetQualificationType operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.GetQualificationType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/GetQualificationType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetQualificationTypeResponse getQualificationType(GetQualificationTypeRequest getQualificationTypeRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetQualificationTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetQualificationTypeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQualificationTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQualificationType");

            return clientHandler.execute(new ClientExecutionParams<GetQualificationTypeRequest, GetQualificationTypeResponse>()
                    .withOperationName("GetQualificationType").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getQualificationTypeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetQualificationTypeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>ListAssignmentsForHIT</code> operation retrieves completed assignments for a HIT. You can use this
     * operation to retrieve the results for a HIT.
     * </p>
     * <p>
     * You can get assignments for a HIT at any time, even if the HIT is not yet Reviewable. If a HIT requested multiple
     * assignments, and has received some results but has not yet become Reviewable, you can still retrieve the partial
     * results with this operation.
     * </p>
     * <p>
     * Use the AssignmentStatus parameter to control which set of assignments for a HIT are returned. The
     * ListAssignmentsForHIT operation can return submitted assignments awaiting approval, or it can return assignments
     * that have already been approved or rejected. You can set AssignmentStatus=Approved,Rejected to get assignments
     * that have already been approved and rejected together in one result set.
     * </p>
     * <p>
     * Only the Requester who created the HIT can retrieve the assignments for that HIT.
     * </p>
     * <p>
     * Results are sorted and divided into numbered pages and the operation returns a single page of results. You can
     * use the parameters of the operation to control sorting and pagination.
     * </p>
     *
     * @param listAssignmentsForHitRequest
     * @return Result of the ListAssignmentsForHIT operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListAssignmentsForHIT
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListAssignmentsForHIT"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAssignmentsForHitResponse listAssignmentsForHIT(ListAssignmentsForHitRequest listAssignmentsForHitRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAssignmentsForHitResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAssignmentsForHitResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAssignmentsForHitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssignmentsForHIT");

            return clientHandler.execute(new ClientExecutionParams<ListAssignmentsForHitRequest, ListAssignmentsForHitResponse>()
                    .withOperationName("ListAssignmentsForHIT").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAssignmentsForHitRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAssignmentsForHitRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>ListAssignmentsForHIT</code> operation retrieves completed assignments for a HIT. You can use this
     * operation to retrieve the results for a HIT.
     * </p>
     * <p>
     * You can get assignments for a HIT at any time, even if the HIT is not yet Reviewable. If a HIT requested multiple
     * assignments, and has received some results but has not yet become Reviewable, you can still retrieve the partial
     * results with this operation.
     * </p>
     * <p>
     * Use the AssignmentStatus parameter to control which set of assignments for a HIT are returned. The
     * ListAssignmentsForHIT operation can return submitted assignments awaiting approval, or it can return assignments
     * that have already been approved or rejected. You can set AssignmentStatus=Approved,Rejected to get assignments
     * that have already been approved and rejected together in one result set.
     * </p>
     * <p>
     * Only the Requester who created the HIT can retrieve the assignments for that HIT.
     * </p>
     * <p>
     * Results are sorted and divided into numbered pages and the operation returns a single page of results. You can
     * use the parameters of the operation to control sorting and pagination.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAssignmentsForHIT(software.amazon.awssdk.services.mturk.model.ListAssignmentsForHitRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListAssignmentsForHITIterable responses = client.listAssignmentsForHITPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mturk.paginators.ListAssignmentsForHITIterable responses = client
     *             .listAssignmentsForHITPaginator(request);
     *     for (software.amazon.awssdk.services.mturk.model.ListAssignmentsForHitResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListAssignmentsForHITIterable responses = client.listAssignmentsForHITPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAssignmentsForHIT(software.amazon.awssdk.services.mturk.model.ListAssignmentsForHitRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAssignmentsForHitRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListAssignmentsForHIT
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListAssignmentsForHIT"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAssignmentsForHITIterable listAssignmentsForHITPaginator(ListAssignmentsForHitRequest listAssignmentsForHitRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        return new ListAssignmentsForHITIterable(this, applyPaginatorUserAgent(listAssignmentsForHitRequest));
    }

    /**
     * <p>
     * The <code>ListBonusPayments</code> operation retrieves the amounts of bonuses you have paid to Workers for a
     * given HIT or assignment.
     * </p>
     *
     * @param listBonusPaymentsRequest
     * @return Result of the ListBonusPayments operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListBonusPayments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListBonusPayments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListBonusPaymentsResponse listBonusPayments(ListBonusPaymentsRequest listBonusPaymentsRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBonusPaymentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListBonusPaymentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBonusPaymentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBonusPayments");

            return clientHandler.execute(new ClientExecutionParams<ListBonusPaymentsRequest, ListBonusPaymentsResponse>()
                    .withOperationName("ListBonusPayments").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listBonusPaymentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListBonusPaymentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>ListBonusPayments</code> operation retrieves the amounts of bonuses you have paid to Workers for a
     * given HIT or assignment.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listBonusPayments(software.amazon.awssdk.services.mturk.model.ListBonusPaymentsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListBonusPaymentsIterable responses = client.listBonusPaymentsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mturk.paginators.ListBonusPaymentsIterable responses = client
     *             .listBonusPaymentsPaginator(request);
     *     for (software.amazon.awssdk.services.mturk.model.ListBonusPaymentsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListBonusPaymentsIterable responses = client.listBonusPaymentsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listBonusPayments(software.amazon.awssdk.services.mturk.model.ListBonusPaymentsRequest)} operation.</b>
     * </p>
     *
     * @param listBonusPaymentsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListBonusPayments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListBonusPayments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListBonusPaymentsIterable listBonusPaymentsPaginator(ListBonusPaymentsRequest listBonusPaymentsRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        return new ListBonusPaymentsIterable(this, applyPaginatorUserAgent(listBonusPaymentsRequest));
    }

    /**
     * <p>
     * The <code>ListHITs</code> operation returns all of a Requester's HITs. The operation returns HITs of any status,
     * except for HITs that have been deleted of with the DeleteHIT operation or that have been auto-deleted.
     * </p>
     *
     * @param listHiTsRequest
     * @return Result of the ListHITs operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListHITs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListHITs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListHiTsResponse listHITs(ListHiTsRequest listHiTsRequest) throws ServiceException, RequestErrorException,
            AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListHiTsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListHiTsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listHiTsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHITs");

            return clientHandler.execute(new ClientExecutionParams<ListHiTsRequest, ListHiTsResponse>()
                    .withOperationName("ListHITs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listHiTsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListHiTsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>ListHITs</code> operation returns all of a Requester's HITs. The operation returns HITs of any status,
     * except for HITs that have been deleted of with the DeleteHIT operation or that have been auto-deleted.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listHITs(software.amazon.awssdk.services.mturk.model.ListHiTsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListHITsIterable responses = client.listHITsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mturk.paginators.ListHITsIterable responses = client.listHITsPaginator(request);
     *     for (software.amazon.awssdk.services.mturk.model.ListHiTsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListHITsIterable responses = client.listHITsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listHITs(software.amazon.awssdk.services.mturk.model.ListHiTsRequest)} operation.</b>
     * </p>
     *
     * @param listHiTsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListHITs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListHITs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListHITsIterable listHITsPaginator(ListHiTsRequest listHiTsRequest) throws ServiceException, RequestErrorException,
            AwsServiceException, SdkClientException, MTurkException {
        return new ListHITsIterable(this, applyPaginatorUserAgent(listHiTsRequest));
    }

    /**
     * <p>
     * The <code>ListHITsForQualificationType</code> operation returns the HITs that use the given Qualification type
     * for a Qualification requirement. The operation returns HITs of any status, except for HITs that have been deleted
     * with the <code>DeleteHIT</code> operation or that have been auto-deleted.
     * </p>
     *
     * @param listHiTsForQualificationTypeRequest
     * @return Result of the ListHITsForQualificationType operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListHITsForQualificationType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListHITsForQualificationType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListHiTsForQualificationTypeResponse listHITsForQualificationType(
            ListHiTsForQualificationTypeRequest listHiTsForQualificationTypeRequest) throws ServiceException,
            RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListHiTsForQualificationTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListHiTsForQualificationTypeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listHiTsForQualificationTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHITsForQualificationType");

            return clientHandler
                    .execute(new ClientExecutionParams<ListHiTsForQualificationTypeRequest, ListHiTsForQualificationTypeResponse>()
                            .withOperationName("ListHITsForQualificationType").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listHiTsForQualificationTypeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListHiTsForQualificationTypeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>ListHITsForQualificationType</code> operation returns the HITs that use the given Qualification type
     * for a Qualification requirement. The operation returns HITs of any status, except for HITs that have been deleted
     * with the <code>DeleteHIT</code> operation or that have been auto-deleted.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listHITsForQualificationType(software.amazon.awssdk.services.mturk.model.ListHiTsForQualificationTypeRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListHITsForQualificationTypeIterable responses = client.listHITsForQualificationTypePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mturk.paginators.ListHITsForQualificationTypeIterable responses = client
     *             .listHITsForQualificationTypePaginator(request);
     *     for (software.amazon.awssdk.services.mturk.model.ListHiTsForQualificationTypeResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListHITsForQualificationTypeIterable responses = client.listHITsForQualificationTypePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listHITsForQualificationType(software.amazon.awssdk.services.mturk.model.ListHiTsForQualificationTypeRequest)}
     * operation.</b>
     * </p>
     *
     * @param listHiTsForQualificationTypeRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListHITsForQualificationType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListHITsForQualificationType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListHITsForQualificationTypeIterable listHITsForQualificationTypePaginator(
            ListHiTsForQualificationTypeRequest listHiTsForQualificationTypeRequest) throws ServiceException,
            RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        return new ListHITsForQualificationTypeIterable(this, applyPaginatorUserAgent(listHiTsForQualificationTypeRequest));
    }

    /**
     * <p>
     * The <code>ListQualificationRequests</code> operation retrieves requests for Qualifications of a particular
     * Qualification type. The owner of the Qualification type calls this operation to poll for pending requests, and
     * accepts them using the AcceptQualification operation.
     * </p>
     *
     * @param listQualificationRequestsRequest
     * @return Result of the ListQualificationRequests operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListQualificationRequests
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListQualificationRequests"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListQualificationRequestsResponse listQualificationRequests(
            ListQualificationRequestsRequest listQualificationRequestsRequest) throws ServiceException, RequestErrorException,
            AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListQualificationRequestsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListQualificationRequestsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listQualificationRequestsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListQualificationRequests");

            return clientHandler
                    .execute(new ClientExecutionParams<ListQualificationRequestsRequest, ListQualificationRequestsResponse>()
                            .withOperationName("ListQualificationRequests").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listQualificationRequestsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListQualificationRequestsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>ListQualificationRequests</code> operation retrieves requests for Qualifications of a particular
     * Qualification type. The owner of the Qualification type calls this operation to poll for pending requests, and
     * accepts them using the AcceptQualification operation.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listQualificationRequests(software.amazon.awssdk.services.mturk.model.ListQualificationRequestsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListQualificationRequestsIterable responses = client.listQualificationRequestsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mturk.paginators.ListQualificationRequestsIterable responses = client
     *             .listQualificationRequestsPaginator(request);
     *     for (software.amazon.awssdk.services.mturk.model.ListQualificationRequestsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListQualificationRequestsIterable responses = client.listQualificationRequestsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listQualificationRequests(software.amazon.awssdk.services.mturk.model.ListQualificationRequestsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listQualificationRequestsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListQualificationRequests
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListQualificationRequests"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListQualificationRequestsIterable listQualificationRequestsPaginator(
            ListQualificationRequestsRequest listQualificationRequestsRequest) throws ServiceException, RequestErrorException,
            AwsServiceException, SdkClientException, MTurkException {
        return new ListQualificationRequestsIterable(this, applyPaginatorUserAgent(listQualificationRequestsRequest));
    }

    /**
     * <p>
     * The <code>ListQualificationTypes</code> operation returns a list of Qualification types, filtered by an optional
     * search term.
     * </p>
     *
     * @param listQualificationTypesRequest
     * @return Result of the ListQualificationTypes operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListQualificationTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListQualificationTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListQualificationTypesResponse listQualificationTypes(ListQualificationTypesRequest listQualificationTypesRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListQualificationTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListQualificationTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listQualificationTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListQualificationTypes");

            return clientHandler
                    .execute(new ClientExecutionParams<ListQualificationTypesRequest, ListQualificationTypesResponse>()
                            .withOperationName("ListQualificationTypes").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listQualificationTypesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListQualificationTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>ListQualificationTypes</code> operation returns a list of Qualification types, filtered by an optional
     * search term.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listQualificationTypes(software.amazon.awssdk.services.mturk.model.ListQualificationTypesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListQualificationTypesIterable responses = client.listQualificationTypesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mturk.paginators.ListQualificationTypesIterable responses = client
     *             .listQualificationTypesPaginator(request);
     *     for (software.amazon.awssdk.services.mturk.model.ListQualificationTypesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListQualificationTypesIterable responses = client.listQualificationTypesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listQualificationTypes(software.amazon.awssdk.services.mturk.model.ListQualificationTypesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listQualificationTypesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListQualificationTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListQualificationTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListQualificationTypesIterable listQualificationTypesPaginator(
            ListQualificationTypesRequest listQualificationTypesRequest) throws ServiceException, RequestErrorException,
            AwsServiceException, SdkClientException, MTurkException {
        return new ListQualificationTypesIterable(this, applyPaginatorUserAgent(listQualificationTypesRequest));
    }

    /**
     * <p>
     * The <code>ListReviewPolicyResultsForHIT</code> operation retrieves the computed results and the actions taken in
     * the course of executing your Review Policies for a given HIT. For information about how to specify Review
     * Policies when you call CreateHIT, see Review Policies. The ListReviewPolicyResultsForHIT operation can return
     * results for both Assignment-level and HIT-level review results.
     * </p>
     *
     * @param listReviewPolicyResultsForHitRequest
     * @return Result of the ListReviewPolicyResultsForHIT operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListReviewPolicyResultsForHIT
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListReviewPolicyResultsForHIT"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListReviewPolicyResultsForHitResponse listReviewPolicyResultsForHIT(
            ListReviewPolicyResultsForHitRequest listReviewPolicyResultsForHitRequest) throws ServiceException,
            RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListReviewPolicyResultsForHitResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListReviewPolicyResultsForHitResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listReviewPolicyResultsForHitRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReviewPolicyResultsForHIT");

            return clientHandler
                    .execute(new ClientExecutionParams<ListReviewPolicyResultsForHitRequest, ListReviewPolicyResultsForHitResponse>()
                            .withOperationName("ListReviewPolicyResultsForHIT").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listReviewPolicyResultsForHitRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListReviewPolicyResultsForHitRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>ListReviewPolicyResultsForHIT</code> operation retrieves the computed results and the actions taken in
     * the course of executing your Review Policies for a given HIT. For information about how to specify Review
     * Policies when you call CreateHIT, see Review Policies. The ListReviewPolicyResultsForHIT operation can return
     * results for both Assignment-level and HIT-level review results.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listReviewPolicyResultsForHIT(software.amazon.awssdk.services.mturk.model.ListReviewPolicyResultsForHitRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListReviewPolicyResultsForHITIterable responses = client.listReviewPolicyResultsForHITPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mturk.paginators.ListReviewPolicyResultsForHITIterable responses = client
     *             .listReviewPolicyResultsForHITPaginator(request);
     *     for (software.amazon.awssdk.services.mturk.model.ListReviewPolicyResultsForHitResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListReviewPolicyResultsForHITIterable responses = client.listReviewPolicyResultsForHITPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listReviewPolicyResultsForHIT(software.amazon.awssdk.services.mturk.model.ListReviewPolicyResultsForHitRequest)}
     * operation.</b>
     * </p>
     *
     * @param listReviewPolicyResultsForHitRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListReviewPolicyResultsForHIT
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListReviewPolicyResultsForHIT"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListReviewPolicyResultsForHITIterable listReviewPolicyResultsForHITPaginator(
            ListReviewPolicyResultsForHitRequest listReviewPolicyResultsForHitRequest) throws ServiceException,
            RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        return new ListReviewPolicyResultsForHITIterable(this, applyPaginatorUserAgent(listReviewPolicyResultsForHitRequest));
    }

    /**
     * <p>
     * The <code>ListReviewableHITs</code> operation retrieves the HITs with Status equal to Reviewable or Status equal
     * to Reviewing that belong to the Requester calling the operation.
     * </p>
     *
     * @param listReviewableHiTsRequest
     * @return Result of the ListReviewableHITs operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListReviewableHITs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListReviewableHITs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListReviewableHiTsResponse listReviewableHITs(ListReviewableHiTsRequest listReviewableHiTsRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListReviewableHiTsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListReviewableHiTsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listReviewableHiTsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReviewableHITs");

            return clientHandler.execute(new ClientExecutionParams<ListReviewableHiTsRequest, ListReviewableHiTsResponse>()
                    .withOperationName("ListReviewableHITs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listReviewableHiTsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListReviewableHiTsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>ListReviewableHITs</code> operation retrieves the HITs with Status equal to Reviewable or Status equal
     * to Reviewing that belong to the Requester calling the operation.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listReviewableHITs(software.amazon.awssdk.services.mturk.model.ListReviewableHiTsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListReviewableHITsIterable responses = client.listReviewableHITsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mturk.paginators.ListReviewableHITsIterable responses = client
     *             .listReviewableHITsPaginator(request);
     *     for (software.amazon.awssdk.services.mturk.model.ListReviewableHiTsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListReviewableHITsIterable responses = client.listReviewableHITsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listReviewableHITs(software.amazon.awssdk.services.mturk.model.ListReviewableHiTsRequest)} operation.</b>
     * </p>
     *
     * @param listReviewableHiTsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListReviewableHITs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListReviewableHITs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListReviewableHITsIterable listReviewableHITsPaginator(ListReviewableHiTsRequest listReviewableHiTsRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        return new ListReviewableHITsIterable(this, applyPaginatorUserAgent(listReviewableHiTsRequest));
    }

    /**
     * <p>
     * The <code>ListWorkersBlocks</code> operation retrieves a list of Workers who are blocked from working on your
     * HITs.
     * </p>
     *
     * @param listWorkerBlocksRequest
     * @return Result of the ListWorkerBlocks operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListWorkerBlocks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListWorkerBlocks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListWorkerBlocksResponse listWorkerBlocks(ListWorkerBlocksRequest listWorkerBlocksRequest) throws ServiceException,
            RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListWorkerBlocksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListWorkerBlocksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkerBlocksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkerBlocks");

            return clientHandler.execute(new ClientExecutionParams<ListWorkerBlocksRequest, ListWorkerBlocksResponse>()
                    .withOperationName("ListWorkerBlocks").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listWorkerBlocksRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListWorkerBlocksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>ListWorkersBlocks</code> operation retrieves a list of Workers who are blocked from working on your
     * HITs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listWorkerBlocks(software.amazon.awssdk.services.mturk.model.ListWorkerBlocksRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListWorkerBlocksIterable responses = client.listWorkerBlocksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mturk.paginators.ListWorkerBlocksIterable responses = client
     *             .listWorkerBlocksPaginator(request);
     *     for (software.amazon.awssdk.services.mturk.model.ListWorkerBlocksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListWorkerBlocksIterable responses = client.listWorkerBlocksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWorkerBlocks(software.amazon.awssdk.services.mturk.model.ListWorkerBlocksRequest)} operation.</b>
     * </p>
     *
     * @param listWorkerBlocksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListWorkerBlocks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListWorkerBlocks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListWorkerBlocksIterable listWorkerBlocksPaginator(ListWorkerBlocksRequest listWorkerBlocksRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        return new ListWorkerBlocksIterable(this, applyPaginatorUserAgent(listWorkerBlocksRequest));
    }

    /**
     * <p>
     * The <code>ListWorkersWithQualificationType</code> operation returns all of the Workers that have been associated
     * with a given Qualification type.
     * </p>
     *
     * @param listWorkersWithQualificationTypeRequest
     * @return Result of the ListWorkersWithQualificationType operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListWorkersWithQualificationType
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListWorkersWithQualificationType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListWorkersWithQualificationTypeResponse listWorkersWithQualificationType(
            ListWorkersWithQualificationTypeRequest listWorkersWithQualificationTypeRequest) throws ServiceException,
            RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListWorkersWithQualificationTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListWorkersWithQualificationTypeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listWorkersWithQualificationTypeRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkersWithQualificationType");

            return clientHandler
                    .execute(new ClientExecutionParams<ListWorkersWithQualificationTypeRequest, ListWorkersWithQualificationTypeResponse>()
                            .withOperationName("ListWorkersWithQualificationType").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listWorkersWithQualificationTypeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListWorkersWithQualificationTypeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>ListWorkersWithQualificationType</code> operation returns all of the Workers that have been associated
     * with a given Qualification type.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listWorkersWithQualificationType(software.amazon.awssdk.services.mturk.model.ListWorkersWithQualificationTypeRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListWorkersWithQualificationTypeIterable responses = client.listWorkersWithQualificationTypePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mturk.paginators.ListWorkersWithQualificationTypeIterable responses = client
     *             .listWorkersWithQualificationTypePaginator(request);
     *     for (software.amazon.awssdk.services.mturk.model.ListWorkersWithQualificationTypeResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mturk.paginators.ListWorkersWithQualificationTypeIterable responses = client.listWorkersWithQualificationTypePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWorkersWithQualificationType(software.amazon.awssdk.services.mturk.model.ListWorkersWithQualificationTypeRequest)}
     * operation.</b>
     * </p>
     *
     * @param listWorkersWithQualificationTypeRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.ListWorkersWithQualificationType
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListWorkersWithQualificationType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListWorkersWithQualificationTypeIterable listWorkersWithQualificationTypePaginator(
            ListWorkersWithQualificationTypeRequest listWorkersWithQualificationTypeRequest) throws ServiceException,
            RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        return new ListWorkersWithQualificationTypeIterable(this,
                applyPaginatorUserAgent(listWorkersWithQualificationTypeRequest));
    }

    /**
     * <p>
     * The <code>NotifyWorkers</code> operation sends an email to one or more Workers that you specify with the Worker
     * ID. You can specify up to 100 Worker IDs to send the same message with a single call to the NotifyWorkers
     * operation. The NotifyWorkers operation will send a notification email to a Worker only if you have previously
     * approved or rejected work from the Worker.
     * </p>
     *
     * @param notifyWorkersRequest
     * @return Result of the NotifyWorkers operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.NotifyWorkers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/NotifyWorkers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public NotifyWorkersResponse notifyWorkers(NotifyWorkersRequest notifyWorkersRequest) throws ServiceException,
            RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<NotifyWorkersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                NotifyWorkersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, notifyWorkersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "NotifyWorkers");

            return clientHandler.execute(new ClientExecutionParams<NotifyWorkersRequest, NotifyWorkersResponse>()
                    .withOperationName("NotifyWorkers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(notifyWorkersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new NotifyWorkersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>RejectAssignment</code> operation rejects the results of a completed assignment.
     * </p>
     * <p>
     * You can include an optional feedback message with the rejection, which the Worker can see in the Status section
     * of the web site. When you include a feedback message with the rejection, it helps the Worker understand why the
     * assignment was rejected, and can improve the quality of the results the Worker submits in the future.
     * </p>
     * <p>
     * Only the Requester who created the HIT can reject an assignment for the HIT.
     * </p>
     *
     * @param rejectAssignmentRequest
     * @return Result of the RejectAssignment operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.RejectAssignment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/RejectAssignment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RejectAssignmentResponse rejectAssignment(RejectAssignmentRequest rejectAssignmentRequest) throws ServiceException,
            RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RejectAssignmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RejectAssignmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rejectAssignmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectAssignment");

            return clientHandler.execute(new ClientExecutionParams<RejectAssignmentRequest, RejectAssignmentResponse>()
                    .withOperationName("RejectAssignment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(rejectAssignmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RejectAssignmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>RejectQualificationRequest</code> operation rejects a user's request for a Qualification.
     * </p>
     * <p>
     * You can provide a text message explaining why the request was rejected. The Worker who made the request can see
     * this message.
     * </p>
     *
     * @param rejectQualificationRequestRequest
     * @return Result of the RejectQualificationRequest operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.RejectQualificationRequest
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/RejectQualificationRequest"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RejectQualificationRequestResponse rejectQualificationRequest(
            RejectQualificationRequestRequest rejectQualificationRequestRequest) throws ServiceException, RequestErrorException,
            AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RejectQualificationRequestResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RejectQualificationRequestResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rejectQualificationRequestRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectQualificationRequest");

            return clientHandler
                    .execute(new ClientExecutionParams<RejectQualificationRequestRequest, RejectQualificationRequestResponse>()
                            .withOperationName("RejectQualificationRequest").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(rejectQualificationRequestRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RejectQualificationRequestRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>SendBonus</code> operation issues a payment of money from your account to a Worker. This payment
     * happens separately from the reward you pay to the Worker when you approve the Worker's assignment. The SendBonus
     * operation requires the Worker's ID and the assignment ID as parameters to initiate payment of the bonus. You must
     * include a message that explains the reason for the bonus payment, as the Worker may not be expecting the payment.
     * Amazon Mechanical Turk collects a fee for bonus payments, similar to the HIT listing fee. This operation fails if
     * your account does not have enough funds to pay for both the bonus and the fees.
     * </p>
     *
     * @param sendBonusRequest
     * @return Result of the SendBonus operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.SendBonus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/SendBonus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SendBonusResponse sendBonus(SendBonusRequest sendBonusRequest) throws ServiceException, RequestErrorException,
            AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SendBonusResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SendBonusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendBonusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendBonus");

            return clientHandler.execute(new ClientExecutionParams<SendBonusRequest, SendBonusResponse>()
                    .withOperationName("SendBonus").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(sendBonusRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new SendBonusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>SendTestEventNotification</code> operation causes Amazon Mechanical Turk to send a notification message
     * as if a HIT event occurred, according to the provided notification specification. This allows you to test
     * notifications without setting up notifications for a real HIT type and trying to trigger them using the website.
     * When you call this operation, the service attempts to send the test notification immediately.
     * </p>
     *
     * @param sendTestEventNotificationRequest
     * @return Result of the SendTestEventNotification operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.SendTestEventNotification
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/SendTestEventNotification"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SendTestEventNotificationResponse sendTestEventNotification(
            SendTestEventNotificationRequest sendTestEventNotificationRequest) throws ServiceException, RequestErrorException,
            AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SendTestEventNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SendTestEventNotificationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendTestEventNotificationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendTestEventNotification");

            return clientHandler
                    .execute(new ClientExecutionParams<SendTestEventNotificationRequest, SendTestEventNotificationResponse>()
                            .withOperationName("SendTestEventNotification").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(sendTestEventNotificationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SendTestEventNotificationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>UpdateExpirationForHIT</code> operation allows you update the expiration time of a HIT. If you update
     * it to a time in the past, the HIT will be immediately expired.
     * </p>
     *
     * @param updateExpirationForHitRequest
     * @return Result of the UpdateExpirationForHIT operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.UpdateExpirationForHIT
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/UpdateExpirationForHIT"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateExpirationForHitResponse updateExpirationForHIT(UpdateExpirationForHitRequest updateExpirationForHitRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateExpirationForHitResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateExpirationForHitResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateExpirationForHitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateExpirationForHIT");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateExpirationForHitRequest, UpdateExpirationForHitResponse>()
                            .withOperationName("UpdateExpirationForHIT").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateExpirationForHitRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateExpirationForHitRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>UpdateHITReviewStatus</code> operation updates the status of a HIT. If the status is Reviewable, this
     * operation can update the status to Reviewing, or it can revert a Reviewing HIT back to the Reviewable status.
     * </p>
     *
     * @param updateHitReviewStatusRequest
     * @return Result of the UpdateHITReviewStatus operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.UpdateHITReviewStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/UpdateHITReviewStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateHitReviewStatusResponse updateHITReviewStatus(UpdateHitReviewStatusRequest updateHitReviewStatusRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateHitReviewStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateHitReviewStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateHitReviewStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateHITReviewStatus");

            return clientHandler.execute(new ClientExecutionParams<UpdateHitReviewStatusRequest, UpdateHitReviewStatusResponse>()
                    .withOperationName("UpdateHITReviewStatus").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateHitReviewStatusRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateHitReviewStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>UpdateHITTypeOfHIT</code> operation allows you to change the HITType properties of a HIT. This
     * operation disassociates the HIT from its old HITType properties and associates it with the new HITType
     * properties. The HIT takes on the properties of the new HITType in place of the old ones.
     * </p>
     *
     * @param updateHitTypeOfHitRequest
     * @return Result of the UpdateHITTypeOfHIT operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.UpdateHITTypeOfHIT
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/UpdateHITTypeOfHIT"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateHitTypeOfHitResponse updateHITTypeOfHIT(UpdateHitTypeOfHitRequest updateHitTypeOfHitRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateHitTypeOfHitResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateHitTypeOfHitResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateHitTypeOfHitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateHITTypeOfHIT");

            return clientHandler.execute(new ClientExecutionParams<UpdateHitTypeOfHitRequest, UpdateHitTypeOfHitResponse>()
                    .withOperationName("UpdateHITTypeOfHIT").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateHitTypeOfHitRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateHitTypeOfHitRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>UpdateNotificationSettings</code> operation creates, updates, disables or re-enables notifications for
     * a HIT type. If you call the UpdateNotificationSettings operation for a HIT type that already has a notification
     * specification, the operation replaces the old specification with a new one. You can call the
     * UpdateNotificationSettings operation to enable or disable notifications for the HIT type, without having to
     * modify the notification specification itself by providing updates to the Active status without specifying a new
     * notification specification. To change the Active status of a HIT type's notifications, the HIT type must already
     * have a notification specification, or one must be provided in the same call to
     * <code>UpdateNotificationSettings</code>.
     * </p>
     *
     * @param updateNotificationSettingsRequest
     * @return Result of the UpdateNotificationSettings operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.UpdateNotificationSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/UpdateNotificationSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateNotificationSettingsResponse updateNotificationSettings(
            UpdateNotificationSettingsRequest updateNotificationSettingsRequest) throws ServiceException, RequestErrorException,
            AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateNotificationSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateNotificationSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateNotificationSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNotificationSettings");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateNotificationSettingsRequest, UpdateNotificationSettingsResponse>()
                            .withOperationName("UpdateNotificationSettings").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateNotificationSettingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateNotificationSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>UpdateQualificationType</code> operation modifies the attributes of an existing Qualification type,
     * which is represented by a QualificationType data structure. Only the owner of a Qualification type can modify its
     * attributes.
     * </p>
     * <p>
     * Most attributes of a Qualification type can be changed after the type has been created. However, the Name and
     * Keywords fields cannot be modified. The RetryDelayInSeconds parameter can be modified or added to change the
     * delay or to enable retries, but RetryDelayInSeconds cannot be used to disable retries.
     * </p>
     * <p>
     * You can use this operation to update the test for a Qualification type. The test is updated based on the values
     * specified for the Test, TestDurationInSeconds and AnswerKey parameters. All three parameters specify the updated
     * test. If you are updating the test for a type, you must specify the Test and TestDurationInSeconds parameters.
     * The AnswerKey parameter is optional; omitting it specifies that the updated test does not have an answer key.
     * </p>
     * <p>
     * If you omit the Test parameter, the test for the Qualification type is unchanged. There is no way to remove a
     * test from a Qualification type that has one. If the type already has a test, you cannot update it to be
     * AutoGranted. If the Qualification type does not have a test and one is provided by an update, the type will
     * henceforth have a test.
     * </p>
     * <p>
     * If you want to update the test duration or answer key for an existing test without changing the questions, you
     * must specify a Test parameter with the original questions, along with the updated values.
     * </p>
     * <p>
     * If you provide an updated Test but no AnswerKey, the new test will not have an answer key. Requests for such
     * Qualifications must be granted manually.
     * </p>
     * <p>
     * You can also update the AutoGranted and AutoGrantedValue attributes of the Qualification type.
     * </p>
     *
     * @param updateQualificationTypeRequest
     * @return Result of the UpdateQualificationType operation returned by the service.
     * @throws ServiceException
     *         Amazon Mechanical Turk is temporarily unable to process your request. Try your call again.
     * @throws RequestErrorException
     *         Your request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MTurkException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MTurkClient.UpdateQualificationType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/UpdateQualificationType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateQualificationTypeResponse updateQualificationType(UpdateQualificationTypeRequest updateQualificationTypeRequest)
            throws ServiceException, RequestErrorException, AwsServiceException, SdkClientException, MTurkException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateQualificationTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateQualificationTypeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateQualificationTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateQualificationType");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateQualificationTypeRequest, UpdateQualificationTypeResponse>()
                            .withOperationName("UpdateQualificationType").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateQualificationTypeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateQualificationTypeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MTurkException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceFault").exceptionBuilderSupplier(ServiceException::builder)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestError")
                                .exceptionBuilderSupplier(RequestErrorException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends MTurkRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
