/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mturk;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.mturk.internal.MTurkServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.mturk.model.AcceptQualificationRequestRequest;
import software.amazon.awssdk.services.mturk.model.AcceptQualificationRequestResponse;
import software.amazon.awssdk.services.mturk.model.ApproveAssignmentRequest;
import software.amazon.awssdk.services.mturk.model.ApproveAssignmentResponse;
import software.amazon.awssdk.services.mturk.model.AssociateQualificationWithWorkerRequest;
import software.amazon.awssdk.services.mturk.model.AssociateQualificationWithWorkerResponse;
import software.amazon.awssdk.services.mturk.model.CreateAdditionalAssignmentsForHitRequest;
import software.amazon.awssdk.services.mturk.model.CreateAdditionalAssignmentsForHitResponse;
import software.amazon.awssdk.services.mturk.model.CreateHitRequest;
import software.amazon.awssdk.services.mturk.model.CreateHitResponse;
import software.amazon.awssdk.services.mturk.model.CreateHitTypeRequest;
import software.amazon.awssdk.services.mturk.model.CreateHitTypeResponse;
import software.amazon.awssdk.services.mturk.model.CreateHitWithHitTypeRequest;
import software.amazon.awssdk.services.mturk.model.CreateHitWithHitTypeResponse;
import software.amazon.awssdk.services.mturk.model.CreateQualificationTypeRequest;
import software.amazon.awssdk.services.mturk.model.CreateQualificationTypeResponse;
import software.amazon.awssdk.services.mturk.model.CreateWorkerBlockRequest;
import software.amazon.awssdk.services.mturk.model.CreateWorkerBlockResponse;
import software.amazon.awssdk.services.mturk.model.DeleteHitRequest;
import software.amazon.awssdk.services.mturk.model.DeleteHitResponse;
import software.amazon.awssdk.services.mturk.model.DeleteQualificationTypeRequest;
import software.amazon.awssdk.services.mturk.model.DeleteQualificationTypeResponse;
import software.amazon.awssdk.services.mturk.model.DeleteWorkerBlockRequest;
import software.amazon.awssdk.services.mturk.model.DeleteWorkerBlockResponse;
import software.amazon.awssdk.services.mturk.model.DisassociateQualificationFromWorkerRequest;
import software.amazon.awssdk.services.mturk.model.DisassociateQualificationFromWorkerResponse;
import software.amazon.awssdk.services.mturk.model.GetAccountBalanceRequest;
import software.amazon.awssdk.services.mturk.model.GetAccountBalanceResponse;
import software.amazon.awssdk.services.mturk.model.GetAssignmentRequest;
import software.amazon.awssdk.services.mturk.model.GetAssignmentResponse;
import software.amazon.awssdk.services.mturk.model.GetFileUploadUrlRequest;
import software.amazon.awssdk.services.mturk.model.GetFileUploadUrlResponse;
import software.amazon.awssdk.services.mturk.model.GetHitRequest;
import software.amazon.awssdk.services.mturk.model.GetHitResponse;
import software.amazon.awssdk.services.mturk.model.GetQualificationScoreRequest;
import software.amazon.awssdk.services.mturk.model.GetQualificationScoreResponse;
import software.amazon.awssdk.services.mturk.model.GetQualificationTypeRequest;
import software.amazon.awssdk.services.mturk.model.GetQualificationTypeResponse;
import software.amazon.awssdk.services.mturk.model.ListAssignmentsForHitRequest;
import software.amazon.awssdk.services.mturk.model.ListAssignmentsForHitResponse;
import software.amazon.awssdk.services.mturk.model.ListBonusPaymentsRequest;
import software.amazon.awssdk.services.mturk.model.ListBonusPaymentsResponse;
import software.amazon.awssdk.services.mturk.model.ListHiTsForQualificationTypeRequest;
import software.amazon.awssdk.services.mturk.model.ListHiTsForQualificationTypeResponse;
import software.amazon.awssdk.services.mturk.model.ListHiTsRequest;
import software.amazon.awssdk.services.mturk.model.ListHiTsResponse;
import software.amazon.awssdk.services.mturk.model.ListQualificationRequestsRequest;
import software.amazon.awssdk.services.mturk.model.ListQualificationRequestsResponse;
import software.amazon.awssdk.services.mturk.model.ListQualificationTypesRequest;
import software.amazon.awssdk.services.mturk.model.ListQualificationTypesResponse;
import software.amazon.awssdk.services.mturk.model.ListReviewPolicyResultsForHitRequest;
import software.amazon.awssdk.services.mturk.model.ListReviewPolicyResultsForHitResponse;
import software.amazon.awssdk.services.mturk.model.ListReviewableHiTsRequest;
import software.amazon.awssdk.services.mturk.model.ListReviewableHiTsResponse;
import software.amazon.awssdk.services.mturk.model.ListWorkerBlocksRequest;
import software.amazon.awssdk.services.mturk.model.ListWorkerBlocksResponse;
import software.amazon.awssdk.services.mturk.model.ListWorkersWithQualificationTypeRequest;
import software.amazon.awssdk.services.mturk.model.ListWorkersWithQualificationTypeResponse;
import software.amazon.awssdk.services.mturk.model.MTurkException;
import software.amazon.awssdk.services.mturk.model.NotifyWorkersRequest;
import software.amazon.awssdk.services.mturk.model.NotifyWorkersResponse;
import software.amazon.awssdk.services.mturk.model.RejectAssignmentRequest;
import software.amazon.awssdk.services.mturk.model.RejectAssignmentResponse;
import software.amazon.awssdk.services.mturk.model.RejectQualificationRequestRequest;
import software.amazon.awssdk.services.mturk.model.RejectQualificationRequestResponse;
import software.amazon.awssdk.services.mturk.model.RequestErrorException;
import software.amazon.awssdk.services.mturk.model.SendBonusRequest;
import software.amazon.awssdk.services.mturk.model.SendBonusResponse;
import software.amazon.awssdk.services.mturk.model.SendTestEventNotificationRequest;
import software.amazon.awssdk.services.mturk.model.SendTestEventNotificationResponse;
import software.amazon.awssdk.services.mturk.model.ServiceException;
import software.amazon.awssdk.services.mturk.model.UpdateExpirationForHitRequest;
import software.amazon.awssdk.services.mturk.model.UpdateExpirationForHitResponse;
import software.amazon.awssdk.services.mturk.model.UpdateHitReviewStatusRequest;
import software.amazon.awssdk.services.mturk.model.UpdateHitReviewStatusResponse;
import software.amazon.awssdk.services.mturk.model.UpdateHitTypeOfHitRequest;
import software.amazon.awssdk.services.mturk.model.UpdateHitTypeOfHitResponse;
import software.amazon.awssdk.services.mturk.model.UpdateNotificationSettingsRequest;
import software.amazon.awssdk.services.mturk.model.UpdateNotificationSettingsResponse;
import software.amazon.awssdk.services.mturk.model.UpdateQualificationTypeRequest;
import software.amazon.awssdk.services.mturk.model.UpdateQualificationTypeResponse;
import software.amazon.awssdk.services.mturk.transform.AcceptQualificationRequestRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ApproveAssignmentRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.AssociateQualificationWithWorkerRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.CreateAdditionalAssignmentsForHitRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.CreateHitRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.CreateHitTypeRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.CreateHitWithHitTypeRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.CreateQualificationTypeRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.CreateWorkerBlockRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.DeleteHitRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.DeleteQualificationTypeRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.DeleteWorkerBlockRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.DisassociateQualificationFromWorkerRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.GetAccountBalanceRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.GetAssignmentRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.GetFileUploadUrlRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.GetHitRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.GetQualificationScoreRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.GetQualificationTypeRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListAssignmentsForHitRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListBonusPaymentsRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListHiTsForQualificationTypeRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListHiTsRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListQualificationRequestsRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListQualificationTypesRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListReviewPolicyResultsForHitRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListReviewableHiTsRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListWorkerBlocksRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.ListWorkersWithQualificationTypeRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.NotifyWorkersRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.RejectAssignmentRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.RejectQualificationRequestRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.SendBonusRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.SendTestEventNotificationRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.UpdateExpirationForHitRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.UpdateHitReviewStatusRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.UpdateHitTypeOfHitRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.UpdateNotificationSettingsRequestMarshaller;
import software.amazon.awssdk.services.mturk.transform.UpdateQualificationTypeRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MTurkAsyncClient}.
 *
 * @see MTurkAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMTurkAsyncClient implements MTurkAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMTurkAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMTurkAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * The <code>AcceptQualificationRequest</code> operation approves a Worker's request for a Qualification.
     * </p>
     * <p>
     * Only the owner of the Qualification type can grant a Qualification request for that type.
     * </p>
     * <p>
     * A successful request for the <code>AcceptQualificationRequest</code> operation returns with no errors and an
     * empty body.
     * </p>
     *
     * @param acceptQualificationRequestRequest
     * @return A Java Future containing the result of the AcceptQualificationRequest operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.AcceptQualificationRequest
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/AcceptQualificationRequest"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptQualificationRequestResponse> acceptQualificationRequest(
            AcceptQualificationRequestRequest acceptQualificationRequestRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(acceptQualificationRequestRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptQualificationRequestRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptQualificationRequest");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptQualificationRequestResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AcceptQualificationRequestResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcceptQualificationRequestResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptQualificationRequestRequest, AcceptQualificationRequestResponse>()
                            .withOperationName("AcceptQualificationRequest").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AcceptQualificationRequestRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(acceptQualificationRequestRequest));
            CompletableFuture<AcceptQualificationRequestResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>ApproveAssignment</code> operation approves the results of a completed assignment.
     * </p>
     * <p>
     * Approving an assignment initiates two payments from the Requester's Amazon.com account
     * </p>
     * <ul>
     * <li>
     * <p>
     * The Worker who submitted the results is paid the reward specified in the HIT.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Mechanical Turk fees are debited.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the Requester's account does not have adequate funds for these payments, the call to ApproveAssignment returns
     * an exception, and the approval is not processed. You can include an optional feedback message with the approval,
     * which the Worker can see in the Status section of the web site.
     * </p>
     * <p>
     * You can also call this operation for assignments that were previous rejected and approve them by explicitly
     * overriding the previous rejection. This only works on rejected assignments that were submitted within the
     * previous 30 days and only if the assignment's related HIT has not been deleted.
     * </p>
     *
     * @param approveAssignmentRequest
     * @return A Java Future containing the result of the ApproveAssignment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.ApproveAssignment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ApproveAssignment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ApproveAssignmentResponse> approveAssignment(ApproveAssignmentRequest approveAssignmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(approveAssignmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, approveAssignmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ApproveAssignment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ApproveAssignmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ApproveAssignmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ApproveAssignmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ApproveAssignmentRequest, ApproveAssignmentResponse>()
                            .withOperationName("ApproveAssignment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ApproveAssignmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(approveAssignmentRequest));
            CompletableFuture<ApproveAssignmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>AssociateQualificationWithWorker</code> operation gives a Worker a Qualification.
     * <code>AssociateQualificationWithWorker</code> does not require that the Worker submit a Qualification request. It
     * gives the Qualification directly to the Worker.
     * </p>
     * <p>
     * You can only assign a Qualification of a Qualification type that you created (using the
     * <code>CreateQualificationType</code> operation).
     * </p>
     * <note>
     * <p>
     * Note: <code>AssociateQualificationWithWorker</code> does not affect any pending Qualification requests for the
     * Qualification by the Worker. If you assign a Qualification to a Worker, then later grant a Qualification request
     * made by the Worker, the granting of the request may modify the Qualification score. To resolve a pending
     * Qualification request without affecting the Qualification the Worker already has, reject the request with the
     * <code>RejectQualificationRequest</code> operation.
     * </p>
     * </note>
     *
     * @param associateQualificationWithWorkerRequest
     * @return A Java Future containing the result of the AssociateQualificationWithWorker operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.AssociateQualificationWithWorker
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/AssociateQualificationWithWorker"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateQualificationWithWorkerResponse> associateQualificationWithWorker(
            AssociateQualificationWithWorkerRequest associateQualificationWithWorkerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateQualificationWithWorkerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateQualificationWithWorkerRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateQualificationWithWorker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateQualificationWithWorkerResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, AssociateQualificationWithWorkerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateQualificationWithWorkerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateQualificationWithWorkerRequest, AssociateQualificationWithWorkerResponse>()
                            .withOperationName("AssociateQualificationWithWorker").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateQualificationWithWorkerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateQualificationWithWorkerRequest));
            CompletableFuture<AssociateQualificationWithWorkerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>CreateAdditionalAssignmentsForHIT</code> operation increases the maximum number of assignments of an
     * existing HIT.
     * </p>
     * <p>
     * To extend the maximum number of assignments, specify the number of additional assignments.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * HITs created with fewer than 10 assignments cannot be extended to have 10 or more assignments. Attempting to add
     * assignments in a way that brings the total number of assignments for a HIT from fewer than 10 assignments to 10
     * or more assignments will result in an <code>AWS.MechanicalTurk.InvalidMaximumAssignmentsIncrease</code>
     * exception.
     * </p>
     * </li>
     * <li>
     * <p>
     * HITs that were created before July 22, 2015 cannot be extended. Attempting to extend HITs that were created
     * before July 22, 2015 will result in an <code>AWS.MechanicalTurk.HITTooOldForExtension</code> exception.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param createAdditionalAssignmentsForHitRequest
     * @return A Java Future containing the result of the CreateAdditionalAssignmentsForHIT operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.CreateAdditionalAssignmentsForHIT
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/CreateAdditionalAssignmentsForHIT"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAdditionalAssignmentsForHitResponse> createAdditionalAssignmentsForHIT(
            CreateAdditionalAssignmentsForHitRequest createAdditionalAssignmentsForHitRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAdditionalAssignmentsForHitRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createAdditionalAssignmentsForHitRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAdditionalAssignmentsForHIT");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAdditionalAssignmentsForHitResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateAdditionalAssignmentsForHitResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAdditionalAssignmentsForHitResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAdditionalAssignmentsForHitRequest, CreateAdditionalAssignmentsForHitResponse>()
                            .withOperationName("CreateAdditionalAssignmentsForHIT").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAdditionalAssignmentsForHitRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAdditionalAssignmentsForHitRequest));
            CompletableFuture<CreateAdditionalAssignmentsForHitResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>CreateHIT</code> operation creates a new Human Intelligence Task (HIT). The new HIT is made available
     * for Workers to find and accept on the Amazon Mechanical Turk website.
     * </p>
     * <p>
     * This operation allows you to specify a new HIT by passing in values for the properties of the HIT, such as its
     * title, reward amount and number of assignments. When you pass these values to <code>CreateHIT</code>, a new HIT
     * is created for you, with a new <code>HITTypeID</code>. The HITTypeID can be used to create additional HITs in the
     * future without needing to specify common parameters such as the title, description and reward amount each time.
     * </p>
     * <p>
     * An alternative way to create HITs is to first generate a HITTypeID using the <code>CreateHITType</code> operation
     * and then call the <code>CreateHITWithHITType</code> operation. This is the recommended best practice for
     * Requesters who are creating large numbers of HITs.
     * </p>
     * <p>
     * CreateHIT also supports several ways to provide question data: by providing a value for the <code>Question</code>
     * parameter that fully specifies the contents of the HIT, or by providing a <code>HitLayoutId</code> and associated
     * <code>HitLayoutParameters</code>.
     * </p>
     * <note>
     * <p>
     * If a HIT is created with 10 or more maximum assignments, there is an additional fee. For more information, see <a
     * href="https://requester.mturk.com/pricing">Amazon Mechanical Turk Pricing</a>.
     * </p>
     * </note>
     *
     * @param createHitRequest
     * @return A Java Future containing the result of the CreateHIT operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.CreateHIT
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/CreateHIT" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateHitResponse> createHIT(CreateHitRequest createHitRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createHitRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHIT");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateHitResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateHitResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateHitResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateHitRequest, CreateHitResponse>().withOperationName("CreateHIT")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateHitRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(createHitRequest));
            CompletableFuture<CreateHitResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>CreateHITType</code> operation creates a new HIT type. This operation allows you to define a standard
     * set of HIT properties to use when creating HITs. If you register a HIT type with values that match an existing
     * HIT type, the HIT type ID of the existing type will be returned.
     * </p>
     *
     * @param createHitTypeRequest
     * @return A Java Future containing the result of the CreateHITType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.CreateHITType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/CreateHITType" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateHitTypeResponse> createHITType(CreateHitTypeRequest createHitTypeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createHitTypeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHitTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHITType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateHitTypeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateHitTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateHitTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateHitTypeRequest, CreateHitTypeResponse>()
                            .withOperationName("CreateHITType").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateHitTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createHitTypeRequest));
            CompletableFuture<CreateHitTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>CreateHITWithHITType</code> operation creates a new Human Intelligence Task (HIT) using an existing
     * HITTypeID generated by the <code>CreateHITType</code> operation.
     * </p>
     * <p>
     * This is an alternative way to create HITs from the <code>CreateHIT</code> operation. This is the recommended best
     * practice for Requesters who are creating large numbers of HITs.
     * </p>
     * <p>
     * CreateHITWithHITType also supports several ways to provide question data: by providing a value for the
     * <code>Question</code> parameter that fully specifies the contents of the HIT, or by providing a
     * <code>HitLayoutId</code> and associated <code>HitLayoutParameters</code>.
     * </p>
     * <note>
     * <p>
     * If a HIT is created with 10 or more maximum assignments, there is an additional fee. For more information, see <a
     * href="https://requester.mturk.com/pricing">Amazon Mechanical Turk Pricing</a>.
     * </p>
     * </note>
     *
     * @param createHitWithHitTypeRequest
     * @return A Java Future containing the result of the CreateHITWithHITType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.CreateHITWithHITType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/CreateHITWithHITType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateHitWithHitTypeResponse> createHITWithHITType(
            CreateHitWithHitTypeRequest createHitWithHitTypeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createHitWithHitTypeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHitWithHitTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHITWithHITType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateHitWithHitTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateHitWithHitTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateHitWithHitTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateHitWithHitTypeRequest, CreateHitWithHitTypeResponse>()
                            .withOperationName("CreateHITWithHITType").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateHitWithHitTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createHitWithHitTypeRequest));
            CompletableFuture<CreateHitWithHitTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>CreateQualificationType</code> operation creates a new Qualification type, which is represented by a
     * <code>QualificationType</code> data structure.
     * </p>
     *
     * @param createQualificationTypeRequest
     * @return A Java Future containing the result of the CreateQualificationType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.CreateQualificationType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/CreateQualificationType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateQualificationTypeResponse> createQualificationType(
            CreateQualificationTypeRequest createQualificationTypeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createQualificationTypeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createQualificationTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateQualificationType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateQualificationTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateQualificationTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateQualificationTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateQualificationTypeRequest, CreateQualificationTypeResponse>()
                            .withOperationName("CreateQualificationType").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateQualificationTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createQualificationTypeRequest));
            CompletableFuture<CreateQualificationTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>CreateWorkerBlock</code> operation allows you to prevent a Worker from working on your HITs. For
     * example, you can block a Worker who is producing poor quality work. You can block up to 100,000 Workers.
     * </p>
     *
     * @param createWorkerBlockRequest
     * @return A Java Future containing the result of the CreateWorkerBlock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.CreateWorkerBlock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/CreateWorkerBlock"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWorkerBlockResponse> createWorkerBlock(CreateWorkerBlockRequest createWorkerBlockRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWorkerBlockRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkerBlockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkerBlock");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWorkerBlockResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateWorkerBlockResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWorkerBlockResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWorkerBlockRequest, CreateWorkerBlockResponse>()
                            .withOperationName("CreateWorkerBlock").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateWorkerBlockRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createWorkerBlockRequest));
            CompletableFuture<CreateWorkerBlockResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>DeleteHIT</code> operation is used to delete HIT that is no longer needed. Only the Requester who
     * created the HIT can delete it.
     * </p>
     * <p>
     * You can only dispose of HITs that are in the <code>Reviewable</code> state, with all of their submitted
     * assignments already either approved or rejected. If you call the DeleteHIT operation on a HIT that is not in the
     * <code>Reviewable</code> state (for example, that has not expired, or still has active assignments), or on a HIT
     * that is Reviewable but without all of its submitted assignments already approved or rejected, the service will
     * return an error.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * HITs are automatically disposed of after 120 days.
     * </p>
     * </li>
     * <li>
     * <p>
     * After you dispose of a HIT, you can no longer approve the HIT's rejected assignments.
     * </p>
     * </li>
     * <li>
     * <p>
     * Disposed HITs are not returned in results for the ListHITs operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * Disposing HITs can improve the performance of operations such as ListReviewableHITs and ListHITs.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param deleteHitRequest
     * @return A Java Future containing the result of the DeleteHIT operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.DeleteHIT
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/DeleteHIT" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteHitResponse> deleteHIT(DeleteHitRequest deleteHitRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteHitRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteHitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHIT");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteHitResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteHitResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteHitResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteHitRequest, DeleteHitResponse>().withOperationName("DeleteHIT")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteHitRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteHitRequest));
            CompletableFuture<DeleteHitResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>DeleteQualificationType</code> deletes a Qualification type and deletes any HIT types that are
     * associated with the Qualification type.
     * </p>
     * <p>
     * This operation does not revoke Qualifications already assigned to Workers because the Qualifications might be
     * needed for active HITs. If there are any pending requests for the Qualification type, Amazon Mechanical Turk
     * rejects those requests. After you delete a Qualification type, you can no longer use it to create HITs or HIT
     * types.
     * </p>
     * <note>
     * <p>
     * DeleteQualificationType must wait for all the HITs that use the deleted Qualification type to be deleted before
     * completing. It may take up to 48 hours before DeleteQualificationType completes and the unique name of the
     * Qualification type is available for reuse with CreateQualificationType.
     * </p>
     * </note>
     *
     * @param deleteQualificationTypeRequest
     * @return A Java Future containing the result of the DeleteQualificationType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.DeleteQualificationType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/DeleteQualificationType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteQualificationTypeResponse> deleteQualificationType(
            DeleteQualificationTypeRequest deleteQualificationTypeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteQualificationTypeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteQualificationTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteQualificationType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteQualificationTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteQualificationTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteQualificationTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteQualificationTypeRequest, DeleteQualificationTypeResponse>()
                            .withOperationName("DeleteQualificationType").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteQualificationTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteQualificationTypeRequest));
            CompletableFuture<DeleteQualificationTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>DeleteWorkerBlock</code> operation allows you to reinstate a blocked Worker to work on your HITs. This
     * operation reverses the effects of the CreateWorkerBlock operation. You need the Worker ID to use this operation.
     * If the Worker ID is missing or invalid, this operation fails and returns the message “WorkerId is invalid.” If
     * the specified Worker is not blocked, this operation returns successfully.
     * </p>
     *
     * @param deleteWorkerBlockRequest
     * @return A Java Future containing the result of the DeleteWorkerBlock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.DeleteWorkerBlock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/DeleteWorkerBlock"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWorkerBlockResponse> deleteWorkerBlock(DeleteWorkerBlockRequest deleteWorkerBlockRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWorkerBlockRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkerBlockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkerBlock");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWorkerBlockResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWorkerBlockResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWorkerBlockResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWorkerBlockRequest, DeleteWorkerBlockResponse>()
                            .withOperationName("DeleteWorkerBlock").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteWorkerBlockRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteWorkerBlockRequest));
            CompletableFuture<DeleteWorkerBlockResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>DisassociateQualificationFromWorker</code> revokes a previously granted Qualification from a user.
     * </p>
     * <p>
     * You can provide a text message explaining why the Qualification was revoked. The user who had the Qualification
     * can see this message.
     * </p>
     *
     * @param disassociateQualificationFromWorkerRequest
     * @return A Java Future containing the result of the DisassociateQualificationFromWorker operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.DisassociateQualificationFromWorker
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/DisassociateQualificationFromWorker"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateQualificationFromWorkerResponse> disassociateQualificationFromWorker(
            DisassociateQualificationFromWorkerRequest disassociateQualificationFromWorkerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateQualificationFromWorkerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateQualificationFromWorkerRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateQualificationFromWorker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateQualificationFromWorkerResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateQualificationFromWorkerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateQualificationFromWorkerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateQualificationFromWorkerRequest, DisassociateQualificationFromWorkerResponse>()
                            .withOperationName("DisassociateQualificationFromWorker").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateQualificationFromWorkerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateQualificationFromWorkerRequest));
            CompletableFuture<DisassociateQualificationFromWorkerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>GetAccountBalance</code> operation retrieves the Prepaid HITs balance in your Amazon Mechanical Turk
     * account if you are a Prepaid Requester. Alternatively, this operation will retrieve the remaining available AWS
     * Billing usage if you have enabled AWS Billing. Note: If you have enabled AWS Billing and still have a remaining
     * Prepaid HITs balance, this balance can be viewed on the My Account page in the Requester console.
     * </p>
     *
     * @param getAccountBalanceRequest
     * @return A Java Future containing the result of the GetAccountBalance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.GetAccountBalance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/GetAccountBalance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccountBalanceResponse> getAccountBalance(GetAccountBalanceRequest getAccountBalanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAccountBalanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccountBalanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccountBalance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAccountBalanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAccountBalanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAccountBalanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccountBalanceRequest, GetAccountBalanceResponse>()
                            .withOperationName("GetAccountBalance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAccountBalanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAccountBalanceRequest));
            CompletableFuture<GetAccountBalanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>GetAssignment</code> operation retrieves the details of the specified Assignment.
     * </p>
     *
     * @param getAssignmentRequest
     * @return A Java Future containing the result of the GetAssignment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.GetAssignment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/GetAssignment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAssignmentResponse> getAssignment(GetAssignmentRequest getAssignmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAssignmentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAssignmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAssignment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAssignmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAssignmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAssignmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAssignmentRequest, GetAssignmentResponse>()
                            .withOperationName("GetAssignment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAssignmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAssignmentRequest));
            CompletableFuture<GetAssignmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>GetFileUploadURL</code> operation generates and returns a temporary URL. You use the temporary URL to
     * retrieve a file uploaded by a Worker as an answer to a FileUploadAnswer question for a HIT. The temporary URL is
     * generated the instant the GetFileUploadURL operation is called, and is valid for 60 seconds. You can get a
     * temporary file upload URL any time until the HIT is disposed. After the HIT is disposed, any uploaded files are
     * deleted, and cannot be retrieved. Pending Deprecation on December 12, 2017. The Answer Specification structure
     * will no longer support the <code>FileUploadAnswer</code> element to be used for the QuestionForm data structure.
     * Instead, we recommend that Requesters who want to create HITs asking Workers to upload files to use Amazon S3.
     * </p>
     *
     * @param getFileUploadUrlRequest
     * @return A Java Future containing the result of the GetFileUploadURL operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.GetFileUploadURL
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/GetFileUploadURL"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetFileUploadUrlResponse> getFileUploadURL(GetFileUploadUrlRequest getFileUploadUrlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getFileUploadUrlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFileUploadUrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFileUploadURL");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFileUploadUrlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetFileUploadUrlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFileUploadUrlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFileUploadUrlRequest, GetFileUploadUrlResponse>()
                            .withOperationName("GetFileUploadURL").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetFileUploadUrlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getFileUploadUrlRequest));
            CompletableFuture<GetFileUploadUrlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>GetHIT</code> operation retrieves the details of the specified HIT.
     * </p>
     *
     * @param getHitRequest
     * @return A Java Future containing the result of the GetHIT operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.GetHIT
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/GetHIT" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetHitResponse> getHIT(GetHitRequest getHitRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getHitRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getHitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetHIT");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetHitResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetHitResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetHitResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetHitRequest, GetHitResponse>().withOperationName("GetHIT")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetHitRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getHitRequest));
            CompletableFuture<GetHitResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>GetQualificationScore</code> operation returns the value of a Worker's Qualification for a given
     * Qualification type.
     * </p>
     * <p>
     * To get a Worker's Qualification, you must know the Worker's ID. The Worker's ID is included in the assignment
     * data returned by the <code>ListAssignmentsForHIT</code> operation.
     * </p>
     * <p>
     * Only the owner of a Qualification type can query the value of a Worker's Qualification of that type.
     * </p>
     *
     * @param getQualificationScoreRequest
     * @return A Java Future containing the result of the GetQualificationScore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.GetQualificationScore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/GetQualificationScore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetQualificationScoreResponse> getQualificationScore(
            GetQualificationScoreRequest getQualificationScoreRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getQualificationScoreRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQualificationScoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQualificationScore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQualificationScoreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetQualificationScoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetQualificationScoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQualificationScoreRequest, GetQualificationScoreResponse>()
                            .withOperationName("GetQualificationScore").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetQualificationScoreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getQualificationScoreRequest));
            CompletableFuture<GetQualificationScoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>GetQualificationType</code>operation retrieves information about a Qualification type using its ID.
     * </p>
     *
     * @param getQualificationTypeRequest
     * @return A Java Future containing the result of the GetQualificationType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.GetQualificationType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/GetQualificationType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetQualificationTypeResponse> getQualificationType(
            GetQualificationTypeRequest getQualificationTypeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getQualificationTypeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQualificationTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQualificationType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQualificationTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetQualificationTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetQualificationTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQualificationTypeRequest, GetQualificationTypeResponse>()
                            .withOperationName("GetQualificationType").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetQualificationTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getQualificationTypeRequest));
            CompletableFuture<GetQualificationTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>ListAssignmentsForHIT</code> operation retrieves completed assignments for a HIT. You can use this
     * operation to retrieve the results for a HIT.
     * </p>
     * <p>
     * You can get assignments for a HIT at any time, even if the HIT is not yet Reviewable. If a HIT requested multiple
     * assignments, and has received some results but has not yet become Reviewable, you can still retrieve the partial
     * results with this operation.
     * </p>
     * <p>
     * Use the AssignmentStatus parameter to control which set of assignments for a HIT are returned. The
     * ListAssignmentsForHIT operation can return submitted assignments awaiting approval, or it can return assignments
     * that have already been approved or rejected. You can set AssignmentStatus=Approved,Rejected to get assignments
     * that have already been approved and rejected together in one result set.
     * </p>
     * <p>
     * Only the Requester who created the HIT can retrieve the assignments for that HIT.
     * </p>
     * <p>
     * Results are sorted and divided into numbered pages and the operation returns a single page of results. You can
     * use the parameters of the operation to control sorting and pagination.
     * </p>
     *
     * @param listAssignmentsForHitRequest
     * @return A Java Future containing the result of the ListAssignmentsForHIT operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.ListAssignmentsForHIT
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListAssignmentsForHIT"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAssignmentsForHitResponse> listAssignmentsForHIT(
            ListAssignmentsForHitRequest listAssignmentsForHitRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAssignmentsForHitRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAssignmentsForHitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssignmentsForHIT");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAssignmentsForHitResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAssignmentsForHitResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAssignmentsForHitResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAssignmentsForHitRequest, ListAssignmentsForHitResponse>()
                            .withOperationName("ListAssignmentsForHIT").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAssignmentsForHitRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAssignmentsForHitRequest));
            CompletableFuture<ListAssignmentsForHitResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>ListBonusPayments</code> operation retrieves the amounts of bonuses you have paid to Workers for a
     * given HIT or assignment.
     * </p>
     *
     * @param listBonusPaymentsRequest
     * @return A Java Future containing the result of the ListBonusPayments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.ListBonusPayments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListBonusPayments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListBonusPaymentsResponse> listBonusPayments(ListBonusPaymentsRequest listBonusPaymentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listBonusPaymentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBonusPaymentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBonusPayments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListBonusPaymentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListBonusPaymentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListBonusPaymentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListBonusPaymentsRequest, ListBonusPaymentsResponse>()
                            .withOperationName("ListBonusPayments").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListBonusPaymentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listBonusPaymentsRequest));
            CompletableFuture<ListBonusPaymentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>ListHITs</code> operation returns all of a Requester's HITs. The operation returns HITs of any status,
     * except for HITs that have been deleted of with the DeleteHIT operation or that have been auto-deleted.
     * </p>
     *
     * @param listHiTsRequest
     * @return A Java Future containing the result of the ListHITs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.ListHITs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListHITs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListHiTsResponse> listHITs(ListHiTsRequest listHiTsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listHiTsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listHiTsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHITs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListHiTsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListHiTsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListHiTsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListHiTsRequest, ListHiTsResponse>().withOperationName("ListHITs")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListHiTsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listHiTsRequest));
            CompletableFuture<ListHiTsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>ListHITsForQualificationType</code> operation returns the HITs that use the given Qualification type
     * for a Qualification requirement. The operation returns HITs of any status, except for HITs that have been deleted
     * with the <code>DeleteHIT</code> operation or that have been auto-deleted.
     * </p>
     *
     * @param listHiTsForQualificationTypeRequest
     * @return A Java Future containing the result of the ListHITsForQualificationType operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.ListHITsForQualificationType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListHITsForQualificationType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListHiTsForQualificationTypeResponse> listHITsForQualificationType(
            ListHiTsForQualificationTypeRequest listHiTsForQualificationTypeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listHiTsForQualificationTypeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listHiTsForQualificationTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHITsForQualificationType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListHiTsForQualificationTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListHiTsForQualificationTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListHiTsForQualificationTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListHiTsForQualificationTypeRequest, ListHiTsForQualificationTypeResponse>()
                            .withOperationName("ListHITsForQualificationType").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListHiTsForQualificationTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listHiTsForQualificationTypeRequest));
            CompletableFuture<ListHiTsForQualificationTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>ListQualificationRequests</code> operation retrieves requests for Qualifications of a particular
     * Qualification type. The owner of the Qualification type calls this operation to poll for pending requests, and
     * accepts them using the AcceptQualification operation.
     * </p>
     *
     * @param listQualificationRequestsRequest
     * @return A Java Future containing the result of the ListQualificationRequests operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.ListQualificationRequests
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListQualificationRequests"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListQualificationRequestsResponse> listQualificationRequests(
            ListQualificationRequestsRequest listQualificationRequestsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listQualificationRequestsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listQualificationRequestsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListQualificationRequests");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListQualificationRequestsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListQualificationRequestsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListQualificationRequestsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListQualificationRequestsRequest, ListQualificationRequestsResponse>()
                            .withOperationName("ListQualificationRequests").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListQualificationRequestsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listQualificationRequestsRequest));
            CompletableFuture<ListQualificationRequestsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>ListQualificationTypes</code> operation returns a list of Qualification types, filtered by an optional
     * search term.
     * </p>
     *
     * @param listQualificationTypesRequest
     * @return A Java Future containing the result of the ListQualificationTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.ListQualificationTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListQualificationTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListQualificationTypesResponse> listQualificationTypes(
            ListQualificationTypesRequest listQualificationTypesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listQualificationTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listQualificationTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListQualificationTypes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListQualificationTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListQualificationTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListQualificationTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListQualificationTypesRequest, ListQualificationTypesResponse>()
                            .withOperationName("ListQualificationTypes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListQualificationTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listQualificationTypesRequest));
            CompletableFuture<ListQualificationTypesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>ListReviewPolicyResultsForHIT</code> operation retrieves the computed results and the actions taken in
     * the course of executing your Review Policies for a given HIT. For information about how to specify Review
     * Policies when you call CreateHIT, see Review Policies. The ListReviewPolicyResultsForHIT operation can return
     * results for both Assignment-level and HIT-level review results.
     * </p>
     *
     * @param listReviewPolicyResultsForHitRequest
     * @return A Java Future containing the result of the ListReviewPolicyResultsForHIT operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.ListReviewPolicyResultsForHIT
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListReviewPolicyResultsForHIT"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListReviewPolicyResultsForHitResponse> listReviewPolicyResultsForHIT(
            ListReviewPolicyResultsForHitRequest listReviewPolicyResultsForHitRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listReviewPolicyResultsForHitRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listReviewPolicyResultsForHitRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReviewPolicyResultsForHIT");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListReviewPolicyResultsForHitResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListReviewPolicyResultsForHitResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListReviewPolicyResultsForHitResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListReviewPolicyResultsForHitRequest, ListReviewPolicyResultsForHitResponse>()
                            .withOperationName("ListReviewPolicyResultsForHIT").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListReviewPolicyResultsForHitRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listReviewPolicyResultsForHitRequest));
            CompletableFuture<ListReviewPolicyResultsForHitResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>ListReviewableHITs</code> operation retrieves the HITs with Status equal to Reviewable or Status equal
     * to Reviewing that belong to the Requester calling the operation.
     * </p>
     *
     * @param listReviewableHiTsRequest
     * @return A Java Future containing the result of the ListReviewableHITs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.ListReviewableHITs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListReviewableHITs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListReviewableHiTsResponse> listReviewableHITs(ListReviewableHiTsRequest listReviewableHiTsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listReviewableHiTsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listReviewableHiTsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReviewableHITs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListReviewableHiTsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListReviewableHiTsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListReviewableHiTsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListReviewableHiTsRequest, ListReviewableHiTsResponse>()
                            .withOperationName("ListReviewableHITs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListReviewableHiTsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listReviewableHiTsRequest));
            CompletableFuture<ListReviewableHiTsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>ListWorkersBlocks</code> operation retrieves a list of Workers who are blocked from working on your
     * HITs.
     * </p>
     *
     * @param listWorkerBlocksRequest
     * @return A Java Future containing the result of the ListWorkerBlocks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.ListWorkerBlocks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListWorkerBlocks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWorkerBlocksResponse> listWorkerBlocks(ListWorkerBlocksRequest listWorkerBlocksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWorkerBlocksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkerBlocksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkerBlocks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWorkerBlocksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListWorkerBlocksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWorkerBlocksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWorkerBlocksRequest, ListWorkerBlocksResponse>()
                            .withOperationName("ListWorkerBlocks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWorkerBlocksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWorkerBlocksRequest));
            CompletableFuture<ListWorkerBlocksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>ListWorkersWithQualificationType</code> operation returns all of the Workers that have been associated
     * with a given Qualification type.
     * </p>
     *
     * @param listWorkersWithQualificationTypeRequest
     * @return A Java Future containing the result of the ListWorkersWithQualificationType operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.ListWorkersWithQualificationType
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/ListWorkersWithQualificationType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWorkersWithQualificationTypeResponse> listWorkersWithQualificationType(
            ListWorkersWithQualificationTypeRequest listWorkersWithQualificationTypeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWorkersWithQualificationTypeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listWorkersWithQualificationTypeRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkersWithQualificationType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWorkersWithQualificationTypeResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListWorkersWithQualificationTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWorkersWithQualificationTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWorkersWithQualificationTypeRequest, ListWorkersWithQualificationTypeResponse>()
                            .withOperationName("ListWorkersWithQualificationType").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWorkersWithQualificationTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWorkersWithQualificationTypeRequest));
            CompletableFuture<ListWorkersWithQualificationTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>NotifyWorkers</code> operation sends an email to one or more Workers that you specify with the Worker
     * ID. You can specify up to 100 Worker IDs to send the same message with a single call to the NotifyWorkers
     * operation. The NotifyWorkers operation will send a notification email to a Worker only if you have previously
     * approved or rejected work from the Worker.
     * </p>
     *
     * @param notifyWorkersRequest
     * @return A Java Future containing the result of the NotifyWorkers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.NotifyWorkers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/NotifyWorkers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<NotifyWorkersResponse> notifyWorkers(NotifyWorkersRequest notifyWorkersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(notifyWorkersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, notifyWorkersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "NotifyWorkers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<NotifyWorkersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    NotifyWorkersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<NotifyWorkersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<NotifyWorkersRequest, NotifyWorkersResponse>()
                            .withOperationName("NotifyWorkers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new NotifyWorkersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(notifyWorkersRequest));
            CompletableFuture<NotifyWorkersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>RejectAssignment</code> operation rejects the results of a completed assignment.
     * </p>
     * <p>
     * You can include an optional feedback message with the rejection, which the Worker can see in the Status section
     * of the web site. When you include a feedback message with the rejection, it helps the Worker understand why the
     * assignment was rejected, and can improve the quality of the results the Worker submits in the future.
     * </p>
     * <p>
     * Only the Requester who created the HIT can reject an assignment for the HIT.
     * </p>
     *
     * @param rejectAssignmentRequest
     * @return A Java Future containing the result of the RejectAssignment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.RejectAssignment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/RejectAssignment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RejectAssignmentResponse> rejectAssignment(RejectAssignmentRequest rejectAssignmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(rejectAssignmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rejectAssignmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectAssignment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RejectAssignmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RejectAssignmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RejectAssignmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RejectAssignmentRequest, RejectAssignmentResponse>()
                            .withOperationName("RejectAssignment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RejectAssignmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(rejectAssignmentRequest));
            CompletableFuture<RejectAssignmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>RejectQualificationRequest</code> operation rejects a user's request for a Qualification.
     * </p>
     * <p>
     * You can provide a text message explaining why the request was rejected. The Worker who made the request can see
     * this message.
     * </p>
     *
     * @param rejectQualificationRequestRequest
     * @return A Java Future containing the result of the RejectQualificationRequest operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.RejectQualificationRequest
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/RejectQualificationRequest"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RejectQualificationRequestResponse> rejectQualificationRequest(
            RejectQualificationRequestRequest rejectQualificationRequestRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(rejectQualificationRequestRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rejectQualificationRequestRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectQualificationRequest");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RejectQualificationRequestResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RejectQualificationRequestResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RejectQualificationRequestResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RejectQualificationRequestRequest, RejectQualificationRequestResponse>()
                            .withOperationName("RejectQualificationRequest").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RejectQualificationRequestRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(rejectQualificationRequestRequest));
            CompletableFuture<RejectQualificationRequestResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>SendBonus</code> operation issues a payment of money from your account to a Worker. This payment
     * happens separately from the reward you pay to the Worker when you approve the Worker's assignment. The SendBonus
     * operation requires the Worker's ID and the assignment ID as parameters to initiate payment of the bonus. You must
     * include a message that explains the reason for the bonus payment, as the Worker may not be expecting the payment.
     * Amazon Mechanical Turk collects a fee for bonus payments, similar to the HIT listing fee. This operation fails if
     * your account does not have enough funds to pay for both the bonus and the fees.
     * </p>
     *
     * @param sendBonusRequest
     * @return A Java Future containing the result of the SendBonus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.SendBonus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/SendBonus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SendBonusResponse> sendBonus(SendBonusRequest sendBonusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendBonusRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendBonusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendBonus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendBonusResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SendBonusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendBonusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendBonusRequest, SendBonusResponse>().withOperationName("SendBonus")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SendBonusRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(sendBonusRequest));
            CompletableFuture<SendBonusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>SendTestEventNotification</code> operation causes Amazon Mechanical Turk to send a notification message
     * as if a HIT event occurred, according to the provided notification specification. This allows you to test
     * notifications without setting up notifications for a real HIT type and trying to trigger them using the website.
     * When you call this operation, the service attempts to send the test notification immediately.
     * </p>
     *
     * @param sendTestEventNotificationRequest
     * @return A Java Future containing the result of the SendTestEventNotification operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.SendTestEventNotification
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/SendTestEventNotification"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SendTestEventNotificationResponse> sendTestEventNotification(
            SendTestEventNotificationRequest sendTestEventNotificationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendTestEventNotificationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendTestEventNotificationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendTestEventNotification");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendTestEventNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendTestEventNotificationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendTestEventNotificationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendTestEventNotificationRequest, SendTestEventNotificationResponse>()
                            .withOperationName("SendTestEventNotification").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SendTestEventNotificationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(sendTestEventNotificationRequest));
            CompletableFuture<SendTestEventNotificationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>UpdateExpirationForHIT</code> operation allows you update the expiration time of a HIT. If you update
     * it to a time in the past, the HIT will be immediately expired.
     * </p>
     *
     * @param updateExpirationForHitRequest
     * @return A Java Future containing the result of the UpdateExpirationForHIT operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.UpdateExpirationForHIT
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/UpdateExpirationForHIT"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateExpirationForHitResponse> updateExpirationForHIT(
            UpdateExpirationForHitRequest updateExpirationForHitRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateExpirationForHitRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateExpirationForHitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateExpirationForHIT");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateExpirationForHitResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateExpirationForHitResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateExpirationForHitResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateExpirationForHitRequest, UpdateExpirationForHitResponse>()
                            .withOperationName("UpdateExpirationForHIT").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateExpirationForHitRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateExpirationForHitRequest));
            CompletableFuture<UpdateExpirationForHitResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>UpdateHITReviewStatus</code> operation updates the status of a HIT. If the status is Reviewable, this
     * operation can update the status to Reviewing, or it can revert a Reviewing HIT back to the Reviewable status.
     * </p>
     *
     * @param updateHitReviewStatusRequest
     * @return A Java Future containing the result of the UpdateHITReviewStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.UpdateHITReviewStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/UpdateHITReviewStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateHitReviewStatusResponse> updateHITReviewStatus(
            UpdateHitReviewStatusRequest updateHitReviewStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateHitReviewStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateHitReviewStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateHITReviewStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateHitReviewStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateHitReviewStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateHitReviewStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateHitReviewStatusRequest, UpdateHitReviewStatusResponse>()
                            .withOperationName("UpdateHITReviewStatus").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateHitReviewStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateHitReviewStatusRequest));
            CompletableFuture<UpdateHitReviewStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>UpdateHITTypeOfHIT</code> operation allows you to change the HITType properties of a HIT. This
     * operation disassociates the HIT from its old HITType properties and associates it with the new HITType
     * properties. The HIT takes on the properties of the new HITType in place of the old ones.
     * </p>
     *
     * @param updateHitTypeOfHitRequest
     * @return A Java Future containing the result of the UpdateHITTypeOfHIT operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.UpdateHITTypeOfHIT
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/UpdateHITTypeOfHIT"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateHitTypeOfHitResponse> updateHITTypeOfHIT(UpdateHitTypeOfHitRequest updateHitTypeOfHitRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateHitTypeOfHitRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateHitTypeOfHitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateHITTypeOfHIT");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateHitTypeOfHitResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateHitTypeOfHitResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateHitTypeOfHitResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateHitTypeOfHitRequest, UpdateHitTypeOfHitResponse>()
                            .withOperationName("UpdateHITTypeOfHIT").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateHitTypeOfHitRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateHitTypeOfHitRequest));
            CompletableFuture<UpdateHitTypeOfHitResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>UpdateNotificationSettings</code> operation creates, updates, disables or re-enables notifications for
     * a HIT type. If you call the UpdateNotificationSettings operation for a HIT type that already has a notification
     * specification, the operation replaces the old specification with a new one. You can call the
     * UpdateNotificationSettings operation to enable or disable notifications for the HIT type, without having to
     * modify the notification specification itself by providing updates to the Active status without specifying a new
     * notification specification. To change the Active status of a HIT type's notifications, the HIT type must already
     * have a notification specification, or one must be provided in the same call to
     * <code>UpdateNotificationSettings</code>.
     * </p>
     *
     * @param updateNotificationSettingsRequest
     * @return A Java Future containing the result of the UpdateNotificationSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.UpdateNotificationSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/UpdateNotificationSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateNotificationSettingsResponse> updateNotificationSettings(
            UpdateNotificationSettingsRequest updateNotificationSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateNotificationSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateNotificationSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNotificationSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateNotificationSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateNotificationSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateNotificationSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateNotificationSettingsRequest, UpdateNotificationSettingsResponse>()
                            .withOperationName("UpdateNotificationSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateNotificationSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateNotificationSettingsRequest));
            CompletableFuture<UpdateNotificationSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>UpdateQualificationType</code> operation modifies the attributes of an existing Qualification type,
     * which is represented by a QualificationType data structure. Only the owner of a Qualification type can modify its
     * attributes.
     * </p>
     * <p>
     * Most attributes of a Qualification type can be changed after the type has been created. However, the Name and
     * Keywords fields cannot be modified. The RetryDelayInSeconds parameter can be modified or added to change the
     * delay or to enable retries, but RetryDelayInSeconds cannot be used to disable retries.
     * </p>
     * <p>
     * You can use this operation to update the test for a Qualification type. The test is updated based on the values
     * specified for the Test, TestDurationInSeconds and AnswerKey parameters. All three parameters specify the updated
     * test. If you are updating the test for a type, you must specify the Test and TestDurationInSeconds parameters.
     * The AnswerKey parameter is optional; omitting it specifies that the updated test does not have an answer key.
     * </p>
     * <p>
     * If you omit the Test parameter, the test for the Qualification type is unchanged. There is no way to remove a
     * test from a Qualification type that has one. If the type already has a test, you cannot update it to be
     * AutoGranted. If the Qualification type does not have a test and one is provided by an update, the type will
     * henceforth have a test.
     * </p>
     * <p>
     * If you want to update the test duration or answer key for an existing test without changing the questions, you
     * must specify a Test parameter with the original questions, along with the updated values.
     * </p>
     * <p>
     * If you provide an updated Test but no AnswerKey, the new test will not have an answer key. Requests for such
     * Qualifications must be granted manually.
     * </p>
     * <p>
     * You can also update the AutoGranted and AutoGrantedValue attributes of the Qualification type.
     * </p>
     *
     * @param updateQualificationTypeRequest
     * @return A Java Future containing the result of the UpdateQualificationType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceException Amazon Mechanical Turk is temporarily unable to process your request. Try your call
     *         again.</li>
     *         <li>RequestErrorException Your request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MTurkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MTurkAsyncClient.UpdateQualificationType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mturk-requester-2017-01-17/UpdateQualificationType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateQualificationTypeResponse> updateQualificationType(
            UpdateQualificationTypeRequest updateQualificationTypeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateQualificationTypeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateQualificationTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MTurk");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateQualificationType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateQualificationTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateQualificationTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateQualificationTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateQualificationTypeRequest, UpdateQualificationTypeResponse>()
                            .withOperationName("UpdateQualificationType").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateQualificationTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateQualificationTypeRequest));
            CompletableFuture<UpdateQualificationTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final MTurkServiceClientConfiguration serviceClientConfiguration() {
        return new MTurkServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MTurkException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceFault").exceptionBuilderSupplier(ServiceException::builder)
                                .httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestError")
                                .exceptionBuilderSupplier(RequestErrorException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        MTurkServiceClientConfigurationBuilder serviceConfigBuilder = new MTurkServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
