/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mturk.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The HIT data structure represents a single HIT, including all the information necessary for a Worker to accept and
 * complete the HIT.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HIT implements SdkPojo, Serializable, ToCopyableBuilder<HIT.Builder, HIT> {
    private static final SdkField<String> HIT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("HITId")
            .getter(getter(HIT::hitId)).setter(setter(Builder::hitId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HITId").build()).build();

    private static final SdkField<String> HIT_TYPE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HITTypeId").getter(getter(HIT::hitTypeId)).setter(setter(Builder::hitTypeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HITTypeId").build()).build();

    private static final SdkField<String> HIT_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HITGroupId").getter(getter(HIT::hitGroupId)).setter(setter(Builder::hitGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HITGroupId").build()).build();

    private static final SdkField<String> HIT_LAYOUT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HITLayoutId").getter(getter(HIT::hitLayoutId)).setter(setter(Builder::hitLayoutId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HITLayoutId").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(HIT::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Title")
            .getter(getter(HIT::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(HIT::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> QUESTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Question").getter(getter(HIT::question)).setter(setter(Builder::question))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Question").build()).build();

    private static final SdkField<String> KEYWORDS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Keywords").getter(getter(HIT::keywords)).setter(setter(Builder::keywords))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Keywords").build()).build();

    private static final SdkField<String> HIT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HITStatus").getter(getter(HIT::hitStatusAsString)).setter(setter(Builder::hitStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HITStatus").build()).build();

    private static final SdkField<Integer> MAX_ASSIGNMENTS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxAssignments").getter(getter(HIT::maxAssignments)).setter(setter(Builder::maxAssignments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxAssignments").build()).build();

    private static final SdkField<String> REWARD_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Reward")
            .getter(getter(HIT::reward)).setter(setter(Builder::reward))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Reward").build()).build();

    private static final SdkField<Long> AUTO_APPROVAL_DELAY_IN_SECONDS_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("AutoApprovalDelayInSeconds")
            .getter(getter(HIT::autoApprovalDelayInSeconds))
            .setter(setter(Builder::autoApprovalDelayInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoApprovalDelayInSeconds").build())
            .build();

    private static final SdkField<Instant> EXPIRATION_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("Expiration").getter(getter(HIT::expiration)).setter(setter(Builder::expiration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Expiration").build()).build();

    private static final SdkField<Long> ASSIGNMENT_DURATION_IN_SECONDS_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("AssignmentDurationInSeconds")
            .getter(getter(HIT::assignmentDurationInSeconds))
            .setter(setter(Builder::assignmentDurationInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssignmentDurationInSeconds")
                    .build()).build();

    private static final SdkField<String> REQUESTER_ANNOTATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RequesterAnnotation").getter(getter(HIT::requesterAnnotation))
            .setter(setter(Builder::requesterAnnotation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequesterAnnotation").build())
            .build();

    private static final SdkField<List<QualificationRequirement>> QUALIFICATION_REQUIREMENTS_FIELD = SdkField
            .<List<QualificationRequirement>> builder(MarshallingType.LIST)
            .memberName("QualificationRequirements")
            .getter(getter(HIT::qualificationRequirements))
            .setter(setter(Builder::qualificationRequirements))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QualificationRequirements").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<QualificationRequirement> builder(MarshallingType.SDK_POJO)
                                            .constructor(QualificationRequirement::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> HIT_REVIEW_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HITReviewStatus").getter(getter(HIT::hitReviewStatusAsString)).setter(setter(Builder::hitReviewStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HITReviewStatus").build()).build();

    private static final SdkField<Integer> NUMBER_OF_ASSIGNMENTS_PENDING_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("NumberOfAssignmentsPending")
            .getter(getter(HIT::numberOfAssignmentsPending))
            .setter(setter(Builder::numberOfAssignmentsPending))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberOfAssignmentsPending").build())
            .build();

    private static final SdkField<Integer> NUMBER_OF_ASSIGNMENTS_AVAILABLE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("NumberOfAssignmentsAvailable")
            .getter(getter(HIT::numberOfAssignmentsAvailable))
            .setter(setter(Builder::numberOfAssignmentsAvailable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberOfAssignmentsAvailable")
                    .build()).build();

    private static final SdkField<Integer> NUMBER_OF_ASSIGNMENTS_COMPLETED_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("NumberOfAssignmentsCompleted")
            .getter(getter(HIT::numberOfAssignmentsCompleted))
            .setter(setter(Builder::numberOfAssignmentsCompleted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberOfAssignmentsCompleted")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(HIT_ID_FIELD,
            HIT_TYPE_ID_FIELD, HIT_GROUP_ID_FIELD, HIT_LAYOUT_ID_FIELD, CREATION_TIME_FIELD, TITLE_FIELD, DESCRIPTION_FIELD,
            QUESTION_FIELD, KEYWORDS_FIELD, HIT_STATUS_FIELD, MAX_ASSIGNMENTS_FIELD, REWARD_FIELD,
            AUTO_APPROVAL_DELAY_IN_SECONDS_FIELD, EXPIRATION_FIELD, ASSIGNMENT_DURATION_IN_SECONDS_FIELD,
            REQUESTER_ANNOTATION_FIELD, QUALIFICATION_REQUIREMENTS_FIELD, HIT_REVIEW_STATUS_FIELD,
            NUMBER_OF_ASSIGNMENTS_PENDING_FIELD, NUMBER_OF_ASSIGNMENTS_AVAILABLE_FIELD, NUMBER_OF_ASSIGNMENTS_COMPLETED_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String hitId;

    private final String hitTypeId;

    private final String hitGroupId;

    private final String hitLayoutId;

    private final Instant creationTime;

    private final String title;

    private final String description;

    private final String question;

    private final String keywords;

    private final String hitStatus;

    private final Integer maxAssignments;

    private final String reward;

    private final Long autoApprovalDelayInSeconds;

    private final Instant expiration;

    private final Long assignmentDurationInSeconds;

    private final String requesterAnnotation;

    private final List<QualificationRequirement> qualificationRequirements;

    private final String hitReviewStatus;

    private final Integer numberOfAssignmentsPending;

    private final Integer numberOfAssignmentsAvailable;

    private final Integer numberOfAssignmentsCompleted;

    private HIT(BuilderImpl builder) {
        this.hitId = builder.hitId;
        this.hitTypeId = builder.hitTypeId;
        this.hitGroupId = builder.hitGroupId;
        this.hitLayoutId = builder.hitLayoutId;
        this.creationTime = builder.creationTime;
        this.title = builder.title;
        this.description = builder.description;
        this.question = builder.question;
        this.keywords = builder.keywords;
        this.hitStatus = builder.hitStatus;
        this.maxAssignments = builder.maxAssignments;
        this.reward = builder.reward;
        this.autoApprovalDelayInSeconds = builder.autoApprovalDelayInSeconds;
        this.expiration = builder.expiration;
        this.assignmentDurationInSeconds = builder.assignmentDurationInSeconds;
        this.requesterAnnotation = builder.requesterAnnotation;
        this.qualificationRequirements = builder.qualificationRequirements;
        this.hitReviewStatus = builder.hitReviewStatus;
        this.numberOfAssignmentsPending = builder.numberOfAssignmentsPending;
        this.numberOfAssignmentsAvailable = builder.numberOfAssignmentsAvailable;
        this.numberOfAssignmentsCompleted = builder.numberOfAssignmentsCompleted;
    }

    /**
     * <p>
     * A unique identifier for the HIT.
     * </p>
     * 
     * @return A unique identifier for the HIT.
     */
    public final String hitId() {
        return hitId;
    }

    /**
     * <p>
     * The ID of the HIT type of this HIT
     * </p>
     * 
     * @return The ID of the HIT type of this HIT
     */
    public final String hitTypeId() {
        return hitTypeId;
    }

    /**
     * <p>
     * The ID of the HIT Group of this HIT.
     * </p>
     * 
     * @return The ID of the HIT Group of this HIT.
     */
    public final String hitGroupId() {
        return hitGroupId;
    }

    /**
     * <p>
     * The ID of the HIT Layout of this HIT.
     * </p>
     * 
     * @return The ID of the HIT Layout of this HIT.
     */
    public final String hitLayoutId() {
        return hitLayoutId;
    }

    /**
     * <p>
     * The date and time the HIT was created.
     * </p>
     * 
     * @return The date and time the HIT was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The title of the HIT.
     * </p>
     * 
     * @return The title of the HIT.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * A general description of the HIT.
     * </p>
     * 
     * @return A general description of the HIT.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The data the Worker completing the HIT uses produce the results. This is either either a QuestionForm,
     * HTMLQuestion or an ExternalQuestion data structure.
     * </p>
     * 
     * @return The data the Worker completing the HIT uses produce the results. This is either either a QuestionForm,
     *         HTMLQuestion or an ExternalQuestion data structure.
     */
    public final String question() {
        return question;
    }

    /**
     * <p>
     * One or more words or phrases that describe the HIT, separated by commas. Search terms similar to the keywords of
     * a HIT are more likely to have the HIT in the search results.
     * </p>
     * 
     * @return One or more words or phrases that describe the HIT, separated by commas. Search terms similar to the
     *         keywords of a HIT are more likely to have the HIT in the search results.
     */
    public final String keywords() {
        return keywords;
    }

    /**
     * <p>
     * The status of the HIT and its assignments. Valid Values are Assignable | Unassignable | Reviewable | Reviewing |
     * Disposed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #hitStatus} will
     * return {@link HITStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #hitStatusAsString}.
     * </p>
     * 
     * @return The status of the HIT and its assignments. Valid Values are Assignable | Unassignable | Reviewable |
     *         Reviewing | Disposed.
     * @see HITStatus
     */
    public final HITStatus hitStatus() {
        return HITStatus.fromValue(hitStatus);
    }

    /**
     * <p>
     * The status of the HIT and its assignments. Valid Values are Assignable | Unassignable | Reviewable | Reviewing |
     * Disposed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #hitStatus} will
     * return {@link HITStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #hitStatusAsString}.
     * </p>
     * 
     * @return The status of the HIT and its assignments. Valid Values are Assignable | Unassignable | Reviewable |
     *         Reviewing | Disposed.
     * @see HITStatus
     */
    public final String hitStatusAsString() {
        return hitStatus;
    }

    /**
     * <p>
     * The number of times the HIT can be accepted and completed before the HIT becomes unavailable.
     * </p>
     * 
     * @return The number of times the HIT can be accepted and completed before the HIT becomes unavailable.
     */
    public final Integer maxAssignments() {
        return maxAssignments;
    }

    /**
     * Returns the value of the Reward property for this object.
     * 
     * @return The value of the Reward property for this object.
     */
    public final String reward() {
        return reward;
    }

    /**
     * <p>
     * The amount of time, in seconds, after the Worker submits an assignment for the HIT that the results are
     * automatically approved by Amazon Mechanical Turk. This is the amount of time the Requester has to reject an
     * assignment submitted by a Worker before the assignment is auto-approved and the Worker is paid.
     * </p>
     * 
     * @return The amount of time, in seconds, after the Worker submits an assignment for the HIT that the results are
     *         automatically approved by Amazon Mechanical Turk. This is the amount of time the Requester has to reject
     *         an assignment submitted by a Worker before the assignment is auto-approved and the Worker is paid.
     */
    public final Long autoApprovalDelayInSeconds() {
        return autoApprovalDelayInSeconds;
    }

    /**
     * <p>
     * The date and time the HIT expires.
     * </p>
     * 
     * @return The date and time the HIT expires.
     */
    public final Instant expiration() {
        return expiration;
    }

    /**
     * <p>
     * The length of time, in seconds, that a Worker has to complete the HIT after accepting it.
     * </p>
     * 
     * @return The length of time, in seconds, that a Worker has to complete the HIT after accepting it.
     */
    public final Long assignmentDurationInSeconds() {
        return assignmentDurationInSeconds;
    }

    /**
     * <p>
     * An arbitrary data field the Requester who created the HIT can use. This field is visible only to the creator of
     * the HIT.
     * </p>
     * 
     * @return An arbitrary data field the Requester who created the HIT can use. This field is visible only to the
     *         creator of the HIT.
     */
    public final String requesterAnnotation() {
        return requesterAnnotation;
    }

    /**
     * For responses, this returns true if the service returned a value for the QualificationRequirements property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasQualificationRequirements() {
        return qualificationRequirements != null && !(qualificationRequirements instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Conditions that a Worker's Qualifications must meet in order to accept the HIT. A HIT can have between zero and
     * ten Qualification requirements. All requirements must be met in order for a Worker to accept the HIT.
     * Additionally, other actions can be restricted using the <code>ActionsGuarded</code> field on each
     * <code>QualificationRequirement</code> structure.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasQualificationRequirements} method.
     * </p>
     * 
     * @return Conditions that a Worker's Qualifications must meet in order to accept the HIT. A HIT can have between
     *         zero and ten Qualification requirements. All requirements must be met in order for a Worker to accept the
     *         HIT. Additionally, other actions can be restricted using the <code>ActionsGuarded</code> field on each
     *         <code>QualificationRequirement</code> structure.
     */
    public final List<QualificationRequirement> qualificationRequirements() {
        return qualificationRequirements;
    }

    /**
     * <p>
     * Indicates the review status of the HIT. Valid Values are NotReviewed | MarkedForReview | ReviewedAppropriate |
     * ReviewedInappropriate.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #hitReviewStatus}
     * will return {@link HITReviewStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #hitReviewStatusAsString}.
     * </p>
     * 
     * @return Indicates the review status of the HIT. Valid Values are NotReviewed | MarkedForReview |
     *         ReviewedAppropriate | ReviewedInappropriate.
     * @see HITReviewStatus
     */
    public final HITReviewStatus hitReviewStatus() {
        return HITReviewStatus.fromValue(hitReviewStatus);
    }

    /**
     * <p>
     * Indicates the review status of the HIT. Valid Values are NotReviewed | MarkedForReview | ReviewedAppropriate |
     * ReviewedInappropriate.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #hitReviewStatus}
     * will return {@link HITReviewStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #hitReviewStatusAsString}.
     * </p>
     * 
     * @return Indicates the review status of the HIT. Valid Values are NotReviewed | MarkedForReview |
     *         ReviewedAppropriate | ReviewedInappropriate.
     * @see HITReviewStatus
     */
    public final String hitReviewStatusAsString() {
        return hitReviewStatus;
    }

    /**
     * <p>
     * The number of assignments for this HIT that are being previewed or have been accepted by Workers, but have not
     * yet been submitted, returned, or abandoned.
     * </p>
     * 
     * @return The number of assignments for this HIT that are being previewed or have been accepted by Workers, but
     *         have not yet been submitted, returned, or abandoned.
     */
    public final Integer numberOfAssignmentsPending() {
        return numberOfAssignmentsPending;
    }

    /**
     * <p>
     * The number of assignments for this HIT that are available for Workers to accept.
     * </p>
     * 
     * @return The number of assignments for this HIT that are available for Workers to accept.
     */
    public final Integer numberOfAssignmentsAvailable() {
        return numberOfAssignmentsAvailable;
    }

    /**
     * <p>
     * The number of assignments for this HIT that have been approved or rejected.
     * </p>
     * 
     * @return The number of assignments for this HIT that have been approved or rejected.
     */
    public final Integer numberOfAssignmentsCompleted() {
        return numberOfAssignmentsCompleted;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hitId());
        hashCode = 31 * hashCode + Objects.hashCode(hitTypeId());
        hashCode = 31 * hashCode + Objects.hashCode(hitGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(hitLayoutId());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(question());
        hashCode = 31 * hashCode + Objects.hashCode(keywords());
        hashCode = 31 * hashCode + Objects.hashCode(hitStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maxAssignments());
        hashCode = 31 * hashCode + Objects.hashCode(reward());
        hashCode = 31 * hashCode + Objects.hashCode(autoApprovalDelayInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(expiration());
        hashCode = 31 * hashCode + Objects.hashCode(assignmentDurationInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(requesterAnnotation());
        hashCode = 31 * hashCode + Objects.hashCode(hasQualificationRequirements() ? qualificationRequirements() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hitReviewStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfAssignmentsPending());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfAssignmentsAvailable());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfAssignmentsCompleted());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HIT)) {
            return false;
        }
        HIT other = (HIT) obj;
        return Objects.equals(hitId(), other.hitId()) && Objects.equals(hitTypeId(), other.hitTypeId())
                && Objects.equals(hitGroupId(), other.hitGroupId()) && Objects.equals(hitLayoutId(), other.hitLayoutId())
                && Objects.equals(creationTime(), other.creationTime()) && Objects.equals(title(), other.title())
                && Objects.equals(description(), other.description()) && Objects.equals(question(), other.question())
                && Objects.equals(keywords(), other.keywords()) && Objects.equals(hitStatusAsString(), other.hitStatusAsString())
                && Objects.equals(maxAssignments(), other.maxAssignments()) && Objects.equals(reward(), other.reward())
                && Objects.equals(autoApprovalDelayInSeconds(), other.autoApprovalDelayInSeconds())
                && Objects.equals(expiration(), other.expiration())
                && Objects.equals(assignmentDurationInSeconds(), other.assignmentDurationInSeconds())
                && Objects.equals(requesterAnnotation(), other.requesterAnnotation())
                && hasQualificationRequirements() == other.hasQualificationRequirements()
                && Objects.equals(qualificationRequirements(), other.qualificationRequirements())
                && Objects.equals(hitReviewStatusAsString(), other.hitReviewStatusAsString())
                && Objects.equals(numberOfAssignmentsPending(), other.numberOfAssignmentsPending())
                && Objects.equals(numberOfAssignmentsAvailable(), other.numberOfAssignmentsAvailable())
                && Objects.equals(numberOfAssignmentsCompleted(), other.numberOfAssignmentsCompleted());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("HIT").add("HITId", hitId()).add("HITTypeId", hitTypeId()).add("HITGroupId", hitGroupId())
                .add("HITLayoutId", hitLayoutId()).add("CreationTime", creationTime()).add("Title", title())
                .add("Description", description()).add("Question", question()).add("Keywords", keywords())
                .add("HITStatus", hitStatusAsString()).add("MaxAssignments", maxAssignments()).add("Reward", reward())
                .add("AutoApprovalDelayInSeconds", autoApprovalDelayInSeconds()).add("Expiration", expiration())
                .add("AssignmentDurationInSeconds", assignmentDurationInSeconds())
                .add("RequesterAnnotation", requesterAnnotation())
                .add("QualificationRequirements", hasQualificationRequirements() ? qualificationRequirements() : null)
                .add("HITReviewStatus", hitReviewStatusAsString())
                .add("NumberOfAssignmentsPending", numberOfAssignmentsPending())
                .add("NumberOfAssignmentsAvailable", numberOfAssignmentsAvailable())
                .add("NumberOfAssignmentsCompleted", numberOfAssignmentsCompleted()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "HITId":
            return Optional.ofNullable(clazz.cast(hitId()));
        case "HITTypeId":
            return Optional.ofNullable(clazz.cast(hitTypeId()));
        case "HITGroupId":
            return Optional.ofNullable(clazz.cast(hitGroupId()));
        case "HITLayoutId":
            return Optional.ofNullable(clazz.cast(hitLayoutId()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Question":
            return Optional.ofNullable(clazz.cast(question()));
        case "Keywords":
            return Optional.ofNullable(clazz.cast(keywords()));
        case "HITStatus":
            return Optional.ofNullable(clazz.cast(hitStatusAsString()));
        case "MaxAssignments":
            return Optional.ofNullable(clazz.cast(maxAssignments()));
        case "Reward":
            return Optional.ofNullable(clazz.cast(reward()));
        case "AutoApprovalDelayInSeconds":
            return Optional.ofNullable(clazz.cast(autoApprovalDelayInSeconds()));
        case "Expiration":
            return Optional.ofNullable(clazz.cast(expiration()));
        case "AssignmentDurationInSeconds":
            return Optional.ofNullable(clazz.cast(assignmentDurationInSeconds()));
        case "RequesterAnnotation":
            return Optional.ofNullable(clazz.cast(requesterAnnotation()));
        case "QualificationRequirements":
            return Optional.ofNullable(clazz.cast(qualificationRequirements()));
        case "HITReviewStatus":
            return Optional.ofNullable(clazz.cast(hitReviewStatusAsString()));
        case "NumberOfAssignmentsPending":
            return Optional.ofNullable(clazz.cast(numberOfAssignmentsPending()));
        case "NumberOfAssignmentsAvailable":
            return Optional.ofNullable(clazz.cast(numberOfAssignmentsAvailable()));
        case "NumberOfAssignmentsCompleted":
            return Optional.ofNullable(clazz.cast(numberOfAssignmentsCompleted()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("HITId", HIT_ID_FIELD);
        map.put("HITTypeId", HIT_TYPE_ID_FIELD);
        map.put("HITGroupId", HIT_GROUP_ID_FIELD);
        map.put("HITLayoutId", HIT_LAYOUT_ID_FIELD);
        map.put("CreationTime", CREATION_TIME_FIELD);
        map.put("Title", TITLE_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("Question", QUESTION_FIELD);
        map.put("Keywords", KEYWORDS_FIELD);
        map.put("HITStatus", HIT_STATUS_FIELD);
        map.put("MaxAssignments", MAX_ASSIGNMENTS_FIELD);
        map.put("Reward", REWARD_FIELD);
        map.put("AutoApprovalDelayInSeconds", AUTO_APPROVAL_DELAY_IN_SECONDS_FIELD);
        map.put("Expiration", EXPIRATION_FIELD);
        map.put("AssignmentDurationInSeconds", ASSIGNMENT_DURATION_IN_SECONDS_FIELD);
        map.put("RequesterAnnotation", REQUESTER_ANNOTATION_FIELD);
        map.put("QualificationRequirements", QUALIFICATION_REQUIREMENTS_FIELD);
        map.put("HITReviewStatus", HIT_REVIEW_STATUS_FIELD);
        map.put("NumberOfAssignmentsPending", NUMBER_OF_ASSIGNMENTS_PENDING_FIELD);
        map.put("NumberOfAssignmentsAvailable", NUMBER_OF_ASSIGNMENTS_AVAILABLE_FIELD);
        map.put("NumberOfAssignmentsCompleted", NUMBER_OF_ASSIGNMENTS_COMPLETED_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<HIT, T> g) {
        return obj -> g.apply((HIT) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HIT> {
        /**
         * <p>
         * A unique identifier for the HIT.
         * </p>
         * 
         * @param hitId
         *        A unique identifier for the HIT.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hitId(String hitId);

        /**
         * <p>
         * The ID of the HIT type of this HIT
         * </p>
         * 
         * @param hitTypeId
         *        The ID of the HIT type of this HIT
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hitTypeId(String hitTypeId);

        /**
         * <p>
         * The ID of the HIT Group of this HIT.
         * </p>
         * 
         * @param hitGroupId
         *        The ID of the HIT Group of this HIT.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hitGroupId(String hitGroupId);

        /**
         * <p>
         * The ID of the HIT Layout of this HIT.
         * </p>
         * 
         * @param hitLayoutId
         *        The ID of the HIT Layout of this HIT.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hitLayoutId(String hitLayoutId);

        /**
         * <p>
         * The date and time the HIT was created.
         * </p>
         * 
         * @param creationTime
         *        The date and time the HIT was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The title of the HIT.
         * </p>
         * 
         * @param title
         *        The title of the HIT.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * A general description of the HIT.
         * </p>
         * 
         * @param description
         *        A general description of the HIT.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The data the Worker completing the HIT uses produce the results. This is either either a QuestionForm,
         * HTMLQuestion or an ExternalQuestion data structure.
         * </p>
         * 
         * @param question
         *        The data the Worker completing the HIT uses produce the results. This is either either a QuestionForm,
         *        HTMLQuestion or an ExternalQuestion data structure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder question(String question);

        /**
         * <p>
         * One or more words or phrases that describe the HIT, separated by commas. Search terms similar to the keywords
         * of a HIT are more likely to have the HIT in the search results.
         * </p>
         * 
         * @param keywords
         *        One or more words or phrases that describe the HIT, separated by commas. Search terms similar to the
         *        keywords of a HIT are more likely to have the HIT in the search results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keywords(String keywords);

        /**
         * <p>
         * The status of the HIT and its assignments. Valid Values are Assignable | Unassignable | Reviewable |
         * Reviewing | Disposed.
         * </p>
         * 
         * @param hitStatus
         *        The status of the HIT and its assignments. Valid Values are Assignable | Unassignable | Reviewable |
         *        Reviewing | Disposed.
         * @see HITStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HITStatus
         */
        Builder hitStatus(String hitStatus);

        /**
         * <p>
         * The status of the HIT and its assignments. Valid Values are Assignable | Unassignable | Reviewable |
         * Reviewing | Disposed.
         * </p>
         * 
         * @param hitStatus
         *        The status of the HIT and its assignments. Valid Values are Assignable | Unassignable | Reviewable |
         *        Reviewing | Disposed.
         * @see HITStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HITStatus
         */
        Builder hitStatus(HITStatus hitStatus);

        /**
         * <p>
         * The number of times the HIT can be accepted and completed before the HIT becomes unavailable.
         * </p>
         * 
         * @param maxAssignments
         *        The number of times the HIT can be accepted and completed before the HIT becomes unavailable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxAssignments(Integer maxAssignments);

        /**
         * Sets the value of the Reward property for this object.
         *
         * @param reward
         *        The new value for the Reward property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reward(String reward);

        /**
         * <p>
         * The amount of time, in seconds, after the Worker submits an assignment for the HIT that the results are
         * automatically approved by Amazon Mechanical Turk. This is the amount of time the Requester has to reject an
         * assignment submitted by a Worker before the assignment is auto-approved and the Worker is paid.
         * </p>
         * 
         * @param autoApprovalDelayInSeconds
         *        The amount of time, in seconds, after the Worker submits an assignment for the HIT that the results
         *        are automatically approved by Amazon Mechanical Turk. This is the amount of time the Requester has to
         *        reject an assignment submitted by a Worker before the assignment is auto-approved and the Worker is
         *        paid.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoApprovalDelayInSeconds(Long autoApprovalDelayInSeconds);

        /**
         * <p>
         * The date and time the HIT expires.
         * </p>
         * 
         * @param expiration
         *        The date and time the HIT expires.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expiration(Instant expiration);

        /**
         * <p>
         * The length of time, in seconds, that a Worker has to complete the HIT after accepting it.
         * </p>
         * 
         * @param assignmentDurationInSeconds
         *        The length of time, in seconds, that a Worker has to complete the HIT after accepting it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assignmentDurationInSeconds(Long assignmentDurationInSeconds);

        /**
         * <p>
         * An arbitrary data field the Requester who created the HIT can use. This field is visible only to the creator
         * of the HIT.
         * </p>
         * 
         * @param requesterAnnotation
         *        An arbitrary data field the Requester who created the HIT can use. This field is visible only to the
         *        creator of the HIT.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requesterAnnotation(String requesterAnnotation);

        /**
         * <p>
         * Conditions that a Worker's Qualifications must meet in order to accept the HIT. A HIT can have between zero
         * and ten Qualification requirements. All requirements must be met in order for a Worker to accept the HIT.
         * Additionally, other actions can be restricted using the <code>ActionsGuarded</code> field on each
         * <code>QualificationRequirement</code> structure.
         * </p>
         * 
         * @param qualificationRequirements
         *        Conditions that a Worker's Qualifications must meet in order to accept the HIT. A HIT can have between
         *        zero and ten Qualification requirements. All requirements must be met in order for a Worker to accept
         *        the HIT. Additionally, other actions can be restricted using the <code>ActionsGuarded</code> field on
         *        each <code>QualificationRequirement</code> structure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder qualificationRequirements(Collection<QualificationRequirement> qualificationRequirements);

        /**
         * <p>
         * Conditions that a Worker's Qualifications must meet in order to accept the HIT. A HIT can have between zero
         * and ten Qualification requirements. All requirements must be met in order for a Worker to accept the HIT.
         * Additionally, other actions can be restricted using the <code>ActionsGuarded</code> field on each
         * <code>QualificationRequirement</code> structure.
         * </p>
         * 
         * @param qualificationRequirements
         *        Conditions that a Worker's Qualifications must meet in order to accept the HIT. A HIT can have between
         *        zero and ten Qualification requirements. All requirements must be met in order for a Worker to accept
         *        the HIT. Additionally, other actions can be restricted using the <code>ActionsGuarded</code> field on
         *        each <code>QualificationRequirement</code> structure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder qualificationRequirements(QualificationRequirement... qualificationRequirements);

        /**
         * <p>
         * Conditions that a Worker's Qualifications must meet in order to accept the HIT. A HIT can have between zero
         * and ten Qualification requirements. All requirements must be met in order for a Worker to accept the HIT.
         * Additionally, other actions can be restricted using the <code>ActionsGuarded</code> field on each
         * <code>QualificationRequirement</code> structure.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.mturk.model.QualificationRequirement.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.mturk.model.QualificationRequirement#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.mturk.model.QualificationRequirement.Builder#build()} is called
         * immediately and its result is passed to {@link #qualificationRequirements(List<QualificationRequirement>)}.
         * 
         * @param qualificationRequirements
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.mturk.model.QualificationRequirement.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #qualificationRequirements(java.util.Collection<QualificationRequirement>)
         */
        Builder qualificationRequirements(Consumer<QualificationRequirement.Builder>... qualificationRequirements);

        /**
         * <p>
         * Indicates the review status of the HIT. Valid Values are NotReviewed | MarkedForReview | ReviewedAppropriate
         * | ReviewedInappropriate.
         * </p>
         * 
         * @param hitReviewStatus
         *        Indicates the review status of the HIT. Valid Values are NotReviewed | MarkedForReview |
         *        ReviewedAppropriate | ReviewedInappropriate.
         * @see HITReviewStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HITReviewStatus
         */
        Builder hitReviewStatus(String hitReviewStatus);

        /**
         * <p>
         * Indicates the review status of the HIT. Valid Values are NotReviewed | MarkedForReview | ReviewedAppropriate
         * | ReviewedInappropriate.
         * </p>
         * 
         * @param hitReviewStatus
         *        Indicates the review status of the HIT. Valid Values are NotReviewed | MarkedForReview |
         *        ReviewedAppropriate | ReviewedInappropriate.
         * @see HITReviewStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HITReviewStatus
         */
        Builder hitReviewStatus(HITReviewStatus hitReviewStatus);

        /**
         * <p>
         * The number of assignments for this HIT that are being previewed or have been accepted by Workers, but have
         * not yet been submitted, returned, or abandoned.
         * </p>
         * 
         * @param numberOfAssignmentsPending
         *        The number of assignments for this HIT that are being previewed or have been accepted by Workers, but
         *        have not yet been submitted, returned, or abandoned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfAssignmentsPending(Integer numberOfAssignmentsPending);

        /**
         * <p>
         * The number of assignments for this HIT that are available for Workers to accept.
         * </p>
         * 
         * @param numberOfAssignmentsAvailable
         *        The number of assignments for this HIT that are available for Workers to accept.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfAssignmentsAvailable(Integer numberOfAssignmentsAvailable);

        /**
         * <p>
         * The number of assignments for this HIT that have been approved or rejected.
         * </p>
         * 
         * @param numberOfAssignmentsCompleted
         *        The number of assignments for this HIT that have been approved or rejected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfAssignmentsCompleted(Integer numberOfAssignmentsCompleted);
    }

    static final class BuilderImpl implements Builder {
        private String hitId;

        private String hitTypeId;

        private String hitGroupId;

        private String hitLayoutId;

        private Instant creationTime;

        private String title;

        private String description;

        private String question;

        private String keywords;

        private String hitStatus;

        private Integer maxAssignments;

        private String reward;

        private Long autoApprovalDelayInSeconds;

        private Instant expiration;

        private Long assignmentDurationInSeconds;

        private String requesterAnnotation;

        private List<QualificationRequirement> qualificationRequirements = DefaultSdkAutoConstructList.getInstance();

        private String hitReviewStatus;

        private Integer numberOfAssignmentsPending;

        private Integer numberOfAssignmentsAvailable;

        private Integer numberOfAssignmentsCompleted;

        private BuilderImpl() {
        }

        private BuilderImpl(HIT model) {
            hitId(model.hitId);
            hitTypeId(model.hitTypeId);
            hitGroupId(model.hitGroupId);
            hitLayoutId(model.hitLayoutId);
            creationTime(model.creationTime);
            title(model.title);
            description(model.description);
            question(model.question);
            keywords(model.keywords);
            hitStatus(model.hitStatus);
            maxAssignments(model.maxAssignments);
            reward(model.reward);
            autoApprovalDelayInSeconds(model.autoApprovalDelayInSeconds);
            expiration(model.expiration);
            assignmentDurationInSeconds(model.assignmentDurationInSeconds);
            requesterAnnotation(model.requesterAnnotation);
            qualificationRequirements(model.qualificationRequirements);
            hitReviewStatus(model.hitReviewStatus);
            numberOfAssignmentsPending(model.numberOfAssignmentsPending);
            numberOfAssignmentsAvailable(model.numberOfAssignmentsAvailable);
            numberOfAssignmentsCompleted(model.numberOfAssignmentsCompleted);
        }

        public final String getHitId() {
            return hitId;
        }

        public final void setHitId(String hitId) {
            this.hitId = hitId;
        }

        @Override
        public final Builder hitId(String hitId) {
            this.hitId = hitId;
            return this;
        }

        public final String getHitTypeId() {
            return hitTypeId;
        }

        public final void setHitTypeId(String hitTypeId) {
            this.hitTypeId = hitTypeId;
        }

        @Override
        public final Builder hitTypeId(String hitTypeId) {
            this.hitTypeId = hitTypeId;
            return this;
        }

        public final String getHitGroupId() {
            return hitGroupId;
        }

        public final void setHitGroupId(String hitGroupId) {
            this.hitGroupId = hitGroupId;
        }

        @Override
        public final Builder hitGroupId(String hitGroupId) {
            this.hitGroupId = hitGroupId;
            return this;
        }

        public final String getHitLayoutId() {
            return hitLayoutId;
        }

        public final void setHitLayoutId(String hitLayoutId) {
            this.hitLayoutId = hitLayoutId;
        }

        @Override
        public final Builder hitLayoutId(String hitLayoutId) {
            this.hitLayoutId = hitLayoutId;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getQuestion() {
            return question;
        }

        public final void setQuestion(String question) {
            this.question = question;
        }

        @Override
        public final Builder question(String question) {
            this.question = question;
            return this;
        }

        public final String getKeywords() {
            return keywords;
        }

        public final void setKeywords(String keywords) {
            this.keywords = keywords;
        }

        @Override
        public final Builder keywords(String keywords) {
            this.keywords = keywords;
            return this;
        }

        public final String getHitStatus() {
            return hitStatus;
        }

        public final void setHitStatus(String hitStatus) {
            this.hitStatus = hitStatus;
        }

        @Override
        public final Builder hitStatus(String hitStatus) {
            this.hitStatus = hitStatus;
            return this;
        }

        @Override
        public final Builder hitStatus(HITStatus hitStatus) {
            this.hitStatus(hitStatus == null ? null : hitStatus.toString());
            return this;
        }

        public final Integer getMaxAssignments() {
            return maxAssignments;
        }

        public final void setMaxAssignments(Integer maxAssignments) {
            this.maxAssignments = maxAssignments;
        }

        @Override
        public final Builder maxAssignments(Integer maxAssignments) {
            this.maxAssignments = maxAssignments;
            return this;
        }

        public final String getReward() {
            return reward;
        }

        public final void setReward(String reward) {
            this.reward = reward;
        }

        @Override
        public final Builder reward(String reward) {
            this.reward = reward;
            return this;
        }

        public final Long getAutoApprovalDelayInSeconds() {
            return autoApprovalDelayInSeconds;
        }

        public final void setAutoApprovalDelayInSeconds(Long autoApprovalDelayInSeconds) {
            this.autoApprovalDelayInSeconds = autoApprovalDelayInSeconds;
        }

        @Override
        public final Builder autoApprovalDelayInSeconds(Long autoApprovalDelayInSeconds) {
            this.autoApprovalDelayInSeconds = autoApprovalDelayInSeconds;
            return this;
        }

        public final Instant getExpiration() {
            return expiration;
        }

        public final void setExpiration(Instant expiration) {
            this.expiration = expiration;
        }

        @Override
        public final Builder expiration(Instant expiration) {
            this.expiration = expiration;
            return this;
        }

        public final Long getAssignmentDurationInSeconds() {
            return assignmentDurationInSeconds;
        }

        public final void setAssignmentDurationInSeconds(Long assignmentDurationInSeconds) {
            this.assignmentDurationInSeconds = assignmentDurationInSeconds;
        }

        @Override
        public final Builder assignmentDurationInSeconds(Long assignmentDurationInSeconds) {
            this.assignmentDurationInSeconds = assignmentDurationInSeconds;
            return this;
        }

        public final String getRequesterAnnotation() {
            return requesterAnnotation;
        }

        public final void setRequesterAnnotation(String requesterAnnotation) {
            this.requesterAnnotation = requesterAnnotation;
        }

        @Override
        public final Builder requesterAnnotation(String requesterAnnotation) {
            this.requesterAnnotation = requesterAnnotation;
            return this;
        }

        public final List<QualificationRequirement.Builder> getQualificationRequirements() {
            List<QualificationRequirement.Builder> result = QualificationRequirementListCopier
                    .copyToBuilder(this.qualificationRequirements);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setQualificationRequirements(Collection<QualificationRequirement.BuilderImpl> qualificationRequirements) {
            this.qualificationRequirements = QualificationRequirementListCopier.copyFromBuilder(qualificationRequirements);
        }

        @Override
        public final Builder qualificationRequirements(Collection<QualificationRequirement> qualificationRequirements) {
            this.qualificationRequirements = QualificationRequirementListCopier.copy(qualificationRequirements);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder qualificationRequirements(QualificationRequirement... qualificationRequirements) {
            qualificationRequirements(Arrays.asList(qualificationRequirements));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder qualificationRequirements(Consumer<QualificationRequirement.Builder>... qualificationRequirements) {
            qualificationRequirements(Stream.of(qualificationRequirements)
                    .map(c -> QualificationRequirement.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getHitReviewStatus() {
            return hitReviewStatus;
        }

        public final void setHitReviewStatus(String hitReviewStatus) {
            this.hitReviewStatus = hitReviewStatus;
        }

        @Override
        public final Builder hitReviewStatus(String hitReviewStatus) {
            this.hitReviewStatus = hitReviewStatus;
            return this;
        }

        @Override
        public final Builder hitReviewStatus(HITReviewStatus hitReviewStatus) {
            this.hitReviewStatus(hitReviewStatus == null ? null : hitReviewStatus.toString());
            return this;
        }

        public final Integer getNumberOfAssignmentsPending() {
            return numberOfAssignmentsPending;
        }

        public final void setNumberOfAssignmentsPending(Integer numberOfAssignmentsPending) {
            this.numberOfAssignmentsPending = numberOfAssignmentsPending;
        }

        @Override
        public final Builder numberOfAssignmentsPending(Integer numberOfAssignmentsPending) {
            this.numberOfAssignmentsPending = numberOfAssignmentsPending;
            return this;
        }

        public final Integer getNumberOfAssignmentsAvailable() {
            return numberOfAssignmentsAvailable;
        }

        public final void setNumberOfAssignmentsAvailable(Integer numberOfAssignmentsAvailable) {
            this.numberOfAssignmentsAvailable = numberOfAssignmentsAvailable;
        }

        @Override
        public final Builder numberOfAssignmentsAvailable(Integer numberOfAssignmentsAvailable) {
            this.numberOfAssignmentsAvailable = numberOfAssignmentsAvailable;
            return this;
        }

        public final Integer getNumberOfAssignmentsCompleted() {
            return numberOfAssignmentsCompleted;
        }

        public final void setNumberOfAssignmentsCompleted(Integer numberOfAssignmentsCompleted) {
            this.numberOfAssignmentsCompleted = numberOfAssignmentsCompleted;
        }

        @Override
        public final Builder numberOfAssignmentsCompleted(Integer numberOfAssignmentsCompleted) {
            this.numberOfAssignmentsCompleted = numberOfAssignmentsCompleted;
            return this;
        }

        @Override
        public HIT build() {
            return new HIT(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
