/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mturk.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The NotificationSpecification data structure describes a HIT event notification for a HIT type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NotificationSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<NotificationSpecification.Builder, NotificationSpecification> {
    private static final SdkField<String> DESTINATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Destination").getter(getter(NotificationSpecification::destination))
            .setter(setter(Builder::destination))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Destination").build()).build();

    private static final SdkField<String> TRANSPORT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Transport").getter(getter(NotificationSpecification::transportAsString))
            .setter(setter(Builder::transport))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Transport").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Version")
            .getter(getter(NotificationSpecification::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<List<String>> EVENT_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EventTypes")
            .getter(getter(NotificationSpecification::eventTypesAsStrings))
            .setter(setter(Builder::eventTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESTINATION_FIELD,
            TRANSPORT_FIELD, VERSION_FIELD, EVENT_TYPES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String destination;

    private final String transport;

    private final String version;

    private final List<String> eventTypes;

    private NotificationSpecification(BuilderImpl builder) {
        this.destination = builder.destination;
        this.transport = builder.transport;
        this.version = builder.version;
        this.eventTypes = builder.eventTypes;
    }

    /**
     * <p>
     * The target for notification messages. The Destination’s format is determined by the specified Transport:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When Transport is Email, the Destination is your email address.
     * </p>
     * </li>
     * <li>
     * <p>
     * When Transport is SQS, the Destination is your queue URL.
     * </p>
     * </li>
     * <li>
     * <p>
     * When Transport is SNS, the Destination is the ARN of your topic.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The target for notification messages. The Destination’s format is determined by the specified Transport:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         When Transport is Email, the Destination is your email address.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         When Transport is SQS, the Destination is your queue URL.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         When Transport is SNS, the Destination is the ARN of your topic.
     *         </p>
     *         </li>
     */
    public final String destination() {
        return destination;
    }

    /**
     * <p>
     * The method Amazon Mechanical Turk uses to send the notification. Valid Values: Email | SQS | SNS.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #transport} will
     * return {@link NotificationTransport#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #transportAsString}.
     * </p>
     * 
     * @return The method Amazon Mechanical Turk uses to send the notification. Valid Values: Email | SQS | SNS.
     * @see NotificationTransport
     */
    public final NotificationTransport transport() {
        return NotificationTransport.fromValue(transport);
    }

    /**
     * <p>
     * The method Amazon Mechanical Turk uses to send the notification. Valid Values: Email | SQS | SNS.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #transport} will
     * return {@link NotificationTransport#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #transportAsString}.
     * </p>
     * 
     * @return The method Amazon Mechanical Turk uses to send the notification. Valid Values: Email | SQS | SNS.
     * @see NotificationTransport
     */
    public final String transportAsString() {
        return transport;
    }

    /**
     * <p>
     * The version of the Notification API to use. Valid value is 2006-05-05.
     * </p>
     * 
     * @return The version of the Notification API to use. Valid value is 2006-05-05.
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * The list of events that should cause notifications to be sent. Valid Values: AssignmentAccepted |
     * AssignmentAbandoned | AssignmentReturned | AssignmentSubmitted | AssignmentRejected | AssignmentApproved |
     * HITCreated | HITExtended | HITDisposed | HITReviewable | HITExpired | Ping. The Ping event is only valid for the
     * SendTestEventNotification operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEventTypes} method.
     * </p>
     * 
     * @return The list of events that should cause notifications to be sent. Valid Values: AssignmentAccepted |
     *         AssignmentAbandoned | AssignmentReturned | AssignmentSubmitted | AssignmentRejected | AssignmentApproved
     *         | HITCreated | HITExtended | HITDisposed | HITReviewable | HITExpired | Ping. The Ping event is only
     *         valid for the SendTestEventNotification operation.
     */
    public final List<EventType> eventTypes() {
        return EventTypeListCopier.copyStringToEnum(eventTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the EventTypes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEventTypes() {
        return eventTypes != null && !(eventTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of events that should cause notifications to be sent. Valid Values: AssignmentAccepted |
     * AssignmentAbandoned | AssignmentReturned | AssignmentSubmitted | AssignmentRejected | AssignmentApproved |
     * HITCreated | HITExtended | HITDisposed | HITReviewable | HITExpired | Ping. The Ping event is only valid for the
     * SendTestEventNotification operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEventTypes} method.
     * </p>
     * 
     * @return The list of events that should cause notifications to be sent. Valid Values: AssignmentAccepted |
     *         AssignmentAbandoned | AssignmentReturned | AssignmentSubmitted | AssignmentRejected | AssignmentApproved
     *         | HITCreated | HITExtended | HITDisposed | HITReviewable | HITExpired | Ping. The Ping event is only
     *         valid for the SendTestEventNotification operation.
     */
    public final List<String> eventTypesAsStrings() {
        return eventTypes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(destination());
        hashCode = 31 * hashCode + Objects.hashCode(transportAsString());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(hasEventTypes() ? eventTypesAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NotificationSpecification)) {
            return false;
        }
        NotificationSpecification other = (NotificationSpecification) obj;
        return Objects.equals(destination(), other.destination())
                && Objects.equals(transportAsString(), other.transportAsString()) && Objects.equals(version(), other.version())
                && hasEventTypes() == other.hasEventTypes() && Objects.equals(eventTypesAsStrings(), other.eventTypesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NotificationSpecification").add("Destination", destination())
                .add("Transport", transportAsString()).add("Version", version())
                .add("EventTypes", hasEventTypes() ? eventTypesAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Destination":
            return Optional.ofNullable(clazz.cast(destination()));
        case "Transport":
            return Optional.ofNullable(clazz.cast(transportAsString()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "EventTypes":
            return Optional.ofNullable(clazz.cast(eventTypesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Destination", DESTINATION_FIELD);
        map.put("Transport", TRANSPORT_FIELD);
        map.put("Version", VERSION_FIELD);
        map.put("EventTypes", EVENT_TYPES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<NotificationSpecification, T> g) {
        return obj -> g.apply((NotificationSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NotificationSpecification> {
        /**
         * <p>
         * The target for notification messages. The Destination’s format is determined by the specified Transport:
         * </p>
         * <ul>
         * <li>
         * <p>
         * When Transport is Email, the Destination is your email address.
         * </p>
         * </li>
         * <li>
         * <p>
         * When Transport is SQS, the Destination is your queue URL.
         * </p>
         * </li>
         * <li>
         * <p>
         * When Transport is SNS, the Destination is the ARN of your topic.
         * </p>
         * </li>
         * </ul>
         * 
         * @param destination
         *        The target for notification messages. The Destination’s format is determined by the specified
         *        Transport: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        When Transport is Email, the Destination is your email address.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        When Transport is SQS, the Destination is your queue URL.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        When Transport is SNS, the Destination is the ARN of your topic.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destination(String destination);

        /**
         * <p>
         * The method Amazon Mechanical Turk uses to send the notification. Valid Values: Email | SQS | SNS.
         * </p>
         * 
         * @param transport
         *        The method Amazon Mechanical Turk uses to send the notification. Valid Values: Email | SQS | SNS.
         * @see NotificationTransport
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotificationTransport
         */
        Builder transport(String transport);

        /**
         * <p>
         * The method Amazon Mechanical Turk uses to send the notification. Valid Values: Email | SQS | SNS.
         * </p>
         * 
         * @param transport
         *        The method Amazon Mechanical Turk uses to send the notification. Valid Values: Email | SQS | SNS.
         * @see NotificationTransport
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotificationTransport
         */
        Builder transport(NotificationTransport transport);

        /**
         * <p>
         * The version of the Notification API to use. Valid value is 2006-05-05.
         * </p>
         * 
         * @param version
         *        The version of the Notification API to use. Valid value is 2006-05-05.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The list of events that should cause notifications to be sent. Valid Values: AssignmentAccepted |
         * AssignmentAbandoned | AssignmentReturned | AssignmentSubmitted | AssignmentRejected | AssignmentApproved |
         * HITCreated | HITExtended | HITDisposed | HITReviewable | HITExpired | Ping. The Ping event is only valid for
         * the SendTestEventNotification operation.
         * </p>
         * 
         * @param eventTypes
         *        The list of events that should cause notifications to be sent. Valid Values: AssignmentAccepted |
         *        AssignmentAbandoned | AssignmentReturned | AssignmentSubmitted | AssignmentRejected |
         *        AssignmentApproved | HITCreated | HITExtended | HITDisposed | HITReviewable | HITExpired | Ping. The
         *        Ping event is only valid for the SendTestEventNotification operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventTypesWithStrings(Collection<String> eventTypes);

        /**
         * <p>
         * The list of events that should cause notifications to be sent. Valid Values: AssignmentAccepted |
         * AssignmentAbandoned | AssignmentReturned | AssignmentSubmitted | AssignmentRejected | AssignmentApproved |
         * HITCreated | HITExtended | HITDisposed | HITReviewable | HITExpired | Ping. The Ping event is only valid for
         * the SendTestEventNotification operation.
         * </p>
         * 
         * @param eventTypes
         *        The list of events that should cause notifications to be sent. Valid Values: AssignmentAccepted |
         *        AssignmentAbandoned | AssignmentReturned | AssignmentSubmitted | AssignmentRejected |
         *        AssignmentApproved | HITCreated | HITExtended | HITDisposed | HITReviewable | HITExpired | Ping. The
         *        Ping event is only valid for the SendTestEventNotification operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventTypesWithStrings(String... eventTypes);

        /**
         * <p>
         * The list of events that should cause notifications to be sent. Valid Values: AssignmentAccepted |
         * AssignmentAbandoned | AssignmentReturned | AssignmentSubmitted | AssignmentRejected | AssignmentApproved |
         * HITCreated | HITExtended | HITDisposed | HITReviewable | HITExpired | Ping. The Ping event is only valid for
         * the SendTestEventNotification operation.
         * </p>
         * 
         * @param eventTypes
         *        The list of events that should cause notifications to be sent. Valid Values: AssignmentAccepted |
         *        AssignmentAbandoned | AssignmentReturned | AssignmentSubmitted | AssignmentRejected |
         *        AssignmentApproved | HITCreated | HITExtended | HITDisposed | HITReviewable | HITExpired | Ping. The
         *        Ping event is only valid for the SendTestEventNotification operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventTypes(Collection<EventType> eventTypes);

        /**
         * <p>
         * The list of events that should cause notifications to be sent. Valid Values: AssignmentAccepted |
         * AssignmentAbandoned | AssignmentReturned | AssignmentSubmitted | AssignmentRejected | AssignmentApproved |
         * HITCreated | HITExtended | HITDisposed | HITReviewable | HITExpired | Ping. The Ping event is only valid for
         * the SendTestEventNotification operation.
         * </p>
         * 
         * @param eventTypes
         *        The list of events that should cause notifications to be sent. Valid Values: AssignmentAccepted |
         *        AssignmentAbandoned | AssignmentReturned | AssignmentSubmitted | AssignmentRejected |
         *        AssignmentApproved | HITCreated | HITExtended | HITDisposed | HITReviewable | HITExpired | Ping. The
         *        Ping event is only valid for the SendTestEventNotification operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventTypes(EventType... eventTypes);
    }

    static final class BuilderImpl implements Builder {
        private String destination;

        private String transport;

        private String version;

        private List<String> eventTypes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(NotificationSpecification model) {
            destination(model.destination);
            transport(model.transport);
            version(model.version);
            eventTypesWithStrings(model.eventTypes);
        }

        public final String getDestination() {
            return destination;
        }

        public final void setDestination(String destination) {
            this.destination = destination;
        }

        @Override
        public final Builder destination(String destination) {
            this.destination = destination;
            return this;
        }

        public final String getTransport() {
            return transport;
        }

        public final void setTransport(String transport) {
            this.transport = transport;
        }

        @Override
        public final Builder transport(String transport) {
            this.transport = transport;
            return this;
        }

        @Override
        public final Builder transport(NotificationTransport transport) {
            this.transport(transport == null ? null : transport.toString());
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final Collection<String> getEventTypes() {
            if (eventTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return eventTypes;
        }

        public final void setEventTypes(Collection<String> eventTypes) {
            this.eventTypes = EventTypeListCopier.copy(eventTypes);
        }

        @Override
        public final Builder eventTypesWithStrings(Collection<String> eventTypes) {
            this.eventTypes = EventTypeListCopier.copy(eventTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder eventTypesWithStrings(String... eventTypes) {
            eventTypesWithStrings(Arrays.asList(eventTypes));
            return this;
        }

        @Override
        public final Builder eventTypes(Collection<EventType> eventTypes) {
            this.eventTypes = EventTypeListCopier.copyEnumToString(eventTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder eventTypes(EventType... eventTypes) {
            eventTypes(Arrays.asList(eventTypes));
            return this;
        }

        @Override
        public NotificationSpecification build() {
            return new NotificationSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
