/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mturk.paginators;

import java.util.Collections;
import java.util.Iterator;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.mturk.MTurkClient;
import software.amazon.awssdk.services.mturk.model.ListReviewableHiTsRequest;
import software.amazon.awssdk.services.mturk.model.ListReviewableHiTsResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.mturk.MTurkClient#listReviewableHITsPaginator(software.amazon.awssdk.services.mturk.model.ListReviewableHiTsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.mturk.model.ListReviewableHiTsResponse} that can be used to iterate through
 * all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.mturk.paginators.ListReviewableHITsIterable responses = client.listReviewableHITsPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.mturk.paginators.ListReviewableHITsIterable responses = client
 *             .listReviewableHITsPaginator(request);
 *     for (software.amazon.awssdk.services.mturk.model.ListReviewableHiTsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.mturk.paginators.ListReviewableHITsIterable responses = client.listReviewableHITsPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listReviewableHITs(software.amazon.awssdk.services.mturk.model.ListReviewableHiTsRequest)} operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListReviewableHITsIterable implements SdkIterable<ListReviewableHiTsResponse> {
    private final MTurkClient client;

    private final ListReviewableHiTsRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListReviewableHITsIterable(MTurkClient client, ListReviewableHiTsRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new ListReviewableHiTsResponseFetcher();
    }

    @Override
    public Iterator<ListReviewableHiTsResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link ListReviewableHITsIterable} that can be used to retrieve the
     * consecutive pages that follows the input page.
     * </p>
     */
    private final ListReviewableHITsIterable resume(ListReviewableHiTsResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new ListReviewableHITsIterable(client, firstRequest.toBuilder().nextToken(lastSuccessfulPage.nextToken())
                    .build());
        }
        return new ListReviewableHITsIterable(client, firstRequest) {
            @Override
            public Iterator<ListReviewableHiTsResponse> iterator() {
                return Collections.emptyIterator();
            }
        };
    }

    private class ListReviewableHiTsResponseFetcher implements SyncPageFetcher<ListReviewableHiTsResponse> {
        @Override
        public boolean hasNextPage(ListReviewableHiTsResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public ListReviewableHiTsResponse nextPage(ListReviewableHiTsResponse previousPage) {
            if (previousPage == null) {
                return client.listReviewableHITs(firstRequest);
            }
            return client.listReviewableHITs(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
