/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mwaa.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This section contains the Amazon Managed Workflows for Apache Airflow (MWAA) API reference documentation to create an
 * environment. For more information, see <a
 * href="https://docs.aws.amazon.com/mwaa/latest/userguide/get-started.html">Get started with Amazon Managed Workflows
 * for Apache Airflow</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateEnvironmentRequest extends MwaaRequest implements
        ToCopyableBuilder<CreateEnvironmentRequest.Builder, CreateEnvironmentRequest> {
    private static final SdkField<Map<String, String>> AIRFLOW_CONFIGURATION_OPTIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("AirflowConfigurationOptions")
            .getter(getter(CreateEnvironmentRequest::airflowConfigurationOptions))
            .setter(setter(Builder::airflowConfigurationOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AirflowConfigurationOptions")
                    .build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> AIRFLOW_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AirflowVersion").getter(getter(CreateEnvironmentRequest::airflowVersion))
            .setter(setter(Builder::airflowVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AirflowVersion").build()).build();

    private static final SdkField<String> DAG_S3_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DagS3Path").getter(getter(CreateEnvironmentRequest::dagS3Path)).setter(setter(Builder::dagS3Path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DagS3Path").build()).build();

    private static final SdkField<String> ENVIRONMENT_CLASS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EnvironmentClass").getter(getter(CreateEnvironmentRequest::environmentClass))
            .setter(setter(Builder::environmentClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnvironmentClass").build()).build();

    private static final SdkField<String> EXECUTION_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExecutionRoleArn").getter(getter(CreateEnvironmentRequest::executionRoleArn))
            .setter(setter(Builder::executionRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionRoleArn").build()).build();

    private static final SdkField<String> KMS_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("KmsKey")
            .getter(getter(CreateEnvironmentRequest::kmsKey)).setter(setter(Builder::kmsKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKey").build()).build();

    private static final SdkField<LoggingConfigurationInput> LOGGING_CONFIGURATION_FIELD = SdkField
            .<LoggingConfigurationInput> builder(MarshallingType.SDK_POJO).memberName("LoggingConfiguration")
            .getter(getter(CreateEnvironmentRequest::loggingConfiguration)).setter(setter(Builder::loggingConfiguration))
            .constructor(LoggingConfigurationInput::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LoggingConfiguration").build())
            .build();

    private static final SdkField<Integer> MAX_WORKERS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxWorkers").getter(getter(CreateEnvironmentRequest::maxWorkers)).setter(setter(Builder::maxWorkers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxWorkers").build()).build();

    private static final SdkField<Integer> MIN_WORKERS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MinWorkers").getter(getter(CreateEnvironmentRequest::minWorkers)).setter(setter(Builder::minWorkers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinWorkers").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(CreateEnvironmentRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("Name").build()).build();

    private static final SdkField<NetworkConfiguration> NETWORK_CONFIGURATION_FIELD = SdkField
            .<NetworkConfiguration> builder(MarshallingType.SDK_POJO).memberName("NetworkConfiguration")
            .getter(getter(CreateEnvironmentRequest::networkConfiguration)).setter(setter(Builder::networkConfiguration))
            .constructor(NetworkConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkConfiguration").build())
            .build();

    private static final SdkField<String> PLUGINS_S3_OBJECT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PluginsS3ObjectVersion").getter(getter(CreateEnvironmentRequest::pluginsS3ObjectVersion))
            .setter(setter(Builder::pluginsS3ObjectVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PluginsS3ObjectVersion").build())
            .build();

    private static final SdkField<String> PLUGINS_S3_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PluginsS3Path").getter(getter(CreateEnvironmentRequest::pluginsS3Path))
            .setter(setter(Builder::pluginsS3Path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PluginsS3Path").build()).build();

    private static final SdkField<String> REQUIREMENTS_S3_OBJECT_VERSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RequirementsS3ObjectVersion")
            .getter(getter(CreateEnvironmentRequest::requirementsS3ObjectVersion))
            .setter(setter(Builder::requirementsS3ObjectVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequirementsS3ObjectVersion")
                    .build()).build();

    private static final SdkField<String> REQUIREMENTS_S3_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RequirementsS3Path").getter(getter(CreateEnvironmentRequest::requirementsS3Path))
            .setter(setter(Builder::requirementsS3Path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequirementsS3Path").build())
            .build();

    private static final SdkField<String> SOURCE_BUCKET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceBucketArn").getter(getter(CreateEnvironmentRequest::sourceBucketArn))
            .setter(setter(Builder::sourceBucketArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceBucketArn").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(CreateEnvironmentRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> WEBSERVER_ACCESS_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WebserverAccessMode").getter(getter(CreateEnvironmentRequest::webserverAccessModeAsString))
            .setter(setter(Builder::webserverAccessMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WebserverAccessMode").build())
            .build();

    private static final SdkField<String> WEEKLY_MAINTENANCE_WINDOW_START_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("WeeklyMaintenanceWindowStart")
            .getter(getter(CreateEnvironmentRequest::weeklyMaintenanceWindowStart))
            .setter(setter(Builder::weeklyMaintenanceWindowStart))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WeeklyMaintenanceWindowStart")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            AIRFLOW_CONFIGURATION_OPTIONS_FIELD, AIRFLOW_VERSION_FIELD, DAG_S3_PATH_FIELD, ENVIRONMENT_CLASS_FIELD,
            EXECUTION_ROLE_ARN_FIELD, KMS_KEY_FIELD, LOGGING_CONFIGURATION_FIELD, MAX_WORKERS_FIELD, MIN_WORKERS_FIELD,
            NAME_FIELD, NETWORK_CONFIGURATION_FIELD, PLUGINS_S3_OBJECT_VERSION_FIELD, PLUGINS_S3_PATH_FIELD,
            REQUIREMENTS_S3_OBJECT_VERSION_FIELD, REQUIREMENTS_S3_PATH_FIELD, SOURCE_BUCKET_ARN_FIELD, TAGS_FIELD,
            WEBSERVER_ACCESS_MODE_FIELD, WEEKLY_MAINTENANCE_WINDOW_START_FIELD));

    private final Map<String, String> airflowConfigurationOptions;

    private final String airflowVersion;

    private final String dagS3Path;

    private final String environmentClass;

    private final String executionRoleArn;

    private final String kmsKey;

    private final LoggingConfigurationInput loggingConfiguration;

    private final Integer maxWorkers;

    private final Integer minWorkers;

    private final String name;

    private final NetworkConfiguration networkConfiguration;

    private final String pluginsS3ObjectVersion;

    private final String pluginsS3Path;

    private final String requirementsS3ObjectVersion;

    private final String requirementsS3Path;

    private final String sourceBucketArn;

    private final Map<String, String> tags;

    private final String webserverAccessMode;

    private final String weeklyMaintenanceWindowStart;

    private CreateEnvironmentRequest(BuilderImpl builder) {
        super(builder);
        this.airflowConfigurationOptions = builder.airflowConfigurationOptions;
        this.airflowVersion = builder.airflowVersion;
        this.dagS3Path = builder.dagS3Path;
        this.environmentClass = builder.environmentClass;
        this.executionRoleArn = builder.executionRoleArn;
        this.kmsKey = builder.kmsKey;
        this.loggingConfiguration = builder.loggingConfiguration;
        this.maxWorkers = builder.maxWorkers;
        this.minWorkers = builder.minWorkers;
        this.name = builder.name;
        this.networkConfiguration = builder.networkConfiguration;
        this.pluginsS3ObjectVersion = builder.pluginsS3ObjectVersion;
        this.pluginsS3Path = builder.pluginsS3Path;
        this.requirementsS3ObjectVersion = builder.requirementsS3ObjectVersion;
        this.requirementsS3Path = builder.requirementsS3Path;
        this.sourceBucketArn = builder.sourceBucketArn;
        this.tags = builder.tags;
        this.webserverAccessMode = builder.webserverAccessMode;
        this.weeklyMaintenanceWindowStart = builder.weeklyMaintenanceWindowStart;
    }

    /**
     * Returns true if the AirflowConfigurationOptions property was specified by the sender (it may be empty), or false
     * if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the
     * AWS service.
     */
    public final boolean hasAirflowConfigurationOptions() {
        return airflowConfigurationOptions != null && !(airflowConfigurationOptions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The Apache Airflow configuration setting you want to override in your environment. For more information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-env-variables.html">Environment
     * configuration</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAirflowConfigurationOptions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The Apache Airflow configuration setting you want to override in your environment. For more information,
     *         see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-env-variables.html">Environment
     *         configuration</a>.
     */
    public final Map<String, String> airflowConfigurationOptions() {
        return airflowConfigurationOptions;
    }

    /**
     * <p>
     * The Apache Airflow version you want to use for your environment.
     * </p>
     * 
     * @return The Apache Airflow version you want to use for your environment.
     */
    public final String airflowVersion() {
        return airflowVersion;
    }

    /**
     * <p>
     * The relative path to the DAG folder on your Amazon S3 storage bucket. For example, <code>dags</code>. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html">Importing DAGs on Amazon
     * MWAA</a>.
     * </p>
     * 
     * @return The relative path to the DAG folder on your Amazon S3 storage bucket. For example, <code>dags</code>. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html">Importing DAGs on
     *         Amazon MWAA</a>.
     */
    public final String dagS3Path() {
        return dagS3Path;
    }

    /**
     * <p>
     * The environment class you want to use for your environment. The environment class determines the size of the
     * containers and database used for your Apache Airflow services.
     * </p>
     * 
     * @return The environment class you want to use for your environment. The environment class determines the size of
     *         the containers and database used for your Apache Airflow services.
     */
    public final String environmentClass() {
        return environmentClass;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the execution role for your environment. An execution role is an AWS Identity
     * and Access Management (IAM) role that grants MWAA permission to access AWS services and resources used by your
     * environment. For example, <code>arn:aws:iam::123456789:role/my-execution-role</code>. For more information, see
     * <a href="https://docs.aws.amazon.com/mwaa/latest/userguide/manage-access.html">Managing access to Amazon Managed
     * Workflows for Apache Airflow</a>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the execution role for your environment. An execution role is an AWS
     *         Identity and Access Management (IAM) role that grants MWAA permission to access AWS services and
     *         resources used by your environment. For example,
     *         <code>arn:aws:iam::123456789:role/my-execution-role</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/manage-access.html">Managing access to Amazon
     *         Managed Workflows for Apache Airflow</a>.
     */
    public final String executionRoleArn() {
        return executionRoleArn;
    }

    /**
     * <p>
     * The AWS Key Management Service (KMS) key to encrypt and decrypt the data in your environment. You can use an AWS
     * KMS key managed by MWAA, or a custom KMS key (advanced). For more information, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html?icmpid=docs_console_unmapped#master_keys"
     * >Customer master keys (CMKs)</a> in the AWS KMS developer guide.
     * </p>
     * 
     * @return The AWS Key Management Service (KMS) key to encrypt and decrypt the data in your environment. You can use
     *         an AWS KMS key managed by MWAA, or a custom KMS key (advanced). For more information, see <a href=
     *         "https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html?icmpid=docs_console_unmapped#master_keys"
     *         >Customer master keys (CMKs)</a> in the AWS KMS developer guide.
     */
    public final String kmsKey() {
        return kmsKey;
    }

    /**
     * <p>
     * The Apache Airflow logs you want to send to Amazon CloudWatch Logs.
     * </p>
     * 
     * @return The Apache Airflow logs you want to send to Amazon CloudWatch Logs.
     */
    public final LoggingConfigurationInput loggingConfiguration() {
        return loggingConfiguration;
    }

    /**
     * <p>
     * The maximum number of workers that you want to run in your environment. MWAA scales the number of Apache Airflow
     * workers and the Fargate containers that run your tasks up to the number you specify in this field. When there are
     * no more tasks running, and no more in the queue, MWAA disposes of the extra containers leaving the one worker
     * that is included with your environment.
     * </p>
     * 
     * @return The maximum number of workers that you want to run in your environment. MWAA scales the number of Apache
     *         Airflow workers and the Fargate containers that run your tasks up to the number you specify in this
     *         field. When there are no more tasks running, and no more in the queue, MWAA disposes of the extra
     *         containers leaving the one worker that is included with your environment.
     */
    public final Integer maxWorkers() {
        return maxWorkers;
    }

    /**
     * <p>
     * The minimum number of workers that you want to run in your environment. MWAA scales the number of Apache Airflow
     * workers and the Fargate containers that run your tasks up to the number you specify in the
     * <code>MaxWorkers</code> field. When there are no more tasks running, and no more in the queue, MWAA disposes of
     * the extra containers leaving the worker count you specify in the <code>MinWorkers</code> field.
     * </p>
     * 
     * @return The minimum number of workers that you want to run in your environment. MWAA scales the number of Apache
     *         Airflow workers and the Fargate containers that run your tasks up to the number you specify in the
     *         <code>MaxWorkers</code> field. When there are no more tasks running, and no more in the queue, MWAA
     *         disposes of the extra containers leaving the worker count you specify in the <code>MinWorkers</code>
     *         field.
     */
    public final Integer minWorkers() {
        return minWorkers;
    }

    /**
     * <p>
     * The name of your MWAA environment.
     * </p>
     * 
     * @return The name of your MWAA environment.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The VPC networking components you want to use for your environment. At least two private subnet identifiers and
     * one VPC security group identifier are required to create an environment. For more information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/vpc-mwaa.html">Creating the VPC network for a MWAA
     * environment</a>.
     * </p>
     * 
     * @return The VPC networking components you want to use for your environment. At least two private subnet
     *         identifiers and one VPC security group identifier are required to create an environment. For more
     *         information, see <a href="https://docs.aws.amazon.com/mwaa/latest/userguide/vpc-mwaa.html">Creating the
     *         VPC network for a MWAA environment</a>.
     */
    public final NetworkConfiguration networkConfiguration() {
        return networkConfiguration;
    }

    /**
     * <p>
     * The <code>plugins.zip</code> file version you want to use.
     * </p>
     * 
     * @return The <code>plugins.zip</code> file version you want to use.
     */
    public final String pluginsS3ObjectVersion() {
        return pluginsS3ObjectVersion;
    }

    /**
     * <p>
     * The relative path to the <code>plugins.zip</code> file on your Amazon S3 storage bucket. For example,
     * <code>plugins.zip</code>. If a relative path is provided in the request, then <code>PluginsS3ObjectVersion</code>
     * is required. For more information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html">Importing DAGs on Amazon
     * MWAA</a>.
     * </p>
     * 
     * @return The relative path to the <code>plugins.zip</code> file on your Amazon S3 storage bucket. For example,
     *         <code>plugins.zip</code>. If a relative path is provided in the request, then
     *         <code>PluginsS3ObjectVersion</code> is required. For more information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html">Importing DAGs on
     *         Amazon MWAA</a>.
     */
    public final String pluginsS3Path() {
        return pluginsS3Path;
    }

    /**
     * <p>
     * The <code>requirements.txt</code> file version you want to use.
     * </p>
     * 
     * @return The <code>requirements.txt</code> file version you want to use.
     */
    public final String requirementsS3ObjectVersion() {
        return requirementsS3ObjectVersion;
    }

    /**
     * <p>
     * The relative path to the <code>requirements.txt</code> file on your Amazon S3 storage bucket. For example,
     * <code>requirements.txt</code>. If a relative path is provided in the request, then
     * <code>RequirementsS3ObjectVersion</code> is required. For more information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html">Importing DAGs on Amazon
     * MWAA</a>.
     * </p>
     * 
     * @return The relative path to the <code>requirements.txt</code> file on your Amazon S3 storage bucket. For
     *         example, <code>requirements.txt</code>. If a relative path is provided in the request, then
     *         <code>RequirementsS3ObjectVersion</code> is required. For more information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html">Importing DAGs on
     *         Amazon MWAA</a>.
     */
    public final String requirementsS3Path() {
        return requirementsS3Path;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of your Amazon S3 storage bucket. For example,
     * <code>arn:aws:s3:::airflow-mybucketname</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of your Amazon S3 storage bucket. For example,
     *         <code>arn:aws:s3:::airflow-mybucketname</code>.
     */
    public final String sourceBucketArn() {
        return sourceBucketArn;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The metadata tags you want to attach to your environment. For more information, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging AWS resources</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The metadata tags you want to attach to your environment. For more information, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging AWS resources</a>.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The networking access of your Apache Airflow web server. A public network allows your Airflow UI to be accessed
     * over the Internet by users granted access in your IAM policy. A private network limits access of your Airflow UI
     * to users within your VPC. For more information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/vpc-mwaa.html">Creating the VPC network for a MWAA
     * environment</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #webserverAccessMode} will return {@link WebserverAccessMode#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #webserverAccessModeAsString}.
     * </p>
     * 
     * @return The networking access of your Apache Airflow web server. A public network allows your Airflow UI to be
     *         accessed over the Internet by users granted access in your IAM policy. A private network limits access of
     *         your Airflow UI to users within your VPC. For more information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/vpc-mwaa.html">Creating the VPC network for a
     *         MWAA environment</a>.
     * @see WebserverAccessMode
     */
    public final WebserverAccessMode webserverAccessMode() {
        return WebserverAccessMode.fromValue(webserverAccessMode);
    }

    /**
     * <p>
     * The networking access of your Apache Airflow web server. A public network allows your Airflow UI to be accessed
     * over the Internet by users granted access in your IAM policy. A private network limits access of your Airflow UI
     * to users within your VPC. For more information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/vpc-mwaa.html">Creating the VPC network for a MWAA
     * environment</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #webserverAccessMode} will return {@link WebserverAccessMode#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #webserverAccessModeAsString}.
     * </p>
     * 
     * @return The networking access of your Apache Airflow web server. A public network allows your Airflow UI to be
     *         accessed over the Internet by users granted access in your IAM policy. A private network limits access of
     *         your Airflow UI to users within your VPC. For more information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/vpc-mwaa.html">Creating the VPC network for a
     *         MWAA environment</a>.
     * @see WebserverAccessMode
     */
    public final String webserverAccessModeAsString() {
        return webserverAccessMode;
    }

    /**
     * <p>
     * The day and time you want MWAA to start weekly maintenance updates on your environment.
     * </p>
     * 
     * @return The day and time you want MWAA to start weekly maintenance updates on your environment.
     */
    public final String weeklyMaintenanceWindowStart() {
        return weeklyMaintenanceWindowStart;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(hasAirflowConfigurationOptions() ? airflowConfigurationOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(airflowVersion());
        hashCode = 31 * hashCode + Objects.hashCode(dagS3Path());
        hashCode = 31 * hashCode + Objects.hashCode(environmentClass());
        hashCode = 31 * hashCode + Objects.hashCode(executionRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKey());
        hashCode = 31 * hashCode + Objects.hashCode(loggingConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(maxWorkers());
        hashCode = 31 * hashCode + Objects.hashCode(minWorkers());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(networkConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(pluginsS3ObjectVersion());
        hashCode = 31 * hashCode + Objects.hashCode(pluginsS3Path());
        hashCode = 31 * hashCode + Objects.hashCode(requirementsS3ObjectVersion());
        hashCode = 31 * hashCode + Objects.hashCode(requirementsS3Path());
        hashCode = 31 * hashCode + Objects.hashCode(sourceBucketArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(webserverAccessModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(weeklyMaintenanceWindowStart());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateEnvironmentRequest)) {
            return false;
        }
        CreateEnvironmentRequest other = (CreateEnvironmentRequest) obj;
        return hasAirflowConfigurationOptions() == other.hasAirflowConfigurationOptions()
                && Objects.equals(airflowConfigurationOptions(), other.airflowConfigurationOptions())
                && Objects.equals(airflowVersion(), other.airflowVersion()) && Objects.equals(dagS3Path(), other.dagS3Path())
                && Objects.equals(environmentClass(), other.environmentClass())
                && Objects.equals(executionRoleArn(), other.executionRoleArn()) && Objects.equals(kmsKey(), other.kmsKey())
                && Objects.equals(loggingConfiguration(), other.loggingConfiguration())
                && Objects.equals(maxWorkers(), other.maxWorkers()) && Objects.equals(minWorkers(), other.minWorkers())
                && Objects.equals(name(), other.name()) && Objects.equals(networkConfiguration(), other.networkConfiguration())
                && Objects.equals(pluginsS3ObjectVersion(), other.pluginsS3ObjectVersion())
                && Objects.equals(pluginsS3Path(), other.pluginsS3Path())
                && Objects.equals(requirementsS3ObjectVersion(), other.requirementsS3ObjectVersion())
                && Objects.equals(requirementsS3Path(), other.requirementsS3Path())
                && Objects.equals(sourceBucketArn(), other.sourceBucketArn()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && Objects.equals(webserverAccessModeAsString(), other.webserverAccessModeAsString())
                && Objects.equals(weeklyMaintenanceWindowStart(), other.weeklyMaintenanceWindowStart());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("CreateEnvironmentRequest")
                .add("AirflowConfigurationOptions",
                        airflowConfigurationOptions() == null ? null : "*** Sensitive Data Redacted ***")
                .add("AirflowVersion", airflowVersion()).add("DagS3Path", dagS3Path())
                .add("EnvironmentClass", environmentClass()).add("ExecutionRoleArn", executionRoleArn()).add("KmsKey", kmsKey())
                .add("LoggingConfiguration", loggingConfiguration()).add("MaxWorkers", maxWorkers())
                .add("MinWorkers", minWorkers()).add("Name", name()).add("NetworkConfiguration", networkConfiguration())
                .add("PluginsS3ObjectVersion", pluginsS3ObjectVersion()).add("PluginsS3Path", pluginsS3Path())
                .add("RequirementsS3ObjectVersion", requirementsS3ObjectVersion())
                .add("RequirementsS3Path", requirementsS3Path()).add("SourceBucketArn", sourceBucketArn())
                .add("Tags", hasTags() ? tags() : null).add("WebserverAccessMode", webserverAccessModeAsString())
                .add("WeeklyMaintenanceWindowStart", weeklyMaintenanceWindowStart()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AirflowConfigurationOptions":
            return Optional.ofNullable(clazz.cast(airflowConfigurationOptions()));
        case "AirflowVersion":
            return Optional.ofNullable(clazz.cast(airflowVersion()));
        case "DagS3Path":
            return Optional.ofNullable(clazz.cast(dagS3Path()));
        case "EnvironmentClass":
            return Optional.ofNullable(clazz.cast(environmentClass()));
        case "ExecutionRoleArn":
            return Optional.ofNullable(clazz.cast(executionRoleArn()));
        case "KmsKey":
            return Optional.ofNullable(clazz.cast(kmsKey()));
        case "LoggingConfiguration":
            return Optional.ofNullable(clazz.cast(loggingConfiguration()));
        case "MaxWorkers":
            return Optional.ofNullable(clazz.cast(maxWorkers()));
        case "MinWorkers":
            return Optional.ofNullable(clazz.cast(minWorkers()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "NetworkConfiguration":
            return Optional.ofNullable(clazz.cast(networkConfiguration()));
        case "PluginsS3ObjectVersion":
            return Optional.ofNullable(clazz.cast(pluginsS3ObjectVersion()));
        case "PluginsS3Path":
            return Optional.ofNullable(clazz.cast(pluginsS3Path()));
        case "RequirementsS3ObjectVersion":
            return Optional.ofNullable(clazz.cast(requirementsS3ObjectVersion()));
        case "RequirementsS3Path":
            return Optional.ofNullable(clazz.cast(requirementsS3Path()));
        case "SourceBucketArn":
            return Optional.ofNullable(clazz.cast(sourceBucketArn()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "WebserverAccessMode":
            return Optional.ofNullable(clazz.cast(webserverAccessModeAsString()));
        case "WeeklyMaintenanceWindowStart":
            return Optional.ofNullable(clazz.cast(weeklyMaintenanceWindowStart()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateEnvironmentRequest, T> g) {
        return obj -> g.apply((CreateEnvironmentRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends MwaaRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateEnvironmentRequest> {
        /**
         * <p>
         * The Apache Airflow configuration setting you want to override in your environment. For more information, see
         * <a href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-env-variables.html">Environment
         * configuration</a>.
         * </p>
         * 
         * @param airflowConfigurationOptions
         *        The Apache Airflow configuration setting you want to override in your environment. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-env-variables.html">Environment
         *        configuration</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder airflowConfigurationOptions(Map<String, String> airflowConfigurationOptions);

        /**
         * <p>
         * The Apache Airflow version you want to use for your environment.
         * </p>
         * 
         * @param airflowVersion
         *        The Apache Airflow version you want to use for your environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder airflowVersion(String airflowVersion);

        /**
         * <p>
         * The relative path to the DAG folder on your Amazon S3 storage bucket. For example, <code>dags</code>. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html">Importing DAGs on Amazon
         * MWAA</a>.
         * </p>
         * 
         * @param dagS3Path
         *        The relative path to the DAG folder on your Amazon S3 storage bucket. For example, <code>dags</code>.
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html">Importing DAGs on
         *        Amazon MWAA</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dagS3Path(String dagS3Path);

        /**
         * <p>
         * The environment class you want to use for your environment. The environment class determines the size of the
         * containers and database used for your Apache Airflow services.
         * </p>
         * 
         * @param environmentClass
         *        The environment class you want to use for your environment. The environment class determines the size
         *        of the containers and database used for your Apache Airflow services.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentClass(String environmentClass);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the execution role for your environment. An execution role is an AWS
         * Identity and Access Management (IAM) role that grants MWAA permission to access AWS services and resources
         * used by your environment. For example, <code>arn:aws:iam::123456789:role/my-execution-role</code>. For more
         * information, see <a href="https://docs.aws.amazon.com/mwaa/latest/userguide/manage-access.html">Managing
         * access to Amazon Managed Workflows for Apache Airflow</a>.
         * </p>
         * 
         * @param executionRoleArn
         *        The Amazon Resource Name (ARN) of the execution role for your environment. An execution role is an AWS
         *        Identity and Access Management (IAM) role that grants MWAA permission to access AWS services and
         *        resources used by your environment. For example,
         *        <code>arn:aws:iam::123456789:role/my-execution-role</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/manage-access.html">Managing access to Amazon
         *        Managed Workflows for Apache Airflow</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionRoleArn(String executionRoleArn);

        /**
         * <p>
         * The AWS Key Management Service (KMS) key to encrypt and decrypt the data in your environment. You can use an
         * AWS KMS key managed by MWAA, or a custom KMS key (advanced). For more information, see <a href=
         * "https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html?icmpid=docs_console_unmapped#master_keys"
         * >Customer master keys (CMKs)</a> in the AWS KMS developer guide.
         * </p>
         * 
         * @param kmsKey
         *        The AWS Key Management Service (KMS) key to encrypt and decrypt the data in your environment. You can
         *        use an AWS KMS key managed by MWAA, or a custom KMS key (advanced). For more information, see <a href=
         *        "https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html?icmpid=docs_console_unmapped#master_keys"
         *        >Customer master keys (CMKs)</a> in the AWS KMS developer guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKey(String kmsKey);

        /**
         * <p>
         * The Apache Airflow logs you want to send to Amazon CloudWatch Logs.
         * </p>
         * 
         * @param loggingConfiguration
         *        The Apache Airflow logs you want to send to Amazon CloudWatch Logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loggingConfiguration(LoggingConfigurationInput loggingConfiguration);

        /**
         * <p>
         * The Apache Airflow logs you want to send to Amazon CloudWatch Logs.
         * </p>
         * This is a convenience that creates an instance of the {@link LoggingConfigurationInput.Builder} avoiding the
         * need to create one manually via {@link LoggingConfigurationInput#builder()}.
         *
         * When the {@link Consumer} completes, {@link LoggingConfigurationInput.Builder#build()} is called immediately
         * and its result is passed to {@link #loggingConfiguration(LoggingConfigurationInput)}.
         * 
         * @param loggingConfiguration
         *        a consumer that will call methods on {@link LoggingConfigurationInput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #loggingConfiguration(LoggingConfigurationInput)
         */
        default Builder loggingConfiguration(Consumer<LoggingConfigurationInput.Builder> loggingConfiguration) {
            return loggingConfiguration(LoggingConfigurationInput.builder().applyMutation(loggingConfiguration).build());
        }

        /**
         * <p>
         * The maximum number of workers that you want to run in your environment. MWAA scales the number of Apache
         * Airflow workers and the Fargate containers that run your tasks up to the number you specify in this field.
         * When there are no more tasks running, and no more in the queue, MWAA disposes of the extra containers leaving
         * the one worker that is included with your environment.
         * </p>
         * 
         * @param maxWorkers
         *        The maximum number of workers that you want to run in your environment. MWAA scales the number of
         *        Apache Airflow workers and the Fargate containers that run your tasks up to the number you specify in
         *        this field. When there are no more tasks running, and no more in the queue, MWAA disposes of the extra
         *        containers leaving the one worker that is included with your environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxWorkers(Integer maxWorkers);

        /**
         * <p>
         * The minimum number of workers that you want to run in your environment. MWAA scales the number of Apache
         * Airflow workers and the Fargate containers that run your tasks up to the number you specify in the
         * <code>MaxWorkers</code> field. When there are no more tasks running, and no more in the queue, MWAA disposes
         * of the extra containers leaving the worker count you specify in the <code>MinWorkers</code> field.
         * </p>
         * 
         * @param minWorkers
         *        The minimum number of workers that you want to run in your environment. MWAA scales the number of
         *        Apache Airflow workers and the Fargate containers that run your tasks up to the number you specify in
         *        the <code>MaxWorkers</code> field. When there are no more tasks running, and no more in the queue,
         *        MWAA disposes of the extra containers leaving the worker count you specify in the
         *        <code>MinWorkers</code> field.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minWorkers(Integer minWorkers);

        /**
         * <p>
         * The name of your MWAA environment.
         * </p>
         * 
         * @param name
         *        The name of your MWAA environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The VPC networking components you want to use for your environment. At least two private subnet identifiers
         * and one VPC security group identifier are required to create an environment. For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/vpc-mwaa.html">Creating the VPC network for a MWAA
         * environment</a>.
         * </p>
         * 
         * @param networkConfiguration
         *        The VPC networking components you want to use for your environment. At least two private subnet
         *        identifiers and one VPC security group identifier are required to create an environment. For more
         *        information, see <a href="https://docs.aws.amazon.com/mwaa/latest/userguide/vpc-mwaa.html">Creating
         *        the VPC network for a MWAA environment</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkConfiguration(NetworkConfiguration networkConfiguration);

        /**
         * <p>
         * The VPC networking components you want to use for your environment. At least two private subnet identifiers
         * and one VPC security group identifier are required to create an environment. For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/vpc-mwaa.html">Creating the VPC network for a MWAA
         * environment</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link NetworkConfiguration.Builder} avoiding the need
         * to create one manually via {@link NetworkConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link NetworkConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #networkConfiguration(NetworkConfiguration)}.
         * 
         * @param networkConfiguration
         *        a consumer that will call methods on {@link NetworkConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkConfiguration(NetworkConfiguration)
         */
        default Builder networkConfiguration(Consumer<NetworkConfiguration.Builder> networkConfiguration) {
            return networkConfiguration(NetworkConfiguration.builder().applyMutation(networkConfiguration).build());
        }

        /**
         * <p>
         * The <code>plugins.zip</code> file version you want to use.
         * </p>
         * 
         * @param pluginsS3ObjectVersion
         *        The <code>plugins.zip</code> file version you want to use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pluginsS3ObjectVersion(String pluginsS3ObjectVersion);

        /**
         * <p>
         * The relative path to the <code>plugins.zip</code> file on your Amazon S3 storage bucket. For example,
         * <code>plugins.zip</code>. If a relative path is provided in the request, then
         * <code>PluginsS3ObjectVersion</code> is required. For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html">Importing DAGs on Amazon
         * MWAA</a>.
         * </p>
         * 
         * @param pluginsS3Path
         *        The relative path to the <code>plugins.zip</code> file on your Amazon S3 storage bucket. For example,
         *        <code>plugins.zip</code>. If a relative path is provided in the request, then
         *        <code>PluginsS3ObjectVersion</code> is required. For more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html">Importing DAGs on
         *        Amazon MWAA</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pluginsS3Path(String pluginsS3Path);

        /**
         * <p>
         * The <code>requirements.txt</code> file version you want to use.
         * </p>
         * 
         * @param requirementsS3ObjectVersion
         *        The <code>requirements.txt</code> file version you want to use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requirementsS3ObjectVersion(String requirementsS3ObjectVersion);

        /**
         * <p>
         * The relative path to the <code>requirements.txt</code> file on your Amazon S3 storage bucket. For example,
         * <code>requirements.txt</code>. If a relative path is provided in the request, then
         * <code>RequirementsS3ObjectVersion</code> is required. For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html">Importing DAGs on Amazon
         * MWAA</a>.
         * </p>
         * 
         * @param requirementsS3Path
         *        The relative path to the <code>requirements.txt</code> file on your Amazon S3 storage bucket. For
         *        example, <code>requirements.txt</code>. If a relative path is provided in the request, then
         *        <code>RequirementsS3ObjectVersion</code> is required. For more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import.html">Importing DAGs on
         *        Amazon MWAA</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requirementsS3Path(String requirementsS3Path);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of your Amazon S3 storage bucket. For example,
         * <code>arn:aws:s3:::airflow-mybucketname</code>.
         * </p>
         * 
         * @param sourceBucketArn
         *        The Amazon Resource Name (ARN) of your Amazon S3 storage bucket. For example,
         *        <code>arn:aws:s3:::airflow-mybucketname</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceBucketArn(String sourceBucketArn);

        /**
         * <p>
         * The metadata tags you want to attach to your environment. For more information, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging AWS resources</a>.
         * </p>
         * 
         * @param tags
         *        The metadata tags you want to attach to your environment. For more information, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging AWS resources</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The networking access of your Apache Airflow web server. A public network allows your Airflow UI to be
         * accessed over the Internet by users granted access in your IAM policy. A private network limits access of
         * your Airflow UI to users within your VPC. For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/vpc-mwaa.html">Creating the VPC network for a MWAA
         * environment</a>.
         * </p>
         * 
         * @param webserverAccessMode
         *        The networking access of your Apache Airflow web server. A public network allows your Airflow UI to be
         *        accessed over the Internet by users granted access in your IAM policy. A private network limits access
         *        of your Airflow UI to users within your VPC. For more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/vpc-mwaa.html">Creating the VPC network for a
         *        MWAA environment</a>.
         * @see WebserverAccessMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WebserverAccessMode
         */
        Builder webserverAccessMode(String webserverAccessMode);

        /**
         * <p>
         * The networking access of your Apache Airflow web server. A public network allows your Airflow UI to be
         * accessed over the Internet by users granted access in your IAM policy. A private network limits access of
         * your Airflow UI to users within your VPC. For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/vpc-mwaa.html">Creating the VPC network for a MWAA
         * environment</a>.
         * </p>
         * 
         * @param webserverAccessMode
         *        The networking access of your Apache Airflow web server. A public network allows your Airflow UI to be
         *        accessed over the Internet by users granted access in your IAM policy. A private network limits access
         *        of your Airflow UI to users within your VPC. For more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/vpc-mwaa.html">Creating the VPC network for a
         *        MWAA environment</a>.
         * @see WebserverAccessMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WebserverAccessMode
         */
        Builder webserverAccessMode(WebserverAccessMode webserverAccessMode);

        /**
         * <p>
         * The day and time you want MWAA to start weekly maintenance updates on your environment.
         * </p>
         * 
         * @param weeklyMaintenanceWindowStart
         *        The day and time you want MWAA to start weekly maintenance updates on your environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder weeklyMaintenanceWindowStart(String weeklyMaintenanceWindowStart);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends MwaaRequest.BuilderImpl implements Builder {
        private Map<String, String> airflowConfigurationOptions = DefaultSdkAutoConstructMap.getInstance();

        private String airflowVersion;

        private String dagS3Path;

        private String environmentClass;

        private String executionRoleArn;

        private String kmsKey;

        private LoggingConfigurationInput loggingConfiguration;

        private Integer maxWorkers;

        private Integer minWorkers;

        private String name;

        private NetworkConfiguration networkConfiguration;

        private String pluginsS3ObjectVersion;

        private String pluginsS3Path;

        private String requirementsS3ObjectVersion;

        private String requirementsS3Path;

        private String sourceBucketArn;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String webserverAccessMode;

        private String weeklyMaintenanceWindowStart;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateEnvironmentRequest model) {
            super(model);
            airflowConfigurationOptions(model.airflowConfigurationOptions);
            airflowVersion(model.airflowVersion);
            dagS3Path(model.dagS3Path);
            environmentClass(model.environmentClass);
            executionRoleArn(model.executionRoleArn);
            kmsKey(model.kmsKey);
            loggingConfiguration(model.loggingConfiguration);
            maxWorkers(model.maxWorkers);
            minWorkers(model.minWorkers);
            name(model.name);
            networkConfiguration(model.networkConfiguration);
            pluginsS3ObjectVersion(model.pluginsS3ObjectVersion);
            pluginsS3Path(model.pluginsS3Path);
            requirementsS3ObjectVersion(model.requirementsS3ObjectVersion);
            requirementsS3Path(model.requirementsS3Path);
            sourceBucketArn(model.sourceBucketArn);
            tags(model.tags);
            webserverAccessMode(model.webserverAccessMode);
            weeklyMaintenanceWindowStart(model.weeklyMaintenanceWindowStart);
        }

        public final Map<String, String> getAirflowConfigurationOptions() {
            if (airflowConfigurationOptions instanceof SdkAutoConstructMap) {
                return null;
            }
            return airflowConfigurationOptions;
        }

        @Override
        public final Builder airflowConfigurationOptions(Map<String, String> airflowConfigurationOptions) {
            this.airflowConfigurationOptions = SyntheticCreateEnvironmentInputAirflowConfigurationOptionsCopier
                    .copy(airflowConfigurationOptions);
            return this;
        }

        public final void setAirflowConfigurationOptions(Map<String, String> airflowConfigurationOptions) {
            this.airflowConfigurationOptions = SyntheticCreateEnvironmentInputAirflowConfigurationOptionsCopier
                    .copy(airflowConfigurationOptions);
        }

        public final String getAirflowVersion() {
            return airflowVersion;
        }

        @Override
        public final Builder airflowVersion(String airflowVersion) {
            this.airflowVersion = airflowVersion;
            return this;
        }

        public final void setAirflowVersion(String airflowVersion) {
            this.airflowVersion = airflowVersion;
        }

        public final String getDagS3Path() {
            return dagS3Path;
        }

        @Override
        public final Builder dagS3Path(String dagS3Path) {
            this.dagS3Path = dagS3Path;
            return this;
        }

        public final void setDagS3Path(String dagS3Path) {
            this.dagS3Path = dagS3Path;
        }

        public final String getEnvironmentClass() {
            return environmentClass;
        }

        @Override
        public final Builder environmentClass(String environmentClass) {
            this.environmentClass = environmentClass;
            return this;
        }

        public final void setEnvironmentClass(String environmentClass) {
            this.environmentClass = environmentClass;
        }

        public final String getExecutionRoleArn() {
            return executionRoleArn;
        }

        @Override
        public final Builder executionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
            return this;
        }

        public final void setExecutionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
        }

        public final String getKmsKey() {
            return kmsKey;
        }

        @Override
        public final Builder kmsKey(String kmsKey) {
            this.kmsKey = kmsKey;
            return this;
        }

        public final void setKmsKey(String kmsKey) {
            this.kmsKey = kmsKey;
        }

        public final LoggingConfigurationInput.Builder getLoggingConfiguration() {
            return loggingConfiguration != null ? loggingConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder loggingConfiguration(LoggingConfigurationInput loggingConfiguration) {
            this.loggingConfiguration = loggingConfiguration;
            return this;
        }

        public final void setLoggingConfiguration(LoggingConfigurationInput.BuilderImpl loggingConfiguration) {
            this.loggingConfiguration = loggingConfiguration != null ? loggingConfiguration.build() : null;
        }

        public final Integer getMaxWorkers() {
            return maxWorkers;
        }

        @Override
        public final Builder maxWorkers(Integer maxWorkers) {
            this.maxWorkers = maxWorkers;
            return this;
        }

        public final void setMaxWorkers(Integer maxWorkers) {
            this.maxWorkers = maxWorkers;
        }

        public final Integer getMinWorkers() {
            return minWorkers;
        }

        @Override
        public final Builder minWorkers(Integer minWorkers) {
            this.minWorkers = minWorkers;
            return this;
        }

        public final void setMinWorkers(Integer minWorkers) {
            this.minWorkers = minWorkers;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final NetworkConfiguration.Builder getNetworkConfiguration() {
            return networkConfiguration != null ? networkConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder networkConfiguration(NetworkConfiguration networkConfiguration) {
            this.networkConfiguration = networkConfiguration;
            return this;
        }

        public final void setNetworkConfiguration(NetworkConfiguration.BuilderImpl networkConfiguration) {
            this.networkConfiguration = networkConfiguration != null ? networkConfiguration.build() : null;
        }

        public final String getPluginsS3ObjectVersion() {
            return pluginsS3ObjectVersion;
        }

        @Override
        public final Builder pluginsS3ObjectVersion(String pluginsS3ObjectVersion) {
            this.pluginsS3ObjectVersion = pluginsS3ObjectVersion;
            return this;
        }

        public final void setPluginsS3ObjectVersion(String pluginsS3ObjectVersion) {
            this.pluginsS3ObjectVersion = pluginsS3ObjectVersion;
        }

        public final String getPluginsS3Path() {
            return pluginsS3Path;
        }

        @Override
        public final Builder pluginsS3Path(String pluginsS3Path) {
            this.pluginsS3Path = pluginsS3Path;
            return this;
        }

        public final void setPluginsS3Path(String pluginsS3Path) {
            this.pluginsS3Path = pluginsS3Path;
        }

        public final String getRequirementsS3ObjectVersion() {
            return requirementsS3ObjectVersion;
        }

        @Override
        public final Builder requirementsS3ObjectVersion(String requirementsS3ObjectVersion) {
            this.requirementsS3ObjectVersion = requirementsS3ObjectVersion;
            return this;
        }

        public final void setRequirementsS3ObjectVersion(String requirementsS3ObjectVersion) {
            this.requirementsS3ObjectVersion = requirementsS3ObjectVersion;
        }

        public final String getRequirementsS3Path() {
            return requirementsS3Path;
        }

        @Override
        public final Builder requirementsS3Path(String requirementsS3Path) {
            this.requirementsS3Path = requirementsS3Path;
            return this;
        }

        public final void setRequirementsS3Path(String requirementsS3Path) {
            this.requirementsS3Path = requirementsS3Path;
        }

        public final String getSourceBucketArn() {
            return sourceBucketArn;
        }

        @Override
        public final Builder sourceBucketArn(String sourceBucketArn) {
            this.sourceBucketArn = sourceBucketArn;
            return this;
        }

        public final void setSourceBucketArn(String sourceBucketArn) {
            this.sourceBucketArn = sourceBucketArn;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        public final String getWebserverAccessMode() {
            return webserverAccessMode;
        }

        @Override
        public final Builder webserverAccessMode(String webserverAccessMode) {
            this.webserverAccessMode = webserverAccessMode;
            return this;
        }

        @Override
        public final Builder webserverAccessMode(WebserverAccessMode webserverAccessMode) {
            this.webserverAccessMode(webserverAccessMode == null ? null : webserverAccessMode.toString());
            return this;
        }

        public final void setWebserverAccessMode(String webserverAccessMode) {
            this.webserverAccessMode = webserverAccessMode;
        }

        public final String getWeeklyMaintenanceWindowStart() {
            return weeklyMaintenanceWindowStart;
        }

        @Override
        public final Builder weeklyMaintenanceWindowStart(String weeklyMaintenanceWindowStart) {
            this.weeklyMaintenanceWindowStart = weeklyMaintenanceWindowStart;
            return this;
        }

        public final void setWeeklyMaintenanceWindowStart(String weeklyMaintenanceWindowStart) {
            this.weeklyMaintenanceWindowStart = weeklyMaintenanceWindowStart;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateEnvironmentRequest build() {
            return new CreateEnvironmentRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
