/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mwaa.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This section contains the Amazon Managed Workflows for Apache Airflow (MWAA) API reference documentation to create an
 * environment. For more information, see <a
 * href="https://docs.aws.amazon.com/mwaa/latest/userguide/get-started.html">Get started with Amazon Managed Workflows
 * for Apache Airflow</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateEnvironmentRequest extends MwaaRequest implements
        ToCopyableBuilder<CreateEnvironmentRequest.Builder, CreateEnvironmentRequest> {
    private static final SdkField<Map<String, String>> AIRFLOW_CONFIGURATION_OPTIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("AirflowConfigurationOptions")
            .getter(getter(CreateEnvironmentRequest::airflowConfigurationOptions))
            .setter(setter(Builder::airflowConfigurationOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AirflowConfigurationOptions")
                    .build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> AIRFLOW_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AirflowVersion").getter(getter(CreateEnvironmentRequest::airflowVersion))
            .setter(setter(Builder::airflowVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AirflowVersion").build()).build();

    private static final SdkField<String> DAG_S3_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DagS3Path").getter(getter(CreateEnvironmentRequest::dagS3Path)).setter(setter(Builder::dagS3Path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DagS3Path").build()).build();

    private static final SdkField<String> ENVIRONMENT_CLASS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EnvironmentClass").getter(getter(CreateEnvironmentRequest::environmentClass))
            .setter(setter(Builder::environmentClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnvironmentClass").build()).build();

    private static final SdkField<String> EXECUTION_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExecutionRoleArn").getter(getter(CreateEnvironmentRequest::executionRoleArn))
            .setter(setter(Builder::executionRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionRoleArn").build()).build();

    private static final SdkField<String> KMS_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("KmsKey")
            .getter(getter(CreateEnvironmentRequest::kmsKey)).setter(setter(Builder::kmsKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKey").build()).build();

    private static final SdkField<LoggingConfigurationInput> LOGGING_CONFIGURATION_FIELD = SdkField
            .<LoggingConfigurationInput> builder(MarshallingType.SDK_POJO).memberName("LoggingConfiguration")
            .getter(getter(CreateEnvironmentRequest::loggingConfiguration)).setter(setter(Builder::loggingConfiguration))
            .constructor(LoggingConfigurationInput::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LoggingConfiguration").build())
            .build();

    private static final SdkField<Integer> MAX_WORKERS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxWorkers").getter(getter(CreateEnvironmentRequest::maxWorkers)).setter(setter(Builder::maxWorkers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxWorkers").build()).build();

    private static final SdkField<Integer> MIN_WORKERS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MinWorkers").getter(getter(CreateEnvironmentRequest::minWorkers)).setter(setter(Builder::minWorkers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinWorkers").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(CreateEnvironmentRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("Name").build()).build();

    private static final SdkField<NetworkConfiguration> NETWORK_CONFIGURATION_FIELD = SdkField
            .<NetworkConfiguration> builder(MarshallingType.SDK_POJO).memberName("NetworkConfiguration")
            .getter(getter(CreateEnvironmentRequest::networkConfiguration)).setter(setter(Builder::networkConfiguration))
            .constructor(NetworkConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkConfiguration").build())
            .build();

    private static final SdkField<String> PLUGINS_S3_OBJECT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PluginsS3ObjectVersion").getter(getter(CreateEnvironmentRequest::pluginsS3ObjectVersion))
            .setter(setter(Builder::pluginsS3ObjectVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PluginsS3ObjectVersion").build())
            .build();

    private static final SdkField<String> PLUGINS_S3_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PluginsS3Path").getter(getter(CreateEnvironmentRequest::pluginsS3Path))
            .setter(setter(Builder::pluginsS3Path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PluginsS3Path").build()).build();

    private static final SdkField<String> REQUIREMENTS_S3_OBJECT_VERSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RequirementsS3ObjectVersion")
            .getter(getter(CreateEnvironmentRequest::requirementsS3ObjectVersion))
            .setter(setter(Builder::requirementsS3ObjectVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequirementsS3ObjectVersion")
                    .build()).build();

    private static final SdkField<String> REQUIREMENTS_S3_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RequirementsS3Path").getter(getter(CreateEnvironmentRequest::requirementsS3Path))
            .setter(setter(Builder::requirementsS3Path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequirementsS3Path").build())
            .build();

    private static final SdkField<Integer> SCHEDULERS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Schedulers").getter(getter(CreateEnvironmentRequest::schedulers)).setter(setter(Builder::schedulers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Schedulers").build()).build();

    private static final SdkField<String> SOURCE_BUCKET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceBucketArn").getter(getter(CreateEnvironmentRequest::sourceBucketArn))
            .setter(setter(Builder::sourceBucketArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceBucketArn").build()).build();

    private static final SdkField<String> STARTUP_SCRIPT_S3_OBJECT_VERSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("StartupScriptS3ObjectVersion")
            .getter(getter(CreateEnvironmentRequest::startupScriptS3ObjectVersion))
            .setter(setter(Builder::startupScriptS3ObjectVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartupScriptS3ObjectVersion")
                    .build()).build();

    private static final SdkField<String> STARTUP_SCRIPT_S3_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StartupScriptS3Path").getter(getter(CreateEnvironmentRequest::startupScriptS3Path))
            .setter(setter(Builder::startupScriptS3Path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartupScriptS3Path").build())
            .build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(CreateEnvironmentRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> WEBSERVER_ACCESS_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WebserverAccessMode").getter(getter(CreateEnvironmentRequest::webserverAccessModeAsString))
            .setter(setter(Builder::webserverAccessMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WebserverAccessMode").build())
            .build();

    private static final SdkField<String> WEEKLY_MAINTENANCE_WINDOW_START_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("WeeklyMaintenanceWindowStart")
            .getter(getter(CreateEnvironmentRequest::weeklyMaintenanceWindowStart))
            .setter(setter(Builder::weeklyMaintenanceWindowStart))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WeeklyMaintenanceWindowStart")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            AIRFLOW_CONFIGURATION_OPTIONS_FIELD, AIRFLOW_VERSION_FIELD, DAG_S3_PATH_FIELD, ENVIRONMENT_CLASS_FIELD,
            EXECUTION_ROLE_ARN_FIELD, KMS_KEY_FIELD, LOGGING_CONFIGURATION_FIELD, MAX_WORKERS_FIELD, MIN_WORKERS_FIELD,
            NAME_FIELD, NETWORK_CONFIGURATION_FIELD, PLUGINS_S3_OBJECT_VERSION_FIELD, PLUGINS_S3_PATH_FIELD,
            REQUIREMENTS_S3_OBJECT_VERSION_FIELD, REQUIREMENTS_S3_PATH_FIELD, SCHEDULERS_FIELD, SOURCE_BUCKET_ARN_FIELD,
            STARTUP_SCRIPT_S3_OBJECT_VERSION_FIELD, STARTUP_SCRIPT_S3_PATH_FIELD, TAGS_FIELD, WEBSERVER_ACCESS_MODE_FIELD,
            WEEKLY_MAINTENANCE_WINDOW_START_FIELD));

    private final Map<String, String> airflowConfigurationOptions;

    private final String airflowVersion;

    private final String dagS3Path;

    private final String environmentClass;

    private final String executionRoleArn;

    private final String kmsKey;

    private final LoggingConfigurationInput loggingConfiguration;

    private final Integer maxWorkers;

    private final Integer minWorkers;

    private final String name;

    private final NetworkConfiguration networkConfiguration;

    private final String pluginsS3ObjectVersion;

    private final String pluginsS3Path;

    private final String requirementsS3ObjectVersion;

    private final String requirementsS3Path;

    private final Integer schedulers;

    private final String sourceBucketArn;

    private final String startupScriptS3ObjectVersion;

    private final String startupScriptS3Path;

    private final Map<String, String> tags;

    private final String webserverAccessMode;

    private final String weeklyMaintenanceWindowStart;

    private CreateEnvironmentRequest(BuilderImpl builder) {
        super(builder);
        this.airflowConfigurationOptions = builder.airflowConfigurationOptions;
        this.airflowVersion = builder.airflowVersion;
        this.dagS3Path = builder.dagS3Path;
        this.environmentClass = builder.environmentClass;
        this.executionRoleArn = builder.executionRoleArn;
        this.kmsKey = builder.kmsKey;
        this.loggingConfiguration = builder.loggingConfiguration;
        this.maxWorkers = builder.maxWorkers;
        this.minWorkers = builder.minWorkers;
        this.name = builder.name;
        this.networkConfiguration = builder.networkConfiguration;
        this.pluginsS3ObjectVersion = builder.pluginsS3ObjectVersion;
        this.pluginsS3Path = builder.pluginsS3Path;
        this.requirementsS3ObjectVersion = builder.requirementsS3ObjectVersion;
        this.requirementsS3Path = builder.requirementsS3Path;
        this.schedulers = builder.schedulers;
        this.sourceBucketArn = builder.sourceBucketArn;
        this.startupScriptS3ObjectVersion = builder.startupScriptS3ObjectVersion;
        this.startupScriptS3Path = builder.startupScriptS3Path;
        this.tags = builder.tags;
        this.webserverAccessMode = builder.webserverAccessMode;
        this.weeklyMaintenanceWindowStart = builder.weeklyMaintenanceWindowStart;
    }

    /**
     * For responses, this returns true if the service returned a value for the AirflowConfigurationOptions property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAirflowConfigurationOptions() {
        return airflowConfigurationOptions != null && !(airflowConfigurationOptions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A list of key-value pairs containing the Apache Airflow configuration options you want to attach to your
     * environment. For more information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-env-variables.html">Apache Airflow
     * configuration options</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAirflowConfigurationOptions} method.
     * </p>
     * 
     * @return A list of key-value pairs containing the Apache Airflow configuration options you want to attach to your
     *         environment. For more information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-env-variables.html">Apache Airflow
     *         configuration options</a>.
     */
    public final Map<String, String> airflowConfigurationOptions() {
        return airflowConfigurationOptions;
    }

    /**
     * <p>
     * The Apache Airflow version for your environment. If no value is specified, it defaults to the latest version.
     * Valid values: <code>1.10.12</code>, <code>2.0.2</code>, <code>2.2.2</code>, and <code>2.4.3</code>. For more
     * information, see <a href="https://docs.aws.amazon.com/mwaa/latest/userguide/airflow-versions.html">Apache Airflow
     * versions on Amazon Managed Workflows for Apache Airflow (MWAA)</a>.
     * </p>
     * 
     * @return The Apache Airflow version for your environment. If no value is specified, it defaults to the latest
     *         version. Valid values: <code>1.10.12</code>, <code>2.0.2</code>, <code>2.2.2</code>, and
     *         <code>2.4.3</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/airflow-versions.html">Apache Airflow versions on
     *         Amazon Managed Workflows for Apache Airflow (MWAA)</a>.
     */
    public final String airflowVersion() {
        return airflowVersion;
    }

    /**
     * <p>
     * The relative path to the DAGs folder on your Amazon S3 bucket. For example, <code>dags</code>. For more
     * information, see <a href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-folder.html">Adding
     * or updating DAGs</a>.
     * </p>
     * 
     * @return The relative path to the DAGs folder on your Amazon S3 bucket. For example, <code>dags</code>. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-folder.html">Adding or updating
     *         DAGs</a>.
     */
    public final String dagS3Path() {
        return dagS3Path;
    }

    /**
     * <p>
     * The environment class type. Valid values: <code>mw1.small</code>, <code>mw1.medium</code>, <code>mw1.large</code>
     * . For more information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/environment-class.html">Amazon MWAA environment
     * class</a>.
     * </p>
     * 
     * @return The environment class type. Valid values: <code>mw1.small</code>, <code>mw1.medium</code>,
     *         <code>mw1.large</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/environment-class.html">Amazon MWAA environment
     *         class</a>.
     */
    public final String environmentClass() {
        return environmentClass;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the execution role for your environment. An execution role is an Amazon Web
     * Services Identity and Access Management (IAM) role that grants MWAA permission to access Amazon Web Services
     * services and resources used by your environment. For example,
     * <code>arn:aws:iam::123456789:role/my-execution-role</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/mwaa-create-role.html">Amazon MWAA Execution role</a>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the execution role for your environment. An execution role is an Amazon
     *         Web Services Identity and Access Management (IAM) role that grants MWAA permission to access Amazon Web
     *         Services services and resources used by your environment. For example,
     *         <code>arn:aws:iam::123456789:role/my-execution-role</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/mwaa-create-role.html">Amazon MWAA Execution
     *         role</a>.
     */
    public final String executionRoleArn() {
        return executionRoleArn;
    }

    /**
     * <p>
     * The Amazon Web Services Key Management Service (KMS) key to encrypt the data in your environment. You can use an
     * Amazon Web Services owned CMK, or a Customer managed CMK (advanced). For more information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/create-environment.html">Create an Amazon MWAA
     * environment</a>.
     * </p>
     * 
     * @return The Amazon Web Services Key Management Service (KMS) key to encrypt the data in your environment. You can
     *         use an Amazon Web Services owned CMK, or a Customer managed CMK (advanced). For more information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/create-environment.html">Create an Amazon MWAA
     *         environment</a>.
     */
    public final String kmsKey() {
        return kmsKey;
    }

    /**
     * <p>
     * Defines the Apache Airflow logs to send to CloudWatch Logs.
     * </p>
     * 
     * @return Defines the Apache Airflow logs to send to CloudWatch Logs.
     */
    public final LoggingConfigurationInput loggingConfiguration() {
        return loggingConfiguration;
    }

    /**
     * <p>
     * The maximum number of workers that you want to run in your environment. MWAA scales the number of Apache Airflow
     * workers up to the number you specify in the <code>MaxWorkers</code> field. For example, <code>20</code>. When
     * there are no more tasks running, and no more in the queue, MWAA disposes of the extra workers leaving the one
     * worker that is included with your environment, or the number you specify in <code>MinWorkers</code>.
     * </p>
     * 
     * @return The maximum number of workers that you want to run in your environment. MWAA scales the number of Apache
     *         Airflow workers up to the number you specify in the <code>MaxWorkers</code> field. For example,
     *         <code>20</code>. When there are no more tasks running, and no more in the queue, MWAA disposes of the
     *         extra workers leaving the one worker that is included with your environment, or the number you specify in
     *         <code>MinWorkers</code>.
     */
    public final Integer maxWorkers() {
        return maxWorkers;
    }

    /**
     * <p>
     * The minimum number of workers that you want to run in your environment. MWAA scales the number of Apache Airflow
     * workers up to the number you specify in the <code>MaxWorkers</code> field. When there are no more tasks running,
     * and no more in the queue, MWAA disposes of the extra workers leaving the worker count you specify in the
     * <code>MinWorkers</code> field. For example, <code>2</code>.
     * </p>
     * 
     * @return The minimum number of workers that you want to run in your environment. MWAA scales the number of Apache
     *         Airflow workers up to the number you specify in the <code>MaxWorkers</code> field. When there are no more
     *         tasks running, and no more in the queue, MWAA disposes of the extra workers leaving the worker count you
     *         specify in the <code>MinWorkers</code> field. For example, <code>2</code>.
     */
    public final Integer minWorkers() {
        return minWorkers;
    }

    /**
     * <p>
     * The name of the Amazon MWAA environment. For example, <code>MyMWAAEnvironment</code>.
     * </p>
     * 
     * @return The name of the Amazon MWAA environment. For example, <code>MyMWAAEnvironment</code>.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The VPC networking components used to secure and enable network traffic between the Amazon Web Services resources
     * for your environment. For more information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/networking-about.html">About networking on Amazon
     * MWAA</a>.
     * </p>
     * 
     * @return The VPC networking components used to secure and enable network traffic between the Amazon Web Services
     *         resources for your environment. For more information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/networking-about.html">About networking on Amazon
     *         MWAA</a>.
     */
    public final NetworkConfiguration networkConfiguration() {
        return networkConfiguration;
    }

    /**
     * <p>
     * The version of the plugins.zip file on your Amazon S3 bucket. You must specify a version each time a plugins.zip
     * file is updated. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/versioning-workflows.html">How S3 Versioning
     * works</a>.
     * </p>
     * 
     * @return The version of the plugins.zip file on your Amazon S3 bucket. You must specify a version each time a
     *         plugins.zip file is updated. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/versioning-workflows.html">How S3 Versioning
     *         works</a>.
     */
    public final String pluginsS3ObjectVersion() {
        return pluginsS3ObjectVersion;
    }

    /**
     * <p>
     * The relative path to the <code>plugins.zip</code> file on your Amazon S3 bucket. For example,
     * <code>plugins.zip</code>. If specified, then the <code>plugins.zip</code> version is required. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import-plugins.html">Installing custom
     * plugins</a>.
     * </p>
     * 
     * @return The relative path to the <code>plugins.zip</code> file on your Amazon S3 bucket. For example,
     *         <code>plugins.zip</code>. If specified, then the <code>plugins.zip</code> version is required. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import-plugins.html">Installing
     *         custom plugins</a>.
     */
    public final String pluginsS3Path() {
        return pluginsS3Path;
    }

    /**
     * <p>
     * The version of the <code>requirements.txt</code> file on your Amazon S3 bucket. You must specify a version each
     * time a requirements.txt file is updated. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/versioning-workflows.html">How S3 Versioning
     * works</a>.
     * </p>
     * 
     * @return The version of the <code>requirements.txt</code> file on your Amazon S3 bucket. You must specify a
     *         version each time a requirements.txt file is updated. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/versioning-workflows.html">How S3 Versioning
     *         works</a>.
     */
    public final String requirementsS3ObjectVersion() {
        return requirementsS3ObjectVersion;
    }

    /**
     * <p>
     * The relative path to the <code>requirements.txt</code> file on your Amazon S3 bucket. For example,
     * <code>requirements.txt</code>. If specified, then a version is required. For more information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/working-dags-dependencies.html">Installing Python
     * dependencies</a>.
     * </p>
     * 
     * @return The relative path to the <code>requirements.txt</code> file on your Amazon S3 bucket. For example,
     *         <code>requirements.txt</code>. If specified, then a version is required. For more information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/working-dags-dependencies.html">Installing Python
     *         dependencies</a>.
     */
    public final String requirementsS3Path() {
        return requirementsS3Path;
    }

    /**
     * <p>
     * The number of Apache Airflow schedulers to run in your environment. Valid values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * v2 - Accepts between 2 to 5. Defaults to 2.
     * </p>
     * </li>
     * <li>
     * <p>
     * v1 - Accepts 1.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The number of Apache Airflow schedulers to run in your environment. Valid values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         v2 - Accepts between 2 to 5. Defaults to 2.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         v1 - Accepts 1.
     *         </p>
     *         </li>
     */
    public final Integer schedulers() {
        return schedulers;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Amazon S3 bucket where your DAG code and supporting files are stored. For
     * example, <code>arn:aws:s3:::my-airflow-bucket-unique-name</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/mwaa-s3-bucket.html">Create an Amazon S3 bucket for
     * Amazon MWAA</a>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Amazon S3 bucket where your DAG code and supporting files are
     *         stored. For example, <code>arn:aws:s3:::my-airflow-bucket-unique-name</code>. For more information, see
     *         <a href="https://docs.aws.amazon.com/mwaa/latest/userguide/mwaa-s3-bucket.html">Create an Amazon S3
     *         bucket for Amazon MWAA</a>.
     */
    public final String sourceBucketArn() {
        return sourceBucketArn;
    }

    /**
     * <p>
     * The version of the startup shell script in your Amazon S3 bucket. You must specify the <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/versioning-workflows.html">version ID</a> that Amazon
     * S3 assigns to the file every time you update the script.
     * </p>
     * <p>
     * Version IDs are Unicode, UTF-8 encoded, URL-ready, opaque strings that are no more than 1,024 bytes long. The
     * following is an example:
     * </p>
     * <p>
     * <code>3sL4kqtJlcpXroDTDmJ+rmSpXd3dIbrHY+MTRCxf3vjVBH40Nr8X8gdRQBpUMLUo</code>
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/using-startup-script.html">Using a startup script</a>.
     * </p>
     * 
     * @return The version of the startup shell script in your Amazon S3 bucket. You must specify the <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/versioning-workflows.html">version ID</a>
     *         that Amazon S3 assigns to the file every time you update the script. </p>
     *         <p>
     *         Version IDs are Unicode, UTF-8 encoded, URL-ready, opaque strings that are no more than 1,024 bytes long.
     *         The following is an example:
     *         </p>
     *         <p>
     *         <code>3sL4kqtJlcpXroDTDmJ+rmSpXd3dIbrHY+MTRCxf3vjVBH40Nr8X8gdRQBpUMLUo</code>
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/using-startup-script.html">Using a startup
     *         script</a>.
     */
    public final String startupScriptS3ObjectVersion() {
        return startupScriptS3ObjectVersion;
    }

    /**
     * <p>
     * The relative path to the startup shell script in your Amazon S3 bucket. For example,
     * <code>s3://mwaa-environment/startup.sh</code>.
     * </p>
     * <p>
     * Amazon MWAA runs the script as your environment starts, and before running the Apache Airflow process. You can
     * use this script to install dependencies, modify Apache Airflow configuration options, and set environment
     * variables. For more information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/using-startup-script.html">Using a startup script</a>.
     * </p>
     * 
     * @return The relative path to the startup shell script in your Amazon S3 bucket. For example,
     *         <code>s3://mwaa-environment/startup.sh</code>.</p>
     *         <p>
     *         Amazon MWAA runs the script as your environment starts, and before running the Apache Airflow process.
     *         You can use this script to install dependencies, modify Apache Airflow configuration options, and set
     *         environment variables. For more information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/using-startup-script.html">Using a startup
     *         script</a>.
     */
    public final String startupScriptS3Path() {
        return startupScriptS3Path;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The key-value tag pairs you want to associate to your environment. For example,
     * <code>"Environment": "Staging"</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services resources</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The key-value tag pairs you want to associate to your environment. For example,
     *         <code>"Environment": "Staging"</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services
     *         resources</a>.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The Apache Airflow <i>Web server</i> access mode. For more information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-networking.html">Apache Airflow access
     * modes</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #webserverAccessMode} will return {@link WebserverAccessMode#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #webserverAccessModeAsString}.
     * </p>
     * 
     * @return The Apache Airflow <i>Web server</i> access mode. For more information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-networking.html">Apache Airflow
     *         access modes</a>.
     * @see WebserverAccessMode
     */
    public final WebserverAccessMode webserverAccessMode() {
        return WebserverAccessMode.fromValue(webserverAccessMode);
    }

    /**
     * <p>
     * The Apache Airflow <i>Web server</i> access mode. For more information, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-networking.html">Apache Airflow access
     * modes</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #webserverAccessMode} will return {@link WebserverAccessMode#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #webserverAccessModeAsString}.
     * </p>
     * 
     * @return The Apache Airflow <i>Web server</i> access mode. For more information, see <a
     *         href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-networking.html">Apache Airflow
     *         access modes</a>.
     * @see WebserverAccessMode
     */
    public final String webserverAccessModeAsString() {
        return webserverAccessMode;
    }

    /**
     * <p>
     * The day and time of the week in Coordinated Universal Time (UTC) 24-hour standard time to start weekly
     * maintenance updates of your environment in the following format: <code>DAY:HH:MM</code>. For example:
     * <code>TUE:03:30</code>. You can specify a start time in 30 minute increments only.
     * </p>
     * 
     * @return The day and time of the week in Coordinated Universal Time (UTC) 24-hour standard time to start weekly
     *         maintenance updates of your environment in the following format: <code>DAY:HH:MM</code>. For example:
     *         <code>TUE:03:30</code>. You can specify a start time in 30 minute increments only.
     */
    public final String weeklyMaintenanceWindowStart() {
        return weeklyMaintenanceWindowStart;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(hasAirflowConfigurationOptions() ? airflowConfigurationOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(airflowVersion());
        hashCode = 31 * hashCode + Objects.hashCode(dagS3Path());
        hashCode = 31 * hashCode + Objects.hashCode(environmentClass());
        hashCode = 31 * hashCode + Objects.hashCode(executionRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKey());
        hashCode = 31 * hashCode + Objects.hashCode(loggingConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(maxWorkers());
        hashCode = 31 * hashCode + Objects.hashCode(minWorkers());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(networkConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(pluginsS3ObjectVersion());
        hashCode = 31 * hashCode + Objects.hashCode(pluginsS3Path());
        hashCode = 31 * hashCode + Objects.hashCode(requirementsS3ObjectVersion());
        hashCode = 31 * hashCode + Objects.hashCode(requirementsS3Path());
        hashCode = 31 * hashCode + Objects.hashCode(schedulers());
        hashCode = 31 * hashCode + Objects.hashCode(sourceBucketArn());
        hashCode = 31 * hashCode + Objects.hashCode(startupScriptS3ObjectVersion());
        hashCode = 31 * hashCode + Objects.hashCode(startupScriptS3Path());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(webserverAccessModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(weeklyMaintenanceWindowStart());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateEnvironmentRequest)) {
            return false;
        }
        CreateEnvironmentRequest other = (CreateEnvironmentRequest) obj;
        return hasAirflowConfigurationOptions() == other.hasAirflowConfigurationOptions()
                && Objects.equals(airflowConfigurationOptions(), other.airflowConfigurationOptions())
                && Objects.equals(airflowVersion(), other.airflowVersion()) && Objects.equals(dagS3Path(), other.dagS3Path())
                && Objects.equals(environmentClass(), other.environmentClass())
                && Objects.equals(executionRoleArn(), other.executionRoleArn()) && Objects.equals(kmsKey(), other.kmsKey())
                && Objects.equals(loggingConfiguration(), other.loggingConfiguration())
                && Objects.equals(maxWorkers(), other.maxWorkers()) && Objects.equals(minWorkers(), other.minWorkers())
                && Objects.equals(name(), other.name()) && Objects.equals(networkConfiguration(), other.networkConfiguration())
                && Objects.equals(pluginsS3ObjectVersion(), other.pluginsS3ObjectVersion())
                && Objects.equals(pluginsS3Path(), other.pluginsS3Path())
                && Objects.equals(requirementsS3ObjectVersion(), other.requirementsS3ObjectVersion())
                && Objects.equals(requirementsS3Path(), other.requirementsS3Path())
                && Objects.equals(schedulers(), other.schedulers()) && Objects.equals(sourceBucketArn(), other.sourceBucketArn())
                && Objects.equals(startupScriptS3ObjectVersion(), other.startupScriptS3ObjectVersion())
                && Objects.equals(startupScriptS3Path(), other.startupScriptS3Path()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && Objects.equals(webserverAccessModeAsString(), other.webserverAccessModeAsString())
                && Objects.equals(weeklyMaintenanceWindowStart(), other.weeklyMaintenanceWindowStart());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("CreateEnvironmentRequest")
                .add("AirflowConfigurationOptions",
                        airflowConfigurationOptions() == null ? null : "*** Sensitive Data Redacted ***")
                .add("AirflowVersion", airflowVersion()).add("DagS3Path", dagS3Path())
                .add("EnvironmentClass", environmentClass()).add("ExecutionRoleArn", executionRoleArn()).add("KmsKey", kmsKey())
                .add("LoggingConfiguration", loggingConfiguration()).add("MaxWorkers", maxWorkers())
                .add("MinWorkers", minWorkers()).add("Name", name()).add("NetworkConfiguration", networkConfiguration())
                .add("PluginsS3ObjectVersion", pluginsS3ObjectVersion()).add("PluginsS3Path", pluginsS3Path())
                .add("RequirementsS3ObjectVersion", requirementsS3ObjectVersion())
                .add("RequirementsS3Path", requirementsS3Path()).add("Schedulers", schedulers())
                .add("SourceBucketArn", sourceBucketArn()).add("StartupScriptS3ObjectVersion", startupScriptS3ObjectVersion())
                .add("StartupScriptS3Path", startupScriptS3Path()).add("Tags", hasTags() ? tags() : null)
                .add("WebserverAccessMode", webserverAccessModeAsString())
                .add("WeeklyMaintenanceWindowStart", weeklyMaintenanceWindowStart()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AirflowConfigurationOptions":
            return Optional.ofNullable(clazz.cast(airflowConfigurationOptions()));
        case "AirflowVersion":
            return Optional.ofNullable(clazz.cast(airflowVersion()));
        case "DagS3Path":
            return Optional.ofNullable(clazz.cast(dagS3Path()));
        case "EnvironmentClass":
            return Optional.ofNullable(clazz.cast(environmentClass()));
        case "ExecutionRoleArn":
            return Optional.ofNullable(clazz.cast(executionRoleArn()));
        case "KmsKey":
            return Optional.ofNullable(clazz.cast(kmsKey()));
        case "LoggingConfiguration":
            return Optional.ofNullable(clazz.cast(loggingConfiguration()));
        case "MaxWorkers":
            return Optional.ofNullable(clazz.cast(maxWorkers()));
        case "MinWorkers":
            return Optional.ofNullable(clazz.cast(minWorkers()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "NetworkConfiguration":
            return Optional.ofNullable(clazz.cast(networkConfiguration()));
        case "PluginsS3ObjectVersion":
            return Optional.ofNullable(clazz.cast(pluginsS3ObjectVersion()));
        case "PluginsS3Path":
            return Optional.ofNullable(clazz.cast(pluginsS3Path()));
        case "RequirementsS3ObjectVersion":
            return Optional.ofNullable(clazz.cast(requirementsS3ObjectVersion()));
        case "RequirementsS3Path":
            return Optional.ofNullable(clazz.cast(requirementsS3Path()));
        case "Schedulers":
            return Optional.ofNullable(clazz.cast(schedulers()));
        case "SourceBucketArn":
            return Optional.ofNullable(clazz.cast(sourceBucketArn()));
        case "StartupScriptS3ObjectVersion":
            return Optional.ofNullable(clazz.cast(startupScriptS3ObjectVersion()));
        case "StartupScriptS3Path":
            return Optional.ofNullable(clazz.cast(startupScriptS3Path()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "WebserverAccessMode":
            return Optional.ofNullable(clazz.cast(webserverAccessModeAsString()));
        case "WeeklyMaintenanceWindowStart":
            return Optional.ofNullable(clazz.cast(weeklyMaintenanceWindowStart()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateEnvironmentRequest, T> g) {
        return obj -> g.apply((CreateEnvironmentRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends MwaaRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateEnvironmentRequest> {
        /**
         * <p>
         * A list of key-value pairs containing the Apache Airflow configuration options you want to attach to your
         * environment. For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-env-variables.html">Apache Airflow
         * configuration options</a>.
         * </p>
         * 
         * @param airflowConfigurationOptions
         *        A list of key-value pairs containing the Apache Airflow configuration options you want to attach to
         *        your environment. For more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-env-variables.html">Apache Airflow
         *        configuration options</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder airflowConfigurationOptions(Map<String, String> airflowConfigurationOptions);

        /**
         * <p>
         * The Apache Airflow version for your environment. If no value is specified, it defaults to the latest version.
         * Valid values: <code>1.10.12</code>, <code>2.0.2</code>, <code>2.2.2</code>, and <code>2.4.3</code>. For more
         * information, see <a href="https://docs.aws.amazon.com/mwaa/latest/userguide/airflow-versions.html">Apache
         * Airflow versions on Amazon Managed Workflows for Apache Airflow (MWAA)</a>.
         * </p>
         * 
         * @param airflowVersion
         *        The Apache Airflow version for your environment. If no value is specified, it defaults to the latest
         *        version. Valid values: <code>1.10.12</code>, <code>2.0.2</code>, <code>2.2.2</code>, and
         *        <code>2.4.3</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/airflow-versions.html">Apache Airflow versions
         *        on Amazon Managed Workflows for Apache Airflow (MWAA)</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder airflowVersion(String airflowVersion);

        /**
         * <p>
         * The relative path to the DAGs folder on your Amazon S3 bucket. For example, <code>dags</code>. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-folder.html">Adding or updating
         * DAGs</a>.
         * </p>
         * 
         * @param dagS3Path
         *        The relative path to the DAGs folder on your Amazon S3 bucket. For example, <code>dags</code>. For
         *        more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-folder.html">Adding or
         *        updating DAGs</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dagS3Path(String dagS3Path);

        /**
         * <p>
         * The environment class type. Valid values: <code>mw1.small</code>, <code>mw1.medium</code>,
         * <code>mw1.large</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/environment-class.html">Amazon MWAA environment
         * class</a>.
         * </p>
         * 
         * @param environmentClass
         *        The environment class type. Valid values: <code>mw1.small</code>, <code>mw1.medium</code>,
         *        <code>mw1.large</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/environment-class.html">Amazon MWAA
         *        environment class</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentClass(String environmentClass);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the execution role for your environment. An execution role is an Amazon Web
         * Services Identity and Access Management (IAM) role that grants MWAA permission to access Amazon Web Services
         * services and resources used by your environment. For example,
         * <code>arn:aws:iam::123456789:role/my-execution-role</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/mwaa-create-role.html">Amazon MWAA Execution
         * role</a>.
         * </p>
         * 
         * @param executionRoleArn
         *        The Amazon Resource Name (ARN) of the execution role for your environment. An execution role is an
         *        Amazon Web Services Identity and Access Management (IAM) role that grants MWAA permission to access
         *        Amazon Web Services services and resources used by your environment. For example,
         *        <code>arn:aws:iam::123456789:role/my-execution-role</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/mwaa-create-role.html">Amazon MWAA Execution
         *        role</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionRoleArn(String executionRoleArn);

        /**
         * <p>
         * The Amazon Web Services Key Management Service (KMS) key to encrypt the data in your environment. You can use
         * an Amazon Web Services owned CMK, or a Customer managed CMK (advanced). For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/create-environment.html">Create an Amazon MWAA
         * environment</a>.
         * </p>
         * 
         * @param kmsKey
         *        The Amazon Web Services Key Management Service (KMS) key to encrypt the data in your environment. You
         *        can use an Amazon Web Services owned CMK, or a Customer managed CMK (advanced). For more information,
         *        see <a href="https://docs.aws.amazon.com/mwaa/latest/userguide/create-environment.html">Create an
         *        Amazon MWAA environment</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKey(String kmsKey);

        /**
         * <p>
         * Defines the Apache Airflow logs to send to CloudWatch Logs.
         * </p>
         * 
         * @param loggingConfiguration
         *        Defines the Apache Airflow logs to send to CloudWatch Logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loggingConfiguration(LoggingConfigurationInput loggingConfiguration);

        /**
         * <p>
         * Defines the Apache Airflow logs to send to CloudWatch Logs.
         * </p>
         * This is a convenience method that creates an instance of the {@link LoggingConfigurationInput.Builder}
         * avoiding the need to create one manually via {@link LoggingConfigurationInput#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LoggingConfigurationInput.Builder#build()} is called immediately
         * and its result is passed to {@link #loggingConfiguration(LoggingConfigurationInput)}.
         * 
         * @param loggingConfiguration
         *        a consumer that will call methods on {@link LoggingConfigurationInput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #loggingConfiguration(LoggingConfigurationInput)
         */
        default Builder loggingConfiguration(Consumer<LoggingConfigurationInput.Builder> loggingConfiguration) {
            return loggingConfiguration(LoggingConfigurationInput.builder().applyMutation(loggingConfiguration).build());
        }

        /**
         * <p>
         * The maximum number of workers that you want to run in your environment. MWAA scales the number of Apache
         * Airflow workers up to the number you specify in the <code>MaxWorkers</code> field. For example,
         * <code>20</code>. When there are no more tasks running, and no more in the queue, MWAA disposes of the extra
         * workers leaving the one worker that is included with your environment, or the number you specify in
         * <code>MinWorkers</code>.
         * </p>
         * 
         * @param maxWorkers
         *        The maximum number of workers that you want to run in your environment. MWAA scales the number of
         *        Apache Airflow workers up to the number you specify in the <code>MaxWorkers</code> field. For example,
         *        <code>20</code>. When there are no more tasks running, and no more in the queue, MWAA disposes of the
         *        extra workers leaving the one worker that is included with your environment, or the number you specify
         *        in <code>MinWorkers</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxWorkers(Integer maxWorkers);

        /**
         * <p>
         * The minimum number of workers that you want to run in your environment. MWAA scales the number of Apache
         * Airflow workers up to the number you specify in the <code>MaxWorkers</code> field. When there are no more
         * tasks running, and no more in the queue, MWAA disposes of the extra workers leaving the worker count you
         * specify in the <code>MinWorkers</code> field. For example, <code>2</code>.
         * </p>
         * 
         * @param minWorkers
         *        The minimum number of workers that you want to run in your environment. MWAA scales the number of
         *        Apache Airflow workers up to the number you specify in the <code>MaxWorkers</code> field. When there
         *        are no more tasks running, and no more in the queue, MWAA disposes of the extra workers leaving the
         *        worker count you specify in the <code>MinWorkers</code> field. For example, <code>2</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minWorkers(Integer minWorkers);

        /**
         * <p>
         * The name of the Amazon MWAA environment. For example, <code>MyMWAAEnvironment</code>.
         * </p>
         * 
         * @param name
         *        The name of the Amazon MWAA environment. For example, <code>MyMWAAEnvironment</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The VPC networking components used to secure and enable network traffic between the Amazon Web Services
         * resources for your environment. For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/networking-about.html">About networking on Amazon
         * MWAA</a>.
         * </p>
         * 
         * @param networkConfiguration
         *        The VPC networking components used to secure and enable network traffic between the Amazon Web
         *        Services resources for your environment. For more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/networking-about.html">About networking on
         *        Amazon MWAA</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkConfiguration(NetworkConfiguration networkConfiguration);

        /**
         * <p>
         * The VPC networking components used to secure and enable network traffic between the Amazon Web Services
         * resources for your environment. For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/networking-about.html">About networking on Amazon
         * MWAA</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link NetworkConfiguration.Builder} avoiding
         * the need to create one manually via {@link NetworkConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NetworkConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #networkConfiguration(NetworkConfiguration)}.
         * 
         * @param networkConfiguration
         *        a consumer that will call methods on {@link NetworkConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkConfiguration(NetworkConfiguration)
         */
        default Builder networkConfiguration(Consumer<NetworkConfiguration.Builder> networkConfiguration) {
            return networkConfiguration(NetworkConfiguration.builder().applyMutation(networkConfiguration).build());
        }

        /**
         * <p>
         * The version of the plugins.zip file on your Amazon S3 bucket. You must specify a version each time a
         * plugins.zip file is updated. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/versioning-workflows.html">How S3 Versioning
         * works</a>.
         * </p>
         * 
         * @param pluginsS3ObjectVersion
         *        The version of the plugins.zip file on your Amazon S3 bucket. You must specify a version each time a
         *        plugins.zip file is updated. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/versioning-workflows.html">How S3
         *        Versioning works</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pluginsS3ObjectVersion(String pluginsS3ObjectVersion);

        /**
         * <p>
         * The relative path to the <code>plugins.zip</code> file on your Amazon S3 bucket. For example,
         * <code>plugins.zip</code>. If specified, then the <code>plugins.zip</code> version is required. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import-plugins.html">Installing
         * custom plugins</a>.
         * </p>
         * 
         * @param pluginsS3Path
         *        The relative path to the <code>plugins.zip</code> file on your Amazon S3 bucket. For example,
         *        <code>plugins.zip</code>. If specified, then the <code>plugins.zip</code> version is required. For
         *        more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-dag-import-plugins.html"
         *        >Installing custom plugins</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pluginsS3Path(String pluginsS3Path);

        /**
         * <p>
         * The version of the <code>requirements.txt</code> file on your Amazon S3 bucket. You must specify a version
         * each time a requirements.txt file is updated. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/versioning-workflows.html">How S3 Versioning
         * works</a>.
         * </p>
         * 
         * @param requirementsS3ObjectVersion
         *        The version of the <code>requirements.txt</code> file on your Amazon S3 bucket. You must specify a
         *        version each time a requirements.txt file is updated. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/versioning-workflows.html">How S3
         *        Versioning works</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requirementsS3ObjectVersion(String requirementsS3ObjectVersion);

        /**
         * <p>
         * The relative path to the <code>requirements.txt</code> file on your Amazon S3 bucket. For example,
         * <code>requirements.txt</code>. If specified, then a version is required. For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/working-dags-dependencies.html">Installing Python
         * dependencies</a>.
         * </p>
         * 
         * @param requirementsS3Path
         *        The relative path to the <code>requirements.txt</code> file on your Amazon S3 bucket. For example,
         *        <code>requirements.txt</code>. If specified, then a version is required. For more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/working-dags-dependencies.html">Installing
         *        Python dependencies</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requirementsS3Path(String requirementsS3Path);

        /**
         * <p>
         * The number of Apache Airflow schedulers to run in your environment. Valid values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * v2 - Accepts between 2 to 5. Defaults to 2.
         * </p>
         * </li>
         * <li>
         * <p>
         * v1 - Accepts 1.
         * </p>
         * </li>
         * </ul>
         * 
         * @param schedulers
         *        The number of Apache Airflow schedulers to run in your environment. Valid values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        v2 - Accepts between 2 to 5. Defaults to 2.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        v1 - Accepts 1.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schedulers(Integer schedulers);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Amazon S3 bucket where your DAG code and supporting files are stored.
         * For example, <code>arn:aws:s3:::my-airflow-bucket-unique-name</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/mwaa-s3-bucket.html">Create an Amazon S3 bucket for
         * Amazon MWAA</a>.
         * </p>
         * 
         * @param sourceBucketArn
         *        The Amazon Resource Name (ARN) of the Amazon S3 bucket where your DAG code and supporting files are
         *        stored. For example, <code>arn:aws:s3:::my-airflow-bucket-unique-name</code>. For more information,
         *        see <a href="https://docs.aws.amazon.com/mwaa/latest/userguide/mwaa-s3-bucket.html">Create an Amazon
         *        S3 bucket for Amazon MWAA</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceBucketArn(String sourceBucketArn);

        /**
         * <p>
         * The version of the startup shell script in your Amazon S3 bucket. You must specify the <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/versioning-workflows.html">version ID</a> that
         * Amazon S3 assigns to the file every time you update the script.
         * </p>
         * <p>
         * Version IDs are Unicode, UTF-8 encoded, URL-ready, opaque strings that are no more than 1,024 bytes long. The
         * following is an example:
         * </p>
         * <p>
         * <code>3sL4kqtJlcpXroDTDmJ+rmSpXd3dIbrHY+MTRCxf3vjVBH40Nr8X8gdRQBpUMLUo</code>
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/using-startup-script.html">Using a startup
         * script</a>.
         * </p>
         * 
         * @param startupScriptS3ObjectVersion
         *        The version of the startup shell script in your Amazon S3 bucket. You must specify the <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/versioning-workflows.html">version ID</a>
         *        that Amazon S3 assigns to the file every time you update the script. </p>
         *        <p>
         *        Version IDs are Unicode, UTF-8 encoded, URL-ready, opaque strings that are no more than 1,024 bytes
         *        long. The following is an example:
         *        </p>
         *        <p>
         *        <code>3sL4kqtJlcpXroDTDmJ+rmSpXd3dIbrHY+MTRCxf3vjVBH40Nr8X8gdRQBpUMLUo</code>
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/using-startup-script.html">Using a startup
         *        script</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startupScriptS3ObjectVersion(String startupScriptS3ObjectVersion);

        /**
         * <p>
         * The relative path to the startup shell script in your Amazon S3 bucket. For example,
         * <code>s3://mwaa-environment/startup.sh</code>.
         * </p>
         * <p>
         * Amazon MWAA runs the script as your environment starts, and before running the Apache Airflow process. You
         * can use this script to install dependencies, modify Apache Airflow configuration options, and set environment
         * variables. For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/using-startup-script.html">Using a startup
         * script</a>.
         * </p>
         * 
         * @param startupScriptS3Path
         *        The relative path to the startup shell script in your Amazon S3 bucket. For example,
         *        <code>s3://mwaa-environment/startup.sh</code>.</p>
         *        <p>
         *        Amazon MWAA runs the script as your environment starts, and before running the Apache Airflow process.
         *        You can use this script to install dependencies, modify Apache Airflow configuration options, and set
         *        environment variables. For more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/using-startup-script.html">Using a startup
         *        script</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startupScriptS3Path(String startupScriptS3Path);

        /**
         * <p>
         * The key-value tag pairs you want to associate to your environment. For example,
         * <code>"Environment": "Staging"</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services
         * resources</a>.
         * </p>
         * 
         * @param tags
         *        The key-value tag pairs you want to associate to your environment. For example,
         *        <code>"Environment": "Staging"</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services
         *        resources</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The Apache Airflow <i>Web server</i> access mode. For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-networking.html">Apache Airflow access
         * modes</a>.
         * </p>
         * 
         * @param webserverAccessMode
         *        The Apache Airflow <i>Web server</i> access mode. For more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-networking.html">Apache Airflow
         *        access modes</a>.
         * @see WebserverAccessMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WebserverAccessMode
         */
        Builder webserverAccessMode(String webserverAccessMode);

        /**
         * <p>
         * The Apache Airflow <i>Web server</i> access mode. For more information, see <a
         * href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-networking.html">Apache Airflow access
         * modes</a>.
         * </p>
         * 
         * @param webserverAccessMode
         *        The Apache Airflow <i>Web server</i> access mode. For more information, see <a
         *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/configuring-networking.html">Apache Airflow
         *        access modes</a>.
         * @see WebserverAccessMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WebserverAccessMode
         */
        Builder webserverAccessMode(WebserverAccessMode webserverAccessMode);

        /**
         * <p>
         * The day and time of the week in Coordinated Universal Time (UTC) 24-hour standard time to start weekly
         * maintenance updates of your environment in the following format: <code>DAY:HH:MM</code>. For example:
         * <code>TUE:03:30</code>. You can specify a start time in 30 minute increments only.
         * </p>
         * 
         * @param weeklyMaintenanceWindowStart
         *        The day and time of the week in Coordinated Universal Time (UTC) 24-hour standard time to start weekly
         *        maintenance updates of your environment in the following format: <code>DAY:HH:MM</code>. For example:
         *        <code>TUE:03:30</code>. You can specify a start time in 30 minute increments only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder weeklyMaintenanceWindowStart(String weeklyMaintenanceWindowStart);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends MwaaRequest.BuilderImpl implements Builder {
        private Map<String, String> airflowConfigurationOptions = DefaultSdkAutoConstructMap.getInstance();

        private String airflowVersion;

        private String dagS3Path;

        private String environmentClass;

        private String executionRoleArn;

        private String kmsKey;

        private LoggingConfigurationInput loggingConfiguration;

        private Integer maxWorkers;

        private Integer minWorkers;

        private String name;

        private NetworkConfiguration networkConfiguration;

        private String pluginsS3ObjectVersion;

        private String pluginsS3Path;

        private String requirementsS3ObjectVersion;

        private String requirementsS3Path;

        private Integer schedulers;

        private String sourceBucketArn;

        private String startupScriptS3ObjectVersion;

        private String startupScriptS3Path;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String webserverAccessMode;

        private String weeklyMaintenanceWindowStart;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateEnvironmentRequest model) {
            super(model);
            airflowConfigurationOptions(model.airflowConfigurationOptions);
            airflowVersion(model.airflowVersion);
            dagS3Path(model.dagS3Path);
            environmentClass(model.environmentClass);
            executionRoleArn(model.executionRoleArn);
            kmsKey(model.kmsKey);
            loggingConfiguration(model.loggingConfiguration);
            maxWorkers(model.maxWorkers);
            minWorkers(model.minWorkers);
            name(model.name);
            networkConfiguration(model.networkConfiguration);
            pluginsS3ObjectVersion(model.pluginsS3ObjectVersion);
            pluginsS3Path(model.pluginsS3Path);
            requirementsS3ObjectVersion(model.requirementsS3ObjectVersion);
            requirementsS3Path(model.requirementsS3Path);
            schedulers(model.schedulers);
            sourceBucketArn(model.sourceBucketArn);
            startupScriptS3ObjectVersion(model.startupScriptS3ObjectVersion);
            startupScriptS3Path(model.startupScriptS3Path);
            tags(model.tags);
            webserverAccessMode(model.webserverAccessMode);
            weeklyMaintenanceWindowStart(model.weeklyMaintenanceWindowStart);
        }

        public final Map<String, String> getAirflowConfigurationOptions() {
            if (airflowConfigurationOptions instanceof SdkAutoConstructMap) {
                return null;
            }
            return airflowConfigurationOptions;
        }

        public final void setAirflowConfigurationOptions(Map<String, String> airflowConfigurationOptions) {
            this.airflowConfigurationOptions = AirflowConfigurationOptionsCopier.copy(airflowConfigurationOptions);
        }

        @Override
        public final Builder airflowConfigurationOptions(Map<String, String> airflowConfigurationOptions) {
            this.airflowConfigurationOptions = AirflowConfigurationOptionsCopier.copy(airflowConfigurationOptions);
            return this;
        }

        public final String getAirflowVersion() {
            return airflowVersion;
        }

        public final void setAirflowVersion(String airflowVersion) {
            this.airflowVersion = airflowVersion;
        }

        @Override
        public final Builder airflowVersion(String airflowVersion) {
            this.airflowVersion = airflowVersion;
            return this;
        }

        public final String getDagS3Path() {
            return dagS3Path;
        }

        public final void setDagS3Path(String dagS3Path) {
            this.dagS3Path = dagS3Path;
        }

        @Override
        public final Builder dagS3Path(String dagS3Path) {
            this.dagS3Path = dagS3Path;
            return this;
        }

        public final String getEnvironmentClass() {
            return environmentClass;
        }

        public final void setEnvironmentClass(String environmentClass) {
            this.environmentClass = environmentClass;
        }

        @Override
        public final Builder environmentClass(String environmentClass) {
            this.environmentClass = environmentClass;
            return this;
        }

        public final String getExecutionRoleArn() {
            return executionRoleArn;
        }

        public final void setExecutionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
        }

        @Override
        public final Builder executionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
            return this;
        }

        public final String getKmsKey() {
            return kmsKey;
        }

        public final void setKmsKey(String kmsKey) {
            this.kmsKey = kmsKey;
        }

        @Override
        public final Builder kmsKey(String kmsKey) {
            this.kmsKey = kmsKey;
            return this;
        }

        public final LoggingConfigurationInput.Builder getLoggingConfiguration() {
            return loggingConfiguration != null ? loggingConfiguration.toBuilder() : null;
        }

        public final void setLoggingConfiguration(LoggingConfigurationInput.BuilderImpl loggingConfiguration) {
            this.loggingConfiguration = loggingConfiguration != null ? loggingConfiguration.build() : null;
        }

        @Override
        public final Builder loggingConfiguration(LoggingConfigurationInput loggingConfiguration) {
            this.loggingConfiguration = loggingConfiguration;
            return this;
        }

        public final Integer getMaxWorkers() {
            return maxWorkers;
        }

        public final void setMaxWorkers(Integer maxWorkers) {
            this.maxWorkers = maxWorkers;
        }

        @Override
        public final Builder maxWorkers(Integer maxWorkers) {
            this.maxWorkers = maxWorkers;
            return this;
        }

        public final Integer getMinWorkers() {
            return minWorkers;
        }

        public final void setMinWorkers(Integer minWorkers) {
            this.minWorkers = minWorkers;
        }

        @Override
        public final Builder minWorkers(Integer minWorkers) {
            this.minWorkers = minWorkers;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final NetworkConfiguration.Builder getNetworkConfiguration() {
            return networkConfiguration != null ? networkConfiguration.toBuilder() : null;
        }

        public final void setNetworkConfiguration(NetworkConfiguration.BuilderImpl networkConfiguration) {
            this.networkConfiguration = networkConfiguration != null ? networkConfiguration.build() : null;
        }

        @Override
        public final Builder networkConfiguration(NetworkConfiguration networkConfiguration) {
            this.networkConfiguration = networkConfiguration;
            return this;
        }

        public final String getPluginsS3ObjectVersion() {
            return pluginsS3ObjectVersion;
        }

        public final void setPluginsS3ObjectVersion(String pluginsS3ObjectVersion) {
            this.pluginsS3ObjectVersion = pluginsS3ObjectVersion;
        }

        @Override
        public final Builder pluginsS3ObjectVersion(String pluginsS3ObjectVersion) {
            this.pluginsS3ObjectVersion = pluginsS3ObjectVersion;
            return this;
        }

        public final String getPluginsS3Path() {
            return pluginsS3Path;
        }

        public final void setPluginsS3Path(String pluginsS3Path) {
            this.pluginsS3Path = pluginsS3Path;
        }

        @Override
        public final Builder pluginsS3Path(String pluginsS3Path) {
            this.pluginsS3Path = pluginsS3Path;
            return this;
        }

        public final String getRequirementsS3ObjectVersion() {
            return requirementsS3ObjectVersion;
        }

        public final void setRequirementsS3ObjectVersion(String requirementsS3ObjectVersion) {
            this.requirementsS3ObjectVersion = requirementsS3ObjectVersion;
        }

        @Override
        public final Builder requirementsS3ObjectVersion(String requirementsS3ObjectVersion) {
            this.requirementsS3ObjectVersion = requirementsS3ObjectVersion;
            return this;
        }

        public final String getRequirementsS3Path() {
            return requirementsS3Path;
        }

        public final void setRequirementsS3Path(String requirementsS3Path) {
            this.requirementsS3Path = requirementsS3Path;
        }

        @Override
        public final Builder requirementsS3Path(String requirementsS3Path) {
            this.requirementsS3Path = requirementsS3Path;
            return this;
        }

        public final Integer getSchedulers() {
            return schedulers;
        }

        public final void setSchedulers(Integer schedulers) {
            this.schedulers = schedulers;
        }

        @Override
        public final Builder schedulers(Integer schedulers) {
            this.schedulers = schedulers;
            return this;
        }

        public final String getSourceBucketArn() {
            return sourceBucketArn;
        }

        public final void setSourceBucketArn(String sourceBucketArn) {
            this.sourceBucketArn = sourceBucketArn;
        }

        @Override
        public final Builder sourceBucketArn(String sourceBucketArn) {
            this.sourceBucketArn = sourceBucketArn;
            return this;
        }

        public final String getStartupScriptS3ObjectVersion() {
            return startupScriptS3ObjectVersion;
        }

        public final void setStartupScriptS3ObjectVersion(String startupScriptS3ObjectVersion) {
            this.startupScriptS3ObjectVersion = startupScriptS3ObjectVersion;
        }

        @Override
        public final Builder startupScriptS3ObjectVersion(String startupScriptS3ObjectVersion) {
            this.startupScriptS3ObjectVersion = startupScriptS3ObjectVersion;
            return this;
        }

        public final String getStartupScriptS3Path() {
            return startupScriptS3Path;
        }

        public final void setStartupScriptS3Path(String startupScriptS3Path) {
            this.startupScriptS3Path = startupScriptS3Path;
        }

        @Override
        public final Builder startupScriptS3Path(String startupScriptS3Path) {
            this.startupScriptS3Path = startupScriptS3Path;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final String getWebserverAccessMode() {
            return webserverAccessMode;
        }

        public final void setWebserverAccessMode(String webserverAccessMode) {
            this.webserverAccessMode = webserverAccessMode;
        }

        @Override
        public final Builder webserverAccessMode(String webserverAccessMode) {
            this.webserverAccessMode = webserverAccessMode;
            return this;
        }

        @Override
        public final Builder webserverAccessMode(WebserverAccessMode webserverAccessMode) {
            this.webserverAccessMode(webserverAccessMode == null ? null : webserverAccessMode.toString());
            return this;
        }

        public final String getWeeklyMaintenanceWindowStart() {
            return weeklyMaintenanceWindowStart;
        }

        public final void setWeeklyMaintenanceWindowStart(String weeklyMaintenanceWindowStart) {
            this.weeklyMaintenanceWindowStart = weeklyMaintenanceWindowStart;
        }

        @Override
        public final Builder weeklyMaintenanceWindowStart(String weeklyMaintenanceWindowStart) {
            this.weeklyMaintenanceWindowStart = weeklyMaintenanceWindowStart;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateEnvironmentRequest build() {
            return new CreateEnvironmentRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
