/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mwaa;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.mwaa.model.AccessDeniedException;
import software.amazon.awssdk.services.mwaa.model.CreateCliTokenRequest;
import software.amazon.awssdk.services.mwaa.model.CreateCliTokenResponse;
import software.amazon.awssdk.services.mwaa.model.CreateEnvironmentRequest;
import software.amazon.awssdk.services.mwaa.model.CreateEnvironmentResponse;
import software.amazon.awssdk.services.mwaa.model.CreateWebLoginTokenRequest;
import software.amazon.awssdk.services.mwaa.model.CreateWebLoginTokenResponse;
import software.amazon.awssdk.services.mwaa.model.DeleteEnvironmentRequest;
import software.amazon.awssdk.services.mwaa.model.DeleteEnvironmentResponse;
import software.amazon.awssdk.services.mwaa.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.mwaa.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.mwaa.model.InternalServerException;
import software.amazon.awssdk.services.mwaa.model.ListEnvironmentsRequest;
import software.amazon.awssdk.services.mwaa.model.ListEnvironmentsResponse;
import software.amazon.awssdk.services.mwaa.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mwaa.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mwaa.model.MwaaException;
import software.amazon.awssdk.services.mwaa.model.PublishMetricsRequest;
import software.amazon.awssdk.services.mwaa.model.PublishMetricsResponse;
import software.amazon.awssdk.services.mwaa.model.ResourceNotFoundException;
import software.amazon.awssdk.services.mwaa.model.TagResourceRequest;
import software.amazon.awssdk.services.mwaa.model.TagResourceResponse;
import software.amazon.awssdk.services.mwaa.model.UntagResourceRequest;
import software.amazon.awssdk.services.mwaa.model.UntagResourceResponse;
import software.amazon.awssdk.services.mwaa.model.UpdateEnvironmentRequest;
import software.amazon.awssdk.services.mwaa.model.UpdateEnvironmentResponse;
import software.amazon.awssdk.services.mwaa.model.ValidationException;
import software.amazon.awssdk.services.mwaa.paginators.ListEnvironmentsIterable;

/**
 * Service client for accessing AmazonMWAA. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * <fullname>Amazon Managed Workflows for Apache Airflow</fullname>
 * <p>
 * This section contains the Amazon Managed Workflows for Apache Airflow (MWAA) API reference documentation. For more
 * information, see <a href="https://docs.aws.amazon.com/mwaa/latest/userguide/what-is-mwaa.html">What is Amazon
 * MWAA?</a>.
 * </p>
 * <p>
 * <b>Endpoints</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>api.airflow.{region}.amazonaws.com</code> - This endpoint is used for environment management.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/mwaa/latest/API/API_CreateEnvironment.html">CreateEnvironment</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/mwaa/latest/API/API_DeleteEnvironment.html">DeleteEnvironment</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/mwaa/latest/API/API_GetEnvironment.html">GetEnvironment</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/mwaa/latest/API/API_ListEnvironments.html">ListEnvironments</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/mwaa/latest/API/API_ListTagsForResource.html">ListTagsForResource</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/mwaa/latest/API/API_TagResource.html">TagResource</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/mwaa/latest/API/API_UntagResource.html">UntagResource</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/mwaa/latest/API/API_UpdateEnvironment.html">UpdateEnvironment</a>
 * </p>
 * </li>
 * </ul>
 * </li>
 * <li>
 * <p>
 * <code>env.airflow.{region}.amazonaws.com</code> - This endpoint is used to operate the Airflow environment.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/mwaa/latest/API/API_CreateCliToken.html ">CreateCliToken</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/mwaa/latest/API/API_CreateWebLoginToken.html">CreateWebLoginToken</a>
 * </p>
 * </li>
 * </ul>
 * </li>
 * <li>
 * <p>
 * <code>ops.airflow.{region}.amazonaws.com</code> - This endpoint is used to push environment metrics that track
 * environment health.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/mwaa/latest/API/API_PublishMetrics.html ">PublishMetrics</a>
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * <p>
 * <b>Regions</b>
 * </p>
 * <p>
 * For a list of regions that Amazon MWAA supports, see <a
 * href="https://docs.aws.amazon.com/mwaa/latest/userguide/what-is-mwaa.html#regions-mwaa">Region availability</a> in
 * the <i>Amazon MWAA User Guide</i>.
 * </p>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface MwaaClient extends AwsClient {
    String SERVICE_NAME = "airflow";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "airflow";

    /**
     * <p>
     * Creates a CLI token for the Airflow CLI. To learn more, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/call-mwaa-apis-cli.html">Creating an Apache Airflow CLI
     * token</a>.
     * </p>
     *
     * @param createCliTokenRequest
     * @return Result of the CreateCliToken operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.CreateCliToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/CreateCliToken" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateCliTokenResponse createCliToken(CreateCliTokenRequest createCliTokenRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, MwaaException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a CLI token for the Airflow CLI. To learn more, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/call-mwaa-apis-cli.html">Creating an Apache Airflow CLI
     * token</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateCliTokenRequest.Builder} avoiding the need to
     * create one manually via {@link CreateCliTokenRequest#builder()}
     * </p>
     *
     * @param createCliTokenRequest
     *        A {@link Consumer} that will call methods on {@link CreateCliTokenRequest.Builder} to create a request.
     * @return Result of the CreateCliToken operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.CreateCliToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/CreateCliToken" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateCliTokenResponse createCliToken(Consumer<CreateCliTokenRequest.Builder> createCliTokenRequest)
            throws ResourceNotFoundException, AwsServiceException, SdkClientException, MwaaException {
        return createCliToken(CreateCliTokenRequest.builder().applyMutation(createCliTokenRequest).build());
    }

    /**
     * <p>
     * Creates an Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * </p>
     *
     * @param createEnvironmentRequest
     *        This section contains the Amazon Managed Workflows for Apache Airflow (MWAA) API reference documentation
     *        to create an environment. For more information, see <a
     *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/get-started.html">Get started with Amazon Managed
     *        Workflows for Apache Airflow</a>.
     * @return Result of the CreateEnvironment operation returned by the service.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.CreateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/CreateEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateEnvironmentResponse createEnvironment(CreateEnvironmentRequest createEnvironmentRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateEnvironmentRequest.Builder} avoiding the need
     * to create one manually via {@link CreateEnvironmentRequest#builder()}
     * </p>
     *
     * @param createEnvironmentRequest
     *        A {@link Consumer} that will call methods on {@link CreateEnvironmentInput.Builder} to create a request.
     *        This section contains the Amazon Managed Workflows for Apache Airflow (MWAA) API reference documentation
     *        to create an environment. For more information, see <a
     *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/get-started.html">Get started with Amazon Managed
     *        Workflows for Apache Airflow</a>.
     * @return Result of the CreateEnvironment operation returned by the service.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.CreateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/CreateEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateEnvironmentResponse createEnvironment(Consumer<CreateEnvironmentRequest.Builder> createEnvironmentRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        return createEnvironment(CreateEnvironmentRequest.builder().applyMutation(createEnvironmentRequest).build());
    }

    /**
     * <p>
     * Creates a web login token for the Airflow Web UI. To learn more, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/call-mwaa-apis-web.html">Creating an Apache Airflow web
     * login token</a>.
     * </p>
     *
     * @param createWebLoginTokenRequest
     * @return Result of the CreateWebLoginToken operation returned by the service.
     * @throws AccessDeniedException
     *         Access to the Apache Airflow Web UI or CLI has been denied due to insufficient permissions. To learn
     *         more, see <a href="https://docs.aws.amazon.com/mwaa/latest/userguide/access-policies.html">Accessing an
     *         Amazon MWAA environment</a>.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.CreateWebLoginToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/CreateWebLoginToken" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateWebLoginTokenResponse createWebLoginToken(CreateWebLoginTokenRequest createWebLoginTokenRequest)
            throws AccessDeniedException, ResourceNotFoundException, ValidationException, InternalServerException,
            AwsServiceException, SdkClientException, MwaaException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a web login token for the Airflow Web UI. To learn more, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/call-mwaa-apis-web.html">Creating an Apache Airflow web
     * login token</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateWebLoginTokenRequest.Builder} avoiding the
     * need to create one manually via {@link CreateWebLoginTokenRequest#builder()}
     * </p>
     *
     * @param createWebLoginTokenRequest
     *        A {@link Consumer} that will call methods on {@link CreateWebLoginTokenRequest.Builder} to create a
     *        request.
     * @return Result of the CreateWebLoginToken operation returned by the service.
     * @throws AccessDeniedException
     *         Access to the Apache Airflow Web UI or CLI has been denied due to insufficient permissions. To learn
     *         more, see <a href="https://docs.aws.amazon.com/mwaa/latest/userguide/access-policies.html">Accessing an
     *         Amazon MWAA environment</a>.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.CreateWebLoginToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/CreateWebLoginToken" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateWebLoginTokenResponse createWebLoginToken(
            Consumer<CreateWebLoginTokenRequest.Builder> createWebLoginTokenRequest) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            MwaaException {
        return createWebLoginToken(CreateWebLoginTokenRequest.builder().applyMutation(createWebLoginTokenRequest).build());
    }

    /**
     * <p>
     * Deletes an Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * </p>
     *
     * @param deleteEnvironmentRequest
     * @return Result of the DeleteEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.DeleteEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/DeleteEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteEnvironmentResponse deleteEnvironment(DeleteEnvironmentRequest deleteEnvironmentRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MwaaException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes an Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteEnvironmentRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteEnvironmentRequest#builder()}
     * </p>
     *
     * @param deleteEnvironmentRequest
     *        A {@link Consumer} that will call methods on {@link DeleteEnvironmentInput.Builder} to create a request.
     * @return Result of the DeleteEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.DeleteEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/DeleteEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteEnvironmentResponse deleteEnvironment(Consumer<DeleteEnvironmentRequest.Builder> deleteEnvironmentRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MwaaException {
        return deleteEnvironment(DeleteEnvironmentRequest.builder().applyMutation(deleteEnvironmentRequest).build());
    }

    /**
     * <p>
     * Describes an Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * </p>
     *
     * @param getEnvironmentRequest
     * @return Result of the GetEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.GetEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/GetEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default GetEnvironmentResponse getEnvironment(GetEnvironmentRequest getEnvironmentRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes an Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetEnvironmentRequest.Builder} avoiding the need to
     * create one manually via {@link GetEnvironmentRequest#builder()}
     * </p>
     *
     * @param getEnvironmentRequest
     *        A {@link Consumer} that will call methods on {@link GetEnvironmentInput.Builder} to create a request.
     * @return Result of the GetEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.GetEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/GetEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default GetEnvironmentResponse getEnvironment(Consumer<GetEnvironmentRequest.Builder> getEnvironmentRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MwaaException {
        return getEnvironment(GetEnvironmentRequest.builder().applyMutation(getEnvironmentRequest).build());
    }

    /**
     * <p>
     * Lists the Amazon Managed Workflows for Apache Airflow (MWAA) environments.
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return Result of the ListEnvironments operation returned by the service.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentsResponse listEnvironments(ListEnvironmentsRequest listEnvironmentsRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the Amazon Managed Workflows for Apache Airflow (MWAA) environments.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEnvironmentsRequest.Builder} avoiding the need
     * to create one manually via {@link ListEnvironmentsRequest#builder()}
     * </p>
     *
     * @param listEnvironmentsRequest
     *        A {@link Consumer} that will call methods on {@link ListEnvironmentsInput.Builder} to create a request.
     * @return Result of the ListEnvironments operation returned by the service.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentsResponse listEnvironments(Consumer<ListEnvironmentsRequest.Builder> listEnvironmentsRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        return listEnvironments(ListEnvironmentsRequest.builder().applyMutation(listEnvironmentsRequest).build());
    }

    /**
     * <p>
     * Lists the Amazon Managed Workflows for Apache Airflow (MWAA) environments.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEnvironments(software.amazon.awssdk.services.mwaa.model.ListEnvironmentsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mwaa.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mwaa.paginators.ListEnvironmentsIterable responses = client
     *             .listEnvironmentsPaginator(request);
     *     for (software.amazon.awssdk.services.mwaa.model.ListEnvironmentsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mwaa.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironments(software.amazon.awssdk.services.mwaa.model.ListEnvironmentsRequest)} operation.</b>
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentsIterable listEnvironmentsPaginator(ListEnvironmentsRequest listEnvironmentsRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the Amazon Managed Workflows for Apache Airflow (MWAA) environments.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEnvironments(software.amazon.awssdk.services.mwaa.model.ListEnvironmentsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mwaa.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mwaa.paginators.ListEnvironmentsIterable responses = client
     *             .listEnvironmentsPaginator(request);
     *     for (software.amazon.awssdk.services.mwaa.model.ListEnvironmentsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mwaa.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironments(software.amazon.awssdk.services.mwaa.model.ListEnvironmentsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEnvironmentsRequest.Builder} avoiding the need
     * to create one manually via {@link ListEnvironmentsRequest#builder()}
     * </p>
     *
     * @param listEnvironmentsRequest
     *        A {@link Consumer} that will call methods on {@link ListEnvironmentsInput.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentsIterable listEnvironmentsPaginator(Consumer<ListEnvironmentsRequest.Builder> listEnvironmentsRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        return listEnvironmentsPaginator(ListEnvironmentsRequest.builder().applyMutation(listEnvironmentsRequest).build());
    }

    /**
     * <p>
     * Lists the key-value tag pairs associated to the Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * For example, <code>"Environment": "Staging"</code>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MwaaException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the key-value tag pairs associated to the Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * For example, <code>"Environment": "Staging"</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceInput.Builder} to create a request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * <b>Internal only</b>. Publishes environment health metrics to Amazon CloudWatch.
     * </p>
     *
     * @param publishMetricsRequest
     * @return Result of the PublishMetrics operation returned by the service.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.PublishMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/PublishMetrics" target="_top">AWS API
     *      Documentation</a>
     */
    default PublishMetricsResponse publishMetrics(PublishMetricsRequest publishMetricsRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * <b>Internal only</b>. Publishes environment health metrics to Amazon CloudWatch.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PublishMetricsRequest.Builder} avoiding the need to
     * create one manually via {@link PublishMetricsRequest#builder()}
     * </p>
     *
     * @param publishMetricsRequest
     *        A {@link Consumer} that will call methods on {@link PublishMetricsInput.Builder} to create a request.
     * @return Result of the PublishMetrics operation returned by the service.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.PublishMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/PublishMetrics" target="_top">AWS API
     *      Documentation</a>
     */
    default PublishMetricsResponse publishMetrics(Consumer<PublishMetricsRequest.Builder> publishMetricsRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        return publishMetrics(PublishMetricsRequest.builder().applyMutation(publishMetricsRequest).build());
    }

    /**
     * <p>
     * Associates key-value tag pairs to your Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Associates key-value tag pairs to your Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on {@link TagResourceInput.Builder} to create a request.
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MwaaException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes key-value tag pairs associated to your Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * For example, <code>"Environment": "Staging"</code>.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes key-value tag pairs associated to your Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * For example, <code>"Environment": "Staging"</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on {@link UntagResourceInput.Builder} to create a request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MwaaException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates an Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * </p>
     *
     * @param updateEnvironmentRequest
     * @return Result of the UpdateEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.UpdateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/UpdateEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateEnvironmentResponse updateEnvironment(UpdateEnvironmentRequest updateEnvironmentRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MwaaException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates an Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateEnvironmentRequest.Builder} avoiding the need
     * to create one manually via {@link UpdateEnvironmentRequest#builder()}
     * </p>
     *
     * @param updateEnvironmentRequest
     *        A {@link Consumer} that will call methods on {@link UpdateEnvironmentInput.Builder} to create a request.
     * @return Result of the UpdateEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.UpdateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/UpdateEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateEnvironmentResponse updateEnvironment(Consumer<UpdateEnvironmentRequest.Builder> updateEnvironmentRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MwaaException {
        return updateEnvironment(UpdateEnvironmentRequest.builder().applyMutation(updateEnvironmentRequest).build());
    }

    /**
     * Create a {@link MwaaClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static MwaaClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link MwaaClient}.
     */
    static MwaaClientBuilder builder() {
        return new DefaultMwaaClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default MwaaServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
