/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mwaa;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.mwaa.internal.MwaaServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.mwaa.model.AccessDeniedException;
import software.amazon.awssdk.services.mwaa.model.CreateCliTokenRequest;
import software.amazon.awssdk.services.mwaa.model.CreateCliTokenResponse;
import software.amazon.awssdk.services.mwaa.model.CreateEnvironmentRequest;
import software.amazon.awssdk.services.mwaa.model.CreateEnvironmentResponse;
import software.amazon.awssdk.services.mwaa.model.CreateWebLoginTokenRequest;
import software.amazon.awssdk.services.mwaa.model.CreateWebLoginTokenResponse;
import software.amazon.awssdk.services.mwaa.model.DeleteEnvironmentRequest;
import software.amazon.awssdk.services.mwaa.model.DeleteEnvironmentResponse;
import software.amazon.awssdk.services.mwaa.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.mwaa.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.mwaa.model.InternalServerException;
import software.amazon.awssdk.services.mwaa.model.ListEnvironmentsRequest;
import software.amazon.awssdk.services.mwaa.model.ListEnvironmentsResponse;
import software.amazon.awssdk.services.mwaa.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mwaa.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mwaa.model.MwaaException;
import software.amazon.awssdk.services.mwaa.model.PublishMetricsRequest;
import software.amazon.awssdk.services.mwaa.model.PublishMetricsResponse;
import software.amazon.awssdk.services.mwaa.model.ResourceNotFoundException;
import software.amazon.awssdk.services.mwaa.model.TagResourceRequest;
import software.amazon.awssdk.services.mwaa.model.TagResourceResponse;
import software.amazon.awssdk.services.mwaa.model.UntagResourceRequest;
import software.amazon.awssdk.services.mwaa.model.UntagResourceResponse;
import software.amazon.awssdk.services.mwaa.model.UpdateEnvironmentRequest;
import software.amazon.awssdk.services.mwaa.model.UpdateEnvironmentResponse;
import software.amazon.awssdk.services.mwaa.model.ValidationException;
import software.amazon.awssdk.services.mwaa.transform.CreateCliTokenRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.CreateEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.CreateWebLoginTokenRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.DeleteEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.GetEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.ListEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.PublishMetricsRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.UpdateEnvironmentRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MwaaClient}.
 *
 * @see MwaaClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMwaaClient implements MwaaClient {
    private static final Logger log = Logger.loggerFor(DefaultMwaaClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMwaaClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a CLI token for the Airflow CLI. To learn more, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/call-mwaa-apis-cli.html">Creating an Apache Airflow CLI
     * token</a>.
     * </p>
     *
     * @param createCliTokenRequest
     * @return Result of the CreateCliToken operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.CreateCliToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/CreateCliToken" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateCliTokenResponse createCliToken(CreateCliTokenRequest createCliTokenRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCliTokenResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateCliTokenResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createCliTokenRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCliTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCliToken");
            String hostPrefix = "env.";
            String resolvedHostExpression = "env.";

            return clientHandler.execute(new ClientExecutionParams<CreateCliTokenRequest, CreateCliTokenResponse>()
                    .withOperationName("CreateCliToken").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .hostPrefixExpression(resolvedHostExpression).withRequestConfiguration(clientConfiguration)
                    .withInput(createCliTokenRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateCliTokenRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * </p>
     *
     * @param createEnvironmentRequest
     *        This section contains the Amazon Managed Workflows for Apache Airflow (MWAA) API reference documentation
     *        to create an environment. For more information, see <a
     *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/get-started.html">Get started with Amazon Managed
     *        Workflows for Apache Airflow</a>.
     * @return Result of the CreateEnvironment operation returned by the service.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.CreateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/CreateEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateEnvironmentResponse createEnvironment(CreateEnvironmentRequest createEnvironmentRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEnvironment");
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            return clientHandler.execute(new ClientExecutionParams<CreateEnvironmentRequest, CreateEnvironmentResponse>()
                    .withOperationName("CreateEnvironment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .hostPrefixExpression(resolvedHostExpression).withRequestConfiguration(clientConfiguration)
                    .withInput(createEnvironmentRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a web login token for the Airflow Web UI. To learn more, see <a
     * href="https://docs.aws.amazon.com/mwaa/latest/userguide/call-mwaa-apis-web.html">Creating an Apache Airflow web
     * login token</a>.
     * </p>
     *
     * @param createWebLoginTokenRequest
     * @return Result of the CreateWebLoginToken operation returned by the service.
     * @throws AccessDeniedException
     *         Access to the Apache Airflow Web UI or CLI has been denied due to insufficient permissions. To learn
     *         more, see <a href="https://docs.aws.amazon.com/mwaa/latest/userguide/access-policies.html">Accessing an
     *         Amazon MWAA environment</a>.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.CreateWebLoginToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/CreateWebLoginToken" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateWebLoginTokenResponse createWebLoginToken(CreateWebLoginTokenRequest createWebLoginTokenRequest)
            throws AccessDeniedException, ResourceNotFoundException, ValidationException, InternalServerException,
            AwsServiceException, SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateWebLoginTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateWebLoginTokenResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWebLoginTokenRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWebLoginTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWebLoginToken");
            String hostPrefix = "env.";
            String resolvedHostExpression = "env.";

            return clientHandler.execute(new ClientExecutionParams<CreateWebLoginTokenRequest, CreateWebLoginTokenResponse>()
                    .withOperationName("CreateWebLoginToken").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .hostPrefixExpression(resolvedHostExpression).withRequestConfiguration(clientConfiguration)
                    .withInput(createWebLoginTokenRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateWebLoginTokenRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * </p>
     *
     * @param deleteEnvironmentRequest
     * @return Result of the DeleteEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.DeleteEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/DeleteEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteEnvironmentResponse deleteEnvironment(DeleteEnvironmentRequest deleteEnvironmentRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEnvironment");
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            return clientHandler.execute(new ClientExecutionParams<DeleteEnvironmentRequest, DeleteEnvironmentResponse>()
                    .withOperationName("DeleteEnvironment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .hostPrefixExpression(resolvedHostExpression).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteEnvironmentRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * </p>
     *
     * @param getEnvironmentRequest
     * @return Result of the GetEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.GetEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/GetEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetEnvironmentResponse getEnvironment(GetEnvironmentRequest getEnvironmentRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEnvironmentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEnvironment");
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            return clientHandler.execute(new ClientExecutionParams<GetEnvironmentRequest, GetEnvironmentResponse>()
                    .withOperationName("GetEnvironment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .hostPrefixExpression(resolvedHostExpression).withRequestConfiguration(clientConfiguration)
                    .withInput(getEnvironmentRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the Amazon Managed Workflows for Apache Airflow (MWAA) environments.
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return Result of the ListEnvironments operation returned by the service.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListEnvironmentsResponse listEnvironments(ListEnvironmentsRequest listEnvironmentsRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListEnvironmentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEnvironmentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEnvironmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEnvironments");
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            return clientHandler.execute(new ClientExecutionParams<ListEnvironmentsRequest, ListEnvironmentsResponse>()
                    .withOperationName("ListEnvironments").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .hostPrefixExpression(resolvedHostExpression).withRequestConfiguration(clientConfiguration)
                    .withInput(listEnvironmentsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListEnvironmentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the key-value tag pairs associated to the Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * For example, <code>"Environment": "Staging"</code>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .hostPrefixExpression(resolvedHostExpression).withRequestConfiguration(clientConfiguration)
                    .withInput(listTagsForResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <b>Internal only</b>. Publishes environment health metrics to Amazon CloudWatch.
     * </p>
     *
     * @param publishMetricsRequest
     * @return Result of the PublishMetrics operation returned by the service.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.PublishMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/PublishMetrics" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PublishMetricsResponse publishMetrics(PublishMetricsRequest publishMetricsRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PublishMetricsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PublishMetricsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(publishMetricsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, publishMetricsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PublishMetrics");
            String hostPrefix = "ops.";
            String resolvedHostExpression = "ops.";

            return clientHandler.execute(new ClientExecutionParams<PublishMetricsRequest, PublishMetricsResponse>()
                    .withOperationName("PublishMetrics").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .hostPrefixExpression(resolvedHostExpression).withRequestConfiguration(clientConfiguration)
                    .withInput(publishMetricsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PublishMetricsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates key-value tag pairs to your Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withRequestConfiguration(clientConfiguration).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes key-value tag pairs associated to your Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * For example, <code>"Environment": "Staging"</code>.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .hostPrefixExpression(resolvedHostExpression).withRequestConfiguration(clientConfiguration)
                    .withInput(untagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an Amazon Managed Workflows for Apache Airflow (MWAA) environment.
     * </p>
     *
     * @param updateEnvironmentRequest
     * @return Result of the UpdateEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.UpdateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/UpdateEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateEnvironmentResponse updateEnvironment(UpdateEnvironmentRequest updateEnvironmentRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnvironment");
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            return clientHandler.execute(new ClientExecutionParams<UpdateEnvironmentRequest, UpdateEnvironmentResponse>()
                    .withOperationName("UpdateEnvironment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .hostPrefixExpression(resolvedHostExpression).withRequestConfiguration(clientConfiguration)
                    .withInput(updateEnvironmentRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        MwaaServiceClientConfigurationBuilder serviceConfigBuilder = new MwaaServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MwaaException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final MwaaServiceClientConfiguration serviceClientConfiguration() {
        return new MwaaServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
