/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mwaaserverless;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.mwaaserverless.internal.MwaaServerlessServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.mwaaserverless.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.mwaaserverless.model.AccessDeniedException;
import software.amazon.awssdk.services.mwaaserverless.model.ConflictException;
import software.amazon.awssdk.services.mwaaserverless.model.CreateWorkflowRequest;
import software.amazon.awssdk.services.mwaaserverless.model.CreateWorkflowResponse;
import software.amazon.awssdk.services.mwaaserverless.model.DeleteWorkflowRequest;
import software.amazon.awssdk.services.mwaaserverless.model.DeleteWorkflowResponse;
import software.amazon.awssdk.services.mwaaserverless.model.GetTaskInstanceRequest;
import software.amazon.awssdk.services.mwaaserverless.model.GetTaskInstanceResponse;
import software.amazon.awssdk.services.mwaaserverless.model.GetWorkflowRequest;
import software.amazon.awssdk.services.mwaaserverless.model.GetWorkflowResponse;
import software.amazon.awssdk.services.mwaaserverless.model.GetWorkflowRunRequest;
import software.amazon.awssdk.services.mwaaserverless.model.GetWorkflowRunResponse;
import software.amazon.awssdk.services.mwaaserverless.model.InternalServerException;
import software.amazon.awssdk.services.mwaaserverless.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mwaaserverless.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mwaaserverless.model.ListTaskInstancesRequest;
import software.amazon.awssdk.services.mwaaserverless.model.ListTaskInstancesResponse;
import software.amazon.awssdk.services.mwaaserverless.model.ListWorkflowRunsRequest;
import software.amazon.awssdk.services.mwaaserverless.model.ListWorkflowRunsResponse;
import software.amazon.awssdk.services.mwaaserverless.model.ListWorkflowVersionsRequest;
import software.amazon.awssdk.services.mwaaserverless.model.ListWorkflowVersionsResponse;
import software.amazon.awssdk.services.mwaaserverless.model.ListWorkflowsRequest;
import software.amazon.awssdk.services.mwaaserverless.model.ListWorkflowsResponse;
import software.amazon.awssdk.services.mwaaserverless.model.MwaaServerlessException;
import software.amazon.awssdk.services.mwaaserverless.model.OperationTimeoutException;
import software.amazon.awssdk.services.mwaaserverless.model.ResourceNotFoundException;
import software.amazon.awssdk.services.mwaaserverless.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.mwaaserverless.model.StartWorkflowRunRequest;
import software.amazon.awssdk.services.mwaaserverless.model.StartWorkflowRunResponse;
import software.amazon.awssdk.services.mwaaserverless.model.StopWorkflowRunRequest;
import software.amazon.awssdk.services.mwaaserverless.model.StopWorkflowRunResponse;
import software.amazon.awssdk.services.mwaaserverless.model.TagResourceRequest;
import software.amazon.awssdk.services.mwaaserverless.model.TagResourceResponse;
import software.amazon.awssdk.services.mwaaserverless.model.ThrottlingException;
import software.amazon.awssdk.services.mwaaserverless.model.UntagResourceRequest;
import software.amazon.awssdk.services.mwaaserverless.model.UntagResourceResponse;
import software.amazon.awssdk.services.mwaaserverless.model.UpdateWorkflowRequest;
import software.amazon.awssdk.services.mwaaserverless.model.UpdateWorkflowResponse;
import software.amazon.awssdk.services.mwaaserverless.model.ValidationException;
import software.amazon.awssdk.services.mwaaserverless.transform.CreateWorkflowRequestMarshaller;
import software.amazon.awssdk.services.mwaaserverless.transform.DeleteWorkflowRequestMarshaller;
import software.amazon.awssdk.services.mwaaserverless.transform.GetTaskInstanceRequestMarshaller;
import software.amazon.awssdk.services.mwaaserverless.transform.GetWorkflowRequestMarshaller;
import software.amazon.awssdk.services.mwaaserverless.transform.GetWorkflowRunRequestMarshaller;
import software.amazon.awssdk.services.mwaaserverless.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mwaaserverless.transform.ListTaskInstancesRequestMarshaller;
import software.amazon.awssdk.services.mwaaserverless.transform.ListWorkflowRunsRequestMarshaller;
import software.amazon.awssdk.services.mwaaserverless.transform.ListWorkflowVersionsRequestMarshaller;
import software.amazon.awssdk.services.mwaaserverless.transform.ListWorkflowsRequestMarshaller;
import software.amazon.awssdk.services.mwaaserverless.transform.StartWorkflowRunRequestMarshaller;
import software.amazon.awssdk.services.mwaaserverless.transform.StopWorkflowRunRequestMarshaller;
import software.amazon.awssdk.services.mwaaserverless.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mwaaserverless.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.mwaaserverless.transform.UpdateWorkflowRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MwaaServerlessAsyncClient}.
 *
 * @see MwaaServerlessAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMwaaServerlessAsyncClient implements MwaaServerlessAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMwaaServerlessAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMwaaServerlessAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "MWAA_Serverless" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new workflow in Amazon Managed Workflows for Apache Airflow Serverless. This operation initializes a
     * workflow with the specified configuration including the workflow definition, execution role, and optional
     * settings for encryption, logging, and networking. You must provide the workflow definition as a YAML file stored
     * in Amazon S3 that defines the DAG structure using supported Amazon Web Services operators. Amazon Managed
     * Workflows for Apache Airflow Serverless automatically creates the first version of the workflow and sets up the
     * necessary execution environment with multi-tenant isolation and security controls.
     * </p>
     *
     * @param createWorkflowRequest
     * @return A Java Future containing the result of the CreateWorkflow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because too many requests were made in a short period,
     *         exceeding the service rate limits. Amazon Managed Workflows for Apache Airflow Serverless implements
     *         throttling controls to ensure fair resource allocation across all customers in the multi-tenant
     *         environment. This helps maintain service stability and performance. If you encounter throttling,
     *         implement exponential backoff and retry logic in your applications, or consider distributing your API
     *         calls over a longer time period.</li>
     *         <li>ValidationException The specified request parameters are invalid, missing, or inconsistent with
     *         Amazon Managed Workflows for Apache Airflow Serverless service requirements. This can occur when workflow
     *         definitions contain unsupported operators, when required IAM permissions are missing, when S3 locations
     *         are inaccessible, or when network configurations are invalid. The service validates workflow definitions,
     *         execution roles, and resource configurations to ensure compatibility with the managed Airflow environment
     *         and security requirements.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An unexpected server-side error occurred during request processing.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>ConflictException You cannot create a resource that already exists, or the resource is in a state
     *         that prevents the requested operation.</li>
     *         <li>ServiceQuotaExceededException The request exceeds the service quota for Amazon Managed Workflows for
     *         Apache Airflow Serverless resources. This can occur when you attempt to create more workflows than
     *         allowed, exceed concurrent workflow run limits, or surpass task execution limits. Amazon Managed
     *         Workflows for Apache Airflow Serverless implements admission control using DynamoDB-based counters to
     *         manage resource utilization across the multi-tenant environment. Contact Amazon Web Services Support to
     *         request quota increases if you need higher limits for your use case.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MwaaServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MwaaServerlessAsyncClient.CreateWorkflow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-serverless-2024-07-26/CreateWorkflow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWorkflowResponse> createWorkflow(CreateWorkflowRequest createWorkflowRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWorkflowRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkflowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkflow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWorkflowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateWorkflowResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "OperationTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OperationTimeoutException").httpStatusCode(500)
                            .exceptionBuilderSupplier(OperationTimeoutException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateWorkflowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWorkflowRequest, CreateWorkflowResponse>()
                            .withOperationName("CreateWorkflow").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateWorkflowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createWorkflowRequest));
            CompletableFuture<CreateWorkflowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a workflow and all its versions. This operation permanently removes the workflow and cannot be undone.
     * Amazon Managed Workflows for Apache Airflow Serverless ensures that all associated resources are properly cleaned
     * up, including stopping any running executions, removing scheduled triggers, and cleaning up execution history.
     * The deletion process respects the multi-tenant isolation boundaries and ensures that no residual data or
     * configurations remain that could affect other customers or workflows.
     * </p>
     *
     * @param deleteWorkflowRequest
     * @return A Java Future containing the result of the DeleteWorkflow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because too many requests were made in a short period,
     *         exceeding the service rate limits. Amazon Managed Workflows for Apache Airflow Serverless implements
     *         throttling controls to ensure fair resource allocation across all customers in the multi-tenant
     *         environment. This helps maintain service stability and performance. If you encounter throttling,
     *         implement exponential backoff and retry logic in your applications, or consider distributing your API
     *         calls over a longer time period.</li>
     *         <li>ValidationException The specified request parameters are invalid, missing, or inconsistent with
     *         Amazon Managed Workflows for Apache Airflow Serverless service requirements. This can occur when workflow
     *         definitions contain unsupported operators, when required IAM permissions are missing, when S3 locations
     *         are inaccessible, or when network configurations are invalid. The service validates workflow definitions,
     *         execution roles, and resource configurations to ensure compatibility with the managed Airflow environment
     *         and security requirements.</li>
     *         <li>ResourceNotFoundException The specified resource was not found. You can only access or modify a
     *         resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An unexpected server-side error occurred during request processing.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MwaaServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MwaaServerlessAsyncClient.DeleteWorkflow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-serverless-2024-07-26/DeleteWorkflow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWorkflowResponse> deleteWorkflow(DeleteWorkflowRequest deleteWorkflowRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWorkflowRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkflowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkflow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWorkflowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWorkflowResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "OperationTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OperationTimeoutException").httpStatusCode(500)
                            .exceptionBuilderSupplier(OperationTimeoutException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteWorkflowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWorkflowRequest, DeleteWorkflowResponse>()
                            .withOperationName("DeleteWorkflow").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteWorkflowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteWorkflowRequest));
            CompletableFuture<DeleteWorkflowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves detailed information about a specific task instance within a workflow run. Task instances represent
     * individual tasks that are executed as part of a workflow in the Amazon Managed Workflows for Apache Airflow
     * Serverless environment. Each task instance runs in an isolated ECS container with dedicated resources and
     * security boundaries. The service tracks task execution state, retry attempts, and provides detailed timing and
     * error information for troubleshooting and monitoring purposes.
     * </p>
     *
     * @param getTaskInstanceRequest
     * @return A Java Future containing the result of the GetTaskInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because too many requests were made in a short period,
     *         exceeding the service rate limits. Amazon Managed Workflows for Apache Airflow Serverless implements
     *         throttling controls to ensure fair resource allocation across all customers in the multi-tenant
     *         environment. This helps maintain service stability and performance. If you encounter throttling,
     *         implement exponential backoff and retry logic in your applications, or consider distributing your API
     *         calls over a longer time period.</li>
     *         <li>ValidationException The specified request parameters are invalid, missing, or inconsistent with
     *         Amazon Managed Workflows for Apache Airflow Serverless service requirements. This can occur when workflow
     *         definitions contain unsupported operators, when required IAM permissions are missing, when S3 locations
     *         are inaccessible, or when network configurations are invalid. The service validates workflow definitions,
     *         execution roles, and resource configurations to ensure compatibility with the managed Airflow environment
     *         and security requirements.</li>
     *         <li>ResourceNotFoundException The specified resource was not found. You can only access or modify a
     *         resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An unexpected server-side error occurred during request processing.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MwaaServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MwaaServerlessAsyncClient.GetTaskInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-serverless-2024-07-26/GetTaskInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTaskInstanceResponse> getTaskInstance(GetTaskInstanceRequest getTaskInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTaskInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTaskInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTaskInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTaskInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTaskInstanceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "OperationTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OperationTimeoutException").httpStatusCode(500)
                            .exceptionBuilderSupplier(OperationTimeoutException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetTaskInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTaskInstanceRequest, GetTaskInstanceResponse>()
                            .withOperationName("GetTaskInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTaskInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTaskInstanceRequest));
            CompletableFuture<GetTaskInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves detailed information about a workflow, including its configuration, status, and metadata.
     * </p>
     *
     * @param getWorkflowRequest
     * @return A Java Future containing the result of the GetWorkflow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because too many requests were made in a short period,
     *         exceeding the service rate limits. Amazon Managed Workflows for Apache Airflow Serverless implements
     *         throttling controls to ensure fair resource allocation across all customers in the multi-tenant
     *         environment. This helps maintain service stability and performance. If you encounter throttling,
     *         implement exponential backoff and retry logic in your applications, or consider distributing your API
     *         calls over a longer time period.</li>
     *         <li>ValidationException The specified request parameters are invalid, missing, or inconsistent with
     *         Amazon Managed Workflows for Apache Airflow Serverless service requirements. This can occur when workflow
     *         definitions contain unsupported operators, when required IAM permissions are missing, when S3 locations
     *         are inaccessible, or when network configurations are invalid. The service validates workflow definitions,
     *         execution roles, and resource configurations to ensure compatibility with the managed Airflow environment
     *         and security requirements.</li>
     *         <li>ResourceNotFoundException The specified resource was not found. You can only access or modify a
     *         resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An unexpected server-side error occurred during request processing.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MwaaServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MwaaServerlessAsyncClient.GetWorkflow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-serverless-2024-07-26/GetWorkflow" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetWorkflowResponse> getWorkflow(GetWorkflowRequest getWorkflowRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getWorkflowRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWorkflowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorkflow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWorkflowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetWorkflowResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "OperationTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OperationTimeoutException").httpStatusCode(500)
                            .exceptionBuilderSupplier(OperationTimeoutException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetWorkflowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWorkflowRequest, GetWorkflowResponse>()
                            .withOperationName("GetWorkflow").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetWorkflowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getWorkflowRequest));
            CompletableFuture<GetWorkflowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves detailed information about a specific workflow run, including its status, execution details, and task
     * instances.
     * </p>
     *
     * @param getWorkflowRunRequest
     * @return A Java Future containing the result of the GetWorkflowRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because too many requests were made in a short period,
     *         exceeding the service rate limits. Amazon Managed Workflows for Apache Airflow Serverless implements
     *         throttling controls to ensure fair resource allocation across all customers in the multi-tenant
     *         environment. This helps maintain service stability and performance. If you encounter throttling,
     *         implement exponential backoff and retry logic in your applications, or consider distributing your API
     *         calls over a longer time period.</li>
     *         <li>ValidationException The specified request parameters are invalid, missing, or inconsistent with
     *         Amazon Managed Workflows for Apache Airflow Serverless service requirements. This can occur when workflow
     *         definitions contain unsupported operators, when required IAM permissions are missing, when S3 locations
     *         are inaccessible, or when network configurations are invalid. The service validates workflow definitions,
     *         execution roles, and resource configurations to ensure compatibility with the managed Airflow environment
     *         and security requirements.</li>
     *         <li>ResourceNotFoundException The specified resource was not found. You can only access or modify a
     *         resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An unexpected server-side error occurred during request processing.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MwaaServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MwaaServerlessAsyncClient.GetWorkflowRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-serverless-2024-07-26/GetWorkflowRun"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetWorkflowRunResponse> getWorkflowRun(GetWorkflowRunRequest getWorkflowRunRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getWorkflowRunRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWorkflowRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorkflowRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWorkflowRunResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetWorkflowRunResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "OperationTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OperationTimeoutException").httpStatusCode(500)
                            .exceptionBuilderSupplier(OperationTimeoutException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetWorkflowRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWorkflowRunRequest, GetWorkflowRunResponse>()
                            .withOperationName("GetWorkflowRun").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetWorkflowRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getWorkflowRunRequest));
            CompletableFuture<GetWorkflowRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all tags that are associated with a specified Amazon Managed Workflows for Apache Airflow Serverless
     * resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because too many requests were made in a short period,
     *         exceeding the service rate limits. Amazon Managed Workflows for Apache Airflow Serverless implements
     *         throttling controls to ensure fair resource allocation across all customers in the multi-tenant
     *         environment. This helps maintain service stability and performance. If you encounter throttling,
     *         implement exponential backoff and retry logic in your applications, or consider distributing your API
     *         calls over a longer time period.</li>
     *         <li>ValidationException The specified request parameters are invalid, missing, or inconsistent with
     *         Amazon Managed Workflows for Apache Airflow Serverless service requirements. This can occur when workflow
     *         definitions contain unsupported operators, when required IAM permissions are missing, when S3 locations
     *         are inaccessible, or when network configurations are invalid. The service validates workflow definitions,
     *         execution roles, and resource configurations to ensure compatibility with the managed Airflow environment
     *         and security requirements.</li>
     *         <li>ResourceNotFoundException The specified resource was not found. You can only access or modify a
     *         resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An unexpected server-side error occurred during request processing.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MwaaServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MwaaServerlessAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-serverless-2024-07-26/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "OperationTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OperationTimeoutException").httpStatusCode(500)
                            .exceptionBuilderSupplier(OperationTimeoutException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all task instances for a specific workflow run, with optional pagination support.
     * </p>
     *
     * @param listTaskInstancesRequest
     * @return A Java Future containing the result of the ListTaskInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because too many requests were made in a short period,
     *         exceeding the service rate limits. Amazon Managed Workflows for Apache Airflow Serverless implements
     *         throttling controls to ensure fair resource allocation across all customers in the multi-tenant
     *         environment. This helps maintain service stability and performance. If you encounter throttling,
     *         implement exponential backoff and retry logic in your applications, or consider distributing your API
     *         calls over a longer time period.</li>
     *         <li>ValidationException The specified request parameters are invalid, missing, or inconsistent with
     *         Amazon Managed Workflows for Apache Airflow Serverless service requirements. This can occur when workflow
     *         definitions contain unsupported operators, when required IAM permissions are missing, when S3 locations
     *         are inaccessible, or when network configurations are invalid. The service validates workflow definitions,
     *         execution roles, and resource configurations to ensure compatibility with the managed Airflow environment
     *         and security requirements.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An unexpected server-side error occurred during request processing.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MwaaServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MwaaServerlessAsyncClient.ListTaskInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-serverless-2024-07-26/ListTaskInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTaskInstancesResponse> listTaskInstances(ListTaskInstancesRequest listTaskInstancesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTaskInstancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTaskInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTaskInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTaskInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTaskInstancesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "OperationTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OperationTimeoutException").httpStatusCode(500)
                            .exceptionBuilderSupplier(OperationTimeoutException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTaskInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTaskInstancesRequest, ListTaskInstancesResponse>()
                            .withOperationName("ListTaskInstances").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTaskInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTaskInstancesRequest));
            CompletableFuture<ListTaskInstancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all runs for a specified workflow, with optional pagination and filtering support.
     * </p>
     *
     * @param listWorkflowRunsRequest
     * @return A Java Future containing the result of the ListWorkflowRuns operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because too many requests were made in a short period,
     *         exceeding the service rate limits. Amazon Managed Workflows for Apache Airflow Serverless implements
     *         throttling controls to ensure fair resource allocation across all customers in the multi-tenant
     *         environment. This helps maintain service stability and performance. If you encounter throttling,
     *         implement exponential backoff and retry logic in your applications, or consider distributing your API
     *         calls over a longer time period.</li>
     *         <li>ValidationException The specified request parameters are invalid, missing, or inconsistent with
     *         Amazon Managed Workflows for Apache Airflow Serverless service requirements. This can occur when workflow
     *         definitions contain unsupported operators, when required IAM permissions are missing, when S3 locations
     *         are inaccessible, or when network configurations are invalid. The service validates workflow definitions,
     *         execution roles, and resource configurations to ensure compatibility with the managed Airflow environment
     *         and security requirements.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An unexpected server-side error occurred during request processing.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MwaaServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MwaaServerlessAsyncClient.ListWorkflowRuns
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-serverless-2024-07-26/ListWorkflowRuns"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWorkflowRunsResponse> listWorkflowRuns(ListWorkflowRunsRequest listWorkflowRunsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWorkflowRunsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkflowRunsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkflowRuns");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWorkflowRunsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListWorkflowRunsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "OperationTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OperationTimeoutException").httpStatusCode(500)
                            .exceptionBuilderSupplier(OperationTimeoutException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListWorkflowRunsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWorkflowRunsRequest, ListWorkflowRunsResponse>()
                            .withOperationName("ListWorkflowRuns").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWorkflowRunsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWorkflowRunsRequest));
            CompletableFuture<ListWorkflowRunsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all versions of a specified workflow, with optional pagination support.
     * </p>
     *
     * @param listWorkflowVersionsRequest
     * @return A Java Future containing the result of the ListWorkflowVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because too many requests were made in a short period,
     *         exceeding the service rate limits. Amazon Managed Workflows for Apache Airflow Serverless implements
     *         throttling controls to ensure fair resource allocation across all customers in the multi-tenant
     *         environment. This helps maintain service stability and performance. If you encounter throttling,
     *         implement exponential backoff and retry logic in your applications, or consider distributing your API
     *         calls over a longer time period.</li>
     *         <li>ValidationException The specified request parameters are invalid, missing, or inconsistent with
     *         Amazon Managed Workflows for Apache Airflow Serverless service requirements. This can occur when workflow
     *         definitions contain unsupported operators, when required IAM permissions are missing, when S3 locations
     *         are inaccessible, or when network configurations are invalid. The service validates workflow definitions,
     *         execution roles, and resource configurations to ensure compatibility with the managed Airflow environment
     *         and security requirements.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An unexpected server-side error occurred during request processing.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MwaaServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MwaaServerlessAsyncClient.ListWorkflowVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-serverless-2024-07-26/ListWorkflowVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWorkflowVersionsResponse> listWorkflowVersions(
            ListWorkflowVersionsRequest listWorkflowVersionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWorkflowVersionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkflowVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkflowVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWorkflowVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListWorkflowVersionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "OperationTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OperationTimeoutException").httpStatusCode(500)
                            .exceptionBuilderSupplier(OperationTimeoutException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListWorkflowVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWorkflowVersionsRequest, ListWorkflowVersionsResponse>()
                            .withOperationName("ListWorkflowVersions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWorkflowVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWorkflowVersionsRequest));
            CompletableFuture<ListWorkflowVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all workflows in your account, with optional pagination support. This operation returns summary information
     * for workflows, showing only the most recently created version of each workflow. Amazon Managed Workflows for
     * Apache Airflow Serverless maintains workflow metadata in a highly available, distributed storage system that
     * enables efficient querying and filtering. The service implements proper access controls to ensure you can only
     * view workflows that you have permissions to access, supporting both individual and team-based workflow management
     * scenarios.
     * </p>
     *
     * @param listWorkflowsRequest
     * @return A Java Future containing the result of the ListWorkflows operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because too many requests were made in a short period,
     *         exceeding the service rate limits. Amazon Managed Workflows for Apache Airflow Serverless implements
     *         throttling controls to ensure fair resource allocation across all customers in the multi-tenant
     *         environment. This helps maintain service stability and performance. If you encounter throttling,
     *         implement exponential backoff and retry logic in your applications, or consider distributing your API
     *         calls over a longer time period.</li>
     *         <li>ValidationException The specified request parameters are invalid, missing, or inconsistent with
     *         Amazon Managed Workflows for Apache Airflow Serverless service requirements. This can occur when workflow
     *         definitions contain unsupported operators, when required IAM permissions are missing, when S3 locations
     *         are inaccessible, or when network configurations are invalid. The service validates workflow definitions,
     *         execution roles, and resource configurations to ensure compatibility with the managed Airflow environment
     *         and security requirements.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An unexpected server-side error occurred during request processing.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MwaaServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MwaaServerlessAsyncClient.ListWorkflows
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-serverless-2024-07-26/ListWorkflows" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWorkflowsResponse> listWorkflows(ListWorkflowsRequest listWorkflowsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWorkflowsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkflowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkflows");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWorkflowsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListWorkflowsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "OperationTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OperationTimeoutException").httpStatusCode(500)
                            .exceptionBuilderSupplier(OperationTimeoutException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListWorkflowsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWorkflowsRequest, ListWorkflowsResponse>()
                            .withOperationName("ListWorkflows").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWorkflowsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWorkflowsRequest));
            CompletableFuture<ListWorkflowsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a new execution of a workflow. This operation creates a workflow run that executes the tasks that are
     * defined in the workflow. Amazon Managed Workflows for Apache Airflow Serverless schedules the workflow execution
     * across its managed Airflow environment, automatically scaling ECS worker tasks based on the workload. The service
     * handles task isolation, dependency resolution, and provides comprehensive monitoring and logging throughout the
     * execution lifecycle.
     * </p>
     *
     * @param startWorkflowRunRequest
     * @return A Java Future containing the result of the StartWorkflowRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because too many requests were made in a short period,
     *         exceeding the service rate limits. Amazon Managed Workflows for Apache Airflow Serverless implements
     *         throttling controls to ensure fair resource allocation across all customers in the multi-tenant
     *         environment. This helps maintain service stability and performance. If you encounter throttling,
     *         implement exponential backoff and retry logic in your applications, or consider distributing your API
     *         calls over a longer time period.</li>
     *         <li>ValidationException The specified request parameters are invalid, missing, or inconsistent with
     *         Amazon Managed Workflows for Apache Airflow Serverless service requirements. This can occur when workflow
     *         definitions contain unsupported operators, when required IAM permissions are missing, when S3 locations
     *         are inaccessible, or when network configurations are invalid. The service validates workflow definitions,
     *         execution roles, and resource configurations to ensure compatibility with the managed Airflow environment
     *         and security requirements.</li>
     *         <li>ResourceNotFoundException The specified resource was not found. You can only access or modify a
     *         resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An unexpected server-side error occurred during request processing.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>ConflictException You cannot create a resource that already exists, or the resource is in a state
     *         that prevents the requested operation.</li>
     *         <li>ServiceQuotaExceededException The request exceeds the service quota for Amazon Managed Workflows for
     *         Apache Airflow Serverless resources. This can occur when you attempt to create more workflows than
     *         allowed, exceed concurrent workflow run limits, or surpass task execution limits. Amazon Managed
     *         Workflows for Apache Airflow Serverless implements admission control using DynamoDB-based counters to
     *         manage resource utilization across the multi-tenant environment. Contact Amazon Web Services Support to
     *         request quota increases if you need higher limits for your use case.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MwaaServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MwaaServerlessAsyncClient.StartWorkflowRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-serverless-2024-07-26/StartWorkflowRun"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartWorkflowRunResponse> startWorkflowRun(StartWorkflowRunRequest startWorkflowRunRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startWorkflowRunRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startWorkflowRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartWorkflowRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartWorkflowRunResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartWorkflowRunResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "OperationTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OperationTimeoutException").httpStatusCode(500)
                            .exceptionBuilderSupplier(OperationTimeoutException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartWorkflowRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartWorkflowRunRequest, StartWorkflowRunResponse>()
                            .withOperationName("StartWorkflowRun").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartWorkflowRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startWorkflowRunRequest));
            CompletableFuture<StartWorkflowRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops a running workflow execution. This operation terminates all running tasks and prevents new tasks from
     * starting. Amazon Managed Workflows for Apache Airflow Serverless gracefully shuts down the workflow execution by
     * stopping task scheduling and terminating active ECS worker containers. The operation transitions the workflow run
     * to a <code>STOPPING</code> state and then to <code>STOPPED</code> once all cleanup is complete. In-flight tasks
     * may complete or be terminated depending on their current execution state.
     * </p>
     *
     * @param stopWorkflowRunRequest
     * @return A Java Future containing the result of the StopWorkflowRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because too many requests were made in a short period,
     *         exceeding the service rate limits. Amazon Managed Workflows for Apache Airflow Serverless implements
     *         throttling controls to ensure fair resource allocation across all customers in the multi-tenant
     *         environment. This helps maintain service stability and performance. If you encounter throttling,
     *         implement exponential backoff and retry logic in your applications, or consider distributing your API
     *         calls over a longer time period.</li>
     *         <li>ValidationException The specified request parameters are invalid, missing, or inconsistent with
     *         Amazon Managed Workflows for Apache Airflow Serverless service requirements. This can occur when workflow
     *         definitions contain unsupported operators, when required IAM permissions are missing, when S3 locations
     *         are inaccessible, or when network configurations are invalid. The service validates workflow definitions,
     *         execution roles, and resource configurations to ensure compatibility with the managed Airflow environment
     *         and security requirements.</li>
     *         <li>ResourceNotFoundException The specified resource was not found. You can only access or modify a
     *         resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An unexpected server-side error occurred during request processing.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MwaaServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MwaaServerlessAsyncClient.StopWorkflowRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-serverless-2024-07-26/StopWorkflowRun"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopWorkflowRunResponse> stopWorkflowRun(StopWorkflowRunRequest stopWorkflowRunRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopWorkflowRunRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopWorkflowRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopWorkflowRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopWorkflowRunResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopWorkflowRunResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "OperationTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OperationTimeoutException").httpStatusCode(500)
                            .exceptionBuilderSupplier(OperationTimeoutException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StopWorkflowRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopWorkflowRunRequest, StopWorkflowRunResponse>()
                            .withOperationName("StopWorkflowRun").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopWorkflowRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopWorkflowRunRequest));
            CompletableFuture<StopWorkflowRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds tags to an Amazon Managed Workflows for Apache Airflow Serverless resource. Tags are key-value pairs that
     * help you organize and categorize your resources.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because too many requests were made in a short period,
     *         exceeding the service rate limits. Amazon Managed Workflows for Apache Airflow Serverless implements
     *         throttling controls to ensure fair resource allocation across all customers in the multi-tenant
     *         environment. This helps maintain service stability and performance. If you encounter throttling,
     *         implement exponential backoff and retry logic in your applications, or consider distributing your API
     *         calls over a longer time period.</li>
     *         <li>ValidationException The specified request parameters are invalid, missing, or inconsistent with
     *         Amazon Managed Workflows for Apache Airflow Serverless service requirements. This can occur when workflow
     *         definitions contain unsupported operators, when required IAM permissions are missing, when S3 locations
     *         are inaccessible, or when network configurations are invalid. The service validates workflow definitions,
     *         execution roles, and resource configurations to ensure compatibility with the managed Airflow environment
     *         and security requirements.</li>
     *         <li>ResourceNotFoundException The specified resource was not found. You can only access or modify a
     *         resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An unexpected server-side error occurred during request processing.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MwaaServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MwaaServerlessAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-serverless-2024-07-26/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "OperationTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OperationTimeoutException").httpStatusCode(500)
                            .exceptionBuilderSupplier(OperationTimeoutException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from an Amazon Managed Workflows for Apache Airflow Serverless resource. This operation removes the
     * specified tags from the resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because too many requests were made in a short period,
     *         exceeding the service rate limits. Amazon Managed Workflows for Apache Airflow Serverless implements
     *         throttling controls to ensure fair resource allocation across all customers in the multi-tenant
     *         environment. This helps maintain service stability and performance. If you encounter throttling,
     *         implement exponential backoff and retry logic in your applications, or consider distributing your API
     *         calls over a longer time period.</li>
     *         <li>ValidationException The specified request parameters are invalid, missing, or inconsistent with
     *         Amazon Managed Workflows for Apache Airflow Serverless service requirements. This can occur when workflow
     *         definitions contain unsupported operators, when required IAM permissions are missing, when S3 locations
     *         are inaccessible, or when network configurations are invalid. The service validates workflow definitions,
     *         execution roles, and resource configurations to ensure compatibility with the managed Airflow environment
     *         and security requirements.</li>
     *         <li>ResourceNotFoundException The specified resource was not found. You can only access or modify a
     *         resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An unexpected server-side error occurred during request processing.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MwaaServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MwaaServerlessAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-serverless-2024-07-26/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "OperationTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OperationTimeoutException").httpStatusCode(500)
                            .exceptionBuilderSupplier(OperationTimeoutException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing workflow with new configuration settings. This operation allows you to modify the workflow
     * definition, role, and other settings. When you update a workflow, Amazon Managed Workflows for Apache Airflow
     * Serverless automatically creates a new version with the updated configuration and disables scheduling on all
     * previous versions to ensure only one version is actively scheduled at a time. The update operation maintains
     * workflow history while providing a clean transition to the new configuration.
     * </p>
     *
     * @param updateWorkflowRequest
     * @return A Java Future containing the result of the UpdateWorkflow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because too many requests were made in a short period,
     *         exceeding the service rate limits. Amazon Managed Workflows for Apache Airflow Serverless implements
     *         throttling controls to ensure fair resource allocation across all customers in the multi-tenant
     *         environment. This helps maintain service stability and performance. If you encounter throttling,
     *         implement exponential backoff and retry logic in your applications, or consider distributing your API
     *         calls over a longer time period.</li>
     *         <li>ValidationException The specified request parameters are invalid, missing, or inconsistent with
     *         Amazon Managed Workflows for Apache Airflow Serverless service requirements. This can occur when workflow
     *         definitions contain unsupported operators, when required IAM permissions are missing, when S3 locations
     *         are inaccessible, or when network configurations are invalid. The service validates workflow definitions,
     *         execution roles, and resource configurations to ensure compatibility with the managed Airflow environment
     *         and security requirements.</li>
     *         <li>ResourceNotFoundException The specified resource was not found. You can only access or modify a
     *         resource that already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>InternalServerException An unexpected server-side error occurred during request processing.</li>
     *         <li>OperationTimeoutException The operation timed out.</li>
     *         <li>ConflictException You cannot create a resource that already exists, or the resource is in a state
     *         that prevents the requested operation.</li>
     *         <li>ServiceQuotaExceededException The request exceeds the service quota for Amazon Managed Workflows for
     *         Apache Airflow Serverless resources. This can occur when you attempt to create more workflows than
     *         allowed, exceed concurrent workflow run limits, or surpass task execution limits. Amazon Managed
     *         Workflows for Apache Airflow Serverless implements admission control using DynamoDB-based counters to
     *         manage resource utilization across the multi-tenant environment. Contact Amazon Web Services Support to
     *         request quota increases if you need higher limits for your use case.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MwaaServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MwaaServerlessAsyncClient.UpdateWorkflow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mwaa-serverless-2024-07-26/UpdateWorkflow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateWorkflowResponse> updateWorkflow(UpdateWorkflowRequest updateWorkflowRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateWorkflowRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWorkflowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWorkflow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateWorkflowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateWorkflowResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "OperationTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OperationTimeoutException").httpStatusCode(500)
                            .exceptionBuilderSupplier(OperationTimeoutException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateWorkflowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateWorkflowRequest, UpdateWorkflowResponse>()
                            .withOperationName("UpdateWorkflow").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateWorkflowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateWorkflowRequest));
            CompletableFuture<UpdateWorkflowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final MwaaServerlessServiceClientConfiguration serviceClientConfiguration() {
        return new MwaaServerlessServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(MwaaServerlessException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON).protocolVersion("1.0");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        MwaaServerlessServiceClientConfigurationBuilder serviceConfigBuilder = new MwaaServerlessServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
