/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.neptune.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This data type is used as a response element in the action <a>DescribeDBEngineVersions</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DBEngineVersion implements SdkPojo, Serializable, ToCopyableBuilder<DBEngineVersion.Builder, DBEngineVersion> {
    private static final SdkField<String> ENGINE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Engine")
            .getter(getter(DBEngineVersion::engine)).setter(setter(Builder::engine))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Engine").build()).build();

    private static final SdkField<String> ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EngineVersion").getter(getter(DBEngineVersion::engineVersion)).setter(setter(Builder::engineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final SdkField<String> DB_PARAMETER_GROUP_FAMILY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBParameterGroupFamily").getter(getter(DBEngineVersion::dbParameterGroupFamily))
            .setter(setter(Builder::dbParameterGroupFamily))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBParameterGroupFamily").build())
            .build();

    private static final SdkField<String> DB_ENGINE_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBEngineDescription").getter(getter(DBEngineVersion::dbEngineDescription))
            .setter(setter(Builder::dbEngineDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBEngineDescription").build())
            .build();

    private static final SdkField<String> DB_ENGINE_VERSION_DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DBEngineVersionDescription")
            .getter(getter(DBEngineVersion::dbEngineVersionDescription))
            .setter(setter(Builder::dbEngineVersionDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBEngineVersionDescription").build())
            .build();

    private static final SdkField<CharacterSet> DEFAULT_CHARACTER_SET_FIELD = SdkField
            .<CharacterSet> builder(MarshallingType.SDK_POJO).memberName("DefaultCharacterSet")
            .getter(getter(DBEngineVersion::defaultCharacterSet)).setter(setter(Builder::defaultCharacterSet))
            .constructor(CharacterSet::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultCharacterSet").build())
            .build();

    private static final SdkField<List<CharacterSet>> SUPPORTED_CHARACTER_SETS_FIELD = SdkField
            .<List<CharacterSet>> builder(MarshallingType.LIST)
            .memberName("SupportedCharacterSets")
            .getter(getter(DBEngineVersion::supportedCharacterSets))
            .setter(setter(Builder::supportedCharacterSets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportedCharacterSets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("CharacterSet")
                            .memberFieldInfo(
                                    SdkField.<CharacterSet> builder(MarshallingType.SDK_POJO)
                                            .constructor(CharacterSet::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("CharacterSet").build()).build()).build()).build();

    private static final SdkField<List<UpgradeTarget>> VALID_UPGRADE_TARGET_FIELD = SdkField
            .<List<UpgradeTarget>> builder(MarshallingType.LIST)
            .memberName("ValidUpgradeTarget")
            .getter(getter(DBEngineVersion::validUpgradeTarget))
            .setter(setter(Builder::validUpgradeTarget))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValidUpgradeTarget").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("UpgradeTarget")
                            .memberFieldInfo(
                                    SdkField.<UpgradeTarget> builder(MarshallingType.SDK_POJO)
                                            .constructor(UpgradeTarget::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("UpgradeTarget").build()).build()).build()).build();

    private static final SdkField<List<Timezone>> SUPPORTED_TIMEZONES_FIELD = SdkField
            .<List<Timezone>> builder(MarshallingType.LIST)
            .memberName("SupportedTimezones")
            .getter(getter(DBEngineVersion::supportedTimezones))
            .setter(setter(Builder::supportedTimezones))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportedTimezones").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Timezone")
                            .memberFieldInfo(
                                    SdkField.<Timezone> builder(MarshallingType.SDK_POJO)
                                            .constructor(Timezone::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Timezone").build()).build()).build()).build();

    private static final SdkField<List<String>> EXPORTABLE_LOG_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExportableLogTypes")
            .getter(getter(DBEngineVersion::exportableLogTypes))
            .setter(setter(Builder::exportableLogTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExportableLogTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> SUPPORTS_LOG_EXPORTS_TO_CLOUDWATCH_LOGS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SupportsLogExportsToCloudwatchLogs")
            .getter(getter(DBEngineVersion::supportsLogExportsToCloudwatchLogs))
            .setter(setter(Builder::supportsLogExportsToCloudwatchLogs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportsLogExportsToCloudwatchLogs")
                    .build()).build();

    private static final SdkField<Boolean> SUPPORTS_READ_REPLICA_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SupportsReadReplica").getter(getter(DBEngineVersion::supportsReadReplica))
            .setter(setter(Builder::supportsReadReplica))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportsReadReplica").build())
            .build();

    private static final SdkField<Boolean> SUPPORTS_GLOBAL_DATABASES_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SupportsGlobalDatabases").getter(getter(DBEngineVersion::supportsGlobalDatabases))
            .setter(setter(Builder::supportsGlobalDatabases))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportsGlobalDatabases").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENGINE_FIELD,
            ENGINE_VERSION_FIELD, DB_PARAMETER_GROUP_FAMILY_FIELD, DB_ENGINE_DESCRIPTION_FIELD,
            DB_ENGINE_VERSION_DESCRIPTION_FIELD, DEFAULT_CHARACTER_SET_FIELD, SUPPORTED_CHARACTER_SETS_FIELD,
            VALID_UPGRADE_TARGET_FIELD, SUPPORTED_TIMEZONES_FIELD, EXPORTABLE_LOG_TYPES_FIELD,
            SUPPORTS_LOG_EXPORTS_TO_CLOUDWATCH_LOGS_FIELD, SUPPORTS_READ_REPLICA_FIELD, SUPPORTS_GLOBAL_DATABASES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String engine;

    private final String engineVersion;

    private final String dbParameterGroupFamily;

    private final String dbEngineDescription;

    private final String dbEngineVersionDescription;

    private final CharacterSet defaultCharacterSet;

    private final List<CharacterSet> supportedCharacterSets;

    private final List<UpgradeTarget> validUpgradeTarget;

    private final List<Timezone> supportedTimezones;

    private final List<String> exportableLogTypes;

    private final Boolean supportsLogExportsToCloudwatchLogs;

    private final Boolean supportsReadReplica;

    private final Boolean supportsGlobalDatabases;

    private DBEngineVersion(BuilderImpl builder) {
        this.engine = builder.engine;
        this.engineVersion = builder.engineVersion;
        this.dbParameterGroupFamily = builder.dbParameterGroupFamily;
        this.dbEngineDescription = builder.dbEngineDescription;
        this.dbEngineVersionDescription = builder.dbEngineVersionDescription;
        this.defaultCharacterSet = builder.defaultCharacterSet;
        this.supportedCharacterSets = builder.supportedCharacterSets;
        this.validUpgradeTarget = builder.validUpgradeTarget;
        this.supportedTimezones = builder.supportedTimezones;
        this.exportableLogTypes = builder.exportableLogTypes;
        this.supportsLogExportsToCloudwatchLogs = builder.supportsLogExportsToCloudwatchLogs;
        this.supportsReadReplica = builder.supportsReadReplica;
        this.supportsGlobalDatabases = builder.supportsGlobalDatabases;
    }

    /**
     * <p>
     * The name of the database engine.
     * </p>
     * 
     * @return The name of the database engine.
     */
    public final String engine() {
        return engine;
    }

    /**
     * <p>
     * The version number of the database engine.
     * </p>
     * 
     * @return The version number of the database engine.
     */
    public final String engineVersion() {
        return engineVersion;
    }

    /**
     * <p>
     * The name of the DB parameter group family for the database engine.
     * </p>
     * 
     * @return The name of the DB parameter group family for the database engine.
     */
    public final String dbParameterGroupFamily() {
        return dbParameterGroupFamily;
    }

    /**
     * <p>
     * The description of the database engine.
     * </p>
     * 
     * @return The description of the database engine.
     */
    public final String dbEngineDescription() {
        return dbEngineDescription;
    }

    /**
     * <p>
     * The description of the database engine version.
     * </p>
     * 
     * @return The description of the database engine version.
     */
    public final String dbEngineVersionDescription() {
        return dbEngineVersionDescription;
    }

    /**
     * <p>
     * <i>(Not supported by Neptune)</i>
     * </p>
     * 
     * @return <i>(Not supported by Neptune)</i>
     */
    public final CharacterSet defaultCharacterSet() {
        return defaultCharacterSet;
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportedCharacterSets property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSupportedCharacterSets() {
        return supportedCharacterSets != null && !(supportedCharacterSets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * <i>(Not supported by Neptune)</i>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedCharacterSets} method.
     * </p>
     * 
     * @return <i>(Not supported by Neptune)</i>
     */
    public final List<CharacterSet> supportedCharacterSets() {
        return supportedCharacterSets;
    }

    /**
     * For responses, this returns true if the service returned a value for the ValidUpgradeTarget property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasValidUpgradeTarget() {
        return validUpgradeTarget != null && !(validUpgradeTarget instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of engine versions that this database engine version can be upgraded to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValidUpgradeTarget} method.
     * </p>
     * 
     * @return A list of engine versions that this database engine version can be upgraded to.
     */
    public final List<UpgradeTarget> validUpgradeTarget() {
        return validUpgradeTarget;
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportedTimezones property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSupportedTimezones() {
        return supportedTimezones != null && !(supportedTimezones instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the time zones supported by this engine for the <code>Timezone</code> parameter of the
     * <code>CreateDBInstance</code> action.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedTimezones} method.
     * </p>
     * 
     * @return A list of the time zones supported by this engine for the <code>Timezone</code> parameter of the
     *         <code>CreateDBInstance</code> action.
     */
    public final List<Timezone> supportedTimezones() {
        return supportedTimezones;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExportableLogTypes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExportableLogTypes() {
        return exportableLogTypes != null && !(exportableLogTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The types of logs that the database engine has available for export to CloudWatch Logs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExportableLogTypes} method.
     * </p>
     * 
     * @return The types of logs that the database engine has available for export to CloudWatch Logs.
     */
    public final List<String> exportableLogTypes() {
        return exportableLogTypes;
    }

    /**
     * <p>
     * A value that indicates whether the engine version supports exporting the log types specified by
     * ExportableLogTypes to CloudWatch Logs.
     * </p>
     * 
     * @return A value that indicates whether the engine version supports exporting the log types specified by
     *         ExportableLogTypes to CloudWatch Logs.
     */
    public final Boolean supportsLogExportsToCloudwatchLogs() {
        return supportsLogExportsToCloudwatchLogs;
    }

    /**
     * <p>
     * Indicates whether the database engine version supports read replicas.
     * </p>
     * 
     * @return Indicates whether the database engine version supports read replicas.
     */
    public final Boolean supportsReadReplica() {
        return supportsReadReplica;
    }

    /**
     * <p>
     * A value that indicates whether you can use Aurora global databases with a specific DB engine version.
     * </p>
     * 
     * @return A value that indicates whether you can use Aurora global databases with a specific DB engine version.
     */
    public final Boolean supportsGlobalDatabases() {
        return supportsGlobalDatabases;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(engine());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(dbParameterGroupFamily());
        hashCode = 31 * hashCode + Objects.hashCode(dbEngineDescription());
        hashCode = 31 * hashCode + Objects.hashCode(dbEngineVersionDescription());
        hashCode = 31 * hashCode + Objects.hashCode(defaultCharacterSet());
        hashCode = 31 * hashCode + Objects.hashCode(hasSupportedCharacterSets() ? supportedCharacterSets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasValidUpgradeTarget() ? validUpgradeTarget() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSupportedTimezones() ? supportedTimezones() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasExportableLogTypes() ? exportableLogTypes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(supportsLogExportsToCloudwatchLogs());
        hashCode = 31 * hashCode + Objects.hashCode(supportsReadReplica());
        hashCode = 31 * hashCode + Objects.hashCode(supportsGlobalDatabases());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DBEngineVersion)) {
            return false;
        }
        DBEngineVersion other = (DBEngineVersion) obj;
        return Objects.equals(engine(), other.engine()) && Objects.equals(engineVersion(), other.engineVersion())
                && Objects.equals(dbParameterGroupFamily(), other.dbParameterGroupFamily())
                && Objects.equals(dbEngineDescription(), other.dbEngineDescription())
                && Objects.equals(dbEngineVersionDescription(), other.dbEngineVersionDescription())
                && Objects.equals(defaultCharacterSet(), other.defaultCharacterSet())
                && hasSupportedCharacterSets() == other.hasSupportedCharacterSets()
                && Objects.equals(supportedCharacterSets(), other.supportedCharacterSets())
                && hasValidUpgradeTarget() == other.hasValidUpgradeTarget()
                && Objects.equals(validUpgradeTarget(), other.validUpgradeTarget())
                && hasSupportedTimezones() == other.hasSupportedTimezones()
                && Objects.equals(supportedTimezones(), other.supportedTimezones())
                && hasExportableLogTypes() == other.hasExportableLogTypes()
                && Objects.equals(exportableLogTypes(), other.exportableLogTypes())
                && Objects.equals(supportsLogExportsToCloudwatchLogs(), other.supportsLogExportsToCloudwatchLogs())
                && Objects.equals(supportsReadReplica(), other.supportsReadReplica())
                && Objects.equals(supportsGlobalDatabases(), other.supportsGlobalDatabases());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DBEngineVersion").add("Engine", engine()).add("EngineVersion", engineVersion())
                .add("DBParameterGroupFamily", dbParameterGroupFamily()).add("DBEngineDescription", dbEngineDescription())
                .add("DBEngineVersionDescription", dbEngineVersionDescription())
                .add("DefaultCharacterSet", defaultCharacterSet())
                .add("SupportedCharacterSets", hasSupportedCharacterSets() ? supportedCharacterSets() : null)
                .add("ValidUpgradeTarget", hasValidUpgradeTarget() ? validUpgradeTarget() : null)
                .add("SupportedTimezones", hasSupportedTimezones() ? supportedTimezones() : null)
                .add("ExportableLogTypes", hasExportableLogTypes() ? exportableLogTypes() : null)
                .add("SupportsLogExportsToCloudwatchLogs", supportsLogExportsToCloudwatchLogs())
                .add("SupportsReadReplica", supportsReadReplica()).add("SupportsGlobalDatabases", supportsGlobalDatabases())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Engine":
            return Optional.ofNullable(clazz.cast(engine()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "DBParameterGroupFamily":
            return Optional.ofNullable(clazz.cast(dbParameterGroupFamily()));
        case "DBEngineDescription":
            return Optional.ofNullable(clazz.cast(dbEngineDescription()));
        case "DBEngineVersionDescription":
            return Optional.ofNullable(clazz.cast(dbEngineVersionDescription()));
        case "DefaultCharacterSet":
            return Optional.ofNullable(clazz.cast(defaultCharacterSet()));
        case "SupportedCharacterSets":
            return Optional.ofNullable(clazz.cast(supportedCharacterSets()));
        case "ValidUpgradeTarget":
            return Optional.ofNullable(clazz.cast(validUpgradeTarget()));
        case "SupportedTimezones":
            return Optional.ofNullable(clazz.cast(supportedTimezones()));
        case "ExportableLogTypes":
            return Optional.ofNullable(clazz.cast(exportableLogTypes()));
        case "SupportsLogExportsToCloudwatchLogs":
            return Optional.ofNullable(clazz.cast(supportsLogExportsToCloudwatchLogs()));
        case "SupportsReadReplica":
            return Optional.ofNullable(clazz.cast(supportsReadReplica()));
        case "SupportsGlobalDatabases":
            return Optional.ofNullable(clazz.cast(supportsGlobalDatabases()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DBEngineVersion, T> g) {
        return obj -> g.apply((DBEngineVersion) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DBEngineVersion> {
        /**
         * <p>
         * The name of the database engine.
         * </p>
         * 
         * @param engine
         *        The name of the database engine.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engine(String engine);

        /**
         * <p>
         * The version number of the database engine.
         * </p>
         * 
         * @param engineVersion
         *        The version number of the database engine.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(String engineVersion);

        /**
         * <p>
         * The name of the DB parameter group family for the database engine.
         * </p>
         * 
         * @param dbParameterGroupFamily
         *        The name of the DB parameter group family for the database engine.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbParameterGroupFamily(String dbParameterGroupFamily);

        /**
         * <p>
         * The description of the database engine.
         * </p>
         * 
         * @param dbEngineDescription
         *        The description of the database engine.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbEngineDescription(String dbEngineDescription);

        /**
         * <p>
         * The description of the database engine version.
         * </p>
         * 
         * @param dbEngineVersionDescription
         *        The description of the database engine version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbEngineVersionDescription(String dbEngineVersionDescription);

        /**
         * <p>
         * <i>(Not supported by Neptune)</i>
         * </p>
         * 
         * @param defaultCharacterSet
         *        <i>(Not supported by Neptune)</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultCharacterSet(CharacterSet defaultCharacterSet);

        /**
         * <p>
         * <i>(Not supported by Neptune)</i>
         * </p>
         * This is a convenience method that creates an instance of the {@link CharacterSet.Builder} avoiding the need
         * to create one manually via {@link CharacterSet#builder()}.
         *
         * When the {@link Consumer} completes, {@link CharacterSet.Builder#build()} is called immediately and its
         * result is passed to {@link #defaultCharacterSet(CharacterSet)}.
         * 
         * @param defaultCharacterSet
         *        a consumer that will call methods on {@link CharacterSet.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #defaultCharacterSet(CharacterSet)
         */
        default Builder defaultCharacterSet(Consumer<CharacterSet.Builder> defaultCharacterSet) {
            return defaultCharacterSet(CharacterSet.builder().applyMutation(defaultCharacterSet).build());
        }

        /**
         * <p>
         * <i>(Not supported by Neptune)</i>
         * </p>
         * 
         * @param supportedCharacterSets
         *        <i>(Not supported by Neptune)</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedCharacterSets(Collection<CharacterSet> supportedCharacterSets);

        /**
         * <p>
         * <i>(Not supported by Neptune)</i>
         * </p>
         * 
         * @param supportedCharacterSets
         *        <i>(Not supported by Neptune)</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedCharacterSets(CharacterSet... supportedCharacterSets);

        /**
         * <p>
         * <i>(Not supported by Neptune)</i>
         * </p>
         * This is a convenience method that creates an instance of the {@link List<CharacterSet>.Builder} avoiding the
         * need to create one manually via {@link List<CharacterSet>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<CharacterSet>.Builder#build()} is called immediately and its
         * result is passed to {@link #supportedCharacterSets(List<CharacterSet>)}.
         * 
         * @param supportedCharacterSets
         *        a consumer that will call methods on {@link List<CharacterSet>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #supportedCharacterSets(List<CharacterSet>)
         */
        Builder supportedCharacterSets(Consumer<CharacterSet.Builder>... supportedCharacterSets);

        /**
         * <p>
         * A list of engine versions that this database engine version can be upgraded to.
         * </p>
         * 
         * @param validUpgradeTarget
         *        A list of engine versions that this database engine version can be upgraded to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validUpgradeTarget(Collection<UpgradeTarget> validUpgradeTarget);

        /**
         * <p>
         * A list of engine versions that this database engine version can be upgraded to.
         * </p>
         * 
         * @param validUpgradeTarget
         *        A list of engine versions that this database engine version can be upgraded to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validUpgradeTarget(UpgradeTarget... validUpgradeTarget);

        /**
         * <p>
         * A list of engine versions that this database engine version can be upgraded to.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<UpgradeTarget>.Builder} avoiding the
         * need to create one manually via {@link List<UpgradeTarget>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<UpgradeTarget>.Builder#build()} is called immediately and
         * its result is passed to {@link #validUpgradeTarget(List<UpgradeTarget>)}.
         * 
         * @param validUpgradeTarget
         *        a consumer that will call methods on {@link List<UpgradeTarget>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #validUpgradeTarget(List<UpgradeTarget>)
         */
        Builder validUpgradeTarget(Consumer<UpgradeTarget.Builder>... validUpgradeTarget);

        /**
         * <p>
         * A list of the time zones supported by this engine for the <code>Timezone</code> parameter of the
         * <code>CreateDBInstance</code> action.
         * </p>
         * 
         * @param supportedTimezones
         *        A list of the time zones supported by this engine for the <code>Timezone</code> parameter of the
         *        <code>CreateDBInstance</code> action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedTimezones(Collection<Timezone> supportedTimezones);

        /**
         * <p>
         * A list of the time zones supported by this engine for the <code>Timezone</code> parameter of the
         * <code>CreateDBInstance</code> action.
         * </p>
         * 
         * @param supportedTimezones
         *        A list of the time zones supported by this engine for the <code>Timezone</code> parameter of the
         *        <code>CreateDBInstance</code> action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedTimezones(Timezone... supportedTimezones);

        /**
         * <p>
         * A list of the time zones supported by this engine for the <code>Timezone</code> parameter of the
         * <code>CreateDBInstance</code> action.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Timezone>.Builder} avoiding the need
         * to create one manually via {@link List<Timezone>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Timezone>.Builder#build()} is called immediately and its
         * result is passed to {@link #supportedTimezones(List<Timezone>)}.
         * 
         * @param supportedTimezones
         *        a consumer that will call methods on {@link List<Timezone>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #supportedTimezones(List<Timezone>)
         */
        Builder supportedTimezones(Consumer<Timezone.Builder>... supportedTimezones);

        /**
         * <p>
         * The types of logs that the database engine has available for export to CloudWatch Logs.
         * </p>
         * 
         * @param exportableLogTypes
         *        The types of logs that the database engine has available for export to CloudWatch Logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportableLogTypes(Collection<String> exportableLogTypes);

        /**
         * <p>
         * The types of logs that the database engine has available for export to CloudWatch Logs.
         * </p>
         * 
         * @param exportableLogTypes
         *        The types of logs that the database engine has available for export to CloudWatch Logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportableLogTypes(String... exportableLogTypes);

        /**
         * <p>
         * A value that indicates whether the engine version supports exporting the log types specified by
         * ExportableLogTypes to CloudWatch Logs.
         * </p>
         * 
         * @param supportsLogExportsToCloudwatchLogs
         *        A value that indicates whether the engine version supports exporting the log types specified by
         *        ExportableLogTypes to CloudWatch Logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportsLogExportsToCloudwatchLogs(Boolean supportsLogExportsToCloudwatchLogs);

        /**
         * <p>
         * Indicates whether the database engine version supports read replicas.
         * </p>
         * 
         * @param supportsReadReplica
         *        Indicates whether the database engine version supports read replicas.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportsReadReplica(Boolean supportsReadReplica);

        /**
         * <p>
         * A value that indicates whether you can use Aurora global databases with a specific DB engine version.
         * </p>
         * 
         * @param supportsGlobalDatabases
         *        A value that indicates whether you can use Aurora global databases with a specific DB engine version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportsGlobalDatabases(Boolean supportsGlobalDatabases);
    }

    static final class BuilderImpl implements Builder {
        private String engine;

        private String engineVersion;

        private String dbParameterGroupFamily;

        private String dbEngineDescription;

        private String dbEngineVersionDescription;

        private CharacterSet defaultCharacterSet;

        private List<CharacterSet> supportedCharacterSets = DefaultSdkAutoConstructList.getInstance();

        private List<UpgradeTarget> validUpgradeTarget = DefaultSdkAutoConstructList.getInstance();

        private List<Timezone> supportedTimezones = DefaultSdkAutoConstructList.getInstance();

        private List<String> exportableLogTypes = DefaultSdkAutoConstructList.getInstance();

        private Boolean supportsLogExportsToCloudwatchLogs;

        private Boolean supportsReadReplica;

        private Boolean supportsGlobalDatabases;

        private BuilderImpl() {
        }

        private BuilderImpl(DBEngineVersion model) {
            engine(model.engine);
            engineVersion(model.engineVersion);
            dbParameterGroupFamily(model.dbParameterGroupFamily);
            dbEngineDescription(model.dbEngineDescription);
            dbEngineVersionDescription(model.dbEngineVersionDescription);
            defaultCharacterSet(model.defaultCharacterSet);
            supportedCharacterSets(model.supportedCharacterSets);
            validUpgradeTarget(model.validUpgradeTarget);
            supportedTimezones(model.supportedTimezones);
            exportableLogTypes(model.exportableLogTypes);
            supportsLogExportsToCloudwatchLogs(model.supportsLogExportsToCloudwatchLogs);
            supportsReadReplica(model.supportsReadReplica);
            supportsGlobalDatabases(model.supportsGlobalDatabases);
        }

        public final String getEngine() {
            return engine;
        }

        public final void setEngine(String engine) {
            this.engine = engine;
        }

        @Override
        public final Builder engine(String engine) {
            this.engine = engine;
            return this;
        }

        public final String getEngineVersion() {
            return engineVersion;
        }

        public final void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        @Override
        public final Builder engineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final String getDbParameterGroupFamily() {
            return dbParameterGroupFamily;
        }

        public final void setDbParameterGroupFamily(String dbParameterGroupFamily) {
            this.dbParameterGroupFamily = dbParameterGroupFamily;
        }

        @Override
        public final Builder dbParameterGroupFamily(String dbParameterGroupFamily) {
            this.dbParameterGroupFamily = dbParameterGroupFamily;
            return this;
        }

        public final String getDbEngineDescription() {
            return dbEngineDescription;
        }

        public final void setDbEngineDescription(String dbEngineDescription) {
            this.dbEngineDescription = dbEngineDescription;
        }

        @Override
        public final Builder dbEngineDescription(String dbEngineDescription) {
            this.dbEngineDescription = dbEngineDescription;
            return this;
        }

        public final String getDbEngineVersionDescription() {
            return dbEngineVersionDescription;
        }

        public final void setDbEngineVersionDescription(String dbEngineVersionDescription) {
            this.dbEngineVersionDescription = dbEngineVersionDescription;
        }

        @Override
        public final Builder dbEngineVersionDescription(String dbEngineVersionDescription) {
            this.dbEngineVersionDescription = dbEngineVersionDescription;
            return this;
        }

        public final CharacterSet.Builder getDefaultCharacterSet() {
            return defaultCharacterSet != null ? defaultCharacterSet.toBuilder() : null;
        }

        public final void setDefaultCharacterSet(CharacterSet.BuilderImpl defaultCharacterSet) {
            this.defaultCharacterSet = defaultCharacterSet != null ? defaultCharacterSet.build() : null;
        }

        @Override
        public final Builder defaultCharacterSet(CharacterSet defaultCharacterSet) {
            this.defaultCharacterSet = defaultCharacterSet;
            return this;
        }

        public final List<CharacterSet.Builder> getSupportedCharacterSets() {
            List<CharacterSet.Builder> result = SupportedCharacterSetsListCopier.copyToBuilder(this.supportedCharacterSets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSupportedCharacterSets(Collection<CharacterSet.BuilderImpl> supportedCharacterSets) {
            this.supportedCharacterSets = SupportedCharacterSetsListCopier.copyFromBuilder(supportedCharacterSets);
        }

        @Override
        public final Builder supportedCharacterSets(Collection<CharacterSet> supportedCharacterSets) {
            this.supportedCharacterSets = SupportedCharacterSetsListCopier.copy(supportedCharacterSets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedCharacterSets(CharacterSet... supportedCharacterSets) {
            supportedCharacterSets(Arrays.asList(supportedCharacterSets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedCharacterSets(Consumer<CharacterSet.Builder>... supportedCharacterSets) {
            supportedCharacterSets(Stream.of(supportedCharacterSets).map(c -> CharacterSet.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<UpgradeTarget.Builder> getValidUpgradeTarget() {
            List<UpgradeTarget.Builder> result = ValidUpgradeTargetListCopier.copyToBuilder(this.validUpgradeTarget);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setValidUpgradeTarget(Collection<UpgradeTarget.BuilderImpl> validUpgradeTarget) {
            this.validUpgradeTarget = ValidUpgradeTargetListCopier.copyFromBuilder(validUpgradeTarget);
        }

        @Override
        public final Builder validUpgradeTarget(Collection<UpgradeTarget> validUpgradeTarget) {
            this.validUpgradeTarget = ValidUpgradeTargetListCopier.copy(validUpgradeTarget);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validUpgradeTarget(UpgradeTarget... validUpgradeTarget) {
            validUpgradeTarget(Arrays.asList(validUpgradeTarget));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validUpgradeTarget(Consumer<UpgradeTarget.Builder>... validUpgradeTarget) {
            validUpgradeTarget(Stream.of(validUpgradeTarget).map(c -> UpgradeTarget.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<Timezone.Builder> getSupportedTimezones() {
            List<Timezone.Builder> result = SupportedTimezonesListCopier.copyToBuilder(this.supportedTimezones);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSupportedTimezones(Collection<Timezone.BuilderImpl> supportedTimezones) {
            this.supportedTimezones = SupportedTimezonesListCopier.copyFromBuilder(supportedTimezones);
        }

        @Override
        public final Builder supportedTimezones(Collection<Timezone> supportedTimezones) {
            this.supportedTimezones = SupportedTimezonesListCopier.copy(supportedTimezones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedTimezones(Timezone... supportedTimezones) {
            supportedTimezones(Arrays.asList(supportedTimezones));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedTimezones(Consumer<Timezone.Builder>... supportedTimezones) {
            supportedTimezones(Stream.of(supportedTimezones).map(c -> Timezone.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getExportableLogTypes() {
            if (exportableLogTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return exportableLogTypes;
        }

        public final void setExportableLogTypes(Collection<String> exportableLogTypes) {
            this.exportableLogTypes = LogTypeListCopier.copy(exportableLogTypes);
        }

        @Override
        public final Builder exportableLogTypes(Collection<String> exportableLogTypes) {
            this.exportableLogTypes = LogTypeListCopier.copy(exportableLogTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder exportableLogTypes(String... exportableLogTypes) {
            exportableLogTypes(Arrays.asList(exportableLogTypes));
            return this;
        }

        public final Boolean getSupportsLogExportsToCloudwatchLogs() {
            return supportsLogExportsToCloudwatchLogs;
        }

        public final void setSupportsLogExportsToCloudwatchLogs(Boolean supportsLogExportsToCloudwatchLogs) {
            this.supportsLogExportsToCloudwatchLogs = supportsLogExportsToCloudwatchLogs;
        }

        @Override
        public final Builder supportsLogExportsToCloudwatchLogs(Boolean supportsLogExportsToCloudwatchLogs) {
            this.supportsLogExportsToCloudwatchLogs = supportsLogExportsToCloudwatchLogs;
            return this;
        }

        public final Boolean getSupportsReadReplica() {
            return supportsReadReplica;
        }

        public final void setSupportsReadReplica(Boolean supportsReadReplica) {
            this.supportsReadReplica = supportsReadReplica;
        }

        @Override
        public final Builder supportsReadReplica(Boolean supportsReadReplica) {
            this.supportsReadReplica = supportsReadReplica;
            return this;
        }

        public final Boolean getSupportsGlobalDatabases() {
            return supportsGlobalDatabases;
        }

        public final void setSupportsGlobalDatabases(Boolean supportsGlobalDatabases) {
            this.supportsGlobalDatabases = supportsGlobalDatabases;
        }

        @Override
        public final Builder supportsGlobalDatabases(Boolean supportsGlobalDatabases) {
            this.supportsGlobalDatabases = supportsGlobalDatabases;
            return this;
        }

        @Override
        public DBEngineVersion build() {
            return new DBEngineVersion(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
