/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.neptune.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the details of an Amazon Neptune DB subnet group.
 * </p>
 * <p>
 * This data type is used as a response element in the <a>DescribeDBSubnetGroups</a> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DBSubnetGroup implements SdkPojo, Serializable, ToCopyableBuilder<DBSubnetGroup.Builder, DBSubnetGroup> {
    private static final SdkField<String> DB_SUBNET_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBSubnetGroupName").getter(getter(DBSubnetGroup::dbSubnetGroupName))
            .setter(setter(Builder::dbSubnetGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBSubnetGroupName").build()).build();

    private static final SdkField<String> DB_SUBNET_GROUP_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBSubnetGroupDescription").getter(getter(DBSubnetGroup::dbSubnetGroupDescription))
            .setter(setter(Builder::dbSubnetGroupDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBSubnetGroupDescription").build())
            .build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VpcId")
            .getter(getter(DBSubnetGroup::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<String> SUBNET_GROUP_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SubnetGroupStatus").getter(getter(DBSubnetGroup::subnetGroupStatus))
            .setter(setter(Builder::subnetGroupStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetGroupStatus").build()).build();

    private static final SdkField<List<Subnet>> SUBNETS_FIELD = SdkField
            .<List<Subnet>> builder(MarshallingType.LIST)
            .memberName("Subnets")
            .getter(getter(DBSubnetGroup::subnets))
            .setter(setter(Builder::subnets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subnets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Subnet")
                            .memberFieldInfo(
                                    SdkField.<Subnet> builder(MarshallingType.SDK_POJO)
                                            .constructor(Subnet::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Subnet").build()).build()).build()).build();

    private static final SdkField<String> DB_SUBNET_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBSubnetGroupArn").getter(getter(DBSubnetGroup::dbSubnetGroupArn))
            .setter(setter(Builder::dbSubnetGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBSubnetGroupArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections
            .unmodifiableList(Arrays.asList(DB_SUBNET_GROUP_NAME_FIELD, DB_SUBNET_GROUP_DESCRIPTION_FIELD, VPC_ID_FIELD,
                    SUBNET_GROUP_STATUS_FIELD, SUBNETS_FIELD, DB_SUBNET_GROUP_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String dbSubnetGroupName;

    private final String dbSubnetGroupDescription;

    private final String vpcId;

    private final String subnetGroupStatus;

    private final List<Subnet> subnets;

    private final String dbSubnetGroupArn;

    private DBSubnetGroup(BuilderImpl builder) {
        this.dbSubnetGroupName = builder.dbSubnetGroupName;
        this.dbSubnetGroupDescription = builder.dbSubnetGroupDescription;
        this.vpcId = builder.vpcId;
        this.subnetGroupStatus = builder.subnetGroupStatus;
        this.subnets = builder.subnets;
        this.dbSubnetGroupArn = builder.dbSubnetGroupArn;
    }

    /**
     * <p>
     * The name of the DB subnet group.
     * </p>
     * 
     * @return The name of the DB subnet group.
     */
    public final String dbSubnetGroupName() {
        return dbSubnetGroupName;
    }

    /**
     * <p>
     * Provides the description of the DB subnet group.
     * </p>
     * 
     * @return Provides the description of the DB subnet group.
     */
    public final String dbSubnetGroupDescription() {
        return dbSubnetGroupDescription;
    }

    /**
     * <p>
     * Provides the VpcId of the DB subnet group.
     * </p>
     * 
     * @return Provides the VpcId of the DB subnet group.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * Provides the status of the DB subnet group.
     * </p>
     * 
     * @return Provides the status of the DB subnet group.
     */
    public final String subnetGroupStatus() {
        return subnetGroupStatus;
    }

    /**
     * For responses, this returns true if the service returned a value for the Subnets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSubnets() {
        return subnets != null && !(subnets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains a list of <a>Subnet</a> elements.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubnets} method.
     * </p>
     * 
     * @return Contains a list of <a>Subnet</a> elements.
     */
    public final List<Subnet> subnets() {
        return subnets;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the DB subnet group.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the DB subnet group.
     */
    public final String dbSubnetGroupArn() {
        return dbSubnetGroupArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dbSubnetGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(dbSubnetGroupDescription());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(subnetGroupStatus());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnets() ? subnets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dbSubnetGroupArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DBSubnetGroup)) {
            return false;
        }
        DBSubnetGroup other = (DBSubnetGroup) obj;
        return Objects.equals(dbSubnetGroupName(), other.dbSubnetGroupName())
                && Objects.equals(dbSubnetGroupDescription(), other.dbSubnetGroupDescription())
                && Objects.equals(vpcId(), other.vpcId()) && Objects.equals(subnetGroupStatus(), other.subnetGroupStatus())
                && hasSubnets() == other.hasSubnets() && Objects.equals(subnets(), other.subnets())
                && Objects.equals(dbSubnetGroupArn(), other.dbSubnetGroupArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DBSubnetGroup").add("DBSubnetGroupName", dbSubnetGroupName())
                .add("DBSubnetGroupDescription", dbSubnetGroupDescription()).add("VpcId", vpcId())
                .add("SubnetGroupStatus", subnetGroupStatus()).add("Subnets", hasSubnets() ? subnets() : null)
                .add("DBSubnetGroupArn", dbSubnetGroupArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DBSubnetGroupName":
            return Optional.ofNullable(clazz.cast(dbSubnetGroupName()));
        case "DBSubnetGroupDescription":
            return Optional.ofNullable(clazz.cast(dbSubnetGroupDescription()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "SubnetGroupStatus":
            return Optional.ofNullable(clazz.cast(subnetGroupStatus()));
        case "Subnets":
            return Optional.ofNullable(clazz.cast(subnets()));
        case "DBSubnetGroupArn":
            return Optional.ofNullable(clazz.cast(dbSubnetGroupArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DBSubnetGroup, T> g) {
        return obj -> g.apply((DBSubnetGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DBSubnetGroup> {
        /**
         * <p>
         * The name of the DB subnet group.
         * </p>
         * 
         * @param dbSubnetGroupName
         *        The name of the DB subnet group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbSubnetGroupName(String dbSubnetGroupName);

        /**
         * <p>
         * Provides the description of the DB subnet group.
         * </p>
         * 
         * @param dbSubnetGroupDescription
         *        Provides the description of the DB subnet group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbSubnetGroupDescription(String dbSubnetGroupDescription);

        /**
         * <p>
         * Provides the VpcId of the DB subnet group.
         * </p>
         * 
         * @param vpcId
         *        Provides the VpcId of the DB subnet group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * Provides the status of the DB subnet group.
         * </p>
         * 
         * @param subnetGroupStatus
         *        Provides the status of the DB subnet group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetGroupStatus(String subnetGroupStatus);

        /**
         * <p>
         * Contains a list of <a>Subnet</a> elements.
         * </p>
         * 
         * @param subnets
         *        Contains a list of <a>Subnet</a> elements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnets(Collection<Subnet> subnets);

        /**
         * <p>
         * Contains a list of <a>Subnet</a> elements.
         * </p>
         * 
         * @param subnets
         *        Contains a list of <a>Subnet</a> elements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnets(Subnet... subnets);

        /**
         * <p>
         * Contains a list of <a>Subnet</a> elements.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Subnet>.Builder} avoiding the need
         * to create one manually via {@link List<Subnet>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Subnet>.Builder#build()} is called immediately and its
         * result is passed to {@link #subnets(List<Subnet>)}.
         * 
         * @param subnets
         *        a consumer that will call methods on {@link List<Subnet>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #subnets(List<Subnet>)
         */
        Builder subnets(Consumer<Subnet.Builder>... subnets);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the DB subnet group.
         * </p>
         * 
         * @param dbSubnetGroupArn
         *        The Amazon Resource Name (ARN) for the DB subnet group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbSubnetGroupArn(String dbSubnetGroupArn);
    }

    static final class BuilderImpl implements Builder {
        private String dbSubnetGroupName;

        private String dbSubnetGroupDescription;

        private String vpcId;

        private String subnetGroupStatus;

        private List<Subnet> subnets = DefaultSdkAutoConstructList.getInstance();

        private String dbSubnetGroupArn;

        private BuilderImpl() {
        }

        private BuilderImpl(DBSubnetGroup model) {
            dbSubnetGroupName(model.dbSubnetGroupName);
            dbSubnetGroupDescription(model.dbSubnetGroupDescription);
            vpcId(model.vpcId);
            subnetGroupStatus(model.subnetGroupStatus);
            subnets(model.subnets);
            dbSubnetGroupArn(model.dbSubnetGroupArn);
        }

        public final String getDbSubnetGroupName() {
            return dbSubnetGroupName;
        }

        public final void setDbSubnetGroupName(String dbSubnetGroupName) {
            this.dbSubnetGroupName = dbSubnetGroupName;
        }

        @Override
        public final Builder dbSubnetGroupName(String dbSubnetGroupName) {
            this.dbSubnetGroupName = dbSubnetGroupName;
            return this;
        }

        public final String getDbSubnetGroupDescription() {
            return dbSubnetGroupDescription;
        }

        public final void setDbSubnetGroupDescription(String dbSubnetGroupDescription) {
            this.dbSubnetGroupDescription = dbSubnetGroupDescription;
        }

        @Override
        public final Builder dbSubnetGroupDescription(String dbSubnetGroupDescription) {
            this.dbSubnetGroupDescription = dbSubnetGroupDescription;
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final String getSubnetGroupStatus() {
            return subnetGroupStatus;
        }

        public final void setSubnetGroupStatus(String subnetGroupStatus) {
            this.subnetGroupStatus = subnetGroupStatus;
        }

        @Override
        public final Builder subnetGroupStatus(String subnetGroupStatus) {
            this.subnetGroupStatus = subnetGroupStatus;
            return this;
        }

        public final List<Subnet.Builder> getSubnets() {
            List<Subnet.Builder> result = SubnetListCopier.copyToBuilder(this.subnets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSubnets(Collection<Subnet.BuilderImpl> subnets) {
            this.subnets = SubnetListCopier.copyFromBuilder(subnets);
        }

        @Override
        public final Builder subnets(Collection<Subnet> subnets) {
            this.subnets = SubnetListCopier.copy(subnets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnets(Subnet... subnets) {
            subnets(Arrays.asList(subnets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnets(Consumer<Subnet.Builder>... subnets) {
            subnets(Stream.of(subnets).map(c -> Subnet.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getDbSubnetGroupArn() {
            return dbSubnetGroupArn;
        }

        public final void setDbSubnetGroupArn(String dbSubnetGroupArn) {
            this.dbSubnetGroupArn = dbSubnetGroupArn;
        }

        @Override
        public final Builder dbSubnetGroupArn(String dbSubnetGroupArn) {
            this.dbSubnetGroupArn = dbSubnetGroupArn;
            return this;
        }

        @Override
        public DBSubnetGroup build() {
            return new DBSubnetGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
