/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.neptune.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class ModifyDbClusterRequest extends NeptuneRequest implements
        ToCopyableBuilder<ModifyDbClusterRequest.Builder, ModifyDbClusterRequest> {
    private static final SdkField<String> DB_CLUSTER_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBClusterIdentifier").getter(getter(ModifyDbClusterRequest::dbClusterIdentifier))
            .setter(setter(Builder::dbClusterIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterIdentifier").build())
            .build();

    private static final SdkField<String> NEW_DB_CLUSTER_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NewDBClusterIdentifier").getter(getter(ModifyDbClusterRequest::newDBClusterIdentifier))
            .setter(setter(Builder::newDBClusterIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NewDBClusterIdentifier").build())
            .build();

    private static final SdkField<Boolean> APPLY_IMMEDIATELY_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ApplyImmediately").getter(getter(ModifyDbClusterRequest::applyImmediately))
            .setter(setter(Builder::applyImmediately))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplyImmediately").build()).build();

    private static final SdkField<Integer> BACKUP_RETENTION_PERIOD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("BackupRetentionPeriod").getter(getter(ModifyDbClusterRequest::backupRetentionPeriod))
            .setter(setter(Builder::backupRetentionPeriod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupRetentionPeriod").build())
            .build();

    private static final SdkField<String> DB_CLUSTER_PARAMETER_GROUP_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DBClusterParameterGroupName")
            .getter(getter(ModifyDbClusterRequest::dbClusterParameterGroupName))
            .setter(setter(Builder::dbClusterParameterGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterParameterGroupName")
                    .build()).build();

    private static final SdkField<List<String>> VPC_SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("VpcSecurityGroupIds")
            .getter(getter(ModifyDbClusterRequest::vpcSecurityGroupIds))
            .setter(setter(Builder::vpcSecurityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcSecurityGroupIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("VpcSecurityGroupId")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("VpcSecurityGroupId").build()).build()).build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Port")
            .getter(getter(ModifyDbClusterRequest::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<String> MASTER_USER_PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MasterUserPassword").getter(getter(ModifyDbClusterRequest::masterUserPassword))
            .setter(setter(Builder::masterUserPassword))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MasterUserPassword").build())
            .build();

    private static final SdkField<String> OPTION_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OptionGroupName").getter(getter(ModifyDbClusterRequest::optionGroupName))
            .setter(setter(Builder::optionGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OptionGroupName").build()).build();

    private static final SdkField<String> PREFERRED_BACKUP_WINDOW_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreferredBackupWindow").getter(getter(ModifyDbClusterRequest::preferredBackupWindow))
            .setter(setter(Builder::preferredBackupWindow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredBackupWindow").build())
            .build();

    private static final SdkField<String> PREFERRED_MAINTENANCE_WINDOW_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PreferredMaintenanceWindow")
            .getter(getter(ModifyDbClusterRequest::preferredMaintenanceWindow))
            .setter(setter(Builder::preferredMaintenanceWindow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredMaintenanceWindow").build())
            .build();

    private static final SdkField<Boolean> ENABLE_IAM_DATABASE_AUTHENTICATION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnableIAMDatabaseAuthentication")
            .getter(getter(ModifyDbClusterRequest::enableIAMDatabaseAuthentication))
            .setter(setter(Builder::enableIAMDatabaseAuthentication))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableIAMDatabaseAuthentication")
                    .build()).build();

    private static final SdkField<CloudwatchLogsExportConfiguration> CLOUDWATCH_LOGS_EXPORT_CONFIGURATION_FIELD = SdkField
            .<CloudwatchLogsExportConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("CloudwatchLogsExportConfiguration")
            .getter(getter(ModifyDbClusterRequest::cloudwatchLogsExportConfiguration))
            .setter(setter(Builder::cloudwatchLogsExportConfiguration))
            .constructor(CloudwatchLogsExportConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudwatchLogsExportConfiguration")
                    .build()).build();

    private static final SdkField<String> ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EngineVersion").getter(getter(ModifyDbClusterRequest::engineVersion))
            .setter(setter(Builder::engineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final SdkField<Boolean> DELETION_PROTECTION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DeletionProtection").getter(getter(ModifyDbClusterRequest::deletionProtection))
            .setter(setter(Builder::deletionProtection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeletionProtection").build())
            .build();

    private static final SdkField<Boolean> COPY_TAGS_TO_SNAPSHOT_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CopyTagsToSnapshot").getter(getter(ModifyDbClusterRequest::copyTagsToSnapshot))
            .setter(setter(Builder::copyTagsToSnapshot))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopyTagsToSnapshot").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DB_CLUSTER_IDENTIFIER_FIELD,
            NEW_DB_CLUSTER_IDENTIFIER_FIELD, APPLY_IMMEDIATELY_FIELD, BACKUP_RETENTION_PERIOD_FIELD,
            DB_CLUSTER_PARAMETER_GROUP_NAME_FIELD, VPC_SECURITY_GROUP_IDS_FIELD, PORT_FIELD, MASTER_USER_PASSWORD_FIELD,
            OPTION_GROUP_NAME_FIELD, PREFERRED_BACKUP_WINDOW_FIELD, PREFERRED_MAINTENANCE_WINDOW_FIELD,
            ENABLE_IAM_DATABASE_AUTHENTICATION_FIELD, CLOUDWATCH_LOGS_EXPORT_CONFIGURATION_FIELD, ENGINE_VERSION_FIELD,
            DELETION_PROTECTION_FIELD, COPY_TAGS_TO_SNAPSHOT_FIELD));

    private final String dbClusterIdentifier;

    private final String newDBClusterIdentifier;

    private final Boolean applyImmediately;

    private final Integer backupRetentionPeriod;

    private final String dbClusterParameterGroupName;

    private final List<String> vpcSecurityGroupIds;

    private final Integer port;

    private final String masterUserPassword;

    private final String optionGroupName;

    private final String preferredBackupWindow;

    private final String preferredMaintenanceWindow;

    private final Boolean enableIAMDatabaseAuthentication;

    private final CloudwatchLogsExportConfiguration cloudwatchLogsExportConfiguration;

    private final String engineVersion;

    private final Boolean deletionProtection;

    private final Boolean copyTagsToSnapshot;

    private ModifyDbClusterRequest(BuilderImpl builder) {
        super(builder);
        this.dbClusterIdentifier = builder.dbClusterIdentifier;
        this.newDBClusterIdentifier = builder.newDBClusterIdentifier;
        this.applyImmediately = builder.applyImmediately;
        this.backupRetentionPeriod = builder.backupRetentionPeriod;
        this.dbClusterParameterGroupName = builder.dbClusterParameterGroupName;
        this.vpcSecurityGroupIds = builder.vpcSecurityGroupIds;
        this.port = builder.port;
        this.masterUserPassword = builder.masterUserPassword;
        this.optionGroupName = builder.optionGroupName;
        this.preferredBackupWindow = builder.preferredBackupWindow;
        this.preferredMaintenanceWindow = builder.preferredMaintenanceWindow;
        this.enableIAMDatabaseAuthentication = builder.enableIAMDatabaseAuthentication;
        this.cloudwatchLogsExportConfiguration = builder.cloudwatchLogsExportConfiguration;
        this.engineVersion = builder.engineVersion;
        this.deletionProtection = builder.deletionProtection;
        this.copyTagsToSnapshot = builder.copyTagsToSnapshot;
    }

    /**
     * <p>
     * The DB cluster identifier for the cluster being modified. This parameter is not case-sensitive.
     * </p>
     * <p>
     * Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must match the identifier of an existing DBCluster.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The DB cluster identifier for the cluster being modified. This parameter is not case-sensitive.</p>
     *         <p>
     *         Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Must match the identifier of an existing DBCluster.
     *         </p>
     *         </li>
     */
    public final String dbClusterIdentifier() {
        return dbClusterIdentifier;
    }

    /**
     * <p>
     * The new DB cluster identifier for the DB cluster when renaming a DB cluster. This value is stored as a lowercase
     * string.
     * </p>
     * <p>
     * Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must contain from 1 to 63 letters, numbers, or hyphens
     * </p>
     * </li>
     * <li>
     * <p>
     * The first character must be a letter
     * </p>
     * </li>
     * <li>
     * <p>
     * Cannot end with a hyphen or contain two consecutive hyphens
     * </p>
     * </li>
     * </ul>
     * <p>
     * Example: <code>my-cluster2</code>
     * </p>
     * 
     * @return The new DB cluster identifier for the DB cluster when renaming a DB cluster. This value is stored as a
     *         lowercase string.</p>
     *         <p>
     *         Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Must contain from 1 to 63 letters, numbers, or hyphens
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The first character must be a letter
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Cannot end with a hyphen or contain two consecutive hyphens
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Example: <code>my-cluster2</code>
     */
    public final String newDBClusterIdentifier() {
        return newDBClusterIdentifier;
    }

    /**
     * <p>
     * A value that specifies whether the modifications in this request and any pending modifications are asynchronously
     * applied as soon as possible, regardless of the <code>PreferredMaintenanceWindow</code> setting for the DB
     * cluster. If this parameter is set to <code>false</code>, changes to the DB cluster are applied during the next
     * maintenance window.
     * </p>
     * <p>
     * The <code>ApplyImmediately</code> parameter only affects <code>NewDBClusterIdentifier</code> values. If you set
     * the <code>ApplyImmediately</code> parameter value to false, then changes to <code>NewDBClusterIdentifier</code>
     * values are applied during the next maintenance window. All other changes are applied immediately, regardless of
     * the value of the <code>ApplyImmediately</code> parameter.
     * </p>
     * <p>
     * Default: <code>false</code>
     * </p>
     * 
     * @return A value that specifies whether the modifications in this request and any pending modifications are
     *         asynchronously applied as soon as possible, regardless of the <code>PreferredMaintenanceWindow</code>
     *         setting for the DB cluster. If this parameter is set to <code>false</code>, changes to the DB cluster are
     *         applied during the next maintenance window.</p>
     *         <p>
     *         The <code>ApplyImmediately</code> parameter only affects <code>NewDBClusterIdentifier</code> values. If
     *         you set the <code>ApplyImmediately</code> parameter value to false, then changes to
     *         <code>NewDBClusterIdentifier</code> values are applied during the next maintenance window. All other
     *         changes are applied immediately, regardless of the value of the <code>ApplyImmediately</code> parameter.
     *         </p>
     *         <p>
     *         Default: <code>false</code>
     */
    public final Boolean applyImmediately() {
        return applyImmediately;
    }

    /**
     * <p>
     * The number of days for which automated backups are retained. You must specify a minimum value of 1.
     * </p>
     * <p>
     * Default: 1
     * </p>
     * <p>
     * Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must be a value from 1 to 35
     * </p>
     * </li>
     * </ul>
     * 
     * @return The number of days for which automated backups are retained. You must specify a minimum value of 1.</p>
     *         <p>
     *         Default: 1
     *         </p>
     *         <p>
     *         Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Must be a value from 1 to 35
     *         </p>
     *         </li>
     */
    public final Integer backupRetentionPeriod() {
        return backupRetentionPeriod;
    }

    /**
     * <p>
     * The name of the DB cluster parameter group to use for the DB cluster.
     * </p>
     * 
     * @return The name of the DB cluster parameter group to use for the DB cluster.
     */
    public final String dbClusterParameterGroupName() {
        return dbClusterParameterGroupName;
    }

    /**
     * Returns true if the VpcSecurityGroupIds property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasVpcSecurityGroupIds() {
        return vpcSecurityGroupIds != null && !(vpcSecurityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of VPC security groups that the DB cluster will belong to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasVpcSecurityGroupIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of VPC security groups that the DB cluster will belong to.
     */
    public final List<String> vpcSecurityGroupIds() {
        return vpcSecurityGroupIds;
    }

    /**
     * <p>
     * The port number on which the DB cluster accepts connections.
     * </p>
     * <p>
     * Constraints: Value must be <code>1150-65535</code>
     * </p>
     * <p>
     * Default: The same port as the original DB cluster.
     * </p>
     * 
     * @return The port number on which the DB cluster accepts connections.</p>
     *         <p>
     *         Constraints: Value must be <code>1150-65535</code>
     *         </p>
     *         <p>
     *         Default: The same port as the original DB cluster.
     */
    public final Integer port() {
        return port;
    }

    /**
     * <p>
     * Not supported by Neptune.
     * </p>
     * 
     * @return Not supported by Neptune.
     */
    public final String masterUserPassword() {
        return masterUserPassword;
    }

    /**
     * <p>
     * <i>Not supported by Neptune.</i>
     * </p>
     * 
     * @return <i>Not supported by Neptune.</i>
     */
    public final String optionGroupName() {
        return optionGroupName;
    }

    /**
     * <p>
     * The daily time range during which automated backups are created if automated backups are enabled, using the
     * <code>BackupRetentionPeriod</code> parameter.
     * </p>
     * <p>
     * The default is a 30-minute window selected at random from an 8-hour block of time for each Amazon Region.
     * </p>
     * <p>
     * Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must be in the format <code>hh24:mi-hh24:mi</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Must be in Universal Coordinated Time (UTC).
     * </p>
     * </li>
     * <li>
     * <p>
     * Must not conflict with the preferred maintenance window.
     * </p>
     * </li>
     * <li>
     * <p>
     * Must be at least 30 minutes.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The daily time range during which automated backups are created if automated backups are enabled, using
     *         the <code>BackupRetentionPeriod</code> parameter.</p>
     *         <p>
     *         The default is a 30-minute window selected at random from an 8-hour block of time for each Amazon Region.
     *         </p>
     *         <p>
     *         Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Must be in the format <code>hh24:mi-hh24:mi</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Must be in Universal Coordinated Time (UTC).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Must not conflict with the preferred maintenance window.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Must be at least 30 minutes.
     *         </p>
     *         </li>
     */
    public final String preferredBackupWindow() {
        return preferredBackupWindow;
    }

    /**
     * <p>
     * The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).
     * </p>
     * <p>
     * Format: <code>ddd:hh24:mi-ddd:hh24:mi</code>
     * </p>
     * <p>
     * The default is a 30-minute window selected at random from an 8-hour block of time for each Amazon Region,
     * occurring on a random day of the week.
     * </p>
     * <p>
     * Valid Days: Mon, Tue, Wed, Thu, Fri, Sat, Sun.
     * </p>
     * <p>
     * Constraints: Minimum 30-minute window.
     * </p>
     * 
     * @return The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).</p>
     *         <p>
     *         Format: <code>ddd:hh24:mi-ddd:hh24:mi</code>
     *         </p>
     *         <p>
     *         The default is a 30-minute window selected at random from an 8-hour block of time for each Amazon Region,
     *         occurring on a random day of the week.
     *         </p>
     *         <p>
     *         Valid Days: Mon, Tue, Wed, Thu, Fri, Sat, Sun.
     *         </p>
     *         <p>
     *         Constraints: Minimum 30-minute window.
     */
    public final String preferredMaintenanceWindow() {
        return preferredMaintenanceWindow;
    }

    /**
     * <p>
     * True to enable mapping of Amazon Identity and Access Management (IAM) accounts to database accounts, and
     * otherwise false.
     * </p>
     * <p>
     * Default: <code>false</code>
     * </p>
     * 
     * @return True to enable mapping of Amazon Identity and Access Management (IAM) accounts to database accounts, and
     *         otherwise false.</p>
     *         <p>
     *         Default: <code>false</code>
     */
    public final Boolean enableIAMDatabaseAuthentication() {
        return enableIAMDatabaseAuthentication;
    }

    /**
     * <p>
     * The configuration setting for the log types to be enabled for export to CloudWatch Logs for a specific DB
     * cluster.
     * </p>
     * 
     * @return The configuration setting for the log types to be enabled for export to CloudWatch Logs for a specific DB
     *         cluster.
     */
    public final CloudwatchLogsExportConfiguration cloudwatchLogsExportConfiguration() {
        return cloudwatchLogsExportConfiguration;
    }

    /**
     * <p>
     * The version number of the database engine to which you want to upgrade. Changing this parameter results in an
     * outage. The change is applied during the next maintenance window unless the <code>ApplyImmediately</code>
     * parameter is set to true.
     * </p>
     * <p>
     * For a list of valid engine versions, see <a
     * href="https://docs.aws.amazon.com/neptune/latest/userguide/engine-releases.html">Engine Releases for Amazon
     * Neptune</a>, or call <a
     * href="https://docs.aws.amazon.com/neptune/latest/userguide/api-other-apis.html#DescribeDBEngineVersions"
     * >DescribeDBEngineVersions</a>.
     * </p>
     * 
     * @return The version number of the database engine to which you want to upgrade. Changing this parameter results
     *         in an outage. The change is applied during the next maintenance window unless the
     *         <code>ApplyImmediately</code> parameter is set to true.</p>
     *         <p>
     *         For a list of valid engine versions, see <a
     *         href="https://docs.aws.amazon.com/neptune/latest/userguide/engine-releases.html">Engine Releases for
     *         Amazon Neptune</a>, or call <a href=
     *         "https://docs.aws.amazon.com/neptune/latest/userguide/api-other-apis.html#DescribeDBEngineVersions"
     *         >DescribeDBEngineVersions</a>.
     */
    public final String engineVersion() {
        return engineVersion;
    }

    /**
     * <p>
     * A value that indicates whether the DB cluster has deletion protection enabled. The database can't be deleted when
     * deletion protection is enabled. By default, deletion protection is disabled.
     * </p>
     * 
     * @return A value that indicates whether the DB cluster has deletion protection enabled. The database can't be
     *         deleted when deletion protection is enabled. By default, deletion protection is disabled.
     */
    public final Boolean deletionProtection() {
        return deletionProtection;
    }

    /**
     * <p>
     * <i>If set to <code>true</code>, tags are copied to any snapshot of the DB cluster that is created.</i>
     * </p>
     * 
     * @return <i>If set to <code>true</code>, tags are copied to any snapshot of the DB cluster that is created.</i>
     */
    public final Boolean copyTagsToSnapshot() {
        return copyTagsToSnapshot;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(newDBClusterIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(applyImmediately());
        hashCode = 31 * hashCode + Objects.hashCode(backupRetentionPeriod());
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterParameterGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(hasVpcSecurityGroupIds() ? vpcSecurityGroupIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(masterUserPassword());
        hashCode = 31 * hashCode + Objects.hashCode(optionGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(preferredBackupWindow());
        hashCode = 31 * hashCode + Objects.hashCode(preferredMaintenanceWindow());
        hashCode = 31 * hashCode + Objects.hashCode(enableIAMDatabaseAuthentication());
        hashCode = 31 * hashCode + Objects.hashCode(cloudwatchLogsExportConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(deletionProtection());
        hashCode = 31 * hashCode + Objects.hashCode(copyTagsToSnapshot());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ModifyDbClusterRequest)) {
            return false;
        }
        ModifyDbClusterRequest other = (ModifyDbClusterRequest) obj;
        return Objects.equals(dbClusterIdentifier(), other.dbClusterIdentifier())
                && Objects.equals(newDBClusterIdentifier(), other.newDBClusterIdentifier())
                && Objects.equals(applyImmediately(), other.applyImmediately())
                && Objects.equals(backupRetentionPeriod(), other.backupRetentionPeriod())
                && Objects.equals(dbClusterParameterGroupName(), other.dbClusterParameterGroupName())
                && hasVpcSecurityGroupIds() == other.hasVpcSecurityGroupIds()
                && Objects.equals(vpcSecurityGroupIds(), other.vpcSecurityGroupIds()) && Objects.equals(port(), other.port())
                && Objects.equals(masterUserPassword(), other.masterUserPassword())
                && Objects.equals(optionGroupName(), other.optionGroupName())
                && Objects.equals(preferredBackupWindow(), other.preferredBackupWindow())
                && Objects.equals(preferredMaintenanceWindow(), other.preferredMaintenanceWindow())
                && Objects.equals(enableIAMDatabaseAuthentication(), other.enableIAMDatabaseAuthentication())
                && Objects.equals(cloudwatchLogsExportConfiguration(), other.cloudwatchLogsExportConfiguration())
                && Objects.equals(engineVersion(), other.engineVersion())
                && Objects.equals(deletionProtection(), other.deletionProtection())
                && Objects.equals(copyTagsToSnapshot(), other.copyTagsToSnapshot());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ModifyDbClusterRequest").add("DBClusterIdentifier", dbClusterIdentifier())
                .add("NewDBClusterIdentifier", newDBClusterIdentifier()).add("ApplyImmediately", applyImmediately())
                .add("BackupRetentionPeriod", backupRetentionPeriod())
                .add("DBClusterParameterGroupName", dbClusterParameterGroupName())
                .add("VpcSecurityGroupIds", hasVpcSecurityGroupIds() ? vpcSecurityGroupIds() : null).add("Port", port())
                .add("MasterUserPassword", masterUserPassword()).add("OptionGroupName", optionGroupName())
                .add("PreferredBackupWindow", preferredBackupWindow())
                .add("PreferredMaintenanceWindow", preferredMaintenanceWindow())
                .add("EnableIAMDatabaseAuthentication", enableIAMDatabaseAuthentication())
                .add("CloudwatchLogsExportConfiguration", cloudwatchLogsExportConfiguration())
                .add("EngineVersion", engineVersion()).add("DeletionProtection", deletionProtection())
                .add("CopyTagsToSnapshot", copyTagsToSnapshot()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DBClusterIdentifier":
            return Optional.ofNullable(clazz.cast(dbClusterIdentifier()));
        case "NewDBClusterIdentifier":
            return Optional.ofNullable(clazz.cast(newDBClusterIdentifier()));
        case "ApplyImmediately":
            return Optional.ofNullable(clazz.cast(applyImmediately()));
        case "BackupRetentionPeriod":
            return Optional.ofNullable(clazz.cast(backupRetentionPeriod()));
        case "DBClusterParameterGroupName":
            return Optional.ofNullable(clazz.cast(dbClusterParameterGroupName()));
        case "VpcSecurityGroupIds":
            return Optional.ofNullable(clazz.cast(vpcSecurityGroupIds()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "MasterUserPassword":
            return Optional.ofNullable(clazz.cast(masterUserPassword()));
        case "OptionGroupName":
            return Optional.ofNullable(clazz.cast(optionGroupName()));
        case "PreferredBackupWindow":
            return Optional.ofNullable(clazz.cast(preferredBackupWindow()));
        case "PreferredMaintenanceWindow":
            return Optional.ofNullable(clazz.cast(preferredMaintenanceWindow()));
        case "EnableIAMDatabaseAuthentication":
            return Optional.ofNullable(clazz.cast(enableIAMDatabaseAuthentication()));
        case "CloudwatchLogsExportConfiguration":
            return Optional.ofNullable(clazz.cast(cloudwatchLogsExportConfiguration()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "DeletionProtection":
            return Optional.ofNullable(clazz.cast(deletionProtection()));
        case "CopyTagsToSnapshot":
            return Optional.ofNullable(clazz.cast(copyTagsToSnapshot()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ModifyDbClusterRequest, T> g) {
        return obj -> g.apply((ModifyDbClusterRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends NeptuneRequest.Builder, SdkPojo, CopyableBuilder<Builder, ModifyDbClusterRequest> {
        /**
         * <p>
         * The DB cluster identifier for the cluster being modified. This parameter is not case-sensitive.
         * </p>
         * <p>
         * Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Must match the identifier of an existing DBCluster.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dbClusterIdentifier
         *        The DB cluster identifier for the cluster being modified. This parameter is not case-sensitive.</p>
         *        <p>
         *        Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Must match the identifier of an existing DBCluster.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterIdentifier(String dbClusterIdentifier);

        /**
         * <p>
         * The new DB cluster identifier for the DB cluster when renaming a DB cluster. This value is stored as a
         * lowercase string.
         * </p>
         * <p>
         * Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Must contain from 1 to 63 letters, numbers, or hyphens
         * </p>
         * </li>
         * <li>
         * <p>
         * The first character must be a letter
         * </p>
         * </li>
         * <li>
         * <p>
         * Cannot end with a hyphen or contain two consecutive hyphens
         * </p>
         * </li>
         * </ul>
         * <p>
         * Example: <code>my-cluster2</code>
         * </p>
         * 
         * @param newDBClusterIdentifier
         *        The new DB cluster identifier for the DB cluster when renaming a DB cluster. This value is stored as a
         *        lowercase string.</p>
         *        <p>
         *        Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Must contain from 1 to 63 letters, numbers, or hyphens
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The first character must be a letter
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cannot end with a hyphen or contain two consecutive hyphens
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Example: <code>my-cluster2</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder newDBClusterIdentifier(String newDBClusterIdentifier);

        /**
         * <p>
         * A value that specifies whether the modifications in this request and any pending modifications are
         * asynchronously applied as soon as possible, regardless of the <code>PreferredMaintenanceWindow</code> setting
         * for the DB cluster. If this parameter is set to <code>false</code>, changes to the DB cluster are applied
         * during the next maintenance window.
         * </p>
         * <p>
         * The <code>ApplyImmediately</code> parameter only affects <code>NewDBClusterIdentifier</code> values. If you
         * set the <code>ApplyImmediately</code> parameter value to false, then changes to
         * <code>NewDBClusterIdentifier</code> values are applied during the next maintenance window. All other changes
         * are applied immediately, regardless of the value of the <code>ApplyImmediately</code> parameter.
         * </p>
         * <p>
         * Default: <code>false</code>
         * </p>
         * 
         * @param applyImmediately
         *        A value that specifies whether the modifications in this request and any pending modifications are
         *        asynchronously applied as soon as possible, regardless of the <code>PreferredMaintenanceWindow</code>
         *        setting for the DB cluster. If this parameter is set to <code>false</code>, changes to the DB cluster
         *        are applied during the next maintenance window.</p>
         *        <p>
         *        The <code>ApplyImmediately</code> parameter only affects <code>NewDBClusterIdentifier</code> values.
         *        If you set the <code>ApplyImmediately</code> parameter value to false, then changes to
         *        <code>NewDBClusterIdentifier</code> values are applied during the next maintenance window. All other
         *        changes are applied immediately, regardless of the value of the <code>ApplyImmediately</code>
         *        parameter.
         *        </p>
         *        <p>
         *        Default: <code>false</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applyImmediately(Boolean applyImmediately);

        /**
         * <p>
         * The number of days for which automated backups are retained. You must specify a minimum value of 1.
         * </p>
         * <p>
         * Default: 1
         * </p>
         * <p>
         * Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Must be a value from 1 to 35
         * </p>
         * </li>
         * </ul>
         * 
         * @param backupRetentionPeriod
         *        The number of days for which automated backups are retained. You must specify a minimum value of
         *        1.</p>
         *        <p>
         *        Default: 1
         *        </p>
         *        <p>
         *        Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Must be a value from 1 to 35
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupRetentionPeriod(Integer backupRetentionPeriod);

        /**
         * <p>
         * The name of the DB cluster parameter group to use for the DB cluster.
         * </p>
         * 
         * @param dbClusterParameterGroupName
         *        The name of the DB cluster parameter group to use for the DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterParameterGroupName(String dbClusterParameterGroupName);

        /**
         * <p>
         * A list of VPC security groups that the DB cluster will belong to.
         * </p>
         * 
         * @param vpcSecurityGroupIds
         *        A list of VPC security groups that the DB cluster will belong to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSecurityGroupIds(Collection<String> vpcSecurityGroupIds);

        /**
         * <p>
         * A list of VPC security groups that the DB cluster will belong to.
         * </p>
         * 
         * @param vpcSecurityGroupIds
         *        A list of VPC security groups that the DB cluster will belong to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSecurityGroupIds(String... vpcSecurityGroupIds);

        /**
         * <p>
         * The port number on which the DB cluster accepts connections.
         * </p>
         * <p>
         * Constraints: Value must be <code>1150-65535</code>
         * </p>
         * <p>
         * Default: The same port as the original DB cluster.
         * </p>
         * 
         * @param port
         *        The port number on which the DB cluster accepts connections.</p>
         *        <p>
         *        Constraints: Value must be <code>1150-65535</code>
         *        </p>
         *        <p>
         *        Default: The same port as the original DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * Not supported by Neptune.
         * </p>
         * 
         * @param masterUserPassword
         *        Not supported by Neptune.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterUserPassword(String masterUserPassword);

        /**
         * <p>
         * <i>Not supported by Neptune.</i>
         * </p>
         * 
         * @param optionGroupName
         *        <i>Not supported by Neptune.</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder optionGroupName(String optionGroupName);

        /**
         * <p>
         * The daily time range during which automated backups are created if automated backups are enabled, using the
         * <code>BackupRetentionPeriod</code> parameter.
         * </p>
         * <p>
         * The default is a 30-minute window selected at random from an 8-hour block of time for each Amazon Region.
         * </p>
         * <p>
         * Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Must be in the format <code>hh24:mi-hh24:mi</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Must be in Universal Coordinated Time (UTC).
         * </p>
         * </li>
         * <li>
         * <p>
         * Must not conflict with the preferred maintenance window.
         * </p>
         * </li>
         * <li>
         * <p>
         * Must be at least 30 minutes.
         * </p>
         * </li>
         * </ul>
         * 
         * @param preferredBackupWindow
         *        The daily time range during which automated backups are created if automated backups are enabled,
         *        using the <code>BackupRetentionPeriod</code> parameter.</p>
         *        <p>
         *        The default is a 30-minute window selected at random from an 8-hour block of time for each Amazon
         *        Region.
         *        </p>
         *        <p>
         *        Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Must be in the format <code>hh24:mi-hh24:mi</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must be in Universal Coordinated Time (UTC).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must not conflict with the preferred maintenance window.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must be at least 30 minutes.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredBackupWindow(String preferredBackupWindow);

        /**
         * <p>
         * The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).
         * </p>
         * <p>
         * Format: <code>ddd:hh24:mi-ddd:hh24:mi</code>
         * </p>
         * <p>
         * The default is a 30-minute window selected at random from an 8-hour block of time for each Amazon Region,
         * occurring on a random day of the week.
         * </p>
         * <p>
         * Valid Days: Mon, Tue, Wed, Thu, Fri, Sat, Sun.
         * </p>
         * <p>
         * Constraints: Minimum 30-minute window.
         * </p>
         * 
         * @param preferredMaintenanceWindow
         *        The weekly time range during which system maintenance can occur, in Universal Coordinated Time
         *        (UTC).</p>
         *        <p>
         *        Format: <code>ddd:hh24:mi-ddd:hh24:mi</code>
         *        </p>
         *        <p>
         *        The default is a 30-minute window selected at random from an 8-hour block of time for each Amazon
         *        Region, occurring on a random day of the week.
         *        </p>
         *        <p>
         *        Valid Days: Mon, Tue, Wed, Thu, Fri, Sat, Sun.
         *        </p>
         *        <p>
         *        Constraints: Minimum 30-minute window.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredMaintenanceWindow(String preferredMaintenanceWindow);

        /**
         * <p>
         * True to enable mapping of Amazon Identity and Access Management (IAM) accounts to database accounts, and
         * otherwise false.
         * </p>
         * <p>
         * Default: <code>false</code>
         * </p>
         * 
         * @param enableIAMDatabaseAuthentication
         *        True to enable mapping of Amazon Identity and Access Management (IAM) accounts to database accounts,
         *        and otherwise false.</p>
         *        <p>
         *        Default: <code>false</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableIAMDatabaseAuthentication(Boolean enableIAMDatabaseAuthentication);

        /**
         * <p>
         * The configuration setting for the log types to be enabled for export to CloudWatch Logs for a specific DB
         * cluster.
         * </p>
         * 
         * @param cloudwatchLogsExportConfiguration
         *        The configuration setting for the log types to be enabled for export to CloudWatch Logs for a specific
         *        DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudwatchLogsExportConfiguration(CloudwatchLogsExportConfiguration cloudwatchLogsExportConfiguration);

        /**
         * <p>
         * The configuration setting for the log types to be enabled for export to CloudWatch Logs for a specific DB
         * cluster.
         * </p>
         * This is a convenience that creates an instance of the {@link CloudwatchLogsExportConfiguration.Builder}
         * avoiding the need to create one manually via {@link CloudwatchLogsExportConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link CloudwatchLogsExportConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #cloudwatchLogsExportConfiguration(CloudwatchLogsExportConfiguration)}.
         * 
         * @param cloudwatchLogsExportConfiguration
         *        a consumer that will call methods on {@link CloudwatchLogsExportConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cloudwatchLogsExportConfiguration(CloudwatchLogsExportConfiguration)
         */
        default Builder cloudwatchLogsExportConfiguration(
                Consumer<CloudwatchLogsExportConfiguration.Builder> cloudwatchLogsExportConfiguration) {
            return cloudwatchLogsExportConfiguration(CloudwatchLogsExportConfiguration.builder()
                    .applyMutation(cloudwatchLogsExportConfiguration).build());
        }

        /**
         * <p>
         * The version number of the database engine to which you want to upgrade. Changing this parameter results in an
         * outage. The change is applied during the next maintenance window unless the <code>ApplyImmediately</code>
         * parameter is set to true.
         * </p>
         * <p>
         * For a list of valid engine versions, see <a
         * href="https://docs.aws.amazon.com/neptune/latest/userguide/engine-releases.html">Engine Releases for Amazon
         * Neptune</a>, or call <a
         * href="https://docs.aws.amazon.com/neptune/latest/userguide/api-other-apis.html#DescribeDBEngineVersions"
         * >DescribeDBEngineVersions</a>.
         * </p>
         * 
         * @param engineVersion
         *        The version number of the database engine to which you want to upgrade. Changing this parameter
         *        results in an outage. The change is applied during the next maintenance window unless the
         *        <code>ApplyImmediately</code> parameter is set to true.</p>
         *        <p>
         *        For a list of valid engine versions, see <a
         *        href="https://docs.aws.amazon.com/neptune/latest/userguide/engine-releases.html">Engine Releases for
         *        Amazon Neptune</a>, or call <a href=
         *        "https://docs.aws.amazon.com/neptune/latest/userguide/api-other-apis.html#DescribeDBEngineVersions"
         *        >DescribeDBEngineVersions</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(String engineVersion);

        /**
         * <p>
         * A value that indicates whether the DB cluster has deletion protection enabled. The database can't be deleted
         * when deletion protection is enabled. By default, deletion protection is disabled.
         * </p>
         * 
         * @param deletionProtection
         *        A value that indicates whether the DB cluster has deletion protection enabled. The database can't be
         *        deleted when deletion protection is enabled. By default, deletion protection is disabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletionProtection(Boolean deletionProtection);

        /**
         * <p>
         * <i>If set to <code>true</code>, tags are copied to any snapshot of the DB cluster that is created.</i>
         * </p>
         * 
         * @param copyTagsToSnapshot
         *        <i>If set to <code>true</code>, tags are copied to any snapshot of the DB cluster that is created.</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder copyTagsToSnapshot(Boolean copyTagsToSnapshot);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends NeptuneRequest.BuilderImpl implements Builder {
        private String dbClusterIdentifier;

        private String newDBClusterIdentifier;

        private Boolean applyImmediately;

        private Integer backupRetentionPeriod;

        private String dbClusterParameterGroupName;

        private List<String> vpcSecurityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private Integer port;

        private String masterUserPassword;

        private String optionGroupName;

        private String preferredBackupWindow;

        private String preferredMaintenanceWindow;

        private Boolean enableIAMDatabaseAuthentication;

        private CloudwatchLogsExportConfiguration cloudwatchLogsExportConfiguration;

        private String engineVersion;

        private Boolean deletionProtection;

        private Boolean copyTagsToSnapshot;

        private BuilderImpl() {
        }

        private BuilderImpl(ModifyDbClusterRequest model) {
            super(model);
            dbClusterIdentifier(model.dbClusterIdentifier);
            newDBClusterIdentifier(model.newDBClusterIdentifier);
            applyImmediately(model.applyImmediately);
            backupRetentionPeriod(model.backupRetentionPeriod);
            dbClusterParameterGroupName(model.dbClusterParameterGroupName);
            vpcSecurityGroupIds(model.vpcSecurityGroupIds);
            port(model.port);
            masterUserPassword(model.masterUserPassword);
            optionGroupName(model.optionGroupName);
            preferredBackupWindow(model.preferredBackupWindow);
            preferredMaintenanceWindow(model.preferredMaintenanceWindow);
            enableIAMDatabaseAuthentication(model.enableIAMDatabaseAuthentication);
            cloudwatchLogsExportConfiguration(model.cloudwatchLogsExportConfiguration);
            engineVersion(model.engineVersion);
            deletionProtection(model.deletionProtection);
            copyTagsToSnapshot(model.copyTagsToSnapshot);
        }

        public final String getDbClusterIdentifier() {
            return dbClusterIdentifier;
        }

        @Override
        public final Builder dbClusterIdentifier(String dbClusterIdentifier) {
            this.dbClusterIdentifier = dbClusterIdentifier;
            return this;
        }

        public final void setDbClusterIdentifier(String dbClusterIdentifier) {
            this.dbClusterIdentifier = dbClusterIdentifier;
        }

        public final String getNewDBClusterIdentifier() {
            return newDBClusterIdentifier;
        }

        @Override
        public final Builder newDBClusterIdentifier(String newDBClusterIdentifier) {
            this.newDBClusterIdentifier = newDBClusterIdentifier;
            return this;
        }

        public final void setNewDBClusterIdentifier(String newDBClusterIdentifier) {
            this.newDBClusterIdentifier = newDBClusterIdentifier;
        }

        public final Boolean getApplyImmediately() {
            return applyImmediately;
        }

        @Override
        public final Builder applyImmediately(Boolean applyImmediately) {
            this.applyImmediately = applyImmediately;
            return this;
        }

        public final void setApplyImmediately(Boolean applyImmediately) {
            this.applyImmediately = applyImmediately;
        }

        public final Integer getBackupRetentionPeriod() {
            return backupRetentionPeriod;
        }

        @Override
        public final Builder backupRetentionPeriod(Integer backupRetentionPeriod) {
            this.backupRetentionPeriod = backupRetentionPeriod;
            return this;
        }

        public final void setBackupRetentionPeriod(Integer backupRetentionPeriod) {
            this.backupRetentionPeriod = backupRetentionPeriod;
        }

        public final String getDbClusterParameterGroupName() {
            return dbClusterParameterGroupName;
        }

        @Override
        public final Builder dbClusterParameterGroupName(String dbClusterParameterGroupName) {
            this.dbClusterParameterGroupName = dbClusterParameterGroupName;
            return this;
        }

        public final void setDbClusterParameterGroupName(String dbClusterParameterGroupName) {
            this.dbClusterParameterGroupName = dbClusterParameterGroupName;
        }

        public final Collection<String> getVpcSecurityGroupIds() {
            if (vpcSecurityGroupIds instanceof SdkAutoConstructList) {
                return null;
            }
            return vpcSecurityGroupIds;
        }

        @Override
        public final Builder vpcSecurityGroupIds(Collection<String> vpcSecurityGroupIds) {
            this.vpcSecurityGroupIds = VpcSecurityGroupIdListCopier.copy(vpcSecurityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vpcSecurityGroupIds(String... vpcSecurityGroupIds) {
            vpcSecurityGroupIds(Arrays.asList(vpcSecurityGroupIds));
            return this;
        }

        public final void setVpcSecurityGroupIds(Collection<String> vpcSecurityGroupIds) {
            this.vpcSecurityGroupIds = VpcSecurityGroupIdListCopier.copy(vpcSecurityGroupIds);
        }

        public final Integer getPort() {
            return port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        public final String getMasterUserPassword() {
            return masterUserPassword;
        }

        @Override
        public final Builder masterUserPassword(String masterUserPassword) {
            this.masterUserPassword = masterUserPassword;
            return this;
        }

        public final void setMasterUserPassword(String masterUserPassword) {
            this.masterUserPassword = masterUserPassword;
        }

        public final String getOptionGroupName() {
            return optionGroupName;
        }

        @Override
        public final Builder optionGroupName(String optionGroupName) {
            this.optionGroupName = optionGroupName;
            return this;
        }

        public final void setOptionGroupName(String optionGroupName) {
            this.optionGroupName = optionGroupName;
        }

        public final String getPreferredBackupWindow() {
            return preferredBackupWindow;
        }

        @Override
        public final Builder preferredBackupWindow(String preferredBackupWindow) {
            this.preferredBackupWindow = preferredBackupWindow;
            return this;
        }

        public final void setPreferredBackupWindow(String preferredBackupWindow) {
            this.preferredBackupWindow = preferredBackupWindow;
        }

        public final String getPreferredMaintenanceWindow() {
            return preferredMaintenanceWindow;
        }

        @Override
        public final Builder preferredMaintenanceWindow(String preferredMaintenanceWindow) {
            this.preferredMaintenanceWindow = preferredMaintenanceWindow;
            return this;
        }

        public final void setPreferredMaintenanceWindow(String preferredMaintenanceWindow) {
            this.preferredMaintenanceWindow = preferredMaintenanceWindow;
        }

        public final Boolean getEnableIAMDatabaseAuthentication() {
            return enableIAMDatabaseAuthentication;
        }

        @Override
        public final Builder enableIAMDatabaseAuthentication(Boolean enableIAMDatabaseAuthentication) {
            this.enableIAMDatabaseAuthentication = enableIAMDatabaseAuthentication;
            return this;
        }

        public final void setEnableIAMDatabaseAuthentication(Boolean enableIAMDatabaseAuthentication) {
            this.enableIAMDatabaseAuthentication = enableIAMDatabaseAuthentication;
        }

        public final CloudwatchLogsExportConfiguration.Builder getCloudwatchLogsExportConfiguration() {
            return cloudwatchLogsExportConfiguration != null ? cloudwatchLogsExportConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder cloudwatchLogsExportConfiguration(CloudwatchLogsExportConfiguration cloudwatchLogsExportConfiguration) {
            this.cloudwatchLogsExportConfiguration = cloudwatchLogsExportConfiguration;
            return this;
        }

        public final void setCloudwatchLogsExportConfiguration(
                CloudwatchLogsExportConfiguration.BuilderImpl cloudwatchLogsExportConfiguration) {
            this.cloudwatchLogsExportConfiguration = cloudwatchLogsExportConfiguration != null ? cloudwatchLogsExportConfiguration
                    .build() : null;
        }

        public final String getEngineVersion() {
            return engineVersion;
        }

        @Override
        public final Builder engineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        public final Boolean getDeletionProtection() {
            return deletionProtection;
        }

        @Override
        public final Builder deletionProtection(Boolean deletionProtection) {
            this.deletionProtection = deletionProtection;
            return this;
        }

        public final void setDeletionProtection(Boolean deletionProtection) {
            this.deletionProtection = deletionProtection;
        }

        public final Boolean getCopyTagsToSnapshot() {
            return copyTagsToSnapshot;
        }

        @Override
        public final Builder copyTagsToSnapshot(Boolean copyTagsToSnapshot) {
            this.copyTagsToSnapshot = copyTagsToSnapshot;
            return this;
        }

        public final void setCopyTagsToSnapshot(Boolean copyTagsToSnapshot) {
            this.copyTagsToSnapshot = copyTagsToSnapshot;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ModifyDbClusterRequest build() {
            return new ModifyDbClusterRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
