/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.neptune.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.neptune.NeptuneClient;
import software.amazon.awssdk.services.neptune.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.neptune.model.DescribeDbInstancesRequest;
import software.amazon.awssdk.services.neptune.model.DescribeDbInstancesResponse;
import software.amazon.awssdk.services.neptune.model.NeptuneRequest;
import software.amazon.awssdk.services.neptune.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultNeptuneWaiter implements NeptuneWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final NeptuneClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeDbInstancesResponse> dBInstanceAvailableWaiter;

    private final Waiter<DescribeDbInstancesResponse> dBInstanceDeletedWaiter;

    private DefaultNeptuneWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = NeptuneClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.dBInstanceAvailableWaiter = Waiter.builder(DescribeDbInstancesResponse.class)
                .acceptors(dBInstanceAvailableWaiterAcceptors())
                .overrideConfiguration(dBInstanceAvailableWaiterConfig(builder.overrideConfiguration)).build();
        this.dBInstanceDeletedWaiter = Waiter.builder(DescribeDbInstancesResponse.class)
                .acceptors(dBInstanceDeletedWaiterAcceptors())
                .overrideConfiguration(dBInstanceDeletedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeDbInstancesResponse> waitUntilDBInstanceAvailable(
            DescribeDbInstancesRequest describeDbInstancesRequest) {
        return dBInstanceAvailableWaiter.run(() -> client.describeDBInstances(applyWaitersUserAgent(describeDbInstancesRequest)));
    }

    @Override
    public WaiterResponse<DescribeDbInstancesResponse> waitUntilDBInstanceAvailable(
            DescribeDbInstancesRequest describeDbInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return dBInstanceAvailableWaiter.run(() -> client.describeDBInstances(applyWaitersUserAgent(describeDbInstancesRequest)),
                dBInstanceAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeDbInstancesResponse> waitUntilDBInstanceDeleted(
            DescribeDbInstancesRequest describeDbInstancesRequest) {
        return dBInstanceDeletedWaiter.run(() -> client.describeDBInstances(applyWaitersUserAgent(describeDbInstancesRequest)));
    }

    @Override
    public WaiterResponse<DescribeDbInstancesResponse> waitUntilDBInstanceDeleted(
            DescribeDbInstancesRequest describeDbInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return dBInstanceDeletedWaiter.run(() -> client.describeDBInstances(applyWaitersUserAgent(describeDbInstancesRequest)),
                dBInstanceDeletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeDbInstancesResponse>> dBInstanceAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDbInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=deleted) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=deleting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-restore"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=incompatible-restore) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-parameters"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=incompatible-parameters) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeDbInstancesResponse>> dBInstanceDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDbInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "DBInstanceNotFound")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "creating"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=creating) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "modifying"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=modifying) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "rebooting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=rebooting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "resetting-master-credentials"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=resetting-master-credentials) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration dBInstanceAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration dBInstanceDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static NeptuneWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends NeptuneRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements NeptuneWaiter.Builder {
        private NeptuneClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public NeptuneWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public NeptuneWaiter.Builder client(NeptuneClient client) {
            this.client = client;
            return this;
        }

        public NeptuneWaiter build() {
            return new DefaultNeptuneWaiter(this);
        }
    }
}
