/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.neptune.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the details for an Amazon Neptune DB cluster snapshot
 * </p>
 * <p>
 * This data type is used as a response element in the <a>DescribeDBClusterSnapshots</a> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DBClusterSnapshot implements SdkPojo, Serializable,
        ToCopyableBuilder<DBClusterSnapshot.Builder, DBClusterSnapshot> {
    private static final SdkField<List<String>> AVAILABILITY_ZONES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DBClusterSnapshot::availabilityZones))
            .setter(setter(Builder::availabilityZones))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZones").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("AvailabilityZone")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("AvailabilityZone").build()).build()).build()).build();

    private static final SdkField<String> DB_CLUSTER_SNAPSHOT_IDENTIFIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(DBClusterSnapshot::dbClusterSnapshotIdentifier))
            .setter(setter(Builder::dbClusterSnapshotIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterSnapshotIdentifier")
                    .build()).build();

    private static final SdkField<String> DB_CLUSTER_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBClusterSnapshot::dbClusterIdentifier)).setter(setter(Builder::dbClusterIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterIdentifier").build())
            .build();

    private static final SdkField<Instant> SNAPSHOT_CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(DBClusterSnapshot::snapshotCreateTime)).setter(setter(Builder::snapshotCreateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotCreateTime").build())
            .build();

    private static final SdkField<String> ENGINE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBClusterSnapshot::engine)).setter(setter(Builder::engine))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Engine").build()).build();

    private static final SdkField<Integer> ALLOCATED_STORAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(DBClusterSnapshot::allocatedStorage)).setter(setter(Builder::allocatedStorage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocatedStorage").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBClusterSnapshot::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(DBClusterSnapshot::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBClusterSnapshot::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<Instant> CLUSTER_CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(DBClusterSnapshot::clusterCreateTime)).setter(setter(Builder::clusterCreateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterCreateTime").build()).build();

    private static final SdkField<String> MASTER_USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBClusterSnapshot::masterUsername)).setter(setter(Builder::masterUsername))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MasterUsername").build()).build();

    private static final SdkField<String> ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBClusterSnapshot::engineVersion)).setter(setter(Builder::engineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final SdkField<String> LICENSE_MODEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBClusterSnapshot::licenseModel)).setter(setter(Builder::licenseModel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LicenseModel").build()).build();

    private static final SdkField<String> SNAPSHOT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBClusterSnapshot::snapshotType)).setter(setter(Builder::snapshotType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotType").build()).build();

    private static final SdkField<Integer> PERCENT_PROGRESS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(DBClusterSnapshot::percentProgress)).setter(setter(Builder::percentProgress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PercentProgress").build()).build();

    private static final SdkField<Boolean> STORAGE_ENCRYPTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(DBClusterSnapshot::storageEncrypted)).setter(setter(Builder::storageEncrypted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageEncrypted").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBClusterSnapshot::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<String> DB_CLUSTER_SNAPSHOT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBClusterSnapshot::dbClusterSnapshotArn)).setter(setter(Builder::dbClusterSnapshotArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterSnapshotArn").build())
            .build();

    private static final SdkField<String> SOURCE_DB_CLUSTER_SNAPSHOT_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(DBClusterSnapshot::sourceDBClusterSnapshotArn))
            .setter(setter(Builder::sourceDBClusterSnapshotArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceDBClusterSnapshotArn").build())
            .build();

    private static final SdkField<Boolean> IAM_DATABASE_AUTHENTICATION_ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(DBClusterSnapshot::iamDatabaseAuthenticationEnabled))
            .setter(setter(Builder::iamDatabaseAuthenticationEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IAMDatabaseAuthenticationEnabled")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AVAILABILITY_ZONES_FIELD,
            DB_CLUSTER_SNAPSHOT_IDENTIFIER_FIELD, DB_CLUSTER_IDENTIFIER_FIELD, SNAPSHOT_CREATE_TIME_FIELD, ENGINE_FIELD,
            ALLOCATED_STORAGE_FIELD, STATUS_FIELD, PORT_FIELD, VPC_ID_FIELD, CLUSTER_CREATE_TIME_FIELD, MASTER_USERNAME_FIELD,
            ENGINE_VERSION_FIELD, LICENSE_MODEL_FIELD, SNAPSHOT_TYPE_FIELD, PERCENT_PROGRESS_FIELD, STORAGE_ENCRYPTED_FIELD,
            KMS_KEY_ID_FIELD, DB_CLUSTER_SNAPSHOT_ARN_FIELD, SOURCE_DB_CLUSTER_SNAPSHOT_ARN_FIELD,
            IAM_DATABASE_AUTHENTICATION_ENABLED_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> availabilityZones;

    private final String dbClusterSnapshotIdentifier;

    private final String dbClusterIdentifier;

    private final Instant snapshotCreateTime;

    private final String engine;

    private final Integer allocatedStorage;

    private final String status;

    private final Integer port;

    private final String vpcId;

    private final Instant clusterCreateTime;

    private final String masterUsername;

    private final String engineVersion;

    private final String licenseModel;

    private final String snapshotType;

    private final Integer percentProgress;

    private final Boolean storageEncrypted;

    private final String kmsKeyId;

    private final String dbClusterSnapshotArn;

    private final String sourceDBClusterSnapshotArn;

    private final Boolean iamDatabaseAuthenticationEnabled;

    private DBClusterSnapshot(BuilderImpl builder) {
        this.availabilityZones = builder.availabilityZones;
        this.dbClusterSnapshotIdentifier = builder.dbClusterSnapshotIdentifier;
        this.dbClusterIdentifier = builder.dbClusterIdentifier;
        this.snapshotCreateTime = builder.snapshotCreateTime;
        this.engine = builder.engine;
        this.allocatedStorage = builder.allocatedStorage;
        this.status = builder.status;
        this.port = builder.port;
        this.vpcId = builder.vpcId;
        this.clusterCreateTime = builder.clusterCreateTime;
        this.masterUsername = builder.masterUsername;
        this.engineVersion = builder.engineVersion;
        this.licenseModel = builder.licenseModel;
        this.snapshotType = builder.snapshotType;
        this.percentProgress = builder.percentProgress;
        this.storageEncrypted = builder.storageEncrypted;
        this.kmsKeyId = builder.kmsKeyId;
        this.dbClusterSnapshotArn = builder.dbClusterSnapshotArn;
        this.sourceDBClusterSnapshotArn = builder.sourceDBClusterSnapshotArn;
        this.iamDatabaseAuthenticationEnabled = builder.iamDatabaseAuthenticationEnabled;
    }

    /**
     * <p>
     * Provides the list of EC2 Availability Zones that instances in the DB cluster snapshot can be restored in.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Provides the list of EC2 Availability Zones that instances in the DB cluster snapshot can be restored in.
     */
    public List<String> availabilityZones() {
        return availabilityZones;
    }

    /**
     * <p>
     * Specifies the identifier for the DB cluster snapshot.
     * </p>
     * 
     * @return Specifies the identifier for the DB cluster snapshot.
     */
    public String dbClusterSnapshotIdentifier() {
        return dbClusterSnapshotIdentifier;
    }

    /**
     * <p>
     * Specifies the DB cluster identifier of the DB cluster that this DB cluster snapshot was created from.
     * </p>
     * 
     * @return Specifies the DB cluster identifier of the DB cluster that this DB cluster snapshot was created from.
     */
    public String dbClusterIdentifier() {
        return dbClusterIdentifier;
    }

    /**
     * <p>
     * Provides the time when the snapshot was taken, in Universal Coordinated Time (UTC).
     * </p>
     * 
     * @return Provides the time when the snapshot was taken, in Universal Coordinated Time (UTC).
     */
    public Instant snapshotCreateTime() {
        return snapshotCreateTime;
    }

    /**
     * <p>
     * Specifies the name of the database engine.
     * </p>
     * 
     * @return Specifies the name of the database engine.
     */
    public String engine() {
        return engine;
    }

    /**
     * <p>
     * Specifies the allocated storage size in gibibytes (GiB).
     * </p>
     * 
     * @return Specifies the allocated storage size in gibibytes (GiB).
     */
    public Integer allocatedStorage() {
        return allocatedStorage;
    }

    /**
     * <p>
     * Specifies the status of this DB cluster snapshot.
     * </p>
     * 
     * @return Specifies the status of this DB cluster snapshot.
     */
    public String status() {
        return status;
    }

    /**
     * <p>
     * Specifies the port that the DB cluster was listening on at the time of the snapshot.
     * </p>
     * 
     * @return Specifies the port that the DB cluster was listening on at the time of the snapshot.
     */
    public Integer port() {
        return port;
    }

    /**
     * <p>
     * Provides the VPC ID associated with the DB cluster snapshot.
     * </p>
     * 
     * @return Provides the VPC ID associated with the DB cluster snapshot.
     */
    public String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * Specifies the time when the DB cluster was created, in Universal Coordinated Time (UTC).
     * </p>
     * 
     * @return Specifies the time when the DB cluster was created, in Universal Coordinated Time (UTC).
     */
    public Instant clusterCreateTime() {
        return clusterCreateTime;
    }

    /**
     * <p>
     * Provides the master username for the DB cluster snapshot.
     * </p>
     * 
     * @return Provides the master username for the DB cluster snapshot.
     */
    public String masterUsername() {
        return masterUsername;
    }

    /**
     * <p>
     * Provides the version of the database engine for this DB cluster snapshot.
     * </p>
     * 
     * @return Provides the version of the database engine for this DB cluster snapshot.
     */
    public String engineVersion() {
        return engineVersion;
    }

    /**
     * <p>
     * Provides the license model information for this DB cluster snapshot.
     * </p>
     * 
     * @return Provides the license model information for this DB cluster snapshot.
     */
    public String licenseModel() {
        return licenseModel;
    }

    /**
     * <p>
     * Provides the type of the DB cluster snapshot.
     * </p>
     * 
     * @return Provides the type of the DB cluster snapshot.
     */
    public String snapshotType() {
        return snapshotType;
    }

    /**
     * <p>
     * Specifies the percentage of the estimated data that has been transferred.
     * </p>
     * 
     * @return Specifies the percentage of the estimated data that has been transferred.
     */
    public Integer percentProgress() {
        return percentProgress;
    }

    /**
     * <p>
     * Specifies whether the DB cluster snapshot is encrypted.
     * </p>
     * 
     * @return Specifies whether the DB cluster snapshot is encrypted.
     */
    public Boolean storageEncrypted() {
        return storageEncrypted;
    }

    /**
     * <p>
     * If <code>StorageEncrypted</code> is true, the AWS KMS key identifier for the encrypted DB cluster snapshot.
     * </p>
     * 
     * @return If <code>StorageEncrypted</code> is true, the AWS KMS key identifier for the encrypted DB cluster
     *         snapshot.
     */
    public String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the DB cluster snapshot.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the DB cluster snapshot.
     */
    public String dbClusterSnapshotArn() {
        return dbClusterSnapshotArn;
    }

    /**
     * <p>
     * If the DB cluster snapshot was copied from a source DB cluster snapshot, the Amazon Resource Name (ARN) for the
     * source DB cluster snapshot, otherwise, a null value.
     * </p>
     * 
     * @return If the DB cluster snapshot was copied from a source DB cluster snapshot, the Amazon Resource Name (ARN)
     *         for the source DB cluster snapshot, otherwise, a null value.
     */
    public String sourceDBClusterSnapshotArn() {
        return sourceDBClusterSnapshotArn;
    }

    /**
     * <p>
     * True if mapping of AWS Identity and Access Management (IAM) accounts to database accounts is enabled, and
     * otherwise false.
     * </p>
     * 
     * @return True if mapping of AWS Identity and Access Management (IAM) accounts to database accounts is enabled, and
     *         otherwise false.
     */
    public Boolean iamDatabaseAuthenticationEnabled() {
        return iamDatabaseAuthenticationEnabled;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZones());
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterSnapshotIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotCreateTime());
        hashCode = 31 * hashCode + Objects.hashCode(engine());
        hashCode = 31 * hashCode + Objects.hashCode(allocatedStorage());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(clusterCreateTime());
        hashCode = 31 * hashCode + Objects.hashCode(masterUsername());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(licenseModel());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotType());
        hashCode = 31 * hashCode + Objects.hashCode(percentProgress());
        hashCode = 31 * hashCode + Objects.hashCode(storageEncrypted());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterSnapshotArn());
        hashCode = 31 * hashCode + Objects.hashCode(sourceDBClusterSnapshotArn());
        hashCode = 31 * hashCode + Objects.hashCode(iamDatabaseAuthenticationEnabled());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DBClusterSnapshot)) {
            return false;
        }
        DBClusterSnapshot other = (DBClusterSnapshot) obj;
        return Objects.equals(availabilityZones(), other.availabilityZones())
                && Objects.equals(dbClusterSnapshotIdentifier(), other.dbClusterSnapshotIdentifier())
                && Objects.equals(dbClusterIdentifier(), other.dbClusterIdentifier())
                && Objects.equals(snapshotCreateTime(), other.snapshotCreateTime()) && Objects.equals(engine(), other.engine())
                && Objects.equals(allocatedStorage(), other.allocatedStorage()) && Objects.equals(status(), other.status())
                && Objects.equals(port(), other.port()) && Objects.equals(vpcId(), other.vpcId())
                && Objects.equals(clusterCreateTime(), other.clusterCreateTime())
                && Objects.equals(masterUsername(), other.masterUsername())
                && Objects.equals(engineVersion(), other.engineVersion()) && Objects.equals(licenseModel(), other.licenseModel())
                && Objects.equals(snapshotType(), other.snapshotType())
                && Objects.equals(percentProgress(), other.percentProgress())
                && Objects.equals(storageEncrypted(), other.storageEncrypted()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(dbClusterSnapshotArn(), other.dbClusterSnapshotArn())
                && Objects.equals(sourceDBClusterSnapshotArn(), other.sourceDBClusterSnapshotArn())
                && Objects.equals(iamDatabaseAuthenticationEnabled(), other.iamDatabaseAuthenticationEnabled());
    }

    @Override
    public String toString() {
        return ToString.builder("DBClusterSnapshot").add("AvailabilityZones", availabilityZones())
                .add("DBClusterSnapshotIdentifier", dbClusterSnapshotIdentifier())
                .add("DBClusterIdentifier", dbClusterIdentifier()).add("SnapshotCreateTime", snapshotCreateTime())
                .add("Engine", engine()).add("AllocatedStorage", allocatedStorage()).add("Status", status()).add("Port", port())
                .add("VpcId", vpcId()).add("ClusterCreateTime", clusterCreateTime()).add("MasterUsername", masterUsername())
                .add("EngineVersion", engineVersion()).add("LicenseModel", licenseModel()).add("SnapshotType", snapshotType())
                .add("PercentProgress", percentProgress()).add("StorageEncrypted", storageEncrypted())
                .add("KmsKeyId", kmsKeyId()).add("DBClusterSnapshotArn", dbClusterSnapshotArn())
                .add("SourceDBClusterSnapshotArn", sourceDBClusterSnapshotArn())
                .add("IAMDatabaseAuthenticationEnabled", iamDatabaseAuthenticationEnabled()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AvailabilityZones":
            return Optional.ofNullable(clazz.cast(availabilityZones()));
        case "DBClusterSnapshotIdentifier":
            return Optional.ofNullable(clazz.cast(dbClusterSnapshotIdentifier()));
        case "DBClusterIdentifier":
            return Optional.ofNullable(clazz.cast(dbClusterIdentifier()));
        case "SnapshotCreateTime":
            return Optional.ofNullable(clazz.cast(snapshotCreateTime()));
        case "Engine":
            return Optional.ofNullable(clazz.cast(engine()));
        case "AllocatedStorage":
            return Optional.ofNullable(clazz.cast(allocatedStorage()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "ClusterCreateTime":
            return Optional.ofNullable(clazz.cast(clusterCreateTime()));
        case "MasterUsername":
            return Optional.ofNullable(clazz.cast(masterUsername()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "LicenseModel":
            return Optional.ofNullable(clazz.cast(licenseModel()));
        case "SnapshotType":
            return Optional.ofNullable(clazz.cast(snapshotType()));
        case "PercentProgress":
            return Optional.ofNullable(clazz.cast(percentProgress()));
        case "StorageEncrypted":
            return Optional.ofNullable(clazz.cast(storageEncrypted()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "DBClusterSnapshotArn":
            return Optional.ofNullable(clazz.cast(dbClusterSnapshotArn()));
        case "SourceDBClusterSnapshotArn":
            return Optional.ofNullable(clazz.cast(sourceDBClusterSnapshotArn()));
        case "IAMDatabaseAuthenticationEnabled":
            return Optional.ofNullable(clazz.cast(iamDatabaseAuthenticationEnabled()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DBClusterSnapshot, T> g) {
        return obj -> g.apply((DBClusterSnapshot) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DBClusterSnapshot> {
        /**
         * <p>
         * Provides the list of EC2 Availability Zones that instances in the DB cluster snapshot can be restored in.
         * </p>
         * 
         * @param availabilityZones
         *        Provides the list of EC2 Availability Zones that instances in the DB cluster snapshot can be restored
         *        in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(Collection<String> availabilityZones);

        /**
         * <p>
         * Provides the list of EC2 Availability Zones that instances in the DB cluster snapshot can be restored in.
         * </p>
         * 
         * @param availabilityZones
         *        Provides the list of EC2 Availability Zones that instances in the DB cluster snapshot can be restored
         *        in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(String... availabilityZones);

        /**
         * <p>
         * Specifies the identifier for the DB cluster snapshot.
         * </p>
         * 
         * @param dbClusterSnapshotIdentifier
         *        Specifies the identifier for the DB cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterSnapshotIdentifier(String dbClusterSnapshotIdentifier);

        /**
         * <p>
         * Specifies the DB cluster identifier of the DB cluster that this DB cluster snapshot was created from.
         * </p>
         * 
         * @param dbClusterIdentifier
         *        Specifies the DB cluster identifier of the DB cluster that this DB cluster snapshot was created from.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterIdentifier(String dbClusterIdentifier);

        /**
         * <p>
         * Provides the time when the snapshot was taken, in Universal Coordinated Time (UTC).
         * </p>
         * 
         * @param snapshotCreateTime
         *        Provides the time when the snapshot was taken, in Universal Coordinated Time (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotCreateTime(Instant snapshotCreateTime);

        /**
         * <p>
         * Specifies the name of the database engine.
         * </p>
         * 
         * @param engine
         *        Specifies the name of the database engine.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engine(String engine);

        /**
         * <p>
         * Specifies the allocated storage size in gibibytes (GiB).
         * </p>
         * 
         * @param allocatedStorage
         *        Specifies the allocated storage size in gibibytes (GiB).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocatedStorage(Integer allocatedStorage);

        /**
         * <p>
         * Specifies the status of this DB cluster snapshot.
         * </p>
         * 
         * @param status
         *        Specifies the status of this DB cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * Specifies the port that the DB cluster was listening on at the time of the snapshot.
         * </p>
         * 
         * @param port
         *        Specifies the port that the DB cluster was listening on at the time of the snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * Provides the VPC ID associated with the DB cluster snapshot.
         * </p>
         * 
         * @param vpcId
         *        Provides the VPC ID associated with the DB cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * Specifies the time when the DB cluster was created, in Universal Coordinated Time (UTC).
         * </p>
         * 
         * @param clusterCreateTime
         *        Specifies the time when the DB cluster was created, in Universal Coordinated Time (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterCreateTime(Instant clusterCreateTime);

        /**
         * <p>
         * Provides the master username for the DB cluster snapshot.
         * </p>
         * 
         * @param masterUsername
         *        Provides the master username for the DB cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterUsername(String masterUsername);

        /**
         * <p>
         * Provides the version of the database engine for this DB cluster snapshot.
         * </p>
         * 
         * @param engineVersion
         *        Provides the version of the database engine for this DB cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(String engineVersion);

        /**
         * <p>
         * Provides the license model information for this DB cluster snapshot.
         * </p>
         * 
         * @param licenseModel
         *        Provides the license model information for this DB cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseModel(String licenseModel);

        /**
         * <p>
         * Provides the type of the DB cluster snapshot.
         * </p>
         * 
         * @param snapshotType
         *        Provides the type of the DB cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotType(String snapshotType);

        /**
         * <p>
         * Specifies the percentage of the estimated data that has been transferred.
         * </p>
         * 
         * @param percentProgress
         *        Specifies the percentage of the estimated data that has been transferred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder percentProgress(Integer percentProgress);

        /**
         * <p>
         * Specifies whether the DB cluster snapshot is encrypted.
         * </p>
         * 
         * @param storageEncrypted
         *        Specifies whether the DB cluster snapshot is encrypted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageEncrypted(Boolean storageEncrypted);

        /**
         * <p>
         * If <code>StorageEncrypted</code> is true, the AWS KMS key identifier for the encrypted DB cluster snapshot.
         * </p>
         * 
         * @param kmsKeyId
         *        If <code>StorageEncrypted</code> is true, the AWS KMS key identifier for the encrypted DB cluster
         *        snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the DB cluster snapshot.
         * </p>
         * 
         * @param dbClusterSnapshotArn
         *        The Amazon Resource Name (ARN) for the DB cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterSnapshotArn(String dbClusterSnapshotArn);

        /**
         * <p>
         * If the DB cluster snapshot was copied from a source DB cluster snapshot, the Amazon Resource Name (ARN) for
         * the source DB cluster snapshot, otherwise, a null value.
         * </p>
         * 
         * @param sourceDBClusterSnapshotArn
         *        If the DB cluster snapshot was copied from a source DB cluster snapshot, the Amazon Resource Name
         *        (ARN) for the source DB cluster snapshot, otherwise, a null value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceDBClusterSnapshotArn(String sourceDBClusterSnapshotArn);

        /**
         * <p>
         * True if mapping of AWS Identity and Access Management (IAM) accounts to database accounts is enabled, and
         * otherwise false.
         * </p>
         * 
         * @param iamDatabaseAuthenticationEnabled
         *        True if mapping of AWS Identity and Access Management (IAM) accounts to database accounts is enabled,
         *        and otherwise false.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamDatabaseAuthenticationEnabled(Boolean iamDatabaseAuthenticationEnabled);
    }

    static final class BuilderImpl implements Builder {
        private List<String> availabilityZones = DefaultSdkAutoConstructList.getInstance();

        private String dbClusterSnapshotIdentifier;

        private String dbClusterIdentifier;

        private Instant snapshotCreateTime;

        private String engine;

        private Integer allocatedStorage;

        private String status;

        private Integer port;

        private String vpcId;

        private Instant clusterCreateTime;

        private String masterUsername;

        private String engineVersion;

        private String licenseModel;

        private String snapshotType;

        private Integer percentProgress;

        private Boolean storageEncrypted;

        private String kmsKeyId;

        private String dbClusterSnapshotArn;

        private String sourceDBClusterSnapshotArn;

        private Boolean iamDatabaseAuthenticationEnabled;

        private BuilderImpl() {
        }

        private BuilderImpl(DBClusterSnapshot model) {
            availabilityZones(model.availabilityZones);
            dbClusterSnapshotIdentifier(model.dbClusterSnapshotIdentifier);
            dbClusterIdentifier(model.dbClusterIdentifier);
            snapshotCreateTime(model.snapshotCreateTime);
            engine(model.engine);
            allocatedStorage(model.allocatedStorage);
            status(model.status);
            port(model.port);
            vpcId(model.vpcId);
            clusterCreateTime(model.clusterCreateTime);
            masterUsername(model.masterUsername);
            engineVersion(model.engineVersion);
            licenseModel(model.licenseModel);
            snapshotType(model.snapshotType);
            percentProgress(model.percentProgress);
            storageEncrypted(model.storageEncrypted);
            kmsKeyId(model.kmsKeyId);
            dbClusterSnapshotArn(model.dbClusterSnapshotArn);
            sourceDBClusterSnapshotArn(model.sourceDBClusterSnapshotArn);
            iamDatabaseAuthenticationEnabled(model.iamDatabaseAuthenticationEnabled);
        }

        public final Collection<String> getAvailabilityZones() {
            return availabilityZones;
        }

        @Override
        public final Builder availabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZonesCopier.copy(availabilityZones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availabilityZones(String... availabilityZones) {
            availabilityZones(Arrays.asList(availabilityZones));
            return this;
        }

        public final void setAvailabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZonesCopier.copy(availabilityZones);
        }

        public final String getDbClusterSnapshotIdentifier() {
            return dbClusterSnapshotIdentifier;
        }

        @Override
        public final Builder dbClusterSnapshotIdentifier(String dbClusterSnapshotIdentifier) {
            this.dbClusterSnapshotIdentifier = dbClusterSnapshotIdentifier;
            return this;
        }

        public final void setDbClusterSnapshotIdentifier(String dbClusterSnapshotIdentifier) {
            this.dbClusterSnapshotIdentifier = dbClusterSnapshotIdentifier;
        }

        public final String getDbClusterIdentifier() {
            return dbClusterIdentifier;
        }

        @Override
        public final Builder dbClusterIdentifier(String dbClusterIdentifier) {
            this.dbClusterIdentifier = dbClusterIdentifier;
            return this;
        }

        public final void setDbClusterIdentifier(String dbClusterIdentifier) {
            this.dbClusterIdentifier = dbClusterIdentifier;
        }

        public final Instant getSnapshotCreateTime() {
            return snapshotCreateTime;
        }

        @Override
        public final Builder snapshotCreateTime(Instant snapshotCreateTime) {
            this.snapshotCreateTime = snapshotCreateTime;
            return this;
        }

        public final void setSnapshotCreateTime(Instant snapshotCreateTime) {
            this.snapshotCreateTime = snapshotCreateTime;
        }

        public final String getEngine() {
            return engine;
        }

        @Override
        public final Builder engine(String engine) {
            this.engine = engine;
            return this;
        }

        public final void setEngine(String engine) {
            this.engine = engine;
        }

        public final Integer getAllocatedStorage() {
            return allocatedStorage;
        }

        @Override
        public final Builder allocatedStorage(Integer allocatedStorage) {
            this.allocatedStorage = allocatedStorage;
            return this;
        }

        public final void setAllocatedStorage(Integer allocatedStorage) {
            this.allocatedStorage = allocatedStorage;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Integer getPort() {
            return port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        public final String getVpcId() {
            return vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        public final Instant getClusterCreateTime() {
            return clusterCreateTime;
        }

        @Override
        public final Builder clusterCreateTime(Instant clusterCreateTime) {
            this.clusterCreateTime = clusterCreateTime;
            return this;
        }

        public final void setClusterCreateTime(Instant clusterCreateTime) {
            this.clusterCreateTime = clusterCreateTime;
        }

        public final String getMasterUsername() {
            return masterUsername;
        }

        @Override
        public final Builder masterUsername(String masterUsername) {
            this.masterUsername = masterUsername;
            return this;
        }

        public final void setMasterUsername(String masterUsername) {
            this.masterUsername = masterUsername;
        }

        public final String getEngineVersion() {
            return engineVersion;
        }

        @Override
        public final Builder engineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        public final String getLicenseModel() {
            return licenseModel;
        }

        @Override
        public final Builder licenseModel(String licenseModel) {
            this.licenseModel = licenseModel;
            return this;
        }

        public final void setLicenseModel(String licenseModel) {
            this.licenseModel = licenseModel;
        }

        public final String getSnapshotType() {
            return snapshotType;
        }

        @Override
        public final Builder snapshotType(String snapshotType) {
            this.snapshotType = snapshotType;
            return this;
        }

        public final void setSnapshotType(String snapshotType) {
            this.snapshotType = snapshotType;
        }

        public final Integer getPercentProgress() {
            return percentProgress;
        }

        @Override
        public final Builder percentProgress(Integer percentProgress) {
            this.percentProgress = percentProgress;
            return this;
        }

        public final void setPercentProgress(Integer percentProgress) {
            this.percentProgress = percentProgress;
        }

        public final Boolean getStorageEncrypted() {
            return storageEncrypted;
        }

        @Override
        public final Builder storageEncrypted(Boolean storageEncrypted) {
            this.storageEncrypted = storageEncrypted;
            return this;
        }

        public final void setStorageEncrypted(Boolean storageEncrypted) {
            this.storageEncrypted = storageEncrypted;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        public final String getDbClusterSnapshotArn() {
            return dbClusterSnapshotArn;
        }

        @Override
        public final Builder dbClusterSnapshotArn(String dbClusterSnapshotArn) {
            this.dbClusterSnapshotArn = dbClusterSnapshotArn;
            return this;
        }

        public final void setDbClusterSnapshotArn(String dbClusterSnapshotArn) {
            this.dbClusterSnapshotArn = dbClusterSnapshotArn;
        }

        public final String getSourceDBClusterSnapshotArn() {
            return sourceDBClusterSnapshotArn;
        }

        @Override
        public final Builder sourceDBClusterSnapshotArn(String sourceDBClusterSnapshotArn) {
            this.sourceDBClusterSnapshotArn = sourceDBClusterSnapshotArn;
            return this;
        }

        public final void setSourceDBClusterSnapshotArn(String sourceDBClusterSnapshotArn) {
            this.sourceDBClusterSnapshotArn = sourceDBClusterSnapshotArn;
        }

        public final Boolean getIamDatabaseAuthenticationEnabled() {
            return iamDatabaseAuthenticationEnabled;
        }

        @Override
        public final Builder iamDatabaseAuthenticationEnabled(Boolean iamDatabaseAuthenticationEnabled) {
            this.iamDatabaseAuthenticationEnabled = iamDatabaseAuthenticationEnabled;
            return this;
        }

        public final void setIamDatabaseAuthenticationEnabled(Boolean iamDatabaseAuthenticationEnabled) {
            this.iamDatabaseAuthenticationEnabled = iamDatabaseAuthenticationEnabled;
        }

        @Override
        public DBClusterSnapshot build() {
            return new DBClusterSnapshot(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
