/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.neptune.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedItemsIterable;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.neptune.NeptuneClient;
import software.amazon.awssdk.services.neptune.model.DescribeEngineDefaultParametersRequest;
import software.amazon.awssdk.services.neptune.model.DescribeEngineDefaultParametersResponse;
import software.amazon.awssdk.services.neptune.model.Parameter;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.neptune.NeptuneClient#describeEngineDefaultParametersPaginator(software.amazon.awssdk.services.neptune.model.DescribeEngineDefaultParametersRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.neptune.model.DescribeEngineDefaultParametersResponse} that can be used to
 * iterate through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.neptune.paginators.DescribeEngineDefaultParametersIterable responses = client.describeEngineDefaultParametersPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.neptune.paginators.DescribeEngineDefaultParametersIterable responses = client
 *             .describeEngineDefaultParametersPaginator(request);
 *     for (software.amazon.awssdk.services.neptune.model.DescribeEngineDefaultParametersResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.neptune.paginators.DescribeEngineDefaultParametersIterable responses = client.describeEngineDefaultParametersPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #describeEngineDefaultParameters(software.amazon.awssdk.services.neptune.model.DescribeEngineDefaultParametersRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class DescribeEngineDefaultParametersIterable implements SdkIterable<DescribeEngineDefaultParametersResponse> {
    private final NeptuneClient client;

    private final DescribeEngineDefaultParametersRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public DescribeEngineDefaultParametersIterable(NeptuneClient client, DescribeEngineDefaultParametersRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new DescribeEngineDefaultParametersResponseFetcher();
    }

    @Override
    public Iterator<DescribeEngineDefaultParametersResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    /**
     * Returns an iterable to iterate through the paginated {@link DescribeEngineDefaultParametersResponse#parameters()}
     * member. The returned iterable is used to iterate through the results across all response pages and not a single
     * page.
     *
     * This method is useful if you are interested in iterating over the paginated member in the response pages instead
     * of the top level pages. Similar to iteration over pages, this method internally makes service calls to get the
     * next list of results until the iteration stops or there are no more results.
     */
    public final SdkIterable<Parameter> parameters() {
        Function<DescribeEngineDefaultParametersResponse, Iterator<Parameter>> getIterator = response -> {
            if (response != null && response.engineDefaults() != null && response.engineDefaults().parameters() != null) {
                return response.engineDefaults().parameters().iterator();
            }
            return Collections.emptyIterator();
        };
        return PaginatedItemsIterable.builder().pagesIterable(this).itemIteratorFunction(getIterator).build();
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link DescribeEngineDefaultParametersIterable} that can be used to
     * retrieve the consecutive pages that follows the input page.
     * </p>
     */
    private final DescribeEngineDefaultParametersIterable resume(DescribeEngineDefaultParametersResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new DescribeEngineDefaultParametersIterable(client, firstRequest.toBuilder()
                    .marker(lastSuccessfulPage.engineDefaults().marker()).build());
        }
        return new DescribeEngineDefaultParametersIterable(client, firstRequest) {
            @Override
            public Iterator<DescribeEngineDefaultParametersResponse> iterator() {
                return Collections.emptyIterator();
            }
        };
    }

    private class DescribeEngineDefaultParametersResponseFetcher implements
            SyncPageFetcher<DescribeEngineDefaultParametersResponse> {
        @Override
        public boolean hasNextPage(DescribeEngineDefaultParametersResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.engineDefaults().marker());
        }

        @Override
        public DescribeEngineDefaultParametersResponse nextPage(DescribeEngineDefaultParametersResponse previousPage) {
            if (previousPage == null) {
                return client.describeEngineDefaultParameters(firstRequest);
            }
            return client.describeEngineDefaultParameters(firstRequest.toBuilder().marker(previousPage.engineDefaults().marker())
                    .build());
        }
    }
}
