/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.neptunedata.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains statistics information. The DFE engine uses information about the data in your Neptune graph to make
 * effective trade-offs when planning query execution. This information takes the form of statistics that include
 * so-called characteristic sets and predicate statistics that can guide query planning. See <a
 * href="https://docs.aws.amazon.com/neptune/latest/userguide/neptune-dfe-statistics.html">Managing statistics for the
 * Neptune DFE to use</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Statistics implements SdkPojo, Serializable, ToCopyableBuilder<Statistics.Builder, Statistics> {
    private static final SdkField<Boolean> AUTO_COMPUTE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("autoCompute").getter(getter(Statistics::autoCompute)).setter(setter(Builder::autoCompute))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autoCompute").build()).build();

    private static final SdkField<Boolean> ACTIVE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("active").getter(getter(Statistics::active)).setter(setter(Builder::active))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("active").build()).build();

    private static final SdkField<String> STATISTICS_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statisticsId").getter(getter(Statistics::statisticsId)).setter(setter(Builder::statisticsId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statisticsId").build()).build();

    private static final SdkField<Instant> DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("date")
            .getter(getter(Statistics::date))
            .setter(setter(Builder::date))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("date").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> NOTE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("note")
            .getter(getter(Statistics::note)).setter(setter(Builder::note))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("note").build()).build();

    private static final SdkField<StatisticsSummary> SIGNATURE_INFO_FIELD = SdkField
            .<StatisticsSummary> builder(MarshallingType.SDK_POJO).memberName("signatureInfo")
            .getter(getter(Statistics::signatureInfo)).setter(setter(Builder::signatureInfo))
            .constructor(StatisticsSummary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("signatureInfo").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AUTO_COMPUTE_FIELD,
            ACTIVE_FIELD, STATISTICS_ID_FIELD, DATE_FIELD, NOTE_FIELD, SIGNATURE_INFO_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean autoCompute;

    private final Boolean active;

    private final String statisticsId;

    private final Instant date;

    private final String note;

    private final StatisticsSummary signatureInfo;

    private Statistics(BuilderImpl builder) {
        this.autoCompute = builder.autoCompute;
        this.active = builder.active;
        this.statisticsId = builder.statisticsId;
        this.date = builder.date;
        this.note = builder.note;
        this.signatureInfo = builder.signatureInfo;
    }

    /**
     * <p>
     * Indicates whether or not automatic statistics generation is enabled.
     * </p>
     * 
     * @return Indicates whether or not automatic statistics generation is enabled.
     */
    public final Boolean autoCompute() {
        return autoCompute;
    }

    /**
     * <p>
     * Indicates whether or not DFE statistics generation is enabled at all.
     * </p>
     * 
     * @return Indicates whether or not DFE statistics generation is enabled at all.
     */
    public final Boolean active() {
        return active;
    }

    /**
     * <p>
     * Reports the ID of the current statistics generation run. A value of -1 indicates that no statistics have been
     * generated.
     * </p>
     * 
     * @return Reports the ID of the current statistics generation run. A value of -1 indicates that no statistics have
     *         been generated.
     */
    public final String statisticsId() {
        return statisticsId;
    }

    /**
     * <p>
     * The UTC time at which DFE statistics have most recently been generated.
     * </p>
     * 
     * @return The UTC time at which DFE statistics have most recently been generated.
     */
    public final Instant date() {
        return date;
    }

    /**
     * <p>
     * A note about problems in the case where statistics are invalid.
     * </p>
     * 
     * @return A note about problems in the case where statistics are invalid.
     */
    public final String note() {
        return note;
    }

    /**
     * <p>
     * A StatisticsSummary structure that contains:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>signatureCount</code> - The total number of signatures across all characteristic sets.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>instanceCount</code> - The total number of characteristic-set instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>predicateCount</code> - The total number of unique predicates.
     * </p>
     * </li>
     * </ul>
     * 
     * @return A StatisticsSummary structure that contains:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>signatureCount</code> - The total number of signatures across all characteristic sets.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>instanceCount</code> - The total number of characteristic-set instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>predicateCount</code> - The total number of unique predicates.
     *         </p>
     *         </li>
     */
    public final StatisticsSummary signatureInfo() {
        return signatureInfo;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(autoCompute());
        hashCode = 31 * hashCode + Objects.hashCode(active());
        hashCode = 31 * hashCode + Objects.hashCode(statisticsId());
        hashCode = 31 * hashCode + Objects.hashCode(date());
        hashCode = 31 * hashCode + Objects.hashCode(note());
        hashCode = 31 * hashCode + Objects.hashCode(signatureInfo());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Statistics)) {
            return false;
        }
        Statistics other = (Statistics) obj;
        return Objects.equals(autoCompute(), other.autoCompute()) && Objects.equals(active(), other.active())
                && Objects.equals(statisticsId(), other.statisticsId()) && Objects.equals(date(), other.date())
                && Objects.equals(note(), other.note()) && Objects.equals(signatureInfo(), other.signatureInfo());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Statistics").add("AutoCompute", autoCompute()).add("Active", active())
                .add("StatisticsId", statisticsId()).add("Date", date()).add("Note", note())
                .add("SignatureInfo", signatureInfo()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "autoCompute":
            return Optional.ofNullable(clazz.cast(autoCompute()));
        case "active":
            return Optional.ofNullable(clazz.cast(active()));
        case "statisticsId":
            return Optional.ofNullable(clazz.cast(statisticsId()));
        case "date":
            return Optional.ofNullable(clazz.cast(date()));
        case "note":
            return Optional.ofNullable(clazz.cast(note()));
        case "signatureInfo":
            return Optional.ofNullable(clazz.cast(signatureInfo()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Statistics, T> g) {
        return obj -> g.apply((Statistics) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Statistics> {
        /**
         * <p>
         * Indicates whether or not automatic statistics generation is enabled.
         * </p>
         * 
         * @param autoCompute
         *        Indicates whether or not automatic statistics generation is enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoCompute(Boolean autoCompute);

        /**
         * <p>
         * Indicates whether or not DFE statistics generation is enabled at all.
         * </p>
         * 
         * @param active
         *        Indicates whether or not DFE statistics generation is enabled at all.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder active(Boolean active);

        /**
         * <p>
         * Reports the ID of the current statistics generation run. A value of -1 indicates that no statistics have been
         * generated.
         * </p>
         * 
         * @param statisticsId
         *        Reports the ID of the current statistics generation run. A value of -1 indicates that no statistics
         *        have been generated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statisticsId(String statisticsId);

        /**
         * <p>
         * The UTC time at which DFE statistics have most recently been generated.
         * </p>
         * 
         * @param date
         *        The UTC time at which DFE statistics have most recently been generated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder date(Instant date);

        /**
         * <p>
         * A note about problems in the case where statistics are invalid.
         * </p>
         * 
         * @param note
         *        A note about problems in the case where statistics are invalid.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder note(String note);

        /**
         * <p>
         * A StatisticsSummary structure that contains:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>signatureCount</code> - The total number of signatures across all characteristic sets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instanceCount</code> - The total number of characteristic-set instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>predicateCount</code> - The total number of unique predicates.
         * </p>
         * </li>
         * </ul>
         * 
         * @param signatureInfo
         *        A StatisticsSummary structure that contains:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>signatureCount</code> - The total number of signatures across all characteristic sets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instanceCount</code> - The total number of characteristic-set instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>predicateCount</code> - The total number of unique predicates.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signatureInfo(StatisticsSummary signatureInfo);

        /**
         * <p>
         * A StatisticsSummary structure that contains:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>signatureCount</code> - The total number of signatures across all characteristic sets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instanceCount</code> - The total number of characteristic-set instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>predicateCount</code> - The total number of unique predicates.
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the {@link StatisticsSummary.Builder} avoiding the
         * need to create one manually via {@link StatisticsSummary#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link StatisticsSummary.Builder#build()} is called immediately and its
         * result is passed to {@link #signatureInfo(StatisticsSummary)}.
         * 
         * @param signatureInfo
         *        a consumer that will call methods on {@link StatisticsSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #signatureInfo(StatisticsSummary)
         */
        default Builder signatureInfo(Consumer<StatisticsSummary.Builder> signatureInfo) {
            return signatureInfo(StatisticsSummary.builder().applyMutation(signatureInfo).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Boolean autoCompute;

        private Boolean active;

        private String statisticsId;

        private Instant date;

        private String note;

        private StatisticsSummary signatureInfo;

        private BuilderImpl() {
        }

        private BuilderImpl(Statistics model) {
            autoCompute(model.autoCompute);
            active(model.active);
            statisticsId(model.statisticsId);
            date(model.date);
            note(model.note);
            signatureInfo(model.signatureInfo);
        }

        public final Boolean getAutoCompute() {
            return autoCompute;
        }

        public final void setAutoCompute(Boolean autoCompute) {
            this.autoCompute = autoCompute;
        }

        @Override
        public final Builder autoCompute(Boolean autoCompute) {
            this.autoCompute = autoCompute;
            return this;
        }

        public final Boolean getActive() {
            return active;
        }

        public final void setActive(Boolean active) {
            this.active = active;
        }

        @Override
        public final Builder active(Boolean active) {
            this.active = active;
            return this;
        }

        public final String getStatisticsId() {
            return statisticsId;
        }

        public final void setStatisticsId(String statisticsId) {
            this.statisticsId = statisticsId;
        }

        @Override
        public final Builder statisticsId(String statisticsId) {
            this.statisticsId = statisticsId;
            return this;
        }

        public final Instant getDate() {
            return date;
        }

        public final void setDate(Instant date) {
            this.date = date;
        }

        @Override
        public final Builder date(Instant date) {
            this.date = date;
            return this;
        }

        public final String getNote() {
            return note;
        }

        public final void setNote(String note) {
            this.note = note;
        }

        @Override
        public final Builder note(String note) {
            this.note = note;
            return this;
        }

        public final StatisticsSummary.Builder getSignatureInfo() {
            return signatureInfo != null ? signatureInfo.toBuilder() : null;
        }

        public final void setSignatureInfo(StatisticsSummary.BuilderImpl signatureInfo) {
            this.signatureInfo = signatureInfo != null ? signatureInfo.build() : null;
        }

        @Override
        public final Builder signatureInfo(StatisticsSummary signatureInfo) {
            this.signatureInfo = signatureInfo;
            return this;
        }

        @Override
        public Statistics build() {
            return new Statistics(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
