/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.neptunegraph;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.neptunegraph.internal.NeptuneGraphServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.neptunegraph.model.CancelImportTaskRequest;
import software.amazon.awssdk.services.neptunegraph.model.CancelImportTaskResponse;
import software.amazon.awssdk.services.neptunegraph.model.ConflictException;
import software.amazon.awssdk.services.neptunegraph.model.CreateGraphRequest;
import software.amazon.awssdk.services.neptunegraph.model.CreateGraphResponse;
import software.amazon.awssdk.services.neptunegraph.model.CreateGraphSnapshotRequest;
import software.amazon.awssdk.services.neptunegraph.model.CreateGraphSnapshotResponse;
import software.amazon.awssdk.services.neptunegraph.model.CreateGraphUsingImportTaskRequest;
import software.amazon.awssdk.services.neptunegraph.model.CreateGraphUsingImportTaskResponse;
import software.amazon.awssdk.services.neptunegraph.model.CreatePrivateGraphEndpointRequest;
import software.amazon.awssdk.services.neptunegraph.model.CreatePrivateGraphEndpointResponse;
import software.amazon.awssdk.services.neptunegraph.model.DeleteGraphRequest;
import software.amazon.awssdk.services.neptunegraph.model.DeleteGraphResponse;
import software.amazon.awssdk.services.neptunegraph.model.DeleteGraphSnapshotRequest;
import software.amazon.awssdk.services.neptunegraph.model.DeleteGraphSnapshotResponse;
import software.amazon.awssdk.services.neptunegraph.model.DeletePrivateGraphEndpointRequest;
import software.amazon.awssdk.services.neptunegraph.model.DeletePrivateGraphEndpointResponse;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphResponse;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphSnapshotRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphSnapshotResponse;
import software.amazon.awssdk.services.neptunegraph.model.GetImportTaskRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetImportTaskResponse;
import software.amazon.awssdk.services.neptunegraph.model.GetPrivateGraphEndpointRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetPrivateGraphEndpointResponse;
import software.amazon.awssdk.services.neptunegraph.model.InternalServerException;
import software.amazon.awssdk.services.neptunegraph.model.ListGraphSnapshotsRequest;
import software.amazon.awssdk.services.neptunegraph.model.ListGraphSnapshotsResponse;
import software.amazon.awssdk.services.neptunegraph.model.ListGraphsRequest;
import software.amazon.awssdk.services.neptunegraph.model.ListGraphsResponse;
import software.amazon.awssdk.services.neptunegraph.model.ListImportTasksRequest;
import software.amazon.awssdk.services.neptunegraph.model.ListImportTasksResponse;
import software.amazon.awssdk.services.neptunegraph.model.ListPrivateGraphEndpointsRequest;
import software.amazon.awssdk.services.neptunegraph.model.ListPrivateGraphEndpointsResponse;
import software.amazon.awssdk.services.neptunegraph.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.neptunegraph.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.neptunegraph.model.NeptuneGraphException;
import software.amazon.awssdk.services.neptunegraph.model.ResetGraphRequest;
import software.amazon.awssdk.services.neptunegraph.model.ResetGraphResponse;
import software.amazon.awssdk.services.neptunegraph.model.ResourceNotFoundException;
import software.amazon.awssdk.services.neptunegraph.model.RestoreGraphFromSnapshotRequest;
import software.amazon.awssdk.services.neptunegraph.model.RestoreGraphFromSnapshotResponse;
import software.amazon.awssdk.services.neptunegraph.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.neptunegraph.model.TagResourceRequest;
import software.amazon.awssdk.services.neptunegraph.model.TagResourceResponse;
import software.amazon.awssdk.services.neptunegraph.model.ThrottlingException;
import software.amazon.awssdk.services.neptunegraph.model.UntagResourceRequest;
import software.amazon.awssdk.services.neptunegraph.model.UntagResourceResponse;
import software.amazon.awssdk.services.neptunegraph.model.UpdateGraphRequest;
import software.amazon.awssdk.services.neptunegraph.model.UpdateGraphResponse;
import software.amazon.awssdk.services.neptunegraph.model.ValidationException;
import software.amazon.awssdk.services.neptunegraph.transform.CancelImportTaskRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.CreateGraphRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.CreateGraphSnapshotRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.CreateGraphUsingImportTaskRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.CreatePrivateGraphEndpointRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.DeleteGraphRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.DeleteGraphSnapshotRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.DeletePrivateGraphEndpointRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.GetGraphRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.GetGraphSnapshotRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.GetImportTaskRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.GetPrivateGraphEndpointRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.ListGraphSnapshotsRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.ListGraphsRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.ListImportTasksRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.ListPrivateGraphEndpointsRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.ResetGraphRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.RestoreGraphFromSnapshotRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.UpdateGraphRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link NeptuneGraphAsyncClient}.
 *
 * @see NeptuneGraphAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultNeptuneGraphAsyncClient implements NeptuneGraphAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultNeptuneGraphAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultNeptuneGraphAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Deletes the specified import task
     * </p>
     *
     * @param cancelImportTaskRequest
     * @return A Java Future containing the result of the CancelImportTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>ConflictException Raised when a conflict is encountered.</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.CancelImportTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/CancelImportTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelImportTaskResponse> cancelImportTask(CancelImportTaskRequest cancelImportTaskRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelImportTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelImportTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelImportTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelImportTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelImportTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelImportTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelImportTaskRequest, CancelImportTaskResponse>()
                            .withOperationName("CancelImportTask").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CancelImportTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(cancelImportTaskRequest));
            CompletableFuture<CancelImportTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new Neptune Analytics graph.
     * </p>
     *
     * @param createGraphRequest
     * @return A Java Future containing the result of the CreateGraph operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException A service quota was exceeded.</li>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>ConflictException Raised when a conflict is encountered.</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.CreateGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/CreateGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGraphResponse> createGraph(CreateGraphRequest createGraphRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createGraphRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGraph");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateGraphResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGraphResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGraphRequest, CreateGraphResponse>()
                            .withOperationName("CreateGraph").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateGraphRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createGraphRequest));
            CompletableFuture<CreateGraphResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a snapshot of the specific graph.
     * </p>
     *
     * @param createGraphSnapshotRequest
     * @return A Java Future containing the result of the CreateGraphSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException A service quota was exceeded.</li>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>ConflictException Raised when a conflict is encountered.</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.CreateGraphSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/CreateGraphSnapshot"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGraphSnapshotResponse> createGraphSnapshot(
            CreateGraphSnapshotRequest createGraphSnapshotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createGraphSnapshotRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGraphSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGraphSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGraphSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateGraphSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGraphSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGraphSnapshotRequest, CreateGraphSnapshotResponse>()
                            .withOperationName("CreateGraphSnapshot").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateGraphSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createGraphSnapshotRequest));
            CompletableFuture<CreateGraphSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new Neptune Analytics graph and imports data into it, either from Amazon Simple Storage Service (S3) or
     * from a Neptune database or a Neptune database snapshot.
     * </p>
     * <p>
     * The data can be loaded from files in S3 that in either the <a
     * href="https://docs.aws.amazon.com/neptune/latest/userguide/bulk-load-tutorial-format-gremlin.html">Gremlin CSV
     * format</a> or the <a
     * href="https://docs.aws.amazon.com/neptune/latest/userguide/bulk-load-tutorial-format-opencypher.html">openCypher
     * load format</a>.
     * </p>
     *
     * @param createGraphUsingImportTaskRequest
     * @return A Java Future containing the result of the CreateGraphUsingImportTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException A service quota was exceeded.</li>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>ConflictException Raised when a conflict is encountered.</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.CreateGraphUsingImportTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/CreateGraphUsingImportTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGraphUsingImportTaskResponse> createGraphUsingImportTask(
            CreateGraphUsingImportTaskRequest createGraphUsingImportTaskRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createGraphUsingImportTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGraphUsingImportTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGraphUsingImportTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGraphUsingImportTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateGraphUsingImportTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGraphUsingImportTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGraphUsingImportTaskRequest, CreateGraphUsingImportTaskResponse>()
                            .withOperationName("CreateGraphUsingImportTask").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateGraphUsingImportTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createGraphUsingImportTaskRequest));
            CompletableFuture<CreateGraphUsingImportTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a private graph endpoint to allow private access from to the graph from within a VPC. You can attach
     * security groups to the private graph endpoint. VPC endpoint charges apply.
     * </p>
     *
     * @param createPrivateGraphEndpointRequest
     * @return A Java Future containing the result of the CreatePrivateGraphEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException A service quota was exceeded.</li>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>ConflictException Raised when a conflict is encountered.</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.CreatePrivateGraphEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/CreatePrivateGraphEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePrivateGraphEndpointResponse> createPrivateGraphEndpoint(
            CreatePrivateGraphEndpointRequest createPrivateGraphEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPrivateGraphEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPrivateGraphEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePrivateGraphEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePrivateGraphEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePrivateGraphEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePrivateGraphEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePrivateGraphEndpointRequest, CreatePrivateGraphEndpointResponse>()
                            .withOperationName("CreatePrivateGraphEndpoint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePrivateGraphEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPrivateGraphEndpointRequest));
            CompletableFuture<CreatePrivateGraphEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified graph. Graphs cannot be deleted if delete-protection is enabled.
     * </p>
     *
     * @param deleteGraphRequest
     * @return A Java Future containing the result of the DeleteGraph operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>ConflictException Raised when a conflict is encountered.</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.DeleteGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/DeleteGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGraphResponse> deleteGraph(DeleteGraphRequest deleteGraphRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteGraphRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGraph");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteGraphResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGraphResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGraphRequest, DeleteGraphResponse>()
                            .withOperationName("DeleteGraph").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteGraphRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteGraphRequest));
            CompletableFuture<DeleteGraphResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specifed graph snapshot.
     * </p>
     *
     * @param deleteGraphSnapshotRequest
     * @return A Java Future containing the result of the DeleteGraphSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>ConflictException Raised when a conflict is encountered.</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.DeleteGraphSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/DeleteGraphSnapshot"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGraphSnapshotResponse> deleteGraphSnapshot(
            DeleteGraphSnapshotRequest deleteGraphSnapshotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteGraphSnapshotRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGraphSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGraphSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGraphSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteGraphSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGraphSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGraphSnapshotRequest, DeleteGraphSnapshotResponse>()
                            .withOperationName("DeleteGraphSnapshot").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteGraphSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteGraphSnapshotRequest));
            CompletableFuture<DeleteGraphSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a private graph endpoint.
     * </p>
     *
     * @param deletePrivateGraphEndpointRequest
     * @return A Java Future containing the result of the DeletePrivateGraphEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>ConflictException Raised when a conflict is encountered.</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.DeletePrivateGraphEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/DeletePrivateGraphEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePrivateGraphEndpointResponse> deletePrivateGraphEndpoint(
            DeletePrivateGraphEndpointRequest deletePrivateGraphEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePrivateGraphEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePrivateGraphEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePrivateGraphEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePrivateGraphEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePrivateGraphEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePrivateGraphEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePrivateGraphEndpointRequest, DeletePrivateGraphEndpointResponse>()
                            .withOperationName("DeletePrivateGraphEndpoint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePrivateGraphEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePrivateGraphEndpointRequest));
            CompletableFuture<DeletePrivateGraphEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a specified graph.
     * </p>
     *
     * @param getGraphRequest
     * @return A Java Future containing the result of the GetGraph operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.GetGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/GetGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetGraphResponse> getGraph(GetGraphRequest getGraphRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getGraphRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGraph");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetGraphResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGraphResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGraphRequest, GetGraphResponse>().withOperationName("GetGraph")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetGraphRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getGraphRequest));
            CompletableFuture<GetGraphResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a specified graph snapshot.
     * </p>
     *
     * @param getGraphSnapshotRequest
     * @return A Java Future containing the result of the GetGraphSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.GetGraphSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/GetGraphSnapshot"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetGraphSnapshotResponse> getGraphSnapshot(GetGraphSnapshotRequest getGraphSnapshotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getGraphSnapshotRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGraphSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGraphSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGraphSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetGraphSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGraphSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGraphSnapshotRequest, GetGraphSnapshotResponse>()
                            .withOperationName("GetGraphSnapshot").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetGraphSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getGraphSnapshotRequest));
            CompletableFuture<GetGraphSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a specified import task.
     * </p>
     *
     * @param getImportTaskRequest
     * @return A Java Future containing the result of the GetImportTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.GetImportTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/GetImportTask" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetImportTaskResponse> getImportTask(GetImportTaskRequest getImportTaskRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getImportTaskRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getImportTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetImportTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetImportTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetImportTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetImportTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetImportTaskRequest, GetImportTaskResponse>()
                            .withOperationName("GetImportTask").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetImportTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getImportTaskRequest));
            CompletableFuture<GetImportTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a specified private endpoint.
     * </p>
     *
     * @param getPrivateGraphEndpointRequest
     * @return A Java Future containing the result of the GetPrivateGraphEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.GetPrivateGraphEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/GetPrivateGraphEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPrivateGraphEndpointResponse> getPrivateGraphEndpoint(
            GetPrivateGraphEndpointRequest getPrivateGraphEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPrivateGraphEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPrivateGraphEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPrivateGraphEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPrivateGraphEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPrivateGraphEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPrivateGraphEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPrivateGraphEndpointRequest, GetPrivateGraphEndpointResponse>()
                            .withOperationName("GetPrivateGraphEndpoint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPrivateGraphEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPrivateGraphEndpointRequest));
            CompletableFuture<GetPrivateGraphEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists available snapshots of a specified Neptune Analytics graph.
     * </p>
     *
     * @param listGraphSnapshotsRequest
     * @return A Java Future containing the result of the ListGraphSnapshots operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.ListGraphSnapshots
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/ListGraphSnapshots"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListGraphSnapshotsResponse> listGraphSnapshots(ListGraphSnapshotsRequest listGraphSnapshotsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listGraphSnapshotsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGraphSnapshotsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGraphSnapshots");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGraphSnapshotsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListGraphSnapshotsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGraphSnapshotsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGraphSnapshotsRequest, ListGraphSnapshotsResponse>()
                            .withOperationName("ListGraphSnapshots").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListGraphSnapshotsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listGraphSnapshotsRequest));
            CompletableFuture<ListGraphSnapshotsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists available Neptune Analytics graphs.
     * </p>
     *
     * @param listGraphsRequest
     * @return A Java Future containing the result of the ListGraphs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.ListGraphs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/ListGraphs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListGraphsResponse> listGraphs(ListGraphsRequest listGraphsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listGraphsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGraphsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGraphs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGraphsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListGraphsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGraphsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGraphsRequest, ListGraphsResponse>().withOperationName("ListGraphs")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListGraphsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listGraphsRequest));
            CompletableFuture<ListGraphsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists import tasks.
     * </p>
     *
     * @param listImportTasksRequest
     * @return A Java Future containing the result of the ListImportTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.ListImportTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/ListImportTasks" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListImportTasksResponse> listImportTasks(ListImportTasksRequest listImportTasksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listImportTasksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listImportTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListImportTasks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListImportTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListImportTasksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListImportTasksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListImportTasksRequest, ListImportTasksResponse>()
                            .withOperationName("ListImportTasks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListImportTasksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listImportTasksRequest));
            CompletableFuture<ListImportTasksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists private endpoints for a specified Neptune Analytics graph.
     * </p>
     *
     * @param listPrivateGraphEndpointsRequest
     * @return A Java Future containing the result of the ListPrivateGraphEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.ListPrivateGraphEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/ListPrivateGraphEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPrivateGraphEndpointsResponse> listPrivateGraphEndpoints(
            ListPrivateGraphEndpointsRequest listPrivateGraphEndpointsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPrivateGraphEndpointsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPrivateGraphEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPrivateGraphEndpoints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPrivateGraphEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPrivateGraphEndpointsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPrivateGraphEndpointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPrivateGraphEndpointsRequest, ListPrivateGraphEndpointsResponse>()
                            .withOperationName("ListPrivateGraphEndpoints").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPrivateGraphEndpointsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPrivateGraphEndpointsRequest));
            CompletableFuture<ListPrivateGraphEndpointsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists tags associated with a specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Empties the data from a specified Neptune Analytics graph.
     * </p>
     *
     * @param resetGraphRequest
     * @return A Java Future containing the result of the ResetGraph operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>ConflictException Raised when a conflict is encountered.</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.ResetGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/ResetGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ResetGraphResponse> resetGraph(ResetGraphRequest resetGraphRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(resetGraphRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resetGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResetGraph");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ResetGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ResetGraphResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ResetGraphResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ResetGraphRequest, ResetGraphResponse>().withOperationName("ResetGraph")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ResetGraphRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(resetGraphRequest));
            CompletableFuture<ResetGraphResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Restores a graph from a snapshot.
     * </p>
     *
     * @param restoreGraphFromSnapshotRequest
     * @return A Java Future containing the result of the RestoreGraphFromSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException A service quota was exceeded.</li>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>ConflictException Raised when a conflict is encountered.</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.RestoreGraphFromSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/RestoreGraphFromSnapshot"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RestoreGraphFromSnapshotResponse> restoreGraphFromSnapshot(
            RestoreGraphFromSnapshotRequest restoreGraphFromSnapshotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(restoreGraphFromSnapshotRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreGraphFromSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreGraphFromSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RestoreGraphFromSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RestoreGraphFromSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RestoreGraphFromSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RestoreGraphFromSnapshotRequest, RestoreGraphFromSnapshotResponse>()
                            .withOperationName("RestoreGraphFromSnapshot").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RestoreGraphFromSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(restoreGraphFromSnapshotRequest));
            CompletableFuture<RestoreGraphFromSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds tags to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the configuration of a specified Neptune Analytics graph
     * </p>
     *
     * @param updateGraphRequest
     * @return A Java Future containing the result of the UpdateGraph operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The exception was interrupted by throttling.</li>
     *         <li>ValidationException A resource could not be validated</li>
     *         <li>ConflictException Raised when a conflict is encountered.</li>
     *         <li>InternalServerException A failure occurred on the server.</li>
     *         <li>ResourceNotFoundException A specified resource could not be located.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>NeptuneGraphException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample NeptuneGraphAsyncClient.UpdateGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/UpdateGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGraphResponse> updateGraph(UpdateGraphRequest updateGraphRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateGraphRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGraph");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateGraphResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGraphResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGraphRequest, UpdateGraphResponse>()
                            .withOperationName("UpdateGraph").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateGraphRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateGraphRequest));
            CompletableFuture<UpdateGraphResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final NeptuneGraphServiceClientConfiguration serviceClientConfiguration() {
        return new NeptuneGraphServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(NeptuneGraphException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        NeptuneGraphServiceClientConfigurationBuilder serviceConfigBuilder = new NeptuneGraphServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
