/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.neptunegraph.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.neptunegraph.NeptuneGraphClient;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphResponse;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphSnapshotRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphSnapshotResponse;
import software.amazon.awssdk.services.neptunegraph.model.GetImportTaskRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetImportTaskResponse;
import software.amazon.awssdk.services.neptunegraph.model.GetPrivateGraphEndpointRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetPrivateGraphEndpointResponse;
import software.amazon.awssdk.services.neptunegraph.model.NeptuneGraphRequest;
import software.amazon.awssdk.services.neptunegraph.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultNeptuneGraphWaiter implements NeptuneGraphWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final NeptuneGraphClient client;

    private final AttributeMap managedResources;

    private final Waiter<GetGraphResponse> graphAvailableWaiter;

    private final Waiter<GetGraphResponse> graphDeletedWaiter;

    private final Waiter<GetGraphSnapshotResponse> graphSnapshotAvailableWaiter;

    private final Waiter<GetGraphSnapshotResponse> graphSnapshotDeletedWaiter;

    private final Waiter<GetImportTaskResponse> importTaskCancelledWaiter;

    private final Waiter<GetImportTaskResponse> importTaskSuccessfulWaiter;

    private final Waiter<GetPrivateGraphEndpointResponse> privateGraphEndpointAvailableWaiter;

    private final Waiter<GetPrivateGraphEndpointResponse> privateGraphEndpointDeletedWaiter;

    private DefaultNeptuneGraphWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = NeptuneGraphClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.graphAvailableWaiter = Waiter.builder(GetGraphResponse.class).acceptors(graphAvailableWaiterAcceptors())
                .overrideConfiguration(graphAvailableWaiterConfig(builder.overrideConfiguration)).build();
        this.graphDeletedWaiter = Waiter.builder(GetGraphResponse.class).acceptors(graphDeletedWaiterAcceptors())
                .overrideConfiguration(graphDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.graphSnapshotAvailableWaiter = Waiter.builder(GetGraphSnapshotResponse.class)
                .acceptors(graphSnapshotAvailableWaiterAcceptors())
                .overrideConfiguration(graphSnapshotAvailableWaiterConfig(builder.overrideConfiguration)).build();
        this.graphSnapshotDeletedWaiter = Waiter.builder(GetGraphSnapshotResponse.class)
                .acceptors(graphSnapshotDeletedWaiterAcceptors())
                .overrideConfiguration(graphSnapshotDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.importTaskCancelledWaiter = Waiter.builder(GetImportTaskResponse.class)
                .acceptors(importTaskCancelledWaiterAcceptors())
                .overrideConfiguration(importTaskCancelledWaiterConfig(builder.overrideConfiguration)).build();
        this.importTaskSuccessfulWaiter = Waiter.builder(GetImportTaskResponse.class)
                .acceptors(importTaskSuccessfulWaiterAcceptors())
                .overrideConfiguration(importTaskSuccessfulWaiterConfig(builder.overrideConfiguration)).build();
        this.privateGraphEndpointAvailableWaiter = Waiter.builder(GetPrivateGraphEndpointResponse.class)
                .acceptors(privateGraphEndpointAvailableWaiterAcceptors())
                .overrideConfiguration(privateGraphEndpointAvailableWaiterConfig(builder.overrideConfiguration)).build();
        this.privateGraphEndpointDeletedWaiter = Waiter.builder(GetPrivateGraphEndpointResponse.class)
                .acceptors(privateGraphEndpointDeletedWaiterAcceptors())
                .overrideConfiguration(privateGraphEndpointDeletedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<GetGraphResponse> waitUntilGraphAvailable(GetGraphRequest getGraphRequest) {
        return graphAvailableWaiter.run(() -> client.getGraph(applyWaitersUserAgent(getGraphRequest)));
    }

    @Override
    public WaiterResponse<GetGraphResponse> waitUntilGraphAvailable(GetGraphRequest getGraphRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return graphAvailableWaiter.run(() -> client.getGraph(applyWaitersUserAgent(getGraphRequest)),
                graphAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetGraphResponse> waitUntilGraphDeleted(GetGraphRequest getGraphRequest) {
        return graphDeletedWaiter.run(() -> client.getGraph(applyWaitersUserAgent(getGraphRequest)));
    }

    @Override
    public WaiterResponse<GetGraphResponse> waitUntilGraphDeleted(GetGraphRequest getGraphRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return graphDeletedWaiter.run(() -> client.getGraph(applyWaitersUserAgent(getGraphRequest)),
                graphDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetGraphSnapshotResponse> waitUntilGraphSnapshotAvailable(
            GetGraphSnapshotRequest getGraphSnapshotRequest) {
        return graphSnapshotAvailableWaiter.run(() -> client.getGraphSnapshot(applyWaitersUserAgent(getGraphSnapshotRequest)));
    }

    @Override
    public WaiterResponse<GetGraphSnapshotResponse> waitUntilGraphSnapshotAvailable(
            GetGraphSnapshotRequest getGraphSnapshotRequest, WaiterOverrideConfiguration overrideConfig) {
        return graphSnapshotAvailableWaiter.run(() -> client.getGraphSnapshot(applyWaitersUserAgent(getGraphSnapshotRequest)),
                graphSnapshotAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetGraphSnapshotResponse> waitUntilGraphSnapshotDeleted(GetGraphSnapshotRequest getGraphSnapshotRequest) {
        return graphSnapshotDeletedWaiter.run(() -> client.getGraphSnapshot(applyWaitersUserAgent(getGraphSnapshotRequest)));
    }

    @Override
    public WaiterResponse<GetGraphSnapshotResponse> waitUntilGraphSnapshotDeleted(
            GetGraphSnapshotRequest getGraphSnapshotRequest, WaiterOverrideConfiguration overrideConfig) {
        return graphSnapshotDeletedWaiter.run(() -> client.getGraphSnapshot(applyWaitersUserAgent(getGraphSnapshotRequest)),
                graphSnapshotDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetImportTaskResponse> waitUntilImportTaskCancelled(GetImportTaskRequest getImportTaskRequest) {
        return importTaskCancelledWaiter.run(() -> client.getImportTask(applyWaitersUserAgent(getImportTaskRequest)));
    }

    @Override
    public WaiterResponse<GetImportTaskResponse> waitUntilImportTaskCancelled(GetImportTaskRequest getImportTaskRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return importTaskCancelledWaiter.run(() -> client.getImportTask(applyWaitersUserAgent(getImportTaskRequest)),
                importTaskCancelledWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetImportTaskResponse> waitUntilImportTaskSuccessful(GetImportTaskRequest getImportTaskRequest) {
        return importTaskSuccessfulWaiter.run(() -> client.getImportTask(applyWaitersUserAgent(getImportTaskRequest)));
    }

    @Override
    public WaiterResponse<GetImportTaskResponse> waitUntilImportTaskSuccessful(GetImportTaskRequest getImportTaskRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return importTaskSuccessfulWaiter.run(() -> client.getImportTask(applyWaitersUserAgent(getImportTaskRequest)),
                importTaskSuccessfulWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetPrivateGraphEndpointResponse> waitUntilPrivateGraphEndpointAvailable(
            GetPrivateGraphEndpointRequest getPrivateGraphEndpointRequest) {
        return privateGraphEndpointAvailableWaiter.run(() -> client
                .getPrivateGraphEndpoint(applyWaitersUserAgent(getPrivateGraphEndpointRequest)));
    }

    @Override
    public WaiterResponse<GetPrivateGraphEndpointResponse> waitUntilPrivateGraphEndpointAvailable(
            GetPrivateGraphEndpointRequest getPrivateGraphEndpointRequest, WaiterOverrideConfiguration overrideConfig) {
        return privateGraphEndpointAvailableWaiter.run(
                () -> client.getPrivateGraphEndpoint(applyWaitersUserAgent(getPrivateGraphEndpointRequest)),
                privateGraphEndpointAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetPrivateGraphEndpointResponse> waitUntilPrivateGraphEndpointDeleted(
            GetPrivateGraphEndpointRequest getPrivateGraphEndpointRequest) {
        return privateGraphEndpointDeletedWaiter.run(() -> client
                .getPrivateGraphEndpoint(applyWaitersUserAgent(getPrivateGraphEndpointRequest)));
    }

    @Override
    public WaiterResponse<GetPrivateGraphEndpointResponse> waitUntilPrivateGraphEndpointDeleted(
            GetPrivateGraphEndpointRequest getPrivateGraphEndpointRequest, WaiterOverrideConfiguration overrideConfig) {
        return privateGraphEndpointDeletedWaiter.run(
                () -> client.getPrivateGraphEndpoint(applyWaitersUserAgent(getPrivateGraphEndpointRequest)),
                privateGraphEndpointDeletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetGraphResponse>> graphAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super GetGraphResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "DELETING");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "FAILED");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "AVAILABLE");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetGraphResponse>> graphDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetGraphResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").compare("!=", input.constant("DELETING")).value(), true);
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetGraphSnapshotResponse>> graphSnapshotAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super GetGraphSnapshotResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "DELETING");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "FAILED");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "AVAILABLE");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetGraphSnapshotResponse>> graphSnapshotDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetGraphSnapshotResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").compare("!=", input.constant("DELETING")).value(), true);
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetImportTaskResponse>> importTaskCancelledWaiterAcceptors() {
        List<WaiterAcceptor<? super GetImportTaskResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(
                    input.field("status").compare("!=", input.constant("CANCELLING"))
                            .and(input.field("status").compare("!=", input.constant("CANCELLED"))).value(), true);
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "CANCELLED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetImportTaskResponse>> importTaskSuccessfulWaiterAcceptors() {
        List<WaiterAcceptor<? super GetImportTaskResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "CANCELLING");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "CANCELLED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "ROLLING_BACK");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "FAILED");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "SUCCEEDED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetPrivateGraphEndpointResponse>> privateGraphEndpointAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super GetPrivateGraphEndpointResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "DELETING");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "FAILED");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "AVAILABLE");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetPrivateGraphEndpointResponse>> privateGraphEndpointDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetPrivateGraphEndpointResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").compare("!=", input.constant("DELETING")).value(), true);
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration graphAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(480);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration graphDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration graphSnapshotAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration graphSnapshotDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration importTaskCancelledWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration importTaskSuccessfulWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(480);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration privateGraphEndpointAvailableWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(180);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration privateGraphEndpointDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(180);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static NeptuneGraphWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends NeptuneGraphRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements NeptuneGraphWaiter.Builder {
        private NeptuneGraphClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public NeptuneGraphWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public NeptuneGraphWaiter.Builder client(NeptuneGraphClient client) {
            this.client = client;
            return this;
        }

        public NeptuneGraphWaiter build() {
            return new DefaultNeptuneGraphWaiter(this);
        }
    }
}
