/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.neptunegraph;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.neptunegraph.internal.NeptuneGraphServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.neptunegraph.model.AccessDeniedException;
import software.amazon.awssdk.services.neptunegraph.model.CancelImportTaskRequest;
import software.amazon.awssdk.services.neptunegraph.model.CancelImportTaskResponse;
import software.amazon.awssdk.services.neptunegraph.model.CancelQueryRequest;
import software.amazon.awssdk.services.neptunegraph.model.CancelQueryResponse;
import software.amazon.awssdk.services.neptunegraph.model.ConflictException;
import software.amazon.awssdk.services.neptunegraph.model.CreateGraphRequest;
import software.amazon.awssdk.services.neptunegraph.model.CreateGraphResponse;
import software.amazon.awssdk.services.neptunegraph.model.CreateGraphSnapshotRequest;
import software.amazon.awssdk.services.neptunegraph.model.CreateGraphSnapshotResponse;
import software.amazon.awssdk.services.neptunegraph.model.CreateGraphUsingImportTaskRequest;
import software.amazon.awssdk.services.neptunegraph.model.CreateGraphUsingImportTaskResponse;
import software.amazon.awssdk.services.neptunegraph.model.CreatePrivateGraphEndpointRequest;
import software.amazon.awssdk.services.neptunegraph.model.CreatePrivateGraphEndpointResponse;
import software.amazon.awssdk.services.neptunegraph.model.DeleteGraphRequest;
import software.amazon.awssdk.services.neptunegraph.model.DeleteGraphResponse;
import software.amazon.awssdk.services.neptunegraph.model.DeleteGraphSnapshotRequest;
import software.amazon.awssdk.services.neptunegraph.model.DeleteGraphSnapshotResponse;
import software.amazon.awssdk.services.neptunegraph.model.DeletePrivateGraphEndpointRequest;
import software.amazon.awssdk.services.neptunegraph.model.DeletePrivateGraphEndpointResponse;
import software.amazon.awssdk.services.neptunegraph.model.ExecuteQueryRequest;
import software.amazon.awssdk.services.neptunegraph.model.ExecuteQueryResponse;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphResponse;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphSnapshotRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphSnapshotResponse;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphSummaryRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphSummaryResponse;
import software.amazon.awssdk.services.neptunegraph.model.GetImportTaskRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetImportTaskResponse;
import software.amazon.awssdk.services.neptunegraph.model.GetPrivateGraphEndpointRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetPrivateGraphEndpointResponse;
import software.amazon.awssdk.services.neptunegraph.model.GetQueryRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetQueryResponse;
import software.amazon.awssdk.services.neptunegraph.model.InternalServerException;
import software.amazon.awssdk.services.neptunegraph.model.ListGraphSnapshotsRequest;
import software.amazon.awssdk.services.neptunegraph.model.ListGraphSnapshotsResponse;
import software.amazon.awssdk.services.neptunegraph.model.ListGraphsRequest;
import software.amazon.awssdk.services.neptunegraph.model.ListGraphsResponse;
import software.amazon.awssdk.services.neptunegraph.model.ListImportTasksRequest;
import software.amazon.awssdk.services.neptunegraph.model.ListImportTasksResponse;
import software.amazon.awssdk.services.neptunegraph.model.ListPrivateGraphEndpointsRequest;
import software.amazon.awssdk.services.neptunegraph.model.ListPrivateGraphEndpointsResponse;
import software.amazon.awssdk.services.neptunegraph.model.ListQueriesRequest;
import software.amazon.awssdk.services.neptunegraph.model.ListQueriesResponse;
import software.amazon.awssdk.services.neptunegraph.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.neptunegraph.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.neptunegraph.model.NeptuneGraphException;
import software.amazon.awssdk.services.neptunegraph.model.ResetGraphRequest;
import software.amazon.awssdk.services.neptunegraph.model.ResetGraphResponse;
import software.amazon.awssdk.services.neptunegraph.model.ResourceNotFoundException;
import software.amazon.awssdk.services.neptunegraph.model.RestoreGraphFromSnapshotRequest;
import software.amazon.awssdk.services.neptunegraph.model.RestoreGraphFromSnapshotResponse;
import software.amazon.awssdk.services.neptunegraph.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.neptunegraph.model.StartImportTaskRequest;
import software.amazon.awssdk.services.neptunegraph.model.StartImportTaskResponse;
import software.amazon.awssdk.services.neptunegraph.model.TagResourceRequest;
import software.amazon.awssdk.services.neptunegraph.model.TagResourceResponse;
import software.amazon.awssdk.services.neptunegraph.model.ThrottlingException;
import software.amazon.awssdk.services.neptunegraph.model.UnprocessableException;
import software.amazon.awssdk.services.neptunegraph.model.UntagResourceRequest;
import software.amazon.awssdk.services.neptunegraph.model.UntagResourceResponse;
import software.amazon.awssdk.services.neptunegraph.model.UpdateGraphRequest;
import software.amazon.awssdk.services.neptunegraph.model.UpdateGraphResponse;
import software.amazon.awssdk.services.neptunegraph.model.ValidationException;
import software.amazon.awssdk.services.neptunegraph.transform.CancelImportTaskRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.CancelQueryRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.CreateGraphRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.CreateGraphSnapshotRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.CreateGraphUsingImportTaskRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.CreatePrivateGraphEndpointRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.DeleteGraphRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.DeleteGraphSnapshotRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.DeletePrivateGraphEndpointRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.ExecuteQueryRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.GetGraphRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.GetGraphSnapshotRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.GetGraphSummaryRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.GetImportTaskRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.GetPrivateGraphEndpointRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.GetQueryRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.ListGraphSnapshotsRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.ListGraphsRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.ListImportTasksRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.ListPrivateGraphEndpointsRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.ListQueriesRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.ResetGraphRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.RestoreGraphFromSnapshotRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.StartImportTaskRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.transform.UpdateGraphRequestMarshaller;
import software.amazon.awssdk.services.neptunegraph.waiters.NeptuneGraphWaiter;
import software.amazon.awssdk.utils.HostnameValidator;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link NeptuneGraphClient}.
 *
 * @see NeptuneGraphClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultNeptuneGraphClient implements NeptuneGraphClient {
    private static final Logger log = Logger.loggerFor(DefaultNeptuneGraphClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultNeptuneGraphClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Deletes the specified import task.
     * </p>
     *
     * @param cancelImportTaskRequest
     * @return Result of the CancelImportTask operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws ConflictException
     *         Raised when a conflict is encountered.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.CancelImportTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/CancelImportTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CancelImportTaskResponse cancelImportTask(CancelImportTaskRequest cancelImportTaskRequest) throws ThrottlingException,
            ValidationException, ConflictException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelImportTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelImportTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelImportTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelImportTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelImportTask");

            return clientHandler.execute(new ClientExecutionParams<CancelImportTaskRequest, CancelImportTaskResponse>()
                    .withOperationName("CancelImportTask").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(cancelImportTaskRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelImportTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Cancels a specified query.
     * </p>
     *
     * @param cancelQueryRequest
     * @return Result of the CancelQuery operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws AccessDeniedException
     *         Raised in case of an authentication or authorization failure.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.CancelQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/CancelQuery" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CancelQueryResponse cancelQuery(CancelQueryRequest cancelQueryRequest) throws ThrottlingException,
            ValidationException, AccessDeniedException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelQueryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelQuery");
            String hostPrefix = "{graphIdentifier}.";
            HostnameValidator.validateHostnameCompliant(cancelQueryRequest.graphIdentifier(), "graphIdentifier",
                    "cancelQueryRequest");
            String resolvedHostExpression = String.format("%s.", cancelQueryRequest.graphIdentifier());

            return clientHandler.execute(new ClientExecutionParams<CancelQueryRequest, CancelQueryResponse>()
                    .withOperationName("CancelQuery").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withRequestConfiguration(clientConfiguration).withInput(cancelQueryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new Neptune Analytics graph.
     * </p>
     *
     * @param createGraphRequest
     * @return Result of the CreateGraph operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         A service quota was exceeded.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws ConflictException
     *         Raised when a conflict is encountered.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.CreateGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/CreateGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateGraphResponse createGraph(CreateGraphRequest createGraphRequest) throws ServiceQuotaExceededException,
            ThrottlingException, ValidationException, ConflictException, InternalServerException, AwsServiceException,
            SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateGraphResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createGraphRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGraph");

            return clientHandler.execute(new ClientExecutionParams<CreateGraphRequest, CreateGraphResponse>()
                    .withOperationName("CreateGraph").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createGraphRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateGraphRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a snapshot of the specific graph.
     * </p>
     *
     * @param createGraphSnapshotRequest
     * @return Result of the CreateGraphSnapshot operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         A service quota was exceeded.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws ConflictException
     *         Raised when a conflict is encountered.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.CreateGraphSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/CreateGraphSnapshot"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateGraphSnapshotResponse createGraphSnapshot(CreateGraphSnapshotRequest createGraphSnapshotRequest)
            throws ServiceQuotaExceededException, ThrottlingException, ValidationException, ConflictException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGraphSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateGraphSnapshotResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createGraphSnapshotRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGraphSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGraphSnapshot");

            return clientHandler.execute(new ClientExecutionParams<CreateGraphSnapshotRequest, CreateGraphSnapshotResponse>()
                    .withOperationName("CreateGraphSnapshot").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createGraphSnapshotRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateGraphSnapshotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new Neptune Analytics graph and imports data into it, either from Amazon Simple Storage Service (S3) or
     * from a Neptune database or a Neptune database snapshot.
     * </p>
     * <p>
     * The data can be loaded from files in S3 that in either the <a
     * href="https://docs.aws.amazon.com/neptune/latest/userguide/bulk-load-tutorial-format-gremlin.html">Gremlin CSV
     * format</a> or the <a
     * href="https://docs.aws.amazon.com/neptune/latest/userguide/bulk-load-tutorial-format-opencypher.html">openCypher
     * load format</a>.
     * </p>
     *
     * @param createGraphUsingImportTaskRequest
     * @return Result of the CreateGraphUsingImportTask operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         A service quota was exceeded.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws ConflictException
     *         Raised when a conflict is encountered.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.CreateGraphUsingImportTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/CreateGraphUsingImportTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateGraphUsingImportTaskResponse createGraphUsingImportTask(
            CreateGraphUsingImportTaskRequest createGraphUsingImportTaskRequest) throws ServiceQuotaExceededException,
            ThrottlingException, ValidationException, ConflictException, InternalServerException, AwsServiceException,
            SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGraphUsingImportTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateGraphUsingImportTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createGraphUsingImportTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGraphUsingImportTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGraphUsingImportTask");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateGraphUsingImportTaskRequest, CreateGraphUsingImportTaskResponse>()
                            .withOperationName("CreateGraphUsingImportTask").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createGraphUsingImportTaskRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateGraphUsingImportTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create a private graph endpoint to allow private access from to the graph from within a VPC. You can attach
     * security groups to the private graph endpoint.
     * </p>
     * <note>
     * <p>
     * VPC endpoint charges apply.
     * </p>
     * </note>
     *
     * @param createPrivateGraphEndpointRequest
     * @return Result of the CreatePrivateGraphEndpoint operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         A service quota was exceeded.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws ConflictException
     *         Raised when a conflict is encountered.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.CreatePrivateGraphEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/CreatePrivateGraphEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreatePrivateGraphEndpointResponse createPrivateGraphEndpoint(
            CreatePrivateGraphEndpointRequest createPrivateGraphEndpointRequest) throws ServiceQuotaExceededException,
            ThrottlingException, ValidationException, ConflictException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePrivateGraphEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreatePrivateGraphEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPrivateGraphEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPrivateGraphEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePrivateGraphEndpoint");

            return clientHandler
                    .execute(new ClientExecutionParams<CreatePrivateGraphEndpointRequest, CreatePrivateGraphEndpointResponse>()
                            .withOperationName("CreatePrivateGraphEndpoint").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createPrivateGraphEndpointRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreatePrivateGraphEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified graph. Graphs cannot be deleted if delete-protection is enabled.
     * </p>
     *
     * @param deleteGraphRequest
     * @return Result of the DeleteGraph operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws ConflictException
     *         Raised when a conflict is encountered.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.DeleteGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/DeleteGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteGraphResponse deleteGraph(DeleteGraphRequest deleteGraphRequest) throws ThrottlingException,
            ValidationException, ConflictException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteGraphResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteGraphRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGraph");

            return clientHandler.execute(new ClientExecutionParams<DeleteGraphRequest, DeleteGraphResponse>()
                    .withOperationName("DeleteGraph").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteGraphRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteGraphRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specifed graph snapshot.
     * </p>
     *
     * @param deleteGraphSnapshotRequest
     * @return Result of the DeleteGraphSnapshot operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws ConflictException
     *         Raised when a conflict is encountered.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.DeleteGraphSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/DeleteGraphSnapshot"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteGraphSnapshotResponse deleteGraphSnapshot(DeleteGraphSnapshotRequest deleteGraphSnapshotRequest)
            throws ThrottlingException, ValidationException, ConflictException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteGraphSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteGraphSnapshotResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteGraphSnapshotRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGraphSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGraphSnapshot");

            return clientHandler.execute(new ClientExecutionParams<DeleteGraphSnapshotRequest, DeleteGraphSnapshotResponse>()
                    .withOperationName("DeleteGraphSnapshot").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteGraphSnapshotRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteGraphSnapshotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a private graph endpoint.
     * </p>
     *
     * @param deletePrivateGraphEndpointRequest
     * @return Result of the DeletePrivateGraphEndpoint operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws ConflictException
     *         Raised when a conflict is encountered.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.DeletePrivateGraphEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/DeletePrivateGraphEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeletePrivateGraphEndpointResponse deletePrivateGraphEndpoint(
            DeletePrivateGraphEndpointRequest deletePrivateGraphEndpointRequest) throws ThrottlingException, ValidationException,
            ConflictException, InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePrivateGraphEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeletePrivateGraphEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePrivateGraphEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePrivateGraphEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePrivateGraphEndpoint");

            return clientHandler
                    .execute(new ClientExecutionParams<DeletePrivateGraphEndpointRequest, DeletePrivateGraphEndpointResponse>()
                            .withOperationName("DeletePrivateGraphEndpoint").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deletePrivateGraphEndpointRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeletePrivateGraphEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Execute an openCypher query.
     * </p>
     * <p>
     * When invoking this operation in a Neptune Analytics cluster, the IAM user or role making the request must have a
     * policy attached that allows one of the following IAM actions in that cluster, depending on the query:
     * </p>
     * <ul>
     * <li>
     * <p>
     * neptune-graph:ReadDataViaQuery
     * </p>
     * </li>
     * <li>
     * <p>
     * neptune-graph:WriteDataViaQuery
     * </p>
     * </li>
     * <li>
     * <p>
     * neptune-graph:DeleteDataViaQuery
     * </p>
     * </li>
     * </ul>
     *
     * @param executeQueryRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled ExecuteQueryResponse
     *        and an InputStream to the response content are provided as parameters to the callback. The callback may
     *        return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The query results.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws UnprocessableException
     *         Request cannot be processed due to known reasons. Eg. partition full.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws ConflictException
     *         Raised when a conflict is encountered.
     * @throws AccessDeniedException
     *         Raised in case of an authentication or authorization failure.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.ExecuteQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/ExecuteQuery" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public <ReturnT> ReturnT executeQuery(ExecuteQueryRequest executeQueryRequest,
            ResponseTransformer<ExecuteQueryResponse, ReturnT> responseTransformer) throws UnprocessableException,
            ThrottlingException, ValidationException, ConflictException, AccessDeniedException, InternalServerException,
            AwsServiceException, SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                .isPayloadJson(false).build();

        HttpResponseHandler<ExecuteQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ExecuteQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(executeQueryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executeQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecuteQuery");
            String hostPrefix = "{graphIdentifier}.";
            HostnameValidator.validateHostnameCompliant(executeQueryRequest.graphIdentifier(), "graphIdentifier",
                    "executeQueryRequest");
            String resolvedHostExpression = String.format("%s.", executeQueryRequest.graphIdentifier());

            return clientHandler.execute(
                    new ClientExecutionParams<ExecuteQueryRequest, ExecuteQueryResponse>().withOperationName("ExecuteQuery")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withRequestConfiguration(clientConfiguration).withInput(executeQueryRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ExecuteQueryRequestMarshaller(protocolFactory)), responseTransformer);
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a specified graph.
     * </p>
     *
     * @param getGraphRequest
     * @return Result of the GetGraph operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.GetGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/GetGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetGraphResponse getGraph(GetGraphRequest getGraphRequest) throws ThrottlingException, ValidationException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetGraphResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getGraphRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGraph");

            return clientHandler.execute(new ClientExecutionParams<GetGraphRequest, GetGraphResponse>()
                    .withOperationName("GetGraph").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getGraphRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetGraphRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a specified graph snapshot.
     * </p>
     *
     * @param getGraphSnapshotRequest
     * @return Result of the GetGraphSnapshot operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.GetGraphSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/GetGraphSnapshot"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetGraphSnapshotResponse getGraphSnapshot(GetGraphSnapshotRequest getGraphSnapshotRequest) throws ThrottlingException,
            ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGraphSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetGraphSnapshotResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getGraphSnapshotRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGraphSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGraphSnapshot");

            return clientHandler.execute(new ClientExecutionParams<GetGraphSnapshotRequest, GetGraphSnapshotResponse>()
                    .withOperationName("GetGraphSnapshot").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getGraphSnapshotRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetGraphSnapshotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a graph summary for a property graph.
     * </p>
     *
     * @param getGraphSummaryRequest
     * @return Result of the GetGraphSummary operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws AccessDeniedException
     *         Raised in case of an authentication or authorization failure.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.GetGraphSummary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/GetGraphSummary" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetGraphSummaryResponse getGraphSummary(GetGraphSummaryRequest getGraphSummaryRequest) throws ThrottlingException,
            ValidationException, AccessDeniedException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGraphSummaryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetGraphSummaryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getGraphSummaryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGraphSummaryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGraphSummary");
            String hostPrefix = "{graphIdentifier}.";
            HostnameValidator.validateHostnameCompliant(getGraphSummaryRequest.graphIdentifier(), "graphIdentifier",
                    "getGraphSummaryRequest");
            String resolvedHostExpression = String.format("%s.", getGraphSummaryRequest.graphIdentifier());

            return clientHandler.execute(new ClientExecutionParams<GetGraphSummaryRequest, GetGraphSummaryResponse>()
                    .withOperationName("GetGraphSummary").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .hostPrefixExpression(resolvedHostExpression).withRequestConfiguration(clientConfiguration)
                    .withInput(getGraphSummaryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetGraphSummaryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a specified import task.
     * </p>
     *
     * @param getImportTaskRequest
     * @return Result of the GetImportTask operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.GetImportTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/GetImportTask" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetImportTaskResponse getImportTask(GetImportTaskRequest getImportTaskRequest) throws ThrottlingException,
            ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetImportTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetImportTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getImportTaskRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getImportTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetImportTask");

            return clientHandler.execute(new ClientExecutionParams<GetImportTaskRequest, GetImportTaskResponse>()
                    .withOperationName("GetImportTask").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getImportTaskRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetImportTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about a specified private endpoint.
     * </p>
     *
     * @param getPrivateGraphEndpointRequest
     * @return Result of the GetPrivateGraphEndpoint operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.GetPrivateGraphEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/GetPrivateGraphEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetPrivateGraphEndpointResponse getPrivateGraphEndpoint(GetPrivateGraphEndpointRequest getPrivateGraphEndpointRequest)
            throws ThrottlingException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPrivateGraphEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetPrivateGraphEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPrivateGraphEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPrivateGraphEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPrivateGraphEndpoint");

            return clientHandler
                    .execute(new ClientExecutionParams<GetPrivateGraphEndpointRequest, GetPrivateGraphEndpointResponse>()
                            .withOperationName("GetPrivateGraphEndpoint").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getPrivateGraphEndpointRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetPrivateGraphEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the status of a specified query.
     * </p>
     * <note>
     * <p>
     * When invoking this operation in a Neptune Analytics cluster, the IAM user or role making the request must have
     * the <code>neptune-graph:GetQueryStatus</code> IAM action attached.
     * </p>
     * </note>
     *
     * @param getQueryRequest
     * @return Result of the GetQuery operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws AccessDeniedException
     *         Raised in case of an authentication or authorization failure.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.GetQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/GetQuery" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetQueryResponse getQuery(GetQueryRequest getQueryRequest) throws ThrottlingException, ValidationException,
            AccessDeniedException, InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getQueryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQuery");
            String hostPrefix = "{graphIdentifier}.";
            HostnameValidator.validateHostnameCompliant(getQueryRequest.graphIdentifier(), "graphIdentifier", "getQueryRequest");
            String resolvedHostExpression = String.format("%s.", getQueryRequest.graphIdentifier());

            return clientHandler.execute(new ClientExecutionParams<GetQueryRequest, GetQueryResponse>()
                    .withOperationName("GetQuery").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withRequestConfiguration(clientConfiguration).withInput(getQueryRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists available snapshots of a specified Neptune Analytics graph.
     * </p>
     *
     * @param listGraphSnapshotsRequest
     * @return Result of the ListGraphSnapshots operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.ListGraphSnapshots
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/ListGraphSnapshots"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListGraphSnapshotsResponse listGraphSnapshots(ListGraphSnapshotsRequest listGraphSnapshotsRequest)
            throws ThrottlingException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGraphSnapshotsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListGraphSnapshotsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listGraphSnapshotsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGraphSnapshotsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGraphSnapshots");

            return clientHandler.execute(new ClientExecutionParams<ListGraphSnapshotsRequest, ListGraphSnapshotsResponse>()
                    .withOperationName("ListGraphSnapshots").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listGraphSnapshotsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListGraphSnapshotsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists available Neptune Analytics graphs.
     * </p>
     *
     * @param listGraphsRequest
     * @return Result of the ListGraphs operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.ListGraphs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/ListGraphs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListGraphsResponse listGraphs(ListGraphsRequest listGraphsRequest) throws ThrottlingException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGraphsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListGraphsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listGraphsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGraphsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGraphs");

            return clientHandler.execute(new ClientExecutionParams<ListGraphsRequest, ListGraphsResponse>()
                    .withOperationName("ListGraphs").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listGraphsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListGraphsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists import tasks.
     * </p>
     *
     * @param listImportTasksRequest
     * @return Result of the ListImportTasks operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.ListImportTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/ListImportTasks" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListImportTasksResponse listImportTasks(ListImportTasksRequest listImportTasksRequest) throws ThrottlingException,
            ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListImportTasksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListImportTasksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listImportTasksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listImportTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListImportTasks");

            return clientHandler.execute(new ClientExecutionParams<ListImportTasksRequest, ListImportTasksResponse>()
                    .withOperationName("ListImportTasks").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listImportTasksRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListImportTasksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists private endpoints for a specified Neptune Analytics graph.
     * </p>
     *
     * @param listPrivateGraphEndpointsRequest
     * @return Result of the ListPrivateGraphEndpoints operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.ListPrivateGraphEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/ListPrivateGraphEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPrivateGraphEndpointsResponse listPrivateGraphEndpoints(
            ListPrivateGraphEndpointsRequest listPrivateGraphEndpointsRequest) throws ThrottlingException, ValidationException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPrivateGraphEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPrivateGraphEndpointsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPrivateGraphEndpointsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPrivateGraphEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPrivateGraphEndpoints");

            return clientHandler
                    .execute(new ClientExecutionParams<ListPrivateGraphEndpointsRequest, ListPrivateGraphEndpointsResponse>()
                            .withOperationName("ListPrivateGraphEndpoints").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listPrivateGraphEndpointsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListPrivateGraphEndpointsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists active openCypher queries.
     * </p>
     *
     * @param listQueriesRequest
     * @return Result of the ListQueries operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws AccessDeniedException
     *         Raised in case of an authentication or authorization failure.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.ListQueries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/ListQueries" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListQueriesResponse listQueries(ListQueriesRequest listQueriesRequest) throws ThrottlingException,
            ValidationException, AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException,
            NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListQueriesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListQueriesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listQueriesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listQueriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListQueries");
            String hostPrefix = "{graphIdentifier}.";
            HostnameValidator.validateHostnameCompliant(listQueriesRequest.graphIdentifier(), "graphIdentifier",
                    "listQueriesRequest");
            String resolvedHostExpression = String.format("%s.", listQueriesRequest.graphIdentifier());

            return clientHandler.execute(new ClientExecutionParams<ListQueriesRequest, ListQueriesResponse>()
                    .withOperationName("ListQueries").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withRequestConfiguration(clientConfiguration).withInput(listQueriesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListQueriesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists tags associated with a specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ThrottlingException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Empties the data from a specified Neptune Analytics graph.
     * </p>
     *
     * @param resetGraphRequest
     * @return Result of the ResetGraph operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws ConflictException
     *         Raised when a conflict is encountered.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.ResetGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/ResetGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ResetGraphResponse resetGraph(ResetGraphRequest resetGraphRequest) throws ThrottlingException, ValidationException,
            ConflictException, InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ResetGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ResetGraphResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(resetGraphRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resetGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResetGraph");

            return clientHandler.execute(new ClientExecutionParams<ResetGraphRequest, ResetGraphResponse>()
                    .withOperationName("ResetGraph").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(resetGraphRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ResetGraphRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Restores a graph from a snapshot.
     * </p>
     *
     * @param restoreGraphFromSnapshotRequest
     * @return Result of the RestoreGraphFromSnapshot operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         A service quota was exceeded.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws ConflictException
     *         Raised when a conflict is encountered.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.RestoreGraphFromSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/RestoreGraphFromSnapshot"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RestoreGraphFromSnapshotResponse restoreGraphFromSnapshot(
            RestoreGraphFromSnapshotRequest restoreGraphFromSnapshotRequest) throws ServiceQuotaExceededException,
            ThrottlingException, ValidationException, ConflictException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RestoreGraphFromSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RestoreGraphFromSnapshotResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(restoreGraphFromSnapshotRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreGraphFromSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreGraphFromSnapshot");

            return clientHandler
                    .execute(new ClientExecutionParams<RestoreGraphFromSnapshotRequest, RestoreGraphFromSnapshotResponse>()
                            .withOperationName("RestoreGraphFromSnapshot").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(restoreGraphFromSnapshotRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RestoreGraphFromSnapshotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Import data into existing Neptune Analytics graph from Amazon Simple Storage Service (S3). The graph needs to be
     * empty and in the AVAILABLE state.
     * </p>
     *
     * @param startImportTaskRequest
     * @return Result of the StartImportTask operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws ConflictException
     *         Raised when a conflict is encountered.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.StartImportTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/StartImportTask" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartImportTaskResponse startImportTask(StartImportTaskRequest startImportTaskRequest) throws ThrottlingException,
            ValidationException, ConflictException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartImportTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartImportTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startImportTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startImportTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartImportTask");

            return clientHandler.execute(new ClientExecutionParams<StartImportTaskRequest, StartImportTaskResponse>()
                    .withOperationName("StartImportTask").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startImportTaskRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartImportTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds tags to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ThrottlingException,
            ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ThrottlingException,
            ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the configuration of a specified Neptune Analytics graph
     * </p>
     *
     * @param updateGraphRequest
     * @return Result of the UpdateGraph operation returned by the service.
     * @throws ThrottlingException
     *         The exception was interrupted by throttling.
     * @throws ValidationException
     *         A resource could not be validated.
     * @throws ConflictException
     *         Raised when a conflict is encountered.
     * @throws InternalServerException
     *         A failure occurred on the server.
     * @throws ResourceNotFoundException
     *         A specified resource could not be located.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NeptuneGraphException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NeptuneGraphClient.UpdateGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/neptune-graph-2023-11-29/UpdateGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateGraphResponse updateGraph(UpdateGraphRequest updateGraphRequest) throws ThrottlingException,
            ValidationException, ConflictException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, NeptuneGraphException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateGraphResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateGraphRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Neptune Graph");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGraph");

            return clientHandler.execute(new ClientExecutionParams<UpdateGraphRequest, UpdateGraphResponse>()
                    .withOperationName("UpdateGraph").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateGraphRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateGraphRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create an instance of {@link NeptuneGraphWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link NeptuneGraphWaiter}
     */
    @Override
    public NeptuneGraphWaiter waiter() {
        return NeptuneGraphWaiter.builder().client(this).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        NeptuneGraphServiceClientConfigurationBuilder serviceConfigBuilder = new NeptuneGraphServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(NeptuneGraphException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnprocessableException")
                                .exceptionBuilderSupplier(UnprocessableException::builder).httpStatusCode(422).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final NeptuneGraphServiceClientConfiguration serviceClientConfiguration() {
        return new NeptuneGraphServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
