/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.neptunegraph.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.neptunegraph.NeptuneGraphAsyncClient;
import software.amazon.awssdk.services.neptunegraph.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.neptunegraph.model.GetExportTaskRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetExportTaskResponse;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphResponse;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphSnapshotRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetGraphSnapshotResponse;
import software.amazon.awssdk.services.neptunegraph.model.GetImportTaskRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetImportTaskResponse;
import software.amazon.awssdk.services.neptunegraph.model.GetPrivateGraphEndpointRequest;
import software.amazon.awssdk.services.neptunegraph.model.GetPrivateGraphEndpointResponse;
import software.amazon.awssdk.services.neptunegraph.model.NeptuneGraphRequest;
import software.amazon.awssdk.services.neptunegraph.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultNeptuneGraphAsyncWaiter implements NeptuneGraphAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final NeptuneGraphAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<GetExportTaskResponse> exportTaskCancelledWaiter;

    private final AsyncWaiter<GetExportTaskResponse> exportTaskSuccessfulWaiter;

    private final AsyncWaiter<GetGraphResponse> graphAvailableWaiter;

    private final AsyncWaiter<GetGraphResponse> graphDeletedWaiter;

    private final AsyncWaiter<GetGraphSnapshotResponse> graphSnapshotAvailableWaiter;

    private final AsyncWaiter<GetGraphSnapshotResponse> graphSnapshotDeletedWaiter;

    private final AsyncWaiter<GetImportTaskResponse> importTaskCancelledWaiter;

    private final AsyncWaiter<GetImportTaskResponse> importTaskSuccessfulWaiter;

    private final AsyncWaiter<GetPrivateGraphEndpointResponse> privateGraphEndpointAvailableWaiter;

    private final AsyncWaiter<GetPrivateGraphEndpointResponse> privateGraphEndpointDeletedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultNeptuneGraphAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = NeptuneGraphAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.exportTaskCancelledWaiter = AsyncWaiter.builder(GetExportTaskResponse.class)
                .acceptors(exportTaskCancelledWaiterAcceptors())
                .overrideConfiguration(exportTaskCancelledWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.exportTaskSuccessfulWaiter = AsyncWaiter.builder(GetExportTaskResponse.class)
                .acceptors(exportTaskSuccessfulWaiterAcceptors())
                .overrideConfiguration(exportTaskSuccessfulWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.graphAvailableWaiter = AsyncWaiter.builder(GetGraphResponse.class).acceptors(graphAvailableWaiterAcceptors())
                .overrideConfiguration(graphAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.graphDeletedWaiter = AsyncWaiter.builder(GetGraphResponse.class).acceptors(graphDeletedWaiterAcceptors())
                .overrideConfiguration(graphDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.graphSnapshotAvailableWaiter = AsyncWaiter.builder(GetGraphSnapshotResponse.class)
                .acceptors(graphSnapshotAvailableWaiterAcceptors())
                .overrideConfiguration(graphSnapshotAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.graphSnapshotDeletedWaiter = AsyncWaiter.builder(GetGraphSnapshotResponse.class)
                .acceptors(graphSnapshotDeletedWaiterAcceptors())
                .overrideConfiguration(graphSnapshotDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.importTaskCancelledWaiter = AsyncWaiter.builder(GetImportTaskResponse.class)
                .acceptors(importTaskCancelledWaiterAcceptors())
                .overrideConfiguration(importTaskCancelledWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.importTaskSuccessfulWaiter = AsyncWaiter.builder(GetImportTaskResponse.class)
                .acceptors(importTaskSuccessfulWaiterAcceptors())
                .overrideConfiguration(importTaskSuccessfulWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.privateGraphEndpointAvailableWaiter = AsyncWaiter.builder(GetPrivateGraphEndpointResponse.class)
                .acceptors(privateGraphEndpointAvailableWaiterAcceptors())
                .overrideConfiguration(privateGraphEndpointAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.privateGraphEndpointDeletedWaiter = AsyncWaiter.builder(GetPrivateGraphEndpointResponse.class)
                .acceptors(privateGraphEndpointDeletedWaiterAcceptors())
                .overrideConfiguration(privateGraphEndpointDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<GetExportTaskResponse>> waitUntilExportTaskCancelled(
            GetExportTaskRequest getExportTaskRequest) {
        return exportTaskCancelledWaiter.runAsync(() -> client.getExportTask(applyWaitersUserAgent(getExportTaskRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetExportTaskResponse>> waitUntilExportTaskCancelled(
            GetExportTaskRequest getExportTaskRequest, WaiterOverrideConfiguration overrideConfig) {
        return exportTaskCancelledWaiter.runAsync(() -> client.getExportTask(applyWaitersUserAgent(getExportTaskRequest)),
                exportTaskCancelledWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetExportTaskResponse>> waitUntilExportTaskSuccessful(
            GetExportTaskRequest getExportTaskRequest) {
        return exportTaskSuccessfulWaiter.runAsync(() -> client.getExportTask(applyWaitersUserAgent(getExportTaskRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetExportTaskResponse>> waitUntilExportTaskSuccessful(
            GetExportTaskRequest getExportTaskRequest, WaiterOverrideConfiguration overrideConfig) {
        return exportTaskSuccessfulWaiter.runAsync(() -> client.getExportTask(applyWaitersUserAgent(getExportTaskRequest)),
                exportTaskSuccessfulWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetGraphResponse>> waitUntilGraphAvailable(GetGraphRequest getGraphRequest) {
        return graphAvailableWaiter.runAsync(() -> client.getGraph(applyWaitersUserAgent(getGraphRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetGraphResponse>> waitUntilGraphAvailable(GetGraphRequest getGraphRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return graphAvailableWaiter.runAsync(() -> client.getGraph(applyWaitersUserAgent(getGraphRequest)),
                graphAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetGraphResponse>> waitUntilGraphDeleted(GetGraphRequest getGraphRequest) {
        return graphDeletedWaiter.runAsync(() -> client.getGraph(applyWaitersUserAgent(getGraphRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetGraphResponse>> waitUntilGraphDeleted(GetGraphRequest getGraphRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return graphDeletedWaiter.runAsync(() -> client.getGraph(applyWaitersUserAgent(getGraphRequest)),
                graphDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetGraphSnapshotResponse>> waitUntilGraphSnapshotAvailable(
            GetGraphSnapshotRequest getGraphSnapshotRequest) {
        return graphSnapshotAvailableWaiter.runAsync(() -> client
                .getGraphSnapshot(applyWaitersUserAgent(getGraphSnapshotRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetGraphSnapshotResponse>> waitUntilGraphSnapshotAvailable(
            GetGraphSnapshotRequest getGraphSnapshotRequest, WaiterOverrideConfiguration overrideConfig) {
        return graphSnapshotAvailableWaiter.runAsync(
                () -> client.getGraphSnapshot(applyWaitersUserAgent(getGraphSnapshotRequest)),
                graphSnapshotAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetGraphSnapshotResponse>> waitUntilGraphSnapshotDeleted(
            GetGraphSnapshotRequest getGraphSnapshotRequest) {
        return graphSnapshotDeletedWaiter.runAsync(() -> client.getGraphSnapshot(applyWaitersUserAgent(getGraphSnapshotRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetGraphSnapshotResponse>> waitUntilGraphSnapshotDeleted(
            GetGraphSnapshotRequest getGraphSnapshotRequest, WaiterOverrideConfiguration overrideConfig) {
        return graphSnapshotDeletedWaiter.runAsync(() -> client.getGraphSnapshot(applyWaitersUserAgent(getGraphSnapshotRequest)),
                graphSnapshotDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetImportTaskResponse>> waitUntilImportTaskCancelled(
            GetImportTaskRequest getImportTaskRequest) {
        return importTaskCancelledWaiter.runAsync(() -> client.getImportTask(applyWaitersUserAgent(getImportTaskRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetImportTaskResponse>> waitUntilImportTaskCancelled(
            GetImportTaskRequest getImportTaskRequest, WaiterOverrideConfiguration overrideConfig) {
        return importTaskCancelledWaiter.runAsync(() -> client.getImportTask(applyWaitersUserAgent(getImportTaskRequest)),
                importTaskCancelledWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetImportTaskResponse>> waitUntilImportTaskSuccessful(
            GetImportTaskRequest getImportTaskRequest) {
        return importTaskSuccessfulWaiter.runAsync(() -> client.getImportTask(applyWaitersUserAgent(getImportTaskRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetImportTaskResponse>> waitUntilImportTaskSuccessful(
            GetImportTaskRequest getImportTaskRequest, WaiterOverrideConfiguration overrideConfig) {
        return importTaskSuccessfulWaiter.runAsync(() -> client.getImportTask(applyWaitersUserAgent(getImportTaskRequest)),
                importTaskSuccessfulWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetPrivateGraphEndpointResponse>> waitUntilPrivateGraphEndpointAvailable(
            GetPrivateGraphEndpointRequest getPrivateGraphEndpointRequest) {
        return privateGraphEndpointAvailableWaiter.runAsync(() -> client
                .getPrivateGraphEndpoint(applyWaitersUserAgent(getPrivateGraphEndpointRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetPrivateGraphEndpointResponse>> waitUntilPrivateGraphEndpointAvailable(
            GetPrivateGraphEndpointRequest getPrivateGraphEndpointRequest, WaiterOverrideConfiguration overrideConfig) {
        return privateGraphEndpointAvailableWaiter.runAsync(
                () -> client.getPrivateGraphEndpoint(applyWaitersUserAgent(getPrivateGraphEndpointRequest)),
                privateGraphEndpointAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetPrivateGraphEndpointResponse>> waitUntilPrivateGraphEndpointDeleted(
            GetPrivateGraphEndpointRequest getPrivateGraphEndpointRequest) {
        return privateGraphEndpointDeletedWaiter.runAsync(() -> client
                .getPrivateGraphEndpoint(applyWaitersUserAgent(getPrivateGraphEndpointRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetPrivateGraphEndpointResponse>> waitUntilPrivateGraphEndpointDeleted(
            GetPrivateGraphEndpointRequest getPrivateGraphEndpointRequest, WaiterOverrideConfiguration overrideConfig) {
        return privateGraphEndpointDeletedWaiter.runAsync(
                () -> client.getPrivateGraphEndpoint(applyWaitersUserAgent(getPrivateGraphEndpointRequest)),
                privateGraphEndpointDeletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetExportTaskResponse>> exportTaskCancelledWaiterAcceptors() {
        List<WaiterAcceptor<? super GetExportTaskResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(
                            input.field("status").compare("!=", input.constant("CANCELLING"))
                                    .and(input.field("status").compare("!=", input.constant("CANCELLED"))).value(), true);
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status != 'CANCELLING' && status != 'CANCELLED'=true) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "CANCELLED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetExportTaskResponse>> exportTaskSuccessfulWaiterAcceptors() {
        List<WaiterAcceptor<? super GetExportTaskResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "CANCELLING");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=CANCELLING) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "CANCELLED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=CANCELLED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "FAILED");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "SUCCEEDED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetGraphResponse>> graphAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super GetGraphResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "DELETING");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=DELETING) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "FAILED");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "AVAILABLE");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetGraphResponse>> graphDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetGraphResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").compare("!=", input.constant("DELETING")).value(), true);
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status != 'DELETING'=true) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetGraphSnapshotResponse>> graphSnapshotAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super GetGraphSnapshotResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "DELETING");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=DELETING) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "FAILED");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "AVAILABLE");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetGraphSnapshotResponse>> graphSnapshotDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetGraphSnapshotResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").compare("!=", input.constant("DELETING")).value(), true);
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status != 'DELETING'=true) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetImportTaskResponse>> importTaskCancelledWaiterAcceptors() {
        List<WaiterAcceptor<? super GetImportTaskResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(
                            input.field("status").compare("!=", input.constant("CANCELLING"))
                                    .and(input.field("status").compare("!=", input.constant("CANCELLED"))).value(), true);
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status != 'CANCELLING' && status != 'CANCELLED'=true) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "CANCELLED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetImportTaskResponse>> importTaskSuccessfulWaiterAcceptors() {
        List<WaiterAcceptor<? super GetImportTaskResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "CANCELLING");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=CANCELLING) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "CANCELLED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=CANCELLED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "ROLLING_BACK");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=ROLLING_BACK) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "FAILED");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "SUCCEEDED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetPrivateGraphEndpointResponse>> privateGraphEndpointAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super GetPrivateGraphEndpointResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "DELETING");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=DELETING) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "FAILED");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "AVAILABLE");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetPrivateGraphEndpointResponse>> privateGraphEndpointDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetPrivateGraphEndpointResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").compare("!=", input.constant("DELETING")).value(), true);
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status != 'DELETING'=true) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration exportTaskCancelledWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration exportTaskSuccessfulWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(480);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration graphAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(480);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration graphDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration graphSnapshotAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration graphSnapshotDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration importTaskCancelledWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration importTaskSuccessfulWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(480);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration privateGraphEndpointAvailableWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(180);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration privateGraphEndpointDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(180);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static NeptuneGraphAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends NeptuneGraphRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements NeptuneGraphAsyncWaiter.Builder {
        private NeptuneGraphAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public NeptuneGraphAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public NeptuneGraphAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public NeptuneGraphAsyncWaiter.Builder client(NeptuneGraphAsyncClient client) {
            this.client = client;
            return this;
        }

        public NeptuneGraphAsyncWaiter build() {
            return new DefaultNeptuneGraphAsyncWaiter(this);
        }
    }
}
