/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.neptunegraph.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Summary information about the graph.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GraphDataSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<GraphDataSummary.Builder, GraphDataSummary> {
    private static final SdkField<Long> NUM_NODES_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("numNodes")
            .getter(getter(GraphDataSummary::numNodes)).setter(setter(Builder::numNodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("numNodes").build()).build();

    private static final SdkField<Long> NUM_EDGES_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("numEdges")
            .getter(getter(GraphDataSummary::numEdges)).setter(setter(Builder::numEdges))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("numEdges").build()).build();

    private static final SdkField<Long> NUM_NODE_LABELS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("numNodeLabels").getter(getter(GraphDataSummary::numNodeLabels)).setter(setter(Builder::numNodeLabels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("numNodeLabels").build()).build();

    private static final SdkField<Long> NUM_EDGE_LABELS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("numEdgeLabels").getter(getter(GraphDataSummary::numEdgeLabels)).setter(setter(Builder::numEdgeLabels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("numEdgeLabels").build()).build();

    private static final SdkField<List<String>> NODE_LABELS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("nodeLabels")
            .getter(getter(GraphDataSummary::nodeLabels))
            .setter(setter(Builder::nodeLabels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nodeLabels").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> EDGE_LABELS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("edgeLabels")
            .getter(getter(GraphDataSummary::edgeLabels))
            .setter(setter(Builder::edgeLabels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("edgeLabels").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Long> NUM_NODE_PROPERTIES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("numNodeProperties").getter(getter(GraphDataSummary::numNodeProperties))
            .setter(setter(Builder::numNodeProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("numNodeProperties").build()).build();

    private static final SdkField<Long> NUM_EDGE_PROPERTIES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("numEdgeProperties").getter(getter(GraphDataSummary::numEdgeProperties))
            .setter(setter(Builder::numEdgeProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("numEdgeProperties").build()).build();

    private static final SdkField<List<Map<String, Long>>> NODE_PROPERTIES_FIELD = SdkField
            .<List<Map<String, Long>>> builder(MarshallingType.LIST)
            .memberName("nodeProperties")
            .getter(getter(GraphDataSummary::nodeProperties))
            .setter(setter(Builder::nodeProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nodeProperties").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Map<String, Long>> builder(MarshallingType.MAP)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build(),
                                                    MapTrait.builder()
                                                            .keyLocationName("key")
                                                            .valueLocationName("value")
                                                            .valueFieldInfo(
                                                                    SdkField.<Long> builder(MarshallingType.LONG)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("value").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<List<Map<String, Long>>> EDGE_PROPERTIES_FIELD = SdkField
            .<List<Map<String, Long>>> builder(MarshallingType.LIST)
            .memberName("edgeProperties")
            .getter(getter(GraphDataSummary::edgeProperties))
            .setter(setter(Builder::edgeProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("edgeProperties").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Map<String, Long>> builder(MarshallingType.MAP)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build(),
                                                    MapTrait.builder()
                                                            .keyLocationName("key")
                                                            .valueLocationName("value")
                                                            .valueFieldInfo(
                                                                    SdkField.<Long> builder(MarshallingType.LONG)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("value").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<Long> TOTAL_NODE_PROPERTY_VALUES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("totalNodePropertyValues").getter(getter(GraphDataSummary::totalNodePropertyValues))
            .setter(setter(Builder::totalNodePropertyValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("totalNodePropertyValues").build())
            .build();

    private static final SdkField<Long> TOTAL_EDGE_PROPERTY_VALUES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("totalEdgePropertyValues").getter(getter(GraphDataSummary::totalEdgePropertyValues))
            .setter(setter(Builder::totalEdgePropertyValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("totalEdgePropertyValues").build())
            .build();

    private static final SdkField<List<NodeStructure>> NODE_STRUCTURES_FIELD = SdkField
            .<List<NodeStructure>> builder(MarshallingType.LIST)
            .memberName("nodeStructures")
            .getter(getter(GraphDataSummary::nodeStructures))
            .setter(setter(Builder::nodeStructures))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nodeStructures").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NodeStructure> builder(MarshallingType.SDK_POJO)
                                            .constructor(NodeStructure::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<EdgeStructure>> EDGE_STRUCTURES_FIELD = SdkField
            .<List<EdgeStructure>> builder(MarshallingType.LIST)
            .memberName("edgeStructures")
            .getter(getter(GraphDataSummary::edgeStructures))
            .setter(setter(Builder::edgeStructures))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("edgeStructures").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EdgeStructure> builder(MarshallingType.SDK_POJO)
                                            .constructor(EdgeStructure::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NUM_NODES_FIELD,
            NUM_EDGES_FIELD, NUM_NODE_LABELS_FIELD, NUM_EDGE_LABELS_FIELD, NODE_LABELS_FIELD, EDGE_LABELS_FIELD,
            NUM_NODE_PROPERTIES_FIELD, NUM_EDGE_PROPERTIES_FIELD, NODE_PROPERTIES_FIELD, EDGE_PROPERTIES_FIELD,
            TOTAL_NODE_PROPERTY_VALUES_FIELD, TOTAL_EDGE_PROPERTY_VALUES_FIELD, NODE_STRUCTURES_FIELD, EDGE_STRUCTURES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Long numNodes;

    private final Long numEdges;

    private final Long numNodeLabels;

    private final Long numEdgeLabels;

    private final List<String> nodeLabels;

    private final List<String> edgeLabels;

    private final Long numNodeProperties;

    private final Long numEdgeProperties;

    private final List<Map<String, Long>> nodeProperties;

    private final List<Map<String, Long>> edgeProperties;

    private final Long totalNodePropertyValues;

    private final Long totalEdgePropertyValues;

    private final List<NodeStructure> nodeStructures;

    private final List<EdgeStructure> edgeStructures;

    private GraphDataSummary(BuilderImpl builder) {
        this.numNodes = builder.numNodes;
        this.numEdges = builder.numEdges;
        this.numNodeLabels = builder.numNodeLabels;
        this.numEdgeLabels = builder.numEdgeLabels;
        this.nodeLabels = builder.nodeLabels;
        this.edgeLabels = builder.edgeLabels;
        this.numNodeProperties = builder.numNodeProperties;
        this.numEdgeProperties = builder.numEdgeProperties;
        this.nodeProperties = builder.nodeProperties;
        this.edgeProperties = builder.edgeProperties;
        this.totalNodePropertyValues = builder.totalNodePropertyValues;
        this.totalEdgePropertyValues = builder.totalEdgePropertyValues;
        this.nodeStructures = builder.nodeStructures;
        this.edgeStructures = builder.edgeStructures;
    }

    /**
     * <p>
     * The number of nodes in the graph.
     * </p>
     * 
     * @return The number of nodes in the graph.
     */
    public final Long numNodes() {
        return numNodes;
    }

    /**
     * <p>
     * The number of edges in the graph.
     * </p>
     * 
     * @return The number of edges in the graph.
     */
    public final Long numEdges() {
        return numEdges;
    }

    /**
     * <p>
     * The number of distinct node labels in the graph.
     * </p>
     * 
     * @return The number of distinct node labels in the graph.
     */
    public final Long numNodeLabels() {
        return numNodeLabels;
    }

    /**
     * <p>
     * The number of unique edge labels in the graph.
     * </p>
     * 
     * @return The number of unique edge labels in the graph.
     */
    public final Long numEdgeLabels() {
        return numEdgeLabels;
    }

    /**
     * For responses, this returns true if the service returned a value for the NodeLabels property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasNodeLabels() {
        return nodeLabels != null && !(nodeLabels instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of distinct node labels in the graph.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNodeLabels} method.
     * </p>
     * 
     * @return A list of distinct node labels in the graph.
     */
    public final List<String> nodeLabels() {
        return nodeLabels;
    }

    /**
     * For responses, this returns true if the service returned a value for the EdgeLabels property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEdgeLabels() {
        return edgeLabels != null && !(edgeLabels instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the edge labels in the graph.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEdgeLabels} method.
     * </p>
     * 
     * @return A list of the edge labels in the graph.
     */
    public final List<String> edgeLabels() {
        return edgeLabels;
    }

    /**
     * <p>
     * The number of distinct node properties in the graph.
     * </p>
     * 
     * @return The number of distinct node properties in the graph.
     */
    public final Long numNodeProperties() {
        return numNodeProperties;
    }

    /**
     * <p>
     * The number of edge properties in the graph.
     * </p>
     * 
     * @return The number of edge properties in the graph.
     */
    public final Long numEdgeProperties() {
        return numEdgeProperties;
    }

    /**
     * For responses, this returns true if the service returned a value for the NodeProperties property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNodeProperties() {
        return nodeProperties != null && !(nodeProperties instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the distinct node properties in the graph, along with the count of nodes where each property is used.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNodeProperties} method.
     * </p>
     * 
     * @return A list of the distinct node properties in the graph, along with the count of nodes where each property is
     *         used.
     */
    public final List<Map<String, Long>> nodeProperties() {
        return nodeProperties;
    }

    /**
     * For responses, this returns true if the service returned a value for the EdgeProperties property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEdgeProperties() {
        return edgeProperties != null && !(edgeProperties instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the distinct edge properties in the graph, along with the count of edges where each property is used.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEdgeProperties} method.
     * </p>
     * 
     * @return A list of the distinct edge properties in the graph, along with the count of edges where each property is
     *         used.
     */
    public final List<Map<String, Long>> edgeProperties() {
        return edgeProperties;
    }

    /**
     * <p>
     * The total number of usages of all node properties.
     * </p>
     * 
     * @return The total number of usages of all node properties.
     */
    public final Long totalNodePropertyValues() {
        return totalNodePropertyValues;
    }

    /**
     * <p>
     * The total number of usages of all edge properties.
     * </p>
     * 
     * @return The total number of usages of all edge properties.
     */
    public final Long totalEdgePropertyValues() {
        return totalEdgePropertyValues;
    }

    /**
     * For responses, this returns true if the service returned a value for the NodeStructures property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNodeStructures() {
        return nodeStructures != null && !(nodeStructures instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * This field is only present when the requested mode is DETAILED. It contains a list of node structures.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNodeStructures} method.
     * </p>
     * 
     * @return This field is only present when the requested mode is DETAILED. It contains a list of node structures.
     */
    public final List<NodeStructure> nodeStructures() {
        return nodeStructures;
    }

    /**
     * For responses, this returns true if the service returned a value for the EdgeStructures property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEdgeStructures() {
        return edgeStructures != null && !(edgeStructures instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * This field is only present when the requested mode is DETAILED. It contains a list of edge structures.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEdgeStructures} method.
     * </p>
     * 
     * @return This field is only present when the requested mode is DETAILED. It contains a list of edge structures.
     */
    public final List<EdgeStructure> edgeStructures() {
        return edgeStructures;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(numNodes());
        hashCode = 31 * hashCode + Objects.hashCode(numEdges());
        hashCode = 31 * hashCode + Objects.hashCode(numNodeLabels());
        hashCode = 31 * hashCode + Objects.hashCode(numEdgeLabels());
        hashCode = 31 * hashCode + Objects.hashCode(hasNodeLabels() ? nodeLabels() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEdgeLabels() ? edgeLabels() : null);
        hashCode = 31 * hashCode + Objects.hashCode(numNodeProperties());
        hashCode = 31 * hashCode + Objects.hashCode(numEdgeProperties());
        hashCode = 31 * hashCode + Objects.hashCode(hasNodeProperties() ? nodeProperties() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEdgeProperties() ? edgeProperties() : null);
        hashCode = 31 * hashCode + Objects.hashCode(totalNodePropertyValues());
        hashCode = 31 * hashCode + Objects.hashCode(totalEdgePropertyValues());
        hashCode = 31 * hashCode + Objects.hashCode(hasNodeStructures() ? nodeStructures() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEdgeStructures() ? edgeStructures() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GraphDataSummary)) {
            return false;
        }
        GraphDataSummary other = (GraphDataSummary) obj;
        return Objects.equals(numNodes(), other.numNodes()) && Objects.equals(numEdges(), other.numEdges())
                && Objects.equals(numNodeLabels(), other.numNodeLabels())
                && Objects.equals(numEdgeLabels(), other.numEdgeLabels()) && hasNodeLabels() == other.hasNodeLabels()
                && Objects.equals(nodeLabels(), other.nodeLabels()) && hasEdgeLabels() == other.hasEdgeLabels()
                && Objects.equals(edgeLabels(), other.edgeLabels())
                && Objects.equals(numNodeProperties(), other.numNodeProperties())
                && Objects.equals(numEdgeProperties(), other.numEdgeProperties())
                && hasNodeProperties() == other.hasNodeProperties() && Objects.equals(nodeProperties(), other.nodeProperties())
                && hasEdgeProperties() == other.hasEdgeProperties() && Objects.equals(edgeProperties(), other.edgeProperties())
                && Objects.equals(totalNodePropertyValues(), other.totalNodePropertyValues())
                && Objects.equals(totalEdgePropertyValues(), other.totalEdgePropertyValues())
                && hasNodeStructures() == other.hasNodeStructures() && Objects.equals(nodeStructures(), other.nodeStructures())
                && hasEdgeStructures() == other.hasEdgeStructures() && Objects.equals(edgeStructures(), other.edgeStructures());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GraphDataSummary").add("NumNodes", numNodes()).add("NumEdges", numEdges())
                .add("NumNodeLabels", numNodeLabels()).add("NumEdgeLabels", numEdgeLabels())
                .add("NodeLabels", hasNodeLabels() ? nodeLabels() : null)
                .add("EdgeLabels", hasEdgeLabels() ? edgeLabels() : null).add("NumNodeProperties", numNodeProperties())
                .add("NumEdgeProperties", numEdgeProperties())
                .add("NodeProperties", hasNodeProperties() ? nodeProperties() : null)
                .add("EdgeProperties", hasEdgeProperties() ? edgeProperties() : null)
                .add("TotalNodePropertyValues", totalNodePropertyValues())
                .add("TotalEdgePropertyValues", totalEdgePropertyValues())
                .add("NodeStructures", hasNodeStructures() ? nodeStructures() : null)
                .add("EdgeStructures", hasEdgeStructures() ? edgeStructures() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "numNodes":
            return Optional.ofNullable(clazz.cast(numNodes()));
        case "numEdges":
            return Optional.ofNullable(clazz.cast(numEdges()));
        case "numNodeLabels":
            return Optional.ofNullable(clazz.cast(numNodeLabels()));
        case "numEdgeLabels":
            return Optional.ofNullable(clazz.cast(numEdgeLabels()));
        case "nodeLabels":
            return Optional.ofNullable(clazz.cast(nodeLabels()));
        case "edgeLabels":
            return Optional.ofNullable(clazz.cast(edgeLabels()));
        case "numNodeProperties":
            return Optional.ofNullable(clazz.cast(numNodeProperties()));
        case "numEdgeProperties":
            return Optional.ofNullable(clazz.cast(numEdgeProperties()));
        case "nodeProperties":
            return Optional.ofNullable(clazz.cast(nodeProperties()));
        case "edgeProperties":
            return Optional.ofNullable(clazz.cast(edgeProperties()));
        case "totalNodePropertyValues":
            return Optional.ofNullable(clazz.cast(totalNodePropertyValues()));
        case "totalEdgePropertyValues":
            return Optional.ofNullable(clazz.cast(totalEdgePropertyValues()));
        case "nodeStructures":
            return Optional.ofNullable(clazz.cast(nodeStructures()));
        case "edgeStructures":
            return Optional.ofNullable(clazz.cast(edgeStructures()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("numNodes", NUM_NODES_FIELD);
        map.put("numEdges", NUM_EDGES_FIELD);
        map.put("numNodeLabels", NUM_NODE_LABELS_FIELD);
        map.put("numEdgeLabels", NUM_EDGE_LABELS_FIELD);
        map.put("nodeLabels", NODE_LABELS_FIELD);
        map.put("edgeLabels", EDGE_LABELS_FIELD);
        map.put("numNodeProperties", NUM_NODE_PROPERTIES_FIELD);
        map.put("numEdgeProperties", NUM_EDGE_PROPERTIES_FIELD);
        map.put("nodeProperties", NODE_PROPERTIES_FIELD);
        map.put("edgeProperties", EDGE_PROPERTIES_FIELD);
        map.put("totalNodePropertyValues", TOTAL_NODE_PROPERTY_VALUES_FIELD);
        map.put("totalEdgePropertyValues", TOTAL_EDGE_PROPERTY_VALUES_FIELD);
        map.put("nodeStructures", NODE_STRUCTURES_FIELD);
        map.put("edgeStructures", EDGE_STRUCTURES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<GraphDataSummary, T> g) {
        return obj -> g.apply((GraphDataSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GraphDataSummary> {
        /**
         * <p>
         * The number of nodes in the graph.
         * </p>
         * 
         * @param numNodes
         *        The number of nodes in the graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numNodes(Long numNodes);

        /**
         * <p>
         * The number of edges in the graph.
         * </p>
         * 
         * @param numEdges
         *        The number of edges in the graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numEdges(Long numEdges);

        /**
         * <p>
         * The number of distinct node labels in the graph.
         * </p>
         * 
         * @param numNodeLabels
         *        The number of distinct node labels in the graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numNodeLabels(Long numNodeLabels);

        /**
         * <p>
         * The number of unique edge labels in the graph.
         * </p>
         * 
         * @param numEdgeLabels
         *        The number of unique edge labels in the graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numEdgeLabels(Long numEdgeLabels);

        /**
         * <p>
         * A list of distinct node labels in the graph.
         * </p>
         * 
         * @param nodeLabels
         *        A list of distinct node labels in the graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeLabels(Collection<String> nodeLabels);

        /**
         * <p>
         * A list of distinct node labels in the graph.
         * </p>
         * 
         * @param nodeLabels
         *        A list of distinct node labels in the graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeLabels(String... nodeLabels);

        /**
         * <p>
         * A list of the edge labels in the graph.
         * </p>
         * 
         * @param edgeLabels
         *        A list of the edge labels in the graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder edgeLabels(Collection<String> edgeLabels);

        /**
         * <p>
         * A list of the edge labels in the graph.
         * </p>
         * 
         * @param edgeLabels
         *        A list of the edge labels in the graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder edgeLabels(String... edgeLabels);

        /**
         * <p>
         * The number of distinct node properties in the graph.
         * </p>
         * 
         * @param numNodeProperties
         *        The number of distinct node properties in the graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numNodeProperties(Long numNodeProperties);

        /**
         * <p>
         * The number of edge properties in the graph.
         * </p>
         * 
         * @param numEdgeProperties
         *        The number of edge properties in the graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numEdgeProperties(Long numEdgeProperties);

        /**
         * <p>
         * A list of the distinct node properties in the graph, along with the count of nodes where each property is
         * used.
         * </p>
         * 
         * @param nodeProperties
         *        A list of the distinct node properties in the graph, along with the count of nodes where each property
         *        is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeProperties(Collection<? extends Map<String, Long>> nodeProperties);

        /**
         * <p>
         * A list of the distinct node properties in the graph, along with the count of nodes where each property is
         * used.
         * </p>
         * 
         * @param nodeProperties
         *        A list of the distinct node properties in the graph, along with the count of nodes where each property
         *        is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeProperties(Map<String, Long>... nodeProperties);

        /**
         * <p>
         * A list of the distinct edge properties in the graph, along with the count of edges where each property is
         * used.
         * </p>
         * 
         * @param edgeProperties
         *        A list of the distinct edge properties in the graph, along with the count of edges where each property
         *        is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder edgeProperties(Collection<? extends Map<String, Long>> edgeProperties);

        /**
         * <p>
         * A list of the distinct edge properties in the graph, along with the count of edges where each property is
         * used.
         * </p>
         * 
         * @param edgeProperties
         *        A list of the distinct edge properties in the graph, along with the count of edges where each property
         *        is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder edgeProperties(Map<String, Long>... edgeProperties);

        /**
         * <p>
         * The total number of usages of all node properties.
         * </p>
         * 
         * @param totalNodePropertyValues
         *        The total number of usages of all node properties.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalNodePropertyValues(Long totalNodePropertyValues);

        /**
         * <p>
         * The total number of usages of all edge properties.
         * </p>
         * 
         * @param totalEdgePropertyValues
         *        The total number of usages of all edge properties.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalEdgePropertyValues(Long totalEdgePropertyValues);

        /**
         * <p>
         * This field is only present when the requested mode is DETAILED. It contains a list of node structures.
         * </p>
         * 
         * @param nodeStructures
         *        This field is only present when the requested mode is DETAILED. It contains a list of node structures.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeStructures(Collection<NodeStructure> nodeStructures);

        /**
         * <p>
         * This field is only present when the requested mode is DETAILED. It contains a list of node structures.
         * </p>
         * 
         * @param nodeStructures
         *        This field is only present when the requested mode is DETAILED. It contains a list of node structures.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeStructures(NodeStructure... nodeStructures);

        /**
         * <p>
         * This field is only present when the requested mode is DETAILED. It contains a list of node structures.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.neptunegraph.model.NodeStructure.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.neptunegraph.model.NodeStructure#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.neptunegraph.model.NodeStructure.Builder#build()} is called
         * immediately and its result is passed to {@link #nodeStructures(List<NodeStructure>)}.
         * 
         * @param nodeStructures
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.neptunegraph.model.NodeStructure.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nodeStructures(java.util.Collection<NodeStructure>)
         */
        Builder nodeStructures(Consumer<NodeStructure.Builder>... nodeStructures);

        /**
         * <p>
         * This field is only present when the requested mode is DETAILED. It contains a list of edge structures.
         * </p>
         * 
         * @param edgeStructures
         *        This field is only present when the requested mode is DETAILED. It contains a list of edge structures.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder edgeStructures(Collection<EdgeStructure> edgeStructures);

        /**
         * <p>
         * This field is only present when the requested mode is DETAILED. It contains a list of edge structures.
         * </p>
         * 
         * @param edgeStructures
         *        This field is only present when the requested mode is DETAILED. It contains a list of edge structures.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder edgeStructures(EdgeStructure... edgeStructures);

        /**
         * <p>
         * This field is only present when the requested mode is DETAILED. It contains a list of edge structures.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.neptunegraph.model.EdgeStructure.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.neptunegraph.model.EdgeStructure#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.neptunegraph.model.EdgeStructure.Builder#build()} is called
         * immediately and its result is passed to {@link #edgeStructures(List<EdgeStructure>)}.
         * 
         * @param edgeStructures
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.neptunegraph.model.EdgeStructure.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #edgeStructures(java.util.Collection<EdgeStructure>)
         */
        Builder edgeStructures(Consumer<EdgeStructure.Builder>... edgeStructures);
    }

    static final class BuilderImpl implements Builder {
        private Long numNodes;

        private Long numEdges;

        private Long numNodeLabels;

        private Long numEdgeLabels;

        private List<String> nodeLabels = DefaultSdkAutoConstructList.getInstance();

        private List<String> edgeLabels = DefaultSdkAutoConstructList.getInstance();

        private Long numNodeProperties;

        private Long numEdgeProperties;

        private List<Map<String, Long>> nodeProperties = DefaultSdkAutoConstructList.getInstance();

        private List<Map<String, Long>> edgeProperties = DefaultSdkAutoConstructList.getInstance();

        private Long totalNodePropertyValues;

        private Long totalEdgePropertyValues;

        private List<NodeStructure> nodeStructures = DefaultSdkAutoConstructList.getInstance();

        private List<EdgeStructure> edgeStructures = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GraphDataSummary model) {
            numNodes(model.numNodes);
            numEdges(model.numEdges);
            numNodeLabels(model.numNodeLabels);
            numEdgeLabels(model.numEdgeLabels);
            nodeLabels(model.nodeLabels);
            edgeLabels(model.edgeLabels);
            numNodeProperties(model.numNodeProperties);
            numEdgeProperties(model.numEdgeProperties);
            nodeProperties(model.nodeProperties);
            edgeProperties(model.edgeProperties);
            totalNodePropertyValues(model.totalNodePropertyValues);
            totalEdgePropertyValues(model.totalEdgePropertyValues);
            nodeStructures(model.nodeStructures);
            edgeStructures(model.edgeStructures);
        }

        public final Long getNumNodes() {
            return numNodes;
        }

        public final void setNumNodes(Long numNodes) {
            this.numNodes = numNodes;
        }

        @Override
        public final Builder numNodes(Long numNodes) {
            this.numNodes = numNodes;
            return this;
        }

        public final Long getNumEdges() {
            return numEdges;
        }

        public final void setNumEdges(Long numEdges) {
            this.numEdges = numEdges;
        }

        @Override
        public final Builder numEdges(Long numEdges) {
            this.numEdges = numEdges;
            return this;
        }

        public final Long getNumNodeLabels() {
            return numNodeLabels;
        }

        public final void setNumNodeLabels(Long numNodeLabels) {
            this.numNodeLabels = numNodeLabels;
        }

        @Override
        public final Builder numNodeLabels(Long numNodeLabels) {
            this.numNodeLabels = numNodeLabels;
            return this;
        }

        public final Long getNumEdgeLabels() {
            return numEdgeLabels;
        }

        public final void setNumEdgeLabels(Long numEdgeLabels) {
            this.numEdgeLabels = numEdgeLabels;
        }

        @Override
        public final Builder numEdgeLabels(Long numEdgeLabels) {
            this.numEdgeLabels = numEdgeLabels;
            return this;
        }

        public final Collection<String> getNodeLabels() {
            if (nodeLabels instanceof SdkAutoConstructList) {
                return null;
            }
            return nodeLabels;
        }

        public final void setNodeLabels(Collection<String> nodeLabels) {
            this.nodeLabels = NodeLabelsCopier.copy(nodeLabels);
        }

        @Override
        public final Builder nodeLabels(Collection<String> nodeLabels) {
            this.nodeLabels = NodeLabelsCopier.copy(nodeLabels);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodeLabels(String... nodeLabels) {
            nodeLabels(Arrays.asList(nodeLabels));
            return this;
        }

        public final Collection<String> getEdgeLabels() {
            if (edgeLabels instanceof SdkAutoConstructList) {
                return null;
            }
            return edgeLabels;
        }

        public final void setEdgeLabels(Collection<String> edgeLabels) {
            this.edgeLabels = EdgeLabelsCopier.copy(edgeLabels);
        }

        @Override
        public final Builder edgeLabels(Collection<String> edgeLabels) {
            this.edgeLabels = EdgeLabelsCopier.copy(edgeLabels);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder edgeLabels(String... edgeLabels) {
            edgeLabels(Arrays.asList(edgeLabels));
            return this;
        }

        public final Long getNumNodeProperties() {
            return numNodeProperties;
        }

        public final void setNumNodeProperties(Long numNodeProperties) {
            this.numNodeProperties = numNodeProperties;
        }

        @Override
        public final Builder numNodeProperties(Long numNodeProperties) {
            this.numNodeProperties = numNodeProperties;
            return this;
        }

        public final Long getNumEdgeProperties() {
            return numEdgeProperties;
        }

        public final void setNumEdgeProperties(Long numEdgeProperties) {
            this.numEdgeProperties = numEdgeProperties;
        }

        @Override
        public final Builder numEdgeProperties(Long numEdgeProperties) {
            this.numEdgeProperties = numEdgeProperties;
            return this;
        }

        public final Collection<? extends Map<String, Long>> getNodeProperties() {
            if (nodeProperties instanceof SdkAutoConstructList) {
                return null;
            }
            return nodeProperties;
        }

        public final void setNodeProperties(Collection<? extends Map<String, Long>> nodeProperties) {
            this.nodeProperties = LongValuedMapListCopier.copy(nodeProperties);
        }

        @Override
        public final Builder nodeProperties(Collection<? extends Map<String, Long>> nodeProperties) {
            this.nodeProperties = LongValuedMapListCopier.copy(nodeProperties);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodeProperties(Map<String, Long>... nodeProperties) {
            nodeProperties(Arrays.asList(nodeProperties));
            return this;
        }

        public final Collection<? extends Map<String, Long>> getEdgeProperties() {
            if (edgeProperties instanceof SdkAutoConstructList) {
                return null;
            }
            return edgeProperties;
        }

        public final void setEdgeProperties(Collection<? extends Map<String, Long>> edgeProperties) {
            this.edgeProperties = LongValuedMapListCopier.copy(edgeProperties);
        }

        @Override
        public final Builder edgeProperties(Collection<? extends Map<String, Long>> edgeProperties) {
            this.edgeProperties = LongValuedMapListCopier.copy(edgeProperties);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder edgeProperties(Map<String, Long>... edgeProperties) {
            edgeProperties(Arrays.asList(edgeProperties));
            return this;
        }

        public final Long getTotalNodePropertyValues() {
            return totalNodePropertyValues;
        }

        public final void setTotalNodePropertyValues(Long totalNodePropertyValues) {
            this.totalNodePropertyValues = totalNodePropertyValues;
        }

        @Override
        public final Builder totalNodePropertyValues(Long totalNodePropertyValues) {
            this.totalNodePropertyValues = totalNodePropertyValues;
            return this;
        }

        public final Long getTotalEdgePropertyValues() {
            return totalEdgePropertyValues;
        }

        public final void setTotalEdgePropertyValues(Long totalEdgePropertyValues) {
            this.totalEdgePropertyValues = totalEdgePropertyValues;
        }

        @Override
        public final Builder totalEdgePropertyValues(Long totalEdgePropertyValues) {
            this.totalEdgePropertyValues = totalEdgePropertyValues;
            return this;
        }

        public final List<NodeStructure.Builder> getNodeStructures() {
            List<NodeStructure.Builder> result = NodeStructuresCopier.copyToBuilder(this.nodeStructures);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNodeStructures(Collection<NodeStructure.BuilderImpl> nodeStructures) {
            this.nodeStructures = NodeStructuresCopier.copyFromBuilder(nodeStructures);
        }

        @Override
        public final Builder nodeStructures(Collection<NodeStructure> nodeStructures) {
            this.nodeStructures = NodeStructuresCopier.copy(nodeStructures);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodeStructures(NodeStructure... nodeStructures) {
            nodeStructures(Arrays.asList(nodeStructures));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodeStructures(Consumer<NodeStructure.Builder>... nodeStructures) {
            nodeStructures(Stream.of(nodeStructures).map(c -> NodeStructure.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<EdgeStructure.Builder> getEdgeStructures() {
            List<EdgeStructure.Builder> result = EdgeStructuresCopier.copyToBuilder(this.edgeStructures);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEdgeStructures(Collection<EdgeStructure.BuilderImpl> edgeStructures) {
            this.edgeStructures = EdgeStructuresCopier.copyFromBuilder(edgeStructures);
        }

        @Override
        public final Builder edgeStructures(Collection<EdgeStructure> edgeStructures) {
            this.edgeStructures = EdgeStructuresCopier.copy(edgeStructures);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder edgeStructures(EdgeStructure... edgeStructures) {
            edgeStructures(Arrays.asList(edgeStructures));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder edgeStructures(Consumer<EdgeStructure.Builder>... edgeStructures) {
            edgeStructures(Stream.of(edgeStructures).map(c -> EdgeStructure.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public GraphDataSummary build() {
            return new GraphDataSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
