/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.http.nio.netty.internal;

import com.typesafe.netty.http.HttpStreamsClientHandler;
import com.typesafe.netty.http.StreamedHttpRequest;
import io.netty.buffer.ByteBuf;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelOption;
import io.netty.handler.codec.DecoderResult;
import io.netty.handler.codec.http.DefaultHttpContent;
import io.netty.handler.codec.http.HttpContent;
import io.netty.handler.codec.http.HttpHeaders;
import io.netty.handler.codec.http.HttpMethod;
import io.netty.handler.codec.http.HttpRequest;
import io.netty.handler.codec.http.HttpVersion;
import io.netty.handler.timeout.ReadTimeoutException;
import io.netty.handler.timeout.ReadTimeoutHandler;
import io.netty.handler.timeout.WriteTimeoutException;
import io.netty.handler.timeout.WriteTimeoutHandler;
import java.io.IOException;
import java.net.URI;
import java.nio.ByteBuffer;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;
import org.reactivestreams.Publisher;
import org.reactivestreams.Subscriber;
import org.reactivestreams.Subscription;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.http.Protocol;
import software.amazon.awssdk.http.async.AbortableRunnable;
import software.amazon.awssdk.http.nio.netty.internal.ChannelAttributeKey;
import software.amazon.awssdk.http.nio.netty.internal.RequestContext;
import software.amazon.awssdk.http.nio.netty.internal.ResponseHandler;
import software.amazon.awssdk.http.nio.netty.internal.http2.Http2ToHttpInboundAdapter;
import software.amazon.awssdk.http.nio.netty.internal.http2.HttpToHttp2OutboundAdapter;
import software.amazon.awssdk.http.nio.netty.internal.utils.ChannelUtils;
import software.amazon.awssdk.utils.FunctionalUtils;

@SdkInternalApi
public final class RunnableRequest
implements AbortableRunnable {
    private static final Logger log = LoggerFactory.getLogger(RunnableRequest.class);
    private final RequestContext context;
    private volatile Channel channel;

    public RunnableRequest(RequestContext context) {
        this.context = context;
    }

    public void run() {
        this.context.channelPool().acquire().addListener(channelFuture -> {
            if (channelFuture.isSuccess()) {
                try {
                    this.channel = (Channel)channelFuture.getNow();
                    this.channel.attr(ChannelAttributeKey.REQUEST_CONTEXT_KEY).set((Object)this.context);
                    this.channel.attr(ChannelAttributeKey.RESPONSE_COMPLETE_KEY).set((Object)false);
                    this.makeRequest(this.context.nettyRequest());
                }
                catch (Exception e) {
                    this.handleFailure(() -> "Failed to make request to " + this.endpoint(), e);
                }
            } else {
                this.handleFailure(() -> "Failed to create connection to " + this.endpoint(), channelFuture.cause());
            }
        });
    }

    public void abort() {
        if (this.channel != null) {
            RunnableRequest.closeAndRelease(this.channel);
        }
    }

    private void makeRequest(HttpRequest request) {
        log.debug("Writing request: {}", (Object)request);
        this.runOrFail(() -> {
            this.configurePipeline();
            this.writeRequest(request);
        }, () -> "Failed to make request to " + this.endpoint());
    }

    private void configurePipeline() {
        Protocol protocol = ChannelAttributeKey.getProtocolNow(this.channel);
        if (Protocol.HTTP2.equals((Object)protocol)) {
            this.channel.pipeline().addLast(new ChannelHandler[]{new Http2ToHttpInboundAdapter()});
            this.channel.pipeline().addLast(new ChannelHandler[]{new HttpToHttp2OutboundAdapter()});
        } else if (!Protocol.HTTP1_1.equals((Object)protocol)) {
            throw new RuntimeException("Unknown protocol: " + protocol);
        }
        this.channel.config().setOption(ChannelOption.AUTO_READ, (Object)false);
        this.channel.pipeline().addLast(new ChannelHandler[]{new HttpStreamsClientHandler()});
        this.channel.pipeline().addLast(new ChannelHandler[]{new ResponseHandler()});
    }

    private void writeRequest(HttpRequest request) {
        this.channel.pipeline().addFirst(new ChannelHandler[]{new WriteTimeoutHandler((long)this.context.configuration().writeTimeoutMillis(), TimeUnit.MILLISECONDS)});
        this.channel.writeAndFlush((Object)new StreamedRequest(request, (Publisher<ByteBuffer>)this.context.sdkRequestProvider(), this.channel)).addListener(wireCall -> {
            ChannelUtils.removeIfExists(this.channel.pipeline(), WriteTimeoutHandler.class);
            if (wireCall.isSuccess()) {
                this.channel.pipeline().addFirst(new ChannelHandler[]{new ReadTimeoutHandler((long)this.context.configuration().readTimeoutMillis(), TimeUnit.MILLISECONDS)});
                this.channel.read();
            } else {
                this.handleFailure(() -> "Failed to make request to " + this.endpoint(), wireCall.cause());
            }
        });
    }

    private URI endpoint() {
        return this.context.sdkRequest().getUri();
    }

    private void runOrFail(Runnable runnable, Supplier<String> errorMsgSupplier) {
        try {
            runnable.run();
        }
        catch (Exception e) {
            this.handleFailure(errorMsgSupplier, e);
        }
    }

    private void handleFailure(Supplier<String> msg, Throwable cause) {
        log.error(msg.get(), cause);
        Throwable throwable = this.decorateException(cause);
        RunnableRequest.runAndLogError("Exception thrown from AsyncResponseHandler", () -> this.context.handler().exceptionOccurred(throwable));
        if (this.channel != null) {
            RunnableRequest.runAndLogError("Unable to release channel back to the pool.", () -> RunnableRequest.closeAndRelease(this.channel));
        }
    }

    private Throwable decorateException(Throwable originalCause) {
        if (this.isAcquireTimeoutException(originalCause)) {
            return new Throwable(this.getMessageForAcquireTimeoutException(), originalCause);
        }
        if (this.isTooManyPendingAcquiresException(originalCause)) {
            return new Throwable(this.getMessageForTooManyAcquireOperationsError(), originalCause);
        }
        if (originalCause instanceof ReadTimeoutException) {
            return new IOException("Read timed out", originalCause);
        }
        if (originalCause instanceof WriteTimeoutException) {
            return new IOException("Write timed out", originalCause);
        }
        return originalCause;
    }

    private boolean isAcquireTimeoutException(Throwable originalCause) {
        return originalCause instanceof TimeoutException && originalCause.getMessage().contains("Acquire operation took longer");
    }

    private boolean isTooManyPendingAcquiresException(Throwable originalCause) {
        return originalCause instanceof IllegalStateException && originalCause.getMessage().contains("Too many outstanding acquire operations");
    }

    private String getMessageForAcquireTimeoutException() {
        return "Acquire operation took longer than the configured maximum time. This indicates that a request cannot get a connection from the pool within the specified maximum time. This can be due to high request rate.\nConsider taking any of the following actions to mitigate the issue: increase max connections, increase acquire timeout, or slowing the request rate.\nIncreasing the max connections can increase client throughput (unless the network interface is already fully utilized), but can eventually start to hit operation system limitations on the number of file descriptors used by the process. If you already are fully utilizing your network interface or cannot further increase your connection count, increasing the acquire timeout gives extra time for requests to acquire a connection before timing out. If the connections doesn't free up, the subsequent requests will still timeout.\nIf the above mechanisms are not able to fix the issue, try smoothing out your requests so that large traffic bursts cannot overload the client, being more efficient with the number of times you need to call AWS, or by increasing the number of hosts sending requests.";
    }

    private String getMessageForTooManyAcquireOperationsError() {
        return "Maximum pending connection acquisitions exceeded. The request rate is too high for the client to keep up.\nConsider taking any of the following actions to mitigate the issue: increase max connections, increase max pending acquire count, decrease pool lease timeout, or slowing the request rate.\nIncreasing the max connections can increase client throughput (unless the network interface is already fully utilized), but can eventually start to hit operation system limitations on the number of file descriptors used by the process. If you already are fully utilizing your network interface or cannot further increase your connection count, increasing the pending acquire count allows extra requests to be buffered by the client, but can cause additional request latency and higher memory usage. If your request latency or memory usage is already too high, decreasing the lease timeout will allow requests to fail more quickly, reducing the number of pending connection acquisitions, but likely won't decrease the total number of failed requests.\nIf the above mechanisms are not able to fix the issue, try smoothing out your requests so that large traffic bursts cannot overload the client, being more efficient with the number of times you need to call AWS, or by increasing the number of hosts sending requests.";
    }

    private static void closeAndRelease(Channel channel) {
        RequestContext requestCtx = (RequestContext)channel.attr(ChannelAttributeKey.REQUEST_CONTEXT_KEY).get();
        channel.close().addListener(ignored -> requestCtx.channelPool().release(channel));
    }

    private static void runAndLogError(String errorMsg, FunctionalUtils.UnsafeRunnable runnable) {
        try {
            runnable.run();
        }
        catch (Exception e) {
            log.error(errorMsg, (Throwable)e);
        }
    }

    private static class StreamedRequest
    extends DelegateHttpRequest
    implements StreamedHttpRequest {
        private final Publisher<ByteBuffer> publisher;
        private final Channel channel;
        private final Optional<Long> requestContentLength;
        private long written = 0L;
        private boolean done;
        private Subscription subscription;

        StreamedRequest(HttpRequest request, Publisher<ByteBuffer> publisher, Channel channel) {
            super(request);
            this.publisher = publisher;
            this.channel = channel;
            this.requestContentLength = StreamedRequest.contentLength(request);
        }

        public void subscribe(final Subscriber<? super HttpContent> subscriber) {
            this.publisher.subscribe((Subscriber)new Subscriber<ByteBuffer>(){

                public void onSubscribe(Subscription subscription) {
                    subscription = subscription;
                    subscriber.onSubscribe(subscription);
                }

                public void onNext(ByteBuffer byteBuffer) {
                    if (done) {
                        return;
                    }
                    int newLimit = this.clampedBufferLimit(byteBuffer.remaining());
                    byteBuffer.limit(newLimit);
                    ByteBuf buffer = channel.alloc().buffer(byteBuffer.remaining());
                    buffer.writeBytes(byteBuffer);
                    DefaultHttpContent content = new DefaultHttpContent(buffer);
                    subscriber.onNext((Object)content);
                    written = written + (long)newLimit;
                    if (!this.shouldContinuePublishing()) {
                        done = true;
                        subscription.cancel();
                        subscriber.onComplete();
                    }
                }

                public void onError(Throwable t) {
                    if (!done) {
                        done = true;
                        subscriber.onError(t);
                    }
                }

                public void onComplete() {
                    if (!done) {
                        done = true;
                        subscriber.onComplete();
                    }
                }
            });
        }

        private int clampedBufferLimit(int bufLen) {
            return this.requestContentLength.map(cl -> (int)Math.min(cl - this.written, (long)bufLen)).orElse(bufLen);
        }

        private boolean shouldContinuePublishing() {
            return this.requestContentLength.map(cl -> this.written < cl).orElse(true);
        }

        private static Optional<Long> contentLength(HttpRequest request) {
            String value = request.headers().get("Content-Length");
            if (value != null) {
                try {
                    return Optional.of(Long.parseLong(value));
                }
                catch (NumberFormatException e) {
                    log.warn("Unable  to parse 'Content-Length' header. Treating it as non existent.");
                }
            }
            return Optional.empty();
        }
    }

    static class DelegateHttpRequest
    implements HttpRequest {
        protected final HttpRequest request;

        DelegateHttpRequest(HttpRequest request) {
            this.request = request;
        }

        public HttpRequest setMethod(HttpMethod method) {
            this.request.setMethod(method);
            return this;
        }

        public HttpRequest setUri(String uri) {
            this.request.setUri(uri);
            return this;
        }

        public HttpMethod getMethod() {
            return this.request.method();
        }

        public HttpMethod method() {
            return this.request.method();
        }

        public String getUri() {
            return this.request.uri();
        }

        public String uri() {
            return this.request.uri();
        }

        public HttpVersion getProtocolVersion() {
            return this.request.protocolVersion();
        }

        public HttpVersion protocolVersion() {
            return this.request.protocolVersion();
        }

        public HttpRequest setProtocolVersion(HttpVersion version) {
            this.request.setProtocolVersion(version);
            return this;
        }

        public HttpHeaders headers() {
            return this.request.headers();
        }

        public DecoderResult getDecoderResult() {
            return this.request.decoderResult();
        }

        public DecoderResult decoderResult() {
            return this.request.decoderResult();
        }

        public void setDecoderResult(DecoderResult result) {
            this.request.setDecoderResult(result);
        }

        public String toString() {
            return this.getClass().getName() + "(" + this.request.toString() + ")";
        }
    }
}

