/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkfirewall.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The status of the firewall endpoint and firewall policy configuration for a single VPC subnet.
 * </p>
 * <p>
 * For each VPC subnet that you associate with a firewall, AWS Network Firewall does the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Instantiates a firewall endpoint in the subnet, ready to take traffic.
 * </p>
 * </li>
 * <li>
 * <p>
 * Configures the endpoint with the current firewall policy settings, to provide the filtering behavior for the
 * endpoint.
 * </p>
 * </li>
 * </ul>
 * <p>
 * When you update a firewall, for example to add a subnet association or change a rule group in the firewall policy,
 * the affected sync states reflect out-of-sync or not ready status until the changes are complete.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SyncState implements SdkPojo, Serializable, ToCopyableBuilder<SyncState.Builder, SyncState> {
    private static final SdkField<Attachment> ATTACHMENT_FIELD = SdkField.<Attachment> builder(MarshallingType.SDK_POJO)
            .memberName("Attachment").getter(getter(SyncState::attachment)).setter(setter(Builder::attachment))
            .constructor(Attachment::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attachment").build()).build();

    private static final SdkField<Map<String, PerObjectStatus>> CONFIG_FIELD = SdkField
            .<Map<String, PerObjectStatus>> builder(MarshallingType.MAP)
            .memberName("Config")
            .getter(getter(SyncState::config))
            .setter(setter(Builder::config))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Config").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<PerObjectStatus> builder(MarshallingType.SDK_POJO)
                                            .constructor(PerObjectStatus::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(ATTACHMENT_FIELD, CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final Attachment attachment;

    private final Map<String, PerObjectStatus> config;

    private SyncState(BuilderImpl builder) {
        this.attachment = builder.attachment;
        this.config = builder.config;
    }

    /**
     * <p>
     * The attachment status of the firewall's association with a single VPC subnet. For each configured subnet, Network
     * Firewall creates the attachment by instantiating the firewall endpoint in the subnet so that it's ready to take
     * traffic. This is part of the <a>FirewallStatus</a>.
     * </p>
     * 
     * @return The attachment status of the firewall's association with a single VPC subnet. For each configured subnet,
     *         Network Firewall creates the attachment by instantiating the firewall endpoint in the subnet so that it's
     *         ready to take traffic. This is part of the <a>FirewallStatus</a>.
     */
    public final Attachment attachment() {
        return attachment;
    }

    /**
     * For responses, this returns true if the service returned a value for the Config property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasConfig() {
        return config != null && !(config instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The configuration status of the firewall endpoint in a single VPC subnet. Network Firewall provides each endpoint
     * with the rules that are configured in the firewall policy. Each time you add a subnet or modify the associated
     * firewall policy, Network Firewall synchronizes the rules in the endpoint, so it can properly filter network
     * traffic. This is part of the <a>FirewallStatus</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConfig} method.
     * </p>
     * 
     * @return The configuration status of the firewall endpoint in a single VPC subnet. Network Firewall provides each
     *         endpoint with the rules that are configured in the firewall policy. Each time you add a subnet or modify
     *         the associated firewall policy, Network Firewall synchronizes the rules in the endpoint, so it can
     *         properly filter network traffic. This is part of the <a>FirewallStatus</a>.
     */
    public final Map<String, PerObjectStatus> config() {
        return config;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(attachment());
        hashCode = 31 * hashCode + Objects.hashCode(hasConfig() ? config() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SyncState)) {
            return false;
        }
        SyncState other = (SyncState) obj;
        return Objects.equals(attachment(), other.attachment()) && hasConfig() == other.hasConfig()
                && Objects.equals(config(), other.config());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SyncState").add("Attachment", attachment()).add("Config", hasConfig() ? config() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Attachment":
            return Optional.ofNullable(clazz.cast(attachment()));
        case "Config":
            return Optional.ofNullable(clazz.cast(config()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SyncState, T> g) {
        return obj -> g.apply((SyncState) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SyncState> {
        /**
         * <p>
         * The attachment status of the firewall's association with a single VPC subnet. For each configured subnet,
         * Network Firewall creates the attachment by instantiating the firewall endpoint in the subnet so that it's
         * ready to take traffic. This is part of the <a>FirewallStatus</a>.
         * </p>
         * 
         * @param attachment
         *        The attachment status of the firewall's association with a single VPC subnet. For each configured
         *        subnet, Network Firewall creates the attachment by instantiating the firewall endpoint in the subnet
         *        so that it's ready to take traffic. This is part of the <a>FirewallStatus</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachment(Attachment attachment);

        /**
         * <p>
         * The attachment status of the firewall's association with a single VPC subnet. For each configured subnet,
         * Network Firewall creates the attachment by instantiating the firewall endpoint in the subnet so that it's
         * ready to take traffic. This is part of the <a>FirewallStatus</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link Attachment.Builder} avoiding the need to create
         * one manually via {@link Attachment#builder()}.
         *
         * When the {@link Consumer} completes, {@link Attachment.Builder#build()} is called immediately and its result
         * is passed to {@link #attachment(Attachment)}.
         * 
         * @param attachment
         *        a consumer that will call methods on {@link Attachment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attachment(Attachment)
         */
        default Builder attachment(Consumer<Attachment.Builder> attachment) {
            return attachment(Attachment.builder().applyMutation(attachment).build());
        }

        /**
         * <p>
         * The configuration status of the firewall endpoint in a single VPC subnet. Network Firewall provides each
         * endpoint with the rules that are configured in the firewall policy. Each time you add a subnet or modify the
         * associated firewall policy, Network Firewall synchronizes the rules in the endpoint, so it can properly
         * filter network traffic. This is part of the <a>FirewallStatus</a>.
         * </p>
         * 
         * @param config
         *        The configuration status of the firewall endpoint in a single VPC subnet. Network Firewall provides
         *        each endpoint with the rules that are configured in the firewall policy. Each time you add a subnet or
         *        modify the associated firewall policy, Network Firewall synchronizes the rules in the endpoint, so it
         *        can properly filter network traffic. This is part of the <a>FirewallStatus</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder config(Map<String, PerObjectStatus> config);
    }

    static final class BuilderImpl implements Builder {
        private Attachment attachment;

        private Map<String, PerObjectStatus> config = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SyncState model) {
            attachment(model.attachment);
            config(model.config);
        }

        public final Attachment.Builder getAttachment() {
            return attachment != null ? attachment.toBuilder() : null;
        }

        public final void setAttachment(Attachment.BuilderImpl attachment) {
            this.attachment = attachment != null ? attachment.build() : null;
        }

        @Override
        @Transient
        public final Builder attachment(Attachment attachment) {
            this.attachment = attachment;
            return this;
        }

        public final Map<String, PerObjectStatus.Builder> getConfig() {
            Map<String, PerObjectStatus.Builder> result = SyncStateConfigCopier.copyToBuilder(this.config);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setConfig(Map<String, PerObjectStatus.BuilderImpl> config) {
            this.config = SyncStateConfigCopier.copyFromBuilder(config);
        }

        @Override
        @Transient
        public final Builder config(Map<String, PerObjectStatus> config) {
            this.config = SyncStateConfigCopier.copy(config);
            return this;
        }

        @Override
        public SyncState build() {
            return new SyncState(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
