/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkfirewall.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A firewall defines the behavior of a firewall, the main VPC where the firewall is used, the Availability Zones where
 * the firewall can be used, and one subnet to use for a firewall endpoint within each of the Availability Zones. The
 * Availability Zones are defined implicitly in the subnet specifications.
 * </p>
 * <p>
 * In addition to the firewall endpoints that you define in this <code>Firewall</code> specification, you can create
 * firewall endpoints in <code>VpcEndpointAssociation</code> resources for any VPC, in any Availability Zone where the
 * firewall is already in use.
 * </p>
 * <p>
 * The status of the firewall, for example whether it's ready to filter network traffic, is provided in the
 * corresponding <a>FirewallStatus</a>. You can retrieve both the firewall and firewall status by calling
 * <a>DescribeFirewall</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Firewall implements SdkPojo, Serializable, ToCopyableBuilder<Firewall.Builder, Firewall> {
    private static final SdkField<String> FIREWALL_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FirewallName").getter(getter(Firewall::firewallName)).setter(setter(Builder::firewallName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FirewallName").build()).build();

    private static final SdkField<String> FIREWALL_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FirewallArn").getter(getter(Firewall::firewallArn)).setter(setter(Builder::firewallArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FirewallArn").build()).build();

    private static final SdkField<String> FIREWALL_POLICY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FirewallPolicyArn").getter(getter(Firewall::firewallPolicyArn))
            .setter(setter(Builder::firewallPolicyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FirewallPolicyArn").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VpcId")
            .getter(getter(Firewall::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<List<SubnetMapping>> SUBNET_MAPPINGS_FIELD = SdkField
            .<List<SubnetMapping>> builder(MarshallingType.LIST)
            .memberName("SubnetMappings")
            .getter(getter(Firewall::subnetMappings))
            .setter(setter(Builder::subnetMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SubnetMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(SubnetMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> DELETE_PROTECTION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DeleteProtection").getter(getter(Firewall::deleteProtection)).setter(setter(Builder::deleteProtection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeleteProtection").build()).build();

    private static final SdkField<Boolean> SUBNET_CHANGE_PROTECTION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SubnetChangeProtection").getter(getter(Firewall::subnetChangeProtection))
            .setter(setter(Builder::subnetChangeProtection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetChangeProtection").build())
            .build();

    private static final SdkField<Boolean> FIREWALL_POLICY_CHANGE_PROTECTION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("FirewallPolicyChangeProtection")
            .getter(getter(Firewall::firewallPolicyChangeProtection))
            .setter(setter(Builder::firewallPolicyChangeProtection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FirewallPolicyChangeProtection")
                    .build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Firewall::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> FIREWALL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FirewallId").getter(getter(Firewall::firewallId)).setter(setter(Builder::firewallId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FirewallId").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(Firewall::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<EncryptionConfiguration> ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<EncryptionConfiguration> builder(MarshallingType.SDK_POJO).memberName("EncryptionConfiguration")
            .getter(getter(Firewall::encryptionConfiguration)).setter(setter(Builder::encryptionConfiguration))
            .constructor(EncryptionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionConfiguration").build())
            .build();

    private static final SdkField<Integer> NUMBER_OF_ASSOCIATIONS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("NumberOfAssociations").getter(getter(Firewall::numberOfAssociations))
            .setter(setter(Builder::numberOfAssociations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberOfAssociations").build())
            .build();

    private static final SdkField<List<String>> ENABLED_ANALYSIS_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EnabledAnalysisTypes")
            .getter(getter(Firewall::enabledAnalysisTypesAsStrings))
            .setter(setter(Builder::enabledAnalysisTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnabledAnalysisTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TRANSIT_GATEWAY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TransitGatewayId").getter(getter(Firewall::transitGatewayId)).setter(setter(Builder::transitGatewayId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransitGatewayId").build()).build();

    private static final SdkField<String> TRANSIT_GATEWAY_OWNER_ACCOUNT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TransitGatewayOwnerAccountId")
            .getter(getter(Firewall::transitGatewayOwnerAccountId))
            .setter(setter(Builder::transitGatewayOwnerAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransitGatewayOwnerAccountId")
                    .build()).build();

    private static final SdkField<List<AvailabilityZoneMapping>> AVAILABILITY_ZONE_MAPPINGS_FIELD = SdkField
            .<List<AvailabilityZoneMapping>> builder(MarshallingType.LIST)
            .memberName("AvailabilityZoneMappings")
            .getter(getter(Firewall::availabilityZoneMappings))
            .setter(setter(Builder::availabilityZoneMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZoneMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AvailabilityZoneMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(AvailabilityZoneMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> AVAILABILITY_ZONE_CHANGE_PROTECTION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AvailabilityZoneChangeProtection")
            .getter(getter(Firewall::availabilityZoneChangeProtection))
            .setter(setter(Builder::availabilityZoneChangeProtection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZoneChangeProtection")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FIREWALL_NAME_FIELD,
            FIREWALL_ARN_FIELD, FIREWALL_POLICY_ARN_FIELD, VPC_ID_FIELD, SUBNET_MAPPINGS_FIELD, DELETE_PROTECTION_FIELD,
            SUBNET_CHANGE_PROTECTION_FIELD, FIREWALL_POLICY_CHANGE_PROTECTION_FIELD, DESCRIPTION_FIELD, FIREWALL_ID_FIELD,
            TAGS_FIELD, ENCRYPTION_CONFIGURATION_FIELD, NUMBER_OF_ASSOCIATIONS_FIELD, ENABLED_ANALYSIS_TYPES_FIELD,
            TRANSIT_GATEWAY_ID_FIELD, TRANSIT_GATEWAY_OWNER_ACCOUNT_ID_FIELD, AVAILABILITY_ZONE_MAPPINGS_FIELD,
            AVAILABILITY_ZONE_CHANGE_PROTECTION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String firewallName;

    private final String firewallArn;

    private final String firewallPolicyArn;

    private final String vpcId;

    private final List<SubnetMapping> subnetMappings;

    private final Boolean deleteProtection;

    private final Boolean subnetChangeProtection;

    private final Boolean firewallPolicyChangeProtection;

    private final String description;

    private final String firewallId;

    private final List<Tag> tags;

    private final EncryptionConfiguration encryptionConfiguration;

    private final Integer numberOfAssociations;

    private final List<String> enabledAnalysisTypes;

    private final String transitGatewayId;

    private final String transitGatewayOwnerAccountId;

    private final List<AvailabilityZoneMapping> availabilityZoneMappings;

    private final Boolean availabilityZoneChangeProtection;

    private Firewall(BuilderImpl builder) {
        this.firewallName = builder.firewallName;
        this.firewallArn = builder.firewallArn;
        this.firewallPolicyArn = builder.firewallPolicyArn;
        this.vpcId = builder.vpcId;
        this.subnetMappings = builder.subnetMappings;
        this.deleteProtection = builder.deleteProtection;
        this.subnetChangeProtection = builder.subnetChangeProtection;
        this.firewallPolicyChangeProtection = builder.firewallPolicyChangeProtection;
        this.description = builder.description;
        this.firewallId = builder.firewallId;
        this.tags = builder.tags;
        this.encryptionConfiguration = builder.encryptionConfiguration;
        this.numberOfAssociations = builder.numberOfAssociations;
        this.enabledAnalysisTypes = builder.enabledAnalysisTypes;
        this.transitGatewayId = builder.transitGatewayId;
        this.transitGatewayOwnerAccountId = builder.transitGatewayOwnerAccountId;
        this.availabilityZoneMappings = builder.availabilityZoneMappings;
        this.availabilityZoneChangeProtection = builder.availabilityZoneChangeProtection;
    }

    /**
     * <p>
     * The descriptive name of the firewall. You can't change the name of a firewall after you create it.
     * </p>
     * 
     * @return The descriptive name of the firewall. You can't change the name of a firewall after you create it.
     */
    public final String firewallName() {
        return firewallName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the firewall.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the firewall.
     */
    public final String firewallArn() {
        return firewallArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the firewall policy.
     * </p>
     * <p>
     * The relationship of firewall to firewall policy is many to one. Each firewall requires one firewall policy
     * association, and you can use the same firewall policy for multiple firewalls.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the firewall policy.</p>
     *         <p>
     *         The relationship of firewall to firewall policy is many to one. Each firewall requires one firewall
     *         policy association, and you can use the same firewall policy for multiple firewalls.
     */
    public final String firewallPolicyArn() {
        return firewallPolicyArn;
    }

    /**
     * <p>
     * The unique identifier of the VPC where the firewall is in use.
     * </p>
     * 
     * @return The unique identifier of the VPC where the firewall is in use.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * For responses, this returns true if the service returned a value for the SubnetMappings property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSubnetMappings() {
        return subnetMappings != null && !(subnetMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The primary public subnets that Network Firewall is using for the firewall. Network Firewall creates a firewall
     * endpoint in each subnet. Create a subnet mapping for each Availability Zone where you want to use the firewall.
     * </p>
     * <p>
     * These subnets are all defined for a single, primary VPC, and each must belong to a different Availability Zone.
     * Each of these subnets establishes the availability of the firewall in its Availability Zone.
     * </p>
     * <p>
     * In addition to these subnets, you can define other endpoints for the firewall in
     * <code>VpcEndpointAssociation</code> resources. You can define these additional endpoints for any VPC, and for any
     * of the Availability Zones where the firewall resource already has a subnet mapping. VPC endpoint associations
     * give you the ability to protect multiple VPCs using a single firewall, and to define multiple firewall endpoints
     * for a VPC in a single Availability Zone.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubnetMappings} method.
     * </p>
     * 
     * @return The primary public subnets that Network Firewall is using for the firewall. Network Firewall creates a
     *         firewall endpoint in each subnet. Create a subnet mapping for each Availability Zone where you want to
     *         use the firewall.</p>
     *         <p>
     *         These subnets are all defined for a single, primary VPC, and each must belong to a different Availability
     *         Zone. Each of these subnets establishes the availability of the firewall in its Availability Zone.
     *         </p>
     *         <p>
     *         In addition to these subnets, you can define other endpoints for the firewall in
     *         <code>VpcEndpointAssociation</code> resources. You can define these additional endpoints for any VPC, and
     *         for any of the Availability Zones where the firewall resource already has a subnet mapping. VPC endpoint
     *         associations give you the ability to protect multiple VPCs using a single firewall, and to define
     *         multiple firewall endpoints for a VPC in a single Availability Zone.
     */
    public final List<SubnetMapping> subnetMappings() {
        return subnetMappings;
    }

    /**
     * <p>
     * A flag indicating whether it is possible to delete the firewall. A setting of <code>TRUE</code> indicates that
     * the firewall is protected against deletion. Use this setting to protect against accidentally deleting a firewall
     * that is in use. When you create a firewall, the operation initializes this flag to <code>TRUE</code>.
     * </p>
     * 
     * @return A flag indicating whether it is possible to delete the firewall. A setting of <code>TRUE</code> indicates
     *         that the firewall is protected against deletion. Use this setting to protect against accidentally
     *         deleting a firewall that is in use. When you create a firewall, the operation initializes this flag to
     *         <code>TRUE</code>.
     */
    public final Boolean deleteProtection() {
        return deleteProtection;
    }

    /**
     * <p>
     * A setting indicating whether the firewall is protected against changes to the subnet associations. Use this
     * setting to protect against accidentally modifying the subnet associations for a firewall that is in use. When you
     * create a firewall, the operation initializes this setting to <code>TRUE</code>.
     * </p>
     * 
     * @return A setting indicating whether the firewall is protected against changes to the subnet associations. Use
     *         this setting to protect against accidentally modifying the subnet associations for a firewall that is in
     *         use. When you create a firewall, the operation initializes this setting to <code>TRUE</code>.
     */
    public final Boolean subnetChangeProtection() {
        return subnetChangeProtection;
    }

    /**
     * <p>
     * A setting indicating whether the firewall is protected against a change to the firewall policy association. Use
     * this setting to protect against accidentally modifying the firewall policy for a firewall that is in use. When
     * you create a firewall, the operation initializes this setting to <code>TRUE</code>.
     * </p>
     * 
     * @return A setting indicating whether the firewall is protected against a change to the firewall policy
     *         association. Use this setting to protect against accidentally modifying the firewall policy for a
     *         firewall that is in use. When you create a firewall, the operation initializes this setting to
     *         <code>TRUE</code>.
     */
    public final Boolean firewallPolicyChangeProtection() {
        return firewallPolicyChangeProtection;
    }

    /**
     * <p>
     * A description of the firewall.
     * </p>
     * 
     * @return A description of the firewall.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The unique identifier for the firewall.
     * </p>
     * 
     * @return The unique identifier for the firewall.
     */
    public final String firewallId() {
        return firewallId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p/>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * A complex type that contains the Amazon Web Services KMS encryption configuration settings for your firewall.
     * </p>
     * 
     * @return A complex type that contains the Amazon Web Services KMS encryption configuration settings for your
     *         firewall.
     */
    public final EncryptionConfiguration encryptionConfiguration() {
        return encryptionConfiguration;
    }

    /**
     * <p>
     * The number of <code>VpcEndpointAssociation</code> resources that use this firewall.
     * </p>
     * 
     * @return The number of <code>VpcEndpointAssociation</code> resources that use this firewall.
     */
    public final Integer numberOfAssociations() {
        return numberOfAssociations;
    }

    /**
     * <p>
     * An optional setting indicating the specific traffic analysis types to enable on the firewall.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnabledAnalysisTypes} method.
     * </p>
     * 
     * @return An optional setting indicating the specific traffic analysis types to enable on the firewall.
     */
    public final List<EnabledAnalysisType> enabledAnalysisTypes() {
        return EnabledAnalysisTypesCopier.copyStringToEnum(enabledAnalysisTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the EnabledAnalysisTypes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnabledAnalysisTypes() {
        return enabledAnalysisTypes != null && !(enabledAnalysisTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An optional setting indicating the specific traffic analysis types to enable on the firewall.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnabledAnalysisTypes} method.
     * </p>
     * 
     * @return An optional setting indicating the specific traffic analysis types to enable on the firewall.
     */
    public final List<String> enabledAnalysisTypesAsStrings() {
        return enabledAnalysisTypes;
    }

    /**
     * <p>
     * The unique identifier of the transit gateway associated with this firewall. This field is only present for
     * transit gateway-attached firewalls.
     * </p>
     * 
     * @return The unique identifier of the transit gateway associated with this firewall. This field is only present
     *         for transit gateway-attached firewalls.
     */
    public final String transitGatewayId() {
        return transitGatewayId;
    }

    /**
     * <p>
     * The Amazon Web Services account ID that owns the transit gateway. This may be different from the firewall owner's
     * account ID when using a shared transit gateway.
     * </p>
     * 
     * @return The Amazon Web Services account ID that owns the transit gateway. This may be different from the firewall
     *         owner's account ID when using a shared transit gateway.
     */
    public final String transitGatewayOwnerAccountId() {
        return transitGatewayOwnerAccountId;
    }

    /**
     * For responses, this returns true if the service returned a value for the AvailabilityZoneMappings property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAvailabilityZoneMappings() {
        return availabilityZoneMappings != null && !(availabilityZoneMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Availability Zones where the firewall endpoints are created for a transit gateway-attached firewall. Each
     * mapping specifies an Availability Zone where the firewall processes traffic.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAvailabilityZoneMappings} method.
     * </p>
     * 
     * @return The Availability Zones where the firewall endpoints are created for a transit gateway-attached firewall.
     *         Each mapping specifies an Availability Zone where the firewall processes traffic.
     */
    public final List<AvailabilityZoneMapping> availabilityZoneMappings() {
        return availabilityZoneMappings;
    }

    /**
     * <p>
     * A setting indicating whether the firewall is protected against changes to its Availability Zone configuration.
     * When set to <code>TRUE</code>, you must first disable this protection before adding or removing Availability
     * Zones.
     * </p>
     * 
     * @return A setting indicating whether the firewall is protected against changes to its Availability Zone
     *         configuration. When set to <code>TRUE</code>, you must first disable this protection before adding or
     *         removing Availability Zones.
     */
    public final Boolean availabilityZoneChangeProtection() {
        return availabilityZoneChangeProtection;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(firewallName());
        hashCode = 31 * hashCode + Objects.hashCode(firewallArn());
        hashCode = 31 * hashCode + Objects.hashCode(firewallPolicyArn());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnetMappings() ? subnetMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(deleteProtection());
        hashCode = 31 * hashCode + Objects.hashCode(subnetChangeProtection());
        hashCode = 31 * hashCode + Objects.hashCode(firewallPolicyChangeProtection());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(firewallId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(encryptionConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfAssociations());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnabledAnalysisTypes() ? enabledAnalysisTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(transitGatewayId());
        hashCode = 31 * hashCode + Objects.hashCode(transitGatewayOwnerAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(hasAvailabilityZoneMappings() ? availabilityZoneMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZoneChangeProtection());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Firewall)) {
            return false;
        }
        Firewall other = (Firewall) obj;
        return Objects.equals(firewallName(), other.firewallName()) && Objects.equals(firewallArn(), other.firewallArn())
                && Objects.equals(firewallPolicyArn(), other.firewallPolicyArn()) && Objects.equals(vpcId(), other.vpcId())
                && hasSubnetMappings() == other.hasSubnetMappings() && Objects.equals(subnetMappings(), other.subnetMappings())
                && Objects.equals(deleteProtection(), other.deleteProtection())
                && Objects.equals(subnetChangeProtection(), other.subnetChangeProtection())
                && Objects.equals(firewallPolicyChangeProtection(), other.firewallPolicyChangeProtection())
                && Objects.equals(description(), other.description()) && Objects.equals(firewallId(), other.firewallId())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(encryptionConfiguration(), other.encryptionConfiguration())
                && Objects.equals(numberOfAssociations(), other.numberOfAssociations())
                && hasEnabledAnalysisTypes() == other.hasEnabledAnalysisTypes()
                && Objects.equals(enabledAnalysisTypesAsStrings(), other.enabledAnalysisTypesAsStrings())
                && Objects.equals(transitGatewayId(), other.transitGatewayId())
                && Objects.equals(transitGatewayOwnerAccountId(), other.transitGatewayOwnerAccountId())
                && hasAvailabilityZoneMappings() == other.hasAvailabilityZoneMappings()
                && Objects.equals(availabilityZoneMappings(), other.availabilityZoneMappings())
                && Objects.equals(availabilityZoneChangeProtection(), other.availabilityZoneChangeProtection());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Firewall").add("FirewallName", firewallName()).add("FirewallArn", firewallArn())
                .add("FirewallPolicyArn", firewallPolicyArn()).add("VpcId", vpcId())
                .add("SubnetMappings", hasSubnetMappings() ? subnetMappings() : null).add("DeleteProtection", deleteProtection())
                .add("SubnetChangeProtection", subnetChangeProtection())
                .add("FirewallPolicyChangeProtection", firewallPolicyChangeProtection()).add("Description", description())
                .add("FirewallId", firewallId()).add("Tags", hasTags() ? tags() : null)
                .add("EncryptionConfiguration", encryptionConfiguration()).add("NumberOfAssociations", numberOfAssociations())
                .add("EnabledAnalysisTypes", hasEnabledAnalysisTypes() ? enabledAnalysisTypesAsStrings() : null)
                .add("TransitGatewayId", transitGatewayId()).add("TransitGatewayOwnerAccountId", transitGatewayOwnerAccountId())
                .add("AvailabilityZoneMappings", hasAvailabilityZoneMappings() ? availabilityZoneMappings() : null)
                .add("AvailabilityZoneChangeProtection", availabilityZoneChangeProtection()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FirewallName":
            return Optional.ofNullable(clazz.cast(firewallName()));
        case "FirewallArn":
            return Optional.ofNullable(clazz.cast(firewallArn()));
        case "FirewallPolicyArn":
            return Optional.ofNullable(clazz.cast(firewallPolicyArn()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "SubnetMappings":
            return Optional.ofNullable(clazz.cast(subnetMappings()));
        case "DeleteProtection":
            return Optional.ofNullable(clazz.cast(deleteProtection()));
        case "SubnetChangeProtection":
            return Optional.ofNullable(clazz.cast(subnetChangeProtection()));
        case "FirewallPolicyChangeProtection":
            return Optional.ofNullable(clazz.cast(firewallPolicyChangeProtection()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "FirewallId":
            return Optional.ofNullable(clazz.cast(firewallId()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "EncryptionConfiguration":
            return Optional.ofNullable(clazz.cast(encryptionConfiguration()));
        case "NumberOfAssociations":
            return Optional.ofNullable(clazz.cast(numberOfAssociations()));
        case "EnabledAnalysisTypes":
            return Optional.ofNullable(clazz.cast(enabledAnalysisTypesAsStrings()));
        case "TransitGatewayId":
            return Optional.ofNullable(clazz.cast(transitGatewayId()));
        case "TransitGatewayOwnerAccountId":
            return Optional.ofNullable(clazz.cast(transitGatewayOwnerAccountId()));
        case "AvailabilityZoneMappings":
            return Optional.ofNullable(clazz.cast(availabilityZoneMappings()));
        case "AvailabilityZoneChangeProtection":
            return Optional.ofNullable(clazz.cast(availabilityZoneChangeProtection()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("FirewallName", FIREWALL_NAME_FIELD);
        map.put("FirewallArn", FIREWALL_ARN_FIELD);
        map.put("FirewallPolicyArn", FIREWALL_POLICY_ARN_FIELD);
        map.put("VpcId", VPC_ID_FIELD);
        map.put("SubnetMappings", SUBNET_MAPPINGS_FIELD);
        map.put("DeleteProtection", DELETE_PROTECTION_FIELD);
        map.put("SubnetChangeProtection", SUBNET_CHANGE_PROTECTION_FIELD);
        map.put("FirewallPolicyChangeProtection", FIREWALL_POLICY_CHANGE_PROTECTION_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("FirewallId", FIREWALL_ID_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("EncryptionConfiguration", ENCRYPTION_CONFIGURATION_FIELD);
        map.put("NumberOfAssociations", NUMBER_OF_ASSOCIATIONS_FIELD);
        map.put("EnabledAnalysisTypes", ENABLED_ANALYSIS_TYPES_FIELD);
        map.put("TransitGatewayId", TRANSIT_GATEWAY_ID_FIELD);
        map.put("TransitGatewayOwnerAccountId", TRANSIT_GATEWAY_OWNER_ACCOUNT_ID_FIELD);
        map.put("AvailabilityZoneMappings", AVAILABILITY_ZONE_MAPPINGS_FIELD);
        map.put("AvailabilityZoneChangeProtection", AVAILABILITY_ZONE_CHANGE_PROTECTION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Firewall, T> g) {
        return obj -> g.apply((Firewall) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Firewall> {
        /**
         * <p>
         * The descriptive name of the firewall. You can't change the name of a firewall after you create it.
         * </p>
         * 
         * @param firewallName
         *        The descriptive name of the firewall. You can't change the name of a firewall after you create it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firewallName(String firewallName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the firewall.
         * </p>
         * 
         * @param firewallArn
         *        The Amazon Resource Name (ARN) of the firewall.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firewallArn(String firewallArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the firewall policy.
         * </p>
         * <p>
         * The relationship of firewall to firewall policy is many to one. Each firewall requires one firewall policy
         * association, and you can use the same firewall policy for multiple firewalls.
         * </p>
         * 
         * @param firewallPolicyArn
         *        The Amazon Resource Name (ARN) of the firewall policy.</p>
         *        <p>
         *        The relationship of firewall to firewall policy is many to one. Each firewall requires one firewall
         *        policy association, and you can use the same firewall policy for multiple firewalls.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firewallPolicyArn(String firewallPolicyArn);

        /**
         * <p>
         * The unique identifier of the VPC where the firewall is in use.
         * </p>
         * 
         * @param vpcId
         *        The unique identifier of the VPC where the firewall is in use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The primary public subnets that Network Firewall is using for the firewall. Network Firewall creates a
         * firewall endpoint in each subnet. Create a subnet mapping for each Availability Zone where you want to use
         * the firewall.
         * </p>
         * <p>
         * These subnets are all defined for a single, primary VPC, and each must belong to a different Availability
         * Zone. Each of these subnets establishes the availability of the firewall in its Availability Zone.
         * </p>
         * <p>
         * In addition to these subnets, you can define other endpoints for the firewall in
         * <code>VpcEndpointAssociation</code> resources. You can define these additional endpoints for any VPC, and for
         * any of the Availability Zones where the firewall resource already has a subnet mapping. VPC endpoint
         * associations give you the ability to protect multiple VPCs using a single firewall, and to define multiple
         * firewall endpoints for a VPC in a single Availability Zone.
         * </p>
         * 
         * @param subnetMappings
         *        The primary public subnets that Network Firewall is using for the firewall. Network Firewall creates a
         *        firewall endpoint in each subnet. Create a subnet mapping for each Availability Zone where you want to
         *        use the firewall.</p>
         *        <p>
         *        These subnets are all defined for a single, primary VPC, and each must belong to a different
         *        Availability Zone. Each of these subnets establishes the availability of the firewall in its
         *        Availability Zone.
         *        </p>
         *        <p>
         *        In addition to these subnets, you can define other endpoints for the firewall in
         *        <code>VpcEndpointAssociation</code> resources. You can define these additional endpoints for any VPC,
         *        and for any of the Availability Zones where the firewall resource already has a subnet mapping. VPC
         *        endpoint associations give you the ability to protect multiple VPCs using a single firewall, and to
         *        define multiple firewall endpoints for a VPC in a single Availability Zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetMappings(Collection<SubnetMapping> subnetMappings);

        /**
         * <p>
         * The primary public subnets that Network Firewall is using for the firewall. Network Firewall creates a
         * firewall endpoint in each subnet. Create a subnet mapping for each Availability Zone where you want to use
         * the firewall.
         * </p>
         * <p>
         * These subnets are all defined for a single, primary VPC, and each must belong to a different Availability
         * Zone. Each of these subnets establishes the availability of the firewall in its Availability Zone.
         * </p>
         * <p>
         * In addition to these subnets, you can define other endpoints for the firewall in
         * <code>VpcEndpointAssociation</code> resources. You can define these additional endpoints for any VPC, and for
         * any of the Availability Zones where the firewall resource already has a subnet mapping. VPC endpoint
         * associations give you the ability to protect multiple VPCs using a single firewall, and to define multiple
         * firewall endpoints for a VPC in a single Availability Zone.
         * </p>
         * 
         * @param subnetMappings
         *        The primary public subnets that Network Firewall is using for the firewall. Network Firewall creates a
         *        firewall endpoint in each subnet. Create a subnet mapping for each Availability Zone where you want to
         *        use the firewall.</p>
         *        <p>
         *        These subnets are all defined for a single, primary VPC, and each must belong to a different
         *        Availability Zone. Each of these subnets establishes the availability of the firewall in its
         *        Availability Zone.
         *        </p>
         *        <p>
         *        In addition to these subnets, you can define other endpoints for the firewall in
         *        <code>VpcEndpointAssociation</code> resources. You can define these additional endpoints for any VPC,
         *        and for any of the Availability Zones where the firewall resource already has a subnet mapping. VPC
         *        endpoint associations give you the ability to protect multiple VPCs using a single firewall, and to
         *        define multiple firewall endpoints for a VPC in a single Availability Zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetMappings(SubnetMapping... subnetMappings);

        /**
         * <p>
         * The primary public subnets that Network Firewall is using for the firewall. Network Firewall creates a
         * firewall endpoint in each subnet. Create a subnet mapping for each Availability Zone where you want to use
         * the firewall.
         * </p>
         * <p>
         * These subnets are all defined for a single, primary VPC, and each must belong to a different Availability
         * Zone. Each of these subnets establishes the availability of the firewall in its Availability Zone.
         * </p>
         * <p>
         * In addition to these subnets, you can define other endpoints for the firewall in
         * <code>VpcEndpointAssociation</code> resources. You can define these additional endpoints for any VPC, and for
         * any of the Availability Zones where the firewall resource already has a subnet mapping. VPC endpoint
         * associations give you the ability to protect multiple VPCs using a single firewall, and to define multiple
         * firewall endpoints for a VPC in a single Availability Zone.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.networkfirewall.model.SubnetMapping.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.networkfirewall.model.SubnetMapping#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.networkfirewall.model.SubnetMapping.Builder#build()} is called
         * immediately and its result is passed to {@link #subnetMappings(List<SubnetMapping>)}.
         * 
         * @param subnetMappings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.networkfirewall.model.SubnetMapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #subnetMappings(java.util.Collection<SubnetMapping>)
         */
        Builder subnetMappings(Consumer<SubnetMapping.Builder>... subnetMappings);

        /**
         * <p>
         * A flag indicating whether it is possible to delete the firewall. A setting of <code>TRUE</code> indicates
         * that the firewall is protected against deletion. Use this setting to protect against accidentally deleting a
         * firewall that is in use. When you create a firewall, the operation initializes this flag to <code>TRUE</code>
         * .
         * </p>
         * 
         * @param deleteProtection
         *        A flag indicating whether it is possible to delete the firewall. A setting of <code>TRUE</code>
         *        indicates that the firewall is protected against deletion. Use this setting to protect against
         *        accidentally deleting a firewall that is in use. When you create a firewall, the operation initializes
         *        this flag to <code>TRUE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteProtection(Boolean deleteProtection);

        /**
         * <p>
         * A setting indicating whether the firewall is protected against changes to the subnet associations. Use this
         * setting to protect against accidentally modifying the subnet associations for a firewall that is in use. When
         * you create a firewall, the operation initializes this setting to <code>TRUE</code>.
         * </p>
         * 
         * @param subnetChangeProtection
         *        A setting indicating whether the firewall is protected against changes to the subnet associations. Use
         *        this setting to protect against accidentally modifying the subnet associations for a firewall that is
         *        in use. When you create a firewall, the operation initializes this setting to <code>TRUE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetChangeProtection(Boolean subnetChangeProtection);

        /**
         * <p>
         * A setting indicating whether the firewall is protected against a change to the firewall policy association.
         * Use this setting to protect against accidentally modifying the firewall policy for a firewall that is in use.
         * When you create a firewall, the operation initializes this setting to <code>TRUE</code>.
         * </p>
         * 
         * @param firewallPolicyChangeProtection
         *        A setting indicating whether the firewall is protected against a change to the firewall policy
         *        association. Use this setting to protect against accidentally modifying the firewall policy for a
         *        firewall that is in use. When you create a firewall, the operation initializes this setting to
         *        <code>TRUE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firewallPolicyChangeProtection(Boolean firewallPolicyChangeProtection);

        /**
         * <p>
         * A description of the firewall.
         * </p>
         * 
         * @param description
         *        A description of the firewall.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The unique identifier for the firewall.
         * </p>
         * 
         * @param firewallId
         *        The unique identifier for the firewall.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firewallId(String firewallId);

        /**
         * <p/>
         * 
         * @param tags
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p/>
         * 
         * @param tags
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p/>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.networkfirewall.model.Tag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.networkfirewall.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.networkfirewall.model.Tag.Builder#build()} is called immediately and
         * its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.networkfirewall.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * A complex type that contains the Amazon Web Services KMS encryption configuration settings for your firewall.
         * </p>
         * 
         * @param encryptionConfiguration
         *        A complex type that contains the Amazon Web Services KMS encryption configuration settings for your
         *        firewall.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration);

        /**
         * <p>
         * A complex type that contains the Amazon Web Services KMS encryption configuration settings for your firewall.
         * </p>
         * This is a convenience method that creates an instance of the {@link EncryptionConfiguration.Builder} avoiding
         * the need to create one manually via {@link EncryptionConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EncryptionConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #encryptionConfiguration(EncryptionConfiguration)}.
         * 
         * @param encryptionConfiguration
         *        a consumer that will call methods on {@link EncryptionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionConfiguration(EncryptionConfiguration)
         */
        default Builder encryptionConfiguration(Consumer<EncryptionConfiguration.Builder> encryptionConfiguration) {
            return encryptionConfiguration(EncryptionConfiguration.builder().applyMutation(encryptionConfiguration).build());
        }

        /**
         * <p>
         * The number of <code>VpcEndpointAssociation</code> resources that use this firewall.
         * </p>
         * 
         * @param numberOfAssociations
         *        The number of <code>VpcEndpointAssociation</code> resources that use this firewall.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfAssociations(Integer numberOfAssociations);

        /**
         * <p>
         * An optional setting indicating the specific traffic analysis types to enable on the firewall.
         * </p>
         * 
         * @param enabledAnalysisTypes
         *        An optional setting indicating the specific traffic analysis types to enable on the firewall.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabledAnalysisTypesWithStrings(Collection<String> enabledAnalysisTypes);

        /**
         * <p>
         * An optional setting indicating the specific traffic analysis types to enable on the firewall.
         * </p>
         * 
         * @param enabledAnalysisTypes
         *        An optional setting indicating the specific traffic analysis types to enable on the firewall.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabledAnalysisTypesWithStrings(String... enabledAnalysisTypes);

        /**
         * <p>
         * An optional setting indicating the specific traffic analysis types to enable on the firewall.
         * </p>
         * 
         * @param enabledAnalysisTypes
         *        An optional setting indicating the specific traffic analysis types to enable on the firewall.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabledAnalysisTypes(Collection<EnabledAnalysisType> enabledAnalysisTypes);

        /**
         * <p>
         * An optional setting indicating the specific traffic analysis types to enable on the firewall.
         * </p>
         * 
         * @param enabledAnalysisTypes
         *        An optional setting indicating the specific traffic analysis types to enable on the firewall.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabledAnalysisTypes(EnabledAnalysisType... enabledAnalysisTypes);

        /**
         * <p>
         * The unique identifier of the transit gateway associated with this firewall. This field is only present for
         * transit gateway-attached firewalls.
         * </p>
         * 
         * @param transitGatewayId
         *        The unique identifier of the transit gateway associated with this firewall. This field is only present
         *        for transit gateway-attached firewalls.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transitGatewayId(String transitGatewayId);

        /**
         * <p>
         * The Amazon Web Services account ID that owns the transit gateway. This may be different from the firewall
         * owner's account ID when using a shared transit gateway.
         * </p>
         * 
         * @param transitGatewayOwnerAccountId
         *        The Amazon Web Services account ID that owns the transit gateway. This may be different from the
         *        firewall owner's account ID when using a shared transit gateway.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transitGatewayOwnerAccountId(String transitGatewayOwnerAccountId);

        /**
         * <p>
         * The Availability Zones where the firewall endpoints are created for a transit gateway-attached firewall. Each
         * mapping specifies an Availability Zone where the firewall processes traffic.
         * </p>
         * 
         * @param availabilityZoneMappings
         *        The Availability Zones where the firewall endpoints are created for a transit gateway-attached
         *        firewall. Each mapping specifies an Availability Zone where the firewall processes traffic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZoneMappings(Collection<AvailabilityZoneMapping> availabilityZoneMappings);

        /**
         * <p>
         * The Availability Zones where the firewall endpoints are created for a transit gateway-attached firewall. Each
         * mapping specifies an Availability Zone where the firewall processes traffic.
         * </p>
         * 
         * @param availabilityZoneMappings
         *        The Availability Zones where the firewall endpoints are created for a transit gateway-attached
         *        firewall. Each mapping specifies an Availability Zone where the firewall processes traffic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZoneMappings(AvailabilityZoneMapping... availabilityZoneMappings);

        /**
         * <p>
         * The Availability Zones where the firewall endpoints are created for a transit gateway-attached firewall. Each
         * mapping specifies an Availability Zone where the firewall processes traffic.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.networkfirewall.model.AvailabilityZoneMapping.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.networkfirewall.model.AvailabilityZoneMapping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.networkfirewall.model.AvailabilityZoneMapping.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #availabilityZoneMappings(List<AvailabilityZoneMapping>)}.
         * 
         * @param availabilityZoneMappings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.networkfirewall.model.AvailabilityZoneMapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #availabilityZoneMappings(java.util.Collection<AvailabilityZoneMapping>)
         */
        Builder availabilityZoneMappings(Consumer<AvailabilityZoneMapping.Builder>... availabilityZoneMappings);

        /**
         * <p>
         * A setting indicating whether the firewall is protected against changes to its Availability Zone
         * configuration. When set to <code>TRUE</code>, you must first disable this protection before adding or
         * removing Availability Zones.
         * </p>
         * 
         * @param availabilityZoneChangeProtection
         *        A setting indicating whether the firewall is protected against changes to its Availability Zone
         *        configuration. When set to <code>TRUE</code>, you must first disable this protection before adding or
         *        removing Availability Zones.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZoneChangeProtection(Boolean availabilityZoneChangeProtection);
    }

    static final class BuilderImpl implements Builder {
        private String firewallName;

        private String firewallArn;

        private String firewallPolicyArn;

        private String vpcId;

        private List<SubnetMapping> subnetMappings = DefaultSdkAutoConstructList.getInstance();

        private Boolean deleteProtection;

        private Boolean subnetChangeProtection;

        private Boolean firewallPolicyChangeProtection;

        private String description;

        private String firewallId;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private EncryptionConfiguration encryptionConfiguration;

        private Integer numberOfAssociations;

        private List<String> enabledAnalysisTypes = DefaultSdkAutoConstructList.getInstance();

        private String transitGatewayId;

        private String transitGatewayOwnerAccountId;

        private List<AvailabilityZoneMapping> availabilityZoneMappings = DefaultSdkAutoConstructList.getInstance();

        private Boolean availabilityZoneChangeProtection;

        private BuilderImpl() {
        }

        private BuilderImpl(Firewall model) {
            firewallName(model.firewallName);
            firewallArn(model.firewallArn);
            firewallPolicyArn(model.firewallPolicyArn);
            vpcId(model.vpcId);
            subnetMappings(model.subnetMappings);
            deleteProtection(model.deleteProtection);
            subnetChangeProtection(model.subnetChangeProtection);
            firewallPolicyChangeProtection(model.firewallPolicyChangeProtection);
            description(model.description);
            firewallId(model.firewallId);
            tags(model.tags);
            encryptionConfiguration(model.encryptionConfiguration);
            numberOfAssociations(model.numberOfAssociations);
            enabledAnalysisTypesWithStrings(model.enabledAnalysisTypes);
            transitGatewayId(model.transitGatewayId);
            transitGatewayOwnerAccountId(model.transitGatewayOwnerAccountId);
            availabilityZoneMappings(model.availabilityZoneMappings);
            availabilityZoneChangeProtection(model.availabilityZoneChangeProtection);
        }

        public final String getFirewallName() {
            return firewallName;
        }

        public final void setFirewallName(String firewallName) {
            this.firewallName = firewallName;
        }

        @Override
        public final Builder firewallName(String firewallName) {
            this.firewallName = firewallName;
            return this;
        }

        public final String getFirewallArn() {
            return firewallArn;
        }

        public final void setFirewallArn(String firewallArn) {
            this.firewallArn = firewallArn;
        }

        @Override
        public final Builder firewallArn(String firewallArn) {
            this.firewallArn = firewallArn;
            return this;
        }

        public final String getFirewallPolicyArn() {
            return firewallPolicyArn;
        }

        public final void setFirewallPolicyArn(String firewallPolicyArn) {
            this.firewallPolicyArn = firewallPolicyArn;
        }

        @Override
        public final Builder firewallPolicyArn(String firewallPolicyArn) {
            this.firewallPolicyArn = firewallPolicyArn;
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final List<SubnetMapping.Builder> getSubnetMappings() {
            List<SubnetMapping.Builder> result = SubnetMappingsCopier.copyToBuilder(this.subnetMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSubnetMappings(Collection<SubnetMapping.BuilderImpl> subnetMappings) {
            this.subnetMappings = SubnetMappingsCopier.copyFromBuilder(subnetMappings);
        }

        @Override
        public final Builder subnetMappings(Collection<SubnetMapping> subnetMappings) {
            this.subnetMappings = SubnetMappingsCopier.copy(subnetMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetMappings(SubnetMapping... subnetMappings) {
            subnetMappings(Arrays.asList(subnetMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetMappings(Consumer<SubnetMapping.Builder>... subnetMappings) {
            subnetMappings(Stream.of(subnetMappings).map(c -> SubnetMapping.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getDeleteProtection() {
            return deleteProtection;
        }

        public final void setDeleteProtection(Boolean deleteProtection) {
            this.deleteProtection = deleteProtection;
        }

        @Override
        public final Builder deleteProtection(Boolean deleteProtection) {
            this.deleteProtection = deleteProtection;
            return this;
        }

        public final Boolean getSubnetChangeProtection() {
            return subnetChangeProtection;
        }

        public final void setSubnetChangeProtection(Boolean subnetChangeProtection) {
            this.subnetChangeProtection = subnetChangeProtection;
        }

        @Override
        public final Builder subnetChangeProtection(Boolean subnetChangeProtection) {
            this.subnetChangeProtection = subnetChangeProtection;
            return this;
        }

        public final Boolean getFirewallPolicyChangeProtection() {
            return firewallPolicyChangeProtection;
        }

        public final void setFirewallPolicyChangeProtection(Boolean firewallPolicyChangeProtection) {
            this.firewallPolicyChangeProtection = firewallPolicyChangeProtection;
        }

        @Override
        public final Builder firewallPolicyChangeProtection(Boolean firewallPolicyChangeProtection) {
            this.firewallPolicyChangeProtection = firewallPolicyChangeProtection;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getFirewallId() {
            return firewallId;
        }

        public final void setFirewallId(String firewallId) {
            this.firewallId = firewallId;
        }

        @Override
        public final Builder firewallId(String firewallId) {
            this.firewallId = firewallId;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final EncryptionConfiguration.Builder getEncryptionConfiguration() {
            return encryptionConfiguration != null ? encryptionConfiguration.toBuilder() : null;
        }

        public final void setEncryptionConfiguration(EncryptionConfiguration.BuilderImpl encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration != null ? encryptionConfiguration.build() : null;
        }

        @Override
        public final Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration;
            return this;
        }

        public final Integer getNumberOfAssociations() {
            return numberOfAssociations;
        }

        public final void setNumberOfAssociations(Integer numberOfAssociations) {
            this.numberOfAssociations = numberOfAssociations;
        }

        @Override
        public final Builder numberOfAssociations(Integer numberOfAssociations) {
            this.numberOfAssociations = numberOfAssociations;
            return this;
        }

        public final Collection<String> getEnabledAnalysisTypes() {
            if (enabledAnalysisTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return enabledAnalysisTypes;
        }

        public final void setEnabledAnalysisTypes(Collection<String> enabledAnalysisTypes) {
            this.enabledAnalysisTypes = EnabledAnalysisTypesCopier.copy(enabledAnalysisTypes);
        }

        @Override
        public final Builder enabledAnalysisTypesWithStrings(Collection<String> enabledAnalysisTypes) {
            this.enabledAnalysisTypes = EnabledAnalysisTypesCopier.copy(enabledAnalysisTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder enabledAnalysisTypesWithStrings(String... enabledAnalysisTypes) {
            enabledAnalysisTypesWithStrings(Arrays.asList(enabledAnalysisTypes));
            return this;
        }

        @Override
        public final Builder enabledAnalysisTypes(Collection<EnabledAnalysisType> enabledAnalysisTypes) {
            this.enabledAnalysisTypes = EnabledAnalysisTypesCopier.copyEnumToString(enabledAnalysisTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder enabledAnalysisTypes(EnabledAnalysisType... enabledAnalysisTypes) {
            enabledAnalysisTypes(Arrays.asList(enabledAnalysisTypes));
            return this;
        }

        public final String getTransitGatewayId() {
            return transitGatewayId;
        }

        public final void setTransitGatewayId(String transitGatewayId) {
            this.transitGatewayId = transitGatewayId;
        }

        @Override
        public final Builder transitGatewayId(String transitGatewayId) {
            this.transitGatewayId = transitGatewayId;
            return this;
        }

        public final String getTransitGatewayOwnerAccountId() {
            return transitGatewayOwnerAccountId;
        }

        public final void setTransitGatewayOwnerAccountId(String transitGatewayOwnerAccountId) {
            this.transitGatewayOwnerAccountId = transitGatewayOwnerAccountId;
        }

        @Override
        public final Builder transitGatewayOwnerAccountId(String transitGatewayOwnerAccountId) {
            this.transitGatewayOwnerAccountId = transitGatewayOwnerAccountId;
            return this;
        }

        public final List<AvailabilityZoneMapping.Builder> getAvailabilityZoneMappings() {
            List<AvailabilityZoneMapping.Builder> result = AvailabilityZoneMappingsCopier
                    .copyToBuilder(this.availabilityZoneMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAvailabilityZoneMappings(Collection<AvailabilityZoneMapping.BuilderImpl> availabilityZoneMappings) {
            this.availabilityZoneMappings = AvailabilityZoneMappingsCopier.copyFromBuilder(availabilityZoneMappings);
        }

        @Override
        public final Builder availabilityZoneMappings(Collection<AvailabilityZoneMapping> availabilityZoneMappings) {
            this.availabilityZoneMappings = AvailabilityZoneMappingsCopier.copy(availabilityZoneMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availabilityZoneMappings(AvailabilityZoneMapping... availabilityZoneMappings) {
            availabilityZoneMappings(Arrays.asList(availabilityZoneMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availabilityZoneMappings(Consumer<AvailabilityZoneMapping.Builder>... availabilityZoneMappings) {
            availabilityZoneMappings(Stream.of(availabilityZoneMappings)
                    .map(c -> AvailabilityZoneMapping.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Boolean getAvailabilityZoneChangeProtection() {
            return availabilityZoneChangeProtection;
        }

        public final void setAvailabilityZoneChangeProtection(Boolean availabilityZoneChangeProtection) {
            this.availabilityZoneChangeProtection = availabilityZoneChangeProtection;
        }

        @Override
        public final Builder availabilityZoneChangeProtection(Boolean availabilityZoneChangeProtection) {
            this.availabilityZoneChangeProtection = availabilityZoneChangeProtection;
            return this;
        }

        @Override
        public Firewall build() {
            return new Firewall(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
