/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkfirewall.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration settings for the handling of the stateful rule groups in a firewall policy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StatefulEngineOptions implements SdkPojo, Serializable,
        ToCopyableBuilder<StatefulEngineOptions.Builder, StatefulEngineOptions> {
    private static final SdkField<String> RULE_ORDER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RuleOrder").getter(getter(StatefulEngineOptions::ruleOrderAsString)).setter(setter(Builder::ruleOrder))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleOrder").build()).build();

    private static final SdkField<String> STREAM_EXCEPTION_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamExceptionPolicy").getter(getter(StatefulEngineOptions::streamExceptionPolicyAsString))
            .setter(setter(Builder::streamExceptionPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamExceptionPolicy").build())
            .build();

    private static final SdkField<FlowTimeouts> FLOW_TIMEOUTS_FIELD = SdkField.<FlowTimeouts> builder(MarshallingType.SDK_POJO)
            .memberName("FlowTimeouts").getter(getter(StatefulEngineOptions::flowTimeouts)).setter(setter(Builder::flowTimeouts))
            .constructor(FlowTimeouts::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FlowTimeouts").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RULE_ORDER_FIELD,
            STREAM_EXCEPTION_POLICY_FIELD, FLOW_TIMEOUTS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String ruleOrder;

    private final String streamExceptionPolicy;

    private final FlowTimeouts flowTimeouts;

    private StatefulEngineOptions(BuilderImpl builder) {
        this.ruleOrder = builder.ruleOrder;
        this.streamExceptionPolicy = builder.streamExceptionPolicy;
        this.flowTimeouts = builder.flowTimeouts;
    }

    /**
     * <p>
     * Indicates how to manage the order of stateful rule evaluation for the policy. <code>STRICT_ORDER</code> is the
     * recommended option, but <code>DEFAULT_ACTION_ORDER</code> is the default option. With <code>STRICT_ORDER</code>,
     * provide your rules in the order that you want them to be evaluated. You can then choose one or more default
     * actions for packets that don't match any rules. Choose <code>STRICT_ORDER</code> to have the stateful rules
     * engine determine the evaluation order of your rules. The default action for this rule order is <code>PASS</code>,
     * followed by <code>DROP</code>, <code>REJECT</code>, and <code>ALERT</code> actions. Stateful rules are provided
     * to the rule engine as Suricata compatible strings, and Suricata evaluates them based on your settings. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/network-firewall/latest/developerguide/suricata-rule-evaluation-order.html"
     * >Evaluation order for stateful rules</a> in the <i>Network Firewall Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ruleOrder} will
     * return {@link RuleOrder#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ruleOrderAsString}.
     * </p>
     * 
     * @return Indicates how to manage the order of stateful rule evaluation for the policy. <code>STRICT_ORDER</code>
     *         is the recommended option, but <code>DEFAULT_ACTION_ORDER</code> is the default option. With
     *         <code>STRICT_ORDER</code>, provide your rules in the order that you want them to be evaluated. You can
     *         then choose one or more default actions for packets that don't match any rules. Choose
     *         <code>STRICT_ORDER</code> to have the stateful rules engine determine the evaluation order of your rules.
     *         The default action for this rule order is <code>PASS</code>, followed by <code>DROP</code>,
     *         <code>REJECT</code>, and <code>ALERT</code> actions. Stateful rules are provided to the rule engine as
     *         Suricata compatible strings, and Suricata evaluates them based on your settings. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/network-firewall/latest/developerguide/suricata-rule-evaluation-order.html"
     *         >Evaluation order for stateful rules</a> in the <i>Network Firewall Developer Guide</i>.
     * @see RuleOrder
     */
    public final RuleOrder ruleOrder() {
        return RuleOrder.fromValue(ruleOrder);
    }

    /**
     * <p>
     * Indicates how to manage the order of stateful rule evaluation for the policy. <code>STRICT_ORDER</code> is the
     * recommended option, but <code>DEFAULT_ACTION_ORDER</code> is the default option. With <code>STRICT_ORDER</code>,
     * provide your rules in the order that you want them to be evaluated. You can then choose one or more default
     * actions for packets that don't match any rules. Choose <code>STRICT_ORDER</code> to have the stateful rules
     * engine determine the evaluation order of your rules. The default action for this rule order is <code>PASS</code>,
     * followed by <code>DROP</code>, <code>REJECT</code>, and <code>ALERT</code> actions. Stateful rules are provided
     * to the rule engine as Suricata compatible strings, and Suricata evaluates them based on your settings. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/network-firewall/latest/developerguide/suricata-rule-evaluation-order.html"
     * >Evaluation order for stateful rules</a> in the <i>Network Firewall Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ruleOrder} will
     * return {@link RuleOrder#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ruleOrderAsString}.
     * </p>
     * 
     * @return Indicates how to manage the order of stateful rule evaluation for the policy. <code>STRICT_ORDER</code>
     *         is the recommended option, but <code>DEFAULT_ACTION_ORDER</code> is the default option. With
     *         <code>STRICT_ORDER</code>, provide your rules in the order that you want them to be evaluated. You can
     *         then choose one or more default actions for packets that don't match any rules. Choose
     *         <code>STRICT_ORDER</code> to have the stateful rules engine determine the evaluation order of your rules.
     *         The default action for this rule order is <code>PASS</code>, followed by <code>DROP</code>,
     *         <code>REJECT</code>, and <code>ALERT</code> actions. Stateful rules are provided to the rule engine as
     *         Suricata compatible strings, and Suricata evaluates them based on your settings. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/network-firewall/latest/developerguide/suricata-rule-evaluation-order.html"
     *         >Evaluation order for stateful rules</a> in the <i>Network Firewall Developer Guide</i>.
     * @see RuleOrder
     */
    public final String ruleOrderAsString() {
        return ruleOrder;
    }

    /**
     * <p>
     * Configures how Network Firewall processes traffic when a network connection breaks midstream. Network connections
     * can break due to disruptions in external networks or within the firewall itself.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DROP</code> - Network Firewall fails closed and drops all subsequent traffic going to the firewall. This is
     * the default behavior.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CONTINUE</code> - Network Firewall continues to apply rules to the subsequent traffic without context from
     * traffic before the break. This impacts the behavior of rules that depend on this context. For example, if you
     * have a stateful rule to <code>drop http</code> traffic, Network Firewall won't match the traffic for this rule
     * because the service won't have the context from session initialization defining the application layer protocol as
     * HTTP. However, this behavior is rule dependent—a TCP-layer rule using a <code>flow:stateless</code> rule would
     * still match, as would the <code>aws:drop_strict</code> default action.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REJECT</code> - Network Firewall fails closed and drops all subsequent traffic going to the firewall.
     * Network Firewall also sends a TCP reject packet back to your client so that the client can immediately establish
     * a new session. Network Firewall will have context about the new session and will apply rules to the subsequent
     * traffic.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #streamExceptionPolicy} will return {@link StreamExceptionPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #streamExceptionPolicyAsString}.
     * </p>
     * 
     * @return Configures how Network Firewall processes traffic when a network connection breaks midstream. Network
     *         connections can break due to disruptions in external networks or within the firewall itself.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DROP</code> - Network Firewall fails closed and drops all subsequent traffic going to the firewall.
     *         This is the default behavior.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CONTINUE</code> - Network Firewall continues to apply rules to the subsequent traffic without
     *         context from traffic before the break. This impacts the behavior of rules that depend on this context.
     *         For example, if you have a stateful rule to <code>drop http</code> traffic, Network Firewall won't match
     *         the traffic for this rule because the service won't have the context from session initialization defining
     *         the application layer protocol as HTTP. However, this behavior is rule dependent—a TCP-layer rule using a
     *         <code>flow:stateless</code> rule would still match, as would the <code>aws:drop_strict</code> default
     *         action.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REJECT</code> - Network Firewall fails closed and drops all subsequent traffic going to the
     *         firewall. Network Firewall also sends a TCP reject packet back to your client so that the client can
     *         immediately establish a new session. Network Firewall will have context about the new session and will
     *         apply rules to the subsequent traffic.
     *         </p>
     *         </li>
     * @see StreamExceptionPolicy
     */
    public final StreamExceptionPolicy streamExceptionPolicy() {
        return StreamExceptionPolicy.fromValue(streamExceptionPolicy);
    }

    /**
     * <p>
     * Configures how Network Firewall processes traffic when a network connection breaks midstream. Network connections
     * can break due to disruptions in external networks or within the firewall itself.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DROP</code> - Network Firewall fails closed and drops all subsequent traffic going to the firewall. This is
     * the default behavior.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CONTINUE</code> - Network Firewall continues to apply rules to the subsequent traffic without context from
     * traffic before the break. This impacts the behavior of rules that depend on this context. For example, if you
     * have a stateful rule to <code>drop http</code> traffic, Network Firewall won't match the traffic for this rule
     * because the service won't have the context from session initialization defining the application layer protocol as
     * HTTP. However, this behavior is rule dependent—a TCP-layer rule using a <code>flow:stateless</code> rule would
     * still match, as would the <code>aws:drop_strict</code> default action.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REJECT</code> - Network Firewall fails closed and drops all subsequent traffic going to the firewall.
     * Network Firewall also sends a TCP reject packet back to your client so that the client can immediately establish
     * a new session. Network Firewall will have context about the new session and will apply rules to the subsequent
     * traffic.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #streamExceptionPolicy} will return {@link StreamExceptionPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #streamExceptionPolicyAsString}.
     * </p>
     * 
     * @return Configures how Network Firewall processes traffic when a network connection breaks midstream. Network
     *         connections can break due to disruptions in external networks or within the firewall itself.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DROP</code> - Network Firewall fails closed and drops all subsequent traffic going to the firewall.
     *         This is the default behavior.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CONTINUE</code> - Network Firewall continues to apply rules to the subsequent traffic without
     *         context from traffic before the break. This impacts the behavior of rules that depend on this context.
     *         For example, if you have a stateful rule to <code>drop http</code> traffic, Network Firewall won't match
     *         the traffic for this rule because the service won't have the context from session initialization defining
     *         the application layer protocol as HTTP. However, this behavior is rule dependent—a TCP-layer rule using a
     *         <code>flow:stateless</code> rule would still match, as would the <code>aws:drop_strict</code> default
     *         action.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REJECT</code> - Network Firewall fails closed and drops all subsequent traffic going to the
     *         firewall. Network Firewall also sends a TCP reject packet back to your client so that the client can
     *         immediately establish a new session. Network Firewall will have context about the new session and will
     *         apply rules to the subsequent traffic.
     *         </p>
     *         </li>
     * @see StreamExceptionPolicy
     */
    public final String streamExceptionPolicyAsString() {
        return streamExceptionPolicy;
    }

    /**
     * <p>
     * Configures the amount of time that can pass without any traffic sent through the firewall before the firewall
     * determines that the connection is idle.
     * </p>
     * 
     * @return Configures the amount of time that can pass without any traffic sent through the firewall before the
     *         firewall determines that the connection is idle.
     */
    public final FlowTimeouts flowTimeouts() {
        return flowTimeouts;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ruleOrderAsString());
        hashCode = 31 * hashCode + Objects.hashCode(streamExceptionPolicyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(flowTimeouts());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StatefulEngineOptions)) {
            return false;
        }
        StatefulEngineOptions other = (StatefulEngineOptions) obj;
        return Objects.equals(ruleOrderAsString(), other.ruleOrderAsString())
                && Objects.equals(streamExceptionPolicyAsString(), other.streamExceptionPolicyAsString())
                && Objects.equals(flowTimeouts(), other.flowTimeouts());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StatefulEngineOptions").add("RuleOrder", ruleOrderAsString())
                .add("StreamExceptionPolicy", streamExceptionPolicyAsString()).add("FlowTimeouts", flowTimeouts()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RuleOrder":
            return Optional.ofNullable(clazz.cast(ruleOrderAsString()));
        case "StreamExceptionPolicy":
            return Optional.ofNullable(clazz.cast(streamExceptionPolicyAsString()));
        case "FlowTimeouts":
            return Optional.ofNullable(clazz.cast(flowTimeouts()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RuleOrder", RULE_ORDER_FIELD);
        map.put("StreamExceptionPolicy", STREAM_EXCEPTION_POLICY_FIELD);
        map.put("FlowTimeouts", FLOW_TIMEOUTS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<StatefulEngineOptions, T> g) {
        return obj -> g.apply((StatefulEngineOptions) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StatefulEngineOptions> {
        /**
         * <p>
         * Indicates how to manage the order of stateful rule evaluation for the policy. <code>STRICT_ORDER</code> is
         * the recommended option, but <code>DEFAULT_ACTION_ORDER</code> is the default option. With
         * <code>STRICT_ORDER</code>, provide your rules in the order that you want them to be evaluated. You can then
         * choose one or more default actions for packets that don't match any rules. Choose <code>STRICT_ORDER</code>
         * to have the stateful rules engine determine the evaluation order of your rules. The default action for this
         * rule order is <code>PASS</code>, followed by <code>DROP</code>, <code>REJECT</code>, and <code>ALERT</code>
         * actions. Stateful rules are provided to the rule engine as Suricata compatible strings, and Suricata
         * evaluates them based on your settings. For more information, see <a href=
         * "https://docs.aws.amazon.com/network-firewall/latest/developerguide/suricata-rule-evaluation-order.html"
         * >Evaluation order for stateful rules</a> in the <i>Network Firewall Developer Guide</i>.
         * </p>
         * 
         * @param ruleOrder
         *        Indicates how to manage the order of stateful rule evaluation for the policy.
         *        <code>STRICT_ORDER</code> is the recommended option, but <code>DEFAULT_ACTION_ORDER</code> is the
         *        default option. With <code>STRICT_ORDER</code>, provide your rules in the order that you want them to
         *        be evaluated. You can then choose one or more default actions for packets that don't match any rules.
         *        Choose <code>STRICT_ORDER</code> to have the stateful rules engine determine the evaluation order of
         *        your rules. The default action for this rule order is <code>PASS</code>, followed by <code>DROP</code>
         *        , <code>REJECT</code>, and <code>ALERT</code> actions. Stateful rules are provided to the rule engine
         *        as Suricata compatible strings, and Suricata evaluates them based on your settings. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/network-firewall/latest/developerguide/suricata-rule-evaluation-order.html"
         *        >Evaluation order for stateful rules</a> in the <i>Network Firewall Developer Guide</i>.
         * @see RuleOrder
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RuleOrder
         */
        Builder ruleOrder(String ruleOrder);

        /**
         * <p>
         * Indicates how to manage the order of stateful rule evaluation for the policy. <code>STRICT_ORDER</code> is
         * the recommended option, but <code>DEFAULT_ACTION_ORDER</code> is the default option. With
         * <code>STRICT_ORDER</code>, provide your rules in the order that you want them to be evaluated. You can then
         * choose one or more default actions for packets that don't match any rules. Choose <code>STRICT_ORDER</code>
         * to have the stateful rules engine determine the evaluation order of your rules. The default action for this
         * rule order is <code>PASS</code>, followed by <code>DROP</code>, <code>REJECT</code>, and <code>ALERT</code>
         * actions. Stateful rules are provided to the rule engine as Suricata compatible strings, and Suricata
         * evaluates them based on your settings. For more information, see <a href=
         * "https://docs.aws.amazon.com/network-firewall/latest/developerguide/suricata-rule-evaluation-order.html"
         * >Evaluation order for stateful rules</a> in the <i>Network Firewall Developer Guide</i>.
         * </p>
         * 
         * @param ruleOrder
         *        Indicates how to manage the order of stateful rule evaluation for the policy.
         *        <code>STRICT_ORDER</code> is the recommended option, but <code>DEFAULT_ACTION_ORDER</code> is the
         *        default option. With <code>STRICT_ORDER</code>, provide your rules in the order that you want them to
         *        be evaluated. You can then choose one or more default actions for packets that don't match any rules.
         *        Choose <code>STRICT_ORDER</code> to have the stateful rules engine determine the evaluation order of
         *        your rules. The default action for this rule order is <code>PASS</code>, followed by <code>DROP</code>
         *        , <code>REJECT</code>, and <code>ALERT</code> actions. Stateful rules are provided to the rule engine
         *        as Suricata compatible strings, and Suricata evaluates them based on your settings. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/network-firewall/latest/developerguide/suricata-rule-evaluation-order.html"
         *        >Evaluation order for stateful rules</a> in the <i>Network Firewall Developer Guide</i>.
         * @see RuleOrder
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RuleOrder
         */
        Builder ruleOrder(RuleOrder ruleOrder);

        /**
         * <p>
         * Configures how Network Firewall processes traffic when a network connection breaks midstream. Network
         * connections can break due to disruptions in external networks or within the firewall itself.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DROP</code> - Network Firewall fails closed and drops all subsequent traffic going to the firewall.
         * This is the default behavior.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CONTINUE</code> - Network Firewall continues to apply rules to the subsequent traffic without context
         * from traffic before the break. This impacts the behavior of rules that depend on this context. For example,
         * if you have a stateful rule to <code>drop http</code> traffic, Network Firewall won't match the traffic for
         * this rule because the service won't have the context from session initialization defining the application
         * layer protocol as HTTP. However, this behavior is rule dependent—a TCP-layer rule using a
         * <code>flow:stateless</code> rule would still match, as would the <code>aws:drop_strict</code> default action.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REJECT</code> - Network Firewall fails closed and drops all subsequent traffic going to the firewall.
         * Network Firewall also sends a TCP reject packet back to your client so that the client can immediately
         * establish a new session. Network Firewall will have context about the new session and will apply rules to the
         * subsequent traffic.
         * </p>
         * </li>
         * </ul>
         * 
         * @param streamExceptionPolicy
         *        Configures how Network Firewall processes traffic when a network connection breaks midstream. Network
         *        connections can break due to disruptions in external networks or within the firewall itself.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DROP</code> - Network Firewall fails closed and drops all subsequent traffic going to the
         *        firewall. This is the default behavior.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CONTINUE</code> - Network Firewall continues to apply rules to the subsequent traffic without
         *        context from traffic before the break. This impacts the behavior of rules that depend on this context.
         *        For example, if you have a stateful rule to <code>drop http</code> traffic, Network Firewall won't
         *        match the traffic for this rule because the service won't have the context from session initialization
         *        defining the application layer protocol as HTTP. However, this behavior is rule dependent—a TCP-layer
         *        rule using a <code>flow:stateless</code> rule would still match, as would the
         *        <code>aws:drop_strict</code> default action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REJECT</code> - Network Firewall fails closed and drops all subsequent traffic going to the
         *        firewall. Network Firewall also sends a TCP reject packet back to your client so that the client can
         *        immediately establish a new session. Network Firewall will have context about the new session and will
         *        apply rules to the subsequent traffic.
         *        </p>
         *        </li>
         * @see StreamExceptionPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StreamExceptionPolicy
         */
        Builder streamExceptionPolicy(String streamExceptionPolicy);

        /**
         * <p>
         * Configures how Network Firewall processes traffic when a network connection breaks midstream. Network
         * connections can break due to disruptions in external networks or within the firewall itself.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DROP</code> - Network Firewall fails closed and drops all subsequent traffic going to the firewall.
         * This is the default behavior.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CONTINUE</code> - Network Firewall continues to apply rules to the subsequent traffic without context
         * from traffic before the break. This impacts the behavior of rules that depend on this context. For example,
         * if you have a stateful rule to <code>drop http</code> traffic, Network Firewall won't match the traffic for
         * this rule because the service won't have the context from session initialization defining the application
         * layer protocol as HTTP. However, this behavior is rule dependent—a TCP-layer rule using a
         * <code>flow:stateless</code> rule would still match, as would the <code>aws:drop_strict</code> default action.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REJECT</code> - Network Firewall fails closed and drops all subsequent traffic going to the firewall.
         * Network Firewall also sends a TCP reject packet back to your client so that the client can immediately
         * establish a new session. Network Firewall will have context about the new session and will apply rules to the
         * subsequent traffic.
         * </p>
         * </li>
         * </ul>
         * 
         * @param streamExceptionPolicy
         *        Configures how Network Firewall processes traffic when a network connection breaks midstream. Network
         *        connections can break due to disruptions in external networks or within the firewall itself.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DROP</code> - Network Firewall fails closed and drops all subsequent traffic going to the
         *        firewall. This is the default behavior.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CONTINUE</code> - Network Firewall continues to apply rules to the subsequent traffic without
         *        context from traffic before the break. This impacts the behavior of rules that depend on this context.
         *        For example, if you have a stateful rule to <code>drop http</code> traffic, Network Firewall won't
         *        match the traffic for this rule because the service won't have the context from session initialization
         *        defining the application layer protocol as HTTP. However, this behavior is rule dependent—a TCP-layer
         *        rule using a <code>flow:stateless</code> rule would still match, as would the
         *        <code>aws:drop_strict</code> default action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REJECT</code> - Network Firewall fails closed and drops all subsequent traffic going to the
         *        firewall. Network Firewall also sends a TCP reject packet back to your client so that the client can
         *        immediately establish a new session. Network Firewall will have context about the new session and will
         *        apply rules to the subsequent traffic.
         *        </p>
         *        </li>
         * @see StreamExceptionPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StreamExceptionPolicy
         */
        Builder streamExceptionPolicy(StreamExceptionPolicy streamExceptionPolicy);

        /**
         * <p>
         * Configures the amount of time that can pass without any traffic sent through the firewall before the firewall
         * determines that the connection is idle.
         * </p>
         * 
         * @param flowTimeouts
         *        Configures the amount of time that can pass without any traffic sent through the firewall before the
         *        firewall determines that the connection is idle.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder flowTimeouts(FlowTimeouts flowTimeouts);

        /**
         * <p>
         * Configures the amount of time that can pass without any traffic sent through the firewall before the firewall
         * determines that the connection is idle.
         * </p>
         * This is a convenience method that creates an instance of the {@link FlowTimeouts.Builder} avoiding the need
         * to create one manually via {@link FlowTimeouts#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FlowTimeouts.Builder#build()} is called immediately and its
         * result is passed to {@link #flowTimeouts(FlowTimeouts)}.
         * 
         * @param flowTimeouts
         *        a consumer that will call methods on {@link FlowTimeouts.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #flowTimeouts(FlowTimeouts)
         */
        default Builder flowTimeouts(Consumer<FlowTimeouts.Builder> flowTimeouts) {
            return flowTimeouts(FlowTimeouts.builder().applyMutation(flowTimeouts).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String ruleOrder;

        private String streamExceptionPolicy;

        private FlowTimeouts flowTimeouts;

        private BuilderImpl() {
        }

        private BuilderImpl(StatefulEngineOptions model) {
            ruleOrder(model.ruleOrder);
            streamExceptionPolicy(model.streamExceptionPolicy);
            flowTimeouts(model.flowTimeouts);
        }

        public final String getRuleOrder() {
            return ruleOrder;
        }

        public final void setRuleOrder(String ruleOrder) {
            this.ruleOrder = ruleOrder;
        }

        @Override
        public final Builder ruleOrder(String ruleOrder) {
            this.ruleOrder = ruleOrder;
            return this;
        }

        @Override
        public final Builder ruleOrder(RuleOrder ruleOrder) {
            this.ruleOrder(ruleOrder == null ? null : ruleOrder.toString());
            return this;
        }

        public final String getStreamExceptionPolicy() {
            return streamExceptionPolicy;
        }

        public final void setStreamExceptionPolicy(String streamExceptionPolicy) {
            this.streamExceptionPolicy = streamExceptionPolicy;
        }

        @Override
        public final Builder streamExceptionPolicy(String streamExceptionPolicy) {
            this.streamExceptionPolicy = streamExceptionPolicy;
            return this;
        }

        @Override
        public final Builder streamExceptionPolicy(StreamExceptionPolicy streamExceptionPolicy) {
            this.streamExceptionPolicy(streamExceptionPolicy == null ? null : streamExceptionPolicy.toString());
            return this;
        }

        public final FlowTimeouts.Builder getFlowTimeouts() {
            return flowTimeouts != null ? flowTimeouts.toBuilder() : null;
        }

        public final void setFlowTimeouts(FlowTimeouts.BuilderImpl flowTimeouts) {
            this.flowTimeouts = flowTimeouts != null ? flowTimeouts.build() : null;
        }

        @Override
        public final Builder flowTimeouts(FlowTimeouts flowTimeouts) {
            this.flowTimeouts = flowTimeouts;
            return this;
        }

        @Override
        public StatefulEngineOptions build() {
            return new StatefulEngineOptions(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
