/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkmanager.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedItemsIterable;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.networkmanager.NetworkManagerClient;
import software.amazon.awssdk.services.networkmanager.model.CustomerGatewayAssociation;
import software.amazon.awssdk.services.networkmanager.model.GetCustomerGatewayAssociationsRequest;
import software.amazon.awssdk.services.networkmanager.model.GetCustomerGatewayAssociationsResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.networkmanager.NetworkManagerClient#getCustomerGatewayAssociationsPaginator(software.amazon.awssdk.services.networkmanager.model.GetCustomerGatewayAssociationsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.networkmanager.model.GetCustomerGatewayAssociationsResponse} that can be used
 * to iterate through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.networkmanager.paginators.GetCustomerGatewayAssociationsIterable responses = client.getCustomerGatewayAssociationsPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.networkmanager.paginators.GetCustomerGatewayAssociationsIterable responses = client
 *             .getCustomerGatewayAssociationsPaginator(request);
 *     for (software.amazon.awssdk.services.networkmanager.model.GetCustomerGatewayAssociationsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.networkmanager.paginators.GetCustomerGatewayAssociationsIterable responses = client.getCustomerGatewayAssociationsPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #getCustomerGatewayAssociations(software.amazon.awssdk.services.networkmanager.model.GetCustomerGatewayAssociationsRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class GetCustomerGatewayAssociationsIterable implements SdkIterable<GetCustomerGatewayAssociationsResponse> {
    private final NetworkManagerClient client;

    private final GetCustomerGatewayAssociationsRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public GetCustomerGatewayAssociationsIterable(NetworkManagerClient client, GetCustomerGatewayAssociationsRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new GetCustomerGatewayAssociationsResponseFetcher();
    }

    @Override
    public Iterator<GetCustomerGatewayAssociationsResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    /**
     * Returns an iterable to iterate through the paginated
     * {@link GetCustomerGatewayAssociationsResponse#customerGatewayAssociations()} member. The returned iterable is
     * used to iterate through the results across all response pages and not a single page.
     *
     * This method is useful if you are interested in iterating over the paginated member in the response pages instead
     * of the top level pages. Similar to iteration over pages, this method internally makes service calls to get the
     * next list of results until the iteration stops or there are no more results.
     */
    public final SdkIterable<CustomerGatewayAssociation> customerGatewayAssociations() {
        Function<GetCustomerGatewayAssociationsResponse, Iterator<CustomerGatewayAssociation>> getIterator = response -> {
            if (response != null && response.customerGatewayAssociations() != null) {
                return response.customerGatewayAssociations().iterator();
            }
            return Collections.emptyIterator();
        };
        return PaginatedItemsIterable.builder().pagesIterable(this).itemIteratorFunction(getIterator).build();
    }

    private class GetCustomerGatewayAssociationsResponseFetcher implements
            SyncPageFetcher<GetCustomerGatewayAssociationsResponse> {
        @Override
        public boolean hasNextPage(GetCustomerGatewayAssociationsResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public GetCustomerGatewayAssociationsResponse nextPage(GetCustomerGatewayAssociationsResponse previousPage) {
            if (previousPage == null) {
                return client.getCustomerGatewayAssociations(firstRequest);
            }
            return client.getCustomerGatewayAssociations(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
