/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkmanager.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the action that the service insertion will take for any segments associated with it.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ServiceInsertionAction implements SdkPojo, Serializable,
        ToCopyableBuilder<ServiceInsertionAction.Builder, ServiceInsertionAction> {
    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Action")
            .getter(getter(ServiceInsertionAction::actionAsString)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<String> MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Mode")
            .getter(getter(ServiceInsertionAction::modeAsString)).setter(setter(Builder::mode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Mode").build()).build();

    private static final SdkField<WhenSentTo> WHEN_SENT_TO_FIELD = SdkField.<WhenSentTo> builder(MarshallingType.SDK_POJO)
            .memberName("WhenSentTo").getter(getter(ServiceInsertionAction::whenSentTo)).setter(setter(Builder::whenSentTo))
            .constructor(WhenSentTo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WhenSentTo").build()).build();

    private static final SdkField<Via> VIA_FIELD = SdkField.<Via> builder(MarshallingType.SDK_POJO).memberName("Via")
            .getter(getter(ServiceInsertionAction::via)).setter(setter(Builder::via)).constructor(Via::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Via").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACTION_FIELD, MODE_FIELD,
            WHEN_SENT_TO_FIELD, VIA_FIELD));

    private static final long serialVersionUID = 1L;

    private final String action;

    private final String mode;

    private final WhenSentTo whenSentTo;

    private final Via via;

    private ServiceInsertionAction(BuilderImpl builder) {
        this.action = builder.action;
        this.mode = builder.mode;
        this.whenSentTo = builder.whenSentTo;
        this.via = builder.via;
    }

    /**
     * <p>
     * The action the service insertion takes for traffic. <code>send-via</code> sends east-west traffic between
     * attachments. <code>send-to</code> sends north-south traffic to the security appliance, and then from that to
     * either the Internet or to an on-premesis location.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link SegmentActionServiceInsertion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #actionAsString}.
     * </p>
     * 
     * @return The action the service insertion takes for traffic. <code>send-via</code> sends east-west traffic between
     *         attachments. <code>send-to</code> sends north-south traffic to the security appliance, and then from that
     *         to either the Internet or to an on-premesis location.
     * @see SegmentActionServiceInsertion
     */
    public final SegmentActionServiceInsertion action() {
        return SegmentActionServiceInsertion.fromValue(action);
    }

    /**
     * <p>
     * The action the service insertion takes for traffic. <code>send-via</code> sends east-west traffic between
     * attachments. <code>send-to</code> sends north-south traffic to the security appliance, and then from that to
     * either the Internet or to an on-premesis location.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link SegmentActionServiceInsertion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #actionAsString}.
     * </p>
     * 
     * @return The action the service insertion takes for traffic. <code>send-via</code> sends east-west traffic between
     *         attachments. <code>send-to</code> sends north-south traffic to the security appliance, and then from that
     *         to either the Internet or to an on-premesis location.
     * @see SegmentActionServiceInsertion
     */
    public final String actionAsString() {
        return action;
    }

    /**
     * <p>
     * Describes the mode packets take for the <code>send-via</code> action. This is not used when the action is
     * <code>send-to</code>. <code>dual-hop</code> packets traverse attachments in both the source to the destination
     * core network edges. This mode requires that an inspection attachment must be present in all Regions of the
     * service insertion-enabled segments. For <code>single-hop</code>, packets traverse a single intermediate inserted
     * attachment. You can use <code>EdgeOverride</code> to specify a specific edge to use.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link SendViaMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return Describes the mode packets take for the <code>send-via</code> action. This is not used when the action is
     *         <code>send-to</code>. <code>dual-hop</code> packets traverse attachments in both the source to the
     *         destination core network edges. This mode requires that an inspection attachment must be present in all
     *         Regions of the service insertion-enabled segments. For <code>single-hop</code>, packets traverse a single
     *         intermediate inserted attachment. You can use <code>EdgeOverride</code> to specify a specific edge to
     *         use.
     * @see SendViaMode
     */
    public final SendViaMode mode() {
        return SendViaMode.fromValue(mode);
    }

    /**
     * <p>
     * Describes the mode packets take for the <code>send-via</code> action. This is not used when the action is
     * <code>send-to</code>. <code>dual-hop</code> packets traverse attachments in both the source to the destination
     * core network edges. This mode requires that an inspection attachment must be present in all Regions of the
     * service insertion-enabled segments. For <code>single-hop</code>, packets traverse a single intermediate inserted
     * attachment. You can use <code>EdgeOverride</code> to specify a specific edge to use.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link SendViaMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return Describes the mode packets take for the <code>send-via</code> action. This is not used when the action is
     *         <code>send-to</code>. <code>dual-hop</code> packets traverse attachments in both the source to the
     *         destination core network edges. This mode requires that an inspection attachment must be present in all
     *         Regions of the service insertion-enabled segments. For <code>single-hop</code>, packets traverse a single
     *         intermediate inserted attachment. You can use <code>EdgeOverride</code> to specify a specific edge to
     *         use.
     * @see SendViaMode
     */
    public final String modeAsString() {
        return mode;
    }

    /**
     * <p>
     * The list of destination segments if the service insertion action is <code>send-via</code>.
     * </p>
     * 
     * @return The list of destination segments if the service insertion action is <code>send-via</code>.
     */
    public final WhenSentTo whenSentTo() {
        return whenSentTo;
    }

    /**
     * <p>
     * The list of network function groups and any edge overrides for the chosen service insertion action. Used for both
     * <code>send-to</code> or <code>send-via</code>.
     * </p>
     * 
     * @return The list of network function groups and any edge overrides for the chosen service insertion action. Used
     *         for both <code>send-to</code> or <code>send-via</code>.
     */
    public final Via via() {
        return via;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(modeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(whenSentTo());
        hashCode = 31 * hashCode + Objects.hashCode(via());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ServiceInsertionAction)) {
            return false;
        }
        ServiceInsertionAction other = (ServiceInsertionAction) obj;
        return Objects.equals(actionAsString(), other.actionAsString()) && Objects.equals(modeAsString(), other.modeAsString())
                && Objects.equals(whenSentTo(), other.whenSentTo()) && Objects.equals(via(), other.via());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ServiceInsertionAction").add("Action", actionAsString()).add("Mode", modeAsString())
                .add("WhenSentTo", whenSentTo()).add("Via", via()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        case "Mode":
            return Optional.ofNullable(clazz.cast(modeAsString()));
        case "WhenSentTo":
            return Optional.ofNullable(clazz.cast(whenSentTo()));
        case "Via":
            return Optional.ofNullable(clazz.cast(via()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ServiceInsertionAction, T> g) {
        return obj -> g.apply((ServiceInsertionAction) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ServiceInsertionAction> {
        /**
         * <p>
         * The action the service insertion takes for traffic. <code>send-via</code> sends east-west traffic between
         * attachments. <code>send-to</code> sends north-south traffic to the security appliance, and then from that to
         * either the Internet or to an on-premesis location.
         * </p>
         * 
         * @param action
         *        The action the service insertion takes for traffic. <code>send-via</code> sends east-west traffic
         *        between attachments. <code>send-to</code> sends north-south traffic to the security appliance, and
         *        then from that to either the Internet or to an on-premesis location.
         * @see SegmentActionServiceInsertion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SegmentActionServiceInsertion
         */
        Builder action(String action);

        /**
         * <p>
         * The action the service insertion takes for traffic. <code>send-via</code> sends east-west traffic between
         * attachments. <code>send-to</code> sends north-south traffic to the security appliance, and then from that to
         * either the Internet or to an on-premesis location.
         * </p>
         * 
         * @param action
         *        The action the service insertion takes for traffic. <code>send-via</code> sends east-west traffic
         *        between attachments. <code>send-to</code> sends north-south traffic to the security appliance, and
         *        then from that to either the Internet or to an on-premesis location.
         * @see SegmentActionServiceInsertion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SegmentActionServiceInsertion
         */
        Builder action(SegmentActionServiceInsertion action);

        /**
         * <p>
         * Describes the mode packets take for the <code>send-via</code> action. This is not used when the action is
         * <code>send-to</code>. <code>dual-hop</code> packets traverse attachments in both the source to the
         * destination core network edges. This mode requires that an inspection attachment must be present in all
         * Regions of the service insertion-enabled segments. For <code>single-hop</code>, packets traverse a single
         * intermediate inserted attachment. You can use <code>EdgeOverride</code> to specify a specific edge to use.
         * </p>
         * 
         * @param mode
         *        Describes the mode packets take for the <code>send-via</code> action. This is not used when the action
         *        is <code>send-to</code>. <code>dual-hop</code> packets traverse attachments in both the source to the
         *        destination core network edges. This mode requires that an inspection attachment must be present in
         *        all Regions of the service insertion-enabled segments. For <code>single-hop</code>, packets traverse a
         *        single intermediate inserted attachment. You can use <code>EdgeOverride</code> to specify a specific
         *        edge to use.
         * @see SendViaMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SendViaMode
         */
        Builder mode(String mode);

        /**
         * <p>
         * Describes the mode packets take for the <code>send-via</code> action. This is not used when the action is
         * <code>send-to</code>. <code>dual-hop</code> packets traverse attachments in both the source to the
         * destination core network edges. This mode requires that an inspection attachment must be present in all
         * Regions of the service insertion-enabled segments. For <code>single-hop</code>, packets traverse a single
         * intermediate inserted attachment. You can use <code>EdgeOverride</code> to specify a specific edge to use.
         * </p>
         * 
         * @param mode
         *        Describes the mode packets take for the <code>send-via</code> action. This is not used when the action
         *        is <code>send-to</code>. <code>dual-hop</code> packets traverse attachments in both the source to the
         *        destination core network edges. This mode requires that an inspection attachment must be present in
         *        all Regions of the service insertion-enabled segments. For <code>single-hop</code>, packets traverse a
         *        single intermediate inserted attachment. You can use <code>EdgeOverride</code> to specify a specific
         *        edge to use.
         * @see SendViaMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SendViaMode
         */
        Builder mode(SendViaMode mode);

        /**
         * <p>
         * The list of destination segments if the service insertion action is <code>send-via</code>.
         * </p>
         * 
         * @param whenSentTo
         *        The list of destination segments if the service insertion action is <code>send-via</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder whenSentTo(WhenSentTo whenSentTo);

        /**
         * <p>
         * The list of destination segments if the service insertion action is <code>send-via</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link WhenSentTo.Builder} avoiding the need to
         * create one manually via {@link WhenSentTo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link WhenSentTo.Builder#build()} is called immediately and its result
         * is passed to {@link #whenSentTo(WhenSentTo)}.
         * 
         * @param whenSentTo
         *        a consumer that will call methods on {@link WhenSentTo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #whenSentTo(WhenSentTo)
         */
        default Builder whenSentTo(Consumer<WhenSentTo.Builder> whenSentTo) {
            return whenSentTo(WhenSentTo.builder().applyMutation(whenSentTo).build());
        }

        /**
         * <p>
         * The list of network function groups and any edge overrides for the chosen service insertion action. Used for
         * both <code>send-to</code> or <code>send-via</code>.
         * </p>
         * 
         * @param via
         *        The list of network function groups and any edge overrides for the chosen service insertion action.
         *        Used for both <code>send-to</code> or <code>send-via</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder via(Via via);

        /**
         * <p>
         * The list of network function groups and any edge overrides for the chosen service insertion action. Used for
         * both <code>send-to</code> or <code>send-via</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link Via.Builder} avoiding the need to create
         * one manually via {@link Via#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Via.Builder#build()} is called immediately and its result is
         * passed to {@link #via(Via)}.
         * 
         * @param via
         *        a consumer that will call methods on {@link Via.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #via(Via)
         */
        default Builder via(Consumer<Via.Builder> via) {
            return via(Via.builder().applyMutation(via).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String action;

        private String mode;

        private WhenSentTo whenSentTo;

        private Via via;

        private BuilderImpl() {
        }

        private BuilderImpl(ServiceInsertionAction model) {
            action(model.action);
            mode(model.mode);
            whenSentTo(model.whenSentTo);
            via(model.via);
        }

        public final String getAction() {
            return action;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        public final Builder action(SegmentActionServiceInsertion action) {
            this.action(action == null ? null : action.toString());
            return this;
        }

        public final String getMode() {
            return mode;
        }

        public final void setMode(String mode) {
            this.mode = mode;
        }

        @Override
        public final Builder mode(String mode) {
            this.mode = mode;
            return this;
        }

        @Override
        public final Builder mode(SendViaMode mode) {
            this.mode(mode == null ? null : mode.toString());
            return this;
        }

        public final WhenSentTo.Builder getWhenSentTo() {
            return whenSentTo != null ? whenSentTo.toBuilder() : null;
        }

        public final void setWhenSentTo(WhenSentTo.BuilderImpl whenSentTo) {
            this.whenSentTo = whenSentTo != null ? whenSentTo.build() : null;
        }

        @Override
        public final Builder whenSentTo(WhenSentTo whenSentTo) {
            this.whenSentTo = whenSentTo;
            return this;
        }

        public final Via.Builder getVia() {
            return via != null ? via.toBuilder() : null;
        }

        public final void setVia(Via.BuilderImpl via) {
            this.via = via != null ? via.build() : null;
        }

        @Override
        public final Builder via(Via via) {
            this.via = via;
            return this;
        }

        @Override
        public ServiceInsertionAction build() {
            return new ServiceInsertionAction(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
