/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.nimble.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration for streaming workstations created using this launch profile.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StreamConfigurationCreate implements SdkPojo, Serializable,
        ToCopyableBuilder<StreamConfigurationCreate.Builder, StreamConfigurationCreate> {
    private static final SdkField<String> CLIPBOARD_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("clipboardMode").getter(getter(StreamConfigurationCreate::clipboardModeAsString))
            .setter(setter(Builder::clipboardMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clipboardMode").build()).build();

    private static final SdkField<List<String>> EC2_INSTANCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ec2InstanceTypes")
            .getter(getter(StreamConfigurationCreate::ec2InstanceTypesAsStrings))
            .setter(setter(Builder::ec2InstanceTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ec2InstanceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> MAX_SESSION_LENGTH_IN_MINUTES_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("maxSessionLengthInMinutes")
            .getter(getter(StreamConfigurationCreate::maxSessionLengthInMinutes))
            .setter(setter(Builder::maxSessionLengthInMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxSessionLengthInMinutes").build())
            .build();

    private static final SdkField<Integer> MAX_STOPPED_SESSION_LENGTH_IN_MINUTES_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxStoppedSessionLengthInMinutes")
            .getter(getter(StreamConfigurationCreate::maxStoppedSessionLengthInMinutes))
            .setter(setter(Builder::maxStoppedSessionLengthInMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxStoppedSessionLengthInMinutes")
                    .build()).build();

    private static final SdkField<StreamConfigurationSessionStorage> SESSION_STORAGE_FIELD = SdkField
            .<StreamConfigurationSessionStorage> builder(MarshallingType.SDK_POJO).memberName("sessionStorage")
            .getter(getter(StreamConfigurationCreate::sessionStorage)).setter(setter(Builder::sessionStorage))
            .constructor(StreamConfigurationSessionStorage::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sessionStorage").build()).build();

    private static final SdkField<List<String>> STREAMING_IMAGE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("streamingImageIds")
            .getter(getter(StreamConfigurationCreate::streamingImageIds))
            .setter(setter(Builder::streamingImageIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("streamingImageIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLIPBOARD_MODE_FIELD,
            EC2_INSTANCE_TYPES_FIELD, MAX_SESSION_LENGTH_IN_MINUTES_FIELD, MAX_STOPPED_SESSION_LENGTH_IN_MINUTES_FIELD,
            SESSION_STORAGE_FIELD, STREAMING_IMAGE_IDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String clipboardMode;

    private final List<String> ec2InstanceTypes;

    private final Integer maxSessionLengthInMinutes;

    private final Integer maxStoppedSessionLengthInMinutes;

    private final StreamConfigurationSessionStorage sessionStorage;

    private final List<String> streamingImageIds;

    private StreamConfigurationCreate(BuilderImpl builder) {
        this.clipboardMode = builder.clipboardMode;
        this.ec2InstanceTypes = builder.ec2InstanceTypes;
        this.maxSessionLengthInMinutes = builder.maxSessionLengthInMinutes;
        this.maxStoppedSessionLengthInMinutes = builder.maxStoppedSessionLengthInMinutes;
        this.sessionStorage = builder.sessionStorage;
        this.streamingImageIds = builder.streamingImageIds;
    }

    /**
     * <p>
     * Enable or disable the use of the system clipboard to copy and paste between the streaming session and streaming
     * client.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #clipboardMode}
     * will return {@link StreamingClipboardMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #clipboardModeAsString}.
     * </p>
     * 
     * @return Enable or disable the use of the system clipboard to copy and paste between the streaming session and
     *         streaming client.
     * @see StreamingClipboardMode
     */
    public final StreamingClipboardMode clipboardMode() {
        return StreamingClipboardMode.fromValue(clipboardMode);
    }

    /**
     * <p>
     * Enable or disable the use of the system clipboard to copy and paste between the streaming session and streaming
     * client.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #clipboardMode}
     * will return {@link StreamingClipboardMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #clipboardModeAsString}.
     * </p>
     * 
     * @return Enable or disable the use of the system clipboard to copy and paste between the streaming session and
     *         streaming client.
     * @see StreamingClipboardMode
     */
    public final String clipboardModeAsString() {
        return clipboardMode;
    }

    /**
     * <p>
     * The EC2 instance types that users can select from when launching a streaming session with this launch profile.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEc2InstanceTypes} method.
     * </p>
     * 
     * @return The EC2 instance types that users can select from when launching a streaming session with this launch
     *         profile.
     */
    public final List<StreamingInstanceType> ec2InstanceTypes() {
        return StreamingInstanceTypeListCopier.copyStringToEnum(ec2InstanceTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the Ec2InstanceTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEc2InstanceTypes() {
        return ec2InstanceTypes != null && !(ec2InstanceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The EC2 instance types that users can select from when launching a streaming session with this launch profile.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEc2InstanceTypes} method.
     * </p>
     * 
     * @return The EC2 instance types that users can select from when launching a streaming session with this launch
     *         profile.
     */
    public final List<String> ec2InstanceTypesAsStrings() {
        return ec2InstanceTypes;
    }

    /**
     * <p>
     * The length of time, in minutes, that a streaming session can be active before it is stopped or terminated. After
     * this point, Nimble Studio automatically terminates or stops the session. The default length of time is 690
     * minutes, and the maximum length of time is 30 days.
     * </p>
     * 
     * @return The length of time, in minutes, that a streaming session can be active before it is stopped or
     *         terminated. After this point, Nimble Studio automatically terminates or stops the session. The default
     *         length of time is 690 minutes, and the maximum length of time is 30 days.
     */
    public final Integer maxSessionLengthInMinutes() {
        return maxSessionLengthInMinutes;
    }

    /**
     * <p>
     * Integer that determines if you can start and stop your sessions and how long a session can stay in the STOPPED
     * state. The default value is 0. The maximum value is 5760.
     * </p>
     * <p>
     * If the value is missing or set to 0, your sessions can’t be stopped. If you then call
     * <code>StopStreamingSession</code>, the session fails. If the time that a session stays in the READY state exceeds
     * the <code>maxSessionLengthInMinutes</code> value, the session will automatically be terminated (instead of
     * stopped).
     * </p>
     * <p>
     * If the value is set to a positive number, the session can be stopped. You can call
     * <code>StopStreamingSession</code> to stop sessions in the READY state. If the time that a session stays in the
     * READY state exceeds the <code>maxSessionLengthInMinutes</code> value, the session will automatically be stopped
     * (instead of terminated).
     * </p>
     * 
     * @return Integer that determines if you can start and stop your sessions and how long a session can stay in the
     *         STOPPED state. The default value is 0. The maximum value is 5760.</p>
     *         <p>
     *         If the value is missing or set to 0, your sessions can’t be stopped. If you then call
     *         <code>StopStreamingSession</code>, the session fails. If the time that a session stays in the READY state
     *         exceeds the <code>maxSessionLengthInMinutes</code> value, the session will automatically be terminated
     *         (instead of stopped).
     *         </p>
     *         <p>
     *         If the value is set to a positive number, the session can be stopped. You can call
     *         <code>StopStreamingSession</code> to stop sessions in the READY state. If the time that a session stays
     *         in the READY state exceeds the <code>maxSessionLengthInMinutes</code> value, the session will
     *         automatically be stopped (instead of terminated).
     */
    public final Integer maxStoppedSessionLengthInMinutes() {
        return maxStoppedSessionLengthInMinutes;
    }

    /**
     * <p>
     * (Optional) The upload storage for a streaming workstation that is created using this launch profile.
     * </p>
     * 
     * @return (Optional) The upload storage for a streaming workstation that is created using this launch profile.
     */
    public final StreamConfigurationSessionStorage sessionStorage() {
        return sessionStorage;
    }

    /**
     * For responses, this returns true if the service returned a value for the StreamingImageIds property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStreamingImageIds() {
        return streamingImageIds != null && !(streamingImageIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The streaming images that users can select from when launching a streaming session with this launch profile.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStreamingImageIds} method.
     * </p>
     * 
     * @return The streaming images that users can select from when launching a streaming session with this launch
     *         profile.
     */
    public final List<String> streamingImageIds() {
        return streamingImageIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(clipboardModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasEc2InstanceTypes() ? ec2InstanceTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(maxSessionLengthInMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(maxStoppedSessionLengthInMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(sessionStorage());
        hashCode = 31 * hashCode + Objects.hashCode(hasStreamingImageIds() ? streamingImageIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StreamConfigurationCreate)) {
            return false;
        }
        StreamConfigurationCreate other = (StreamConfigurationCreate) obj;
        return Objects.equals(clipboardModeAsString(), other.clipboardModeAsString())
                && hasEc2InstanceTypes() == other.hasEc2InstanceTypes()
                && Objects.equals(ec2InstanceTypesAsStrings(), other.ec2InstanceTypesAsStrings())
                && Objects.equals(maxSessionLengthInMinutes(), other.maxSessionLengthInMinutes())
                && Objects.equals(maxStoppedSessionLengthInMinutes(), other.maxStoppedSessionLengthInMinutes())
                && Objects.equals(sessionStorage(), other.sessionStorage())
                && hasStreamingImageIds() == other.hasStreamingImageIds()
                && Objects.equals(streamingImageIds(), other.streamingImageIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StreamConfigurationCreate").add("ClipboardMode", clipboardModeAsString())
                .add("Ec2InstanceTypes", hasEc2InstanceTypes() ? ec2InstanceTypesAsStrings() : null)
                .add("MaxSessionLengthInMinutes", maxSessionLengthInMinutes())
                .add("MaxStoppedSessionLengthInMinutes", maxStoppedSessionLengthInMinutes())
                .add("SessionStorage", sessionStorage())
                .add("StreamingImageIds", hasStreamingImageIds() ? streamingImageIds() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "clipboardMode":
            return Optional.ofNullable(clazz.cast(clipboardModeAsString()));
        case "ec2InstanceTypes":
            return Optional.ofNullable(clazz.cast(ec2InstanceTypesAsStrings()));
        case "maxSessionLengthInMinutes":
            return Optional.ofNullable(clazz.cast(maxSessionLengthInMinutes()));
        case "maxStoppedSessionLengthInMinutes":
            return Optional.ofNullable(clazz.cast(maxStoppedSessionLengthInMinutes()));
        case "sessionStorage":
            return Optional.ofNullable(clazz.cast(sessionStorage()));
        case "streamingImageIds":
            return Optional.ofNullable(clazz.cast(streamingImageIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StreamConfigurationCreate, T> g) {
        return obj -> g.apply((StreamConfigurationCreate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StreamConfigurationCreate> {
        /**
         * <p>
         * Enable or disable the use of the system clipboard to copy and paste between the streaming session and
         * streaming client.
         * </p>
         * 
         * @param clipboardMode
         *        Enable or disable the use of the system clipboard to copy and paste between the streaming session and
         *        streaming client.
         * @see StreamingClipboardMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StreamingClipboardMode
         */
        Builder clipboardMode(String clipboardMode);

        /**
         * <p>
         * Enable or disable the use of the system clipboard to copy and paste between the streaming session and
         * streaming client.
         * </p>
         * 
         * @param clipboardMode
         *        Enable or disable the use of the system clipboard to copy and paste between the streaming session and
         *        streaming client.
         * @see StreamingClipboardMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StreamingClipboardMode
         */
        Builder clipboardMode(StreamingClipboardMode clipboardMode);

        /**
         * <p>
         * The EC2 instance types that users can select from when launching a streaming session with this launch
         * profile.
         * </p>
         * 
         * @param ec2InstanceTypes
         *        The EC2 instance types that users can select from when launching a streaming session with this launch
         *        profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2InstanceTypesWithStrings(Collection<String> ec2InstanceTypes);

        /**
         * <p>
         * The EC2 instance types that users can select from when launching a streaming session with this launch
         * profile.
         * </p>
         * 
         * @param ec2InstanceTypes
         *        The EC2 instance types that users can select from when launching a streaming session with this launch
         *        profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2InstanceTypesWithStrings(String... ec2InstanceTypes);

        /**
         * <p>
         * The EC2 instance types that users can select from when launching a streaming session with this launch
         * profile.
         * </p>
         * 
         * @param ec2InstanceTypes
         *        The EC2 instance types that users can select from when launching a streaming session with this launch
         *        profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2InstanceTypes(Collection<StreamingInstanceType> ec2InstanceTypes);

        /**
         * <p>
         * The EC2 instance types that users can select from when launching a streaming session with this launch
         * profile.
         * </p>
         * 
         * @param ec2InstanceTypes
         *        The EC2 instance types that users can select from when launching a streaming session with this launch
         *        profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2InstanceTypes(StreamingInstanceType... ec2InstanceTypes);

        /**
         * <p>
         * The length of time, in minutes, that a streaming session can be active before it is stopped or terminated.
         * After this point, Nimble Studio automatically terminates or stops the session. The default length of time is
         * 690 minutes, and the maximum length of time is 30 days.
         * </p>
         * 
         * @param maxSessionLengthInMinutes
         *        The length of time, in minutes, that a streaming session can be active before it is stopped or
         *        terminated. After this point, Nimble Studio automatically terminates or stops the session. The default
         *        length of time is 690 minutes, and the maximum length of time is 30 days.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxSessionLengthInMinutes(Integer maxSessionLengthInMinutes);

        /**
         * <p>
         * Integer that determines if you can start and stop your sessions and how long a session can stay in the
         * STOPPED state. The default value is 0. The maximum value is 5760.
         * </p>
         * <p>
         * If the value is missing or set to 0, your sessions can’t be stopped. If you then call
         * <code>StopStreamingSession</code>, the session fails. If the time that a session stays in the READY state
         * exceeds the <code>maxSessionLengthInMinutes</code> value, the session will automatically be terminated
         * (instead of stopped).
         * </p>
         * <p>
         * If the value is set to a positive number, the session can be stopped. You can call
         * <code>StopStreamingSession</code> to stop sessions in the READY state. If the time that a session stays in
         * the READY state exceeds the <code>maxSessionLengthInMinutes</code> value, the session will automatically be
         * stopped (instead of terminated).
         * </p>
         * 
         * @param maxStoppedSessionLengthInMinutes
         *        Integer that determines if you can start and stop your sessions and how long a session can stay in the
         *        STOPPED state. The default value is 0. The maximum value is 5760.</p>
         *        <p>
         *        If the value is missing or set to 0, your sessions can’t be stopped. If you then call
         *        <code>StopStreamingSession</code>, the session fails. If the time that a session stays in the READY
         *        state exceeds the <code>maxSessionLengthInMinutes</code> value, the session will automatically be
         *        terminated (instead of stopped).
         *        </p>
         *        <p>
         *        If the value is set to a positive number, the session can be stopped. You can call
         *        <code>StopStreamingSession</code> to stop sessions in the READY state. If the time that a session
         *        stays in the READY state exceeds the <code>maxSessionLengthInMinutes</code> value, the session will
         *        automatically be stopped (instead of terminated).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxStoppedSessionLengthInMinutes(Integer maxStoppedSessionLengthInMinutes);

        /**
         * <p>
         * (Optional) The upload storage for a streaming workstation that is created using this launch profile.
         * </p>
         * 
         * @param sessionStorage
         *        (Optional) The upload storage for a streaming workstation that is created using this launch profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sessionStorage(StreamConfigurationSessionStorage sessionStorage);

        /**
         * <p>
         * (Optional) The upload storage for a streaming workstation that is created using this launch profile.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link StreamConfigurationSessionStorage.Builder} avoiding the need to create one manually via
         * {@link StreamConfigurationSessionStorage#builder()}.
         *
         * When the {@link Consumer} completes, {@link StreamConfigurationSessionStorage.Builder#build()} is called
         * immediately and its result is passed to {@link #sessionStorage(StreamConfigurationSessionStorage)}.
         * 
         * @param sessionStorage
         *        a consumer that will call methods on {@link StreamConfigurationSessionStorage.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sessionStorage(StreamConfigurationSessionStorage)
         */
        default Builder sessionStorage(Consumer<StreamConfigurationSessionStorage.Builder> sessionStorage) {
            return sessionStorage(StreamConfigurationSessionStorage.builder().applyMutation(sessionStorage).build());
        }

        /**
         * <p>
         * The streaming images that users can select from when launching a streaming session with this launch profile.
         * </p>
         * 
         * @param streamingImageIds
         *        The streaming images that users can select from when launching a streaming session with this launch
         *        profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamingImageIds(Collection<String> streamingImageIds);

        /**
         * <p>
         * The streaming images that users can select from when launching a streaming session with this launch profile.
         * </p>
         * 
         * @param streamingImageIds
         *        The streaming images that users can select from when launching a streaming session with this launch
         *        profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamingImageIds(String... streamingImageIds);
    }

    static final class BuilderImpl implements Builder {
        private String clipboardMode;

        private List<String> ec2InstanceTypes = DefaultSdkAutoConstructList.getInstance();

        private Integer maxSessionLengthInMinutes;

        private Integer maxStoppedSessionLengthInMinutes;

        private StreamConfigurationSessionStorage sessionStorage;

        private List<String> streamingImageIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(StreamConfigurationCreate model) {
            clipboardMode(model.clipboardMode);
            ec2InstanceTypesWithStrings(model.ec2InstanceTypes);
            maxSessionLengthInMinutes(model.maxSessionLengthInMinutes);
            maxStoppedSessionLengthInMinutes(model.maxStoppedSessionLengthInMinutes);
            sessionStorage(model.sessionStorage);
            streamingImageIds(model.streamingImageIds);
        }

        public final String getClipboardMode() {
            return clipboardMode;
        }

        public final void setClipboardMode(String clipboardMode) {
            this.clipboardMode = clipboardMode;
        }

        @Override
        public final Builder clipboardMode(String clipboardMode) {
            this.clipboardMode = clipboardMode;
            return this;
        }

        @Override
        public final Builder clipboardMode(StreamingClipboardMode clipboardMode) {
            this.clipboardMode(clipboardMode == null ? null : clipboardMode.toString());
            return this;
        }

        public final Collection<String> getEc2InstanceTypes() {
            if (ec2InstanceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return ec2InstanceTypes;
        }

        public final void setEc2InstanceTypes(Collection<String> ec2InstanceTypes) {
            this.ec2InstanceTypes = StreamingInstanceTypeListCopier.copy(ec2InstanceTypes);
        }

        @Override
        public final Builder ec2InstanceTypesWithStrings(Collection<String> ec2InstanceTypes) {
            this.ec2InstanceTypes = StreamingInstanceTypeListCopier.copy(ec2InstanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ec2InstanceTypesWithStrings(String... ec2InstanceTypes) {
            ec2InstanceTypesWithStrings(Arrays.asList(ec2InstanceTypes));
            return this;
        }

        @Override
        public final Builder ec2InstanceTypes(Collection<StreamingInstanceType> ec2InstanceTypes) {
            this.ec2InstanceTypes = StreamingInstanceTypeListCopier.copyEnumToString(ec2InstanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ec2InstanceTypes(StreamingInstanceType... ec2InstanceTypes) {
            ec2InstanceTypes(Arrays.asList(ec2InstanceTypes));
            return this;
        }

        public final Integer getMaxSessionLengthInMinutes() {
            return maxSessionLengthInMinutes;
        }

        public final void setMaxSessionLengthInMinutes(Integer maxSessionLengthInMinutes) {
            this.maxSessionLengthInMinutes = maxSessionLengthInMinutes;
        }

        @Override
        public final Builder maxSessionLengthInMinutes(Integer maxSessionLengthInMinutes) {
            this.maxSessionLengthInMinutes = maxSessionLengthInMinutes;
            return this;
        }

        public final Integer getMaxStoppedSessionLengthInMinutes() {
            return maxStoppedSessionLengthInMinutes;
        }

        public final void setMaxStoppedSessionLengthInMinutes(Integer maxStoppedSessionLengthInMinutes) {
            this.maxStoppedSessionLengthInMinutes = maxStoppedSessionLengthInMinutes;
        }

        @Override
        public final Builder maxStoppedSessionLengthInMinutes(Integer maxStoppedSessionLengthInMinutes) {
            this.maxStoppedSessionLengthInMinutes = maxStoppedSessionLengthInMinutes;
            return this;
        }

        public final StreamConfigurationSessionStorage.Builder getSessionStorage() {
            return sessionStorage != null ? sessionStorage.toBuilder() : null;
        }

        public final void setSessionStorage(StreamConfigurationSessionStorage.BuilderImpl sessionStorage) {
            this.sessionStorage = sessionStorage != null ? sessionStorage.build() : null;
        }

        @Override
        public final Builder sessionStorage(StreamConfigurationSessionStorage sessionStorage) {
            this.sessionStorage = sessionStorage;
            return this;
        }

        public final Collection<String> getStreamingImageIds() {
            if (streamingImageIds instanceof SdkAutoConstructList) {
                return null;
            }
            return streamingImageIds;
        }

        public final void setStreamingImageIds(Collection<String> streamingImageIds) {
            this.streamingImageIds = StreamingImageIdListCopier.copy(streamingImageIds);
        }

        @Override
        public final Builder streamingImageIds(Collection<String> streamingImageIds) {
            this.streamingImageIds = StreamingImageIdListCopier.copy(streamingImageIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder streamingImageIds(String... streamingImageIds) {
            streamingImageIds(Arrays.asList(streamingImageIds));
            return this;
        }

        @Override
        public StreamConfigurationCreate build() {
            return new StreamConfigurationCreate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
