/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.nimble.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a studio resource.
 * </p>
 * <p>
 * A studio is the core resource used with Nimble Studio. You must create a studio first, before any other resource type
 * can be created. All other resources you create and manage in Nimble Studio are contained within a studio.
 * </p>
 * <p>
 * When creating a studio, you must provides two IAM roles for use with the Nimble Studio portal. These roles are
 * assumed by your users when they log in to the Nimble Studio portal via Amazon Web Services SSO and your identity
 * source.
 * </p>
 * <p>
 * The user role must have the AmazonNimbleStudio-StudioUser managed policy attached for the portal to function
 * properly.
 * </p>
 * <p>
 * The admin role must have the AmazonNimbleStudio-StudioAdmin managed policy attached for the portal to function
 * properly.
 * </p>
 * <p>
 * Your studio roles must trust the identity.nimble.amazonaws.com service principal to function properly.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Studio implements SdkPojo, Serializable, ToCopyableBuilder<Studio.Builder, Studio> {
    private static final SdkField<String> ADMIN_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("adminRoleArn").getter(getter(Studio::adminRoleArn)).setter(setter(Builder::adminRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("adminRoleArn").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Studio::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(Studio::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> DISPLAY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("displayName").getter(getter(Studio::displayName)).setter(setter(Builder::displayName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("displayName").build()).build();

    private static final SdkField<String> HOME_REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("homeRegion").getter(getter(Studio::homeRegion)).setter(setter(Builder::homeRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("homeRegion").build()).build();

    private static final SdkField<String> SSO_CLIENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ssoClientId").getter(getter(Studio::ssoClientId)).setter(setter(Builder::ssoClientId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ssoClientId").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(Studio::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> STATUS_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusCode").getter(getter(Studio::statusCodeAsString)).setter(setter(Builder::statusCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusCode").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusMessage").getter(getter(Studio::statusMessage)).setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusMessage").build()).build();

    private static final SdkField<StudioEncryptionConfiguration> STUDIO_ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<StudioEncryptionConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("studioEncryptionConfiguration")
            .getter(getter(Studio::studioEncryptionConfiguration))
            .setter(setter(Builder::studioEncryptionConfiguration))
            .constructor(StudioEncryptionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("studioEncryptionConfiguration")
                    .build()).build();

    private static final SdkField<String> STUDIO_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("studioId").getter(getter(Studio::studioId)).setter(setter(Builder::studioId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("studioId").build()).build();

    private static final SdkField<String> STUDIO_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("studioName").getter(getter(Studio::studioName)).setter(setter(Builder::studioName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("studioName").build()).build();

    private static final SdkField<String> STUDIO_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("studioUrl").getter(getter(Studio::studioUrl)).setter(setter(Builder::studioUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("studioUrl").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(Studio::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt")
            .getter(getter(Studio::updatedAt))
            .setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> USER_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("userRoleArn").getter(getter(Studio::userRoleArn)).setter(setter(Builder::userRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("userRoleArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ADMIN_ROLE_ARN_FIELD,
            ARN_FIELD, CREATED_AT_FIELD, DISPLAY_NAME_FIELD, HOME_REGION_FIELD, SSO_CLIENT_ID_FIELD, STATE_FIELD,
            STATUS_CODE_FIELD, STATUS_MESSAGE_FIELD, STUDIO_ENCRYPTION_CONFIGURATION_FIELD, STUDIO_ID_FIELD, STUDIO_NAME_FIELD,
            STUDIO_URL_FIELD, TAGS_FIELD, UPDATED_AT_FIELD, USER_ROLE_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String adminRoleArn;

    private final String arn;

    private final Instant createdAt;

    private final String displayName;

    private final String homeRegion;

    private final String ssoClientId;

    private final String state;

    private final String statusCodeValue;

    private final String statusMessage;

    private final StudioEncryptionConfiguration studioEncryptionConfiguration;

    private final String studioId;

    private final String studioName;

    private final String studioUrl;

    private final Map<String, String> tags;

    private final Instant updatedAt;

    private final String userRoleArn;

    private Studio(BuilderImpl builder) {
        this.adminRoleArn = builder.adminRoleArn;
        this.arn = builder.arn;
        this.createdAt = builder.createdAt;
        this.displayName = builder.displayName;
        this.homeRegion = builder.homeRegion;
        this.ssoClientId = builder.ssoClientId;
        this.state = builder.state;
        this.statusCodeValue = builder.statusCodeValue;
        this.statusMessage = builder.statusMessage;
        this.studioEncryptionConfiguration = builder.studioEncryptionConfiguration;
        this.studioId = builder.studioId;
        this.studioName = builder.studioName;
        this.studioUrl = builder.studioUrl;
        this.tags = builder.tags;
        this.updatedAt = builder.updatedAt;
        this.userRoleArn = builder.userRoleArn;
    }

    /**
     * <p>
     * The IAM role that studio admins assume when logging in to the Nimble Studio portal.
     * </p>
     * 
     * @return The IAM role that studio admins assume when logging in to the Nimble Studio portal.
     */
    public final String adminRoleArn() {
        return adminRoleArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that is assigned to a studio resource and uniquely identifies it. ARNs are unique
     * across all Regions.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that is assigned to a studio resource and uniquely identifies it. ARNs are
     *         unique across all Regions.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The Unix epoch timestamp in seconds for when the resource was created.
     * </p>
     * 
     * @return The Unix epoch timestamp in seconds for when the resource was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * A friendly name for the studio.
     * </p>
     * 
     * @return A friendly name for the studio.
     */
    public final String displayName() {
        return displayName;
    }

    /**
     * <p>
     * The Amazon Web Services Region where the studio resource is located.
     * </p>
     * 
     * @return The Amazon Web Services Region where the studio resource is located.
     */
    public final String homeRegion() {
        return homeRegion;
    }

    /**
     * <p>
     * The Amazon Web Services SSO application client ID used to integrate with Amazon Web Services SSO to enable Amazon
     * Web Services SSO users to log in to Nimble Studio portal.
     * </p>
     * 
     * @return The Amazon Web Services SSO application client ID used to integrate with Amazon Web Services SSO to
     *         enable Amazon Web Services SSO users to log in to Nimble Studio portal.
     */
    public final String ssoClientId() {
        return ssoClientId;
    }

    /**
     * <p>
     * The current state of the studio resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link StudioState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the studio resource.
     * @see StudioState
     */
    public final StudioState state() {
        return StudioState.fromValue(state);
    }

    /**
     * <p>
     * The current state of the studio resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link StudioState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the studio resource.
     * @see StudioState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * Status codes that provide additional detail on the studio state.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link StudioStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusCodeAsString}.
     * </p>
     * 
     * @return Status codes that provide additional detail on the studio state.
     * @see StudioStatusCode
     */
    public final StudioStatusCode statusCode() {
        return StudioStatusCode.fromValue(statusCodeValue);
    }

    /**
     * <p>
     * Status codes that provide additional detail on the studio state.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link StudioStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusCodeAsString}.
     * </p>
     * 
     * @return Status codes that provide additional detail on the studio state.
     * @see StudioStatusCode
     */
    public final String statusCodeAsString() {
        return statusCodeValue;
    }

    /**
     * <p>
     * Additional detail on the studio state.
     * </p>
     * 
     * @return Additional detail on the studio state.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    /**
     * <p>
     * Configuration of the encryption method that is used for the studio.
     * </p>
     * 
     * @return Configuration of the encryption method that is used for the studio.
     */
    public final StudioEncryptionConfiguration studioEncryptionConfiguration() {
        return studioEncryptionConfiguration;
    }

    /**
     * <p>
     * The unique identifier for a studio resource. In Nimble Studio, all other resources are contained in a studio
     * resource.
     * </p>
     * 
     * @return The unique identifier for a studio resource. In Nimble Studio, all other resources are contained in a
     *         studio resource.
     */
    public final String studioId() {
        return studioId;
    }

    /**
     * <p>
     * The name of the studio, as included in the URL when accessing it in the Nimble Studio portal.
     * </p>
     * 
     * @return The name of the studio, as included in the URL when accessing it in the Nimble Studio portal.
     */
    public final String studioName() {
        return studioName;
    }

    /**
     * <p>
     * The address of the web page for the studio.
     * </p>
     * 
     * @return The address of the web page for the studio.
     */
    public final String studioUrl() {
        return studioUrl;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A collection of labels, in the form of key:value pairs, that apply to this resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A collection of labels, in the form of key:value pairs, that apply to this resource.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The Unix epoch timestamp in seconds for when the resource was updated.
     * </p>
     * 
     * @return The Unix epoch timestamp in seconds for when the resource was updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    /**
     * <p>
     * The IAM role that studio users assume when logging in to the Nimble Studio portal.
     * </p>
     * 
     * @return The IAM role that studio users assume when logging in to the Nimble Studio portal.
     */
    public final String userRoleArn() {
        return userRoleArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(adminRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(displayName());
        hashCode = 31 * hashCode + Objects.hashCode(homeRegion());
        hashCode = 31 * hashCode + Objects.hashCode(ssoClientId());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(studioEncryptionConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(studioId());
        hashCode = 31 * hashCode + Objects.hashCode(studioName());
        hashCode = 31 * hashCode + Objects.hashCode(studioUrl());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(userRoleArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Studio)) {
            return false;
        }
        Studio other = (Studio) obj;
        return Objects.equals(adminRoleArn(), other.adminRoleArn()) && Objects.equals(arn(), other.arn())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(displayName(), other.displayName())
                && Objects.equals(homeRegion(), other.homeRegion()) && Objects.equals(ssoClientId(), other.ssoClientId())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(statusCodeAsString(), other.statusCodeAsString())
                && Objects.equals(statusMessage(), other.statusMessage())
                && Objects.equals(studioEncryptionConfiguration(), other.studioEncryptionConfiguration())
                && Objects.equals(studioId(), other.studioId()) && Objects.equals(studioName(), other.studioName())
                && Objects.equals(studioUrl(), other.studioUrl()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(updatedAt(), other.updatedAt())
                && Objects.equals(userRoleArn(), other.userRoleArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Studio").add("AdminRoleArn", adminRoleArn()).add("Arn", arn()).add("CreatedAt", createdAt())
                .add("DisplayName", displayName() == null ? null : "*** Sensitive Data Redacted ***")
                .add("HomeRegion", homeRegion()).add("SsoClientId", ssoClientId()).add("State", stateAsString())
                .add("StatusCode", statusCodeAsString()).add("StatusMessage", statusMessage())
                .add("StudioEncryptionConfiguration", studioEncryptionConfiguration()).add("StudioId", studioId())
                .add("StudioName", studioName()).add("StudioUrl", studioUrl()).add("Tags", hasTags() ? tags() : null)
                .add("UpdatedAt", updatedAt()).add("UserRoleArn", userRoleArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "adminRoleArn":
            return Optional.ofNullable(clazz.cast(adminRoleArn()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "displayName":
            return Optional.ofNullable(clazz.cast(displayName()));
        case "homeRegion":
            return Optional.ofNullable(clazz.cast(homeRegion()));
        case "ssoClientId":
            return Optional.ofNullable(clazz.cast(ssoClientId()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "statusCode":
            return Optional.ofNullable(clazz.cast(statusCodeAsString()));
        case "statusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        case "studioEncryptionConfiguration":
            return Optional.ofNullable(clazz.cast(studioEncryptionConfiguration()));
        case "studioId":
            return Optional.ofNullable(clazz.cast(studioId()));
        case "studioName":
            return Optional.ofNullable(clazz.cast(studioName()));
        case "studioUrl":
            return Optional.ofNullable(clazz.cast(studioUrl()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "userRoleArn":
            return Optional.ofNullable(clazz.cast(userRoleArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Studio, T> g) {
        return obj -> g.apply((Studio) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Studio> {
        /**
         * <p>
         * The IAM role that studio admins assume when logging in to the Nimble Studio portal.
         * </p>
         * 
         * @param adminRoleArn
         *        The IAM role that studio admins assume when logging in to the Nimble Studio portal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder adminRoleArn(String adminRoleArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) that is assigned to a studio resource and uniquely identifies it. ARNs are
         * unique across all Regions.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) that is assigned to a studio resource and uniquely identifies it. ARNs
         *        are unique across all Regions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The Unix epoch timestamp in seconds for when the resource was created.
         * </p>
         * 
         * @param createdAt
         *        The Unix epoch timestamp in seconds for when the resource was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * A friendly name for the studio.
         * </p>
         * 
         * @param displayName
         *        A friendly name for the studio.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder displayName(String displayName);

        /**
         * <p>
         * The Amazon Web Services Region where the studio resource is located.
         * </p>
         * 
         * @param homeRegion
         *        The Amazon Web Services Region where the studio resource is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder homeRegion(String homeRegion);

        /**
         * <p>
         * The Amazon Web Services SSO application client ID used to integrate with Amazon Web Services SSO to enable
         * Amazon Web Services SSO users to log in to Nimble Studio portal.
         * </p>
         * 
         * @param ssoClientId
         *        The Amazon Web Services SSO application client ID used to integrate with Amazon Web Services SSO to
         *        enable Amazon Web Services SSO users to log in to Nimble Studio portal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ssoClientId(String ssoClientId);

        /**
         * <p>
         * The current state of the studio resource.
         * </p>
         * 
         * @param state
         *        The current state of the studio resource.
         * @see StudioState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StudioState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the studio resource.
         * </p>
         * 
         * @param state
         *        The current state of the studio resource.
         * @see StudioState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StudioState
         */
        Builder state(StudioState state);

        /**
         * <p>
         * Status codes that provide additional detail on the studio state.
         * </p>
         * 
         * @param statusCodeValue
         *        Status codes that provide additional detail on the studio state.
         * @see StudioStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StudioStatusCode
         */
        Builder statusCode(String statusCodeValue);

        /**
         * <p>
         * Status codes that provide additional detail on the studio state.
         * </p>
         * 
         * @param statusCodeValue
         *        Status codes that provide additional detail on the studio state.
         * @see StudioStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StudioStatusCode
         */
        Builder statusCode(StudioStatusCode statusCodeValue);

        /**
         * <p>
         * Additional detail on the studio state.
         * </p>
         * 
         * @param statusMessage
         *        Additional detail on the studio state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * Configuration of the encryption method that is used for the studio.
         * </p>
         * 
         * @param studioEncryptionConfiguration
         *        Configuration of the encryption method that is used for the studio.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder studioEncryptionConfiguration(StudioEncryptionConfiguration studioEncryptionConfiguration);

        /**
         * <p>
         * Configuration of the encryption method that is used for the studio.
         * </p>
         * This is a convenience method that creates an instance of the {@link StudioEncryptionConfiguration.Builder}
         * avoiding the need to create one manually via {@link StudioEncryptionConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link StudioEncryptionConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #studioEncryptionConfiguration(StudioEncryptionConfiguration)}
         * .
         * 
         * @param studioEncryptionConfiguration
         *        a consumer that will call methods on {@link StudioEncryptionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #studioEncryptionConfiguration(StudioEncryptionConfiguration)
         */
        default Builder studioEncryptionConfiguration(
                Consumer<StudioEncryptionConfiguration.Builder> studioEncryptionConfiguration) {
            return studioEncryptionConfiguration(StudioEncryptionConfiguration.builder()
                    .applyMutation(studioEncryptionConfiguration).build());
        }

        /**
         * <p>
         * The unique identifier for a studio resource. In Nimble Studio, all other resources are contained in a studio
         * resource.
         * </p>
         * 
         * @param studioId
         *        The unique identifier for a studio resource. In Nimble Studio, all other resources are contained in a
         *        studio resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder studioId(String studioId);

        /**
         * <p>
         * The name of the studio, as included in the URL when accessing it in the Nimble Studio portal.
         * </p>
         * 
         * @param studioName
         *        The name of the studio, as included in the URL when accessing it in the Nimble Studio portal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder studioName(String studioName);

        /**
         * <p>
         * The address of the web page for the studio.
         * </p>
         * 
         * @param studioUrl
         *        The address of the web page for the studio.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder studioUrl(String studioUrl);

        /**
         * <p>
         * A collection of labels, in the form of key:value pairs, that apply to this resource.
         * </p>
         * 
         * @param tags
         *        A collection of labels, in the form of key:value pairs, that apply to this resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The Unix epoch timestamp in seconds for when the resource was updated.
         * </p>
         * 
         * @param updatedAt
         *        The Unix epoch timestamp in seconds for when the resource was updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);

        /**
         * <p>
         * The IAM role that studio users assume when logging in to the Nimble Studio portal.
         * </p>
         * 
         * @param userRoleArn
         *        The IAM role that studio users assume when logging in to the Nimble Studio portal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userRoleArn(String userRoleArn);
    }

    static final class BuilderImpl implements Builder {
        private String adminRoleArn;

        private String arn;

        private Instant createdAt;

        private String displayName;

        private String homeRegion;

        private String ssoClientId;

        private String state;

        private String statusCodeValue;

        private String statusMessage;

        private StudioEncryptionConfiguration studioEncryptionConfiguration;

        private String studioId;

        private String studioName;

        private String studioUrl;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private Instant updatedAt;

        private String userRoleArn;

        private BuilderImpl() {
        }

        private BuilderImpl(Studio model) {
            adminRoleArn(model.adminRoleArn);
            arn(model.arn);
            createdAt(model.createdAt);
            displayName(model.displayName);
            homeRegion(model.homeRegion);
            ssoClientId(model.ssoClientId);
            state(model.state);
            statusCode(model.statusCodeValue);
            statusMessage(model.statusMessage);
            studioEncryptionConfiguration(model.studioEncryptionConfiguration);
            studioId(model.studioId);
            studioName(model.studioName);
            studioUrl(model.studioUrl);
            tags(model.tags);
            updatedAt(model.updatedAt);
            userRoleArn(model.userRoleArn);
        }

        public final String getAdminRoleArn() {
            return adminRoleArn;
        }

        public final void setAdminRoleArn(String adminRoleArn) {
            this.adminRoleArn = adminRoleArn;
        }

        @Override
        public final Builder adminRoleArn(String adminRoleArn) {
            this.adminRoleArn = adminRoleArn;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getDisplayName() {
            return displayName;
        }

        public final void setDisplayName(String displayName) {
            this.displayName = displayName;
        }

        @Override
        public final Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public final String getHomeRegion() {
            return homeRegion;
        }

        public final void setHomeRegion(String homeRegion) {
            this.homeRegion = homeRegion;
        }

        @Override
        public final Builder homeRegion(String homeRegion) {
            this.homeRegion = homeRegion;
            return this;
        }

        public final String getSsoClientId() {
            return ssoClientId;
        }

        public final void setSsoClientId(String ssoClientId) {
            this.ssoClientId = ssoClientId;
        }

        @Override
        public final Builder ssoClientId(String ssoClientId) {
            this.ssoClientId = ssoClientId;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(StudioState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getStatusCode() {
            return statusCodeValue;
        }

        public final void setStatusCode(String statusCodeValue) {
            this.statusCodeValue = statusCodeValue;
        }

        @Override
        public final Builder statusCode(String statusCodeValue) {
            this.statusCodeValue = statusCodeValue;
            return this;
        }

        @Override
        public final Builder statusCode(StudioStatusCode statusCodeValue) {
            this.statusCode(statusCodeValue == null ? null : statusCodeValue.toString());
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final StudioEncryptionConfiguration.Builder getStudioEncryptionConfiguration() {
            return studioEncryptionConfiguration != null ? studioEncryptionConfiguration.toBuilder() : null;
        }

        public final void setStudioEncryptionConfiguration(StudioEncryptionConfiguration.BuilderImpl studioEncryptionConfiguration) {
            this.studioEncryptionConfiguration = studioEncryptionConfiguration != null ? studioEncryptionConfiguration.build()
                    : null;
        }

        @Override
        public final Builder studioEncryptionConfiguration(StudioEncryptionConfiguration studioEncryptionConfiguration) {
            this.studioEncryptionConfiguration = studioEncryptionConfiguration;
            return this;
        }

        public final String getStudioId() {
            return studioId;
        }

        public final void setStudioId(String studioId) {
            this.studioId = studioId;
        }

        @Override
        public final Builder studioId(String studioId) {
            this.studioId = studioId;
            return this;
        }

        public final String getStudioName() {
            return studioName;
        }

        public final void setStudioName(String studioName) {
            this.studioName = studioName;
        }

        @Override
        public final Builder studioName(String studioName) {
            this.studioName = studioName;
            return this;
        }

        public final String getStudioUrl() {
            return studioUrl;
        }

        public final void setStudioUrl(String studioUrl) {
            this.studioUrl = studioUrl;
        }

        @Override
        public final Builder studioUrl(String studioUrl) {
            this.studioUrl = studioUrl;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final String getUserRoleArn() {
            return userRoleArn;
        }

        public final void setUserRoleArn(String userRoleArn) {
            this.userRoleArn = userRoleArn;
        }

        @Override
        public final Builder userRoleArn(String userRoleArn) {
            this.userRoleArn = userRoleArn;
            return this;
        }

        @Override
        public Studio build() {
            return new Studio(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
