/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.nimble.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A launch profile controls your artist workforce’s access to studio components, like compute farms, shared file
 * systems, managed file systems, and license server configurations, as well as instance types and Amazon Machine Images
 * (AMIs).
 * </p>
 * 
 * <pre>
 * <code> &lt;p&gt;Studio administrators create launch profiles in the Nimble Studio console. Artists can use their launch profiles to launch an instance from the Nimble Studio portal. Each user’s launch profile defines how they can launch a streaming session. By default, studio admins can use all launch profiles.&lt;/p&gt; </code>
 * </pre>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LaunchProfile implements SdkPojo, Serializable, ToCopyableBuilder<LaunchProfile.Builder, LaunchProfile> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(LaunchProfile::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(LaunchProfile::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> CREATED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("createdBy").getter(getter(LaunchProfile::createdBy)).setter(setter(Builder::createdBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdBy").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(LaunchProfile::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<String>> EC2_SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ec2SubnetIds")
            .getter(getter(LaunchProfile::ec2SubnetIds))
            .setter(setter(Builder::ec2SubnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ec2SubnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> LAUNCH_PROFILE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("launchProfileId").getter(getter(LaunchProfile::launchProfileId))
            .setter(setter(Builder::launchProfileId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("launchProfileId").build()).build();

    private static final SdkField<List<String>> LAUNCH_PROFILE_PROTOCOL_VERSIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("launchProfileProtocolVersions")
            .getter(getter(LaunchProfile::launchProfileProtocolVersions))
            .setter(setter(Builder::launchProfileProtocolVersions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("launchProfileProtocolVersions")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(LaunchProfile::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(LaunchProfile::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> STATUS_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusCode").getter(getter(LaunchProfile::statusCodeAsString)).setter(setter(Builder::statusCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusCode").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusMessage").getter(getter(LaunchProfile::statusMessage)).setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusMessage").build()).build();

    private static final SdkField<StreamConfiguration> STREAM_CONFIGURATION_FIELD = SdkField
            .<StreamConfiguration> builder(MarshallingType.SDK_POJO).memberName("streamConfiguration")
            .getter(getter(LaunchProfile::streamConfiguration)).setter(setter(Builder::streamConfiguration))
            .constructor(StreamConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("streamConfiguration").build())
            .build();

    private static final SdkField<List<String>> STUDIO_COMPONENT_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("studioComponentIds")
            .getter(getter(LaunchProfile::studioComponentIds))
            .setter(setter(Builder::studioComponentIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("studioComponentIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(LaunchProfile::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt")
            .getter(getter(LaunchProfile::updatedAt))
            .setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> UPDATED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("updatedBy").getter(getter(LaunchProfile::updatedBy)).setter(setter(Builder::updatedBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedBy").build()).build();

    private static final SdkField<List<ValidationResult>> VALIDATION_RESULTS_FIELD = SdkField
            .<List<ValidationResult>> builder(MarshallingType.LIST)
            .memberName("validationResults")
            .getter(getter(LaunchProfile::validationResults))
            .setter(setter(Builder::validationResults))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("validationResults").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ValidationResult> builder(MarshallingType.SDK_POJO)
                                            .constructor(ValidationResult::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, CREATED_AT_FIELD,
            CREATED_BY_FIELD, DESCRIPTION_FIELD, EC2_SUBNET_IDS_FIELD, LAUNCH_PROFILE_ID_FIELD,
            LAUNCH_PROFILE_PROTOCOL_VERSIONS_FIELD, NAME_FIELD, STATE_FIELD, STATUS_CODE_FIELD, STATUS_MESSAGE_FIELD,
            STREAM_CONFIGURATION_FIELD, STUDIO_COMPONENT_IDS_FIELD, TAGS_FIELD, UPDATED_AT_FIELD, UPDATED_BY_FIELD,
            VALIDATION_RESULTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final Instant createdAt;

    private final String createdBy;

    private final String description;

    private final List<String> ec2SubnetIds;

    private final String launchProfileId;

    private final List<String> launchProfileProtocolVersions;

    private final String name;

    private final String state;

    private final String statusCodeValue;

    private final String statusMessage;

    private final StreamConfiguration streamConfiguration;

    private final List<String> studioComponentIds;

    private final Map<String, String> tags;

    private final Instant updatedAt;

    private final String updatedBy;

    private final List<ValidationResult> validationResults;

    private LaunchProfile(BuilderImpl builder) {
        this.arn = builder.arn;
        this.createdAt = builder.createdAt;
        this.createdBy = builder.createdBy;
        this.description = builder.description;
        this.ec2SubnetIds = builder.ec2SubnetIds;
        this.launchProfileId = builder.launchProfileId;
        this.launchProfileProtocolVersions = builder.launchProfileProtocolVersions;
        this.name = builder.name;
        this.state = builder.state;
        this.statusCodeValue = builder.statusCodeValue;
        this.statusMessage = builder.statusMessage;
        this.streamConfiguration = builder.streamConfiguration;
        this.studioComponentIds = builder.studioComponentIds;
        this.tags = builder.tags;
        this.updatedAt = builder.updatedAt;
        this.updatedBy = builder.updatedBy;
        this.validationResults = builder.validationResults;
    }

    /**
     * <p>
     * The ARN of the resource.
     * </p>
     * 
     * @return The ARN of the resource.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The Unix epoch timestamp in seconds for when the resource was created.
     * </p>
     * 
     * @return The Unix epoch timestamp in seconds for when the resource was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The user ID of the user that created the launch profile.
     * </p>
     * 
     * @return The user ID of the user that created the launch profile.
     */
    public final String createdBy() {
        return createdBy;
    }

    /**
     * <p>
     * A human-readable description of the launch profile.
     * </p>
     * 
     * @return A human-readable description of the launch profile.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the Ec2SubnetIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEc2SubnetIds() {
        return ec2SubnetIds != null && !(ec2SubnetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Unique identifiers for a collection of EC2 subnets.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEc2SubnetIds} method.
     * </p>
     * 
     * @return Unique identifiers for a collection of EC2 subnets.
     */
    public final List<String> ec2SubnetIds() {
        return ec2SubnetIds;
    }

    /**
     * <p>
     * The launch profile ID.
     * </p>
     * 
     * @return The launch profile ID.
     */
    public final String launchProfileId() {
        return launchProfileId;
    }

    /**
     * For responses, this returns true if the service returned a value for the LaunchProfileProtocolVersions property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasLaunchProfileProtocolVersions() {
        return launchProfileProtocolVersions != null && !(launchProfileProtocolVersions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The version number of the protocol that is used by the launch profile. The only valid version is "2021-03-31".
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLaunchProfileProtocolVersions} method.
     * </p>
     * 
     * @return The version number of the protocol that is used by the launch profile. The only valid version is
     *         "2021-03-31".
     */
    public final List<String> launchProfileProtocolVersions() {
        return launchProfileProtocolVersions;
    }

    /**
     * <p>
     * A friendly name for the launch profile.
     * </p>
     * 
     * @return A friendly name for the launch profile.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The current state.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link LaunchProfileState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state.
     * @see LaunchProfileState
     */
    public final LaunchProfileState state() {
        return LaunchProfileState.fromValue(state);
    }

    /**
     * <p>
     * The current state.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link LaunchProfileState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state.
     * @see LaunchProfileState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The status code.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link LaunchProfileStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusCodeAsString}.
     * </p>
     * 
     * @return The status code.
     * @see LaunchProfileStatusCode
     */
    public final LaunchProfileStatusCode statusCode() {
        return LaunchProfileStatusCode.fromValue(statusCodeValue);
    }

    /**
     * <p>
     * The status code.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link LaunchProfileStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusCodeAsString}.
     * </p>
     * 
     * @return The status code.
     * @see LaunchProfileStatusCode
     */
    public final String statusCodeAsString() {
        return statusCodeValue;
    }

    /**
     * <p>
     * The status message for the launch profile.
     * </p>
     * 
     * @return The status message for the launch profile.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    /**
     * <p>
     * A configuration for a streaming session.
     * </p>
     * 
     * @return A configuration for a streaming session.
     */
    public final StreamConfiguration streamConfiguration() {
        return streamConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the StudioComponentIds property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStudioComponentIds() {
        return studioComponentIds != null && !(studioComponentIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Unique identifiers for a collection of studio components that can be used with this launch profile.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStudioComponentIds} method.
     * </p>
     * 
     * @return Unique identifiers for a collection of studio components that can be used with this launch profile.
     */
    public final List<String> studioComponentIds() {
        return studioComponentIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A collection of labels, in the form of key:value pairs, that apply to this resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A collection of labels, in the form of key:value pairs, that apply to this resource.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The Unix epoch timestamp in seconds for when the resource was updated.
     * </p>
     * 
     * @return The Unix epoch timestamp in seconds for when the resource was updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    /**
     * <p>
     * The user ID of the user that most recently updated the resource.
     * </p>
     * 
     * @return The user ID of the user that most recently updated the resource.
     */
    public final String updatedBy() {
        return updatedBy;
    }

    /**
     * For responses, this returns true if the service returned a value for the ValidationResults property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasValidationResults() {
        return validationResults != null && !(validationResults instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of the latest validation results.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValidationResults} method.
     * </p>
     * 
     * @return The list of the latest validation results.
     */
    public final List<ValidationResult> validationResults() {
        return validationResults;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(createdBy());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasEc2SubnetIds() ? ec2SubnetIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(launchProfileId());
        hashCode = 31 * hashCode + Objects.hashCode(hasLaunchProfileProtocolVersions() ? launchProfileProtocolVersions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(streamConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasStudioComponentIds() ? studioComponentIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(updatedBy());
        hashCode = 31 * hashCode + Objects.hashCode(hasValidationResults() ? validationResults() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LaunchProfile)) {
            return false;
        }
        LaunchProfile other = (LaunchProfile) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(createdBy(), other.createdBy()) && Objects.equals(description(), other.description())
                && hasEc2SubnetIds() == other.hasEc2SubnetIds() && Objects.equals(ec2SubnetIds(), other.ec2SubnetIds())
                && Objects.equals(launchProfileId(), other.launchProfileId())
                && hasLaunchProfileProtocolVersions() == other.hasLaunchProfileProtocolVersions()
                && Objects.equals(launchProfileProtocolVersions(), other.launchProfileProtocolVersions())
                && Objects.equals(name(), other.name()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(statusCodeAsString(), other.statusCodeAsString())
                && Objects.equals(statusMessage(), other.statusMessage())
                && Objects.equals(streamConfiguration(), other.streamConfiguration())
                && hasStudioComponentIds() == other.hasStudioComponentIds()
                && Objects.equals(studioComponentIds(), other.studioComponentIds()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(updatedAt(), other.updatedAt())
                && Objects.equals(updatedBy(), other.updatedBy()) && hasValidationResults() == other.hasValidationResults()
                && Objects.equals(validationResults(), other.validationResults());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("LaunchProfile")
                .add("Arn", arn())
                .add("CreatedAt", createdAt())
                .add("CreatedBy", createdBy())
                .add("Description", description() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Ec2SubnetIds", hasEc2SubnetIds() ? ec2SubnetIds() : null)
                .add("LaunchProfileId", launchProfileId())
                .add("LaunchProfileProtocolVersions", hasLaunchProfileProtocolVersions() ? launchProfileProtocolVersions() : null)
                .add("Name", name() == null ? null : "*** Sensitive Data Redacted ***").add("State", stateAsString())
                .add("StatusCode", statusCodeAsString()).add("StatusMessage", statusMessage())
                .add("StreamConfiguration", streamConfiguration())
                .add("StudioComponentIds", hasStudioComponentIds() ? studioComponentIds() : null)
                .add("Tags", hasTags() ? tags() : null).add("UpdatedAt", updatedAt()).add("UpdatedBy", updatedBy())
                .add("ValidationResults", hasValidationResults() ? validationResults() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "createdBy":
            return Optional.ofNullable(clazz.cast(createdBy()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "ec2SubnetIds":
            return Optional.ofNullable(clazz.cast(ec2SubnetIds()));
        case "launchProfileId":
            return Optional.ofNullable(clazz.cast(launchProfileId()));
        case "launchProfileProtocolVersions":
            return Optional.ofNullable(clazz.cast(launchProfileProtocolVersions()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "statusCode":
            return Optional.ofNullable(clazz.cast(statusCodeAsString()));
        case "statusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        case "streamConfiguration":
            return Optional.ofNullable(clazz.cast(streamConfiguration()));
        case "studioComponentIds":
            return Optional.ofNullable(clazz.cast(studioComponentIds()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "updatedBy":
            return Optional.ofNullable(clazz.cast(updatedBy()));
        case "validationResults":
            return Optional.ofNullable(clazz.cast(validationResults()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LaunchProfile, T> g) {
        return obj -> g.apply((LaunchProfile) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LaunchProfile> {
        /**
         * <p>
         * The ARN of the resource.
         * </p>
         * 
         * @param arn
         *        The ARN of the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The Unix epoch timestamp in seconds for when the resource was created.
         * </p>
         * 
         * @param createdAt
         *        The Unix epoch timestamp in seconds for when the resource was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The user ID of the user that created the launch profile.
         * </p>
         * 
         * @param createdBy
         *        The user ID of the user that created the launch profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdBy(String createdBy);

        /**
         * <p>
         * A human-readable description of the launch profile.
         * </p>
         * 
         * @param description
         *        A human-readable description of the launch profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Unique identifiers for a collection of EC2 subnets.
         * </p>
         * 
         * @param ec2SubnetIds
         *        Unique identifiers for a collection of EC2 subnets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2SubnetIds(Collection<String> ec2SubnetIds);

        /**
         * <p>
         * Unique identifiers for a collection of EC2 subnets.
         * </p>
         * 
         * @param ec2SubnetIds
         *        Unique identifiers for a collection of EC2 subnets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2SubnetIds(String... ec2SubnetIds);

        /**
         * <p>
         * The launch profile ID.
         * </p>
         * 
         * @param launchProfileId
         *        The launch profile ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchProfileId(String launchProfileId);

        /**
         * <p>
         * The version number of the protocol that is used by the launch profile. The only valid version is
         * "2021-03-31".
         * </p>
         * 
         * @param launchProfileProtocolVersions
         *        The version number of the protocol that is used by the launch profile. The only valid version is
         *        "2021-03-31".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchProfileProtocolVersions(Collection<String> launchProfileProtocolVersions);

        /**
         * <p>
         * The version number of the protocol that is used by the launch profile. The only valid version is
         * "2021-03-31".
         * </p>
         * 
         * @param launchProfileProtocolVersions
         *        The version number of the protocol that is used by the launch profile. The only valid version is
         *        "2021-03-31".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchProfileProtocolVersions(String... launchProfileProtocolVersions);

        /**
         * <p>
         * A friendly name for the launch profile.
         * </p>
         * 
         * @param name
         *        A friendly name for the launch profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The current state.
         * </p>
         * 
         * @param state
         *        The current state.
         * @see LaunchProfileState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchProfileState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state.
         * </p>
         * 
         * @param state
         *        The current state.
         * @see LaunchProfileState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchProfileState
         */
        Builder state(LaunchProfileState state);

        /**
         * <p>
         * The status code.
         * </p>
         * 
         * @param statusCodeValue
         *        The status code.
         * @see LaunchProfileStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchProfileStatusCode
         */
        Builder statusCode(String statusCodeValue);

        /**
         * <p>
         * The status code.
         * </p>
         * 
         * @param statusCodeValue
         *        The status code.
         * @see LaunchProfileStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchProfileStatusCode
         */
        Builder statusCode(LaunchProfileStatusCode statusCodeValue);

        /**
         * <p>
         * The status message for the launch profile.
         * </p>
         * 
         * @param statusMessage
         *        The status message for the launch profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * A configuration for a streaming session.
         * </p>
         * 
         * @param streamConfiguration
         *        A configuration for a streaming session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamConfiguration(StreamConfiguration streamConfiguration);

        /**
         * <p>
         * A configuration for a streaming session.
         * </p>
         * This is a convenience method that creates an instance of the {@link StreamConfiguration.Builder} avoiding the
         * need to create one manually via {@link StreamConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link StreamConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #streamConfiguration(StreamConfiguration)}.
         * 
         * @param streamConfiguration
         *        a consumer that will call methods on {@link StreamConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #streamConfiguration(StreamConfiguration)
         */
        default Builder streamConfiguration(Consumer<StreamConfiguration.Builder> streamConfiguration) {
            return streamConfiguration(StreamConfiguration.builder().applyMutation(streamConfiguration).build());
        }

        /**
         * <p>
         * Unique identifiers for a collection of studio components that can be used with this launch profile.
         * </p>
         * 
         * @param studioComponentIds
         *        Unique identifiers for a collection of studio components that can be used with this launch profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder studioComponentIds(Collection<String> studioComponentIds);

        /**
         * <p>
         * Unique identifiers for a collection of studio components that can be used with this launch profile.
         * </p>
         * 
         * @param studioComponentIds
         *        Unique identifiers for a collection of studio components that can be used with this launch profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder studioComponentIds(String... studioComponentIds);

        /**
         * <p>
         * A collection of labels, in the form of key:value pairs, that apply to this resource.
         * </p>
         * 
         * @param tags
         *        A collection of labels, in the form of key:value pairs, that apply to this resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The Unix epoch timestamp in seconds for when the resource was updated.
         * </p>
         * 
         * @param updatedAt
         *        The Unix epoch timestamp in seconds for when the resource was updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);

        /**
         * <p>
         * The user ID of the user that most recently updated the resource.
         * </p>
         * 
         * @param updatedBy
         *        The user ID of the user that most recently updated the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedBy(String updatedBy);

        /**
         * <p>
         * The list of the latest validation results.
         * </p>
         * 
         * @param validationResults
         *        The list of the latest validation results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationResults(Collection<ValidationResult> validationResults);

        /**
         * <p>
         * The list of the latest validation results.
         * </p>
         * 
         * @param validationResults
         *        The list of the latest validation results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationResults(ValidationResult... validationResults);

        /**
         * <p>
         * The list of the latest validation results.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.nimble.model.ValidationResult.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.nimble.model.ValidationResult#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.nimble.model.ValidationResult.Builder#build()} is called immediately
         * and its result is passed to {@link #validationResults(List<ValidationResult>)}.
         * 
         * @param validationResults
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.nimble.model.ValidationResult.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #validationResults(java.util.Collection<ValidationResult>)
         */
        Builder validationResults(Consumer<ValidationResult.Builder>... validationResults);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private Instant createdAt;

        private String createdBy;

        private String description;

        private List<String> ec2SubnetIds = DefaultSdkAutoConstructList.getInstance();

        private String launchProfileId;

        private List<String> launchProfileProtocolVersions = DefaultSdkAutoConstructList.getInstance();

        private String name;

        private String state;

        private String statusCodeValue;

        private String statusMessage;

        private StreamConfiguration streamConfiguration;

        private List<String> studioComponentIds = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private Instant updatedAt;

        private String updatedBy;

        private List<ValidationResult> validationResults = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LaunchProfile model) {
            arn(model.arn);
            createdAt(model.createdAt);
            createdBy(model.createdBy);
            description(model.description);
            ec2SubnetIds(model.ec2SubnetIds);
            launchProfileId(model.launchProfileId);
            launchProfileProtocolVersions(model.launchProfileProtocolVersions);
            name(model.name);
            state(model.state);
            statusCode(model.statusCodeValue);
            statusMessage(model.statusMessage);
            streamConfiguration(model.streamConfiguration);
            studioComponentIds(model.studioComponentIds);
            tags(model.tags);
            updatedAt(model.updatedAt);
            updatedBy(model.updatedBy);
            validationResults(model.validationResults);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getCreatedBy() {
            return createdBy;
        }

        public final void setCreatedBy(String createdBy) {
            this.createdBy = createdBy;
        }

        @Override
        public final Builder createdBy(String createdBy) {
            this.createdBy = createdBy;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Collection<String> getEc2SubnetIds() {
            if (ec2SubnetIds instanceof SdkAutoConstructList) {
                return null;
            }
            return ec2SubnetIds;
        }

        public final void setEc2SubnetIds(Collection<String> ec2SubnetIds) {
            this.ec2SubnetIds = EC2SubnetIdListCopier.copy(ec2SubnetIds);
        }

        @Override
        public final Builder ec2SubnetIds(Collection<String> ec2SubnetIds) {
            this.ec2SubnetIds = EC2SubnetIdListCopier.copy(ec2SubnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ec2SubnetIds(String... ec2SubnetIds) {
            ec2SubnetIds(Arrays.asList(ec2SubnetIds));
            return this;
        }

        public final String getLaunchProfileId() {
            return launchProfileId;
        }

        public final void setLaunchProfileId(String launchProfileId) {
            this.launchProfileId = launchProfileId;
        }

        @Override
        public final Builder launchProfileId(String launchProfileId) {
            this.launchProfileId = launchProfileId;
            return this;
        }

        public final Collection<String> getLaunchProfileProtocolVersions() {
            if (launchProfileProtocolVersions instanceof SdkAutoConstructList) {
                return null;
            }
            return launchProfileProtocolVersions;
        }

        public final void setLaunchProfileProtocolVersions(Collection<String> launchProfileProtocolVersions) {
            this.launchProfileProtocolVersions = LaunchProfileProtocolVersionListCopier.copy(launchProfileProtocolVersions);
        }

        @Override
        public final Builder launchProfileProtocolVersions(Collection<String> launchProfileProtocolVersions) {
            this.launchProfileProtocolVersions = LaunchProfileProtocolVersionListCopier.copy(launchProfileProtocolVersions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder launchProfileProtocolVersions(String... launchProfileProtocolVersions) {
            launchProfileProtocolVersions(Arrays.asList(launchProfileProtocolVersions));
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(LaunchProfileState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getStatusCode() {
            return statusCodeValue;
        }

        public final void setStatusCode(String statusCodeValue) {
            this.statusCodeValue = statusCodeValue;
        }

        @Override
        public final Builder statusCode(String statusCodeValue) {
            this.statusCodeValue = statusCodeValue;
            return this;
        }

        @Override
        public final Builder statusCode(LaunchProfileStatusCode statusCodeValue) {
            this.statusCode(statusCodeValue == null ? null : statusCodeValue.toString());
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final StreamConfiguration.Builder getStreamConfiguration() {
            return streamConfiguration != null ? streamConfiguration.toBuilder() : null;
        }

        public final void setStreamConfiguration(StreamConfiguration.BuilderImpl streamConfiguration) {
            this.streamConfiguration = streamConfiguration != null ? streamConfiguration.build() : null;
        }

        @Override
        public final Builder streamConfiguration(StreamConfiguration streamConfiguration) {
            this.streamConfiguration = streamConfiguration;
            return this;
        }

        public final Collection<String> getStudioComponentIds() {
            if (studioComponentIds instanceof SdkAutoConstructList) {
                return null;
            }
            return studioComponentIds;
        }

        public final void setStudioComponentIds(Collection<String> studioComponentIds) {
            this.studioComponentIds = LaunchProfileStudioComponentIdListCopier.copy(studioComponentIds);
        }

        @Override
        public final Builder studioComponentIds(Collection<String> studioComponentIds) {
            this.studioComponentIds = LaunchProfileStudioComponentIdListCopier.copy(studioComponentIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder studioComponentIds(String... studioComponentIds) {
            studioComponentIds(Arrays.asList(studioComponentIds));
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final String getUpdatedBy() {
            return updatedBy;
        }

        public final void setUpdatedBy(String updatedBy) {
            this.updatedBy = updatedBy;
        }

        @Override
        public final Builder updatedBy(String updatedBy) {
            this.updatedBy = updatedBy;
            return this;
        }

        public final List<ValidationResult.Builder> getValidationResults() {
            List<ValidationResult.Builder> result = ValidationResultsCopier.copyToBuilder(this.validationResults);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setValidationResults(Collection<ValidationResult.BuilderImpl> validationResults) {
            this.validationResults = ValidationResultsCopier.copyFromBuilder(validationResults);
        }

        @Override
        public final Builder validationResults(Collection<ValidationResult> validationResults) {
            this.validationResults = ValidationResultsCopier.copy(validationResults);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validationResults(ValidationResult... validationResults) {
            validationResults(Arrays.asList(validationResults));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validationResults(Consumer<ValidationResult.Builder>... validationResults) {
            validationResults(Stream.of(validationResults).map(c -> ValidationResult.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public LaunchProfile build() {
            return new LaunchProfile(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
