/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.nimble.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.nimble.NimbleClient;
import software.amazon.awssdk.services.nimble.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.nimble.model.GetLaunchProfileRequest;
import software.amazon.awssdk.services.nimble.model.GetLaunchProfileResponse;
import software.amazon.awssdk.services.nimble.model.GetStreamingImageRequest;
import software.amazon.awssdk.services.nimble.model.GetStreamingImageResponse;
import software.amazon.awssdk.services.nimble.model.GetStreamingSessionRequest;
import software.amazon.awssdk.services.nimble.model.GetStreamingSessionResponse;
import software.amazon.awssdk.services.nimble.model.GetStreamingSessionStreamRequest;
import software.amazon.awssdk.services.nimble.model.GetStreamingSessionStreamResponse;
import software.amazon.awssdk.services.nimble.model.GetStudioComponentRequest;
import software.amazon.awssdk.services.nimble.model.GetStudioComponentResponse;
import software.amazon.awssdk.services.nimble.model.GetStudioRequest;
import software.amazon.awssdk.services.nimble.model.GetStudioResponse;
import software.amazon.awssdk.services.nimble.model.NimbleRequest;
import software.amazon.awssdk.services.nimble.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultNimbleWaiter implements NimbleWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final NimbleClient client;

    private final AttributeMap managedResources;

    private final Waiter<GetLaunchProfileResponse> launchProfileDeletedWaiter;

    private final Waiter<GetLaunchProfileResponse> launchProfileReadyWaiter;

    private final Waiter<GetStreamingImageResponse> streamingImageDeletedWaiter;

    private final Waiter<GetStreamingImageResponse> streamingImageReadyWaiter;

    private final Waiter<GetStreamingSessionResponse> streamingSessionDeletedWaiter;

    private final Waiter<GetStreamingSessionResponse> streamingSessionReadyWaiter;

    private final Waiter<GetStreamingSessionResponse> streamingSessionStoppedWaiter;

    private final Waiter<GetStreamingSessionStreamResponse> streamingSessionStreamReadyWaiter;

    private final Waiter<GetStudioComponentResponse> studioComponentDeletedWaiter;

    private final Waiter<GetStudioComponentResponse> studioComponentReadyWaiter;

    private final Waiter<GetStudioResponse> studioDeletedWaiter;

    private final Waiter<GetStudioResponse> studioReadyWaiter;

    private DefaultNimbleWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = NimbleClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.launchProfileDeletedWaiter = Waiter.builder(GetLaunchProfileResponse.class)
                .acceptors(launchProfileDeletedWaiterAcceptors())
                .overrideConfiguration(launchProfileDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.launchProfileReadyWaiter = Waiter.builder(GetLaunchProfileResponse.class)
                .acceptors(launchProfileReadyWaiterAcceptors())
                .overrideConfiguration(launchProfileReadyWaiterConfig(builder.overrideConfiguration)).build();
        this.streamingImageDeletedWaiter = Waiter.builder(GetStreamingImageResponse.class)
                .acceptors(streamingImageDeletedWaiterAcceptors())
                .overrideConfiguration(streamingImageDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.streamingImageReadyWaiter = Waiter.builder(GetStreamingImageResponse.class)
                .acceptors(streamingImageReadyWaiterAcceptors())
                .overrideConfiguration(streamingImageReadyWaiterConfig(builder.overrideConfiguration)).build();
        this.streamingSessionDeletedWaiter = Waiter.builder(GetStreamingSessionResponse.class)
                .acceptors(streamingSessionDeletedWaiterAcceptors())
                .overrideConfiguration(streamingSessionDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.streamingSessionReadyWaiter = Waiter.builder(GetStreamingSessionResponse.class)
                .acceptors(streamingSessionReadyWaiterAcceptors())
                .overrideConfiguration(streamingSessionReadyWaiterConfig(builder.overrideConfiguration)).build();
        this.streamingSessionStoppedWaiter = Waiter.builder(GetStreamingSessionResponse.class)
                .acceptors(streamingSessionStoppedWaiterAcceptors())
                .overrideConfiguration(streamingSessionStoppedWaiterConfig(builder.overrideConfiguration)).build();
        this.streamingSessionStreamReadyWaiter = Waiter.builder(GetStreamingSessionStreamResponse.class)
                .acceptors(streamingSessionStreamReadyWaiterAcceptors())
                .overrideConfiguration(streamingSessionStreamReadyWaiterConfig(builder.overrideConfiguration)).build();
        this.studioComponentDeletedWaiter = Waiter.builder(GetStudioComponentResponse.class)
                .acceptors(studioComponentDeletedWaiterAcceptors())
                .overrideConfiguration(studioComponentDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.studioComponentReadyWaiter = Waiter.builder(GetStudioComponentResponse.class)
                .acceptors(studioComponentReadyWaiterAcceptors())
                .overrideConfiguration(studioComponentReadyWaiterConfig(builder.overrideConfiguration)).build();
        this.studioDeletedWaiter = Waiter.builder(GetStudioResponse.class).acceptors(studioDeletedWaiterAcceptors())
                .overrideConfiguration(studioDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.studioReadyWaiter = Waiter.builder(GetStudioResponse.class).acceptors(studioReadyWaiterAcceptors())
                .overrideConfiguration(studioReadyWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<GetLaunchProfileResponse> waitUntilLaunchProfileDeleted(GetLaunchProfileRequest getLaunchProfileRequest) {
        return launchProfileDeletedWaiter.run(() -> client.getLaunchProfile(applyWaitersUserAgent(getLaunchProfileRequest)));
    }

    @Override
    public WaiterResponse<GetLaunchProfileResponse> waitUntilLaunchProfileDeleted(
            GetLaunchProfileRequest getLaunchProfileRequest, WaiterOverrideConfiguration overrideConfig) {
        return launchProfileDeletedWaiter.run(() -> client.getLaunchProfile(applyWaitersUserAgent(getLaunchProfileRequest)),
                launchProfileDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetLaunchProfileResponse> waitUntilLaunchProfileReady(GetLaunchProfileRequest getLaunchProfileRequest) {
        return launchProfileReadyWaiter.run(() -> client.getLaunchProfile(applyWaitersUserAgent(getLaunchProfileRequest)));
    }

    @Override
    public WaiterResponse<GetLaunchProfileResponse> waitUntilLaunchProfileReady(GetLaunchProfileRequest getLaunchProfileRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return launchProfileReadyWaiter.run(() -> client.getLaunchProfile(applyWaitersUserAgent(getLaunchProfileRequest)),
                launchProfileReadyWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetStreamingImageResponse> waitUntilStreamingImageDeleted(
            GetStreamingImageRequest getStreamingImageRequest) {
        return streamingImageDeletedWaiter.run(() -> client.getStreamingImage(applyWaitersUserAgent(getStreamingImageRequest)));
    }

    @Override
    public WaiterResponse<GetStreamingImageResponse> waitUntilStreamingImageDeleted(
            GetStreamingImageRequest getStreamingImageRequest, WaiterOverrideConfiguration overrideConfig) {
        return streamingImageDeletedWaiter.run(() -> client.getStreamingImage(applyWaitersUserAgent(getStreamingImageRequest)),
                streamingImageDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetStreamingImageResponse> waitUntilStreamingImageReady(
            GetStreamingImageRequest getStreamingImageRequest) {
        return streamingImageReadyWaiter.run(() -> client.getStreamingImage(applyWaitersUserAgent(getStreamingImageRequest)));
    }

    @Override
    public WaiterResponse<GetStreamingImageResponse> waitUntilStreamingImageReady(
            GetStreamingImageRequest getStreamingImageRequest, WaiterOverrideConfiguration overrideConfig) {
        return streamingImageReadyWaiter.run(() -> client.getStreamingImage(applyWaitersUserAgent(getStreamingImageRequest)),
                streamingImageReadyWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetStreamingSessionResponse> waitUntilStreamingSessionDeleted(
            GetStreamingSessionRequest getStreamingSessionRequest) {
        return streamingSessionDeletedWaiter.run(() -> client
                .getStreamingSession(applyWaitersUserAgent(getStreamingSessionRequest)));
    }

    @Override
    public WaiterResponse<GetStreamingSessionResponse> waitUntilStreamingSessionDeleted(
            GetStreamingSessionRequest getStreamingSessionRequest, WaiterOverrideConfiguration overrideConfig) {
        return streamingSessionDeletedWaiter.run(
                () -> client.getStreamingSession(applyWaitersUserAgent(getStreamingSessionRequest)),
                streamingSessionDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetStreamingSessionResponse> waitUntilStreamingSessionReady(
            GetStreamingSessionRequest getStreamingSessionRequest) {
        return streamingSessionReadyWaiter.run(() -> client
                .getStreamingSession(applyWaitersUserAgent(getStreamingSessionRequest)));
    }

    @Override
    public WaiterResponse<GetStreamingSessionResponse> waitUntilStreamingSessionReady(
            GetStreamingSessionRequest getStreamingSessionRequest, WaiterOverrideConfiguration overrideConfig) {
        return streamingSessionReadyWaiter.run(
                () -> client.getStreamingSession(applyWaitersUserAgent(getStreamingSessionRequest)),
                streamingSessionReadyWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetStreamingSessionResponse> waitUntilStreamingSessionStopped(
            GetStreamingSessionRequest getStreamingSessionRequest) {
        return streamingSessionStoppedWaiter.run(() -> client
                .getStreamingSession(applyWaitersUserAgent(getStreamingSessionRequest)));
    }

    @Override
    public WaiterResponse<GetStreamingSessionResponse> waitUntilStreamingSessionStopped(
            GetStreamingSessionRequest getStreamingSessionRequest, WaiterOverrideConfiguration overrideConfig) {
        return streamingSessionStoppedWaiter.run(
                () -> client.getStreamingSession(applyWaitersUserAgent(getStreamingSessionRequest)),
                streamingSessionStoppedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetStreamingSessionStreamResponse> waitUntilStreamingSessionStreamReady(
            GetStreamingSessionStreamRequest getStreamingSessionStreamRequest) {
        return streamingSessionStreamReadyWaiter.run(() -> client
                .getStreamingSessionStream(applyWaitersUserAgent(getStreamingSessionStreamRequest)));
    }

    @Override
    public WaiterResponse<GetStreamingSessionStreamResponse> waitUntilStreamingSessionStreamReady(
            GetStreamingSessionStreamRequest getStreamingSessionStreamRequest, WaiterOverrideConfiguration overrideConfig) {
        return streamingSessionStreamReadyWaiter.run(
                () -> client.getStreamingSessionStream(applyWaitersUserAgent(getStreamingSessionStreamRequest)),
                streamingSessionStreamReadyWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetStudioComponentResponse> waitUntilStudioComponentDeleted(
            GetStudioComponentRequest getStudioComponentRequest) {
        return studioComponentDeletedWaiter
                .run(() -> client.getStudioComponent(applyWaitersUserAgent(getStudioComponentRequest)));
    }

    @Override
    public WaiterResponse<GetStudioComponentResponse> waitUntilStudioComponentDeleted(
            GetStudioComponentRequest getStudioComponentRequest, WaiterOverrideConfiguration overrideConfig) {
        return studioComponentDeletedWaiter.run(
                () -> client.getStudioComponent(applyWaitersUserAgent(getStudioComponentRequest)),
                studioComponentDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetStudioComponentResponse> waitUntilStudioComponentReady(
            GetStudioComponentRequest getStudioComponentRequest) {
        return studioComponentReadyWaiter.run(() -> client.getStudioComponent(applyWaitersUserAgent(getStudioComponentRequest)));
    }

    @Override
    public WaiterResponse<GetStudioComponentResponse> waitUntilStudioComponentReady(
            GetStudioComponentRequest getStudioComponentRequest, WaiterOverrideConfiguration overrideConfig) {
        return studioComponentReadyWaiter.run(() -> client.getStudioComponent(applyWaitersUserAgent(getStudioComponentRequest)),
                studioComponentReadyWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetStudioResponse> waitUntilStudioDeleted(GetStudioRequest getStudioRequest) {
        return studioDeletedWaiter.run(() -> client.getStudio(applyWaitersUserAgent(getStudioRequest)));
    }

    @Override
    public WaiterResponse<GetStudioResponse> waitUntilStudioDeleted(GetStudioRequest getStudioRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return studioDeletedWaiter.run(() -> client.getStudio(applyWaitersUserAgent(getStudioRequest)),
                studioDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetStudioResponse> waitUntilStudioReady(GetStudioRequest getStudioRequest) {
        return studioReadyWaiter.run(() -> client.getStudio(applyWaitersUserAgent(getStudioRequest)));
    }

    @Override
    public WaiterResponse<GetStudioResponse> waitUntilStudioReady(GetStudioRequest getStudioRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return studioReadyWaiter.run(() -> client.getStudio(applyWaitersUserAgent(getStudioRequest)),
                studioReadyWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetLaunchProfileResponse>> launchProfileDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetLaunchProfileResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("launchProfile").field("state").value(), "DELETED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("launchProfile").field("state").value(), "DELETE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (launchProfile.state=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetLaunchProfileResponse>> launchProfileReadyWaiterAcceptors() {
        List<WaiterAcceptor<? super GetLaunchProfileResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("launchProfile").field("state").value(), "READY");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("launchProfile").field("state").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (launchProfile.state=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("launchProfile").field("state").value(), "UPDATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (launchProfile.state=UPDATE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetStreamingImageResponse>> streamingImageDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetStreamingImageResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("streamingImage").field("state").value(), "DELETED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("streamingImage").field("state").value(), "DELETE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (streamingImage.state=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetStreamingImageResponse>> streamingImageReadyWaiterAcceptors() {
        List<WaiterAcceptor<? super GetStreamingImageResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("streamingImage").field("state").value(), "READY");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("streamingImage").field("state").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (streamingImage.state=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("streamingImage").field("state").value(), "UPDATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (streamingImage.state=UPDATE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetStreamingSessionResponse>> streamingSessionDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetStreamingSessionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("session").field("state").value(), "DELETED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("session").field("state").value(), "DELETE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (session.state=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetStreamingSessionResponse>> streamingSessionReadyWaiterAcceptors() {
        List<WaiterAcceptor<? super GetStreamingSessionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("session").field("state").value(), "READY");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("session").field("state").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (session.state=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("session").field("state").value(), "START_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (session.state=START_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetStreamingSessionResponse>> streamingSessionStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetStreamingSessionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("session").field("state").value(), "STOPPED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("session").field("state").value(), "STOP_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (session.state=STOP_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetStreamingSessionStreamResponse>> streamingSessionStreamReadyWaiterAcceptors() {
        List<WaiterAcceptor<? super GetStreamingSessionStreamResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("stream").field("state").value(), "READY");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("stream").field("state").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (stream.state=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetStudioComponentResponse>> studioComponentDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetStudioComponentResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("studioComponent").field("state").value(), "DELETED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("studioComponent").field("state").value(), "DELETE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (studioComponent.state=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetStudioComponentResponse>> studioComponentReadyWaiterAcceptors() {
        List<WaiterAcceptor<? super GetStudioComponentResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("studioComponent").field("state").value(), "READY");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("studioComponent").field("state").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (studioComponent.state=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("studioComponent").field("state").value(), "UPDATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (studioComponent.state=UPDATE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetStudioResponse>> studioDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetStudioResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("studio").field("state").value(), "DELETED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("studio").field("state").value(), "DELETE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (studio.state=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetStudioResponse>> studioReadyWaiterAcceptors() {
        List<WaiterAcceptor<? super GetStudioResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("studio").field("state").value(), "READY");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("studio").field("state").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (studio.state=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("studio").field("state").value(), "UPDATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (studio.state=UPDATE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration launchProfileDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(150);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration launchProfileReadyWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(150);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration streamingImageDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration streamingImageReadyWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration streamingSessionDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(180);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration streamingSessionReadyWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(180);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration streamingSessionStoppedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(180);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration streamingSessionStreamReadyWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(30);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration studioComponentDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(1)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration studioComponentReadyWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration studioDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration studioReadyWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static NimbleWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends NimbleRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements NimbleWaiter.Builder {
        private NimbleClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public NimbleWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public NimbleWaiter.Builder client(NimbleClient client) {
            this.client = client;
            return this;
        }

        public NimbleWaiter build() {
            return new DefaultNimbleWaiter(this);
        }
    }
}
