/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.notifications;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.notifications.internal.NotificationsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.notifications.model.AccessDeniedException;
import software.amazon.awssdk.services.notifications.model.AssociateChannelRequest;
import software.amazon.awssdk.services.notifications.model.AssociateChannelResponse;
import software.amazon.awssdk.services.notifications.model.AssociateManagedNotificationAccountContactRequest;
import software.amazon.awssdk.services.notifications.model.AssociateManagedNotificationAccountContactResponse;
import software.amazon.awssdk.services.notifications.model.AssociateManagedNotificationAdditionalChannelRequest;
import software.amazon.awssdk.services.notifications.model.AssociateManagedNotificationAdditionalChannelResponse;
import software.amazon.awssdk.services.notifications.model.ConflictException;
import software.amazon.awssdk.services.notifications.model.CreateEventRuleRequest;
import software.amazon.awssdk.services.notifications.model.CreateEventRuleResponse;
import software.amazon.awssdk.services.notifications.model.CreateNotificationConfigurationRequest;
import software.amazon.awssdk.services.notifications.model.CreateNotificationConfigurationResponse;
import software.amazon.awssdk.services.notifications.model.DeleteEventRuleRequest;
import software.amazon.awssdk.services.notifications.model.DeleteEventRuleResponse;
import software.amazon.awssdk.services.notifications.model.DeleteNotificationConfigurationRequest;
import software.amazon.awssdk.services.notifications.model.DeleteNotificationConfigurationResponse;
import software.amazon.awssdk.services.notifications.model.DeregisterNotificationHubRequest;
import software.amazon.awssdk.services.notifications.model.DeregisterNotificationHubResponse;
import software.amazon.awssdk.services.notifications.model.DisableNotificationsAccessForOrganizationRequest;
import software.amazon.awssdk.services.notifications.model.DisableNotificationsAccessForOrganizationResponse;
import software.amazon.awssdk.services.notifications.model.DisassociateChannelRequest;
import software.amazon.awssdk.services.notifications.model.DisassociateChannelResponse;
import software.amazon.awssdk.services.notifications.model.DisassociateManagedNotificationAccountContactRequest;
import software.amazon.awssdk.services.notifications.model.DisassociateManagedNotificationAccountContactResponse;
import software.amazon.awssdk.services.notifications.model.DisassociateManagedNotificationAdditionalChannelRequest;
import software.amazon.awssdk.services.notifications.model.DisassociateManagedNotificationAdditionalChannelResponse;
import software.amazon.awssdk.services.notifications.model.EnableNotificationsAccessForOrganizationRequest;
import software.amazon.awssdk.services.notifications.model.EnableNotificationsAccessForOrganizationResponse;
import software.amazon.awssdk.services.notifications.model.GetEventRuleRequest;
import software.amazon.awssdk.services.notifications.model.GetEventRuleResponse;
import software.amazon.awssdk.services.notifications.model.GetManagedNotificationChildEventRequest;
import software.amazon.awssdk.services.notifications.model.GetManagedNotificationChildEventResponse;
import software.amazon.awssdk.services.notifications.model.GetManagedNotificationConfigurationRequest;
import software.amazon.awssdk.services.notifications.model.GetManagedNotificationConfigurationResponse;
import software.amazon.awssdk.services.notifications.model.GetManagedNotificationEventRequest;
import software.amazon.awssdk.services.notifications.model.GetManagedNotificationEventResponse;
import software.amazon.awssdk.services.notifications.model.GetNotificationConfigurationRequest;
import software.amazon.awssdk.services.notifications.model.GetNotificationConfigurationResponse;
import software.amazon.awssdk.services.notifications.model.GetNotificationEventRequest;
import software.amazon.awssdk.services.notifications.model.GetNotificationEventResponse;
import software.amazon.awssdk.services.notifications.model.GetNotificationsAccessForOrganizationRequest;
import software.amazon.awssdk.services.notifications.model.GetNotificationsAccessForOrganizationResponse;
import software.amazon.awssdk.services.notifications.model.InternalServerException;
import software.amazon.awssdk.services.notifications.model.ListChannelsRequest;
import software.amazon.awssdk.services.notifications.model.ListChannelsResponse;
import software.amazon.awssdk.services.notifications.model.ListEventRulesRequest;
import software.amazon.awssdk.services.notifications.model.ListEventRulesResponse;
import software.amazon.awssdk.services.notifications.model.ListManagedNotificationChannelAssociationsRequest;
import software.amazon.awssdk.services.notifications.model.ListManagedNotificationChannelAssociationsResponse;
import software.amazon.awssdk.services.notifications.model.ListManagedNotificationChildEventsRequest;
import software.amazon.awssdk.services.notifications.model.ListManagedNotificationChildEventsResponse;
import software.amazon.awssdk.services.notifications.model.ListManagedNotificationConfigurationsRequest;
import software.amazon.awssdk.services.notifications.model.ListManagedNotificationConfigurationsResponse;
import software.amazon.awssdk.services.notifications.model.ListManagedNotificationEventsRequest;
import software.amazon.awssdk.services.notifications.model.ListManagedNotificationEventsResponse;
import software.amazon.awssdk.services.notifications.model.ListNotificationConfigurationsRequest;
import software.amazon.awssdk.services.notifications.model.ListNotificationConfigurationsResponse;
import software.amazon.awssdk.services.notifications.model.ListNotificationEventsRequest;
import software.amazon.awssdk.services.notifications.model.ListNotificationEventsResponse;
import software.amazon.awssdk.services.notifications.model.ListNotificationHubsRequest;
import software.amazon.awssdk.services.notifications.model.ListNotificationHubsResponse;
import software.amazon.awssdk.services.notifications.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.notifications.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.notifications.model.NotificationsException;
import software.amazon.awssdk.services.notifications.model.RegisterNotificationHubRequest;
import software.amazon.awssdk.services.notifications.model.RegisterNotificationHubResponse;
import software.amazon.awssdk.services.notifications.model.ResourceNotFoundException;
import software.amazon.awssdk.services.notifications.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.notifications.model.TagResourceRequest;
import software.amazon.awssdk.services.notifications.model.TagResourceResponse;
import software.amazon.awssdk.services.notifications.model.ThrottlingException;
import software.amazon.awssdk.services.notifications.model.UntagResourceRequest;
import software.amazon.awssdk.services.notifications.model.UntagResourceResponse;
import software.amazon.awssdk.services.notifications.model.UpdateEventRuleRequest;
import software.amazon.awssdk.services.notifications.model.UpdateEventRuleResponse;
import software.amazon.awssdk.services.notifications.model.UpdateNotificationConfigurationRequest;
import software.amazon.awssdk.services.notifications.model.UpdateNotificationConfigurationResponse;
import software.amazon.awssdk.services.notifications.model.ValidationException;
import software.amazon.awssdk.services.notifications.transform.AssociateChannelRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.AssociateManagedNotificationAccountContactRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.AssociateManagedNotificationAdditionalChannelRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.CreateEventRuleRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.CreateNotificationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.DeleteEventRuleRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.DeleteNotificationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.DeregisterNotificationHubRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.DisableNotificationsAccessForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.DisassociateChannelRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.DisassociateManagedNotificationAccountContactRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.DisassociateManagedNotificationAdditionalChannelRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.EnableNotificationsAccessForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.GetEventRuleRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.GetManagedNotificationChildEventRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.GetManagedNotificationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.GetManagedNotificationEventRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.GetNotificationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.GetNotificationEventRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.GetNotificationsAccessForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.ListChannelsRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.ListEventRulesRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.ListManagedNotificationChannelAssociationsRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.ListManagedNotificationChildEventsRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.ListManagedNotificationConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.ListManagedNotificationEventsRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.ListNotificationConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.ListNotificationEventsRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.ListNotificationHubsRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.RegisterNotificationHubRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.UpdateEventRuleRequestMarshaller;
import software.amazon.awssdk.services.notifications.transform.UpdateNotificationConfigurationRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link NotificationsClient}.
 *
 * @see NotificationsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultNotificationsClient implements NotificationsClient {
    private static final Logger log = Logger.loggerFor(DefaultNotificationsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultNotificationsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates a delivery <a
     * href="https://docs.aws.amazon.com/notifications/latest/userguide/managing-delivery-channels.html">Channel</a>
     * with a particular <code>NotificationConfiguration</code>. Supported Channels include Chatbot, the Console Mobile
     * Application, and emails (notifications-contacts).
     * </p>
     *
     * @param associateChannelRequest
     * @return Result of the AssociateChannel operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.AssociateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/AssociateChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateChannelResponse associateChannel(AssociateChannelRequest associateChannelRequest)
            throws ServiceQuotaExceededException, AccessDeniedException, ValidationException, InternalServerException,
            ThrottlingException, ConflictException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AssociateChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateChannel");

            return clientHandler.execute(new ClientExecutionParams<AssociateChannelRequest, AssociateChannelResponse>()
                    .withOperationName("AssociateChannel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(associateChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AssociateChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates an Account Contact with a particular <code>ManagedNotificationConfiguration</code>.
     * </p>
     *
     * @param associateManagedNotificationAccountContactRequest
     * @return Result of the AssociateManagedNotificationAccountContact operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.AssociateManagedNotificationAccountContact
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/AssociateManagedNotificationAccountContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateManagedNotificationAccountContactResponse associateManagedNotificationAccountContact(
            AssociateManagedNotificationAccountContactRequest associateManagedNotificationAccountContactRequest)
            throws ServiceQuotaExceededException, AccessDeniedException, ValidationException, InternalServerException,
            ThrottlingException, ConflictException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateManagedNotificationAccountContactResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, AssociateManagedNotificationAccountContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                associateManagedNotificationAccountContactRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateManagedNotificationAccountContactRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateManagedNotificationAccountContact");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateManagedNotificationAccountContactRequest, AssociateManagedNotificationAccountContactResponse>()
                            .withOperationName("AssociateManagedNotificationAccountContact")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(associateManagedNotificationAccountContactRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateManagedNotificationAccountContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates an additional Channel with a particular <code>ManagedNotificationConfiguration</code>.
     * </p>
     * <p>
     * Supported Channels include Chatbot, the Console Mobile Application, and emails (notifications-contacts).
     * </p>
     *
     * @param associateManagedNotificationAdditionalChannelRequest
     * @return Result of the AssociateManagedNotificationAdditionalChannel operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.AssociateManagedNotificationAdditionalChannel
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/AssociateManagedNotificationAdditionalChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateManagedNotificationAdditionalChannelResponse associateManagedNotificationAdditionalChannel(
            AssociateManagedNotificationAdditionalChannelRequest associateManagedNotificationAdditionalChannelRequest)
            throws ServiceQuotaExceededException, AccessDeniedException, ValidationException, InternalServerException,
            ThrottlingException, ConflictException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateManagedNotificationAdditionalChannelResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, AssociateManagedNotificationAdditionalChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                associateManagedNotificationAdditionalChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateManagedNotificationAdditionalChannelRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateManagedNotificationAdditionalChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateManagedNotificationAdditionalChannelRequest, AssociateManagedNotificationAdditionalChannelResponse>()
                            .withOperationName("AssociateManagedNotificationAdditionalChannel")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(associateManagedNotificationAdditionalChannelRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateManagedNotificationAdditionalChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an <a href="https://docs.aws.amazon.com/notifications/latest/userguide/glossary.html">
     * <code>EventRule</code> </a> that is associated with a specified <code>NotificationConfiguration</code>.
     * </p>
     *
     * @param createEventRuleRequest
     * @return Result of the CreateEventRule operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.CreateEventRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/CreateEventRule" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateEventRuleResponse createEventRule(CreateEventRuleRequest createEventRuleRequest)
            throws ServiceQuotaExceededException, AccessDeniedException, ValidationException, InternalServerException,
            ThrottlingException, ConflictException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateEventRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateEventRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEventRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEventRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEventRule");

            return clientHandler.execute(new ClientExecutionParams<CreateEventRuleRequest, CreateEventRuleResponse>()
                    .withOperationName("CreateEventRule").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createEventRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateEventRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new <code>NotificationConfiguration</code>.
     * </p>
     *
     * @param createNotificationConfigurationRequest
     * @return Result of the CreateNotificationConfiguration operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.CreateNotificationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/CreateNotificationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateNotificationConfigurationResponse createNotificationConfiguration(
            CreateNotificationConfigurationRequest createNotificationConfigurationRequest) throws ServiceQuotaExceededException,
            AccessDeniedException, ValidationException, InternalServerException, ThrottlingException, ConflictException,
            AwsServiceException, SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateNotificationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateNotificationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createNotificationConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createNotificationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNotificationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateNotificationConfigurationRequest, CreateNotificationConfigurationResponse>()
                            .withOperationName("CreateNotificationConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createNotificationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateNotificationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an <code>EventRule</code>.
     * </p>
     *
     * @param deleteEventRuleRequest
     * @return Result of the DeleteEventRule operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.DeleteEventRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/DeleteEventRule" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteEventRuleResponse deleteEventRule(DeleteEventRuleRequest deleteEventRuleRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, ConflictException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteEventRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteEventRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEventRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEventRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEventRule");

            return clientHandler.execute(new ClientExecutionParams<DeleteEventRuleRequest, DeleteEventRuleResponse>()
                    .withOperationName("DeleteEventRule").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteEventRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteEventRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a <code>NotificationConfiguration</code>.
     * </p>
     *
     * @param deleteNotificationConfigurationRequest
     * @return Result of the DeleteNotificationConfiguration operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.DeleteNotificationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/DeleteNotificationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteNotificationConfigurationResponse deleteNotificationConfiguration(
            DeleteNotificationConfigurationRequest deleteNotificationConfigurationRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, ConflictException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteNotificationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteNotificationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteNotificationConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteNotificationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNotificationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteNotificationConfigurationRequest, DeleteNotificationConfigurationResponse>()
                            .withOperationName("DeleteNotificationConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteNotificationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteNotificationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deregisters a <code>NotificationConfiguration</code> in the specified Region.
     * </p>
     * <note>
     * <p>
     * You can't deregister the last <code>NotificationHub</code> in the account. <code>NotificationEvents</code> stored
     * in the deregistered <code>NotificationConfiguration</code> are no longer be visible. Recreating a new
     * <code>NotificationConfiguration</code> in the same Region restores access to those
     * <code>NotificationEvents</code>.
     * </p>
     * </note>
     *
     * @param deregisterNotificationHubRequest
     * @return Result of the DeregisterNotificationHub operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.DeregisterNotificationHub
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/DeregisterNotificationHub"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeregisterNotificationHubResponse deregisterNotificationHub(
            DeregisterNotificationHubRequest deregisterNotificationHubRequest) throws AccessDeniedException, ValidationException,
            InternalServerException, ThrottlingException, ConflictException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeregisterNotificationHubResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeregisterNotificationHubResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deregisterNotificationHubRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterNotificationHubRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterNotificationHub");

            return clientHandler
                    .execute(new ClientExecutionParams<DeregisterNotificationHubRequest, DeregisterNotificationHubResponse>()
                            .withOperationName("DeregisterNotificationHub").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deregisterNotificationHubRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeregisterNotificationHubRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disables service trust between User Notifications and Amazon Web Services Organizations.
     * </p>
     *
     * @param disableNotificationsAccessForOrganizationRequest
     * @return Result of the DisableNotificationsAccessForOrganization operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.DisableNotificationsAccessForOrganization
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/DisableNotificationsAccessForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisableNotificationsAccessForOrganizationResponse disableNotificationsAccessForOrganization(
            DisableNotificationsAccessForOrganizationRequest disableNotificationsAccessForOrganizationRequest)
            throws ServiceQuotaExceededException, AccessDeniedException, ValidationException, InternalServerException,
            ThrottlingException, ConflictException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableNotificationsAccessForOrganizationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DisableNotificationsAccessForOrganizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                disableNotificationsAccessForOrganizationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disableNotificationsAccessForOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableNotificationsAccessForOrganization");

            return clientHandler
                    .execute(new ClientExecutionParams<DisableNotificationsAccessForOrganizationRequest, DisableNotificationsAccessForOrganizationResponse>()
                            .withOperationName("DisableNotificationsAccessForOrganization")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(disableNotificationsAccessForOrganizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisableNotificationsAccessForOrganizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates a Channel from a specified <code>NotificationConfiguration</code>. Supported Channels include
     * Chatbot, the Console Mobile Application, and emails (notifications-contacts).
     * </p>
     *
     * @param disassociateChannelRequest
     * @return Result of the DisassociateChannel operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.DisassociateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/DisassociateChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateChannelResponse disassociateChannel(DisassociateChannelRequest disassociateChannelRequest)
            throws AccessDeniedException, ValidationException, InternalServerException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateChannel");

            return clientHandler.execute(new ClientExecutionParams<DisassociateChannelRequest, DisassociateChannelResponse>()
                    .withOperationName("DisassociateChannel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(disassociateChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisassociateChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates an Account Contact with a particular <code>ManagedNotificationConfiguration</code>.
     * </p>
     *
     * @param disassociateManagedNotificationAccountContactRequest
     * @return Result of the DisassociateManagedNotificationAccountContact operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.DisassociateManagedNotificationAccountContact
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/DisassociateManagedNotificationAccountContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateManagedNotificationAccountContactResponse disassociateManagedNotificationAccountContact(
            DisassociateManagedNotificationAccountContactRequest disassociateManagedNotificationAccountContactRequest)
            throws AccessDeniedException, ValidationException, InternalServerException, ThrottlingException, ConflictException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateManagedNotificationAccountContactResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DisassociateManagedNotificationAccountContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                disassociateManagedNotificationAccountContactRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateManagedNotificationAccountContactRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateManagedNotificationAccountContact");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateManagedNotificationAccountContactRequest, DisassociateManagedNotificationAccountContactResponse>()
                            .withOperationName("DisassociateManagedNotificationAccountContact")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(disassociateManagedNotificationAccountContactRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateManagedNotificationAccountContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates an additional Channel from a particular <code>ManagedNotificationConfiguration</code>.
     * </p>
     * <p>
     * Supported Channels include Chatbot, the Console Mobile Application, and emails (notifications-contacts).
     * </p>
     *
     * @param disassociateManagedNotificationAdditionalChannelRequest
     * @return Result of the DisassociateManagedNotificationAdditionalChannel operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.DisassociateManagedNotificationAdditionalChannel
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/DisassociateManagedNotificationAdditionalChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateManagedNotificationAdditionalChannelResponse disassociateManagedNotificationAdditionalChannel(
            DisassociateManagedNotificationAdditionalChannelRequest disassociateManagedNotificationAdditionalChannelRequest)
            throws AccessDeniedException, ValidationException, InternalServerException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateManagedNotificationAdditionalChannelResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DisassociateManagedNotificationAdditionalChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                disassociateManagedNotificationAdditionalChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateManagedNotificationAdditionalChannelRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateManagedNotificationAdditionalChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateManagedNotificationAdditionalChannelRequest, DisassociateManagedNotificationAdditionalChannelResponse>()
                            .withOperationName("DisassociateManagedNotificationAdditionalChannel")
                            .withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(disassociateManagedNotificationAdditionalChannelRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(
                                    new DisassociateManagedNotificationAdditionalChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables service trust between User Notifications and Amazon Web Services Organizations.
     * </p>
     *
     * @param enableNotificationsAccessForOrganizationRequest
     * @return Result of the EnableNotificationsAccessForOrganization operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.EnableNotificationsAccessForOrganization
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/EnableNotificationsAccessForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public EnableNotificationsAccessForOrganizationResponse enableNotificationsAccessForOrganization(
            EnableNotificationsAccessForOrganizationRequest enableNotificationsAccessForOrganizationRequest)
            throws ServiceQuotaExceededException, AccessDeniedException, ValidationException, InternalServerException,
            ThrottlingException, ConflictException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableNotificationsAccessForOrganizationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, EnableNotificationsAccessForOrganizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                enableNotificationsAccessForOrganizationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                enableNotificationsAccessForOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableNotificationsAccessForOrganization");

            return clientHandler
                    .execute(new ClientExecutionParams<EnableNotificationsAccessForOrganizationRequest, EnableNotificationsAccessForOrganizationResponse>()
                            .withOperationName("EnableNotificationsAccessForOrganization").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(enableNotificationsAccessForOrganizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new EnableNotificationsAccessForOrganizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a specified <code>EventRule</code>.
     * </p>
     *
     * @param getEventRuleRequest
     * @return Result of the GetEventRule operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.GetEventRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/GetEventRule" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetEventRuleResponse getEventRule(GetEventRuleRequest getEventRuleRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEventRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetEventRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEventRuleRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEventRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEventRule");

            return clientHandler.execute(new ClientExecutionParams<GetEventRuleRequest, GetEventRuleResponse>()
                    .withOperationName("GetEventRule").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getEventRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetEventRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the child event of a specific given <code>ManagedNotificationEvent</code>.
     * </p>
     *
     * @param getManagedNotificationChildEventRequest
     * @return Result of the GetManagedNotificationChildEvent operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.GetManagedNotificationChildEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/GetManagedNotificationChildEvent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetManagedNotificationChildEventResponse getManagedNotificationChildEvent(
            GetManagedNotificationChildEventRequest getManagedNotificationChildEventRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetManagedNotificationChildEventResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetManagedNotificationChildEventResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getManagedNotificationChildEventRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getManagedNotificationChildEventRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetManagedNotificationChildEvent");

            return clientHandler
                    .execute(new ClientExecutionParams<GetManagedNotificationChildEventRequest, GetManagedNotificationChildEventResponse>()
                            .withOperationName("GetManagedNotificationChildEvent").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getManagedNotificationChildEventRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetManagedNotificationChildEventRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a specified <code>ManagedNotificationConfiguration</code>.
     * </p>
     *
     * @param getManagedNotificationConfigurationRequest
     * @return Result of the GetManagedNotificationConfiguration operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.GetManagedNotificationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/GetManagedNotificationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetManagedNotificationConfigurationResponse getManagedNotificationConfiguration(
            GetManagedNotificationConfigurationRequest getManagedNotificationConfigurationRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetManagedNotificationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetManagedNotificationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getManagedNotificationConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getManagedNotificationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetManagedNotificationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<GetManagedNotificationConfigurationRequest, GetManagedNotificationConfigurationResponse>()
                            .withOperationName("GetManagedNotificationConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getManagedNotificationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetManagedNotificationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a specified <code>ManagedNotificationEvent</code>.
     * </p>
     *
     * @param getManagedNotificationEventRequest
     * @return Result of the GetManagedNotificationEvent operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.GetManagedNotificationEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/GetManagedNotificationEvent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetManagedNotificationEventResponse getManagedNotificationEvent(
            GetManagedNotificationEventRequest getManagedNotificationEventRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetManagedNotificationEventResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetManagedNotificationEventResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getManagedNotificationEventRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getManagedNotificationEventRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetManagedNotificationEvent");

            return clientHandler
                    .execute(new ClientExecutionParams<GetManagedNotificationEventRequest, GetManagedNotificationEventResponse>()
                            .withOperationName("GetManagedNotificationEvent").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getManagedNotificationEventRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetManagedNotificationEventRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a specified <code>NotificationConfiguration</code>.
     * </p>
     *
     * @param getNotificationConfigurationRequest
     * @return Result of the GetNotificationConfiguration operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.GetNotificationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/GetNotificationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetNotificationConfigurationResponse getNotificationConfiguration(
            GetNotificationConfigurationRequest getNotificationConfigurationRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetNotificationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetNotificationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getNotificationConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNotificationConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNotificationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<GetNotificationConfigurationRequest, GetNotificationConfigurationResponse>()
                            .withOperationName("GetNotificationConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getNotificationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetNotificationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a specified <code>NotificationEvent</code>.
     * </p>
     * <important>
     * <p>
     * User Notifications stores notifications in the individual Regions you register as notification hubs and the
     * Region of the source event rule. <code>GetNotificationEvent</code> only returns notifications stored in the same
     * Region in which the action is called. User Notifications doesn't backfill notifications to new Regions selected
     * as notification hubs. For this reason, we recommend that you make calls in your oldest registered notification
     * hub. For more information, see <a
     * href="https://docs.aws.amazon.com/notifications/latest/userguide/notification-hubs.html">Notification hubs</a> in
     * the <i>Amazon Web Services User Notifications User Guide</i>.
     * </p>
     * </important>
     *
     * @param getNotificationEventRequest
     * @return Result of the GetNotificationEvent operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.GetNotificationEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/GetNotificationEvent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetNotificationEventResponse getNotificationEvent(GetNotificationEventRequest getNotificationEventRequest)
            throws AccessDeniedException, ValidationException, InternalServerException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetNotificationEventResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetNotificationEventResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getNotificationEventRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNotificationEventRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNotificationEvent");

            return clientHandler.execute(new ClientExecutionParams<GetNotificationEventRequest, GetNotificationEventResponse>()
                    .withOperationName("GetNotificationEvent").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getNotificationEventRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetNotificationEventRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the AccessStatus of Service Trust Enablement for User Notifications and Amazon Web Services
     * Organizations.
     * </p>
     *
     * @param getNotificationsAccessForOrganizationRequest
     * @return Result of the GetNotificationsAccessForOrganization operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.GetNotificationsAccessForOrganization
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/GetNotificationsAccessForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetNotificationsAccessForOrganizationResponse getNotificationsAccessForOrganization(
            GetNotificationsAccessForOrganizationRequest getNotificationsAccessForOrganizationRequest)
            throws AccessDeniedException, ValidationException, InternalServerException, ThrottlingException, AwsServiceException,
            SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetNotificationsAccessForOrganizationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, GetNotificationsAccessForOrganizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getNotificationsAccessForOrganizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getNotificationsAccessForOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNotificationsAccessForOrganization");

            return clientHandler
                    .execute(new ClientExecutionParams<GetNotificationsAccessForOrganizationRequest, GetNotificationsAccessForOrganizationResponse>()
                            .withOperationName("GetNotificationsAccessForOrganization").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(getNotificationsAccessForOrganizationRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetNotificationsAccessForOrganizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of Channels for a <code>NotificationConfiguration</code>.
     * </p>
     *
     * @param listChannelsRequest
     * @return Result of the ListChannels operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/ListChannels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListChannelsResponse listChannels(ListChannelsRequest listChannelsRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListChannelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListChannelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listChannelsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannels");

            return clientHandler.execute(new ClientExecutionParams<ListChannelsRequest, ListChannelsResponse>()
                    .withOperationName("ListChannels").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listChannelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListChannelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of <code>EventRules</code> according to specified filters, in reverse chronological order (newest
     * first).
     * </p>
     *
     * @param listEventRulesRequest
     * @return Result of the ListEventRules operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.ListEventRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/ListEventRules" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListEventRulesResponse listEventRules(ListEventRulesRequest listEventRulesRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEventRulesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListEventRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEventRulesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEventRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEventRules");

            return clientHandler.execute(new ClientExecutionParams<ListEventRulesRequest, ListEventRulesResponse>()
                    .withOperationName("ListEventRules").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listEventRulesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListEventRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of Account contacts and Channels associated with a <code>ManagedNotificationConfiguration</code>,
     * in paginated format.
     * </p>
     *
     * @param listManagedNotificationChannelAssociationsRequest
     * @return Result of the ListManagedNotificationChannelAssociations operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.ListManagedNotificationChannelAssociations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/ListManagedNotificationChannelAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListManagedNotificationChannelAssociationsResponse listManagedNotificationChannelAssociations(
            ListManagedNotificationChannelAssociationsRequest listManagedNotificationChannelAssociationsRequest)
            throws AccessDeniedException, ValidationException, InternalServerException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListManagedNotificationChannelAssociationsResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, ListManagedNotificationChannelAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                listManagedNotificationChannelAssociationsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listManagedNotificationChannelAssociationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListManagedNotificationChannelAssociations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListManagedNotificationChannelAssociationsRequest, ListManagedNotificationChannelAssociationsResponse>()
                            .withOperationName("ListManagedNotificationChannelAssociations")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(listManagedNotificationChannelAssociationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListManagedNotificationChannelAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of <code>ManagedNotificationChildEvents</code> for a specified aggregate
     * <code>ManagedNotificationEvent</code>, ordered by creation time in reverse chronological order (newest first).
     * </p>
     *
     * @param listManagedNotificationChildEventsRequest
     * @return Result of the ListManagedNotificationChildEvents operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.ListManagedNotificationChildEvents
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/ListManagedNotificationChildEvents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListManagedNotificationChildEventsResponse listManagedNotificationChildEvents(
            ListManagedNotificationChildEventsRequest listManagedNotificationChildEventsRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, AwsServiceException, SdkClientException,
            NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListManagedNotificationChildEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListManagedNotificationChildEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listManagedNotificationChildEventsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listManagedNotificationChildEventsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListManagedNotificationChildEvents");

            return clientHandler
                    .execute(new ClientExecutionParams<ListManagedNotificationChildEventsRequest, ListManagedNotificationChildEventsResponse>()
                            .withOperationName("ListManagedNotificationChildEvents").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listManagedNotificationChildEventsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListManagedNotificationChildEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of Managed Notification Configurations according to specified filters, ordered by creation time in
     * reverse chronological order (newest first).
     * </p>
     *
     * @param listManagedNotificationConfigurationsRequest
     * @return Result of the ListManagedNotificationConfigurations operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.ListManagedNotificationConfigurations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/ListManagedNotificationConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListManagedNotificationConfigurationsResponse listManagedNotificationConfigurations(
            ListManagedNotificationConfigurationsRequest listManagedNotificationConfigurationsRequest)
            throws AccessDeniedException, ValidationException, InternalServerException, ThrottlingException, AwsServiceException,
            SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListManagedNotificationConfigurationsResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, ListManagedNotificationConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listManagedNotificationConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listManagedNotificationConfigurationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListManagedNotificationConfigurations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListManagedNotificationConfigurationsRequest, ListManagedNotificationConfigurationsResponse>()
                            .withOperationName("ListManagedNotificationConfigurations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(listManagedNotificationConfigurationsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListManagedNotificationConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of Managed Notification Events according to specified filters, ordered by creation time in reverse
     * chronological order (newest first).
     * </p>
     *
     * @param listManagedNotificationEventsRequest
     * @return Result of the ListManagedNotificationEvents operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.ListManagedNotificationEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/ListManagedNotificationEvents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListManagedNotificationEventsResponse listManagedNotificationEvents(
            ListManagedNotificationEventsRequest listManagedNotificationEventsRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, AwsServiceException, SdkClientException,
            NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListManagedNotificationEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListManagedNotificationEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listManagedNotificationEventsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listManagedNotificationEventsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListManagedNotificationEvents");

            return clientHandler
                    .execute(new ClientExecutionParams<ListManagedNotificationEventsRequest, ListManagedNotificationEventsResponse>()
                            .withOperationName("ListManagedNotificationEvents").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listManagedNotificationEventsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListManagedNotificationEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of abbreviated <code>NotificationConfigurations</code> according to specified filters, in reverse
     * chronological order (newest first).
     * </p>
     *
     * @param listNotificationConfigurationsRequest
     * @return Result of the ListNotificationConfigurations operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.ListNotificationConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/ListNotificationConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNotificationConfigurationsResponse listNotificationConfigurations(
            ListNotificationConfigurationsRequest listNotificationConfigurationsRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, AwsServiceException, SdkClientException,
            NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNotificationConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListNotificationConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listNotificationConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listNotificationConfigurationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNotificationConfigurations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListNotificationConfigurationsRequest, ListNotificationConfigurationsResponse>()
                            .withOperationName("ListNotificationConfigurations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listNotificationConfigurationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListNotificationConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of <code>NotificationEvents</code> according to specified filters, in reverse chronological order
     * (newest first).
     * </p>
     * <important>
     * <p>
     * User Notifications stores notifications in the individual Regions you register as notification hubs and the
     * Region of the source event rule. ListNotificationEvents only returns notifications stored in the same Region in
     * which the action is called. User Notifications doesn't backfill notifications to new Regions selected as
     * notification hubs. For this reason, we recommend that you make calls in your oldest registered notification hub.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/notifications/latest/userguide/notification-hubs.html">Notification hubs</a> in
     * the <i>Amazon Web Services User Notifications User Guide</i>.
     * </p>
     * </important>
     *
     * @param listNotificationEventsRequest
     * @return Result of the ListNotificationEvents operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.ListNotificationEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/ListNotificationEvents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNotificationEventsResponse listNotificationEvents(ListNotificationEventsRequest listNotificationEventsRequest)
            throws AccessDeniedException, ValidationException, InternalServerException, ThrottlingException, AwsServiceException,
            SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNotificationEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListNotificationEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listNotificationEventsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNotificationEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNotificationEvents");

            return clientHandler
                    .execute(new ClientExecutionParams<ListNotificationEventsRequest, ListNotificationEventsResponse>()
                            .withOperationName("ListNotificationEvents").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listNotificationEventsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListNotificationEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of <code>NotificationHubs</code>.
     * </p>
     *
     * @param listNotificationHubsRequest
     * @return Result of the ListNotificationHubs operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.ListNotificationHubs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/ListNotificationHubs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNotificationHubsResponse listNotificationHubs(ListNotificationHubsRequest listNotificationHubsRequest)
            throws AccessDeniedException, ValidationException, InternalServerException, ThrottlingException, AwsServiceException,
            SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNotificationHubsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListNotificationHubsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listNotificationHubsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNotificationHubsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNotificationHubs");

            return clientHandler.execute(new ClientExecutionParams<ListNotificationHubsRequest, ListNotificationHubsResponse>()
                    .withOperationName("ListNotificationHubs").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listNotificationHubsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListNotificationHubsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of tags for a specified Amazon Resource Name (ARN).
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html">Tagging
     * your Amazon Web Services resources</a> in the <i>Tagging Amazon Web Services Resources User Guide</i>.
     * </p>
     * <note>
     * <p>
     * This is only supported for <code>NotificationConfigurations</code>.
     * </p>
     * </note>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws AccessDeniedException, ValidationException, InternalServerException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Registers a <code>NotificationConfiguration</code> in the specified Region.
     * </p>
     * <p>
     * There is a maximum of one <code>NotificationConfiguration</code> per Region. You can have a maximum of 3
     * <code>NotificationHub</code> resources at a time.
     * </p>
     *
     * @param registerNotificationHubRequest
     * @return Result of the RegisterNotificationHub operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.RegisterNotificationHub
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/RegisterNotificationHub"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RegisterNotificationHubResponse registerNotificationHub(RegisterNotificationHubRequest registerNotificationHubRequest)
            throws ServiceQuotaExceededException, AccessDeniedException, ValidationException, InternalServerException,
            ThrottlingException, ConflictException, AwsServiceException, SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterNotificationHubResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RegisterNotificationHubResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerNotificationHubRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerNotificationHubRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterNotificationHub");

            return clientHandler
                    .execute(new ClientExecutionParams<RegisterNotificationHubRequest, RegisterNotificationHubResponse>()
                            .withOperationName("RegisterNotificationHub").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(registerNotificationHubRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RegisterNotificationHubRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tags the resource with a tag key and value.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html">Tagging
     * your Amazon Web Services resources</a> in the <i>Tagging Amazon Web Services Resources User Guide</i>.
     * </p>
     * <note>
     * <p>
     * This is only supported for <code>NotificationConfigurations</code>.
     * </p>
     * </note>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Untags a resource with a specified Amazon Resource Name (ARN).
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html">Tagging
     * your Amazon Web Services resources</a> in the <i>Tagging Amazon Web Services Resources User Guide</i>.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing <code>EventRule</code>.
     * </p>
     *
     * @param updateEventRuleRequest
     * @return Result of the UpdateEventRule operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.UpdateEventRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/UpdateEventRule" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateEventRuleResponse updateEventRule(UpdateEventRuleRequest updateEventRuleRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, ConflictException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEventRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateEventRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEventRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEventRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEventRule");

            return clientHandler.execute(new ClientExecutionParams<UpdateEventRuleRequest, UpdateEventRuleResponse>()
                    .withOperationName("UpdateEventRule").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateEventRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateEventRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a <code>NotificationConfiguration</code>.
     * </p>
     *
     * @param updateNotificationConfigurationRequest
     * @return Result of the UpdateNotificationConfiguration operation returned by the service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ValidationException
     *         This exception is thrown when the notification event fails validation.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws NotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample NotificationsClient.UpdateNotificationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/notifications-2018-05-10/UpdateNotificationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateNotificationConfigurationResponse updateNotificationConfiguration(
            UpdateNotificationConfigurationRequest updateNotificationConfigurationRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, ConflictException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, NotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateNotificationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateNotificationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateNotificationConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateNotificationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Notifications");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNotificationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateNotificationConfigurationRequest, UpdateNotificationConfigurationResponse>()
                            .withOperationName("UpdateNotificationConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateNotificationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateNotificationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        NotificationsServiceClientConfigurationBuilder serviceConfigBuilder = new NotificationsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(NotificationsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    @Override
    public final NotificationsServiceClientConfiguration serviceClientConfiguration() {
        return new NotificationsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
