/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.notifications.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the metadata for a source event.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-events-structure.html">Event structure
 * reference</a> in the <i>Amazon EventBridge User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SourceEventMetadata implements SdkPojo, Serializable,
        ToCopyableBuilder<SourceEventMetadata.Builder, SourceEventMetadata> {
    private static final SdkField<String> EVENT_TYPE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("eventTypeVersion").getter(getter(SourceEventMetadata::eventTypeVersion))
            .setter(setter(Builder::eventTypeVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventTypeVersion").build()).build();

    private static final SdkField<String> SOURCE_EVENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sourceEventId").getter(getter(SourceEventMetadata::sourceEventId))
            .setter(setter(Builder::sourceEventId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceEventId").build()).build();

    private static final SdkField<String> EVENT_ORIGIN_REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("eventOriginRegion").getter(getter(SourceEventMetadata::eventOriginRegion))
            .setter(setter(Builder::eventOriginRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventOriginRegion").build()).build();

    private static final SdkField<String> RELATED_ACCOUNT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("relatedAccount").getter(getter(SourceEventMetadata::relatedAccount))
            .setter(setter(Builder::relatedAccount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("relatedAccount").build()).build();

    private static final SdkField<String> SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("source")
            .getter(getter(SourceEventMetadata::source)).setter(setter(Builder::source))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("source").build()).build();

    private static final SdkField<Instant> EVENT_OCCURRENCE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("eventOccurrenceTime")
            .getter(getter(SourceEventMetadata::eventOccurrenceTime))
            .setter(setter(Builder::eventOccurrenceTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventOccurrenceTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> EVENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("eventType").getter(getter(SourceEventMetadata::eventType)).setter(setter(Builder::eventType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventType").build()).build();

    private static final SdkField<List<Resource>> RELATED_RESOURCES_FIELD = SdkField
            .<List<Resource>> builder(MarshallingType.LIST)
            .memberName("relatedResources")
            .getter(getter(SourceEventMetadata::relatedResources))
            .setter(setter(Builder::relatedResources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("relatedResources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Resource> builder(MarshallingType.SDK_POJO)
                                            .constructor(Resource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVENT_TYPE_VERSION_FIELD,
            SOURCE_EVENT_ID_FIELD, EVENT_ORIGIN_REGION_FIELD, RELATED_ACCOUNT_FIELD, SOURCE_FIELD, EVENT_OCCURRENCE_TIME_FIELD,
            EVENT_TYPE_FIELD, RELATED_RESOURCES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String eventTypeVersion;

    private final String sourceEventId;

    private final String eventOriginRegion;

    private final String relatedAccount;

    private final String source;

    private final Instant eventOccurrenceTime;

    private final String eventType;

    private final List<Resource> relatedResources;

    private SourceEventMetadata(BuilderImpl builder) {
        this.eventTypeVersion = builder.eventTypeVersion;
        this.sourceEventId = builder.sourceEventId;
        this.eventOriginRegion = builder.eventOriginRegion;
        this.relatedAccount = builder.relatedAccount;
        this.source = builder.source;
        this.eventOccurrenceTime = builder.eventOccurrenceTime;
        this.eventType = builder.eventType;
        this.relatedResources = builder.relatedResources;
    }

    /**
     * <p>
     * The version of the type of event.
     * </p>
     * 
     * @return The version of the type of event.
     */
    public final String eventTypeVersion() {
        return eventTypeVersion;
    }

    /**
     * <p>
     * The source event id.
     * </p>
     * 
     * @return The source event id.
     */
    public final String sourceEventId() {
        return sourceEventId;
    }

    /**
     * <p>
     * The Region the event originated from.
     * </p>
     * 
     * @return The Region the event originated from.
     */
    public final String eventOriginRegion() {
        return eventOriginRegion;
    }

    /**
     * <p>
     * The primary Amazon Web Services account of <code>SourceEvent</code>.
     * </p>
     * 
     * @return The primary Amazon Web Services account of <code>SourceEvent</code>.
     */
    public final String relatedAccount() {
        return relatedAccount;
    }

    /**
     * <p>
     * The Amazon Web Services service the event originates from. For example <code>aws.cloudwatch</code>.
     * </p>
     * 
     * @return The Amazon Web Services service the event originates from. For example <code>aws.cloudwatch</code>.
     */
    public final String source() {
        return source;
    }

    /**
     * <p>
     * The date and time the source event occurred. This is based on the Source Event.
     * </p>
     * 
     * @return The date and time the source event occurred. This is based on the Source Event.
     */
    public final Instant eventOccurrenceTime() {
        return eventOccurrenceTime;
    }

    /**
     * <p>
     * The type of event. For example, an Amazon CloudWatch state change.
     * </p>
     * 
     * @return The type of event. For example, an Amazon CloudWatch state change.
     */
    public final String eventType() {
        return eventType;
    }

    /**
     * For responses, this returns true if the service returned a value for the RelatedResources property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRelatedResources() {
        return relatedResources != null && !(relatedResources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of resources related to this <code>NotificationEvent</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRelatedResources} method.
     * </p>
     * 
     * @return A list of resources related to this <code>NotificationEvent</code>.
     */
    public final List<Resource> relatedResources() {
        return relatedResources;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(eventTypeVersion());
        hashCode = 31 * hashCode + Objects.hashCode(sourceEventId());
        hashCode = 31 * hashCode + Objects.hashCode(eventOriginRegion());
        hashCode = 31 * hashCode + Objects.hashCode(relatedAccount());
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(eventOccurrenceTime());
        hashCode = 31 * hashCode + Objects.hashCode(eventType());
        hashCode = 31 * hashCode + Objects.hashCode(hasRelatedResources() ? relatedResources() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SourceEventMetadata)) {
            return false;
        }
        SourceEventMetadata other = (SourceEventMetadata) obj;
        return Objects.equals(eventTypeVersion(), other.eventTypeVersion())
                && Objects.equals(sourceEventId(), other.sourceEventId())
                && Objects.equals(eventOriginRegion(), other.eventOriginRegion())
                && Objects.equals(relatedAccount(), other.relatedAccount()) && Objects.equals(source(), other.source())
                && Objects.equals(eventOccurrenceTime(), other.eventOccurrenceTime())
                && Objects.equals(eventType(), other.eventType()) && hasRelatedResources() == other.hasRelatedResources()
                && Objects.equals(relatedResources(), other.relatedResources());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SourceEventMetadata").add("EventTypeVersion", eventTypeVersion())
                .add("SourceEventId", sourceEventId()).add("EventOriginRegion", eventOriginRegion())
                .add("RelatedAccount", relatedAccount()).add("Source", source())
                .add("EventOccurrenceTime", eventOccurrenceTime()).add("EventType", eventType())
                .add("RelatedResources", hasRelatedResources() ? relatedResources() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "eventTypeVersion":
            return Optional.ofNullable(clazz.cast(eventTypeVersion()));
        case "sourceEventId":
            return Optional.ofNullable(clazz.cast(sourceEventId()));
        case "eventOriginRegion":
            return Optional.ofNullable(clazz.cast(eventOriginRegion()));
        case "relatedAccount":
            return Optional.ofNullable(clazz.cast(relatedAccount()));
        case "source":
            return Optional.ofNullable(clazz.cast(source()));
        case "eventOccurrenceTime":
            return Optional.ofNullable(clazz.cast(eventOccurrenceTime()));
        case "eventType":
            return Optional.ofNullable(clazz.cast(eventType()));
        case "relatedResources":
            return Optional.ofNullable(clazz.cast(relatedResources()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("eventTypeVersion", EVENT_TYPE_VERSION_FIELD);
        map.put("sourceEventId", SOURCE_EVENT_ID_FIELD);
        map.put("eventOriginRegion", EVENT_ORIGIN_REGION_FIELD);
        map.put("relatedAccount", RELATED_ACCOUNT_FIELD);
        map.put("source", SOURCE_FIELD);
        map.put("eventOccurrenceTime", EVENT_OCCURRENCE_TIME_FIELD);
        map.put("eventType", EVENT_TYPE_FIELD);
        map.put("relatedResources", RELATED_RESOURCES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SourceEventMetadata, T> g) {
        return obj -> g.apply((SourceEventMetadata) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SourceEventMetadata> {
        /**
         * <p>
         * The version of the type of event.
         * </p>
         * 
         * @param eventTypeVersion
         *        The version of the type of event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventTypeVersion(String eventTypeVersion);

        /**
         * <p>
         * The source event id.
         * </p>
         * 
         * @param sourceEventId
         *        The source event id.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceEventId(String sourceEventId);

        /**
         * <p>
         * The Region the event originated from.
         * </p>
         * 
         * @param eventOriginRegion
         *        The Region the event originated from.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventOriginRegion(String eventOriginRegion);

        /**
         * <p>
         * The primary Amazon Web Services account of <code>SourceEvent</code>.
         * </p>
         * 
         * @param relatedAccount
         *        The primary Amazon Web Services account of <code>SourceEvent</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedAccount(String relatedAccount);

        /**
         * <p>
         * The Amazon Web Services service the event originates from. For example <code>aws.cloudwatch</code>.
         * </p>
         * 
         * @param source
         *        The Amazon Web Services service the event originates from. For example <code>aws.cloudwatch</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * The date and time the source event occurred. This is based on the Source Event.
         * </p>
         * 
         * @param eventOccurrenceTime
         *        The date and time the source event occurred. This is based on the Source Event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventOccurrenceTime(Instant eventOccurrenceTime);

        /**
         * <p>
         * The type of event. For example, an Amazon CloudWatch state change.
         * </p>
         * 
         * @param eventType
         *        The type of event. For example, an Amazon CloudWatch state change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventType(String eventType);

        /**
         * <p>
         * A list of resources related to this <code>NotificationEvent</code>.
         * </p>
         * 
         * @param relatedResources
         *        A list of resources related to this <code>NotificationEvent</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedResources(Collection<Resource> relatedResources);

        /**
         * <p>
         * A list of resources related to this <code>NotificationEvent</code>.
         * </p>
         * 
         * @param relatedResources
         *        A list of resources related to this <code>NotificationEvent</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedResources(Resource... relatedResources);

        /**
         * <p>
         * A list of resources related to this <code>NotificationEvent</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.notifications.model.Resource.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.notifications.model.Resource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.notifications.model.Resource.Builder#build()} is called immediately
         * and its result is passed to {@link #relatedResources(List<Resource>)}.
         * 
         * @param relatedResources
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.notifications.model.Resource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #relatedResources(java.util.Collection<Resource>)
         */
        Builder relatedResources(Consumer<Resource.Builder>... relatedResources);
    }

    static final class BuilderImpl implements Builder {
        private String eventTypeVersion;

        private String sourceEventId;

        private String eventOriginRegion;

        private String relatedAccount;

        private String source;

        private Instant eventOccurrenceTime;

        private String eventType;

        private List<Resource> relatedResources = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SourceEventMetadata model) {
            eventTypeVersion(model.eventTypeVersion);
            sourceEventId(model.sourceEventId);
            eventOriginRegion(model.eventOriginRegion);
            relatedAccount(model.relatedAccount);
            source(model.source);
            eventOccurrenceTime(model.eventOccurrenceTime);
            eventType(model.eventType);
            relatedResources(model.relatedResources);
        }

        public final String getEventTypeVersion() {
            return eventTypeVersion;
        }

        public final void setEventTypeVersion(String eventTypeVersion) {
            this.eventTypeVersion = eventTypeVersion;
        }

        @Override
        public final Builder eventTypeVersion(String eventTypeVersion) {
            this.eventTypeVersion = eventTypeVersion;
            return this;
        }

        public final String getSourceEventId() {
            return sourceEventId;
        }

        public final void setSourceEventId(String sourceEventId) {
            this.sourceEventId = sourceEventId;
        }

        @Override
        public final Builder sourceEventId(String sourceEventId) {
            this.sourceEventId = sourceEventId;
            return this;
        }

        public final String getEventOriginRegion() {
            return eventOriginRegion;
        }

        public final void setEventOriginRegion(String eventOriginRegion) {
            this.eventOriginRegion = eventOriginRegion;
        }

        @Override
        public final Builder eventOriginRegion(String eventOriginRegion) {
            this.eventOriginRegion = eventOriginRegion;
            return this;
        }

        public final String getRelatedAccount() {
            return relatedAccount;
        }

        public final void setRelatedAccount(String relatedAccount) {
            this.relatedAccount = relatedAccount;
        }

        @Override
        public final Builder relatedAccount(String relatedAccount) {
            this.relatedAccount = relatedAccount;
            return this;
        }

        public final String getSource() {
            return source;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final Instant getEventOccurrenceTime() {
            return eventOccurrenceTime;
        }

        public final void setEventOccurrenceTime(Instant eventOccurrenceTime) {
            this.eventOccurrenceTime = eventOccurrenceTime;
        }

        @Override
        public final Builder eventOccurrenceTime(Instant eventOccurrenceTime) {
            this.eventOccurrenceTime = eventOccurrenceTime;
            return this;
        }

        public final String getEventType() {
            return eventType;
        }

        public final void setEventType(String eventType) {
            this.eventType = eventType;
        }

        @Override
        public final Builder eventType(String eventType) {
            this.eventType = eventType;
            return this;
        }

        public final List<Resource.Builder> getRelatedResources() {
            List<Resource.Builder> result = ResourcesCopier.copyToBuilder(this.relatedResources);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRelatedResources(Collection<Resource.BuilderImpl> relatedResources) {
            this.relatedResources = ResourcesCopier.copyFromBuilder(relatedResources);
        }

        @Override
        public final Builder relatedResources(Collection<Resource> relatedResources) {
            this.relatedResources = ResourcesCopier.copy(relatedResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedResources(Resource... relatedResources) {
            relatedResources(Arrays.asList(relatedResources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedResources(Consumer<Resource.Builder>... relatedResources) {
            relatedResources(Stream.of(relatedResources).map(c -> Resource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public SourceEventMetadata build() {
            return new SourceEventMetadata(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
