/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.notifications.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the components of a notification message.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MessageComponents implements SdkPojo, Serializable,
        ToCopyableBuilder<MessageComponents.Builder, MessageComponents> {
    private static final SdkField<String> HEADLINE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("headline").getter(getter(MessageComponents::headline)).setter(setter(Builder::headline))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("headline").build()).build();

    private static final SdkField<String> PARAGRAPH_SUMMARY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("paragraphSummary").getter(getter(MessageComponents::paragraphSummary))
            .setter(setter(Builder::paragraphSummary))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("paragraphSummary").build()).build();

    private static final SdkField<String> COMPLETE_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("completeDescription").getter(getter(MessageComponents::completeDescription))
            .setter(setter(Builder::completeDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("completeDescription").build())
            .build();

    private static final SdkField<List<Dimension>> DIMENSIONS_FIELD = SdkField
            .<List<Dimension>> builder(MarshallingType.LIST)
            .memberName("dimensions")
            .getter(getter(MessageComponents::dimensions))
            .setter(setter(Builder::dimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dimensions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Dimension> builder(MarshallingType.SDK_POJO)
                                            .constructor(Dimension::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(HEADLINE_FIELD,
            PARAGRAPH_SUMMARY_FIELD, COMPLETE_DESCRIPTION_FIELD, DIMENSIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String headline;

    private final String paragraphSummary;

    private final String completeDescription;

    private final List<Dimension> dimensions;

    private MessageComponents(BuilderImpl builder) {
        this.headline = builder.headline;
        this.paragraphSummary = builder.paragraphSummary;
        this.completeDescription = builder.completeDescription;
        this.dimensions = builder.dimensions;
    }

    /**
     * <p>
     * A sentence long summary. For example, titles or an email subject line.
     * </p>
     * 
     * @return A sentence long summary. For example, titles or an email subject line.
     */
    public final String headline() {
        return headline;
    }

    /**
     * <p>
     * A paragraph long or multiple sentence summary. For example, Amazon Q Developer in chat applications
     * notifications.
     * </p>
     * 
     * @return A paragraph long or multiple sentence summary. For example, Amazon Q Developer in chat applications
     *         notifications.
     */
    public final String paragraphSummary() {
        return paragraphSummary;
    }

    /**
     * <p>
     * A complete summary with all possible relevant information.
     * </p>
     * 
     * @return A complete summary with all possible relevant information.
     */
    public final String completeDescription() {
        return completeDescription;
    }

    /**
     * For responses, this returns true if the service returned a value for the Dimensions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDimensions() {
        return dimensions != null && !(dimensions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of properties in key-value pairs. Pairs are shown in order of importance from most important to least
     * important. Channels may limit the number of dimensions shown to the notification viewer.
     * </p>
     * <note>
     * <p>
     * Included dimensions, keys, and values are subject to change.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDimensions} method.
     * </p>
     * 
     * @return A list of properties in key-value pairs. Pairs are shown in order of importance from most important to
     *         least important. Channels may limit the number of dimensions shown to the notification viewer.</p> <note>
     *         <p>
     *         Included dimensions, keys, and values are subject to change.
     *         </p>
     */
    public final List<Dimension> dimensions() {
        return dimensions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(headline());
        hashCode = 31 * hashCode + Objects.hashCode(paragraphSummary());
        hashCode = 31 * hashCode + Objects.hashCode(completeDescription());
        hashCode = 31 * hashCode + Objects.hashCode(hasDimensions() ? dimensions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MessageComponents)) {
            return false;
        }
        MessageComponents other = (MessageComponents) obj;
        return Objects.equals(headline(), other.headline()) && Objects.equals(paragraphSummary(), other.paragraphSummary())
                && Objects.equals(completeDescription(), other.completeDescription()) && hasDimensions() == other.hasDimensions()
                && Objects.equals(dimensions(), other.dimensions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MessageComponents").add("Headline", headline()).add("ParagraphSummary", paragraphSummary())
                .add("CompleteDescription", completeDescription()).add("Dimensions", hasDimensions() ? dimensions() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "headline":
            return Optional.ofNullable(clazz.cast(headline()));
        case "paragraphSummary":
            return Optional.ofNullable(clazz.cast(paragraphSummary()));
        case "completeDescription":
            return Optional.ofNullable(clazz.cast(completeDescription()));
        case "dimensions":
            return Optional.ofNullable(clazz.cast(dimensions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("headline", HEADLINE_FIELD);
        map.put("paragraphSummary", PARAGRAPH_SUMMARY_FIELD);
        map.put("completeDescription", COMPLETE_DESCRIPTION_FIELD);
        map.put("dimensions", DIMENSIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<MessageComponents, T> g) {
        return obj -> g.apply((MessageComponents) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MessageComponents> {
        /**
         * <p>
         * A sentence long summary. For example, titles or an email subject line.
         * </p>
         * 
         * @param headline
         *        A sentence long summary. For example, titles or an email subject line.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder headline(String headline);

        /**
         * <p>
         * A paragraph long or multiple sentence summary. For example, Amazon Q Developer in chat applications
         * notifications.
         * </p>
         * 
         * @param paragraphSummary
         *        A paragraph long or multiple sentence summary. For example, Amazon Q Developer in chat applications
         *        notifications.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder paragraphSummary(String paragraphSummary);

        /**
         * <p>
         * A complete summary with all possible relevant information.
         * </p>
         * 
         * @param completeDescription
         *        A complete summary with all possible relevant information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completeDescription(String completeDescription);

        /**
         * <p>
         * A list of properties in key-value pairs. Pairs are shown in order of importance from most important to least
         * important. Channels may limit the number of dimensions shown to the notification viewer.
         * </p>
         * <note>
         * <p>
         * Included dimensions, keys, and values are subject to change.
         * </p>
         * </note>
         * 
         * @param dimensions
         *        A list of properties in key-value pairs. Pairs are shown in order of importance from most important to
         *        least important. Channels may limit the number of dimensions shown to the notification viewer.</p>
         *        <note>
         *        <p>
         *        Included dimensions, keys, and values are subject to change.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<Dimension> dimensions);

        /**
         * <p>
         * A list of properties in key-value pairs. Pairs are shown in order of importance from most important to least
         * important. Channels may limit the number of dimensions shown to the notification viewer.
         * </p>
         * <note>
         * <p>
         * Included dimensions, keys, and values are subject to change.
         * </p>
         * </note>
         * 
         * @param dimensions
         *        A list of properties in key-value pairs. Pairs are shown in order of importance from most important to
         *        least important. Channels may limit the number of dimensions shown to the notification viewer.</p>
         *        <note>
         *        <p>
         *        Included dimensions, keys, and values are subject to change.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Dimension... dimensions);

        /**
         * <p>
         * A list of properties in key-value pairs. Pairs are shown in order of importance from most important to least
         * important. Channels may limit the number of dimensions shown to the notification viewer.
         * </p>
         * <note>
         * <p>
         * Included dimensions, keys, and values are subject to change.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.notifications.model.Dimension.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.notifications.model.Dimension#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.notifications.model.Dimension.Builder#build()} is called immediately
         * and its result is passed to {@link #dimensions(List<Dimension>)}.
         * 
         * @param dimensions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.notifications.model.Dimension.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dimensions(java.util.Collection<Dimension>)
         */
        Builder dimensions(Consumer<Dimension.Builder>... dimensions);
    }

    static final class BuilderImpl implements Builder {
        private String headline;

        private String paragraphSummary;

        private String completeDescription;

        private List<Dimension> dimensions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MessageComponents model) {
            headline(model.headline);
            paragraphSummary(model.paragraphSummary);
            completeDescription(model.completeDescription);
            dimensions(model.dimensions);
        }

        public final String getHeadline() {
            return headline;
        }

        public final void setHeadline(String headline) {
            this.headline = headline;
        }

        @Override
        public final Builder headline(String headline) {
            this.headline = headline;
            return this;
        }

        public final String getParagraphSummary() {
            return paragraphSummary;
        }

        public final void setParagraphSummary(String paragraphSummary) {
            this.paragraphSummary = paragraphSummary;
        }

        @Override
        public final Builder paragraphSummary(String paragraphSummary) {
            this.paragraphSummary = paragraphSummary;
            return this;
        }

        public final String getCompleteDescription() {
            return completeDescription;
        }

        public final void setCompleteDescription(String completeDescription) {
            this.completeDescription = completeDescription;
        }

        @Override
        public final Builder completeDescription(String completeDescription) {
            this.completeDescription = completeDescription;
            return this;
        }

        public final List<Dimension.Builder> getDimensions() {
            List<Dimension.Builder> result = DimensionsCopier.copyToBuilder(this.dimensions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDimensions(Collection<Dimension.BuilderImpl> dimensions) {
            this.dimensions = DimensionsCopier.copyFromBuilder(dimensions);
        }

        @Override
        public final Builder dimensions(Collection<Dimension> dimensions) {
            this.dimensions = DimensionsCopier.copy(dimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Dimension... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Consumer<Dimension.Builder>... dimensions) {
            dimensions(Stream.of(dimensions).map(c -> Dimension.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public MessageComponents build() {
            return new MessageComponents(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
