/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.notifications.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A <code>NotificationEvent</code> is a notification-focused representation of an event. They contain semantic
 * information used by Channels to create end-user notifications.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NotificationEvent implements SdkPojo, Serializable,
        ToCopyableBuilder<NotificationEvent.Builder, NotificationEvent> {
    private static final SdkField<String> SCHEMA_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("schemaVersion").getter(getter(NotificationEvent::schemaVersionAsString))
            .setter(setter(Builder::schemaVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("schemaVersion").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(NotificationEvent::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<SourceEventMetadata> SOURCE_EVENT_METADATA_FIELD = SdkField
            .<SourceEventMetadata> builder(MarshallingType.SDK_POJO).memberName("sourceEventMetadata")
            .getter(getter(NotificationEvent::sourceEventMetadata)).setter(setter(Builder::sourceEventMetadata))
            .constructor(SourceEventMetadata::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceEventMetadata").build())
            .build();

    private static final SdkField<MessageComponents> MESSAGE_COMPONENTS_FIELD = SdkField
            .<MessageComponents> builder(MarshallingType.SDK_POJO).memberName("messageComponents")
            .getter(getter(NotificationEvent::messageComponents)).setter(setter(Builder::messageComponents))
            .constructor(MessageComponents::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("messageComponents").build()).build();

    private static final SdkField<String> SOURCE_EVENT_DETAIL_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sourceEventDetailUrl").getter(getter(NotificationEvent::sourceEventDetailUrl))
            .setter(setter(Builder::sourceEventDetailUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceEventDetailUrl").build())
            .build();

    private static final SdkField<String> SOURCE_EVENT_DETAIL_URL_DISPLAY_TEXT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("sourceEventDetailUrlDisplayText")
            .getter(getter(NotificationEvent::sourceEventDetailUrlDisplayText))
            .setter(setter(Builder::sourceEventDetailUrlDisplayText))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceEventDetailUrlDisplayText")
                    .build()).build();

    private static final SdkField<String> NOTIFICATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("notificationType").getter(getter(NotificationEvent::notificationTypeAsString))
            .setter(setter(Builder::notificationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("notificationType").build()).build();

    private static final SdkField<String> EVENT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("eventStatus").getter(getter(NotificationEvent::eventStatusAsString))
            .setter(setter(Builder::eventStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventStatus").build()).build();

    private static final SdkField<String> AGGREGATION_EVENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("aggregationEventType").getter(getter(NotificationEvent::aggregationEventTypeAsString))
            .setter(setter(Builder::aggregationEventType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("aggregationEventType").build())
            .build();

    private static final SdkField<String> AGGREGATE_NOTIFICATION_EVENT_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("aggregateNotificationEventArn")
            .getter(getter(NotificationEvent::aggregateNotificationEventArn))
            .setter(setter(Builder::aggregateNotificationEventArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("aggregateNotificationEventArn")
                    .build()).build();

    private static final SdkField<AggregationSummary> AGGREGATION_SUMMARY_FIELD = SdkField
            .<AggregationSummary> builder(MarshallingType.SDK_POJO).memberName("aggregationSummary")
            .getter(getter(NotificationEvent::aggregationSummary)).setter(setter(Builder::aggregationSummary))
            .constructor(AggregationSummary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("aggregationSummary").build())
            .build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime")
            .getter(getter(NotificationEvent::startTime))
            .setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("endTime")
            .getter(getter(NotificationEvent::endTime))
            .setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Map<String, TextPartValue>> TEXT_PARTS_FIELD = SdkField
            .<Map<String, TextPartValue>> builder(MarshallingType.MAP)
            .memberName("textParts")
            .getter(getter(NotificationEvent::textParts))
            .setter(setter(Builder::textParts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("textParts").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<TextPartValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(TextPartValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<MediaElement>> MEDIA_FIELD = SdkField
            .<List<MediaElement>> builder(MarshallingType.LIST)
            .memberName("media")
            .getter(getter(NotificationEvent::media))
            .setter(setter(Builder::media))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("media").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MediaElement> builder(MarshallingType.SDK_POJO)
                                            .constructor(MediaElement::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ORGANIZATIONAL_UNIT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("organizationalUnitId").getter(getter(NotificationEvent::organizationalUnitId))
            .setter(setter(Builder::organizationalUnitId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("organizationalUnitId").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SCHEMA_VERSION_FIELD,
            ID_FIELD, SOURCE_EVENT_METADATA_FIELD, MESSAGE_COMPONENTS_FIELD, SOURCE_EVENT_DETAIL_URL_FIELD,
            SOURCE_EVENT_DETAIL_URL_DISPLAY_TEXT_FIELD, NOTIFICATION_TYPE_FIELD, EVENT_STATUS_FIELD,
            AGGREGATION_EVENT_TYPE_FIELD, AGGREGATE_NOTIFICATION_EVENT_ARN_FIELD, AGGREGATION_SUMMARY_FIELD, START_TIME_FIELD,
            END_TIME_FIELD, TEXT_PARTS_FIELD, MEDIA_FIELD, ORGANIZATIONAL_UNIT_ID_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String schemaVersion;

    private final String id;

    private final SourceEventMetadata sourceEventMetadata;

    private final MessageComponents messageComponents;

    private final String sourceEventDetailUrl;

    private final String sourceEventDetailUrlDisplayText;

    private final String notificationType;

    private final String eventStatus;

    private final String aggregationEventType;

    private final String aggregateNotificationEventArn;

    private final AggregationSummary aggregationSummary;

    private final Instant startTime;

    private final Instant endTime;

    private final Map<String, TextPartValue> textParts;

    private final List<MediaElement> media;

    private final String organizationalUnitId;

    private NotificationEvent(BuilderImpl builder) {
        this.schemaVersion = builder.schemaVersion;
        this.id = builder.id;
        this.sourceEventMetadata = builder.sourceEventMetadata;
        this.messageComponents = builder.messageComponents;
        this.sourceEventDetailUrl = builder.sourceEventDetailUrl;
        this.sourceEventDetailUrlDisplayText = builder.sourceEventDetailUrlDisplayText;
        this.notificationType = builder.notificationType;
        this.eventStatus = builder.eventStatus;
        this.aggregationEventType = builder.aggregationEventType;
        this.aggregateNotificationEventArn = builder.aggregateNotificationEventArn;
        this.aggregationSummary = builder.aggregationSummary;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.textParts = builder.textParts;
        this.media = builder.media;
        this.organizationalUnitId = builder.organizationalUnitId;
    }

    /**
     * <p>
     * The schema version of the Notification Event.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #schemaVersion}
     * will return {@link SchemaVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #schemaVersionAsString}.
     * </p>
     * 
     * @return The schema version of the Notification Event.
     * @see SchemaVersion
     */
    public final SchemaVersion schemaVersion() {
        return SchemaVersion.fromValue(schemaVersion);
    }

    /**
     * <p>
     * The schema version of the Notification Event.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #schemaVersion}
     * will return {@link SchemaVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #schemaVersionAsString}.
     * </p>
     * 
     * @return The schema version of the Notification Event.
     * @see SchemaVersion
     */
    public final String schemaVersionAsString() {
        return schemaVersion;
    }

    /**
     * <p>
     * The unique identifier for a <code>NotificationEvent</code>.
     * </p>
     * 
     * @return The unique identifier for a <code>NotificationEvent</code>.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The source event metadata.
     * </p>
     * 
     * @return The source event metadata.
     */
    public final SourceEventMetadata sourceEventMetadata() {
        return sourceEventMetadata;
    }

    /**
     * Returns the value of the MessageComponents property for this object.
     * 
     * @return The value of the MessageComponents property for this object.
     */
    public final MessageComponents messageComponents() {
        return messageComponents;
    }

    /**
     * <p>
     * The source event URL.
     * </p>
     * 
     * @return The source event URL.
     */
    public final String sourceEventDetailUrl() {
        return sourceEventDetailUrl;
    }

    /**
     * <p>
     * The detailed URL for the source event.
     * </p>
     * 
     * @return The detailed URL for the source event.
     */
    public final String sourceEventDetailUrlDisplayText() {
        return sourceEventDetailUrlDisplayText;
    }

    /**
     * <p>
     * The type of event causing the notification.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ALERT</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * A notification about an event where something was triggered, initiated, reopened, deployed, or a threshold was
     * breached.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>WARNING</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * A notification about an event where an issue is about to arise. For example, something is approaching a
     * threshold.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>ANNOUNCEMENT</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * A notification about an important event. For example, a step in a workflow or escalation path or that a workflow
     * was updated.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>INFORMATIONAL</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * A notification about informational messages. For example, recommendations, service announcements, or reminders.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #notificationType}
     * will return {@link NotificationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #notificationTypeAsString}.
     * </p>
     * 
     * @return The type of event causing the notification.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ALERT</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A notification about an event where something was triggered, initiated, reopened, deployed, or a
     *         threshold was breached.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>WARNING</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A notification about an event where an issue is about to arise. For example, something is approaching a
     *         threshold.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ANNOUNCEMENT</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A notification about an important event. For example, a step in a workflow or escalation path or that a
     *         workflow was updated.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INFORMATIONAL</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A notification about informational messages. For example, recommendations, service announcements, or
     *         reminders.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     * @see NotificationType
     */
    public final NotificationType notificationType() {
        return NotificationType.fromValue(notificationType);
    }

    /**
     * <p>
     * The type of event causing the notification.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ALERT</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * A notification about an event where something was triggered, initiated, reopened, deployed, or a threshold was
     * breached.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>WARNING</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * A notification about an event where an issue is about to arise. For example, something is approaching a
     * threshold.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>ANNOUNCEMENT</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * A notification about an important event. For example, a step in a workflow or escalation path or that a workflow
     * was updated.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>INFORMATIONAL</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * A notification about informational messages. For example, recommendations, service announcements, or reminders.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #notificationType}
     * will return {@link NotificationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #notificationTypeAsString}.
     * </p>
     * 
     * @return The type of event causing the notification.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ALERT</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A notification about an event where something was triggered, initiated, reopened, deployed, or a
     *         threshold was breached.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>WARNING</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A notification about an event where an issue is about to arise. For example, something is approaching a
     *         threshold.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ANNOUNCEMENT</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A notification about an important event. For example, a step in a workflow or escalation path or that a
     *         workflow was updated.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INFORMATIONAL</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A notification about informational messages. For example, recommendations, service announcements, or
     *         reminders.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     * @see NotificationType
     */
    public final String notificationTypeAsString() {
        return notificationType;
    }

    /**
     * <p>
     * The assessed nature of the event.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>HEALTHY</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * All <code>EventRules</code> are <code>ACTIVE</code> and any call can be run.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>UNHEALTHY</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Some <code>EventRules</code> are <code>ACTIVE</code> and some are <code>INACTIVE</code>. Any call can be run.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventStatus} will
     * return {@link EventStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #eventStatusAsString}.
     * </p>
     * 
     * @return The assessed nature of the event.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>HEALTHY</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         All <code>EventRules</code> are <code>ACTIVE</code> and any call can be run.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UNHEALTHY</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Some <code>EventRules</code> are <code>ACTIVE</code> and some are <code>INACTIVE</code>. Any call can be
     *         run.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     * @see EventStatus
     */
    public final EventStatus eventStatus() {
        return EventStatus.fromValue(eventStatus);
    }

    /**
     * <p>
     * The assessed nature of the event.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>HEALTHY</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * All <code>EventRules</code> are <code>ACTIVE</code> and any call can be run.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>UNHEALTHY</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Some <code>EventRules</code> are <code>ACTIVE</code> and some are <code>INACTIVE</code>. Any call can be run.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventStatus} will
     * return {@link EventStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #eventStatusAsString}.
     * </p>
     * 
     * @return The assessed nature of the event.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>HEALTHY</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         All <code>EventRules</code> are <code>ACTIVE</code> and any call can be run.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UNHEALTHY</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Some <code>EventRules</code> are <code>ACTIVE</code> and some are <code>INACTIVE</code>. Any call can be
     *         run.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     * @see EventStatus
     */
    public final String eventStatusAsString() {
        return eventStatus;
    }

    /**
     * <p>
     * The aggregation type of the <code>NotificationConfiguration</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AGGREGATE</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The notification event is an aggregate notification. Aggregate notifications summarize grouped events over a
     * specified time period.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>CHILD</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Some <code>EventRules</code> are <code>ACTIVE</code> and some are <code>INACTIVE</code>. Any call can be run.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>NONE</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The notification isn't aggregated.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #aggregationEventType} will return {@link AggregationEventType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #aggregationEventTypeAsString}.
     * </p>
     * 
     * @return The aggregation type of the <code>NotificationConfiguration</code>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AGGREGATE</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The notification event is an aggregate notification. Aggregate notifications summarize grouped events
     *         over a specified time period.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CHILD</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Some <code>EventRules</code> are <code>ACTIVE</code> and some are <code>INACTIVE</code>. Any call can be
     *         run.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NONE</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The notification isn't aggregated.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     * @see AggregationEventType
     */
    public final AggregationEventType aggregationEventType() {
        return AggregationEventType.fromValue(aggregationEventType);
    }

    /**
     * <p>
     * The aggregation type of the <code>NotificationConfiguration</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AGGREGATE</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The notification event is an aggregate notification. Aggregate notifications summarize grouped events over a
     * specified time period.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>CHILD</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Some <code>EventRules</code> are <code>ACTIVE</code> and some are <code>INACTIVE</code>. Any call can be run.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>NONE</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The notification isn't aggregated.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #aggregationEventType} will return {@link AggregationEventType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #aggregationEventTypeAsString}.
     * </p>
     * 
     * @return The aggregation type of the <code>NotificationConfiguration</code>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AGGREGATE</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The notification event is an aggregate notification. Aggregate notifications summarize grouped events
     *         over a specified time period.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CHILD</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Some <code>EventRules</code> are <code>ACTIVE</code> and some are <code>INACTIVE</code>. Any call can be
     *         run.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NONE</code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The notification isn't aggregated.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     * @see AggregationEventType
     */
    public final String aggregationEventTypeAsString() {
        return aggregationEventType;
    }

    /**
     * <p>
     * If the value of <code>aggregationEventType</code> is not <code>NONE</code>, this is the Amazon Resource Event
     * (ARN) of the parent aggregate notification.
     * </p>
     * <p>
     * This is omitted if notification isn't aggregated.
     * </p>
     * 
     * @return If the value of <code>aggregationEventType</code> is not <code>NONE</code>, this is the Amazon Resource
     *         Event (ARN) of the parent aggregate notification.</p>
     *         <p>
     *         This is omitted if notification isn't aggregated.
     */
    public final String aggregateNotificationEventArn() {
        return aggregateNotificationEventArn;
    }

    /**
     * <p>
     * Provides additional information about how multiple notifications are grouped.
     * </p>
     * 
     * @return Provides additional information about how multiple notifications are grouped.
     */
    public final AggregationSummary aggregationSummary() {
        return aggregationSummary;
    }

    /**
     * <p>
     * The notification event start time.
     * </p>
     * 
     * @return The notification event start time.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The end time of the event.
     * </p>
     * 
     * @return The end time of the event.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the TextParts property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTextParts() {
        return textParts != null && !(textParts instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A list of text values.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTextParts} method.
     * </p>
     * 
     * @return A list of text values.
     */
    public final Map<String, TextPartValue> textParts() {
        return textParts;
    }

    /**
     * For responses, this returns true if the service returned a value for the Media property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasMedia() {
        return media != null && !(media instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of media elements.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMedia} method.
     * </p>
     * 
     * @return A list of media elements.
     */
    public final List<MediaElement> media() {
        return media;
    }

    /**
     * <p>
     * The unique identifier of the organizational unit associated with the notification event.
     * </p>
     * 
     * @return The unique identifier of the organizational unit associated with the notification event.
     */
    public final String organizationalUnitId() {
        return organizationalUnitId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(schemaVersionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(sourceEventMetadata());
        hashCode = 31 * hashCode + Objects.hashCode(messageComponents());
        hashCode = 31 * hashCode + Objects.hashCode(sourceEventDetailUrl());
        hashCode = 31 * hashCode + Objects.hashCode(sourceEventDetailUrlDisplayText());
        hashCode = 31 * hashCode + Objects.hashCode(notificationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(eventStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(aggregationEventTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(aggregateNotificationEventArn());
        hashCode = 31 * hashCode + Objects.hashCode(aggregationSummary());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasTextParts() ? textParts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasMedia() ? media() : null);
        hashCode = 31 * hashCode + Objects.hashCode(organizationalUnitId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NotificationEvent)) {
            return false;
        }
        NotificationEvent other = (NotificationEvent) obj;
        return Objects.equals(schemaVersionAsString(), other.schemaVersionAsString()) && Objects.equals(id(), other.id())
                && Objects.equals(sourceEventMetadata(), other.sourceEventMetadata())
                && Objects.equals(messageComponents(), other.messageComponents())
                && Objects.equals(sourceEventDetailUrl(), other.sourceEventDetailUrl())
                && Objects.equals(sourceEventDetailUrlDisplayText(), other.sourceEventDetailUrlDisplayText())
                && Objects.equals(notificationTypeAsString(), other.notificationTypeAsString())
                && Objects.equals(eventStatusAsString(), other.eventStatusAsString())
                && Objects.equals(aggregationEventTypeAsString(), other.aggregationEventTypeAsString())
                && Objects.equals(aggregateNotificationEventArn(), other.aggregateNotificationEventArn())
                && Objects.equals(aggregationSummary(), other.aggregationSummary())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(endTime(), other.endTime())
                && hasTextParts() == other.hasTextParts() && Objects.equals(textParts(), other.textParts())
                && hasMedia() == other.hasMedia() && Objects.equals(media(), other.media())
                && Objects.equals(organizationalUnitId(), other.organizationalUnitId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NotificationEvent").add("SchemaVersion", schemaVersionAsString()).add("Id", id())
                .add("SourceEventMetadata", sourceEventMetadata()).add("MessageComponents", messageComponents())
                .add("SourceEventDetailUrl", sourceEventDetailUrl())
                .add("SourceEventDetailUrlDisplayText", sourceEventDetailUrlDisplayText())
                .add("NotificationType", notificationTypeAsString()).add("EventStatus", eventStatusAsString())
                .add("AggregationEventType", aggregationEventTypeAsString())
                .add("AggregateNotificationEventArn", aggregateNotificationEventArn())
                .add("AggregationSummary", aggregationSummary()).add("StartTime", startTime()).add("EndTime", endTime())
                .add("TextParts", hasTextParts() ? textParts() : null).add("Media", hasMedia() ? media() : null)
                .add("OrganizationalUnitId", organizationalUnitId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "schemaVersion":
            return Optional.ofNullable(clazz.cast(schemaVersionAsString()));
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "sourceEventMetadata":
            return Optional.ofNullable(clazz.cast(sourceEventMetadata()));
        case "messageComponents":
            return Optional.ofNullable(clazz.cast(messageComponents()));
        case "sourceEventDetailUrl":
            return Optional.ofNullable(clazz.cast(sourceEventDetailUrl()));
        case "sourceEventDetailUrlDisplayText":
            return Optional.ofNullable(clazz.cast(sourceEventDetailUrlDisplayText()));
        case "notificationType":
            return Optional.ofNullable(clazz.cast(notificationTypeAsString()));
        case "eventStatus":
            return Optional.ofNullable(clazz.cast(eventStatusAsString()));
        case "aggregationEventType":
            return Optional.ofNullable(clazz.cast(aggregationEventTypeAsString()));
        case "aggregateNotificationEventArn":
            return Optional.ofNullable(clazz.cast(aggregateNotificationEventArn()));
        case "aggregationSummary":
            return Optional.ofNullable(clazz.cast(aggregationSummary()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "textParts":
            return Optional.ofNullable(clazz.cast(textParts()));
        case "media":
            return Optional.ofNullable(clazz.cast(media()));
        case "organizationalUnitId":
            return Optional.ofNullable(clazz.cast(organizationalUnitId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("schemaVersion", SCHEMA_VERSION_FIELD);
        map.put("id", ID_FIELD);
        map.put("sourceEventMetadata", SOURCE_EVENT_METADATA_FIELD);
        map.put("messageComponents", MESSAGE_COMPONENTS_FIELD);
        map.put("sourceEventDetailUrl", SOURCE_EVENT_DETAIL_URL_FIELD);
        map.put("sourceEventDetailUrlDisplayText", SOURCE_EVENT_DETAIL_URL_DISPLAY_TEXT_FIELD);
        map.put("notificationType", NOTIFICATION_TYPE_FIELD);
        map.put("eventStatus", EVENT_STATUS_FIELD);
        map.put("aggregationEventType", AGGREGATION_EVENT_TYPE_FIELD);
        map.put("aggregateNotificationEventArn", AGGREGATE_NOTIFICATION_EVENT_ARN_FIELD);
        map.put("aggregationSummary", AGGREGATION_SUMMARY_FIELD);
        map.put("startTime", START_TIME_FIELD);
        map.put("endTime", END_TIME_FIELD);
        map.put("textParts", TEXT_PARTS_FIELD);
        map.put("media", MEDIA_FIELD);
        map.put("organizationalUnitId", ORGANIZATIONAL_UNIT_ID_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<NotificationEvent, T> g) {
        return obj -> g.apply((NotificationEvent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NotificationEvent> {
        /**
         * <p>
         * The schema version of the Notification Event.
         * </p>
         * 
         * @param schemaVersion
         *        The schema version of the Notification Event.
         * @see SchemaVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SchemaVersion
         */
        Builder schemaVersion(String schemaVersion);

        /**
         * <p>
         * The schema version of the Notification Event.
         * </p>
         * 
         * @param schemaVersion
         *        The schema version of the Notification Event.
         * @see SchemaVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SchemaVersion
         */
        Builder schemaVersion(SchemaVersion schemaVersion);

        /**
         * <p>
         * The unique identifier for a <code>NotificationEvent</code>.
         * </p>
         * 
         * @param id
         *        The unique identifier for a <code>NotificationEvent</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The source event metadata.
         * </p>
         * 
         * @param sourceEventMetadata
         *        The source event metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceEventMetadata(SourceEventMetadata sourceEventMetadata);

        /**
         * <p>
         * The source event metadata.
         * </p>
         * This is a convenience method that creates an instance of the {@link SourceEventMetadata.Builder} avoiding the
         * need to create one manually via {@link SourceEventMetadata#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SourceEventMetadata.Builder#build()} is called immediately and
         * its result is passed to {@link #sourceEventMetadata(SourceEventMetadata)}.
         * 
         * @param sourceEventMetadata
         *        a consumer that will call methods on {@link SourceEventMetadata.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceEventMetadata(SourceEventMetadata)
         */
        default Builder sourceEventMetadata(Consumer<SourceEventMetadata.Builder> sourceEventMetadata) {
            return sourceEventMetadata(SourceEventMetadata.builder().applyMutation(sourceEventMetadata).build());
        }

        /**
         * Sets the value of the MessageComponents property for this object.
         *
         * @param messageComponents
         *        The new value for the MessageComponents property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageComponents(MessageComponents messageComponents);

        /**
         * Sets the value of the MessageComponents property for this object.
         *
         * This is a convenience method that creates an instance of the {@link MessageComponents.Builder} avoiding the
         * need to create one manually via {@link MessageComponents#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MessageComponents.Builder#build()} is called immediately and its
         * result is passed to {@link #messageComponents(MessageComponents)}.
         * 
         * @param messageComponents
         *        a consumer that will call methods on {@link MessageComponents.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #messageComponents(MessageComponents)
         */
        default Builder messageComponents(Consumer<MessageComponents.Builder> messageComponents) {
            return messageComponents(MessageComponents.builder().applyMutation(messageComponents).build());
        }

        /**
         * <p>
         * The source event URL.
         * </p>
         * 
         * @param sourceEventDetailUrl
         *        The source event URL.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceEventDetailUrl(String sourceEventDetailUrl);

        /**
         * <p>
         * The detailed URL for the source event.
         * </p>
         * 
         * @param sourceEventDetailUrlDisplayText
         *        The detailed URL for the source event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceEventDetailUrlDisplayText(String sourceEventDetailUrlDisplayText);

        /**
         * <p>
         * The type of event causing the notification.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ALERT</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * A notification about an event where something was triggered, initiated, reopened, deployed, or a threshold
         * was breached.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>WARNING</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * A notification about an event where an issue is about to arise. For example, something is approaching a
         * threshold.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>ANNOUNCEMENT</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * A notification about an important event. For example, a step in a workflow or escalation path or that a
         * workflow was updated.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>INFORMATIONAL</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * A notification about informational messages. For example, recommendations, service announcements, or
         * reminders.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param notificationType
         *        The type of event causing the notification.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ALERT</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A notification about an event where something was triggered, initiated, reopened, deployed, or a
         *        threshold was breached.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>WARNING</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A notification about an event where an issue is about to arise. For example, something is approaching
         *        a threshold.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ANNOUNCEMENT</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A notification about an important event. For example, a step in a workflow or escalation path or that
         *        a workflow was updated.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INFORMATIONAL</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A notification about informational messages. For example, recommendations, service announcements, or
         *        reminders.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        </li>
         * @see NotificationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotificationType
         */
        Builder notificationType(String notificationType);

        /**
         * <p>
         * The type of event causing the notification.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ALERT</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * A notification about an event where something was triggered, initiated, reopened, deployed, or a threshold
         * was breached.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>WARNING</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * A notification about an event where an issue is about to arise. For example, something is approaching a
         * threshold.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>ANNOUNCEMENT</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * A notification about an important event. For example, a step in a workflow or escalation path or that a
         * workflow was updated.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>INFORMATIONAL</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * A notification about informational messages. For example, recommendations, service announcements, or
         * reminders.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param notificationType
         *        The type of event causing the notification.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ALERT</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A notification about an event where something was triggered, initiated, reopened, deployed, or a
         *        threshold was breached.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>WARNING</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A notification about an event where an issue is about to arise. For example, something is approaching
         *        a threshold.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ANNOUNCEMENT</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A notification about an important event. For example, a step in a workflow or escalation path or that
         *        a workflow was updated.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INFORMATIONAL</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A notification about informational messages. For example, recommendations, service announcements, or
         *        reminders.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        </li>
         * @see NotificationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotificationType
         */
        Builder notificationType(NotificationType notificationType);

        /**
         * <p>
         * The assessed nature of the event.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HEALTHY</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * All <code>EventRules</code> are <code>ACTIVE</code> and any call can be run.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>UNHEALTHY</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * Some <code>EventRules</code> are <code>ACTIVE</code> and some are <code>INACTIVE</code>. Any call can be run.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param eventStatus
         *        The assessed nature of the event.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HEALTHY</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        All <code>EventRules</code> are <code>ACTIVE</code> and any call can be run.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UNHEALTHY</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Some <code>EventRules</code> are <code>ACTIVE</code> and some are <code>INACTIVE</code>. Any call can
         *        be run.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        </li>
         * @see EventStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventStatus
         */
        Builder eventStatus(String eventStatus);

        /**
         * <p>
         * The assessed nature of the event.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HEALTHY</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * All <code>EventRules</code> are <code>ACTIVE</code> and any call can be run.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>UNHEALTHY</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * Some <code>EventRules</code> are <code>ACTIVE</code> and some are <code>INACTIVE</code>. Any call can be run.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param eventStatus
         *        The assessed nature of the event.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HEALTHY</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        All <code>EventRules</code> are <code>ACTIVE</code> and any call can be run.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UNHEALTHY</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Some <code>EventRules</code> are <code>ACTIVE</code> and some are <code>INACTIVE</code>. Any call can
         *        be run.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        </li>
         * @see EventStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventStatus
         */
        Builder eventStatus(EventStatus eventStatus);

        /**
         * <p>
         * The aggregation type of the <code>NotificationConfiguration</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AGGREGATE</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * The notification event is an aggregate notification. Aggregate notifications summarize grouped events over a
         * specified time period.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>CHILD</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * Some <code>EventRules</code> are <code>ACTIVE</code> and some are <code>INACTIVE</code>. Any call can be run.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * The notification isn't aggregated.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param aggregationEventType
         *        The aggregation type of the <code>NotificationConfiguration</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AGGREGATE</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The notification event is an aggregate notification. Aggregate notifications summarize grouped events
         *        over a specified time period.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CHILD</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Some <code>EventRules</code> are <code>ACTIVE</code> and some are <code>INACTIVE</code>. Any call can
         *        be run.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NONE</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The notification isn't aggregated.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        </li>
         * @see AggregationEventType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AggregationEventType
         */
        Builder aggregationEventType(String aggregationEventType);

        /**
         * <p>
         * The aggregation type of the <code>NotificationConfiguration</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AGGREGATE</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * The notification event is an aggregate notification. Aggregate notifications summarize grouped events over a
         * specified time period.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>CHILD</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * Some <code>EventRules</code> are <code>ACTIVE</code> and some are <code>INACTIVE</code>. Any call can be run.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * The notification isn't aggregated.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param aggregationEventType
         *        The aggregation type of the <code>NotificationConfiguration</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AGGREGATE</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The notification event is an aggregate notification. Aggregate notifications summarize grouped events
         *        over a specified time period.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CHILD</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Some <code>EventRules</code> are <code>ACTIVE</code> and some are <code>INACTIVE</code>. Any call can
         *        be run.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NONE</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The notification isn't aggregated.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        </li>
         * @see AggregationEventType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AggregationEventType
         */
        Builder aggregationEventType(AggregationEventType aggregationEventType);

        /**
         * <p>
         * If the value of <code>aggregationEventType</code> is not <code>NONE</code>, this is the Amazon Resource Event
         * (ARN) of the parent aggregate notification.
         * </p>
         * <p>
         * This is omitted if notification isn't aggregated.
         * </p>
         * 
         * @param aggregateNotificationEventArn
         *        If the value of <code>aggregationEventType</code> is not <code>NONE</code>, this is the Amazon
         *        Resource Event (ARN) of the parent aggregate notification.</p>
         *        <p>
         *        This is omitted if notification isn't aggregated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aggregateNotificationEventArn(String aggregateNotificationEventArn);

        /**
         * <p>
         * Provides additional information about how multiple notifications are grouped.
         * </p>
         * 
         * @param aggregationSummary
         *        Provides additional information about how multiple notifications are grouped.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aggregationSummary(AggregationSummary aggregationSummary);

        /**
         * <p>
         * Provides additional information about how multiple notifications are grouped.
         * </p>
         * This is a convenience method that creates an instance of the {@link AggregationSummary.Builder} avoiding the
         * need to create one manually via {@link AggregationSummary#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AggregationSummary.Builder#build()} is called immediately and its
         * result is passed to {@link #aggregationSummary(AggregationSummary)}.
         * 
         * @param aggregationSummary
         *        a consumer that will call methods on {@link AggregationSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #aggregationSummary(AggregationSummary)
         */
        default Builder aggregationSummary(Consumer<AggregationSummary.Builder> aggregationSummary) {
            return aggregationSummary(AggregationSummary.builder().applyMutation(aggregationSummary).build());
        }

        /**
         * <p>
         * The notification event start time.
         * </p>
         * 
         * @param startTime
         *        The notification event start time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The end time of the event.
         * </p>
         * 
         * @param endTime
         *        The end time of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * A list of text values.
         * </p>
         * 
         * @param textParts
         *        A list of text values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder textParts(Map<String, TextPartValue> textParts);

        /**
         * <p>
         * A list of media elements.
         * </p>
         * 
         * @param media
         *        A list of media elements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder media(Collection<MediaElement> media);

        /**
         * <p>
         * A list of media elements.
         * </p>
         * 
         * @param media
         *        A list of media elements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder media(MediaElement... media);

        /**
         * <p>
         * A list of media elements.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.notifications.model.MediaElement.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.notifications.model.MediaElement#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.notifications.model.MediaElement.Builder#build()} is called
         * immediately and its result is passed to {@link #media(List<MediaElement>)}.
         * 
         * @param media
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.notifications.model.MediaElement.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #media(java.util.Collection<MediaElement>)
         */
        Builder media(Consumer<MediaElement.Builder>... media);

        /**
         * <p>
         * The unique identifier of the organizational unit associated with the notification event.
         * </p>
         * 
         * @param organizationalUnitId
         *        The unique identifier of the organizational unit associated with the notification event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationalUnitId(String organizationalUnitId);
    }

    static final class BuilderImpl implements Builder {
        private String schemaVersion;

        private String id;

        private SourceEventMetadata sourceEventMetadata;

        private MessageComponents messageComponents;

        private String sourceEventDetailUrl;

        private String sourceEventDetailUrlDisplayText;

        private String notificationType;

        private String eventStatus;

        private String aggregationEventType;

        private String aggregateNotificationEventArn;

        private AggregationSummary aggregationSummary;

        private Instant startTime;

        private Instant endTime;

        private Map<String, TextPartValue> textParts = DefaultSdkAutoConstructMap.getInstance();

        private List<MediaElement> media = DefaultSdkAutoConstructList.getInstance();

        private String organizationalUnitId;

        private BuilderImpl() {
        }

        private BuilderImpl(NotificationEvent model) {
            schemaVersion(model.schemaVersion);
            id(model.id);
            sourceEventMetadata(model.sourceEventMetadata);
            messageComponents(model.messageComponents);
            sourceEventDetailUrl(model.sourceEventDetailUrl);
            sourceEventDetailUrlDisplayText(model.sourceEventDetailUrlDisplayText);
            notificationType(model.notificationType);
            eventStatus(model.eventStatus);
            aggregationEventType(model.aggregationEventType);
            aggregateNotificationEventArn(model.aggregateNotificationEventArn);
            aggregationSummary(model.aggregationSummary);
            startTime(model.startTime);
            endTime(model.endTime);
            textParts(model.textParts);
            media(model.media);
            organizationalUnitId(model.organizationalUnitId);
        }

        public final String getSchemaVersion() {
            return schemaVersion;
        }

        public final void setSchemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
        }

        @Override
        public final Builder schemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
            return this;
        }

        @Override
        public final Builder schemaVersion(SchemaVersion schemaVersion) {
            this.schemaVersion(schemaVersion == null ? null : schemaVersion.toString());
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final SourceEventMetadata.Builder getSourceEventMetadata() {
            return sourceEventMetadata != null ? sourceEventMetadata.toBuilder() : null;
        }

        public final void setSourceEventMetadata(SourceEventMetadata.BuilderImpl sourceEventMetadata) {
            this.sourceEventMetadata = sourceEventMetadata != null ? sourceEventMetadata.build() : null;
        }

        @Override
        public final Builder sourceEventMetadata(SourceEventMetadata sourceEventMetadata) {
            this.sourceEventMetadata = sourceEventMetadata;
            return this;
        }

        public final MessageComponents.Builder getMessageComponents() {
            return messageComponents != null ? messageComponents.toBuilder() : null;
        }

        public final void setMessageComponents(MessageComponents.BuilderImpl messageComponents) {
            this.messageComponents = messageComponents != null ? messageComponents.build() : null;
        }

        @Override
        public final Builder messageComponents(MessageComponents messageComponents) {
            this.messageComponents = messageComponents;
            return this;
        }

        public final String getSourceEventDetailUrl() {
            return sourceEventDetailUrl;
        }

        public final void setSourceEventDetailUrl(String sourceEventDetailUrl) {
            this.sourceEventDetailUrl = sourceEventDetailUrl;
        }

        @Override
        public final Builder sourceEventDetailUrl(String sourceEventDetailUrl) {
            this.sourceEventDetailUrl = sourceEventDetailUrl;
            return this;
        }

        public final String getSourceEventDetailUrlDisplayText() {
            return sourceEventDetailUrlDisplayText;
        }

        public final void setSourceEventDetailUrlDisplayText(String sourceEventDetailUrlDisplayText) {
            this.sourceEventDetailUrlDisplayText = sourceEventDetailUrlDisplayText;
        }

        @Override
        public final Builder sourceEventDetailUrlDisplayText(String sourceEventDetailUrlDisplayText) {
            this.sourceEventDetailUrlDisplayText = sourceEventDetailUrlDisplayText;
            return this;
        }

        public final String getNotificationType() {
            return notificationType;
        }

        public final void setNotificationType(String notificationType) {
            this.notificationType = notificationType;
        }

        @Override
        public final Builder notificationType(String notificationType) {
            this.notificationType = notificationType;
            return this;
        }

        @Override
        public final Builder notificationType(NotificationType notificationType) {
            this.notificationType(notificationType == null ? null : notificationType.toString());
            return this;
        }

        public final String getEventStatus() {
            return eventStatus;
        }

        public final void setEventStatus(String eventStatus) {
            this.eventStatus = eventStatus;
        }

        @Override
        public final Builder eventStatus(String eventStatus) {
            this.eventStatus = eventStatus;
            return this;
        }

        @Override
        public final Builder eventStatus(EventStatus eventStatus) {
            this.eventStatus(eventStatus == null ? null : eventStatus.toString());
            return this;
        }

        public final String getAggregationEventType() {
            return aggregationEventType;
        }

        public final void setAggregationEventType(String aggregationEventType) {
            this.aggregationEventType = aggregationEventType;
        }

        @Override
        public final Builder aggregationEventType(String aggregationEventType) {
            this.aggregationEventType = aggregationEventType;
            return this;
        }

        @Override
        public final Builder aggregationEventType(AggregationEventType aggregationEventType) {
            this.aggregationEventType(aggregationEventType == null ? null : aggregationEventType.toString());
            return this;
        }

        public final String getAggregateNotificationEventArn() {
            return aggregateNotificationEventArn;
        }

        public final void setAggregateNotificationEventArn(String aggregateNotificationEventArn) {
            this.aggregateNotificationEventArn = aggregateNotificationEventArn;
        }

        @Override
        public final Builder aggregateNotificationEventArn(String aggregateNotificationEventArn) {
            this.aggregateNotificationEventArn = aggregateNotificationEventArn;
            return this;
        }

        public final AggregationSummary.Builder getAggregationSummary() {
            return aggregationSummary != null ? aggregationSummary.toBuilder() : null;
        }

        public final void setAggregationSummary(AggregationSummary.BuilderImpl aggregationSummary) {
            this.aggregationSummary = aggregationSummary != null ? aggregationSummary.build() : null;
        }

        @Override
        public final Builder aggregationSummary(AggregationSummary aggregationSummary) {
            this.aggregationSummary = aggregationSummary;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final Map<String, TextPartValue.Builder> getTextParts() {
            Map<String, TextPartValue.Builder> result = TextPartsCopier.copyToBuilder(this.textParts);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setTextParts(Map<String, TextPartValue.BuilderImpl> textParts) {
            this.textParts = TextPartsCopier.copyFromBuilder(textParts);
        }

        @Override
        public final Builder textParts(Map<String, TextPartValue> textParts) {
            this.textParts = TextPartsCopier.copy(textParts);
            return this;
        }

        public final List<MediaElement.Builder> getMedia() {
            List<MediaElement.Builder> result = MediaCopier.copyToBuilder(this.media);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMedia(Collection<MediaElement.BuilderImpl> media) {
            this.media = MediaCopier.copyFromBuilder(media);
        }

        @Override
        public final Builder media(Collection<MediaElement> media) {
            this.media = MediaCopier.copy(media);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder media(MediaElement... media) {
            media(Arrays.asList(media));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder media(Consumer<MediaElement.Builder>... media) {
            media(Stream.of(media).map(c -> MediaElement.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getOrganizationalUnitId() {
            return organizationalUnitId;
        }

        public final void setOrganizationalUnitId(String organizationalUnitId) {
            this.organizationalUnitId = organizationalUnitId;
        }

        @Override
        public final Builder organizationalUnitId(String organizationalUnitId) {
            this.organizationalUnitId = organizationalUnitId;
            return this;
        }

        @Override
        public NotificationEvent build() {
            return new NotificationEvent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
