/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.oam;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.oam.model.ConflictException;
import software.amazon.awssdk.services.oam.model.CreateLinkRequest;
import software.amazon.awssdk.services.oam.model.CreateLinkResponse;
import software.amazon.awssdk.services.oam.model.CreateSinkRequest;
import software.amazon.awssdk.services.oam.model.CreateSinkResponse;
import software.amazon.awssdk.services.oam.model.DeleteLinkRequest;
import software.amazon.awssdk.services.oam.model.DeleteLinkResponse;
import software.amazon.awssdk.services.oam.model.DeleteSinkRequest;
import software.amazon.awssdk.services.oam.model.DeleteSinkResponse;
import software.amazon.awssdk.services.oam.model.GetLinkRequest;
import software.amazon.awssdk.services.oam.model.GetLinkResponse;
import software.amazon.awssdk.services.oam.model.GetSinkPolicyRequest;
import software.amazon.awssdk.services.oam.model.GetSinkPolicyResponse;
import software.amazon.awssdk.services.oam.model.GetSinkRequest;
import software.amazon.awssdk.services.oam.model.GetSinkResponse;
import software.amazon.awssdk.services.oam.model.InternalServiceException;
import software.amazon.awssdk.services.oam.model.InvalidParameterException;
import software.amazon.awssdk.services.oam.model.ListAttachedLinksRequest;
import software.amazon.awssdk.services.oam.model.ListAttachedLinksResponse;
import software.amazon.awssdk.services.oam.model.ListLinksRequest;
import software.amazon.awssdk.services.oam.model.ListLinksResponse;
import software.amazon.awssdk.services.oam.model.ListSinksRequest;
import software.amazon.awssdk.services.oam.model.ListSinksResponse;
import software.amazon.awssdk.services.oam.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.oam.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.oam.model.MissingRequiredParameterException;
import software.amazon.awssdk.services.oam.model.OamException;
import software.amazon.awssdk.services.oam.model.PutSinkPolicyRequest;
import software.amazon.awssdk.services.oam.model.PutSinkPolicyResponse;
import software.amazon.awssdk.services.oam.model.ResourceNotFoundException;
import software.amazon.awssdk.services.oam.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.oam.model.TagResourceRequest;
import software.amazon.awssdk.services.oam.model.TagResourceResponse;
import software.amazon.awssdk.services.oam.model.TooManyTagsException;
import software.amazon.awssdk.services.oam.model.UntagResourceRequest;
import software.amazon.awssdk.services.oam.model.UntagResourceResponse;
import software.amazon.awssdk.services.oam.model.UpdateLinkRequest;
import software.amazon.awssdk.services.oam.model.UpdateLinkResponse;
import software.amazon.awssdk.services.oam.model.ValidationException;
import software.amazon.awssdk.services.oam.paginators.ListAttachedLinksIterable;
import software.amazon.awssdk.services.oam.paginators.ListLinksIterable;
import software.amazon.awssdk.services.oam.paginators.ListSinksIterable;

/**
 * Service client for accessing CloudWatch Observability Access Manager. This can be created using the static
 * {@link #builder()} method.
 *
 * <p>
 * Use Amazon CloudWatch Observability Access Manager to create and manage links between source accounts and monitoring
 * accounts by using <i>CloudWatch cross-account observability</i>. With CloudWatch cross-account observability, you can
 * monitor and troubleshoot applications that span multiple accounts within a Region. Seamlessly search, visualize, and
 * analyze your metrics, logs, and traces in any of the linked accounts without account boundaries.
 * </p>
 * 
 * <pre>
 * <code> &lt;p&gt;Set up one or more Amazon Web Services accounts as &lt;i&gt;monitoring accounts&lt;/i&gt; and link them with multiple &lt;i&gt;source accounts&lt;/i&gt;. A monitoring account is a central Amazon Web Services account that can view and interact with observability data generated from source accounts. A source account is an individual Amazon Web Services account that generates observability data for the resources that reside in it. Source accounts share their observability data with the monitoring account. The shared observability data can include metrics in Amazon CloudWatch, logs in Amazon CloudWatch Logs, and traces in X-Ray.&lt;/p&gt; </code>
 * </pre>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface OamClient extends AwsClient {
    String SERVICE_NAME = "oam";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "oam";

    /**
     * <p>
     * Creates a link between a source account and a sink that you have created in a monitoring account.
     * </p>
     * <p>
     * Before you create a link, you must create a sink in the monitoring account and create a sink policy in that
     * account. The sink policy must permit the source account to link to it. You can grant permission to source
     * accounts by granting permission to an entire organization or to individual accounts.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_CreateSink.html">CreateSink</a> and <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_PutSinkPolicy.html">PutSinkPolicy</a>.
     * </p>
     * <p>
     * Each monitoring account can be linked to as many as 100,000 source accounts.
     * </p>
     * <p>
     * Each source account can be linked to as many as five monitoring accounts.
     * </p>
     *
     * @param createLinkRequest
     * @return Result of the CreateLink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.CreateLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/CreateLink" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateLinkResponse createLink(CreateLinkRequest createLinkRequest) throws InternalServiceException,
            ConflictException, MissingRequiredParameterException, ServiceQuotaExceededException, InvalidParameterException,
            AwsServiceException, SdkClientException, OamException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a link between a source account and a sink that you have created in a monitoring account.
     * </p>
     * <p>
     * Before you create a link, you must create a sink in the monitoring account and create a sink policy in that
     * account. The sink policy must permit the source account to link to it. You can grant permission to source
     * accounts by granting permission to an entire organization or to individual accounts.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_CreateSink.html">CreateSink</a> and <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_PutSinkPolicy.html">PutSinkPolicy</a>.
     * </p>
     * <p>
     * Each monitoring account can be linked to as many as 100,000 source accounts.
     * </p>
     * <p>
     * Each source account can be linked to as many as five monitoring accounts.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateLinkRequest.Builder} avoiding the need to
     * create one manually via {@link CreateLinkRequest#builder()}
     * </p>
     *
     * @param createLinkRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.CreateLinkRequest.Builder} to create a request.
     * @return Result of the CreateLink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.CreateLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/CreateLink" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateLinkResponse createLink(Consumer<CreateLinkRequest.Builder> createLinkRequest) throws InternalServiceException,
            ConflictException, MissingRequiredParameterException, ServiceQuotaExceededException, InvalidParameterException,
            AwsServiceException, SdkClientException, OamException {
        return createLink(CreateLinkRequest.builder().applyMutation(createLinkRequest).build());
    }

    /**
     * <p>
     * Use this to create a <i>sink</i> in the current account, so that it can be used as a monitoring account in
     * CloudWatch cross-account observability. A sink is a resource that represents an attachment point in a monitoring
     * account. Source accounts can link to the sink to send observability data.
     * </p>
     * <p>
     * After you create a sink, you must create a sink policy that allows source accounts to attach to it. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_PutSinkPolicy.html">PutSinkPolicy</a>.
     * </p>
     * <p>
     * Each account can contain one sink. If you delete a sink, you can then create a new one in that account.
     * </p>
     *
     * @param createSinkRequest
     * @return Result of the CreateSink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.CreateSink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/CreateSink" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateSinkResponse createSink(CreateSinkRequest createSinkRequest) throws InternalServiceException,
            ConflictException, MissingRequiredParameterException, ServiceQuotaExceededException, InvalidParameterException,
            AwsServiceException, SdkClientException, OamException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Use this to create a <i>sink</i> in the current account, so that it can be used as a monitoring account in
     * CloudWatch cross-account observability. A sink is a resource that represents an attachment point in a monitoring
     * account. Source accounts can link to the sink to send observability data.
     * </p>
     * <p>
     * After you create a sink, you must create a sink policy that allows source accounts to attach to it. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_PutSinkPolicy.html">PutSinkPolicy</a>.
     * </p>
     * <p>
     * Each account can contain one sink. If you delete a sink, you can then create a new one in that account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateSinkRequest.Builder} avoiding the need to
     * create one manually via {@link CreateSinkRequest#builder()}
     * </p>
     *
     * @param createSinkRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.CreateSinkRequest.Builder} to create a request.
     * @return Result of the CreateSink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.CreateSink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/CreateSink" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateSinkResponse createSink(Consumer<CreateSinkRequest.Builder> createSinkRequest) throws InternalServiceException,
            ConflictException, MissingRequiredParameterException, ServiceQuotaExceededException, InvalidParameterException,
            AwsServiceException, SdkClientException, OamException {
        return createSink(CreateSinkRequest.builder().applyMutation(createSinkRequest).build());
    }

    /**
     * <p>
     * Deletes a link between a monitoring account sink and a source account. You must run this operation in the source
     * account.
     * </p>
     *
     * @param deleteLinkRequest
     * @return Result of the DeleteLink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.DeleteLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/DeleteLink" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteLinkResponse deleteLink(DeleteLinkRequest deleteLinkRequest) throws InternalServiceException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a link between a monitoring account sink and a source account. You must run this operation in the source
     * account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteLinkRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteLinkRequest#builder()}
     * </p>
     *
     * @param deleteLinkRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.DeleteLinkRequest.Builder} to create a request.
     * @return Result of the DeleteLink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.DeleteLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/DeleteLink" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteLinkResponse deleteLink(Consumer<DeleteLinkRequest.Builder> deleteLinkRequest) throws InternalServiceException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        return deleteLink(DeleteLinkRequest.builder().applyMutation(deleteLinkRequest).build());
    }

    /**
     * <p>
     * Deletes a sink. You must delete all links to a sink before you can delete that sink.
     * </p>
     *
     * @param deleteSinkRequest
     * @return Result of the DeleteSink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.DeleteSink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/DeleteSink" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteSinkResponse deleteSink(DeleteSinkRequest deleteSinkRequest) throws InternalServiceException,
            ConflictException, MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, OamException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a sink. You must delete all links to a sink before you can delete that sink.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteSinkRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteSinkRequest#builder()}
     * </p>
     *
     * @param deleteSinkRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.DeleteSinkRequest.Builder} to create a request.
     * @return Result of the DeleteSink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.DeleteSink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/DeleteSink" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteSinkResponse deleteSink(Consumer<DeleteSinkRequest.Builder> deleteSinkRequest) throws InternalServiceException,
            ConflictException, MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, OamException {
        return deleteSink(DeleteSinkRequest.builder().applyMutation(deleteSinkRequest).build());
    }

    /**
     * <p>
     * Returns complete information about one link.
     * </p>
     * <p>
     * To use this operation, provide the link ARN. To retrieve a list of link ARNs, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListLinks.html">ListLinks</a>.
     * </p>
     *
     * @param getLinkRequest
     * @return Result of the GetLink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.GetLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/GetLink" target="_top">AWS API
     *      Documentation</a>
     */
    default GetLinkResponse getLink(GetLinkRequest getLinkRequest) throws InternalServiceException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns complete information about one link.
     * </p>
     * <p>
     * To use this operation, provide the link ARN. To retrieve a list of link ARNs, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListLinks.html">ListLinks</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetLinkRequest.Builder} avoiding the need to create
     * one manually via {@link GetLinkRequest#builder()}
     * </p>
     *
     * @param getLinkRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.GetLinkRequest.Builder} to create a request.
     * @return Result of the GetLink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.GetLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/GetLink" target="_top">AWS API
     *      Documentation</a>
     */
    default GetLinkResponse getLink(Consumer<GetLinkRequest.Builder> getLinkRequest) throws InternalServiceException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        return getLink(GetLinkRequest.builder().applyMutation(getLinkRequest).build());
    }

    /**
     * <p>
     * Returns complete information about one monitoring account sink.
     * </p>
     * <p>
     * To use this operation, provide the sink ARN. To retrieve a list of sink ARNs, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListSinks.html">ListSinks</a>.
     * </p>
     *
     * @param getSinkRequest
     * @return Result of the GetSink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.GetSink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/GetSink" target="_top">AWS API
     *      Documentation</a>
     */
    default GetSinkResponse getSink(GetSinkRequest getSinkRequest) throws InternalServiceException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns complete information about one monitoring account sink.
     * </p>
     * <p>
     * To use this operation, provide the sink ARN. To retrieve a list of sink ARNs, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListSinks.html">ListSinks</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetSinkRequest.Builder} avoiding the need to create
     * one manually via {@link GetSinkRequest#builder()}
     * </p>
     *
     * @param getSinkRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.GetSinkRequest.Builder} to create a request.
     * @return Result of the GetSink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.GetSink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/GetSink" target="_top">AWS API
     *      Documentation</a>
     */
    default GetSinkResponse getSink(Consumer<GetSinkRequest.Builder> getSinkRequest) throws InternalServiceException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        return getSink(GetSinkRequest.builder().applyMutation(getSinkRequest).build());
    }

    /**
     * <p>
     * Returns the current sink policy attached to this sink. The sink policy specifies what accounts can attach to this
     * sink as source accounts, and what types of data they can share.
     * </p>
     *
     * @param getSinkPolicyRequest
     * @return Result of the GetSinkPolicy operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.GetSinkPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/GetSinkPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    default GetSinkPolicyResponse getSinkPolicy(GetSinkPolicyRequest getSinkPolicyRequest) throws InternalServiceException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the current sink policy attached to this sink. The sink policy specifies what accounts can attach to this
     * sink as source accounts, and what types of data they can share.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetSinkPolicyRequest.Builder} avoiding the need to
     * create one manually via {@link GetSinkPolicyRequest#builder()}
     * </p>
     *
     * @param getSinkPolicyRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.GetSinkPolicyRequest.Builder} to create a request.
     * @return Result of the GetSinkPolicy operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.GetSinkPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/GetSinkPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    default GetSinkPolicyResponse getSinkPolicy(Consumer<GetSinkPolicyRequest.Builder> getSinkPolicyRequest)
            throws InternalServiceException, MissingRequiredParameterException, InvalidParameterException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        return getSinkPolicy(GetSinkPolicyRequest.builder().applyMutation(getSinkPolicyRequest).build());
    }

    /**
     * <p>
     * Returns a list of source account links that are linked to this monitoring account sink.
     * </p>
     * <p>
     * To use this operation, provide the sink ARN. To retrieve a list of sink ARNs, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListSinks.html">ListSinks</a>.
     * </p>
     * <p>
     * To find a list of links for one source account, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListLinks.html">ListLinks</a>.
     * </p>
     *
     * @param listAttachedLinksRequest
     * @return Result of the ListAttachedLinks operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListAttachedLinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListAttachedLinks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListAttachedLinksResponse listAttachedLinks(ListAttachedLinksRequest listAttachedLinksRequest)
            throws InternalServiceException, MissingRequiredParameterException, InvalidParameterException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of source account links that are linked to this monitoring account sink.
     * </p>
     * <p>
     * To use this operation, provide the sink ARN. To retrieve a list of sink ARNs, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListSinks.html">ListSinks</a>.
     * </p>
     * <p>
     * To find a list of links for one source account, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListLinks.html">ListLinks</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListAttachedLinksRequest.Builder} avoiding the need
     * to create one manually via {@link ListAttachedLinksRequest#builder()}
     * </p>
     *
     * @param listAttachedLinksRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.ListAttachedLinksRequest.Builder} to create a request.
     * @return Result of the ListAttachedLinks operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListAttachedLinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListAttachedLinks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListAttachedLinksResponse listAttachedLinks(Consumer<ListAttachedLinksRequest.Builder> listAttachedLinksRequest)
            throws InternalServiceException, MissingRequiredParameterException, InvalidParameterException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        return listAttachedLinks(ListAttachedLinksRequest.builder().applyMutation(listAttachedLinksRequest).build());
    }

    /**
     * <p>
     * Returns a list of source account links that are linked to this monitoring account sink.
     * </p>
     * <p>
     * To use this operation, provide the sink ARN. To retrieve a list of sink ARNs, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListSinks.html">ListSinks</a>.
     * </p>
     * <p>
     * To find a list of links for one source account, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListLinks.html">ListLinks</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAttachedLinks(software.amazon.awssdk.services.oam.model.ListAttachedLinksRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListAttachedLinksIterable responses = client.listAttachedLinksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.oam.paginators.ListAttachedLinksIterable responses = client
     *             .listAttachedLinksPaginator(request);
     *     for (software.amazon.awssdk.services.oam.model.ListAttachedLinksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListAttachedLinksIterable responses = client.listAttachedLinksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAttachedLinks(software.amazon.awssdk.services.oam.model.ListAttachedLinksRequest)} operation.</b>
     * </p>
     *
     * @param listAttachedLinksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListAttachedLinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListAttachedLinks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListAttachedLinksIterable listAttachedLinksPaginator(ListAttachedLinksRequest listAttachedLinksRequest)
            throws InternalServiceException, MissingRequiredParameterException, InvalidParameterException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        return new ListAttachedLinksIterable(this, listAttachedLinksRequest);
    }

    /**
     * <p>
     * Returns a list of source account links that are linked to this monitoring account sink.
     * </p>
     * <p>
     * To use this operation, provide the sink ARN. To retrieve a list of sink ARNs, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListSinks.html">ListSinks</a>.
     * </p>
     * <p>
     * To find a list of links for one source account, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListLinks.html">ListLinks</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAttachedLinks(software.amazon.awssdk.services.oam.model.ListAttachedLinksRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListAttachedLinksIterable responses = client.listAttachedLinksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.oam.paginators.ListAttachedLinksIterable responses = client
     *             .listAttachedLinksPaginator(request);
     *     for (software.amazon.awssdk.services.oam.model.ListAttachedLinksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListAttachedLinksIterable responses = client.listAttachedLinksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAttachedLinks(software.amazon.awssdk.services.oam.model.ListAttachedLinksRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListAttachedLinksRequest.Builder} avoiding the need
     * to create one manually via {@link ListAttachedLinksRequest#builder()}
     * </p>
     *
     * @param listAttachedLinksRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.ListAttachedLinksRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListAttachedLinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListAttachedLinks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListAttachedLinksIterable listAttachedLinksPaginator(
            Consumer<ListAttachedLinksRequest.Builder> listAttachedLinksRequest) throws InternalServiceException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        return listAttachedLinksPaginator(ListAttachedLinksRequest.builder().applyMutation(listAttachedLinksRequest).build());
    }

    /**
     * <p>
     * Use this operation in a source account to return a list of links to monitoring account sinks that this source
     * account has.
     * </p>
     * <p>
     * To find a list of links for one monitoring account sink, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListAttachedLinks.html">ListAttachedLinks</a> from
     * within the monitoring account.
     * </p>
     *
     * @param listLinksRequest
     * @return Result of the ListLinks operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListLinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListLinks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListLinksResponse listLinks(ListLinksRequest listLinksRequest) throws InternalServiceException,
            InvalidParameterException, ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Use this operation in a source account to return a list of links to monitoring account sinks that this source
     * account has.
     * </p>
     * <p>
     * To find a list of links for one monitoring account sink, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListAttachedLinks.html">ListAttachedLinks</a> from
     * within the monitoring account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListLinksRequest.Builder} avoiding the need to
     * create one manually via {@link ListLinksRequest#builder()}
     * </p>
     *
     * @param listLinksRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.ListLinksRequest.Builder} to create a request.
     * @return Result of the ListLinks operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListLinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListLinks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListLinksResponse listLinks(Consumer<ListLinksRequest.Builder> listLinksRequest) throws InternalServiceException,
            InvalidParameterException, ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        return listLinks(ListLinksRequest.builder().applyMutation(listLinksRequest).build());
    }

    /**
     * <p>
     * Use this operation in a source account to return a list of links to monitoring account sinks that this source
     * account has.
     * </p>
     * <p>
     * To find a list of links for one monitoring account sink, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListAttachedLinks.html">ListAttachedLinks</a> from
     * within the monitoring account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listLinks(software.amazon.awssdk.services.oam.model.ListLinksRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListLinksIterable responses = client.listLinksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.oam.paginators.ListLinksIterable responses = client.listLinksPaginator(request);
     *     for (software.amazon.awssdk.services.oam.model.ListLinksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListLinksIterable responses = client.listLinksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLinks(software.amazon.awssdk.services.oam.model.ListLinksRequest)} operation.</b>
     * </p>
     *
     * @param listLinksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListLinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListLinks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListLinksIterable listLinksPaginator(ListLinksRequest listLinksRequest) throws InternalServiceException,
            InvalidParameterException, ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        return new ListLinksIterable(this, listLinksRequest);
    }

    /**
     * <p>
     * Use this operation in a source account to return a list of links to monitoring account sinks that this source
     * account has.
     * </p>
     * <p>
     * To find a list of links for one monitoring account sink, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListAttachedLinks.html">ListAttachedLinks</a> from
     * within the monitoring account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listLinks(software.amazon.awssdk.services.oam.model.ListLinksRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListLinksIterable responses = client.listLinksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.oam.paginators.ListLinksIterable responses = client.listLinksPaginator(request);
     *     for (software.amazon.awssdk.services.oam.model.ListLinksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListLinksIterable responses = client.listLinksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLinks(software.amazon.awssdk.services.oam.model.ListLinksRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListLinksRequest.Builder} avoiding the need to
     * create one manually via {@link ListLinksRequest#builder()}
     * </p>
     *
     * @param listLinksRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.ListLinksRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListLinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListLinks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListLinksIterable listLinksPaginator(Consumer<ListLinksRequest.Builder> listLinksRequest)
            throws InternalServiceException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        return listLinksPaginator(ListLinksRequest.builder().applyMutation(listLinksRequest).build());
    }

    /**
     * <p>
     * Use this operation in a monitoring account to return the list of sinks created in that account.
     * </p>
     *
     * @param listSinksRequest
     * @return Result of the ListSinks operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListSinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListSinks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSinksResponse listSinks(ListSinksRequest listSinksRequest) throws InternalServiceException,
            InvalidParameterException, ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Use this operation in a monitoring account to return the list of sinks created in that account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSinksRequest.Builder} avoiding the need to
     * create one manually via {@link ListSinksRequest#builder()}
     * </p>
     *
     * @param listSinksRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.ListSinksRequest.Builder} to create a request.
     * @return Result of the ListSinks operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListSinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListSinks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSinksResponse listSinks(Consumer<ListSinksRequest.Builder> listSinksRequest) throws InternalServiceException,
            InvalidParameterException, ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        return listSinks(ListSinksRequest.builder().applyMutation(listSinksRequest).build());
    }

    /**
     * <p>
     * Use this operation in a monitoring account to return the list of sinks created in that account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listSinks(software.amazon.awssdk.services.oam.model.ListSinksRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListSinksIterable responses = client.listSinksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.oam.paginators.ListSinksIterable responses = client.listSinksPaginator(request);
     *     for (software.amazon.awssdk.services.oam.model.ListSinksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListSinksIterable responses = client.listSinksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSinks(software.amazon.awssdk.services.oam.model.ListSinksRequest)} operation.</b>
     * </p>
     *
     * @param listSinksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListSinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListSinks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSinksIterable listSinksPaginator(ListSinksRequest listSinksRequest) throws InternalServiceException,
            InvalidParameterException, ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        return new ListSinksIterable(this, listSinksRequest);
    }

    /**
     * <p>
     * Use this operation in a monitoring account to return the list of sinks created in that account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listSinks(software.amazon.awssdk.services.oam.model.ListSinksRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListSinksIterable responses = client.listSinksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.oam.paginators.ListSinksIterable responses = client.listSinksPaginator(request);
     *     for (software.amazon.awssdk.services.oam.model.ListSinksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListSinksIterable responses = client.listSinksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSinks(software.amazon.awssdk.services.oam.model.ListSinksRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSinksRequest.Builder} avoiding the need to
     * create one manually via {@link ListSinksRequest#builder()}
     * </p>
     *
     * @param listSinksRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.ListSinksRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListSinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListSinks" target="_top">AWS API
     *      Documentation</a>
     */
    default ListSinksIterable listSinksPaginator(Consumer<ListSinksRequest.Builder> listSinksRequest)
            throws InternalServiceException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        return listSinksPaginator(ListSinksRequest.builder().applyMutation(listSinksRequest).build());
    }

    /**
     * <p>
     * Displays the tags associated with a resource. Both sinks and links support tagging.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Displays the tags associated with a resource. Both sinks and links support tagging.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.ListTagsForResourceRequest.Builder} to create a request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Creates or updates the resource policy that grants permissions to source accounts to link to the monitoring
     * account sink. When you create a sink policy, you can grant permissions to all accounts in an organization or to
     * individual accounts.
     * </p>
     * <p>
     * You can also use a sink policy to limit the types of data that is shared. The three types that you can allow or
     * deny are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Metrics</b> - Specify with <code>AWS::CloudWatch::Metric</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Log groups</b> - Specify with <code>AWS::Logs::LogGroup</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Traces</b> - Specify with <code>AWS::XRay::Trace</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * See the examples in this section to see how to specify permitted source accounts and data types.
     * </p>
     *
     * @param putSinkPolicyRequest
     * @return Result of the PutSinkPolicy operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.PutSinkPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/PutSinkPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    default PutSinkPolicyResponse putSinkPolicy(PutSinkPolicyRequest putSinkPolicyRequest) throws InternalServiceException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates or updates the resource policy that grants permissions to source accounts to link to the monitoring
     * account sink. When you create a sink policy, you can grant permissions to all accounts in an organization or to
     * individual accounts.
     * </p>
     * <p>
     * You can also use a sink policy to limit the types of data that is shared. The three types that you can allow or
     * deny are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Metrics</b> - Specify with <code>AWS::CloudWatch::Metric</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Log groups</b> - Specify with <code>AWS::Logs::LogGroup</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Traces</b> - Specify with <code>AWS::XRay::Trace</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * See the examples in this section to see how to specify permitted source accounts and data types.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutSinkPolicyRequest.Builder} avoiding the need to
     * create one manually via {@link PutSinkPolicyRequest#builder()}
     * </p>
     *
     * @param putSinkPolicyRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.PutSinkPolicyRequest.Builder} to create a request.
     * @return Result of the PutSinkPolicy operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.PutSinkPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/PutSinkPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    default PutSinkPolicyResponse putSinkPolicy(Consumer<PutSinkPolicyRequest.Builder> putSinkPolicyRequest)
            throws InternalServiceException, MissingRequiredParameterException, InvalidParameterException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        return putSinkPolicy(PutSinkPolicyRequest.builder().applyMutation(putSinkPolicyRequest).build());
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified resource. Both sinks and links can be tagged.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a resource that already has tags. If you specify a new tag
     * key for the alarm, this tag is appended to the list of tags associated with the alarm. If you specify a tag key
     * that is already associated with the alarm, the new tag value that you specify replaces the previous value for
     * that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a resource.
     * </p>
     * <important>
     * <p>
     * Unlike tagging permissions in other Amazon Web Services services, to tag or untag links and sinks you must have
     * the <code>oam:ResourceTag</code> permission. The <code>iam:ResourceTag</code> permission does not allow you to
     * tag and untag links and sinks.
     * </p>
     * </important>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws TooManyTagsException
     *         A resource can have no more than 50 tags.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            TooManyTagsException, ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified resource. Both sinks and links can be tagged.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a resource that already has tags. If you specify a new tag
     * key for the alarm, this tag is appended to the list of tags associated with the alarm. If you specify a tag key
     * that is already associated with the alarm, the new tag value that you specify replaces the previous value for
     * that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a resource.
     * </p>
     * <important>
     * <p>
     * Unlike tagging permissions in other Amazon Web Services services, to tag or untag links and sinks you must have
     * the <code>oam:ResourceTag</code> permission. The <code>iam:ResourceTag</code> permission does not allow you to
     * tag and untag links and sinks.
     * </p>
     * </important><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.TagResourceRequest.Builder} to create a request.
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws TooManyTagsException
     *         A resource can have no more than 50 tags.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) throws ValidationException,
            TooManyTagsException, ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     * <important>
     * <p>
     * Unlike tagging permissions in other Amazon Web Services services, to tag or untag links and sinks you must have
     * the <code>oam:ResourceTag</code> permission. The <code>iam:TagResource</code> permission does not allow you to
     * tag and untag links and sinks.
     * </p>
     * </important>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     * <important>
     * <p>
     * Unlike tagging permissions in other Amazon Web Services services, to tag or untag links and sinks you must have
     * the <code>oam:ResourceTag</code> permission. The <code>iam:TagResource</code> permission does not allow you to
     * tag and untag links and sinks.
     * </p>
     * </important><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.UntagResourceRequest.Builder} to create a request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Use this operation to change what types of data are shared from a source account to its linked monitoring account
     * sink. You can't change the sink or change the monitoring account with this operation.
     * </p>
     * <p>
     * To update the list of tags associated with the sink, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     *
     * @param updateLinkRequest
     * @return Result of the UpdateLink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.UpdateLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/UpdateLink" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateLinkResponse updateLink(UpdateLinkRequest updateLinkRequest) throws InternalServiceException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Use this operation to change what types of data are shared from a source account to its linked monitoring account
     * sink. You can't change the sink or change the monitoring account with this operation.
     * </p>
     * <p>
     * To update the list of tags associated with the sink, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateLinkRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateLinkRequest#builder()}
     * </p>
     *
     * @param updateLinkRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.oam.model.UpdateLinkRequest.Builder} to create a request.
     * @return Result of the UpdateLink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.UpdateLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/UpdateLink" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateLinkResponse updateLink(Consumer<UpdateLinkRequest.Builder> updateLinkRequest) throws InternalServiceException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        return updateLink(UpdateLinkRequest.builder().applyMutation(updateLinkRequest).build());
    }

    /**
     * Create a {@link OamClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static OamClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link OamClient}.
     */
    static OamClientBuilder builder() {
        return new DefaultOamClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default OamServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
