/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.oam;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.oam.model.ConflictException;
import software.amazon.awssdk.services.oam.model.CreateLinkRequest;
import software.amazon.awssdk.services.oam.model.CreateLinkResponse;
import software.amazon.awssdk.services.oam.model.CreateSinkRequest;
import software.amazon.awssdk.services.oam.model.CreateSinkResponse;
import software.amazon.awssdk.services.oam.model.DeleteLinkRequest;
import software.amazon.awssdk.services.oam.model.DeleteLinkResponse;
import software.amazon.awssdk.services.oam.model.DeleteSinkRequest;
import software.amazon.awssdk.services.oam.model.DeleteSinkResponse;
import software.amazon.awssdk.services.oam.model.GetLinkRequest;
import software.amazon.awssdk.services.oam.model.GetLinkResponse;
import software.amazon.awssdk.services.oam.model.GetSinkPolicyRequest;
import software.amazon.awssdk.services.oam.model.GetSinkPolicyResponse;
import software.amazon.awssdk.services.oam.model.GetSinkRequest;
import software.amazon.awssdk.services.oam.model.GetSinkResponse;
import software.amazon.awssdk.services.oam.model.InternalServiceException;
import software.amazon.awssdk.services.oam.model.InvalidParameterException;
import software.amazon.awssdk.services.oam.model.ListAttachedLinksRequest;
import software.amazon.awssdk.services.oam.model.ListAttachedLinksResponse;
import software.amazon.awssdk.services.oam.model.ListLinksRequest;
import software.amazon.awssdk.services.oam.model.ListLinksResponse;
import software.amazon.awssdk.services.oam.model.ListSinksRequest;
import software.amazon.awssdk.services.oam.model.ListSinksResponse;
import software.amazon.awssdk.services.oam.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.oam.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.oam.model.MissingRequiredParameterException;
import software.amazon.awssdk.services.oam.model.OamException;
import software.amazon.awssdk.services.oam.model.OamRequest;
import software.amazon.awssdk.services.oam.model.PutSinkPolicyRequest;
import software.amazon.awssdk.services.oam.model.PutSinkPolicyResponse;
import software.amazon.awssdk.services.oam.model.ResourceNotFoundException;
import software.amazon.awssdk.services.oam.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.oam.model.TagResourceRequest;
import software.amazon.awssdk.services.oam.model.TagResourceResponse;
import software.amazon.awssdk.services.oam.model.TooManyTagsException;
import software.amazon.awssdk.services.oam.model.UntagResourceRequest;
import software.amazon.awssdk.services.oam.model.UntagResourceResponse;
import software.amazon.awssdk.services.oam.model.UpdateLinkRequest;
import software.amazon.awssdk.services.oam.model.UpdateLinkResponse;
import software.amazon.awssdk.services.oam.model.ValidationException;
import software.amazon.awssdk.services.oam.paginators.ListAttachedLinksIterable;
import software.amazon.awssdk.services.oam.paginators.ListLinksIterable;
import software.amazon.awssdk.services.oam.paginators.ListSinksIterable;
import software.amazon.awssdk.services.oam.transform.CreateLinkRequestMarshaller;
import software.amazon.awssdk.services.oam.transform.CreateSinkRequestMarshaller;
import software.amazon.awssdk.services.oam.transform.DeleteLinkRequestMarshaller;
import software.amazon.awssdk.services.oam.transform.DeleteSinkRequestMarshaller;
import software.amazon.awssdk.services.oam.transform.GetLinkRequestMarshaller;
import software.amazon.awssdk.services.oam.transform.GetSinkPolicyRequestMarshaller;
import software.amazon.awssdk.services.oam.transform.GetSinkRequestMarshaller;
import software.amazon.awssdk.services.oam.transform.ListAttachedLinksRequestMarshaller;
import software.amazon.awssdk.services.oam.transform.ListLinksRequestMarshaller;
import software.amazon.awssdk.services.oam.transform.ListSinksRequestMarshaller;
import software.amazon.awssdk.services.oam.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.oam.transform.PutSinkPolicyRequestMarshaller;
import software.amazon.awssdk.services.oam.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.oam.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.oam.transform.UpdateLinkRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link OamClient}.
 *
 * @see OamClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultOamClient implements OamClient {
    private static final Logger log = Logger.loggerFor(DefaultOamClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultOamClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a link between a source account and a sink that you have created in a monitoring account.
     * </p>
     * <p>
     * Before you create a link, you must create a sink in the monitoring account and create a sink policy in that
     * account. The sink policy must permit the source account to link to it. You can grant permission to source
     * accounts by granting permission to an entire organization or to individual accounts.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_CreateSink.html">CreateSink</a> and <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_PutSinkPolicy.html">PutSinkPolicy</a>.
     * </p>
     * <p>
     * Each monitoring account can be linked to as many as 100,000 source accounts.
     * </p>
     * <p>
     * Each source account can be linked to as many as five monitoring accounts.
     * </p>
     *
     * @param createLinkRequest
     * @return Result of the CreateLink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.CreateLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/CreateLink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateLinkResponse createLink(CreateLinkRequest createLinkRequest) throws InternalServiceException, ConflictException,
            MissingRequiredParameterException, ServiceQuotaExceededException, InvalidParameterException, AwsServiceException,
            SdkClientException, OamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateLinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateLinkResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLink");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateLinkRequest, CreateLinkResponse>().withOperationName("CreateLink")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createLinkRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use this to create a <i>sink</i> in the current account, so that it can be used as a monitoring account in
     * CloudWatch cross-account observability. A sink is a resource that represents an attachment point in a monitoring
     * account. Source accounts can link to the sink to send observability data.
     * </p>
     * <p>
     * After you create a sink, you must create a sink policy that allows source accounts to attach to it. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_PutSinkPolicy.html">PutSinkPolicy</a>.
     * </p>
     * <p>
     * Each account can contain one sink. If you delete a sink, you can then create a new one in that account.
     * </p>
     *
     * @param createSinkRequest
     * @return Result of the CreateSink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.CreateSink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/CreateSink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateSinkResponse createSink(CreateSinkRequest createSinkRequest) throws InternalServiceException, ConflictException,
            MissingRequiredParameterException, ServiceQuotaExceededException, InvalidParameterException, AwsServiceException,
            SdkClientException, OamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateSinkResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSink");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateSinkRequest, CreateSinkResponse>().withOperationName("CreateSink")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createSinkRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateSinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a link between a monitoring account sink and a source account. You must run this operation in the source
     * account.
     * </p>
     *
     * @param deleteLinkRequest
     * @return Result of the DeleteLink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.DeleteLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/DeleteLink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteLinkResponse deleteLink(DeleteLinkRequest deleteLinkRequest) throws InternalServiceException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteLinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteLinkResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLink");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteLinkRequest, DeleteLinkResponse>().withOperationName("DeleteLink")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteLinkRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a sink. You must delete all links to a sink before you can delete that sink.
     * </p>
     *
     * @param deleteSinkRequest
     * @return Result of the DeleteSink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.DeleteSink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/DeleteSink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteSinkResponse deleteSink(DeleteSinkRequest deleteSinkRequest) throws InternalServiceException, ConflictException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteSinkResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSink");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteSinkRequest, DeleteSinkResponse>().withOperationName("DeleteSink")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteSinkRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteSinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns complete information about one link.
     * </p>
     * <p>
     * To use this operation, provide the link ARN. To retrieve a list of link ARNs, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListLinks.html">ListLinks</a>.
     * </p>
     *
     * @param getLinkRequest
     * @return Result of the GetLink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.GetLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/GetLink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetLinkResponse getLink(GetLinkRequest getLinkRequest) throws InternalServiceException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetLinkResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLink");

            return clientHandler.execute(new ClientExecutionParams<GetLinkRequest, GetLinkResponse>()
                    .withOperationName("GetLink").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getLinkRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns complete information about one monitoring account sink.
     * </p>
     * <p>
     * To use this operation, provide the sink ARN. To retrieve a list of sink ARNs, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListSinks.html">ListSinks</a>.
     * </p>
     *
     * @param getSinkRequest
     * @return Result of the GetSink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.GetSink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/GetSink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSinkResponse getSink(GetSinkRequest getSinkRequest) throws InternalServiceException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSinkResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSink");

            return clientHandler.execute(new ClientExecutionParams<GetSinkRequest, GetSinkResponse>()
                    .withOperationName("GetSink").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSinkRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetSinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the current sink policy attached to this sink. The sink policy specifies what accounts can attach to this
     * sink as source accounts, and what types of data they can share.
     * </p>
     *
     * @param getSinkPolicyRequest
     * @return Result of the GetSinkPolicy operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.GetSinkPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/GetSinkPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSinkPolicyResponse getSinkPolicy(GetSinkPolicyRequest getSinkPolicyRequest) throws InternalServiceException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSinkPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSinkPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSinkPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSinkPolicy");

            return clientHandler.execute(new ClientExecutionParams<GetSinkPolicyRequest, GetSinkPolicyResponse>()
                    .withOperationName("GetSinkPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSinkPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSinkPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of source account links that are linked to this monitoring account sink.
     * </p>
     * <p>
     * To use this operation, provide the sink ARN. To retrieve a list of sink ARNs, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListSinks.html">ListSinks</a>.
     * </p>
     * <p>
     * To find a list of links for one source account, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListLinks.html">ListLinks</a>.
     * </p>
     *
     * @param listAttachedLinksRequest
     * @return Result of the ListAttachedLinks operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListAttachedLinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListAttachedLinks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAttachedLinksResponse listAttachedLinks(ListAttachedLinksRequest listAttachedLinksRequest)
            throws InternalServiceException, MissingRequiredParameterException, InvalidParameterException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAttachedLinksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAttachedLinksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAttachedLinksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAttachedLinks");

            return clientHandler.execute(new ClientExecutionParams<ListAttachedLinksRequest, ListAttachedLinksResponse>()
                    .withOperationName("ListAttachedLinks").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAttachedLinksRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAttachedLinksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of source account links that are linked to this monitoring account sink.
     * </p>
     * <p>
     * To use this operation, provide the sink ARN. To retrieve a list of sink ARNs, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListSinks.html">ListSinks</a>.
     * </p>
     * <p>
     * To find a list of links for one source account, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListLinks.html">ListLinks</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAttachedLinks(software.amazon.awssdk.services.oam.model.ListAttachedLinksRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListAttachedLinksIterable responses = client.listAttachedLinksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.oam.paginators.ListAttachedLinksIterable responses = client
     *             .listAttachedLinksPaginator(request);
     *     for (software.amazon.awssdk.services.oam.model.ListAttachedLinksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListAttachedLinksIterable responses = client.listAttachedLinksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAttachedLinks(software.amazon.awssdk.services.oam.model.ListAttachedLinksRequest)} operation.</b>
     * </p>
     *
     * @param listAttachedLinksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListAttachedLinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListAttachedLinks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAttachedLinksIterable listAttachedLinksPaginator(ListAttachedLinksRequest listAttachedLinksRequest)
            throws InternalServiceException, MissingRequiredParameterException, InvalidParameterException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        return new ListAttachedLinksIterable(this, applyPaginatorUserAgent(listAttachedLinksRequest));
    }

    /**
     * <p>
     * Use this operation in a source account to return a list of links to monitoring account sinks that this source
     * account has.
     * </p>
     * <p>
     * To find a list of links for one monitoring account sink, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListAttachedLinks.html">ListAttachedLinks</a> from
     * within the monitoring account.
     * </p>
     *
     * @param listLinksRequest
     * @return Result of the ListLinks operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListLinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListLinks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListLinksResponse listLinks(ListLinksRequest listLinksRequest) throws InternalServiceException,
            InvalidParameterException, ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLinksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListLinksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLinksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLinks");

            return clientHandler.execute(new ClientExecutionParams<ListLinksRequest, ListLinksResponse>()
                    .withOperationName("ListLinks").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listLinksRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListLinksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use this operation in a source account to return a list of links to monitoring account sinks that this source
     * account has.
     * </p>
     * <p>
     * To find a list of links for one monitoring account sink, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListAttachedLinks.html">ListAttachedLinks</a> from
     * within the monitoring account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listLinks(software.amazon.awssdk.services.oam.model.ListLinksRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListLinksIterable responses = client.listLinksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.oam.paginators.ListLinksIterable responses = client.listLinksPaginator(request);
     *     for (software.amazon.awssdk.services.oam.model.ListLinksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListLinksIterable responses = client.listLinksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLinks(software.amazon.awssdk.services.oam.model.ListLinksRequest)} operation.</b>
     * </p>
     *
     * @param listLinksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListLinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListLinks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListLinksIterable listLinksPaginator(ListLinksRequest listLinksRequest) throws InternalServiceException,
            InvalidParameterException, ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        return new ListLinksIterable(this, applyPaginatorUserAgent(listLinksRequest));
    }

    /**
     * <p>
     * Use this operation in a monitoring account to return the list of sinks created in that account.
     * </p>
     *
     * @param listSinksRequest
     * @return Result of the ListSinks operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListSinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListSinks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListSinksResponse listSinks(ListSinksRequest listSinksRequest) throws InternalServiceException,
            InvalidParameterException, ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSinksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSinksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSinksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSinks");

            return clientHandler.execute(new ClientExecutionParams<ListSinksRequest, ListSinksResponse>()
                    .withOperationName("ListSinks").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSinksRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListSinksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use this operation in a monitoring account to return the list of sinks created in that account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listSinks(software.amazon.awssdk.services.oam.model.ListSinksRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListSinksIterable responses = client.listSinksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.oam.paginators.ListSinksIterable responses = client.listSinksPaginator(request);
     *     for (software.amazon.awssdk.services.oam.model.ListSinksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.oam.paginators.ListSinksIterable responses = client.listSinksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSinks(software.amazon.awssdk.services.oam.model.ListSinksRequest)} operation.</b>
     * </p>
     *
     * @param listSinksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListSinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListSinks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListSinksIterable listSinksPaginator(ListSinksRequest listSinksRequest) throws InternalServiceException,
            InvalidParameterException, ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        return new ListSinksIterable(this, applyPaginatorUserAgent(listSinksRequest));
    }

    /**
     * <p>
     * Displays the tags associated with a resource. Both sinks and links support tagging.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates the resource policy that grants permissions to source accounts to link to the monitoring
     * account sink. When you create a sink policy, you can grant permissions to all accounts in an organization or to
     * individual accounts.
     * </p>
     * <p>
     * You can also use a sink policy to limit the types of data that is shared. The three types that you can allow or
     * deny are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Metrics</b> - Specify with <code>AWS::CloudWatch::Metric</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Log groups</b> - Specify with <code>AWS::Logs::LogGroup</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Traces</b> - Specify with <code>AWS::XRay::Trace</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * See the examples in this section to see how to specify permitted source accounts and data types.
     * </p>
     *
     * @param putSinkPolicyRequest
     * @return Result of the PutSinkPolicy operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.PutSinkPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/PutSinkPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutSinkPolicyResponse putSinkPolicy(PutSinkPolicyRequest putSinkPolicyRequest) throws InternalServiceException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutSinkPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutSinkPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putSinkPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutSinkPolicy");

            return clientHandler.execute(new ClientExecutionParams<PutSinkPolicyRequest, PutSinkPolicyResponse>()
                    .withOperationName("PutSinkPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putSinkPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutSinkPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified resource. Both sinks and links can be tagged.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a resource that already has tags. If you specify a new tag
     * key for the alarm, this tag is appended to the list of tags associated with the alarm. If you specify a tag key
     * that is already associated with the alarm, the new tag value that you specify replaces the previous value for
     * that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a resource.
     * </p>
     * <important>
     * <p>
     * Unlike tagging permissions in other Amazon Web Services services, to tag or untag links and sinks you must have
     * the <code>oam:ResourceTag</code> permission. The <code>iam:ResourceTag</code> permission does not allow you to
     * tag and untag links and sinks.
     * </p>
     * </important>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws TooManyTagsException
     *         A resource can have no more than 50 tags.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            TooManyTagsException, ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     * <important>
     * <p>
     * Unlike tagging permissions in other Amazon Web Services services, to tag or untag links and sinks you must have
     * the <code>oam:ResourceTag</code> permission. The <code>iam:TagResource</code> permission does not allow you to
     * tag and untag links and sinks.
     * </p>
     * </important>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, OamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use this operation to change what types of data are shared from a source account to its linked monitoring account
     * sink. You can't change the sink or change the monitoring account with this operation.
     * </p>
     * <p>
     * To update the list of tags associated with the sink, use <a
     * href="https://docs.aws.amazon.com/OAM/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     *
     * @param updateLinkRequest
     * @return Result of the UpdateLink operation returned by the service.
     * @throws InternalServiceException
     *         Unexpected error while processing the request. Retry the request.
     * @throws MissingRequiredParameterException
     *         A required parameter is missing from the request.
     * @throws InvalidParameterException
     *         A parameter is specified incorrectly.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OamException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OamClient.UpdateLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/oam-2022-06-10/UpdateLink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateLinkResponse updateLink(UpdateLinkRequest updateLinkRequest) throws InternalServiceException,
            MissingRequiredParameterException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, OamException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateLinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateLinkResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OAM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLink");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateLinkRequest, UpdateLinkResponse>().withOperationName("UpdateLink")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateLinkRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends OamRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(OamException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingRequiredParameterException")
                                .exceptionBuilderSupplier(MissingRequiredParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceFault")
                                .exceptionBuilderSupplier(InternalServiceException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
