/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.odb.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A summary of an ODB peering connection.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OdbPeeringConnectionSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<OdbPeeringConnectionSummary.Builder, OdbPeeringConnectionSummary> {
    private static final SdkField<String> ODB_PEERING_CONNECTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("odbPeeringConnectionId").getter(getter(OdbPeeringConnectionSummary::odbPeeringConnectionId))
            .setter(setter(Builder::odbPeeringConnectionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("odbPeeringConnectionId").build())
            .build();

    private static final SdkField<String> DISPLAY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("displayName").getter(getter(OdbPeeringConnectionSummary::displayName))
            .setter(setter(Builder::displayName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("displayName").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(OdbPeeringConnectionSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusReason").getter(getter(OdbPeeringConnectionSummary::statusReason))
            .setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusReason").build()).build();

    private static final SdkField<String> ODB_PEERING_CONNECTION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("odbPeeringConnectionArn").getter(getter(OdbPeeringConnectionSummary::odbPeeringConnectionArn))
            .setter(setter(Builder::odbPeeringConnectionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("odbPeeringConnectionArn").build())
            .build();

    private static final SdkField<String> ODB_NETWORK_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("odbNetworkArn").getter(getter(OdbPeeringConnectionSummary::odbNetworkArn))
            .setter(setter(Builder::odbNetworkArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("odbNetworkArn").build()).build();

    private static final SdkField<String> PEER_NETWORK_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("peerNetworkArn").getter(getter(OdbPeeringConnectionSummary::peerNetworkArn))
            .setter(setter(Builder::peerNetworkArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("peerNetworkArn").build()).build();

    private static final SdkField<String> ODB_PEERING_CONNECTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("odbPeeringConnectionType").getter(getter(OdbPeeringConnectionSummary::odbPeeringConnectionType))
            .setter(setter(Builder::odbPeeringConnectionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("odbPeeringConnectionType").build())
            .build();

    private static final SdkField<List<String>> PEER_NETWORK_CIDRS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("peerNetworkCidrs")
            .getter(getter(OdbPeeringConnectionSummary::peerNetworkCidrs))
            .setter(setter(Builder::peerNetworkCidrs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("peerNetworkCidrs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(OdbPeeringConnectionSummary::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Float> PERCENT_PROGRESS_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("percentProgress").getter(getter(OdbPeeringConnectionSummary::percentProgress))
            .setter(setter(Builder::percentProgress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("percentProgress").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            ODB_PEERING_CONNECTION_ID_FIELD, DISPLAY_NAME_FIELD, STATUS_FIELD, STATUS_REASON_FIELD,
            ODB_PEERING_CONNECTION_ARN_FIELD, ODB_NETWORK_ARN_FIELD, PEER_NETWORK_ARN_FIELD, ODB_PEERING_CONNECTION_TYPE_FIELD,
            PEER_NETWORK_CIDRS_FIELD, CREATED_AT_FIELD, PERCENT_PROGRESS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String odbPeeringConnectionId;

    private final String displayName;

    private final String status;

    private final String statusReason;

    private final String odbPeeringConnectionArn;

    private final String odbNetworkArn;

    private final String peerNetworkArn;

    private final String odbPeeringConnectionType;

    private final List<String> peerNetworkCidrs;

    private final Instant createdAt;

    private final Float percentProgress;

    private OdbPeeringConnectionSummary(BuilderImpl builder) {
        this.odbPeeringConnectionId = builder.odbPeeringConnectionId;
        this.displayName = builder.displayName;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
        this.odbPeeringConnectionArn = builder.odbPeeringConnectionArn;
        this.odbNetworkArn = builder.odbNetworkArn;
        this.peerNetworkArn = builder.peerNetworkArn;
        this.odbPeeringConnectionType = builder.odbPeeringConnectionType;
        this.peerNetworkCidrs = builder.peerNetworkCidrs;
        this.createdAt = builder.createdAt;
        this.percentProgress = builder.percentProgress;
    }

    /**
     * <p>
     * The unique identifier of the ODB peering connection. A sample ID is <code>odbpcx-abcdefgh12345678</code>.
     * </p>
     * 
     * @return The unique identifier of the ODB peering connection. A sample ID is <code>odbpcx-abcdefgh12345678</code>.
     */
    public final String odbPeeringConnectionId() {
        return odbPeeringConnectionId;
    }

    /**
     * <p>
     * The display name of the ODB peering connection.
     * </p>
     * 
     * @return The display name of the ODB peering connection.
     */
    public final String displayName() {
        return displayName;
    }

    /**
     * <p>
     * The status of the ODB peering connection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ResourceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the ODB peering connection.
     * @see ResourceStatus
     */
    public final ResourceStatus status() {
        return ResourceStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the ODB peering connection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ResourceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the ODB peering connection.
     * @see ResourceStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The reason for the current status of the ODB peering connection.
     * </p>
     * 
     * @return The reason for the current status of the ODB peering connection.
     */
    public final String statusReason() {
        return statusReason;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the ODB peering connection.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the ODB peering connection.
     */
    public final String odbPeeringConnectionArn() {
        return odbPeeringConnectionArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the ODB network that initiated the peering connection.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the ODB network that initiated the peering connection.
     */
    public final String odbNetworkArn() {
        return odbNetworkArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the peer network.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the peer network.
     */
    public final String peerNetworkArn() {
        return peerNetworkArn;
    }

    /**
     * <p>
     * The type of the ODB peering connection.
     * </p>
     * <p>
     * Valid Values: <code>ODB-VPC | ODB-ODB</code>
     * </p>
     * 
     * @return The type of the ODB peering connection.</p>
     *         <p>
     *         Valid Values: <code>ODB-VPC | ODB-ODB</code>
     */
    public final String odbPeeringConnectionType() {
        return odbPeeringConnectionType;
    }

    /**
     * For responses, this returns true if the service returned a value for the PeerNetworkCidrs property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPeerNetworkCidrs() {
        return peerNetworkCidrs != null && !(peerNetworkCidrs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The CIDR blocks associated with the peering connection. These CIDR blocks define the IP address ranges that can
     * communicate through the peering connection.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPeerNetworkCidrs} method.
     * </p>
     * 
     * @return The CIDR blocks associated with the peering connection. These CIDR blocks define the IP address ranges
     *         that can communicate through the peering connection.
     */
    public final List<String> peerNetworkCidrs() {
        return peerNetworkCidrs;
    }

    /**
     * <p>
     * The timestamp when the ODB peering connection was created.
     * </p>
     * 
     * @return The timestamp when the ODB peering connection was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The percentage progress of the ODB peering connection creation or deletion.
     * </p>
     * 
     * @return The percentage progress of the ODB peering connection creation or deletion.
     */
    public final Float percentProgress() {
        return percentProgress;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(odbPeeringConnectionId());
        hashCode = 31 * hashCode + Objects.hashCode(displayName());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(odbPeeringConnectionArn());
        hashCode = 31 * hashCode + Objects.hashCode(odbNetworkArn());
        hashCode = 31 * hashCode + Objects.hashCode(peerNetworkArn());
        hashCode = 31 * hashCode + Objects.hashCode(odbPeeringConnectionType());
        hashCode = 31 * hashCode + Objects.hashCode(hasPeerNetworkCidrs() ? peerNetworkCidrs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(percentProgress());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OdbPeeringConnectionSummary)) {
            return false;
        }
        OdbPeeringConnectionSummary other = (OdbPeeringConnectionSummary) obj;
        return Objects.equals(odbPeeringConnectionId(), other.odbPeeringConnectionId())
                && Objects.equals(displayName(), other.displayName()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReason(), other.statusReason())
                && Objects.equals(odbPeeringConnectionArn(), other.odbPeeringConnectionArn())
                && Objects.equals(odbNetworkArn(), other.odbNetworkArn())
                && Objects.equals(peerNetworkArn(), other.peerNetworkArn())
                && Objects.equals(odbPeeringConnectionType(), other.odbPeeringConnectionType())
                && hasPeerNetworkCidrs() == other.hasPeerNetworkCidrs()
                && Objects.equals(peerNetworkCidrs(), other.peerNetworkCidrs()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(percentProgress(), other.percentProgress());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OdbPeeringConnectionSummary").add("OdbPeeringConnectionId", odbPeeringConnectionId())
                .add("DisplayName", displayName()).add("Status", statusAsString()).add("StatusReason", statusReason())
                .add("OdbPeeringConnectionArn", odbPeeringConnectionArn()).add("OdbNetworkArn", odbNetworkArn())
                .add("PeerNetworkArn", peerNetworkArn()).add("OdbPeeringConnectionType", odbPeeringConnectionType())
                .add("PeerNetworkCidrs", hasPeerNetworkCidrs() ? peerNetworkCidrs() : null).add("CreatedAt", createdAt())
                .add("PercentProgress", percentProgress()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "odbPeeringConnectionId":
            return Optional.ofNullable(clazz.cast(odbPeeringConnectionId()));
        case "displayName":
            return Optional.ofNullable(clazz.cast(displayName()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "odbPeeringConnectionArn":
            return Optional.ofNullable(clazz.cast(odbPeeringConnectionArn()));
        case "odbNetworkArn":
            return Optional.ofNullable(clazz.cast(odbNetworkArn()));
        case "peerNetworkArn":
            return Optional.ofNullable(clazz.cast(peerNetworkArn()));
        case "odbPeeringConnectionType":
            return Optional.ofNullable(clazz.cast(odbPeeringConnectionType()));
        case "peerNetworkCidrs":
            return Optional.ofNullable(clazz.cast(peerNetworkCidrs()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "percentProgress":
            return Optional.ofNullable(clazz.cast(percentProgress()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("odbPeeringConnectionId", ODB_PEERING_CONNECTION_ID_FIELD);
        map.put("displayName", DISPLAY_NAME_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("statusReason", STATUS_REASON_FIELD);
        map.put("odbPeeringConnectionArn", ODB_PEERING_CONNECTION_ARN_FIELD);
        map.put("odbNetworkArn", ODB_NETWORK_ARN_FIELD);
        map.put("peerNetworkArn", PEER_NETWORK_ARN_FIELD);
        map.put("odbPeeringConnectionType", ODB_PEERING_CONNECTION_TYPE_FIELD);
        map.put("peerNetworkCidrs", PEER_NETWORK_CIDRS_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("percentProgress", PERCENT_PROGRESS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<OdbPeeringConnectionSummary, T> g) {
        return obj -> g.apply((OdbPeeringConnectionSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OdbPeeringConnectionSummary> {
        /**
         * <p>
         * The unique identifier of the ODB peering connection. A sample ID is <code>odbpcx-abcdefgh12345678</code>.
         * </p>
         * 
         * @param odbPeeringConnectionId
         *        The unique identifier of the ODB peering connection. A sample ID is
         *        <code>odbpcx-abcdefgh12345678</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder odbPeeringConnectionId(String odbPeeringConnectionId);

        /**
         * <p>
         * The display name of the ODB peering connection.
         * </p>
         * 
         * @param displayName
         *        The display name of the ODB peering connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder displayName(String displayName);

        /**
         * <p>
         * The status of the ODB peering connection.
         * </p>
         * 
         * @param status
         *        The status of the ODB peering connection.
         * @see ResourceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the ODB peering connection.
         * </p>
         * 
         * @param status
         *        The status of the ODB peering connection.
         * @see ResourceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceStatus
         */
        Builder status(ResourceStatus status);

        /**
         * <p>
         * The reason for the current status of the ODB peering connection.
         * </p>
         * 
         * @param statusReason
         *        The reason for the current status of the ODB peering connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the ODB peering connection.
         * </p>
         * 
         * @param odbPeeringConnectionArn
         *        The Amazon Resource Name (ARN) of the ODB peering connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder odbPeeringConnectionArn(String odbPeeringConnectionArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the ODB network that initiated the peering connection.
         * </p>
         * 
         * @param odbNetworkArn
         *        The Amazon Resource Name (ARN) of the ODB network that initiated the peering connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder odbNetworkArn(String odbNetworkArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the peer network.
         * </p>
         * 
         * @param peerNetworkArn
         *        The Amazon Resource Name (ARN) of the peer network.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder peerNetworkArn(String peerNetworkArn);

        /**
         * <p>
         * The type of the ODB peering connection.
         * </p>
         * <p>
         * Valid Values: <code>ODB-VPC | ODB-ODB</code>
         * </p>
         * 
         * @param odbPeeringConnectionType
         *        The type of the ODB peering connection.</p>
         *        <p>
         *        Valid Values: <code>ODB-VPC | ODB-ODB</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder odbPeeringConnectionType(String odbPeeringConnectionType);

        /**
         * <p>
         * The CIDR blocks associated with the peering connection. These CIDR blocks define the IP address ranges that
         * can communicate through the peering connection.
         * </p>
         * 
         * @param peerNetworkCidrs
         *        The CIDR blocks associated with the peering connection. These CIDR blocks define the IP address ranges
         *        that can communicate through the peering connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder peerNetworkCidrs(Collection<String> peerNetworkCidrs);

        /**
         * <p>
         * The CIDR blocks associated with the peering connection. These CIDR blocks define the IP address ranges that
         * can communicate through the peering connection.
         * </p>
         * 
         * @param peerNetworkCidrs
         *        The CIDR blocks associated with the peering connection. These CIDR blocks define the IP address ranges
         *        that can communicate through the peering connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder peerNetworkCidrs(String... peerNetworkCidrs);

        /**
         * <p>
         * The timestamp when the ODB peering connection was created.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the ODB peering connection was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The percentage progress of the ODB peering connection creation or deletion.
         * </p>
         * 
         * @param percentProgress
         *        The percentage progress of the ODB peering connection creation or deletion.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder percentProgress(Float percentProgress);
    }

    static final class BuilderImpl implements Builder {
        private String odbPeeringConnectionId;

        private String displayName;

        private String status;

        private String statusReason;

        private String odbPeeringConnectionArn;

        private String odbNetworkArn;

        private String peerNetworkArn;

        private String odbPeeringConnectionType;

        private List<String> peerNetworkCidrs = DefaultSdkAutoConstructList.getInstance();

        private Instant createdAt;

        private Float percentProgress;

        private BuilderImpl() {
        }

        private BuilderImpl(OdbPeeringConnectionSummary model) {
            odbPeeringConnectionId(model.odbPeeringConnectionId);
            displayName(model.displayName);
            status(model.status);
            statusReason(model.statusReason);
            odbPeeringConnectionArn(model.odbPeeringConnectionArn);
            odbNetworkArn(model.odbNetworkArn);
            peerNetworkArn(model.peerNetworkArn);
            odbPeeringConnectionType(model.odbPeeringConnectionType);
            peerNetworkCidrs(model.peerNetworkCidrs);
            createdAt(model.createdAt);
            percentProgress(model.percentProgress);
        }

        public final String getOdbPeeringConnectionId() {
            return odbPeeringConnectionId;
        }

        public final void setOdbPeeringConnectionId(String odbPeeringConnectionId) {
            this.odbPeeringConnectionId = odbPeeringConnectionId;
        }

        @Override
        public final Builder odbPeeringConnectionId(String odbPeeringConnectionId) {
            this.odbPeeringConnectionId = odbPeeringConnectionId;
            return this;
        }

        public final String getDisplayName() {
            return displayName;
        }

        public final void setDisplayName(String displayName) {
            this.displayName = displayName;
        }

        @Override
        public final Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ResourceStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final String getOdbPeeringConnectionArn() {
            return odbPeeringConnectionArn;
        }

        public final void setOdbPeeringConnectionArn(String odbPeeringConnectionArn) {
            this.odbPeeringConnectionArn = odbPeeringConnectionArn;
        }

        @Override
        public final Builder odbPeeringConnectionArn(String odbPeeringConnectionArn) {
            this.odbPeeringConnectionArn = odbPeeringConnectionArn;
            return this;
        }

        public final String getOdbNetworkArn() {
            return odbNetworkArn;
        }

        public final void setOdbNetworkArn(String odbNetworkArn) {
            this.odbNetworkArn = odbNetworkArn;
        }

        @Override
        public final Builder odbNetworkArn(String odbNetworkArn) {
            this.odbNetworkArn = odbNetworkArn;
            return this;
        }

        public final String getPeerNetworkArn() {
            return peerNetworkArn;
        }

        public final void setPeerNetworkArn(String peerNetworkArn) {
            this.peerNetworkArn = peerNetworkArn;
        }

        @Override
        public final Builder peerNetworkArn(String peerNetworkArn) {
            this.peerNetworkArn = peerNetworkArn;
            return this;
        }

        public final String getOdbPeeringConnectionType() {
            return odbPeeringConnectionType;
        }

        public final void setOdbPeeringConnectionType(String odbPeeringConnectionType) {
            this.odbPeeringConnectionType = odbPeeringConnectionType;
        }

        @Override
        public final Builder odbPeeringConnectionType(String odbPeeringConnectionType) {
            this.odbPeeringConnectionType = odbPeeringConnectionType;
            return this;
        }

        public final Collection<String> getPeerNetworkCidrs() {
            if (peerNetworkCidrs instanceof SdkAutoConstructList) {
                return null;
            }
            return peerNetworkCidrs;
        }

        public final void setPeerNetworkCidrs(Collection<String> peerNetworkCidrs) {
            this.peerNetworkCidrs = PeeredCidrListCopier.copy(peerNetworkCidrs);
        }

        @Override
        public final Builder peerNetworkCidrs(Collection<String> peerNetworkCidrs) {
            this.peerNetworkCidrs = PeeredCidrListCopier.copy(peerNetworkCidrs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder peerNetworkCidrs(String... peerNetworkCidrs) {
            peerNetworkCidrs(Arrays.asList(peerNetworkCidrs));
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Float getPercentProgress() {
            return percentProgress;
        }

        public final void setPercentProgress(Float percentProgress) {
            this.percentProgress = percentProgress;
        }

        @Override
        public final Builder percentProgress(Float percentProgress) {
            this.percentProgress = percentProgress;
            return this;
        }

        @Override
        public OdbPeeringConnectionSummary build() {
            return new OdbPeeringConnectionSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
