/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.opensearch.model;

import java.beans.Transient;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Container for the parameters to the <code> <a>UpdateDomain</a> </code> operation. Specifies the type and number of
 * instances in the domain cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateDomainConfigRequest extends OpenSearchRequest implements
        ToCopyableBuilder<UpdateDomainConfigRequest.Builder, UpdateDomainConfigRequest> {
    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DomainName").getter(getter(UpdateDomainConfigRequest::domainName)).setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("DomainName").build()).build();

    private static final SdkField<ClusterConfig> CLUSTER_CONFIG_FIELD = SdkField
            .<ClusterConfig> builder(MarshallingType.SDK_POJO).memberName("ClusterConfig")
            .getter(getter(UpdateDomainConfigRequest::clusterConfig)).setter(setter(Builder::clusterConfig))
            .constructor(ClusterConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterConfig").build()).build();

    private static final SdkField<EBSOptions> EBS_OPTIONS_FIELD = SdkField.<EBSOptions> builder(MarshallingType.SDK_POJO)
            .memberName("EBSOptions").getter(getter(UpdateDomainConfigRequest::ebsOptions)).setter(setter(Builder::ebsOptions))
            .constructor(EBSOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EBSOptions").build()).build();

    private static final SdkField<SnapshotOptions> SNAPSHOT_OPTIONS_FIELD = SdkField
            .<SnapshotOptions> builder(MarshallingType.SDK_POJO).memberName("SnapshotOptions")
            .getter(getter(UpdateDomainConfigRequest::snapshotOptions)).setter(setter(Builder::snapshotOptions))
            .constructor(SnapshotOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotOptions").build()).build();

    private static final SdkField<VPCOptions> VPC_OPTIONS_FIELD = SdkField.<VPCOptions> builder(MarshallingType.SDK_POJO)
            .memberName("VPCOptions").getter(getter(UpdateDomainConfigRequest::vpcOptions)).setter(setter(Builder::vpcOptions))
            .constructor(VPCOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VPCOptions").build()).build();

    private static final SdkField<CognitoOptions> COGNITO_OPTIONS_FIELD = SdkField
            .<CognitoOptions> builder(MarshallingType.SDK_POJO).memberName("CognitoOptions")
            .getter(getter(UpdateDomainConfigRequest::cognitoOptions)).setter(setter(Builder::cognitoOptions))
            .constructor(CognitoOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CognitoOptions").build()).build();

    private static final SdkField<Map<String, String>> ADVANCED_OPTIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("AdvancedOptions")
            .getter(getter(UpdateDomainConfigRequest::advancedOptions))
            .setter(setter(Builder::advancedOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdvancedOptions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> ACCESS_POLICIES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccessPolicies").getter(getter(UpdateDomainConfigRequest::accessPolicies))
            .setter(setter(Builder::accessPolicies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessPolicies").build()).build();

    private static final SdkField<Map<String, LogPublishingOption>> LOG_PUBLISHING_OPTIONS_FIELD = SdkField
            .<Map<String, LogPublishingOption>> builder(MarshallingType.MAP)
            .memberName("LogPublishingOptions")
            .getter(getter(UpdateDomainConfigRequest::logPublishingOptionsAsStrings))
            .setter(setter(Builder::logPublishingOptionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogPublishingOptions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<LogPublishingOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(LogPublishingOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<EncryptionAtRestOptions> ENCRYPTION_AT_REST_OPTIONS_FIELD = SdkField
            .<EncryptionAtRestOptions> builder(MarshallingType.SDK_POJO).memberName("EncryptionAtRestOptions")
            .getter(getter(UpdateDomainConfigRequest::encryptionAtRestOptions)).setter(setter(Builder::encryptionAtRestOptions))
            .constructor(EncryptionAtRestOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionAtRestOptions").build())
            .build();

    private static final SdkField<DomainEndpointOptions> DOMAIN_ENDPOINT_OPTIONS_FIELD = SdkField
            .<DomainEndpointOptions> builder(MarshallingType.SDK_POJO).memberName("DomainEndpointOptions")
            .getter(getter(UpdateDomainConfigRequest::domainEndpointOptions)).setter(setter(Builder::domainEndpointOptions))
            .constructor(DomainEndpointOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainEndpointOptions").build())
            .build();

    private static final SdkField<NodeToNodeEncryptionOptions> NODE_TO_NODE_ENCRYPTION_OPTIONS_FIELD = SdkField
            .<NodeToNodeEncryptionOptions> builder(MarshallingType.SDK_POJO)
            .memberName("NodeToNodeEncryptionOptions")
            .getter(getter(UpdateDomainConfigRequest::nodeToNodeEncryptionOptions))
            .setter(setter(Builder::nodeToNodeEncryptionOptions))
            .constructor(NodeToNodeEncryptionOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NodeToNodeEncryptionOptions")
                    .build()).build();

    private static final SdkField<AdvancedSecurityOptionsInput> ADVANCED_SECURITY_OPTIONS_FIELD = SdkField
            .<AdvancedSecurityOptionsInput> builder(MarshallingType.SDK_POJO).memberName("AdvancedSecurityOptions")
            .getter(getter(UpdateDomainConfigRequest::advancedSecurityOptions)).setter(setter(Builder::advancedSecurityOptions))
            .constructor(AdvancedSecurityOptionsInput::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdvancedSecurityOptions").build())
            .build();

    private static final SdkField<AutoTuneOptions> AUTO_TUNE_OPTIONS_FIELD = SdkField
            .<AutoTuneOptions> builder(MarshallingType.SDK_POJO).memberName("AutoTuneOptions")
            .getter(getter(UpdateDomainConfigRequest::autoTuneOptions)).setter(setter(Builder::autoTuneOptions))
            .constructor(AutoTuneOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoTuneOptions").build()).build();

    private static final SdkField<Boolean> DRY_RUN_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DryRun").getter(getter(UpdateDomainConfigRequest::dryRun)).setter(setter(Builder::dryRun))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DryRun").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOMAIN_NAME_FIELD,
            CLUSTER_CONFIG_FIELD, EBS_OPTIONS_FIELD, SNAPSHOT_OPTIONS_FIELD, VPC_OPTIONS_FIELD, COGNITO_OPTIONS_FIELD,
            ADVANCED_OPTIONS_FIELD, ACCESS_POLICIES_FIELD, LOG_PUBLISHING_OPTIONS_FIELD, ENCRYPTION_AT_REST_OPTIONS_FIELD,
            DOMAIN_ENDPOINT_OPTIONS_FIELD, NODE_TO_NODE_ENCRYPTION_OPTIONS_FIELD, ADVANCED_SECURITY_OPTIONS_FIELD,
            AUTO_TUNE_OPTIONS_FIELD, DRY_RUN_FIELD));

    private final String domainName;

    private final ClusterConfig clusterConfig;

    private final EBSOptions ebsOptions;

    private final SnapshotOptions snapshotOptions;

    private final VPCOptions vpcOptions;

    private final CognitoOptions cognitoOptions;

    private final Map<String, String> advancedOptions;

    private final String accessPolicies;

    private final Map<String, LogPublishingOption> logPublishingOptions;

    private final EncryptionAtRestOptions encryptionAtRestOptions;

    private final DomainEndpointOptions domainEndpointOptions;

    private final NodeToNodeEncryptionOptions nodeToNodeEncryptionOptions;

    private final AdvancedSecurityOptionsInput advancedSecurityOptions;

    private final AutoTuneOptions autoTuneOptions;

    private final Boolean dryRun;

    private UpdateDomainConfigRequest(BuilderImpl builder) {
        super(builder);
        this.domainName = builder.domainName;
        this.clusterConfig = builder.clusterConfig;
        this.ebsOptions = builder.ebsOptions;
        this.snapshotOptions = builder.snapshotOptions;
        this.vpcOptions = builder.vpcOptions;
        this.cognitoOptions = builder.cognitoOptions;
        this.advancedOptions = builder.advancedOptions;
        this.accessPolicies = builder.accessPolicies;
        this.logPublishingOptions = builder.logPublishingOptions;
        this.encryptionAtRestOptions = builder.encryptionAtRestOptions;
        this.domainEndpointOptions = builder.domainEndpointOptions;
        this.nodeToNodeEncryptionOptions = builder.nodeToNodeEncryptionOptions;
        this.advancedSecurityOptions = builder.advancedSecurityOptions;
        this.autoTuneOptions = builder.autoTuneOptions;
        this.dryRun = builder.dryRun;
    }

    /**
     * <p>
     * The name of the domain you're updating.
     * </p>
     * 
     * @return The name of the domain you're updating.
     */
    public final String domainName() {
        return domainName;
    }

    /**
     * <p>
     * The type and number of instances to instantiate for the domain cluster.
     * </p>
     * 
     * @return The type and number of instances to instantiate for the domain cluster.
     */
    public final ClusterConfig clusterConfig() {
        return clusterConfig;
    }

    /**
     * <p>
     * Specify the type and size of the EBS volume to use.
     * </p>
     * 
     * @return Specify the type and size of the EBS volume to use.
     */
    public final EBSOptions ebsOptions() {
        return ebsOptions;
    }

    /**
     * <p>
     * Option to set the time, in UTC format, for the daily automated snapshot. Default value is <code>0</code> hours.
     * </p>
     * 
     * @return Option to set the time, in UTC format, for the daily automated snapshot. Default value is <code>0</code>
     *         hours.
     */
    public final SnapshotOptions snapshotOptions() {
        return snapshotOptions;
    }

    /**
     * <p>
     * Options to specify the subnets and security groups for the VPC endpoint. For more information, see <a
     * href="http://docs.aws.amazon.com/opensearch-service/latest/developerguide/vpc.html" target="_blank">Launching
     * your Amazon OpenSearch Service domains using a VPC </a>.
     * </p>
     * 
     * @return Options to specify the subnets and security groups for the VPC endpoint. For more information, see <a
     *         href="http://docs.aws.amazon.com/opensearch-service/latest/developerguide/vpc.html"
     *         target="_blank">Launching your Amazon OpenSearch Service domains using a VPC </a>.
     */
    public final VPCOptions vpcOptions() {
        return vpcOptions;
    }

    /**
     * <p>
     * Options to specify the Cognito user and identity pools for OpenSearch Dashboards authentication. For more
     * information, see <a href="http://docs.aws.amazon.com/opensearch-service/latest/developerguide/cognito-auth.html"
     * target="_blank">Configuring Amazon Cognito authentication for OpenSearch Dashboards</a>.
     * </p>
     * 
     * @return Options to specify the Cognito user and identity pools for OpenSearch Dashboards authentication. For more
     *         information, see <a
     *         href="http://docs.aws.amazon.com/opensearch-service/latest/developerguide/cognito-auth.html"
     *         target="_blank">Configuring Amazon Cognito authentication for OpenSearch Dashboards</a>.
     */
    public final CognitoOptions cognitoOptions() {
        return cognitoOptions;
    }

    /**
     * For responses, this returns true if the service returned a value for the AdvancedOptions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAdvancedOptions() {
        return advancedOptions != null && !(advancedOptions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Modifies the advanced option to allow references to indices in an HTTP request body. Must be <code>false</code>
     * when configuring access to individual sub-resources. By default, the value is <code>true</code>. See <a href=
     * "http://docs.aws.amazon.com/opensearch-service/latest/developerguide/createupdatedomains.html#createdomain-configure-advanced-options"
     * target="_blank">Advanced options </a> for more information.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAdvancedOptions} method.
     * </p>
     * 
     * @return Modifies the advanced option to allow references to indices in an HTTP request body. Must be
     *         <code>false</code> when configuring access to individual sub-resources. By default, the value is
     *         <code>true</code>. See <a href=
     *         "http://docs.aws.amazon.com/opensearch-service/latest/developerguide/createupdatedomains.html#createdomain-configure-advanced-options"
     *         target="_blank">Advanced options </a> for more information.
     */
    public final Map<String, String> advancedOptions() {
        return advancedOptions;
    }

    /**
     * <p>
     * IAM access policy as a JSON-formatted string.
     * </p>
     * 
     * @return IAM access policy as a JSON-formatted string.
     */
    public final String accessPolicies() {
        return accessPolicies;
    }

    /**
     * <p>
     * Map of <code>LogType</code> and <code>LogPublishingOption</code>, each containing options to publish a given type
     * of OpenSearch log.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLogPublishingOptions} method.
     * </p>
     * 
     * @return Map of <code>LogType</code> and <code>LogPublishingOption</code>, each containing options to publish a
     *         given type of OpenSearch log.
     */
    public final Map<LogType, LogPublishingOption> logPublishingOptions() {
        return LogPublishingOptionsCopier.copyStringToEnum(logPublishingOptions);
    }

    /**
     * For responses, this returns true if the service returned a value for the LogPublishingOptions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLogPublishingOptions() {
        return logPublishingOptions != null && !(logPublishingOptions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Map of <code>LogType</code> and <code>LogPublishingOption</code>, each containing options to publish a given type
     * of OpenSearch log.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLogPublishingOptions} method.
     * </p>
     * 
     * @return Map of <code>LogType</code> and <code>LogPublishingOption</code>, each containing options to publish a
     *         given type of OpenSearch log.
     */
    public final Map<String, LogPublishingOption> logPublishingOptionsAsStrings() {
        return logPublishingOptions;
    }

    /**
     * <p>
     * Specifies encryption of data at rest options.
     * </p>
     * 
     * @return Specifies encryption of data at rest options.
     */
    public final EncryptionAtRestOptions encryptionAtRestOptions() {
        return encryptionAtRestOptions;
    }

    /**
     * <p>
     * Options to specify configuration that will be applied to the domain endpoint.
     * </p>
     * 
     * @return Options to specify configuration that will be applied to the domain endpoint.
     */
    public final DomainEndpointOptions domainEndpointOptions() {
        return domainEndpointOptions;
    }

    /**
     * <p>
     * Specifies node-to-node encryption options.
     * </p>
     * 
     * @return Specifies node-to-node encryption options.
     */
    public final NodeToNodeEncryptionOptions nodeToNodeEncryptionOptions() {
        return nodeToNodeEncryptionOptions;
    }

    /**
     * <p>
     * Specifies advanced security options.
     * </p>
     * 
     * @return Specifies advanced security options.
     */
    public final AdvancedSecurityOptionsInput advancedSecurityOptions() {
        return advancedSecurityOptions;
    }

    /**
     * <p>
     * Specifies Auto-Tune options.
     * </p>
     * 
     * @return Specifies Auto-Tune options.
     */
    public final AutoTuneOptions autoTuneOptions() {
        return autoTuneOptions;
    }

    /**
     * <p>
     * This flag, when set to True, specifies whether the <code>UpdateDomain</code> request should return the results of
     * validation checks (DryRunResults) without actually applying the change.
     * </p>
     * 
     * @return This flag, when set to True, specifies whether the <code>UpdateDomain</code> request should return the
     *         results of validation checks (DryRunResults) without actually applying the change.
     */
    public final Boolean dryRun() {
        return dryRun;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(clusterConfig());
        hashCode = 31 * hashCode + Objects.hashCode(ebsOptions());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotOptions());
        hashCode = 31 * hashCode + Objects.hashCode(vpcOptions());
        hashCode = 31 * hashCode + Objects.hashCode(cognitoOptions());
        hashCode = 31 * hashCode + Objects.hashCode(hasAdvancedOptions() ? advancedOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(accessPolicies());
        hashCode = 31 * hashCode + Objects.hashCode(hasLogPublishingOptions() ? logPublishingOptionsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(encryptionAtRestOptions());
        hashCode = 31 * hashCode + Objects.hashCode(domainEndpointOptions());
        hashCode = 31 * hashCode + Objects.hashCode(nodeToNodeEncryptionOptions());
        hashCode = 31 * hashCode + Objects.hashCode(advancedSecurityOptions());
        hashCode = 31 * hashCode + Objects.hashCode(autoTuneOptions());
        hashCode = 31 * hashCode + Objects.hashCode(dryRun());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateDomainConfigRequest)) {
            return false;
        }
        UpdateDomainConfigRequest other = (UpdateDomainConfigRequest) obj;
        return Objects.equals(domainName(), other.domainName()) && Objects.equals(clusterConfig(), other.clusterConfig())
                && Objects.equals(ebsOptions(), other.ebsOptions()) && Objects.equals(snapshotOptions(), other.snapshotOptions())
                && Objects.equals(vpcOptions(), other.vpcOptions()) && Objects.equals(cognitoOptions(), other.cognitoOptions())
                && hasAdvancedOptions() == other.hasAdvancedOptions()
                && Objects.equals(advancedOptions(), other.advancedOptions())
                && Objects.equals(accessPolicies(), other.accessPolicies())
                && hasLogPublishingOptions() == other.hasLogPublishingOptions()
                && Objects.equals(logPublishingOptionsAsStrings(), other.logPublishingOptionsAsStrings())
                && Objects.equals(encryptionAtRestOptions(), other.encryptionAtRestOptions())
                && Objects.equals(domainEndpointOptions(), other.domainEndpointOptions())
                && Objects.equals(nodeToNodeEncryptionOptions(), other.nodeToNodeEncryptionOptions())
                && Objects.equals(advancedSecurityOptions(), other.advancedSecurityOptions())
                && Objects.equals(autoTuneOptions(), other.autoTuneOptions()) && Objects.equals(dryRun(), other.dryRun());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UpdateDomainConfigRequest").add("DomainName", domainName())
                .add("ClusterConfig", clusterConfig()).add("EBSOptions", ebsOptions()).add("SnapshotOptions", snapshotOptions())
                .add("VPCOptions", vpcOptions()).add("CognitoOptions", cognitoOptions())
                .add("AdvancedOptions", hasAdvancedOptions() ? advancedOptions() : null).add("AccessPolicies", accessPolicies())
                .add("LogPublishingOptions", hasLogPublishingOptions() ? logPublishingOptionsAsStrings() : null)
                .add("EncryptionAtRestOptions", encryptionAtRestOptions()).add("DomainEndpointOptions", domainEndpointOptions())
                .add("NodeToNodeEncryptionOptions", nodeToNodeEncryptionOptions())
                .add("AdvancedSecurityOptions", advancedSecurityOptions()).add("AutoTuneOptions", autoTuneOptions())
                .add("DryRun", dryRun()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DomainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "ClusterConfig":
            return Optional.ofNullable(clazz.cast(clusterConfig()));
        case "EBSOptions":
            return Optional.ofNullable(clazz.cast(ebsOptions()));
        case "SnapshotOptions":
            return Optional.ofNullable(clazz.cast(snapshotOptions()));
        case "VPCOptions":
            return Optional.ofNullable(clazz.cast(vpcOptions()));
        case "CognitoOptions":
            return Optional.ofNullable(clazz.cast(cognitoOptions()));
        case "AdvancedOptions":
            return Optional.ofNullable(clazz.cast(advancedOptions()));
        case "AccessPolicies":
            return Optional.ofNullable(clazz.cast(accessPolicies()));
        case "LogPublishingOptions":
            return Optional.ofNullable(clazz.cast(logPublishingOptionsAsStrings()));
        case "EncryptionAtRestOptions":
            return Optional.ofNullable(clazz.cast(encryptionAtRestOptions()));
        case "DomainEndpointOptions":
            return Optional.ofNullable(clazz.cast(domainEndpointOptions()));
        case "NodeToNodeEncryptionOptions":
            return Optional.ofNullable(clazz.cast(nodeToNodeEncryptionOptions()));
        case "AdvancedSecurityOptions":
            return Optional.ofNullable(clazz.cast(advancedSecurityOptions()));
        case "AutoTuneOptions":
            return Optional.ofNullable(clazz.cast(autoTuneOptions()));
        case "DryRun":
            return Optional.ofNullable(clazz.cast(dryRun()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UpdateDomainConfigRequest, T> g) {
        return obj -> g.apply((UpdateDomainConfigRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends OpenSearchRequest.Builder, SdkPojo, CopyableBuilder<Builder, UpdateDomainConfigRequest> {
        /**
         * <p>
         * The name of the domain you're updating.
         * </p>
         * 
         * @param domainName
         *        The name of the domain you're updating.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * The type and number of instances to instantiate for the domain cluster.
         * </p>
         * 
         * @param clusterConfig
         *        The type and number of instances to instantiate for the domain cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterConfig(ClusterConfig clusterConfig);

        /**
         * <p>
         * The type and number of instances to instantiate for the domain cluster.
         * </p>
         * This is a convenience that creates an instance of the {@link ClusterConfig.Builder} avoiding the need to
         * create one manually via {@link ClusterConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link ClusterConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #clusterConfig(ClusterConfig)}.
         * 
         * @param clusterConfig
         *        a consumer that will call methods on {@link ClusterConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #clusterConfig(ClusterConfig)
         */
        default Builder clusterConfig(Consumer<ClusterConfig.Builder> clusterConfig) {
            return clusterConfig(ClusterConfig.builder().applyMutation(clusterConfig).build());
        }

        /**
         * <p>
         * Specify the type and size of the EBS volume to use.
         * </p>
         * 
         * @param ebsOptions
         *        Specify the type and size of the EBS volume to use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsOptions(EBSOptions ebsOptions);

        /**
         * <p>
         * Specify the type and size of the EBS volume to use.
         * </p>
         * This is a convenience that creates an instance of the {@link EBSOptions.Builder} avoiding the need to create
         * one manually via {@link EBSOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link EBSOptions.Builder#build()} is called immediately and its result
         * is passed to {@link #ebsOptions(EBSOptions)}.
         * 
         * @param ebsOptions
         *        a consumer that will call methods on {@link EBSOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ebsOptions(EBSOptions)
         */
        default Builder ebsOptions(Consumer<EBSOptions.Builder> ebsOptions) {
            return ebsOptions(EBSOptions.builder().applyMutation(ebsOptions).build());
        }

        /**
         * <p>
         * Option to set the time, in UTC format, for the daily automated snapshot. Default value is <code>0</code>
         * hours.
         * </p>
         * 
         * @param snapshotOptions
         *        Option to set the time, in UTC format, for the daily automated snapshot. Default value is
         *        <code>0</code> hours.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotOptions(SnapshotOptions snapshotOptions);

        /**
         * <p>
         * Option to set the time, in UTC format, for the daily automated snapshot. Default value is <code>0</code>
         * hours.
         * </p>
         * This is a convenience that creates an instance of the {@link SnapshotOptions.Builder} avoiding the need to
         * create one manually via {@link SnapshotOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link SnapshotOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #snapshotOptions(SnapshotOptions)}.
         * 
         * @param snapshotOptions
         *        a consumer that will call methods on {@link SnapshotOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #snapshotOptions(SnapshotOptions)
         */
        default Builder snapshotOptions(Consumer<SnapshotOptions.Builder> snapshotOptions) {
            return snapshotOptions(SnapshotOptions.builder().applyMutation(snapshotOptions).build());
        }

        /**
         * <p>
         * Options to specify the subnets and security groups for the VPC endpoint. For more information, see <a
         * href="http://docs.aws.amazon.com/opensearch-service/latest/developerguide/vpc.html" target="_blank">Launching
         * your Amazon OpenSearch Service domains using a VPC </a>.
         * </p>
         * 
         * @param vpcOptions
         *        Options to specify the subnets and security groups for the VPC endpoint. For more information, see <a
         *        href="http://docs.aws.amazon.com/opensearch-service/latest/developerguide/vpc.html"
         *        target="_blank">Launching your Amazon OpenSearch Service domains using a VPC </a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcOptions(VPCOptions vpcOptions);

        /**
         * <p>
         * Options to specify the subnets and security groups for the VPC endpoint. For more information, see <a
         * href="http://docs.aws.amazon.com/opensearch-service/latest/developerguide/vpc.html" target="_blank">Launching
         * your Amazon OpenSearch Service domains using a VPC </a>.
         * </p>
         * This is a convenience that creates an instance of the {@link VPCOptions.Builder} avoiding the need to create
         * one manually via {@link VPCOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link VPCOptions.Builder#build()} is called immediately and its result
         * is passed to {@link #vpcOptions(VPCOptions)}.
         * 
         * @param vpcOptions
         *        a consumer that will call methods on {@link VPCOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcOptions(VPCOptions)
         */
        default Builder vpcOptions(Consumer<VPCOptions.Builder> vpcOptions) {
            return vpcOptions(VPCOptions.builder().applyMutation(vpcOptions).build());
        }

        /**
         * <p>
         * Options to specify the Cognito user and identity pools for OpenSearch Dashboards authentication. For more
         * information, see <a
         * href="http://docs.aws.amazon.com/opensearch-service/latest/developerguide/cognito-auth.html"
         * target="_blank">Configuring Amazon Cognito authentication for OpenSearch Dashboards</a>.
         * </p>
         * 
         * @param cognitoOptions
         *        Options to specify the Cognito user and identity pools for OpenSearch Dashboards authentication. For
         *        more information, see <a
         *        href="http://docs.aws.amazon.com/opensearch-service/latest/developerguide/cognito-auth.html"
         *        target="_blank">Configuring Amazon Cognito authentication for OpenSearch Dashboards</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cognitoOptions(CognitoOptions cognitoOptions);

        /**
         * <p>
         * Options to specify the Cognito user and identity pools for OpenSearch Dashboards authentication. For more
         * information, see <a
         * href="http://docs.aws.amazon.com/opensearch-service/latest/developerguide/cognito-auth.html"
         * target="_blank">Configuring Amazon Cognito authentication for OpenSearch Dashboards</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link CognitoOptions.Builder} avoiding the need to
         * create one manually via {@link CognitoOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link CognitoOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #cognitoOptions(CognitoOptions)}.
         * 
         * @param cognitoOptions
         *        a consumer that will call methods on {@link CognitoOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cognitoOptions(CognitoOptions)
         */
        default Builder cognitoOptions(Consumer<CognitoOptions.Builder> cognitoOptions) {
            return cognitoOptions(CognitoOptions.builder().applyMutation(cognitoOptions).build());
        }

        /**
         * <p>
         * Modifies the advanced option to allow references to indices in an HTTP request body. Must be
         * <code>false</code> when configuring access to individual sub-resources. By default, the value is
         * <code>true</code>. See <a href=
         * "http://docs.aws.amazon.com/opensearch-service/latest/developerguide/createupdatedomains.html#createdomain-configure-advanced-options"
         * target="_blank">Advanced options </a> for more information.
         * </p>
         * 
         * @param advancedOptions
         *        Modifies the advanced option to allow references to indices in an HTTP request body. Must be
         *        <code>false</code> when configuring access to individual sub-resources. By default, the value is
         *        <code>true</code>. See <a href=
         *        "http://docs.aws.amazon.com/opensearch-service/latest/developerguide/createupdatedomains.html#createdomain-configure-advanced-options"
         *        target="_blank">Advanced options </a> for more information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder advancedOptions(Map<String, String> advancedOptions);

        /**
         * <p>
         * IAM access policy as a JSON-formatted string.
         * </p>
         * 
         * @param accessPolicies
         *        IAM access policy as a JSON-formatted string.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessPolicies(String accessPolicies);

        /**
         * <p>
         * Map of <code>LogType</code> and <code>LogPublishingOption</code>, each containing options to publish a given
         * type of OpenSearch log.
         * </p>
         * 
         * @param logPublishingOptions
         *        Map of <code>LogType</code> and <code>LogPublishingOption</code>, each containing options to publish a
         *        given type of OpenSearch log.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logPublishingOptionsWithStrings(Map<String, LogPublishingOption> logPublishingOptions);

        /**
         * <p>
         * Map of <code>LogType</code> and <code>LogPublishingOption</code>, each containing options to publish a given
         * type of OpenSearch log.
         * </p>
         * 
         * @param logPublishingOptions
         *        Map of <code>LogType</code> and <code>LogPublishingOption</code>, each containing options to publish a
         *        given type of OpenSearch log.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logPublishingOptions(Map<LogType, LogPublishingOption> logPublishingOptions);

        /**
         * <p>
         * Specifies encryption of data at rest options.
         * </p>
         * 
         * @param encryptionAtRestOptions
         *        Specifies encryption of data at rest options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionAtRestOptions(EncryptionAtRestOptions encryptionAtRestOptions);

        /**
         * <p>
         * Specifies encryption of data at rest options.
         * </p>
         * This is a convenience that creates an instance of the {@link EncryptionAtRestOptions.Builder} avoiding the
         * need to create one manually via {@link EncryptionAtRestOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link EncryptionAtRestOptions.Builder#build()} is called immediately
         * and its result is passed to {@link #encryptionAtRestOptions(EncryptionAtRestOptions)}.
         * 
         * @param encryptionAtRestOptions
         *        a consumer that will call methods on {@link EncryptionAtRestOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionAtRestOptions(EncryptionAtRestOptions)
         */
        default Builder encryptionAtRestOptions(Consumer<EncryptionAtRestOptions.Builder> encryptionAtRestOptions) {
            return encryptionAtRestOptions(EncryptionAtRestOptions.builder().applyMutation(encryptionAtRestOptions).build());
        }

        /**
         * <p>
         * Options to specify configuration that will be applied to the domain endpoint.
         * </p>
         * 
         * @param domainEndpointOptions
         *        Options to specify configuration that will be applied to the domain endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainEndpointOptions(DomainEndpointOptions domainEndpointOptions);

        /**
         * <p>
         * Options to specify configuration that will be applied to the domain endpoint.
         * </p>
         * This is a convenience that creates an instance of the {@link DomainEndpointOptions.Builder} avoiding the need
         * to create one manually via {@link DomainEndpointOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link DomainEndpointOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #domainEndpointOptions(DomainEndpointOptions)}.
         * 
         * @param domainEndpointOptions
         *        a consumer that will call methods on {@link DomainEndpointOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #domainEndpointOptions(DomainEndpointOptions)
         */
        default Builder domainEndpointOptions(Consumer<DomainEndpointOptions.Builder> domainEndpointOptions) {
            return domainEndpointOptions(DomainEndpointOptions.builder().applyMutation(domainEndpointOptions).build());
        }

        /**
         * <p>
         * Specifies node-to-node encryption options.
         * </p>
         * 
         * @param nodeToNodeEncryptionOptions
         *        Specifies node-to-node encryption options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions nodeToNodeEncryptionOptions);

        /**
         * <p>
         * Specifies node-to-node encryption options.
         * </p>
         * This is a convenience that creates an instance of the {@link NodeToNodeEncryptionOptions.Builder} avoiding
         * the need to create one manually via {@link NodeToNodeEncryptionOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link NodeToNodeEncryptionOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #nodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions)}.
         * 
         * @param nodeToNodeEncryptionOptions
         *        a consumer that will call methods on {@link NodeToNodeEncryptionOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions)
         */
        default Builder nodeToNodeEncryptionOptions(Consumer<NodeToNodeEncryptionOptions.Builder> nodeToNodeEncryptionOptions) {
            return nodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions.builder().applyMutation(nodeToNodeEncryptionOptions)
                    .build());
        }

        /**
         * <p>
         * Specifies advanced security options.
         * </p>
         * 
         * @param advancedSecurityOptions
         *        Specifies advanced security options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder advancedSecurityOptions(AdvancedSecurityOptionsInput advancedSecurityOptions);

        /**
         * <p>
         * Specifies advanced security options.
         * </p>
         * This is a convenience that creates an instance of the {@link AdvancedSecurityOptionsInput.Builder} avoiding
         * the need to create one manually via {@link AdvancedSecurityOptionsInput#builder()}.
         *
         * When the {@link Consumer} completes, {@link AdvancedSecurityOptionsInput.Builder#build()} is called
         * immediately and its result is passed to {@link #advancedSecurityOptions(AdvancedSecurityOptionsInput)}.
         * 
         * @param advancedSecurityOptions
         *        a consumer that will call methods on {@link AdvancedSecurityOptionsInput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #advancedSecurityOptions(AdvancedSecurityOptionsInput)
         */
        default Builder advancedSecurityOptions(Consumer<AdvancedSecurityOptionsInput.Builder> advancedSecurityOptions) {
            return advancedSecurityOptions(AdvancedSecurityOptionsInput.builder().applyMutation(advancedSecurityOptions).build());
        }

        /**
         * <p>
         * Specifies Auto-Tune options.
         * </p>
         * 
         * @param autoTuneOptions
         *        Specifies Auto-Tune options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoTuneOptions(AutoTuneOptions autoTuneOptions);

        /**
         * <p>
         * Specifies Auto-Tune options.
         * </p>
         * This is a convenience that creates an instance of the {@link AutoTuneOptions.Builder} avoiding the need to
         * create one manually via {@link AutoTuneOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link AutoTuneOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #autoTuneOptions(AutoTuneOptions)}.
         * 
         * @param autoTuneOptions
         *        a consumer that will call methods on {@link AutoTuneOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #autoTuneOptions(AutoTuneOptions)
         */
        default Builder autoTuneOptions(Consumer<AutoTuneOptions.Builder> autoTuneOptions) {
            return autoTuneOptions(AutoTuneOptions.builder().applyMutation(autoTuneOptions).build());
        }

        /**
         * <p>
         * This flag, when set to True, specifies whether the <code>UpdateDomain</code> request should return the
         * results of validation checks (DryRunResults) without actually applying the change.
         * </p>
         * 
         * @param dryRun
         *        This flag, when set to True, specifies whether the <code>UpdateDomain</code> request should return the
         *        results of validation checks (DryRunResults) without actually applying the change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dryRun(Boolean dryRun);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends OpenSearchRequest.BuilderImpl implements Builder {
        private String domainName;

        private ClusterConfig clusterConfig;

        private EBSOptions ebsOptions;

        private SnapshotOptions snapshotOptions;

        private VPCOptions vpcOptions;

        private CognitoOptions cognitoOptions;

        private Map<String, String> advancedOptions = DefaultSdkAutoConstructMap.getInstance();

        private String accessPolicies;

        private Map<String, LogPublishingOption> logPublishingOptions = DefaultSdkAutoConstructMap.getInstance();

        private EncryptionAtRestOptions encryptionAtRestOptions;

        private DomainEndpointOptions domainEndpointOptions;

        private NodeToNodeEncryptionOptions nodeToNodeEncryptionOptions;

        private AdvancedSecurityOptionsInput advancedSecurityOptions;

        private AutoTuneOptions autoTuneOptions;

        private Boolean dryRun;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateDomainConfigRequest model) {
            super(model);
            domainName(model.domainName);
            clusterConfig(model.clusterConfig);
            ebsOptions(model.ebsOptions);
            snapshotOptions(model.snapshotOptions);
            vpcOptions(model.vpcOptions);
            cognitoOptions(model.cognitoOptions);
            advancedOptions(model.advancedOptions);
            accessPolicies(model.accessPolicies);
            logPublishingOptionsWithStrings(model.logPublishingOptions);
            encryptionAtRestOptions(model.encryptionAtRestOptions);
            domainEndpointOptions(model.domainEndpointOptions);
            nodeToNodeEncryptionOptions(model.nodeToNodeEncryptionOptions);
            advancedSecurityOptions(model.advancedSecurityOptions);
            autoTuneOptions(model.autoTuneOptions);
            dryRun(model.dryRun);
        }

        public final String getDomainName() {
            return domainName;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        @Override
        @Transient
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final ClusterConfig.Builder getClusterConfig() {
            return clusterConfig != null ? clusterConfig.toBuilder() : null;
        }

        public final void setClusterConfig(ClusterConfig.BuilderImpl clusterConfig) {
            this.clusterConfig = clusterConfig != null ? clusterConfig.build() : null;
        }

        @Override
        @Transient
        public final Builder clusterConfig(ClusterConfig clusterConfig) {
            this.clusterConfig = clusterConfig;
            return this;
        }

        public final EBSOptions.Builder getEbsOptions() {
            return ebsOptions != null ? ebsOptions.toBuilder() : null;
        }

        public final void setEbsOptions(EBSOptions.BuilderImpl ebsOptions) {
            this.ebsOptions = ebsOptions != null ? ebsOptions.build() : null;
        }

        @Override
        @Transient
        public final Builder ebsOptions(EBSOptions ebsOptions) {
            this.ebsOptions = ebsOptions;
            return this;
        }

        public final SnapshotOptions.Builder getSnapshotOptions() {
            return snapshotOptions != null ? snapshotOptions.toBuilder() : null;
        }

        public final void setSnapshotOptions(SnapshotOptions.BuilderImpl snapshotOptions) {
            this.snapshotOptions = snapshotOptions != null ? snapshotOptions.build() : null;
        }

        @Override
        @Transient
        public final Builder snapshotOptions(SnapshotOptions snapshotOptions) {
            this.snapshotOptions = snapshotOptions;
            return this;
        }

        public final VPCOptions.Builder getVpcOptions() {
            return vpcOptions != null ? vpcOptions.toBuilder() : null;
        }

        public final void setVpcOptions(VPCOptions.BuilderImpl vpcOptions) {
            this.vpcOptions = vpcOptions != null ? vpcOptions.build() : null;
        }

        @Override
        @Transient
        public final Builder vpcOptions(VPCOptions vpcOptions) {
            this.vpcOptions = vpcOptions;
            return this;
        }

        public final CognitoOptions.Builder getCognitoOptions() {
            return cognitoOptions != null ? cognitoOptions.toBuilder() : null;
        }

        public final void setCognitoOptions(CognitoOptions.BuilderImpl cognitoOptions) {
            this.cognitoOptions = cognitoOptions != null ? cognitoOptions.build() : null;
        }

        @Override
        @Transient
        public final Builder cognitoOptions(CognitoOptions cognitoOptions) {
            this.cognitoOptions = cognitoOptions;
            return this;
        }

        public final Map<String, String> getAdvancedOptions() {
            if (advancedOptions instanceof SdkAutoConstructMap) {
                return null;
            }
            return advancedOptions;
        }

        public final void setAdvancedOptions(Map<String, String> advancedOptions) {
            this.advancedOptions = AdvancedOptionsCopier.copy(advancedOptions);
        }

        @Override
        @Transient
        public final Builder advancedOptions(Map<String, String> advancedOptions) {
            this.advancedOptions = AdvancedOptionsCopier.copy(advancedOptions);
            return this;
        }

        public final String getAccessPolicies() {
            return accessPolicies;
        }

        public final void setAccessPolicies(String accessPolicies) {
            this.accessPolicies = accessPolicies;
        }

        @Override
        @Transient
        public final Builder accessPolicies(String accessPolicies) {
            this.accessPolicies = accessPolicies;
            return this;
        }

        public final Map<String, LogPublishingOption.Builder> getLogPublishingOptions() {
            Map<String, LogPublishingOption.Builder> result = LogPublishingOptionsCopier.copyToBuilder(this.logPublishingOptions);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setLogPublishingOptions(Map<String, LogPublishingOption.BuilderImpl> logPublishingOptions) {
            this.logPublishingOptions = LogPublishingOptionsCopier.copyFromBuilder(logPublishingOptions);
        }

        @Override
        @Transient
        public final Builder logPublishingOptionsWithStrings(Map<String, LogPublishingOption> logPublishingOptions) {
            this.logPublishingOptions = LogPublishingOptionsCopier.copy(logPublishingOptions);
            return this;
        }

        @Override
        @Transient
        public final Builder logPublishingOptions(Map<LogType, LogPublishingOption> logPublishingOptions) {
            this.logPublishingOptions = LogPublishingOptionsCopier.copyEnumToString(logPublishingOptions);
            return this;
        }

        public final EncryptionAtRestOptions.Builder getEncryptionAtRestOptions() {
            return encryptionAtRestOptions != null ? encryptionAtRestOptions.toBuilder() : null;
        }

        public final void setEncryptionAtRestOptions(EncryptionAtRestOptions.BuilderImpl encryptionAtRestOptions) {
            this.encryptionAtRestOptions = encryptionAtRestOptions != null ? encryptionAtRestOptions.build() : null;
        }

        @Override
        @Transient
        public final Builder encryptionAtRestOptions(EncryptionAtRestOptions encryptionAtRestOptions) {
            this.encryptionAtRestOptions = encryptionAtRestOptions;
            return this;
        }

        public final DomainEndpointOptions.Builder getDomainEndpointOptions() {
            return domainEndpointOptions != null ? domainEndpointOptions.toBuilder() : null;
        }

        public final void setDomainEndpointOptions(DomainEndpointOptions.BuilderImpl domainEndpointOptions) {
            this.domainEndpointOptions = domainEndpointOptions != null ? domainEndpointOptions.build() : null;
        }

        @Override
        @Transient
        public final Builder domainEndpointOptions(DomainEndpointOptions domainEndpointOptions) {
            this.domainEndpointOptions = domainEndpointOptions;
            return this;
        }

        public final NodeToNodeEncryptionOptions.Builder getNodeToNodeEncryptionOptions() {
            return nodeToNodeEncryptionOptions != null ? nodeToNodeEncryptionOptions.toBuilder() : null;
        }

        public final void setNodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions.BuilderImpl nodeToNodeEncryptionOptions) {
            this.nodeToNodeEncryptionOptions = nodeToNodeEncryptionOptions != null ? nodeToNodeEncryptionOptions.build() : null;
        }

        @Override
        @Transient
        public final Builder nodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions nodeToNodeEncryptionOptions) {
            this.nodeToNodeEncryptionOptions = nodeToNodeEncryptionOptions;
            return this;
        }

        public final AdvancedSecurityOptionsInput.Builder getAdvancedSecurityOptions() {
            return advancedSecurityOptions != null ? advancedSecurityOptions.toBuilder() : null;
        }

        public final void setAdvancedSecurityOptions(AdvancedSecurityOptionsInput.BuilderImpl advancedSecurityOptions) {
            this.advancedSecurityOptions = advancedSecurityOptions != null ? advancedSecurityOptions.build() : null;
        }

        @Override
        @Transient
        public final Builder advancedSecurityOptions(AdvancedSecurityOptionsInput advancedSecurityOptions) {
            this.advancedSecurityOptions = advancedSecurityOptions;
            return this;
        }

        public final AutoTuneOptions.Builder getAutoTuneOptions() {
            return autoTuneOptions != null ? autoTuneOptions.toBuilder() : null;
        }

        public final void setAutoTuneOptions(AutoTuneOptions.BuilderImpl autoTuneOptions) {
            this.autoTuneOptions = autoTuneOptions != null ? autoTuneOptions.build() : null;
        }

        @Override
        @Transient
        public final Builder autoTuneOptions(AutoTuneOptions autoTuneOptions) {
            this.autoTuneOptions = autoTuneOptions;
            return this;
        }

        public final Boolean getDryRun() {
            return dryRun;
        }

        public final void setDryRun(Boolean dryRun) {
            this.dryRun = dryRun;
        }

        @Override
        @Transient
        public final Builder dryRun(Boolean dryRun) {
            this.dryRun = dryRun;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateDomainConfigRequest build() {
            return new UpdateDomainConfigRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
