/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.opensearch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A map from an <code> <a>EngineVersion</a> </code> to a list of compatible <code> <a>EngineVersion</a> </code> s to
 * which the domain can be upgraded.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CompatibleVersionsMap implements SdkPojo, Serializable,
        ToCopyableBuilder<CompatibleVersionsMap.Builder, CompatibleVersionsMap> {
    private static final SdkField<String> SOURCE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceVersion").getter(getter(CompatibleVersionsMap::sourceVersion))
            .setter(setter(Builder::sourceVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceVersion").build()).build();

    private static final SdkField<List<String>> TARGET_VERSIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TargetVersions")
            .getter(getter(CompatibleVersionsMap::targetVersions))
            .setter(setter(Builder::targetVersions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetVersions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SOURCE_VERSION_FIELD,
            TARGET_VERSIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String sourceVersion;

    private final List<String> targetVersions;

    private CompatibleVersionsMap(BuilderImpl builder) {
        this.sourceVersion = builder.sourceVersion;
        this.targetVersions = builder.targetVersions;
    }

    /**
     * <p>
     * The current version of OpenSearch a domain is on.
     * </p>
     * 
     * @return The current version of OpenSearch a domain is on.
     */
    public final String sourceVersion() {
        return sourceVersion;
    }

    /**
     * For responses, this returns true if the service returned a value for the TargetVersions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTargetVersions() {
        return targetVersions != null && !(targetVersions instanceof SdkAutoConstructList);
    }

    /**
     * Returns the value of the TargetVersions property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargetVersions} method.
     * </p>
     * 
     * @return The value of the TargetVersions property for this object.
     */
    public final List<String> targetVersions() {
        return targetVersions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sourceVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargetVersions() ? targetVersions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CompatibleVersionsMap)) {
            return false;
        }
        CompatibleVersionsMap other = (CompatibleVersionsMap) obj;
        return Objects.equals(sourceVersion(), other.sourceVersion()) && hasTargetVersions() == other.hasTargetVersions()
                && Objects.equals(targetVersions(), other.targetVersions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CompatibleVersionsMap").add("SourceVersion", sourceVersion())
                .add("TargetVersions", hasTargetVersions() ? targetVersions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SourceVersion":
            return Optional.ofNullable(clazz.cast(sourceVersion()));
        case "TargetVersions":
            return Optional.ofNullable(clazz.cast(targetVersions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CompatibleVersionsMap, T> g) {
        return obj -> g.apply((CompatibleVersionsMap) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CompatibleVersionsMap> {
        /**
         * <p>
         * The current version of OpenSearch a domain is on.
         * </p>
         * 
         * @param sourceVersion
         *        The current version of OpenSearch a domain is on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceVersion(String sourceVersion);

        /**
         * Sets the value of the TargetVersions property for this object.
         *
         * @param targetVersions
         *        The new value for the TargetVersions property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetVersions(Collection<String> targetVersions);

        /**
         * Sets the value of the TargetVersions property for this object.
         *
         * @param targetVersions
         *        The new value for the TargetVersions property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetVersions(String... targetVersions);
    }

    static final class BuilderImpl implements Builder {
        private String sourceVersion;

        private List<String> targetVersions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CompatibleVersionsMap model) {
            sourceVersion(model.sourceVersion);
            targetVersions(model.targetVersions);
        }

        public final String getSourceVersion() {
            return sourceVersion;
        }

        public final void setSourceVersion(String sourceVersion) {
            this.sourceVersion = sourceVersion;
        }

        @Override
        public final Builder sourceVersion(String sourceVersion) {
            this.sourceVersion = sourceVersion;
            return this;
        }

        public final Collection<String> getTargetVersions() {
            if (targetVersions instanceof SdkAutoConstructList) {
                return null;
            }
            return targetVersions;
        }

        public final void setTargetVersions(Collection<String> targetVersions) {
            this.targetVersions = VersionListCopier.copy(targetVersions);
        }

        @Override
        public final Builder targetVersions(Collection<String> targetVersions) {
            this.targetVersions = VersionListCopier.copy(targetVersions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetVersions(String... targetVersions) {
            targetVersions(Arrays.asList(targetVersions));
            return this;
        }

        @Override
        public CompatibleVersionsMap build() {
            return new CompatibleVersionsMap(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
