/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.opensearch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The current status of an OpenSearch Service domain.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DomainStatus implements SdkPojo, Serializable, ToCopyableBuilder<DomainStatus.Builder, DomainStatus> {
    private static final SdkField<String> DOMAIN_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DomainId").getter(getter(DomainStatus::domainId)).setter(setter(Builder::domainId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainId").build()).build();

    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DomainName").getter(getter(DomainStatus::domainName)).setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainName").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ARN")
            .getter(getter(DomainStatus::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ARN").build()).build();

    private static final SdkField<Boolean> CREATED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Created").getter(getter(DomainStatus::created)).setter(setter(Builder::created))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Created").build()).build();

    private static final SdkField<Boolean> DELETED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Deleted").getter(getter(DomainStatus::deleted)).setter(setter(Builder::deleted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Deleted").build()).build();

    private static final SdkField<String> ENDPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Endpoint").getter(getter(DomainStatus::endpoint)).setter(setter(Builder::endpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Endpoint").build()).build();

    private static final SdkField<String> ENDPOINT_V2_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointV2").getter(getter(DomainStatus::endpointV2)).setter(setter(Builder::endpointV2))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointV2").build()).build();

    private static final SdkField<Map<String, String>> ENDPOINTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Endpoints")
            .getter(getter(DomainStatus::endpoints))
            .setter(setter(Builder::endpoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Endpoints").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Boolean> PROCESSING_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Processing").getter(getter(DomainStatus::processing)).setter(setter(Builder::processing))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Processing").build()).build();

    private static final SdkField<Boolean> UPGRADE_PROCESSING_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("UpgradeProcessing").getter(getter(DomainStatus::upgradeProcessing))
            .setter(setter(Builder::upgradeProcessing))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpgradeProcessing").build()).build();

    private static final SdkField<String> ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EngineVersion").getter(getter(DomainStatus::engineVersion)).setter(setter(Builder::engineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final SdkField<ClusterConfig> CLUSTER_CONFIG_FIELD = SdkField
            .<ClusterConfig> builder(MarshallingType.SDK_POJO).memberName("ClusterConfig")
            .getter(getter(DomainStatus::clusterConfig)).setter(setter(Builder::clusterConfig))
            .constructor(ClusterConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterConfig").build()).build();

    private static final SdkField<EBSOptions> EBS_OPTIONS_FIELD = SdkField.<EBSOptions> builder(MarshallingType.SDK_POJO)
            .memberName("EBSOptions").getter(getter(DomainStatus::ebsOptions)).setter(setter(Builder::ebsOptions))
            .constructor(EBSOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EBSOptions").build()).build();

    private static final SdkField<String> ACCESS_POLICIES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccessPolicies").getter(getter(DomainStatus::accessPolicies)).setter(setter(Builder::accessPolicies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessPolicies").build()).build();

    private static final SdkField<String> IP_ADDRESS_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IPAddressType").getter(getter(DomainStatus::ipAddressTypeAsString))
            .setter(setter(Builder::ipAddressType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IPAddressType").build()).build();

    private static final SdkField<SnapshotOptions> SNAPSHOT_OPTIONS_FIELD = SdkField
            .<SnapshotOptions> builder(MarshallingType.SDK_POJO).memberName("SnapshotOptions")
            .getter(getter(DomainStatus::snapshotOptions)).setter(setter(Builder::snapshotOptions))
            .constructor(SnapshotOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotOptions").build()).build();

    private static final SdkField<VPCDerivedInfo> VPC_OPTIONS_FIELD = SdkField.<VPCDerivedInfo> builder(MarshallingType.SDK_POJO)
            .memberName("VPCOptions").getter(getter(DomainStatus::vpcOptions)).setter(setter(Builder::vpcOptions))
            .constructor(VPCDerivedInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VPCOptions").build()).build();

    private static final SdkField<CognitoOptions> COGNITO_OPTIONS_FIELD = SdkField
            .<CognitoOptions> builder(MarshallingType.SDK_POJO).memberName("CognitoOptions")
            .getter(getter(DomainStatus::cognitoOptions)).setter(setter(Builder::cognitoOptions))
            .constructor(CognitoOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CognitoOptions").build()).build();

    private static final SdkField<EncryptionAtRestOptions> ENCRYPTION_AT_REST_OPTIONS_FIELD = SdkField
            .<EncryptionAtRestOptions> builder(MarshallingType.SDK_POJO).memberName("EncryptionAtRestOptions")
            .getter(getter(DomainStatus::encryptionAtRestOptions)).setter(setter(Builder::encryptionAtRestOptions))
            .constructor(EncryptionAtRestOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionAtRestOptions").build())
            .build();

    private static final SdkField<NodeToNodeEncryptionOptions> NODE_TO_NODE_ENCRYPTION_OPTIONS_FIELD = SdkField
            .<NodeToNodeEncryptionOptions> builder(MarshallingType.SDK_POJO)
            .memberName("NodeToNodeEncryptionOptions")
            .getter(getter(DomainStatus::nodeToNodeEncryptionOptions))
            .setter(setter(Builder::nodeToNodeEncryptionOptions))
            .constructor(NodeToNodeEncryptionOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NodeToNodeEncryptionOptions")
                    .build()).build();

    private static final SdkField<Map<String, String>> ADVANCED_OPTIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("AdvancedOptions")
            .getter(getter(DomainStatus::advancedOptions))
            .setter(setter(Builder::advancedOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdvancedOptions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, LogPublishingOption>> LOG_PUBLISHING_OPTIONS_FIELD = SdkField
            .<Map<String, LogPublishingOption>> builder(MarshallingType.MAP)
            .memberName("LogPublishingOptions")
            .getter(getter(DomainStatus::logPublishingOptionsAsStrings))
            .setter(setter(Builder::logPublishingOptionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogPublishingOptions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<LogPublishingOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(LogPublishingOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<ServiceSoftwareOptions> SERVICE_SOFTWARE_OPTIONS_FIELD = SdkField
            .<ServiceSoftwareOptions> builder(MarshallingType.SDK_POJO).memberName("ServiceSoftwareOptions")
            .getter(getter(DomainStatus::serviceSoftwareOptions)).setter(setter(Builder::serviceSoftwareOptions))
            .constructor(ServiceSoftwareOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceSoftwareOptions").build())
            .build();

    private static final SdkField<DomainEndpointOptions> DOMAIN_ENDPOINT_OPTIONS_FIELD = SdkField
            .<DomainEndpointOptions> builder(MarshallingType.SDK_POJO).memberName("DomainEndpointOptions")
            .getter(getter(DomainStatus::domainEndpointOptions)).setter(setter(Builder::domainEndpointOptions))
            .constructor(DomainEndpointOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainEndpointOptions").build())
            .build();

    private static final SdkField<AdvancedSecurityOptions> ADVANCED_SECURITY_OPTIONS_FIELD = SdkField
            .<AdvancedSecurityOptions> builder(MarshallingType.SDK_POJO).memberName("AdvancedSecurityOptions")
            .getter(getter(DomainStatus::advancedSecurityOptions)).setter(setter(Builder::advancedSecurityOptions))
            .constructor(AdvancedSecurityOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdvancedSecurityOptions").build())
            .build();

    private static final SdkField<AutoTuneOptionsOutput> AUTO_TUNE_OPTIONS_FIELD = SdkField
            .<AutoTuneOptionsOutput> builder(MarshallingType.SDK_POJO).memberName("AutoTuneOptions")
            .getter(getter(DomainStatus::autoTuneOptions)).setter(setter(Builder::autoTuneOptions))
            .constructor(AutoTuneOptionsOutput::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoTuneOptions").build()).build();

    private static final SdkField<ChangeProgressDetails> CHANGE_PROGRESS_DETAILS_FIELD = SdkField
            .<ChangeProgressDetails> builder(MarshallingType.SDK_POJO).memberName("ChangeProgressDetails")
            .getter(getter(DomainStatus::changeProgressDetails)).setter(setter(Builder::changeProgressDetails))
            .constructor(ChangeProgressDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChangeProgressDetails").build())
            .build();

    private static final SdkField<OffPeakWindowOptions> OFF_PEAK_WINDOW_OPTIONS_FIELD = SdkField
            .<OffPeakWindowOptions> builder(MarshallingType.SDK_POJO).memberName("OffPeakWindowOptions")
            .getter(getter(DomainStatus::offPeakWindowOptions)).setter(setter(Builder::offPeakWindowOptions))
            .constructor(OffPeakWindowOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OffPeakWindowOptions").build())
            .build();

    private static final SdkField<SoftwareUpdateOptions> SOFTWARE_UPDATE_OPTIONS_FIELD = SdkField
            .<SoftwareUpdateOptions> builder(MarshallingType.SDK_POJO).memberName("SoftwareUpdateOptions")
            .getter(getter(DomainStatus::softwareUpdateOptions)).setter(setter(Builder::softwareUpdateOptions))
            .constructor(SoftwareUpdateOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SoftwareUpdateOptions").build())
            .build();

    private static final SdkField<String> DOMAIN_PROCESSING_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DomainProcessingStatus").getter(getter(DomainStatus::domainProcessingStatusAsString))
            .setter(setter(Builder::domainProcessingStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainProcessingStatus").build())
            .build();

    private static final SdkField<List<ModifyingProperties>> MODIFYING_PROPERTIES_FIELD = SdkField
            .<List<ModifyingProperties>> builder(MarshallingType.LIST)
            .memberName("ModifyingProperties")
            .getter(getter(DomainStatus::modifyingProperties))
            .setter(setter(Builder::modifyingProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModifyingProperties").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ModifyingProperties> builder(MarshallingType.SDK_POJO)
                                            .constructor(ModifyingProperties::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOMAIN_ID_FIELD,
            DOMAIN_NAME_FIELD, ARN_FIELD, CREATED_FIELD, DELETED_FIELD, ENDPOINT_FIELD, ENDPOINT_V2_FIELD, ENDPOINTS_FIELD,
            PROCESSING_FIELD, UPGRADE_PROCESSING_FIELD, ENGINE_VERSION_FIELD, CLUSTER_CONFIG_FIELD, EBS_OPTIONS_FIELD,
            ACCESS_POLICIES_FIELD, IP_ADDRESS_TYPE_FIELD, SNAPSHOT_OPTIONS_FIELD, VPC_OPTIONS_FIELD, COGNITO_OPTIONS_FIELD,
            ENCRYPTION_AT_REST_OPTIONS_FIELD, NODE_TO_NODE_ENCRYPTION_OPTIONS_FIELD, ADVANCED_OPTIONS_FIELD,
            LOG_PUBLISHING_OPTIONS_FIELD, SERVICE_SOFTWARE_OPTIONS_FIELD, DOMAIN_ENDPOINT_OPTIONS_FIELD,
            ADVANCED_SECURITY_OPTIONS_FIELD, AUTO_TUNE_OPTIONS_FIELD, CHANGE_PROGRESS_DETAILS_FIELD,
            OFF_PEAK_WINDOW_OPTIONS_FIELD, SOFTWARE_UPDATE_OPTIONS_FIELD, DOMAIN_PROCESSING_STATUS_FIELD,
            MODIFYING_PROPERTIES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String domainId;

    private final String domainName;

    private final String arn;

    private final Boolean created;

    private final Boolean deleted;

    private final String endpoint;

    private final String endpointV2;

    private final Map<String, String> endpoints;

    private final Boolean processing;

    private final Boolean upgradeProcessing;

    private final String engineVersion;

    private final ClusterConfig clusterConfig;

    private final EBSOptions ebsOptions;

    private final String accessPolicies;

    private final String ipAddressType;

    private final SnapshotOptions snapshotOptions;

    private final VPCDerivedInfo vpcOptions;

    private final CognitoOptions cognitoOptions;

    private final EncryptionAtRestOptions encryptionAtRestOptions;

    private final NodeToNodeEncryptionOptions nodeToNodeEncryptionOptions;

    private final Map<String, String> advancedOptions;

    private final Map<String, LogPublishingOption> logPublishingOptions;

    private final ServiceSoftwareOptions serviceSoftwareOptions;

    private final DomainEndpointOptions domainEndpointOptions;

    private final AdvancedSecurityOptions advancedSecurityOptions;

    private final AutoTuneOptionsOutput autoTuneOptions;

    private final ChangeProgressDetails changeProgressDetails;

    private final OffPeakWindowOptions offPeakWindowOptions;

    private final SoftwareUpdateOptions softwareUpdateOptions;

    private final String domainProcessingStatus;

    private final List<ModifyingProperties> modifyingProperties;

    private DomainStatus(BuilderImpl builder) {
        this.domainId = builder.domainId;
        this.domainName = builder.domainName;
        this.arn = builder.arn;
        this.created = builder.created;
        this.deleted = builder.deleted;
        this.endpoint = builder.endpoint;
        this.endpointV2 = builder.endpointV2;
        this.endpoints = builder.endpoints;
        this.processing = builder.processing;
        this.upgradeProcessing = builder.upgradeProcessing;
        this.engineVersion = builder.engineVersion;
        this.clusterConfig = builder.clusterConfig;
        this.ebsOptions = builder.ebsOptions;
        this.accessPolicies = builder.accessPolicies;
        this.ipAddressType = builder.ipAddressType;
        this.snapshotOptions = builder.snapshotOptions;
        this.vpcOptions = builder.vpcOptions;
        this.cognitoOptions = builder.cognitoOptions;
        this.encryptionAtRestOptions = builder.encryptionAtRestOptions;
        this.nodeToNodeEncryptionOptions = builder.nodeToNodeEncryptionOptions;
        this.advancedOptions = builder.advancedOptions;
        this.logPublishingOptions = builder.logPublishingOptions;
        this.serviceSoftwareOptions = builder.serviceSoftwareOptions;
        this.domainEndpointOptions = builder.domainEndpointOptions;
        this.advancedSecurityOptions = builder.advancedSecurityOptions;
        this.autoTuneOptions = builder.autoTuneOptions;
        this.changeProgressDetails = builder.changeProgressDetails;
        this.offPeakWindowOptions = builder.offPeakWindowOptions;
        this.softwareUpdateOptions = builder.softwareUpdateOptions;
        this.domainProcessingStatus = builder.domainProcessingStatus;
        this.modifyingProperties = builder.modifyingProperties;
    }

    /**
     * <p>
     * Unique identifier for the domain.
     * </p>
     * 
     * @return Unique identifier for the domain.
     */
    public final String domainId() {
        return domainId;
    }

    /**
     * <p>
     * Name of the domain. Domain names are unique across all domains owned by the same account within an Amazon Web
     * Services Region.
     * </p>
     * 
     * @return Name of the domain. Domain names are unique across all domains owned by the same account within an Amazon
     *         Web Services Region.
     */
    public final String domainName() {
        return domainName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the domain. For more information, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html">IAM identifiers </a> in the
     * <i>AWS Identity and Access Management User Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the domain. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html">IAM identifiers </a>
     *         in the <i>AWS Identity and Access Management User Guide</i>.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * Creation status of an OpenSearch Service domain. True if domain creation is complete. False if domain creation is
     * still in progress.
     * </p>
     * 
     * @return Creation status of an OpenSearch Service domain. True if domain creation is complete. False if domain
     *         creation is still in progress.
     */
    public final Boolean created() {
        return created;
    }

    /**
     * <p>
     * Deletion status of an OpenSearch Service domain. True if domain deletion is complete. False if domain deletion is
     * still in progress. Once deletion is complete, the status of the domain is no longer returned.
     * </p>
     * 
     * @return Deletion status of an OpenSearch Service domain. True if domain deletion is complete. False if domain
     *         deletion is still in progress. Once deletion is complete, the status of the domain is no longer returned.
     */
    public final Boolean deleted() {
        return deleted;
    }

    /**
     * <p>
     * Domain-specific endpoint used to submit index, search, and data upload requests to the domain.
     * </p>
     * 
     * @return Domain-specific endpoint used to submit index, search, and data upload requests to the domain.
     */
    public final String endpoint() {
        return endpoint;
    }

    /**
     * <p>
     * If <code>IPAddressType</code> to set to <code>dualstack</code>, a version 2 domain endpoint is provisioned. This
     * endpoint functions like a normal endpoint, except that it works with both IPv4 and IPv6 IP addresses. Normal
     * endpoints work only with IPv4 IP addresses.
     * </p>
     * 
     * @return If <code>IPAddressType</code> to set to <code>dualstack</code>, a version 2 domain endpoint is
     *         provisioned. This endpoint functions like a normal endpoint, except that it works with both IPv4 and IPv6
     *         IP addresses. Normal endpoints work only with IPv4 IP addresses.
     */
    public final String endpointV2() {
        return endpointV2;
    }

    /**
     * For responses, this returns true if the service returned a value for the Endpoints property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEndpoints() {
        return endpoints != null && !(endpoints instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The key-value pair that exists if the OpenSearch Service domain uses VPC endpoints. Example
     * <code>key, value</code>:
     * <code>'vpc','vpc-endpoint-h2dsd34efgyghrtguk5gt6j2foh4.us-east-1.es.amazonaws.com'</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEndpoints} method.
     * </p>
     * 
     * @return The key-value pair that exists if the OpenSearch Service domain uses VPC endpoints. Example
     *         <code>key, value</code>:
     *         <code>'vpc','vpc-endpoint-h2dsd34efgyghrtguk5gt6j2foh4.us-east-1.es.amazonaws.com'</code>.
     */
    public final Map<String, String> endpoints() {
        return endpoints;
    }

    /**
     * <p>
     * The status of the domain configuration. True if OpenSearch Service is processing configuration changes. False if
     * the configuration is active.
     * </p>
     * 
     * @return The status of the domain configuration. True if OpenSearch Service is processing configuration changes.
     *         False if the configuration is active.
     */
    public final Boolean processing() {
        return processing;
    }

    /**
     * <p>
     * The status of a domain version upgrade to a new version of OpenSearch or Elasticsearch. True if OpenSearch
     * Service is in the process of a version upgrade. False if the configuration is active.
     * </p>
     * 
     * @return The status of a domain version upgrade to a new version of OpenSearch or Elasticsearch. True if
     *         OpenSearch Service is in the process of a version upgrade. False if the configuration is active.
     */
    public final Boolean upgradeProcessing() {
        return upgradeProcessing;
    }

    /**
     * <p>
     * Version of OpenSearch or Elasticsearch that the domain is running, in the format <code>Elasticsearch_X.Y</code>
     * or <code>OpenSearch_X.Y</code>.
     * </p>
     * 
     * @return Version of OpenSearch or Elasticsearch that the domain is running, in the format
     *         <code>Elasticsearch_X.Y</code> or <code>OpenSearch_X.Y</code>.
     */
    public final String engineVersion() {
        return engineVersion;
    }

    /**
     * <p>
     * Container for the cluster configuration of the domain.
     * </p>
     * 
     * @return Container for the cluster configuration of the domain.
     */
    public final ClusterConfig clusterConfig() {
        return clusterConfig;
    }

    /**
     * <p>
     * Container for EBS-based storage settings for the domain.
     * </p>
     * 
     * @return Container for EBS-based storage settings for the domain.
     */
    public final EBSOptions ebsOptions() {
        return ebsOptions;
    }

    /**
     * <p>
     * Identity and Access Management (IAM) policy document specifying the access policies for the domain.
     * </p>
     * 
     * @return Identity and Access Management (IAM) policy document specifying the access policies for the domain.
     */
    public final String accessPolicies() {
        return accessPolicies;
    }

    /**
     * <p>
     * The type of IP addresses supported by the endpoint for the domain.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressType}
     * will return {@link IPAddressType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipAddressTypeAsString}.
     * </p>
     * 
     * @return The type of IP addresses supported by the endpoint for the domain.
     * @see IPAddressType
     */
    public final IPAddressType ipAddressType() {
        return IPAddressType.fromValue(ipAddressType);
    }

    /**
     * <p>
     * The type of IP addresses supported by the endpoint for the domain.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressType}
     * will return {@link IPAddressType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipAddressTypeAsString}.
     * </p>
     * 
     * @return The type of IP addresses supported by the endpoint for the domain.
     * @see IPAddressType
     */
    public final String ipAddressTypeAsString() {
        return ipAddressType;
    }

    /**
     * <p>
     * DEPRECATED. Container for parameters required to configure automated snapshots of domain indexes.
     * </p>
     * 
     * @return DEPRECATED. Container for parameters required to configure automated snapshots of domain indexes.
     */
    public final SnapshotOptions snapshotOptions() {
        return snapshotOptions;
    }

    /**
     * <p>
     * The VPC configuration for the domain.
     * </p>
     * 
     * @return The VPC configuration for the domain.
     */
    public final VPCDerivedInfo vpcOptions() {
        return vpcOptions;
    }

    /**
     * <p>
     * Key-value pairs to configure Amazon Cognito authentication for OpenSearch Dashboards.
     * </p>
     * 
     * @return Key-value pairs to configure Amazon Cognito authentication for OpenSearch Dashboards.
     */
    public final CognitoOptions cognitoOptions() {
        return cognitoOptions;
    }

    /**
     * <p>
     * Encryption at rest settings for the domain.
     * </p>
     * 
     * @return Encryption at rest settings for the domain.
     */
    public final EncryptionAtRestOptions encryptionAtRestOptions() {
        return encryptionAtRestOptions;
    }

    /**
     * <p>
     * Whether node-to-node encryption is enabled or disabled.
     * </p>
     * 
     * @return Whether node-to-node encryption is enabled or disabled.
     */
    public final NodeToNodeEncryptionOptions nodeToNodeEncryptionOptions() {
        return nodeToNodeEncryptionOptions;
    }

    /**
     * For responses, this returns true if the service returned a value for the AdvancedOptions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAdvancedOptions() {
        return advancedOptions != null && !(advancedOptions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Key-value pairs that specify advanced configuration options.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAdvancedOptions} method.
     * </p>
     * 
     * @return Key-value pairs that specify advanced configuration options.
     */
    public final Map<String, String> advancedOptions() {
        return advancedOptions;
    }

    /**
     * <p>
     * Log publishing options for the domain.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLogPublishingOptions} method.
     * </p>
     * 
     * @return Log publishing options for the domain.
     */
    public final Map<LogType, LogPublishingOption> logPublishingOptions() {
        return LogPublishingOptionsCopier.copyStringToEnum(logPublishingOptions);
    }

    /**
     * For responses, this returns true if the service returned a value for the LogPublishingOptions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLogPublishingOptions() {
        return logPublishingOptions != null && !(logPublishingOptions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Log publishing options for the domain.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLogPublishingOptions} method.
     * </p>
     * 
     * @return Log publishing options for the domain.
     */
    public final Map<String, LogPublishingOption> logPublishingOptionsAsStrings() {
        return logPublishingOptions;
    }

    /**
     * <p>
     * The current status of the domain's service software.
     * </p>
     * 
     * @return The current status of the domain's service software.
     */
    public final ServiceSoftwareOptions serviceSoftwareOptions() {
        return serviceSoftwareOptions;
    }

    /**
     * <p>
     * Additional options for the domain endpoint, such as whether to require HTTPS for all traffic.
     * </p>
     * 
     * @return Additional options for the domain endpoint, such as whether to require HTTPS for all traffic.
     */
    public final DomainEndpointOptions domainEndpointOptions() {
        return domainEndpointOptions;
    }

    /**
     * <p>
     * Settings for fine-grained access control.
     * </p>
     * 
     * @return Settings for fine-grained access control.
     */
    public final AdvancedSecurityOptions advancedSecurityOptions() {
        return advancedSecurityOptions;
    }

    /**
     * <p>
     * Auto-Tune settings for the domain.
     * </p>
     * 
     * @return Auto-Tune settings for the domain.
     */
    public final AutoTuneOptionsOutput autoTuneOptions() {
        return autoTuneOptions;
    }

    /**
     * <p>
     * Information about a configuration change happening on the domain.
     * </p>
     * 
     * @return Information about a configuration change happening on the domain.
     */
    public final ChangeProgressDetails changeProgressDetails() {
        return changeProgressDetails;
    }

    /**
     * <p>
     * Options that specify a custom 10-hour window during which OpenSearch Service can perform configuration changes on
     * the domain.
     * </p>
     * 
     * @return Options that specify a custom 10-hour window during which OpenSearch Service can perform configuration
     *         changes on the domain.
     */
    public final OffPeakWindowOptions offPeakWindowOptions() {
        return offPeakWindowOptions;
    }

    /**
     * <p>
     * Service software update options for the domain.
     * </p>
     * 
     * @return Service software update options for the domain.
     */
    public final SoftwareUpdateOptions softwareUpdateOptions() {
        return softwareUpdateOptions;
    }

    /**
     * <p>
     * The status of any changes that are currently in progress for the domain.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #domainProcessingStatus} will return {@link DomainProcessingStatusType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #domainProcessingStatusAsString}.
     * </p>
     * 
     * @return The status of any changes that are currently in progress for the domain.
     * @see DomainProcessingStatusType
     */
    public final DomainProcessingStatusType domainProcessingStatus() {
        return DomainProcessingStatusType.fromValue(domainProcessingStatus);
    }

    /**
     * <p>
     * The status of any changes that are currently in progress for the domain.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #domainProcessingStatus} will return {@link DomainProcessingStatusType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #domainProcessingStatusAsString}.
     * </p>
     * 
     * @return The status of any changes that are currently in progress for the domain.
     * @see DomainProcessingStatusType
     */
    public final String domainProcessingStatusAsString() {
        return domainProcessingStatus;
    }

    /**
     * For responses, this returns true if the service returned a value for the ModifyingProperties property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasModifyingProperties() {
        return modifyingProperties != null && !(modifyingProperties instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the domain properties that are currently being modified.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasModifyingProperties} method.
     * </p>
     * 
     * @return Information about the domain properties that are currently being modified.
     */
    public final List<ModifyingProperties> modifyingProperties() {
        return modifyingProperties;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(domainId());
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(created());
        hashCode = 31 * hashCode + Objects.hashCode(deleted());
        hashCode = 31 * hashCode + Objects.hashCode(endpoint());
        hashCode = 31 * hashCode + Objects.hashCode(endpointV2());
        hashCode = 31 * hashCode + Objects.hashCode(hasEndpoints() ? endpoints() : null);
        hashCode = 31 * hashCode + Objects.hashCode(processing());
        hashCode = 31 * hashCode + Objects.hashCode(upgradeProcessing());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(clusterConfig());
        hashCode = 31 * hashCode + Objects.hashCode(ebsOptions());
        hashCode = 31 * hashCode + Objects.hashCode(accessPolicies());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddressTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotOptions());
        hashCode = 31 * hashCode + Objects.hashCode(vpcOptions());
        hashCode = 31 * hashCode + Objects.hashCode(cognitoOptions());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionAtRestOptions());
        hashCode = 31 * hashCode + Objects.hashCode(nodeToNodeEncryptionOptions());
        hashCode = 31 * hashCode + Objects.hashCode(hasAdvancedOptions() ? advancedOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasLogPublishingOptions() ? logPublishingOptionsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(serviceSoftwareOptions());
        hashCode = 31 * hashCode + Objects.hashCode(domainEndpointOptions());
        hashCode = 31 * hashCode + Objects.hashCode(advancedSecurityOptions());
        hashCode = 31 * hashCode + Objects.hashCode(autoTuneOptions());
        hashCode = 31 * hashCode + Objects.hashCode(changeProgressDetails());
        hashCode = 31 * hashCode + Objects.hashCode(offPeakWindowOptions());
        hashCode = 31 * hashCode + Objects.hashCode(softwareUpdateOptions());
        hashCode = 31 * hashCode + Objects.hashCode(domainProcessingStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasModifyingProperties() ? modifyingProperties() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DomainStatus)) {
            return false;
        }
        DomainStatus other = (DomainStatus) obj;
        return Objects.equals(domainId(), other.domainId()) && Objects.equals(domainName(), other.domainName())
                && Objects.equals(arn(), other.arn()) && Objects.equals(created(), other.created())
                && Objects.equals(deleted(), other.deleted()) && Objects.equals(endpoint(), other.endpoint())
                && Objects.equals(endpointV2(), other.endpointV2()) && hasEndpoints() == other.hasEndpoints()
                && Objects.equals(endpoints(), other.endpoints()) && Objects.equals(processing(), other.processing())
                && Objects.equals(upgradeProcessing(), other.upgradeProcessing())
                && Objects.equals(engineVersion(), other.engineVersion())
                && Objects.equals(clusterConfig(), other.clusterConfig()) && Objects.equals(ebsOptions(), other.ebsOptions())
                && Objects.equals(accessPolicies(), other.accessPolicies())
                && Objects.equals(ipAddressTypeAsString(), other.ipAddressTypeAsString())
                && Objects.equals(snapshotOptions(), other.snapshotOptions()) && Objects.equals(vpcOptions(), other.vpcOptions())
                && Objects.equals(cognitoOptions(), other.cognitoOptions())
                && Objects.equals(encryptionAtRestOptions(), other.encryptionAtRestOptions())
                && Objects.equals(nodeToNodeEncryptionOptions(), other.nodeToNodeEncryptionOptions())
                && hasAdvancedOptions() == other.hasAdvancedOptions()
                && Objects.equals(advancedOptions(), other.advancedOptions())
                && hasLogPublishingOptions() == other.hasLogPublishingOptions()
                && Objects.equals(logPublishingOptionsAsStrings(), other.logPublishingOptionsAsStrings())
                && Objects.equals(serviceSoftwareOptions(), other.serviceSoftwareOptions())
                && Objects.equals(domainEndpointOptions(), other.domainEndpointOptions())
                && Objects.equals(advancedSecurityOptions(), other.advancedSecurityOptions())
                && Objects.equals(autoTuneOptions(), other.autoTuneOptions())
                && Objects.equals(changeProgressDetails(), other.changeProgressDetails())
                && Objects.equals(offPeakWindowOptions(), other.offPeakWindowOptions())
                && Objects.equals(softwareUpdateOptions(), other.softwareUpdateOptions())
                && Objects.equals(domainProcessingStatusAsString(), other.domainProcessingStatusAsString())
                && hasModifyingProperties() == other.hasModifyingProperties()
                && Objects.equals(modifyingProperties(), other.modifyingProperties());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DomainStatus").add("DomainId", domainId()).add("DomainName", domainName()).add("ARN", arn())
                .add("Created", created()).add("Deleted", deleted()).add("Endpoint", endpoint()).add("EndpointV2", endpointV2())
                .add("Endpoints", hasEndpoints() ? endpoints() : null).add("Processing", processing())
                .add("UpgradeProcessing", upgradeProcessing()).add("EngineVersion", engineVersion())
                .add("ClusterConfig", clusterConfig()).add("EBSOptions", ebsOptions()).add("AccessPolicies", accessPolicies())
                .add("IPAddressType", ipAddressTypeAsString()).add("SnapshotOptions", snapshotOptions())
                .add("VPCOptions", vpcOptions()).add("CognitoOptions", cognitoOptions())
                .add("EncryptionAtRestOptions", encryptionAtRestOptions())
                .add("NodeToNodeEncryptionOptions", nodeToNodeEncryptionOptions())
                .add("AdvancedOptions", hasAdvancedOptions() ? advancedOptions() : null)
                .add("LogPublishingOptions", hasLogPublishingOptions() ? logPublishingOptionsAsStrings() : null)
                .add("ServiceSoftwareOptions", serviceSoftwareOptions()).add("DomainEndpointOptions", domainEndpointOptions())
                .add("AdvancedSecurityOptions", advancedSecurityOptions()).add("AutoTuneOptions", autoTuneOptions())
                .add("ChangeProgressDetails", changeProgressDetails()).add("OffPeakWindowOptions", offPeakWindowOptions())
                .add("SoftwareUpdateOptions", softwareUpdateOptions())
                .add("DomainProcessingStatus", domainProcessingStatusAsString())
                .add("ModifyingProperties", hasModifyingProperties() ? modifyingProperties() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DomainId":
            return Optional.ofNullable(clazz.cast(domainId()));
        case "DomainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "ARN":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Created":
            return Optional.ofNullable(clazz.cast(created()));
        case "Deleted":
            return Optional.ofNullable(clazz.cast(deleted()));
        case "Endpoint":
            return Optional.ofNullable(clazz.cast(endpoint()));
        case "EndpointV2":
            return Optional.ofNullable(clazz.cast(endpointV2()));
        case "Endpoints":
            return Optional.ofNullable(clazz.cast(endpoints()));
        case "Processing":
            return Optional.ofNullable(clazz.cast(processing()));
        case "UpgradeProcessing":
            return Optional.ofNullable(clazz.cast(upgradeProcessing()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "ClusterConfig":
            return Optional.ofNullable(clazz.cast(clusterConfig()));
        case "EBSOptions":
            return Optional.ofNullable(clazz.cast(ebsOptions()));
        case "AccessPolicies":
            return Optional.ofNullable(clazz.cast(accessPolicies()));
        case "IPAddressType":
            return Optional.ofNullable(clazz.cast(ipAddressTypeAsString()));
        case "SnapshotOptions":
            return Optional.ofNullable(clazz.cast(snapshotOptions()));
        case "VPCOptions":
            return Optional.ofNullable(clazz.cast(vpcOptions()));
        case "CognitoOptions":
            return Optional.ofNullable(clazz.cast(cognitoOptions()));
        case "EncryptionAtRestOptions":
            return Optional.ofNullable(clazz.cast(encryptionAtRestOptions()));
        case "NodeToNodeEncryptionOptions":
            return Optional.ofNullable(clazz.cast(nodeToNodeEncryptionOptions()));
        case "AdvancedOptions":
            return Optional.ofNullable(clazz.cast(advancedOptions()));
        case "LogPublishingOptions":
            return Optional.ofNullable(clazz.cast(logPublishingOptionsAsStrings()));
        case "ServiceSoftwareOptions":
            return Optional.ofNullable(clazz.cast(serviceSoftwareOptions()));
        case "DomainEndpointOptions":
            return Optional.ofNullable(clazz.cast(domainEndpointOptions()));
        case "AdvancedSecurityOptions":
            return Optional.ofNullable(clazz.cast(advancedSecurityOptions()));
        case "AutoTuneOptions":
            return Optional.ofNullable(clazz.cast(autoTuneOptions()));
        case "ChangeProgressDetails":
            return Optional.ofNullable(clazz.cast(changeProgressDetails()));
        case "OffPeakWindowOptions":
            return Optional.ofNullable(clazz.cast(offPeakWindowOptions()));
        case "SoftwareUpdateOptions":
            return Optional.ofNullable(clazz.cast(softwareUpdateOptions()));
        case "DomainProcessingStatus":
            return Optional.ofNullable(clazz.cast(domainProcessingStatusAsString()));
        case "ModifyingProperties":
            return Optional.ofNullable(clazz.cast(modifyingProperties()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DomainStatus, T> g) {
        return obj -> g.apply((DomainStatus) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DomainStatus> {
        /**
         * <p>
         * Unique identifier for the domain.
         * </p>
         * 
         * @param domainId
         *        Unique identifier for the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainId(String domainId);

        /**
         * <p>
         * Name of the domain. Domain names are unique across all domains owned by the same account within an Amazon Web
         * Services Region.
         * </p>
         * 
         * @param domainName
         *        Name of the domain. Domain names are unique across all domains owned by the same account within an
         *        Amazon Web Services Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the domain. For more information, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html">IAM identifiers </a> in
         * the <i>AWS Identity and Access Management User Guide</i>.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the domain. For more information, see <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html">IAM identifiers
         *        </a> in the <i>AWS Identity and Access Management User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * Creation status of an OpenSearch Service domain. True if domain creation is complete. False if domain
         * creation is still in progress.
         * </p>
         * 
         * @param created
         *        Creation status of an OpenSearch Service domain. True if domain creation is complete. False if domain
         *        creation is still in progress.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder created(Boolean created);

        /**
         * <p>
         * Deletion status of an OpenSearch Service domain. True if domain deletion is complete. False if domain
         * deletion is still in progress. Once deletion is complete, the status of the domain is no longer returned.
         * </p>
         * 
         * @param deleted
         *        Deletion status of an OpenSearch Service domain. True if domain deletion is complete. False if domain
         *        deletion is still in progress. Once deletion is complete, the status of the domain is no longer
         *        returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleted(Boolean deleted);

        /**
         * <p>
         * Domain-specific endpoint used to submit index, search, and data upload requests to the domain.
         * </p>
         * 
         * @param endpoint
         *        Domain-specific endpoint used to submit index, search, and data upload requests to the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoint(String endpoint);

        /**
         * <p>
         * If <code>IPAddressType</code> to set to <code>dualstack</code>, a version 2 domain endpoint is provisioned.
         * This endpoint functions like a normal endpoint, except that it works with both IPv4 and IPv6 IP addresses.
         * Normal endpoints work only with IPv4 IP addresses.
         * </p>
         * 
         * @param endpointV2
         *        If <code>IPAddressType</code> to set to <code>dualstack</code>, a version 2 domain endpoint is
         *        provisioned. This endpoint functions like a normal endpoint, except that it works with both IPv4 and
         *        IPv6 IP addresses. Normal endpoints work only with IPv4 IP addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointV2(String endpointV2);

        /**
         * <p>
         * The key-value pair that exists if the OpenSearch Service domain uses VPC endpoints. Example
         * <code>key, value</code>:
         * <code>'vpc','vpc-endpoint-h2dsd34efgyghrtguk5gt6j2foh4.us-east-1.es.amazonaws.com'</code>.
         * </p>
         * 
         * @param endpoints
         *        The key-value pair that exists if the OpenSearch Service domain uses VPC endpoints. Example
         *        <code>key, value</code>:
         *        <code>'vpc','vpc-endpoint-h2dsd34efgyghrtguk5gt6j2foh4.us-east-1.es.amazonaws.com'</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoints(Map<String, String> endpoints);

        /**
         * <p>
         * The status of the domain configuration. True if OpenSearch Service is processing configuration changes. False
         * if the configuration is active.
         * </p>
         * 
         * @param processing
         *        The status of the domain configuration. True if OpenSearch Service is processing configuration
         *        changes. False if the configuration is active.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder processing(Boolean processing);

        /**
         * <p>
         * The status of a domain version upgrade to a new version of OpenSearch or Elasticsearch. True if OpenSearch
         * Service is in the process of a version upgrade. False if the configuration is active.
         * </p>
         * 
         * @param upgradeProcessing
         *        The status of a domain version upgrade to a new version of OpenSearch or Elasticsearch. True if
         *        OpenSearch Service is in the process of a version upgrade. False if the configuration is active.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder upgradeProcessing(Boolean upgradeProcessing);

        /**
         * <p>
         * Version of OpenSearch or Elasticsearch that the domain is running, in the format
         * <code>Elasticsearch_X.Y</code> or <code>OpenSearch_X.Y</code>.
         * </p>
         * 
         * @param engineVersion
         *        Version of OpenSearch or Elasticsearch that the domain is running, in the format
         *        <code>Elasticsearch_X.Y</code> or <code>OpenSearch_X.Y</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(String engineVersion);

        /**
         * <p>
         * Container for the cluster configuration of the domain.
         * </p>
         * 
         * @param clusterConfig
         *        Container for the cluster configuration of the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterConfig(ClusterConfig clusterConfig);

        /**
         * <p>
         * Container for the cluster configuration of the domain.
         * </p>
         * This is a convenience method that creates an instance of the {@link ClusterConfig.Builder} avoiding the need
         * to create one manually via {@link ClusterConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ClusterConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #clusterConfig(ClusterConfig)}.
         * 
         * @param clusterConfig
         *        a consumer that will call methods on {@link ClusterConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #clusterConfig(ClusterConfig)
         */
        default Builder clusterConfig(Consumer<ClusterConfig.Builder> clusterConfig) {
            return clusterConfig(ClusterConfig.builder().applyMutation(clusterConfig).build());
        }

        /**
         * <p>
         * Container for EBS-based storage settings for the domain.
         * </p>
         * 
         * @param ebsOptions
         *        Container for EBS-based storage settings for the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsOptions(EBSOptions ebsOptions);

        /**
         * <p>
         * Container for EBS-based storage settings for the domain.
         * </p>
         * This is a convenience method that creates an instance of the {@link EBSOptions.Builder} avoiding the need to
         * create one manually via {@link EBSOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EBSOptions.Builder#build()} is called immediately and its result
         * is passed to {@link #ebsOptions(EBSOptions)}.
         * 
         * @param ebsOptions
         *        a consumer that will call methods on {@link EBSOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ebsOptions(EBSOptions)
         */
        default Builder ebsOptions(Consumer<EBSOptions.Builder> ebsOptions) {
            return ebsOptions(EBSOptions.builder().applyMutation(ebsOptions).build());
        }

        /**
         * <p>
         * Identity and Access Management (IAM) policy document specifying the access policies for the domain.
         * </p>
         * 
         * @param accessPolicies
         *        Identity and Access Management (IAM) policy document specifying the access policies for the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessPolicies(String accessPolicies);

        /**
         * <p>
         * The type of IP addresses supported by the endpoint for the domain.
         * </p>
         * 
         * @param ipAddressType
         *        The type of IP addresses supported by the endpoint for the domain.
         * @see IPAddressType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IPAddressType
         */
        Builder ipAddressType(String ipAddressType);

        /**
         * <p>
         * The type of IP addresses supported by the endpoint for the domain.
         * </p>
         * 
         * @param ipAddressType
         *        The type of IP addresses supported by the endpoint for the domain.
         * @see IPAddressType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IPAddressType
         */
        Builder ipAddressType(IPAddressType ipAddressType);

        /**
         * <p>
         * DEPRECATED. Container for parameters required to configure automated snapshots of domain indexes.
         * </p>
         * 
         * @param snapshotOptions
         *        DEPRECATED. Container for parameters required to configure automated snapshots of domain indexes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotOptions(SnapshotOptions snapshotOptions);

        /**
         * <p>
         * DEPRECATED. Container for parameters required to configure automated snapshots of domain indexes.
         * </p>
         * This is a convenience method that creates an instance of the {@link SnapshotOptions.Builder} avoiding the
         * need to create one manually via {@link SnapshotOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SnapshotOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #snapshotOptions(SnapshotOptions)}.
         * 
         * @param snapshotOptions
         *        a consumer that will call methods on {@link SnapshotOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #snapshotOptions(SnapshotOptions)
         */
        default Builder snapshotOptions(Consumer<SnapshotOptions.Builder> snapshotOptions) {
            return snapshotOptions(SnapshotOptions.builder().applyMutation(snapshotOptions).build());
        }

        /**
         * <p>
         * The VPC configuration for the domain.
         * </p>
         * 
         * @param vpcOptions
         *        The VPC configuration for the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcOptions(VPCDerivedInfo vpcOptions);

        /**
         * <p>
         * The VPC configuration for the domain.
         * </p>
         * This is a convenience method that creates an instance of the {@link VPCDerivedInfo.Builder} avoiding the need
         * to create one manually via {@link VPCDerivedInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VPCDerivedInfo.Builder#build()} is called immediately and its
         * result is passed to {@link #vpcOptions(VPCDerivedInfo)}.
         * 
         * @param vpcOptions
         *        a consumer that will call methods on {@link VPCDerivedInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcOptions(VPCDerivedInfo)
         */
        default Builder vpcOptions(Consumer<VPCDerivedInfo.Builder> vpcOptions) {
            return vpcOptions(VPCDerivedInfo.builder().applyMutation(vpcOptions).build());
        }

        /**
         * <p>
         * Key-value pairs to configure Amazon Cognito authentication for OpenSearch Dashboards.
         * </p>
         * 
         * @param cognitoOptions
         *        Key-value pairs to configure Amazon Cognito authentication for OpenSearch Dashboards.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cognitoOptions(CognitoOptions cognitoOptions);

        /**
         * <p>
         * Key-value pairs to configure Amazon Cognito authentication for OpenSearch Dashboards.
         * </p>
         * This is a convenience method that creates an instance of the {@link CognitoOptions.Builder} avoiding the need
         * to create one manually via {@link CognitoOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CognitoOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #cognitoOptions(CognitoOptions)}.
         * 
         * @param cognitoOptions
         *        a consumer that will call methods on {@link CognitoOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cognitoOptions(CognitoOptions)
         */
        default Builder cognitoOptions(Consumer<CognitoOptions.Builder> cognitoOptions) {
            return cognitoOptions(CognitoOptions.builder().applyMutation(cognitoOptions).build());
        }

        /**
         * <p>
         * Encryption at rest settings for the domain.
         * </p>
         * 
         * @param encryptionAtRestOptions
         *        Encryption at rest settings for the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionAtRestOptions(EncryptionAtRestOptions encryptionAtRestOptions);

        /**
         * <p>
         * Encryption at rest settings for the domain.
         * </p>
         * This is a convenience method that creates an instance of the {@link EncryptionAtRestOptions.Builder} avoiding
         * the need to create one manually via {@link EncryptionAtRestOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EncryptionAtRestOptions.Builder#build()} is called immediately
         * and its result is passed to {@link #encryptionAtRestOptions(EncryptionAtRestOptions)}.
         * 
         * @param encryptionAtRestOptions
         *        a consumer that will call methods on {@link EncryptionAtRestOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionAtRestOptions(EncryptionAtRestOptions)
         */
        default Builder encryptionAtRestOptions(Consumer<EncryptionAtRestOptions.Builder> encryptionAtRestOptions) {
            return encryptionAtRestOptions(EncryptionAtRestOptions.builder().applyMutation(encryptionAtRestOptions).build());
        }

        /**
         * <p>
         * Whether node-to-node encryption is enabled or disabled.
         * </p>
         * 
         * @param nodeToNodeEncryptionOptions
         *        Whether node-to-node encryption is enabled or disabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions nodeToNodeEncryptionOptions);

        /**
         * <p>
         * Whether node-to-node encryption is enabled or disabled.
         * </p>
         * This is a convenience method that creates an instance of the {@link NodeToNodeEncryptionOptions.Builder}
         * avoiding the need to create one manually via {@link NodeToNodeEncryptionOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NodeToNodeEncryptionOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #nodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions)}.
         * 
         * @param nodeToNodeEncryptionOptions
         *        a consumer that will call methods on {@link NodeToNodeEncryptionOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions)
         */
        default Builder nodeToNodeEncryptionOptions(Consumer<NodeToNodeEncryptionOptions.Builder> nodeToNodeEncryptionOptions) {
            return nodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions.builder().applyMutation(nodeToNodeEncryptionOptions)
                    .build());
        }

        /**
         * <p>
         * Key-value pairs that specify advanced configuration options.
         * </p>
         * 
         * @param advancedOptions
         *        Key-value pairs that specify advanced configuration options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder advancedOptions(Map<String, String> advancedOptions);

        /**
         * <p>
         * Log publishing options for the domain.
         * </p>
         * 
         * @param logPublishingOptions
         *        Log publishing options for the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logPublishingOptionsWithStrings(Map<String, LogPublishingOption> logPublishingOptions);

        /**
         * <p>
         * Log publishing options for the domain.
         * </p>
         * 
         * @param logPublishingOptions
         *        Log publishing options for the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logPublishingOptions(Map<LogType, LogPublishingOption> logPublishingOptions);

        /**
         * <p>
         * The current status of the domain's service software.
         * </p>
         * 
         * @param serviceSoftwareOptions
         *        The current status of the domain's service software.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceSoftwareOptions(ServiceSoftwareOptions serviceSoftwareOptions);

        /**
         * <p>
         * The current status of the domain's service software.
         * </p>
         * This is a convenience method that creates an instance of the {@link ServiceSoftwareOptions.Builder} avoiding
         * the need to create one manually via {@link ServiceSoftwareOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ServiceSoftwareOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #serviceSoftwareOptions(ServiceSoftwareOptions)}.
         * 
         * @param serviceSoftwareOptions
         *        a consumer that will call methods on {@link ServiceSoftwareOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serviceSoftwareOptions(ServiceSoftwareOptions)
         */
        default Builder serviceSoftwareOptions(Consumer<ServiceSoftwareOptions.Builder> serviceSoftwareOptions) {
            return serviceSoftwareOptions(ServiceSoftwareOptions.builder().applyMutation(serviceSoftwareOptions).build());
        }

        /**
         * <p>
         * Additional options for the domain endpoint, such as whether to require HTTPS for all traffic.
         * </p>
         * 
         * @param domainEndpointOptions
         *        Additional options for the domain endpoint, such as whether to require HTTPS for all traffic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainEndpointOptions(DomainEndpointOptions domainEndpointOptions);

        /**
         * <p>
         * Additional options for the domain endpoint, such as whether to require HTTPS for all traffic.
         * </p>
         * This is a convenience method that creates an instance of the {@link DomainEndpointOptions.Builder} avoiding
         * the need to create one manually via {@link DomainEndpointOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DomainEndpointOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #domainEndpointOptions(DomainEndpointOptions)}.
         * 
         * @param domainEndpointOptions
         *        a consumer that will call methods on {@link DomainEndpointOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #domainEndpointOptions(DomainEndpointOptions)
         */
        default Builder domainEndpointOptions(Consumer<DomainEndpointOptions.Builder> domainEndpointOptions) {
            return domainEndpointOptions(DomainEndpointOptions.builder().applyMutation(domainEndpointOptions).build());
        }

        /**
         * <p>
         * Settings for fine-grained access control.
         * </p>
         * 
         * @param advancedSecurityOptions
         *        Settings for fine-grained access control.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder advancedSecurityOptions(AdvancedSecurityOptions advancedSecurityOptions);

        /**
         * <p>
         * Settings for fine-grained access control.
         * </p>
         * This is a convenience method that creates an instance of the {@link AdvancedSecurityOptions.Builder} avoiding
         * the need to create one manually via {@link AdvancedSecurityOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AdvancedSecurityOptions.Builder#build()} is called immediately
         * and its result is passed to {@link #advancedSecurityOptions(AdvancedSecurityOptions)}.
         * 
         * @param advancedSecurityOptions
         *        a consumer that will call methods on {@link AdvancedSecurityOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #advancedSecurityOptions(AdvancedSecurityOptions)
         */
        default Builder advancedSecurityOptions(Consumer<AdvancedSecurityOptions.Builder> advancedSecurityOptions) {
            return advancedSecurityOptions(AdvancedSecurityOptions.builder().applyMutation(advancedSecurityOptions).build());
        }

        /**
         * <p>
         * Auto-Tune settings for the domain.
         * </p>
         * 
         * @param autoTuneOptions
         *        Auto-Tune settings for the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoTuneOptions(AutoTuneOptionsOutput autoTuneOptions);

        /**
         * <p>
         * Auto-Tune settings for the domain.
         * </p>
         * This is a convenience method that creates an instance of the {@link AutoTuneOptionsOutput.Builder} avoiding
         * the need to create one manually via {@link AutoTuneOptionsOutput#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AutoTuneOptionsOutput.Builder#build()} is called immediately and
         * its result is passed to {@link #autoTuneOptions(AutoTuneOptionsOutput)}.
         * 
         * @param autoTuneOptions
         *        a consumer that will call methods on {@link AutoTuneOptionsOutput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #autoTuneOptions(AutoTuneOptionsOutput)
         */
        default Builder autoTuneOptions(Consumer<AutoTuneOptionsOutput.Builder> autoTuneOptions) {
            return autoTuneOptions(AutoTuneOptionsOutput.builder().applyMutation(autoTuneOptions).build());
        }

        /**
         * <p>
         * Information about a configuration change happening on the domain.
         * </p>
         * 
         * @param changeProgressDetails
         *        Information about a configuration change happening on the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder changeProgressDetails(ChangeProgressDetails changeProgressDetails);

        /**
         * <p>
         * Information about a configuration change happening on the domain.
         * </p>
         * This is a convenience method that creates an instance of the {@link ChangeProgressDetails.Builder} avoiding
         * the need to create one manually via {@link ChangeProgressDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ChangeProgressDetails.Builder#build()} is called immediately and
         * its result is passed to {@link #changeProgressDetails(ChangeProgressDetails)}.
         * 
         * @param changeProgressDetails
         *        a consumer that will call methods on {@link ChangeProgressDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #changeProgressDetails(ChangeProgressDetails)
         */
        default Builder changeProgressDetails(Consumer<ChangeProgressDetails.Builder> changeProgressDetails) {
            return changeProgressDetails(ChangeProgressDetails.builder().applyMutation(changeProgressDetails).build());
        }

        /**
         * <p>
         * Options that specify a custom 10-hour window during which OpenSearch Service can perform configuration
         * changes on the domain.
         * </p>
         * 
         * @param offPeakWindowOptions
         *        Options that specify a custom 10-hour window during which OpenSearch Service can perform configuration
         *        changes on the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder offPeakWindowOptions(OffPeakWindowOptions offPeakWindowOptions);

        /**
         * <p>
         * Options that specify a custom 10-hour window during which OpenSearch Service can perform configuration
         * changes on the domain.
         * </p>
         * This is a convenience method that creates an instance of the {@link OffPeakWindowOptions.Builder} avoiding
         * the need to create one manually via {@link OffPeakWindowOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OffPeakWindowOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #offPeakWindowOptions(OffPeakWindowOptions)}.
         * 
         * @param offPeakWindowOptions
         *        a consumer that will call methods on {@link OffPeakWindowOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #offPeakWindowOptions(OffPeakWindowOptions)
         */
        default Builder offPeakWindowOptions(Consumer<OffPeakWindowOptions.Builder> offPeakWindowOptions) {
            return offPeakWindowOptions(OffPeakWindowOptions.builder().applyMutation(offPeakWindowOptions).build());
        }

        /**
         * <p>
         * Service software update options for the domain.
         * </p>
         * 
         * @param softwareUpdateOptions
         *        Service software update options for the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder softwareUpdateOptions(SoftwareUpdateOptions softwareUpdateOptions);

        /**
         * <p>
         * Service software update options for the domain.
         * </p>
         * This is a convenience method that creates an instance of the {@link SoftwareUpdateOptions.Builder} avoiding
         * the need to create one manually via {@link SoftwareUpdateOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SoftwareUpdateOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #softwareUpdateOptions(SoftwareUpdateOptions)}.
         * 
         * @param softwareUpdateOptions
         *        a consumer that will call methods on {@link SoftwareUpdateOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #softwareUpdateOptions(SoftwareUpdateOptions)
         */
        default Builder softwareUpdateOptions(Consumer<SoftwareUpdateOptions.Builder> softwareUpdateOptions) {
            return softwareUpdateOptions(SoftwareUpdateOptions.builder().applyMutation(softwareUpdateOptions).build());
        }

        /**
         * <p>
         * The status of any changes that are currently in progress for the domain.
         * </p>
         * 
         * @param domainProcessingStatus
         *        The status of any changes that are currently in progress for the domain.
         * @see DomainProcessingStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DomainProcessingStatusType
         */
        Builder domainProcessingStatus(String domainProcessingStatus);

        /**
         * <p>
         * The status of any changes that are currently in progress for the domain.
         * </p>
         * 
         * @param domainProcessingStatus
         *        The status of any changes that are currently in progress for the domain.
         * @see DomainProcessingStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DomainProcessingStatusType
         */
        Builder domainProcessingStatus(DomainProcessingStatusType domainProcessingStatus);

        /**
         * <p>
         * Information about the domain properties that are currently being modified.
         * </p>
         * 
         * @param modifyingProperties
         *        Information about the domain properties that are currently being modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modifyingProperties(Collection<ModifyingProperties> modifyingProperties);

        /**
         * <p>
         * Information about the domain properties that are currently being modified.
         * </p>
         * 
         * @param modifyingProperties
         *        Information about the domain properties that are currently being modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modifyingProperties(ModifyingProperties... modifyingProperties);

        /**
         * <p>
         * Information about the domain properties that are currently being modified.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.opensearch.model.ModifyingProperties.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.opensearch.model.ModifyingProperties#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.opensearch.model.ModifyingProperties.Builder#build()} is called
         * immediately and its result is passed to {@link #modifyingProperties(List<ModifyingProperties>)}.
         * 
         * @param modifyingProperties
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.opensearch.model.ModifyingProperties.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #modifyingProperties(java.util.Collection<ModifyingProperties>)
         */
        Builder modifyingProperties(Consumer<ModifyingProperties.Builder>... modifyingProperties);
    }

    static final class BuilderImpl implements Builder {
        private String domainId;

        private String domainName;

        private String arn;

        private Boolean created;

        private Boolean deleted;

        private String endpoint;

        private String endpointV2;

        private Map<String, String> endpoints = DefaultSdkAutoConstructMap.getInstance();

        private Boolean processing;

        private Boolean upgradeProcessing;

        private String engineVersion;

        private ClusterConfig clusterConfig;

        private EBSOptions ebsOptions;

        private String accessPolicies;

        private String ipAddressType;

        private SnapshotOptions snapshotOptions;

        private VPCDerivedInfo vpcOptions;

        private CognitoOptions cognitoOptions;

        private EncryptionAtRestOptions encryptionAtRestOptions;

        private NodeToNodeEncryptionOptions nodeToNodeEncryptionOptions;

        private Map<String, String> advancedOptions = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, LogPublishingOption> logPublishingOptions = DefaultSdkAutoConstructMap.getInstance();

        private ServiceSoftwareOptions serviceSoftwareOptions;

        private DomainEndpointOptions domainEndpointOptions;

        private AdvancedSecurityOptions advancedSecurityOptions;

        private AutoTuneOptionsOutput autoTuneOptions;

        private ChangeProgressDetails changeProgressDetails;

        private OffPeakWindowOptions offPeakWindowOptions;

        private SoftwareUpdateOptions softwareUpdateOptions;

        private String domainProcessingStatus;

        private List<ModifyingProperties> modifyingProperties = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DomainStatus model) {
            domainId(model.domainId);
            domainName(model.domainName);
            arn(model.arn);
            created(model.created);
            deleted(model.deleted);
            endpoint(model.endpoint);
            endpointV2(model.endpointV2);
            endpoints(model.endpoints);
            processing(model.processing);
            upgradeProcessing(model.upgradeProcessing);
            engineVersion(model.engineVersion);
            clusterConfig(model.clusterConfig);
            ebsOptions(model.ebsOptions);
            accessPolicies(model.accessPolicies);
            ipAddressType(model.ipAddressType);
            snapshotOptions(model.snapshotOptions);
            vpcOptions(model.vpcOptions);
            cognitoOptions(model.cognitoOptions);
            encryptionAtRestOptions(model.encryptionAtRestOptions);
            nodeToNodeEncryptionOptions(model.nodeToNodeEncryptionOptions);
            advancedOptions(model.advancedOptions);
            logPublishingOptionsWithStrings(model.logPublishingOptions);
            serviceSoftwareOptions(model.serviceSoftwareOptions);
            domainEndpointOptions(model.domainEndpointOptions);
            advancedSecurityOptions(model.advancedSecurityOptions);
            autoTuneOptions(model.autoTuneOptions);
            changeProgressDetails(model.changeProgressDetails);
            offPeakWindowOptions(model.offPeakWindowOptions);
            softwareUpdateOptions(model.softwareUpdateOptions);
            domainProcessingStatus(model.domainProcessingStatus);
            modifyingProperties(model.modifyingProperties);
        }

        public final String getDomainId() {
            return domainId;
        }

        public final void setDomainId(String domainId) {
            this.domainId = domainId;
        }

        @Override
        public final Builder domainId(String domainId) {
            this.domainId = domainId;
            return this;
        }

        public final String getDomainName() {
            return domainName;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        @Override
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final Boolean getCreated() {
            return created;
        }

        public final void setCreated(Boolean created) {
            this.created = created;
        }

        @Override
        public final Builder created(Boolean created) {
            this.created = created;
            return this;
        }

        public final Boolean getDeleted() {
            return deleted;
        }

        public final void setDeleted(Boolean deleted) {
            this.deleted = deleted;
        }

        @Override
        public final Builder deleted(Boolean deleted) {
            this.deleted = deleted;
            return this;
        }

        public final String getEndpoint() {
            return endpoint;
        }

        public final void setEndpoint(String endpoint) {
            this.endpoint = endpoint;
        }

        @Override
        public final Builder endpoint(String endpoint) {
            this.endpoint = endpoint;
            return this;
        }

        public final String getEndpointV2() {
            return endpointV2;
        }

        public final void setEndpointV2(String endpointV2) {
            this.endpointV2 = endpointV2;
        }

        @Override
        public final Builder endpointV2(String endpointV2) {
            this.endpointV2 = endpointV2;
            return this;
        }

        public final Map<String, String> getEndpoints() {
            if (endpoints instanceof SdkAutoConstructMap) {
                return null;
            }
            return endpoints;
        }

        public final void setEndpoints(Map<String, String> endpoints) {
            this.endpoints = EndpointsMapCopier.copy(endpoints);
        }

        @Override
        public final Builder endpoints(Map<String, String> endpoints) {
            this.endpoints = EndpointsMapCopier.copy(endpoints);
            return this;
        }

        public final Boolean getProcessing() {
            return processing;
        }

        public final void setProcessing(Boolean processing) {
            this.processing = processing;
        }

        @Override
        public final Builder processing(Boolean processing) {
            this.processing = processing;
            return this;
        }

        public final Boolean getUpgradeProcessing() {
            return upgradeProcessing;
        }

        public final void setUpgradeProcessing(Boolean upgradeProcessing) {
            this.upgradeProcessing = upgradeProcessing;
        }

        @Override
        public final Builder upgradeProcessing(Boolean upgradeProcessing) {
            this.upgradeProcessing = upgradeProcessing;
            return this;
        }

        public final String getEngineVersion() {
            return engineVersion;
        }

        public final void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        @Override
        public final Builder engineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final ClusterConfig.Builder getClusterConfig() {
            return clusterConfig != null ? clusterConfig.toBuilder() : null;
        }

        public final void setClusterConfig(ClusterConfig.BuilderImpl clusterConfig) {
            this.clusterConfig = clusterConfig != null ? clusterConfig.build() : null;
        }

        @Override
        public final Builder clusterConfig(ClusterConfig clusterConfig) {
            this.clusterConfig = clusterConfig;
            return this;
        }

        public final EBSOptions.Builder getEbsOptions() {
            return ebsOptions != null ? ebsOptions.toBuilder() : null;
        }

        public final void setEbsOptions(EBSOptions.BuilderImpl ebsOptions) {
            this.ebsOptions = ebsOptions != null ? ebsOptions.build() : null;
        }

        @Override
        public final Builder ebsOptions(EBSOptions ebsOptions) {
            this.ebsOptions = ebsOptions;
            return this;
        }

        public final String getAccessPolicies() {
            return accessPolicies;
        }

        public final void setAccessPolicies(String accessPolicies) {
            this.accessPolicies = accessPolicies;
        }

        @Override
        public final Builder accessPolicies(String accessPolicies) {
            this.accessPolicies = accessPolicies;
            return this;
        }

        public final String getIpAddressType() {
            return ipAddressType;
        }

        public final void setIpAddressType(String ipAddressType) {
            this.ipAddressType = ipAddressType;
        }

        @Override
        public final Builder ipAddressType(String ipAddressType) {
            this.ipAddressType = ipAddressType;
            return this;
        }

        @Override
        public final Builder ipAddressType(IPAddressType ipAddressType) {
            this.ipAddressType(ipAddressType == null ? null : ipAddressType.toString());
            return this;
        }

        public final SnapshotOptions.Builder getSnapshotOptions() {
            return snapshotOptions != null ? snapshotOptions.toBuilder() : null;
        }

        public final void setSnapshotOptions(SnapshotOptions.BuilderImpl snapshotOptions) {
            this.snapshotOptions = snapshotOptions != null ? snapshotOptions.build() : null;
        }

        @Override
        public final Builder snapshotOptions(SnapshotOptions snapshotOptions) {
            this.snapshotOptions = snapshotOptions;
            return this;
        }

        public final VPCDerivedInfo.Builder getVpcOptions() {
            return vpcOptions != null ? vpcOptions.toBuilder() : null;
        }

        public final void setVpcOptions(VPCDerivedInfo.BuilderImpl vpcOptions) {
            this.vpcOptions = vpcOptions != null ? vpcOptions.build() : null;
        }

        @Override
        public final Builder vpcOptions(VPCDerivedInfo vpcOptions) {
            this.vpcOptions = vpcOptions;
            return this;
        }

        public final CognitoOptions.Builder getCognitoOptions() {
            return cognitoOptions != null ? cognitoOptions.toBuilder() : null;
        }

        public final void setCognitoOptions(CognitoOptions.BuilderImpl cognitoOptions) {
            this.cognitoOptions = cognitoOptions != null ? cognitoOptions.build() : null;
        }

        @Override
        public final Builder cognitoOptions(CognitoOptions cognitoOptions) {
            this.cognitoOptions = cognitoOptions;
            return this;
        }

        public final EncryptionAtRestOptions.Builder getEncryptionAtRestOptions() {
            return encryptionAtRestOptions != null ? encryptionAtRestOptions.toBuilder() : null;
        }

        public final void setEncryptionAtRestOptions(EncryptionAtRestOptions.BuilderImpl encryptionAtRestOptions) {
            this.encryptionAtRestOptions = encryptionAtRestOptions != null ? encryptionAtRestOptions.build() : null;
        }

        @Override
        public final Builder encryptionAtRestOptions(EncryptionAtRestOptions encryptionAtRestOptions) {
            this.encryptionAtRestOptions = encryptionAtRestOptions;
            return this;
        }

        public final NodeToNodeEncryptionOptions.Builder getNodeToNodeEncryptionOptions() {
            return nodeToNodeEncryptionOptions != null ? nodeToNodeEncryptionOptions.toBuilder() : null;
        }

        public final void setNodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions.BuilderImpl nodeToNodeEncryptionOptions) {
            this.nodeToNodeEncryptionOptions = nodeToNodeEncryptionOptions != null ? nodeToNodeEncryptionOptions.build() : null;
        }

        @Override
        public final Builder nodeToNodeEncryptionOptions(NodeToNodeEncryptionOptions nodeToNodeEncryptionOptions) {
            this.nodeToNodeEncryptionOptions = nodeToNodeEncryptionOptions;
            return this;
        }

        public final Map<String, String> getAdvancedOptions() {
            if (advancedOptions instanceof SdkAutoConstructMap) {
                return null;
            }
            return advancedOptions;
        }

        public final void setAdvancedOptions(Map<String, String> advancedOptions) {
            this.advancedOptions = AdvancedOptionsCopier.copy(advancedOptions);
        }

        @Override
        public final Builder advancedOptions(Map<String, String> advancedOptions) {
            this.advancedOptions = AdvancedOptionsCopier.copy(advancedOptions);
            return this;
        }

        public final Map<String, LogPublishingOption.Builder> getLogPublishingOptions() {
            Map<String, LogPublishingOption.Builder> result = LogPublishingOptionsCopier.copyToBuilder(this.logPublishingOptions);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setLogPublishingOptions(Map<String, LogPublishingOption.BuilderImpl> logPublishingOptions) {
            this.logPublishingOptions = LogPublishingOptionsCopier.copyFromBuilder(logPublishingOptions);
        }

        @Override
        public final Builder logPublishingOptionsWithStrings(Map<String, LogPublishingOption> logPublishingOptions) {
            this.logPublishingOptions = LogPublishingOptionsCopier.copy(logPublishingOptions);
            return this;
        }

        @Override
        public final Builder logPublishingOptions(Map<LogType, LogPublishingOption> logPublishingOptions) {
            this.logPublishingOptions = LogPublishingOptionsCopier.copyEnumToString(logPublishingOptions);
            return this;
        }

        public final ServiceSoftwareOptions.Builder getServiceSoftwareOptions() {
            return serviceSoftwareOptions != null ? serviceSoftwareOptions.toBuilder() : null;
        }

        public final void setServiceSoftwareOptions(ServiceSoftwareOptions.BuilderImpl serviceSoftwareOptions) {
            this.serviceSoftwareOptions = serviceSoftwareOptions != null ? serviceSoftwareOptions.build() : null;
        }

        @Override
        public final Builder serviceSoftwareOptions(ServiceSoftwareOptions serviceSoftwareOptions) {
            this.serviceSoftwareOptions = serviceSoftwareOptions;
            return this;
        }

        public final DomainEndpointOptions.Builder getDomainEndpointOptions() {
            return domainEndpointOptions != null ? domainEndpointOptions.toBuilder() : null;
        }

        public final void setDomainEndpointOptions(DomainEndpointOptions.BuilderImpl domainEndpointOptions) {
            this.domainEndpointOptions = domainEndpointOptions != null ? domainEndpointOptions.build() : null;
        }

        @Override
        public final Builder domainEndpointOptions(DomainEndpointOptions domainEndpointOptions) {
            this.domainEndpointOptions = domainEndpointOptions;
            return this;
        }

        public final AdvancedSecurityOptions.Builder getAdvancedSecurityOptions() {
            return advancedSecurityOptions != null ? advancedSecurityOptions.toBuilder() : null;
        }

        public final void setAdvancedSecurityOptions(AdvancedSecurityOptions.BuilderImpl advancedSecurityOptions) {
            this.advancedSecurityOptions = advancedSecurityOptions != null ? advancedSecurityOptions.build() : null;
        }

        @Override
        public final Builder advancedSecurityOptions(AdvancedSecurityOptions advancedSecurityOptions) {
            this.advancedSecurityOptions = advancedSecurityOptions;
            return this;
        }

        public final AutoTuneOptionsOutput.Builder getAutoTuneOptions() {
            return autoTuneOptions != null ? autoTuneOptions.toBuilder() : null;
        }

        public final void setAutoTuneOptions(AutoTuneOptionsOutput.BuilderImpl autoTuneOptions) {
            this.autoTuneOptions = autoTuneOptions != null ? autoTuneOptions.build() : null;
        }

        @Override
        public final Builder autoTuneOptions(AutoTuneOptionsOutput autoTuneOptions) {
            this.autoTuneOptions = autoTuneOptions;
            return this;
        }

        public final ChangeProgressDetails.Builder getChangeProgressDetails() {
            return changeProgressDetails != null ? changeProgressDetails.toBuilder() : null;
        }

        public final void setChangeProgressDetails(ChangeProgressDetails.BuilderImpl changeProgressDetails) {
            this.changeProgressDetails = changeProgressDetails != null ? changeProgressDetails.build() : null;
        }

        @Override
        public final Builder changeProgressDetails(ChangeProgressDetails changeProgressDetails) {
            this.changeProgressDetails = changeProgressDetails;
            return this;
        }

        public final OffPeakWindowOptions.Builder getOffPeakWindowOptions() {
            return offPeakWindowOptions != null ? offPeakWindowOptions.toBuilder() : null;
        }

        public final void setOffPeakWindowOptions(OffPeakWindowOptions.BuilderImpl offPeakWindowOptions) {
            this.offPeakWindowOptions = offPeakWindowOptions != null ? offPeakWindowOptions.build() : null;
        }

        @Override
        public final Builder offPeakWindowOptions(OffPeakWindowOptions offPeakWindowOptions) {
            this.offPeakWindowOptions = offPeakWindowOptions;
            return this;
        }

        public final SoftwareUpdateOptions.Builder getSoftwareUpdateOptions() {
            return softwareUpdateOptions != null ? softwareUpdateOptions.toBuilder() : null;
        }

        public final void setSoftwareUpdateOptions(SoftwareUpdateOptions.BuilderImpl softwareUpdateOptions) {
            this.softwareUpdateOptions = softwareUpdateOptions != null ? softwareUpdateOptions.build() : null;
        }

        @Override
        public final Builder softwareUpdateOptions(SoftwareUpdateOptions softwareUpdateOptions) {
            this.softwareUpdateOptions = softwareUpdateOptions;
            return this;
        }

        public final String getDomainProcessingStatus() {
            return domainProcessingStatus;
        }

        public final void setDomainProcessingStatus(String domainProcessingStatus) {
            this.domainProcessingStatus = domainProcessingStatus;
        }

        @Override
        public final Builder domainProcessingStatus(String domainProcessingStatus) {
            this.domainProcessingStatus = domainProcessingStatus;
            return this;
        }

        @Override
        public final Builder domainProcessingStatus(DomainProcessingStatusType domainProcessingStatus) {
            this.domainProcessingStatus(domainProcessingStatus == null ? null : domainProcessingStatus.toString());
            return this;
        }

        public final List<ModifyingProperties.Builder> getModifyingProperties() {
            List<ModifyingProperties.Builder> result = ModifyingPropertiesListCopier.copyToBuilder(this.modifyingProperties);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setModifyingProperties(Collection<ModifyingProperties.BuilderImpl> modifyingProperties) {
            this.modifyingProperties = ModifyingPropertiesListCopier.copyFromBuilder(modifyingProperties);
        }

        @Override
        public final Builder modifyingProperties(Collection<ModifyingProperties> modifyingProperties) {
            this.modifyingProperties = ModifyingPropertiesListCopier.copy(modifyingProperties);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder modifyingProperties(ModifyingProperties... modifyingProperties) {
            modifyingProperties(Arrays.asList(modifyingProperties));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder modifyingProperties(Consumer<ModifyingProperties.Builder>... modifyingProperties) {
            modifyingProperties(Stream.of(modifyingProperties).map(c -> ModifyingProperties.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public DomainStatus build() {
            return new DomainStatus(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
