/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.opensearch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Lists all instance types and available features for a given OpenSearch or Elasticsearch version.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceTypeDetails implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceTypeDetails.Builder, InstanceTypeDetails> {
    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(InstanceTypeDetails::instanceTypeAsString))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<Boolean> ENCRYPTION_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EncryptionEnabled").getter(getter(InstanceTypeDetails::encryptionEnabled))
            .setter(setter(Builder::encryptionEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionEnabled").build()).build();

    private static final SdkField<Boolean> COGNITO_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CognitoEnabled").getter(getter(InstanceTypeDetails::cognitoEnabled))
            .setter(setter(Builder::cognitoEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CognitoEnabled").build()).build();

    private static final SdkField<Boolean> APP_LOGS_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AppLogsEnabled").getter(getter(InstanceTypeDetails::appLogsEnabled))
            .setter(setter(Builder::appLogsEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AppLogsEnabled").build()).build();

    private static final SdkField<Boolean> ADVANCED_SECURITY_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AdvancedSecurityEnabled").getter(getter(InstanceTypeDetails::advancedSecurityEnabled))
            .setter(setter(Builder::advancedSecurityEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdvancedSecurityEnabled").build())
            .build();

    private static final SdkField<Boolean> WARM_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("WarmEnabled").getter(getter(InstanceTypeDetails::warmEnabled)).setter(setter(Builder::warmEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WarmEnabled").build()).build();

    private static final SdkField<List<String>> INSTANCE_ROLE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("InstanceRole")
            .getter(getter(InstanceTypeDetails::instanceRole))
            .setter(setter(Builder::instanceRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceRole").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> AVAILABILITY_ZONES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AvailabilityZones")
            .getter(getter(InstanceTypeDetails::availabilityZones))
            .setter(setter(Builder::availabilityZones))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZones").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_TYPE_FIELD,
            ENCRYPTION_ENABLED_FIELD, COGNITO_ENABLED_FIELD, APP_LOGS_ENABLED_FIELD, ADVANCED_SECURITY_ENABLED_FIELD,
            WARM_ENABLED_FIELD, INSTANCE_ROLE_FIELD, AVAILABILITY_ZONES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String instanceType;

    private final Boolean encryptionEnabled;

    private final Boolean cognitoEnabled;

    private final Boolean appLogsEnabled;

    private final Boolean advancedSecurityEnabled;

    private final Boolean warmEnabled;

    private final List<String> instanceRole;

    private final List<String> availabilityZones;

    private InstanceTypeDetails(BuilderImpl builder) {
        this.instanceType = builder.instanceType;
        this.encryptionEnabled = builder.encryptionEnabled;
        this.cognitoEnabled = builder.cognitoEnabled;
        this.appLogsEnabled = builder.appLogsEnabled;
        this.advancedSecurityEnabled = builder.advancedSecurityEnabled;
        this.warmEnabled = builder.warmEnabled;
        this.instanceRole = builder.instanceRole;
        this.availabilityZones = builder.availabilityZones;
    }

    /**
     * <p>
     * The instance type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link OpenSearchPartitionInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The instance type.
     * @see OpenSearchPartitionInstanceType
     */
    public final OpenSearchPartitionInstanceType instanceType() {
        return OpenSearchPartitionInstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * The instance type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link OpenSearchPartitionInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The instance type.
     * @see OpenSearchPartitionInstanceType
     */
    public final String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * Whether encryption at rest and node-to-node encryption are supported for the instance type.
     * </p>
     * 
     * @return Whether encryption at rest and node-to-node encryption are supported for the instance type.
     */
    public final Boolean encryptionEnabled() {
        return encryptionEnabled;
    }

    /**
     * <p>
     * Whether Amazon Cognito access is supported for the instance type.
     * </p>
     * 
     * @return Whether Amazon Cognito access is supported for the instance type.
     */
    public final Boolean cognitoEnabled() {
        return cognitoEnabled;
    }

    /**
     * <p>
     * Whether logging is supported for the instance type.
     * </p>
     * 
     * @return Whether logging is supported for the instance type.
     */
    public final Boolean appLogsEnabled() {
        return appLogsEnabled;
    }

    /**
     * <p>
     * Whether fine-grained access control is supported for the instance type.
     * </p>
     * 
     * @return Whether fine-grained access control is supported for the instance type.
     */
    public final Boolean advancedSecurityEnabled() {
        return advancedSecurityEnabled;
    }

    /**
     * <p>
     * Whether UltraWarm is supported for the instance type.
     * </p>
     * 
     * @return Whether UltraWarm is supported for the instance type.
     */
    public final Boolean warmEnabled() {
        return warmEnabled;
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceRole property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInstanceRole() {
        return instanceRole != null && !(instanceRole instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Whether the instance acts as a data node, a dedicated master node, or an UltraWarm node.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceRole} method.
     * </p>
     * 
     * @return Whether the instance acts as a data node, a dedicated master node, or an UltraWarm node.
     */
    public final List<String> instanceRole() {
        return instanceRole;
    }

    /**
     * For responses, this returns true if the service returned a value for the AvailabilityZones property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAvailabilityZones() {
        return availabilityZones != null && !(availabilityZones instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The supported Availability Zones for the instance type.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAvailabilityZones} method.
     * </p>
     * 
     * @return The supported Availability Zones for the instance type.
     */
    public final List<String> availabilityZones() {
        return availabilityZones;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(cognitoEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(appLogsEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(advancedSecurityEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(warmEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceRole() ? instanceRole() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAvailabilityZones() ? availabilityZones() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceTypeDetails)) {
            return false;
        }
        InstanceTypeDetails other = (InstanceTypeDetails) obj;
        return Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(encryptionEnabled(), other.encryptionEnabled())
                && Objects.equals(cognitoEnabled(), other.cognitoEnabled())
                && Objects.equals(appLogsEnabled(), other.appLogsEnabled())
                && Objects.equals(advancedSecurityEnabled(), other.advancedSecurityEnabled())
                && Objects.equals(warmEnabled(), other.warmEnabled()) && hasInstanceRole() == other.hasInstanceRole()
                && Objects.equals(instanceRole(), other.instanceRole()) && hasAvailabilityZones() == other.hasAvailabilityZones()
                && Objects.equals(availabilityZones(), other.availabilityZones());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceTypeDetails").add("InstanceType", instanceTypeAsString())
                .add("EncryptionEnabled", encryptionEnabled()).add("CognitoEnabled", cognitoEnabled())
                .add("AppLogsEnabled", appLogsEnabled()).add("AdvancedSecurityEnabled", advancedSecurityEnabled())
                .add("WarmEnabled", warmEnabled()).add("InstanceRole", hasInstanceRole() ? instanceRole() : null)
                .add("AvailabilityZones", hasAvailabilityZones() ? availabilityZones() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "EncryptionEnabled":
            return Optional.ofNullable(clazz.cast(encryptionEnabled()));
        case "CognitoEnabled":
            return Optional.ofNullable(clazz.cast(cognitoEnabled()));
        case "AppLogsEnabled":
            return Optional.ofNullable(clazz.cast(appLogsEnabled()));
        case "AdvancedSecurityEnabled":
            return Optional.ofNullable(clazz.cast(advancedSecurityEnabled()));
        case "WarmEnabled":
            return Optional.ofNullable(clazz.cast(warmEnabled()));
        case "InstanceRole":
            return Optional.ofNullable(clazz.cast(instanceRole()));
        case "AvailabilityZones":
            return Optional.ofNullable(clazz.cast(availabilityZones()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("InstanceType", INSTANCE_TYPE_FIELD);
        map.put("EncryptionEnabled", ENCRYPTION_ENABLED_FIELD);
        map.put("CognitoEnabled", COGNITO_ENABLED_FIELD);
        map.put("AppLogsEnabled", APP_LOGS_ENABLED_FIELD);
        map.put("AdvancedSecurityEnabled", ADVANCED_SECURITY_ENABLED_FIELD);
        map.put("WarmEnabled", WARM_ENABLED_FIELD);
        map.put("InstanceRole", INSTANCE_ROLE_FIELD);
        map.put("AvailabilityZones", AVAILABILITY_ZONES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InstanceTypeDetails, T> g) {
        return obj -> g.apply((InstanceTypeDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceTypeDetails> {
        /**
         * <p>
         * The instance type.
         * </p>
         * 
         * @param instanceType
         *        The instance type.
         * @see OpenSearchPartitionInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenSearchPartitionInstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The instance type.
         * </p>
         * 
         * @param instanceType
         *        The instance type.
         * @see OpenSearchPartitionInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenSearchPartitionInstanceType
         */
        Builder instanceType(OpenSearchPartitionInstanceType instanceType);

        /**
         * <p>
         * Whether encryption at rest and node-to-node encryption are supported for the instance type.
         * </p>
         * 
         * @param encryptionEnabled
         *        Whether encryption at rest and node-to-node encryption are supported for the instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionEnabled(Boolean encryptionEnabled);

        /**
         * <p>
         * Whether Amazon Cognito access is supported for the instance type.
         * </p>
         * 
         * @param cognitoEnabled
         *        Whether Amazon Cognito access is supported for the instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cognitoEnabled(Boolean cognitoEnabled);

        /**
         * <p>
         * Whether logging is supported for the instance type.
         * </p>
         * 
         * @param appLogsEnabled
         *        Whether logging is supported for the instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appLogsEnabled(Boolean appLogsEnabled);

        /**
         * <p>
         * Whether fine-grained access control is supported for the instance type.
         * </p>
         * 
         * @param advancedSecurityEnabled
         *        Whether fine-grained access control is supported for the instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder advancedSecurityEnabled(Boolean advancedSecurityEnabled);

        /**
         * <p>
         * Whether UltraWarm is supported for the instance type.
         * </p>
         * 
         * @param warmEnabled
         *        Whether UltraWarm is supported for the instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder warmEnabled(Boolean warmEnabled);

        /**
         * <p>
         * Whether the instance acts as a data node, a dedicated master node, or an UltraWarm node.
         * </p>
         * 
         * @param instanceRole
         *        Whether the instance acts as a data node, a dedicated master node, or an UltraWarm node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceRole(Collection<String> instanceRole);

        /**
         * <p>
         * Whether the instance acts as a data node, a dedicated master node, or an UltraWarm node.
         * </p>
         * 
         * @param instanceRole
         *        Whether the instance acts as a data node, a dedicated master node, or an UltraWarm node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceRole(String... instanceRole);

        /**
         * <p>
         * The supported Availability Zones for the instance type.
         * </p>
         * 
         * @param availabilityZones
         *        The supported Availability Zones for the instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(Collection<String> availabilityZones);

        /**
         * <p>
         * The supported Availability Zones for the instance type.
         * </p>
         * 
         * @param availabilityZones
         *        The supported Availability Zones for the instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(String... availabilityZones);
    }

    static final class BuilderImpl implements Builder {
        private String instanceType;

        private Boolean encryptionEnabled;

        private Boolean cognitoEnabled;

        private Boolean appLogsEnabled;

        private Boolean advancedSecurityEnabled;

        private Boolean warmEnabled;

        private List<String> instanceRole = DefaultSdkAutoConstructList.getInstance();

        private List<String> availabilityZones = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceTypeDetails model) {
            instanceType(model.instanceType);
            encryptionEnabled(model.encryptionEnabled);
            cognitoEnabled(model.cognitoEnabled);
            appLogsEnabled(model.appLogsEnabled);
            advancedSecurityEnabled(model.advancedSecurityEnabled);
            warmEnabled(model.warmEnabled);
            instanceRole(model.instanceRole);
            availabilityZones(model.availabilityZones);
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        public final Builder instanceType(OpenSearchPartitionInstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final Boolean getEncryptionEnabled() {
            return encryptionEnabled;
        }

        public final void setEncryptionEnabled(Boolean encryptionEnabled) {
            this.encryptionEnabled = encryptionEnabled;
        }

        @Override
        public final Builder encryptionEnabled(Boolean encryptionEnabled) {
            this.encryptionEnabled = encryptionEnabled;
            return this;
        }

        public final Boolean getCognitoEnabled() {
            return cognitoEnabled;
        }

        public final void setCognitoEnabled(Boolean cognitoEnabled) {
            this.cognitoEnabled = cognitoEnabled;
        }

        @Override
        public final Builder cognitoEnabled(Boolean cognitoEnabled) {
            this.cognitoEnabled = cognitoEnabled;
            return this;
        }

        public final Boolean getAppLogsEnabled() {
            return appLogsEnabled;
        }

        public final void setAppLogsEnabled(Boolean appLogsEnabled) {
            this.appLogsEnabled = appLogsEnabled;
        }

        @Override
        public final Builder appLogsEnabled(Boolean appLogsEnabled) {
            this.appLogsEnabled = appLogsEnabled;
            return this;
        }

        public final Boolean getAdvancedSecurityEnabled() {
            return advancedSecurityEnabled;
        }

        public final void setAdvancedSecurityEnabled(Boolean advancedSecurityEnabled) {
            this.advancedSecurityEnabled = advancedSecurityEnabled;
        }

        @Override
        public final Builder advancedSecurityEnabled(Boolean advancedSecurityEnabled) {
            this.advancedSecurityEnabled = advancedSecurityEnabled;
            return this;
        }

        public final Boolean getWarmEnabled() {
            return warmEnabled;
        }

        public final void setWarmEnabled(Boolean warmEnabled) {
            this.warmEnabled = warmEnabled;
        }

        @Override
        public final Builder warmEnabled(Boolean warmEnabled) {
            this.warmEnabled = warmEnabled;
            return this;
        }

        public final Collection<String> getInstanceRole() {
            if (instanceRole instanceof SdkAutoConstructList) {
                return null;
            }
            return instanceRole;
        }

        public final void setInstanceRole(Collection<String> instanceRole) {
            this.instanceRole = InstanceRoleListCopier.copy(instanceRole);
        }

        @Override
        public final Builder instanceRole(Collection<String> instanceRole) {
            this.instanceRole = InstanceRoleListCopier.copy(instanceRole);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceRole(String... instanceRole) {
            instanceRole(Arrays.asList(instanceRole));
            return this;
        }

        public final Collection<String> getAvailabilityZones() {
            if (availabilityZones instanceof SdkAutoConstructList) {
                return null;
            }
            return availabilityZones;
        }

        public final void setAvailabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZoneListCopier.copy(availabilityZones);
        }

        @Override
        public final Builder availabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZoneListCopier.copy(availabilityZones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availabilityZones(String... availabilityZones) {
            availabilityZones(Arrays.asList(availabilityZones));
            return this;
        }

        @Override
        public InstanceTypeDetails build() {
            return new InstanceTypeDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
